(in-package "ZENO")
(use-package "VARIABLE")

" 1993 Copyright (c) University of Washington
  Written by J. Scott Penberthy

  Parts herein were derived from code written jointly by
  Tony Barrett, Daniel Weld, J. Scott Penberthy, and Stephen Soderland."

(defun PLAN (start end initial goals
             &key
	     (rank-fun #'rank3)
	     (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (zeno start end initial goals rank-fun search-fun)
    (declare (ignore q-len))
    (record-stat .cpu-time. time)
    (record-stat .branch. (float av-branch))
    (record-stat .success. done?)
    (values plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Main control level of ZENO

;;; Returns 5 values: final plan, t or nil denoting success, run time
;;; length of q at termination, average branching factor

(defun ZENO (start end init goal rank-fun search-fun)
  (init-zeno)
  (main-loop (make-goal-plan start end init goal) rank-fun search-fun))

(defun INIT-ZENO ()
  (reset-stat-vars)
  (init-zeno-time-map))
  
(defun MAIN-LOOP (goal-plan rank-fun search-fun)
  (let ((init-time (get-internal-run-time)))
    (multiple-value-bind (plan bfactor qlength)
	(funcall search-fun goal-plan
		 #'plan-refinements #'plan-test rank-fun *search-limit*)
      (values plan			; the plan itself
	      (if plan (plan-test plan))
	      (- (get-internal-run-time) init-time) ; time
	      qlength			; final length of the queue
	      bfactor))))

(defun MAKE-GOAL-PLAN (start end init goal)
  ;; Returns the goal plan for a planning problem. 
;  (reset-stat-vars)			; clear globals for metering
  (test-wd goal nil)
  (setf start (instantiate-term start 0))
  (setf end (instantiate-term end 0))
  (init-zeno-time-map)
  (let* ((g (instantiate-term goal 0))
	 (istep (make-initial-step start end g init))
	 (init-plan (tweak-plan 
		     nil
		     :names nil
		     :constraints nil
		     :links nil
		     :flaws nil
		     :reason '(:init)
		     :steps (list istep)
		     :add-goal
		      (make-openc :condition (p-step-precond istep)
				  :step 0
				  :time nil)
		     :ordering (new-ztm)
		     :bindings (new-bindings)
		     :high-step 0)))
    (dolist (c (p-step-ca istep))
      (add-metric-constraint init-plan c))
    (set-time< init-plan start end)
    (constrain-step-times init-plan istep)
    (values init-plan)))

(defun MAKE-INITIAL-STEP (start end goals effects)
  (let ((step (make-p-step :id 0
			   :action '(:init)
			   :start start
			   :end end
			   :parms nil
			   :precond (krsl-parse goals))))
    (parse&install-effects step (zenofy-effect effects nil))
    (when *debug*
    (format t "~&Init precond:~s~%Init efx:~s~%"
	    (p-step-precond step)
	    (p-step-add step)))
    (values step)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Goal test function for a search.
(defun PLAN-TEST (plan)
  (and
   plan
   (null (plan-flaws plan))
   (if (metric-constraints-sound? plan)
       t
       (progn (count-stat .invalid.) nil))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a list of one step refinements to the current plan.
;;;

(defun PLAN-REFINEMENTS (plan)
  (new-unsafes plan)
  (preprocess-plan plan)
  (postprocess-children
   plan (handle-flaw (select-flaw plan) plan)))

(defun select-flaw (plan)
  (or (get-unsafe plan) (get-open plan)
      (car (plan-flaws plan))))

(defun get-unsafe (plan)
  (find-if #'unsafe-p (plan-flaws plan) :from-end *fifo-flaws*))

(defun ok-goal? (thing plan)
  (and (openc-p thing)
       (if (procedurally-defined? (openc-condition thing))
	   (not (eq :no-match-attempted
		    (match-fact (openc-condition thing) plan)))
	 t)))

(defun get-open (plan)
  (find-if #'(lambda (f)
	       (ok-goal? f plan))
	   (plan-flaws plan) :from-end *fifo-flaws*))

(defun handle-flaw (flaw plan)
  (if (openc-p flaw)
      (handle-open flaw plan)
    (handle-unsafe flaw plan)))

(defun PREPROCESS-PLAN (plan)
  ;; Some quick checks on the plan *before* refinements are called.
  (declare (special p))
  (setf p plan) 
  (when *debug* (display-plan p))
  (push (rank3 p) *rank-history*)
  (check-zeno-time-map plan)
  )

(defun update-temporal-constraints (plan)
  (when (and *deadline-goals*
	     (plan-constraints plan))  ;; only do this if absolutely necessary
    (let ((ztm (plan-ordering plan)))
      (when (ztm-history ztm)
	(do ((constraint (pop (ztm-history ztm)) 
			 (pop (ztm-history ztm))))
	    ((null constraint))
	  (unless (and (eq (car constraint) '<=)
		       (eq (second constraint) (third constraint)))
	    (add-metric-constraint plan constraint)
	    )))
      plan)))

(defun POSTPROCESS-CHILDREN (plan kids)
  ;; After KIDS are generated as children of PLAN, check their
  ;; constraints for consistency, add debugging info, etc.
  (let ((good-kids nil))
    (dolist (k1 kids)
      (unless (plan-flaws k1) (new-unsafes k1))
      (dolist (k (check-resource-constraints (remove-forever-goals k1)))
        (update-temporal-constraints k)
	(when (and *show-complete-plans* (null (plan-flaws k)))
	  (let ((*debug* t)) (display-plan k)))
;;;	(equality-prune k)
	(cond ((metric-constraints-ok? k)
	       (push k good-kids)
	       (if (not (eq k plan))
		   (setf (plan-other k)
		     (acons :parent plan (plan-other k)))))
	      (t
	       (count-stat .inconsistent.)))))
    (setf (plan-other plan)
      (acons :kids (remove plan good-kids) (plan-other plan)))
    (unless good-kids
      (when *show-dead-ends* (display-plan plan))
      (count-stat .dead-end.))
    (values good-kids)))

(defun check-resource-constraints (plans)
  (cond ((not (eq *resource-check* :late))
	 plans)
	(t
	 (let ((result nil))
	   (dolist (plan plans)
	     (cond ((plan-test plan)
		    (setf result
		      (nconc (secure-all-resources plan) result)))
		   (t
		    (push plan result))))
	   (values result)))))

;;
;;; Greedy binding of atemporal goals
;;

(defun REMOVE-FOREVER-GOALS (plan)
  (cond (*greedy*
	 (remove-forever-goals-1 plan))
	(t
	 (list plan))))

(defun GROUNDED-FOREVER-GOAL-P (goal plan)
  (let ((thing (openc-condition goal)))
    (cond ((eq *greedy* :desperate)
	    (eq :forever (theta-time thing)))
	  (t
	   (and (eq :forever (theta-time thing))
		(grounded-p (theta-args thing) (plan-bindings plan)))))))

(defun REMOVE-FOREVER-GOALS-1 (plan)
  (let ((forever-goals nil)
	(new-plans nil)
	(new-flaws nil)
	(plans nil)
	(temp nil))
    (dolist (g (plan-flaws plan))
      (cond ((openc-p g)
	     (if (grounded-forever-goal-p g plan)
		 (push g forever-goals)
	       (push g new-flaws)))
	    (t
	     (push g new-flaws))))
    (cond (forever-goals
	   (setf plans (list plan))
	   (setf (plan-flaws plan) (nreverse new-flaws))
	   (dolist (g forever-goals)
	     (dolist (p plans)
	       (setf temp
		 (link-to-forever nil (openc-condition g) p nil))
	       (when temp
		 (setf new-plans (nconc temp new-plans))))
	     (setf plans new-plans)
	     (setf new-plans nil))
	   (values plans))
	  (t
	   (values (list plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. ranking partial plans

(defun METRIC-PENALTY (plan)
  ;; Penalize a plan for having nonlinear constraints
  (let ((c (plan-constraints plan)))
    (cond ((null c)
	   0)
	  (t
	   (if (consp c) (setf c (car c)))
	   (+ (* 10 (length (nlin-table (cs-nlin c))))
	      (* 2 (length (ineq-tableau (cs-ineq c))))
	      (length (eqn-forms (cs-eqn c))))))))

(defun COUNT-GOALS (plan)
  (let ((count 0))
    (dolist (o (plan-flaws plan))
      (when (openc-p o)
	(if (not (eq :forever (openc-time o))) 
	    (incf count))))
    (values count)))

(defun COUNT-UNSAFE (plan)
  (let ((count 0))
    (dolist (o (plan-flaws plan))
      (when (unsafe-p o) (incf count)))
    (values count)))

(defun simple-rank (plan) (+ (length (plan-flaws plan))
			     (length (plan-steps plan))))

(defun RANK3 (plan)
  (let ((num-steps (length (plan-steps plan)))
	(unsafe (count-unsafe plan))
	(open (count-goals plan))
	(links (length (plan-links plan)))
	(metric (metric-penalty plan)))
;    num-steps
     (+ (* (elt *ranker* 0) unsafe)
        (* (elt *ranker* 1) open)
        (* (elt *ranker* 2) num-steps)
        (* (elt *ranker* 3) links)
	(* (elt *ranker* 4) metric))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Handling open conditions

;;
;; Need to upgrade to a "splitting" routine.  We're missing
;; it in this implementation.
;;

(defun HANDLE-OPEN (open-cond plan)
  (when open-cond   ;; NIL when plan is final, but bad constraints
   (let ((open (openc-condition open-cond))
         (plans nil))
     (case (car open)
       (:and
	(setf (plan-flaws plan) (remove-1 open-cond (plan-flaws plan)))
        (push-non-nil
  	  (handle-and open
		      (openc-step open-cond)
		      (openc-time open-cond)
		      plan
		      (openc-marked? open-cond)) plans))
	(:or       (handle-or open-cond plan))
       ((= <= >= < > :eq :neq)  
	(handle-constraint open-cond plan))
       ((:forall :exists)
	(error "Lone ~s on the goal stack:~%~s" (car open) open))
       (otherwise
	(handle-simple-goal open-cond plan))))))

(defun DEREF-FN (fn time)
  ;; Fn is of the form
  ;; (== (f t x1...xn) v)
  ;;
  ;; Transform it by replacing "t" with TIME, then
  ;; its variable with v@time.
  ;;
  `(== (,(caadr fn) ,time ,@(cddr (second fn)))
       ,(deref-fnvar (third fn) time)))

(defun HANDLE-SIMPLE-GOAL (open-cond plan)
  ;; Open-cond is one of
  ;;
  ;; (R t x1...xn)        over time
  ;; (== (f t x1..xn) v)  over time
  ;;
  (let ((theta (openc-condition open-cond))
	(time (openc-time open-cond)))
    (cond ((is-a-fn? theta)
	   (cond ((or (null time) (eq :point (i-type time)))
		  (satisfy-goal open-cond plan))
		 ((not (eq :closed (i-type time)))
		  (error "Illegal interval goal for functions:~%~s" 
		  	 open-cond))
		 ((or (not *goal-splitting*)
		      (openc-marked? open-cond))
		  (satisfy-goal open-cond plan))
                 (t (split&solve open-cond plan))))
	  ((or (not *goal-splitting*)
	       (openc-marked? open-cond))
	   (satisfy-goal open-cond plan))
	  (t
	   (split&solve open-cond plan)))))

(defun SPLIT&SOLVE (open-cond plan)
  ;; Return one (or more) plans:
  ;;
  ;; 1. Split the original goal into two pieces.
  ;; 2. Mark the current goal, then solve.
  ;;
  (let ((result nil)
	(split nil))
    (setf (openc-marked? open-cond) t)
    (nconc
     (satisfy-goal open-cond plan)
     (split-goal open-cond plan))))
	 
(defvar *GOAL-SPLITTING* t  "Set to NIL to disable temporal goal splitting")

(defvar *max-goal-splits* 0
  "This is the maximum depth (N) of binary 'goal-split' tree.  A goal
will only be split N times.")

(defvar *t-counter* 0)

(defun SPLIT-GOAL (open-cond plan)
  ;;
  ;; Split the given goal (open-cond) into two, temporally
  ;; adjacent pieces.
  ;;
  (let* ((gtime (openc-time open-cond))
	 (level (1+ (openc-level open-cond)))
	 (gtype (i-type gtime)))
    (unless (or (not *goal-splitting*)
		(> level *max-goal-splits*)
		(eq :point gtype))
      (let ((tx (variable::make-variable '?T (incf *t-counter*)))
	    (t0 (i-start gtime))
	    (t1 (i-end gtime))
	    (clause (openc-condition open-cond))
	    (new-flaws (remove-1 open-cond (plan-flaws plan)))
	    (newp nil))
	(setf newp
	  (tweak-plan
	   plan
	   :reason (list :split (list gtype t0 t1) :at tx)
	   :ordering (copy-ztm (plan-ordering plan))
	   :flaws new-flaws
	   ))
	(macrolet ((split-it (time)
		     `(let ((value ,time))
			(push
			 (make-openc
			  :level level
			  :condition clause
			  :time value
			  :step (openc-step open-cond)
			  :marked? nil)
			 (plan-flaws newp))
			(when *debug*
			  (format t "~&Splitting goal ~s (level ~d): ~s~%"
				  clause level
				  (print-interval value nil)))
			(set-time< newp t0 tx)
			(set-time< newp tx t1))))
	  (split-it (list (if (is-a-fn? clause)
			      gtype
			    :open-end) t0 tx))
	  (split-it (list gtype tx t1))
	  (values (list newp)))))))

(defun SATISFY-GOAL (open-cond plan)
  ;;
  ;; The old fashioned way.
  ;;
  (let ((gtheta (openc-condition open-cond)))
    ;; we're committed, so mark the time.
    (when (null (openc-time open-cond))
      (setf (openc-time open-cond) (theta-time gtheta)))
    (cond ((eq :forever (theta-time gtheta))
	   (link-to-forever open-cond gtheta plan T))
	  ((procedurally-defined? gtheta)
	   (handle-fact gtheta plan))
	  (t
	   (nconc (when (eq :not (car gtheta))
		    (cw-assume open-cond plan))
		  (add-step open-cond plan)
		  (reuse-step open-cond plan))))))

(defun handle-fact (gtheta plan)
  (let ((result (match-fact gtheta plan)))
    (cond ((or (null result)
	       (eq result :no-match-attempted))
	   nil)
	  ((eq result :ok)
	   (values (list plan)))
	  (t
	   (setf result
	     (tweak-plan
	      :reason `(:fact ,gtheta)
	      :bindings (add-bind result (plan-bindings plan))
	      ))
	   (when result
	     (list result))))))

(defun LINK-TO-FOREVER (open gtheta plan record?)
  (let ((plans nil)
	(entries (cdr (assoc (theta-pred gtheta) *forever*)))
	(bind (plan-bindings plan))
	(fn? (is-a-fn? gtheta))
	(new-flaws (if open (remove-1 open (plan-flaws plan))
		     (plan-flaws plan)))
	(p1 nil)
	(b nil))
    (dolist (e entries)
      (when (setf b (mgu gtheta e bind))
	(setf p1 
	  (tweak-plan plan
		      :reason (if record?
				  `(:fact ,gtheta)
				:same)
		      :flaws new-flaws
		      :bindings (add-bind (car b) bind)))
	(when p1
;	  (print b)
;	  (display-plan p1)
	  (when fn?
	    (setf (plan-constraints p1)
	      (copy-metric-constraints plan))
	    (add-metric-constraint p1 `(= ,(theta-var gtheta)
					 ,(theta-var e)))
	    (add-metric-constraint p1 `(:eq ,(theta-var gtheta)
					    ,(theta-var e)))
;	    (show (plan-constraints p1))
	    (push e (plan-names p1))
	    (push gtheta (plan-names p1)))
	  (push p1 plans))))
    (values plans)))
		
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Handling constraints: 
;;;    Post them to the collective set of constraints for
;;;  consistency.
;;;

(defun HANDLE-CONSTRAINT (open-cond plan)
  "Adds the given CONSTRAINT to PLAN.
   A singleton set of the newly modified plan is returned."
  (let ((constraint (openc-condition open-cond)))
    ;(pop (plan-flaws plan))
    (add-metric-constraint plan constraint)
    (setf (plan-flaws plan) (remove-1 open-cond (plan-flaws plan)))
    (values (list plan))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Handling disjunction
(defun HANDLE-OR (goal plan &aux ret)
  (dolist (g (cdr (openc-condition goal)) (delete nil ret))
    (push-non-nil (tweak-plan plan 
		      :reason `(:goal ,g ,(openc-step goal))
		      :flaws (remove-1 goal (plan-flaws plan))
		      :add-goal (make-openc :condition g
					    :step (openc-step goal)
					    :time (openc-time goal)
					    :marked?
					    (openc-marked? goal)))
	           ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.1. Adding new steps and reusing steps

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Returns a list of pairs of new plans and step#.  The new plan
;;; establishes the condition by adding a new step, instantiated 
;;; from a step in *templates* with post-conditions that unify with
;;; the condition. 
(defun ADD-STEP (open-cond plan)
  (let ((new-step-num (1+ (plan-high-step plan))))
    (new-link open-cond
	      (mapcar #'(lambda (templ) (instantiate-step templ new-step-num))
		      (get-opers (openc-condition open-cond)))
	      plan)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Quickly get those operators in the templates that match a condition
(defun SAME-PREDICATE-P (theta1 theta2)
  (and (eq (car theta1) (car theta2))
       (eq (theta-pred theta1) (theta-pred theta2))))

(defun GET-OPERS (theta &aux (ret nil))
  (let ((key (theta-pred theta)))
    (labels ((test-templ (templ)
	       (dolist (e (cdr (assoc key (p-step-add templ))) nil)
		 (when (same-predicate-p (effect-post e) theta)
		   (return-from test-templ t)))))
      (dolist (templ *templates*)
	(when (test-templ templ)
	  (push templ ret)))
      (if (and (null ret)
	       (not (eq :not (car theta))))
	  (error "Oops!  You haven't defined any actions that affect~%~
                  clauses beginning with '~s'" (theta-pred theta))
	ret))))

;;;;;;;;;;;;;;;;;;;
;;; 
(defun REUSE-STEP (open-cond plan)
  (let ((ztm (plan-ordering plan)))
    (let ((bad-times (decode-ztime
		      ztm
		      (nec-> ztm (i-start (openc-time open-cond))))))
    ;; you can reuse any step, provided it doesn't come too late!
      (new-link
       open-cond
       (remove-if #'(lambda (s)
		      (eq-member (p-step-start s) bad-times))
		  (plan-steps plan))
       plan))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.2. Adding links to steps

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates new plans to achieve an open condition by creating a causal link.
;;;  If STEP is not nil then a new step is added to the plan.
;;;  Each effect tested to see if it has an add condition matching the 
;;;  open condition.  If so, a new plan is created with that link added.
;;;  Only the effects associated with STEPS are tried

(defun NEW-LINK (open-cond steps plan &aux (new-plans nil))
  (dolist (step steps)
    (dolist (effect (cdr (assoc (theta-pred (openc-condition open-cond))
				(p-step-add step))))
      (setf new-plans
	(nconc
	 (maybe-establish-link plan step effect open-cond)
	 new-plans))))
  new-plans)
	  
(defun EFFECT-TIME-OVERLAPS-GOAL-TIME-P (plan inf? etime gtime)
  ;; GTIME and ETIME are both interval specs.  Return T
  ;; if GTIME possibly precedes ETIME.
  ;;
  ;; INF? is T if the effect is an influence.
  ;;
  (let ((eztime (find-ztime! (plan-ordering plan) (i-start etime)))
	(gztime (find-ztime! (plan-ordering plan) (i-start gtime))))
    (or
     (null eztime)			;a new time can go anywhere
     (zerop
      (logand (ztime-mask eztime)
	      (ztime->= gztime)
              (if (and (eq-member (i-type etime) '(:open-start :open))
                       (not inf?))
                  all-time
 	          (ztime-neq gztime)))))))

(defun INFLUENCE-CHECK (inf? etime gtime)
  ;; Return T if its ok for the influence at ETIME to support
  ;; a goal at GTIME.  This function is a filter that
  ;; eliminates ``no-ops'' caused by an influence over [t0,t1]
  ;; satisfying a goal at [t0,t0].
  ;;
  (or (not inf?)
      ;; should we check to see if the timepoints are semantically
      ;; equivalent?  probably so.  But, then again, this is only
      ;; an empirical filter...
      (not
       (and inf?
             (eq (i-start etime) (i-start gtime))
             (eq (i-start etime) (i-end gtime)))))
  )

(defun MAYBE-ESTABLISH-LINK (plan step effect open-cond)
  ;; Return a list of plans that link open-cond to effect,
  ;; if possible.
  (let ((new-step? (> (effect-id effect) (plan-high-step plan)))
        (inf? (effect-influence-p effect))
        (etime (effect-time effect))
        (gtime (openc-time open-cond))
	(b nil))
    (when (and (effect-time-overlaps-goal-time-p plan inf? etime gtime)
               (influence-check inf? etime gtime)
	       (setf b (mgu (effect-post effect)
			    (openc-condition open-cond)
			    (plan-bindings plan))))
      (let ((peeled-goal (peel-goal (car b) effect))
	    (peeled-ca (peel-ca (car b) effect)))
	(when (and new-step? (p-step-precond step))
	  (setf peeled-goal (if (not (null peeled-goal))
				`(:and ,peeled-goal
				       ,(p-step-precond step))
			      (p-step-precond step))))
	(establish-links
	 plan step effect open-cond
	 new-step? b peeled-goal peeled-ca)))))

(defun ESTABLISH-LINKS (plan step effect open-cond
			new-step? bindings peeled-goal peeled-ca)
  (let ((theta-g (openc-condition open-cond))
	(et (effect-time effect))
	(gt (openc-time open-cond))
	(gcode (i-type (openc-time open-cond)))
	(ecode (i-type (effect-time effect))))
    (when *debug*
      (format t "~&Effect at ~s, goal at ~s~%" et gt))
  (cond ((is-a-fn? theta-g)
	 (funcall
	  (case ecode
	    (:point 'rlink-pt->goal)
	    (:open 'rlink-open->goal)
	    (:open-start 'rlink-open-start->goal)
	    (otherwise (error "~s metric effects are illegal:~%~s."
			      ecode effect)))
	  plan step effect open-cond
	  new-step? bindings peeled-goal peeled-ca))
	(t
	 (let ((stuff (getf (getf *logical-link-table* gcode) ecode)))
	   (cond ((null stuff)
		  (error "All relational goals must start at a well-defined~%~
                          time point.  All effects must start after the~%~
                          beginning of a step.  You've specified:~%~%~
                          Effect of ~a: ~s at time ~s~%~
                          Goal of ~a:   ~s at time ~s~%"
			 (car (p-step-action step))
			 (effect-post effect) (effect-time effect)
			 (car (p-step-action
			       (get-step-with-id plan (openc-step open-cond))))
			 (openc-condition open-cond)
			 (openc-time open-cond)

			 ))
		 (t
		  (setf stuff
			(sublis
			 `((t0 . ,(i-start et))
			   (t1 . ,(i-end et))
			   (t2 . ,(i-start gt))
			   (t3 . ,(i-end gt))) stuff))
		  (follow-link-instructions
		   stuff plan step effect open-cond new-step?
		   bindings peeled-goal peeled-ca))))))))

(defun FOLLOW-LINK-INSTRUCTIONS
    (stuff plan step effect open-cond
     new-step? bindings peeled-goal peeled-ca)
  (let ((plans nil)
	(newp nil)
	(theta-g (openc-condition open-cond))
	(reason (if new-step? 
		    `(:step ,(p-step-id step)
			    ,(openc-condition open-cond))
		  `(:link ,(p-step-id step)
			  ,(openc-condition open-cond))))
	(flaws (remove-1 open-cond (plan-flaws plan)))
	(bind (add-bind (peel-binds (car bindings) effect) 
                        (plan-bindings plan)))
	(high-step (if new-step? (effect-id effect) (plan-high-step plan)))
	(steps (if new-step? (cons step (plan-steps plan)) 
	                     (plan-steps plan)))
	(goal (when peeled-goal
		(make-openc :condition peeled-goal
			    :step (p-step-id step) :time nil)))
	(ztm nil)
	(con nil)
	(link nil)
	)
    (dolist (instr stuff)
      (if peeled-ca 
          (setf con (copy-metric-constraints plan))
         (setf con (plan-constraints plan)))
      (setf ztm (copy-ztm (plan-ordering plan)))
      (dolist (time (getf instr :time))
	;; time is (order t1 t2)
	(let ((t1 (find-ztime ztm (second time)))
	      (t2 (find-ztime ztm (third time))))
	  (ecase (car time)
	    (<  (zset< ztm t1 t2))
	    (<= (zset<= ztm t1 t2))))) 
      (when (ztimes-consistent? ztm)
	(setf newp
	  (tweak-plan plan
		      :reason reason :flaws flaws :steps steps
		      :constraints con :ordering ztm :bindings bind
		      :add-goal goal :high-step high-step))
	(when *debug* (format t "~%New plan is ~s" newp))
	(when newp
         (when new-step?
           (set-time< newp (p-step-start step) (p-step-end step)))
	  (dolist (c (p-step-ca step))
	    (add-metric-constraint newp c))
	  (dolist (c peeled-ca) (add-metric-constraint newp c))
	  (setf link (getf instr :link))
	  ;; link is (interval-type from-time to-time)
	  (push
	   (make-link :time link
		      :condition theta-g
		      :Si (p-step-id step)
                      :effect effect
		      :Sj (openc-step open-cond))
	   (plan-links newp))
	  ;; record new link & steps for later threat detection.
	  (setf (plan-other newp)
	    (acons :new (list (car (plan-links newp))
			      (if new-step? step))
		   (plan-other newp)))
	  (when *debug*
	      (format t "~&Linking ~s for ~s~%"
		      (getf instr :link) theta-g)
	    ;(display-plan newp)
	    )
          ;; returns all possibilities for legal Step instances
	  (cond (new-step?
		 (setf plans (nconc plans (install-new-step newp step))))
		(t
		 (push newp plans))))))
    (values plans)))

(defun NEW-UNSAFES (plan)
  (let ((info (assoc :new (plan-other plan))))
    (cond ((null info) nil)
	  (t
	   (setf (plan-other plan)
	     (remove info (plan-other plan)))
	   (setf (plan-flaws plan)
	     (append (test-link plan (second info))
		     (when (third info)
		       (test-effects plan (p-step-add (third info))))
		     (plan-flaws plan)))))))

(defun INSTALL-NEW-STEP (plan step)
  ;; Return a plan with STEP installed, making sure
  ;; that all resources are safely allocated.
  (let ((init (get-step-with-id plan 0)))
    (set-time< plan (p-step-start init) (p-step-start step))
    (set-time< plan (p-step-end step) (p-step-end init))
  (set-time< plan (p-step-start step) (p-step-end step))
  (cond ((eq *resource-check* :early)
	 (secure-resources plan step))
	(t
	 (cons plan nil)))))


;;
;;; Handling of atomicity conditions.  We use the more common
;;; term "RESOURCE."
;;

(defun REMOVE-RESOURCE-CONFLICTS (good-plans new-step old-step bind)
  ;;
  ;; See Wilkins, D. "Practical Planning", pp.114-117
  ;;
  ;; I've done something very similar here.  We can order steps before
  ;; or after one another when a resource conflict occurs.  We can also
  ;; post non-codesignation constraints among the resource descriptor's
  ;; arguments.  The latter corresponds to what Wilkins called
  ;; 'optional, not-same' constraints.
  ;;
  (let ((result nil))
    (dolist (plan good-plans)
      (setf result
	(nconc result
	       (induce-step-order plan new-step old-step)
	       (induce-step-order plan old-step new-step)
	       (separate-for-resource plan bind))))
    (values result)))
	       
(defun SECURE-RESOURCES (plan step)
  (let ((bind (plan-bindings plan))
	(b nil)
	(good-plans (list plan)))
    (dolist (other (plan-steps plan))
      ;; Spec 'describes' the resource used by the OTHER step
      (unless (eq other step)
	;; feh..  We should be smarter and check for the
	;; existing temporal constraints.
	(dolist (in-use (p-step-res other))
	  (dolist (needed (p-step-res step))
	    (setf b (mgu in-use needed bind))
	    ;; we have a match... now remove the conflict!
	    (if *debug*
		(format t "~&Resource conflict on ~s:~%   ~a vs.~%   ~a~%"
			(bind-variable in-use bind)
			(bind-variable (p-step-action step) bind)
			(bind-variable (p-step-action other) bind)))
	    (when b
	      (setf good-plans
	(remove-resource-conflicts good-plans step other (car b))))))))
    (values good-plans)))


(defun SECURE-ALL-RESOURCES (plan)
  (let ((bind (plan-bindings plan))
	(remaining-steps (cdr (plan-steps plan)))
	(b nil)
	(good-plans (list plan)))
    (unless *ignore-resources* 
    (dolist (other (plan-steps plan))
      ;; Spec 'describes' the resource used by the OTHER step
      (dolist (step remaining-steps)
	;; feh..  We should be smarter and check for the
	;; existing temporal constraints.
	(dolist (in-use (p-step-res other))
	  (dolist (needed (p-step-res step))
	    (setf b (mgu in-use needed bind))
	    ;; we have a match... now remove the conflict!
	    (if *debug*
		(format t "~&Resource conflict on ~s:~%   ~a vs.~%   ~a~%"
			(bind-variable in-use bind)
			(bind-variable (p-step-action step) bind)
			(bind-variable (p-step-action other) bind)))
	    (when b
	      (setf good-plans
		(remove-resource-conflicts good-plans step other (car b)))))))
      (setf remaining-steps (cdr remaining-steps))))
    (values good-plans)))

(defun INDUCE-STEP-ORDER (plan step1 step2)
  (let ((i1 `(:closed ,(p-step-start step1) ,(p-step-end step1)))
	(i2 `(:closed ,(p-step-start step2) ,(p-step-end step2)))
	(ztm (plan-ordering plan)))
    (cond ((ztime-within-p ztm (i-end i1) (nec-< ztm (i-start i2)))
	   (values (list plan)))
	  ((order-ok? ztm i1 i2)
	   (setf ztm (copy-ztm ztm))
	   (order-times ztm i1 i2)
	   (values
	    (list
	     (tweak-plan plan
		 :reason (append (cdr (assoc :reason (plan-other plan)))
                          `(:resource-conflict :step-order ,step1 ,step2))
			 :ordering ztm ))))
	  (t
	   (values nil)))))

(defun SEPARATE-FOR-RESOURCE (plan bind)
  (cond ((null bind) nil)
	(t
	 (let ((subgoal `(:or ,@(mapcar #'(lambda (x) 
                                            `(:neq ,(car x) ,(cdr x)))
					bind))))
	   (list
	    (tweak-plan plan
			:reason `(:resource-conflict :separate ,subgoal)
			:add-goal subgoal))))))

