;;
;;; A new, faster approach to Time using bitvectors.
;;;
;;; Times are stored in a static vector whose last element
;;; represents the "last" available time.  This is a "ZTM" or
;;; Zeno Time Map.  You can extend the size by varying
;;; .MAX-TIME.
;;

(in-package "ZENO")
;(proclaim '(optimize (speed 3) (space 1) (safety 1) (compile-speed 0)))

(defconstant empty-set 0)
(defconstant full-set -1)
(defconstant all-time -1)

(defvar .max-time. 8)
(defvar .default-max-time. 8)
(defvar .time-threshold. 4)
(defvar .time-increment. 8)

(eval-when (load compile eval)
  (defmacro last-ztime (ztm)
    `(svref ,ztm 0))
  (defmacro ztm-history (ztm) `(cdr ,ztm))
      )

(defun init-zeno-time-map ()
  (setf .max-time. .default-max-time.))

(defun check-zeno-time-map (plan)
  (if (< (- .max-time. (last-ztime (car (plan-ordering plan))))
	 .time-threshold.)
      (incf .max-time. .time-increment.)))

(defstruct (ztime
	    (:copier emitz-ypoc)
	    (:type vector))
	   ;(:print-function print-ztime)
  (id nil)       ;some symbolic reference(s)
  (mask 0)
  (<= empty-set)
  (>= empty-set)
  (neq empty-set))

(defun print-ztime (z s d)
  (declare (ignore d))
  (princ 'z s)
  (princ (ztime-id z) s))


(defun new-ztm ()
  (let ((ztm (make-array (1+ .max-time.))))
    (setf (last-ztime ztm) 0)
    (cons ztm nil)))

(eval-when (load eval compile)
  (defmacro dotimes1 ((var listform) &body body)
    (let ((end (gensym)))
      `(do ((,end ,listform)
	    (,var 1 (1+ ,var)))
	   ((>= ,var ,end))
	 ,@body))))

(defun copy-ztm (ztm)
  (let ((new (new-ztm))
         (newz nil))
    (setf newz (car new))
    (setf (ztm-history new) (ztm-history ztm))
    (setf ztm (car ztm))
    (dotimes1 (i (1+ (last-ztime ztm)))
      (setf (elt newz i) (copy-seq (elt ztm i))))
    (setf (last-ztime newz) (last-ztime ztm))
    new))

(defun find-ztime! (ztm sym)
  ;; returns the index of SYM, or NIL if none exists
  (declare (optimize (speed 3) (safety 1)))
  (setf ztm (car ztm))
  (let ((elt nil))
    (dotimes1 (i (1+ (last-ztime ztm)))
      (setf elt (svref ztm i))
      (if (eq sym (ztime-id elt))
	(return-from find-ztime! elt)))
  (values nil)))

(defun find-ztime (ztm sym)
  ;; returns the index of SYM
  (declare (optimize (speed 3) (safety 1)))
  (or (find-ztime! ztm sym)
      (progn
               (setf ztm (car ztm))
	(let* ((end (last-ztime ztm))
	       (mask (ash 1 end))
	       (ztime (make-ztime :id sym
				  :mask mask
				  :<= mask
				  :>= mask
				  :neq empty-set)))
	  (setf (elt ztm (1+ end)) ztime)
	  (incf (last-ztime ztm))
	  ztime))))
	  
(defmacro looping-through-ztimes ((ztm time time-set) &body body)
  ;; *LAST-TIME* had better be bound to something!!
  (let ((index (gensym))
	(mask (gensym))
	(setval (gensym))
	)
    `(do* ((,setval ,time-set)
	   (,index 1 (1+ ,index))
	   (,mask 1)
	   (,time nil))
	 ((> ,index *last-time*))
       (declare (type integer ,index) (type integer ,setval)
		(type integer ,mask) )
       (setf ,time (elt ,ztm ,index))
       (setf ,mask (ztime-mask ,time))
       (when (= ,mask (logand ,mask ,setval))
	 ,@body))))

(defun zshow (ztm)
  (setf ztm (car ztm))
  (let ((*last-time* (last-ztime ztm)))
    (cond ((< *last-time* 0)
	   nil)
	  (t
	   (let ((time nil))
	     (do ((i 1 (1+ i)))
		 ((> i *last-time*))
	       (setf time (elt ztm i))
	       (format t "~&~a: (>=" (ztime-id time))
	       (looping-through-ztimes (ztm other (ztime-<= time))
                 (princ #\space)
		 (princ (ztime-id other)))
	     (format t ") (<=")
	     (looping-through-ztimes (ztm other (ztime->= time))
	       (princ #\space)
	       (princ (ztime-id other)))
	     (princ #\))
	     (terpri)))))))

(defun make<= (ztm sym1 sym2)
  (setf ztm (car ztm))
  (let ((t1 (find-ztime ztm sym1))
	(t2 (find-ztime ztm sym2)))
    (zset<= ztm t1 t2)))

(defun make< (ztm sym1 sym2)
  (setf ztm (car ztm))
  (let ((t1 (find-ztime ztm sym1))
	(t2 (find-ztime ztm sym2)))
    (zset< ztm t1 t2)))

(defun make= (ztm sym1 sym2)
   (setf ztm (car ztm))
      (let ((t1 (find-ztime ztm sym1))
	(t2 (find-ztime ztm sym2)))
    (zset<= ztm t1 t2)
    (zset<= ztm t2 t1)))

(defun zset< (ztm b a)
  (declare (optimize (speed 3) (safety 1)))
  ;; don't do anything if this is redundant!
  (when (zerop (logand (ztime-<= a) (ztime-neq a) (ztime-mask b)))
    (push `(< ,(ztime-id b) ,(ztime-id a)) (ztm-history ztm))
    (let ((*last-time* 0)
	  (>=a 0)
	  (<=b 0))
      (declare (type integer >=a)
	       (type integer <=b)
	       (type integer *last-time*))
      (zset<= ztm b a nil)
      (setf >=a (ztime->= a))
      (setf <=b (ztime-<= b))
					;
      ;; Every TIME <= B is not equal to all times >= A.
					;
      (setf ztm (car ztm))
      (setf *last-time* (last-ztime ztm))
      (setf (ztime-neq a) (logior (ztime-neq a) (ztime-<= b)))
      (looping-through-ztimes (ztm time (ztime-<= b))
			      (setf (ztime-neq time)
				(logior (ztime-neq time) >=a)))
					;
      ;; Every TIME >= A is not equal to B.
					;
      (setf (ztime-neq b) (logior (ztime-neq b) (ztime->= a)))
      (looping-through-ztimes (ztm time (ztime->= a))
			      (setf (ztime-neq time)
				(logior (ztime-neq time) <=b)))
      (values))))

(defun ztm-canonicalize (ztm symbol)
  ;; return the earliest ID that is equal to SYMBOL.
  (let ((eq-mask (nec-= ztm symbol)))
      (setf ztm (car ztm))
    (do ((i 1 (1+ i))
	 (elt nil)
	 (*last-time* (last-ztime ztm)))
	((> i *last-time*))
      (setf elt (elt ztm i))
      (if (= (ztime-mask elt)
	     (logand eq-mask (ztime-mask elt)))
	  (return-from ztm-canonicalize (ztime-id elt))))
    (values symbol)))

(defun ztimes-consistent? (ztm)
  (setf ztm (car ztm))
      (do ((i 1 (1+ i))
       (elt nil)
       (*last-time* (last-ztime ztm)))
      ((> i *last-time*))
    (setf elt (elt ztm i))
    (if (not (= (logand (ztime-mask elt) (ztime-neq elt)) 0))
	(return-from ztimes-consistent?
	  (progn
;	    (format t  "~& Egads! inconsistent constraints on ~s~%"
;	    (error  "~& Egads! inconsistent constraints on ~s~%")
;		     (ztime-id elt))
	      nil))))
  (values t))

(defun zset<= (ztm b a &optional (record? t))
  ;; array contains ztimes. both a and b are ztime structs.
  ;; bi and ai are their indices.
;  (declare (optimize (speed 3) (safety 1)))
  (when (zerop (logand (ztime-<= a) (ztime-mask b)))
    (if record?
      (push `(<= ,(ztime-id b) ,(ztime-id a)) (ztm-history ztm)))
  (setf ztm (car ztm))
       (let ((b-time (ztime-mask b))
	(a-time (ztime-mask a))
	(<=b (ztime-<= b))
	(<b 0)
	(>=a (ztime->= a))
	(>a 0)
	(*last-time* (last-ztime ztm)))
    (declare (type integer >=b) (type integer <=b) (type integer <=a)
	     (type integer b-time) (type integer a-time) (type integer >a)
	     (type integer <b) (type integer time)
	     (optimize (speed 3) (safety 3)))
    ;
    ;; Every TIME <= b must know that it is less than every time >= a
    ;
    (looping-through-ztimes (ztm time <=b)
      (setf (ztime->= time)
	(logior (ztime->= time) >=a)))
    ;
    ;; Every TIME >= a must know that it is greater than every time <= b
    ;
     (looping-through-ztimes (ztm time >=a)
      (setf (ztime-<= time)
	(logior (ztime-<= time) <=b)))
    ;
    ;; Every TIME > a must know that it is strictly > b.
    ;
    (setf >a (logand >=a (ztime-neq a)))
    (setf (ztime-neq b) (logior (ztime-neq b) >a))
    (looping-through-ztimes (ztm time >a)
      (setf (ztime-neq time)
	(logior (ztime-neq time) <=b)))
    ;
    ;; Every TIME < b must know that it is strictly < a.
    ;
    (setf <b (logand <=b (ztime-neq b)))
    (setf (ztime-neq a) (logior (ztime-neq a) <b))
    (looping-through-ztimes (ztm time <b)
      (setf (ztime-neq time)
	(logior (ztime-neq time) >=a)))
    t)))


;;
;;; Querying the temporal database -- all these fns return encoded ztimes.
;;; 
;;

(defun explode-ztm (orig-ztm)
  ;; takes a ztm and produces a list of lists, each
  ;; of the form (id all-that-precede all-that-succed)
  ;;
  (let ((result nil)
         (dups 0)
         (ztm (car orig-ztm)))
    (do ((i 1 (1+ i))
	 (done-so-far 0)
	 (elt nil)
	 (<=i 0)
	 (maski 0)
	 (>=i 0)
	 (temp 0)
	 (*last-time* (last-ztime ztm)))
	((> i *last-time*) result)
      (declare (type integer temp) (type integer i)
	       (type integer >=i) (type integer <=i)
	       (type integer neqi))
      (setf elt (elt ztm i))
      (setf maski (ztime-mask elt))
      (cond ((= 0 (logand done-so-far maski))   ;; new timepoint
	     (setf <=i (ztime-<= elt))
	     (setf >=i (ztime->= elt))
	     ;;                     Everything not equal to i
	     ;;                     +-----------------------+
	     (setf temp (logand <=i >=i))
	     (setf done-so-far (logior done-so-far temp))
	     (setf temp (lognot temp))
	     (push (list (ztime-id elt)
			 (logand <=i temp)
			 (logand >=i temp))
		   result))
	    (t
	     ;; record this as a duplicate for later removal
	     (setf dups (logior dups maski)))))
    (setf dups (lognot dups))
    (mapcar #'(lambda (entry)
		(list (car entry)
		      (decode-ztime orig-ztm (logand (second entry) dups))
		      (decode-ztime orig-ztm (logand (third entry) dups))))
	    result)))
	  

(defun ztime-within-p (ztm symbol encoded-set)
  (let ((temp (find-ztime! ztm symbol)))
    (cond ((null temp)
	   nil)
	  (t
	   (setf temp (ztime-mask temp))
	   (= temp (logand temp encoded-set))))))

(defun nec-< (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (cond ((null zt)
	   empty-set)
	  (t
	   (logand (ztime-<= zt) (ztime-neq zt))))))

(defun nec-<= (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (if (null zt)
	   empty-set
      (ztime-<= zt))))

(defun nec-> (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (cond ((null zt)
	   empty-set)
	  (t
	   (logand (ztime->= zt) (ztime-neq zt))))))

(defun nec->= (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (if (null zt)
	empty-set
      (ztime->= zt))))
  
(defun nec-= (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (cond ((null zt)
	   empty-set)
	  (t
	   (logand (ztime-<= zt) (ztime->= zt))))))
  
(defun nec-neq (ztm time)
  (let ((zt (find-ztime! ztm time)))
    (if (null zt)
	empty-set)
    (ztime-neq zt)))

(defun pos-<= (ztm time)
  (lognot (nec-> ztm time)))

(defun pos-< (ztm time)
  (lognot (nec->= ztm time)))

(defun pos->= (ztm time)
  (lognot (nec-< ztm time)))

(defun pos-> (ztm time)
  (lognot (nec-<= ztm time)))

(defun pos-= (ztm time)
  (lognot (nec-neq ztm time)))

(defun pos-neq (ztm time)
  (lognot (nec-= ztm time)))

(defun decode-ztime (ztm time-set)
  (setf ztm (car ztm))
  (let ((result nil)
	(*last-time* (last-ztime ztm)))
    (looping-through-ztimes (ztm time time-set)
      (push (ztime-id time) result))
    result))

;;
;;;  Interval stuff
;;

(defun i-start (time-spec)
  (cond ((atom time-spec) time-spec)
	(t
	 (second time-spec))))

(defun i-end (time-spec)
  (cond ((atom time-spec) time-spec)
	(t
	 (third time-spec))))

(defun i-type (time-spec)
  (cond ((atom time-spec)
	 :point)
	(t
	 (car time-spec))))

;;
;;; More comparisons
;;

(defun TIMES-POSSIBLY-OVERLAP-P (ztm time1 time2)
  (let ((t2 (i-start time2))
	(t3 (i-end time2))
	(c2 (i-type time2)))
    (ecase c2
      (:point
       (pos-overlap-point-p ztm time1 t2))
      (:open
       (pos-overlap-open-p ztm time1 t2 t3))
      (:open-start
       (pos-overlap-open-start-p ztm time1 t2 t3))
      (:open-end
       (pos-overlap-open-end-p ztm time1 t2 t3))
      (:closed
       (pos-overlap-closed-p ztm time1 t2 t3)))))

(defun POS-OVERLAP-POINT-P (ztm time1 t2)
  (let ((t0 (i-start time1))
	(t1 (i-end time1))
	(c1 (i-type time1)))
    (ecase c1
      (:point 
       (not
	(or (ztime-within-p ztm t1 (nec-< ztm t2))
	    (ztime-within-p ztm t0 (nec-> ztm t2)))))
      (:open
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t2)))))
      (:open-start
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t2)))))
      (:open-end
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->  ztm t2)))))
      (:closed
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->  ztm t2)))))
      )))

(defun POS-OVERLAP-OPEN-P (ztm time1 t2 t3)
  (let ((t0 (i-start time1))
	(t1 (i-end time1)))
    (not
     (or (ztime-within-p ztm t1 (nec-<= ztm t2))
	 (ztime-within-p ztm t0 (nec->= ztm t3))))))


(defun POS-OVERLAP-OPEN-START-P (ztm time1 t2 t3)
  ;; interval is (t2 t3] )
  ;;
  (let ((t0 (i-start time1))
	(t1 (i-end time1))
	(c1 (i-type time1)))
    (ecase c1
      (:point 
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec-> ztm t3)))))
      (:open
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-start
       (not
	(or (ztime-within-p ztm t1 (nec-<=  ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-end
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->  ztm t3)))))
      (:closed
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->  ztm t3)))))
      )))

(defun POS-OVERLAP-OPEN-END-P (ztm time1 t2 t3)
  ;; (interval is  [t2 t3) 
  ;;
  (let ((t0 (i-start time1))
	(t1 (i-end time1))
	(c1 (i-type time1)))
    (ecase c1
      (:point 
       (not
	(or (ztime-within-p ztm t1 (nec-< ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-start
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-end
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:closed
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->=  ztm t3)))))
      )))

(defun POS-OVERLAP-CLOSED-P (ztm time1 t2 t3)
  ;; interval is [t2 t3] 
  ;;
  (let ((t0 (i-start time1))
	(t1 (i-end time1))
	(c1 (i-type time1)))
    (ecase c1
      (:point 
       (not
	(or (ztime-within-p ztm t1 (nec-< ztm t2))
	    (ztime-within-p ztm t0 (nec-> ztm t3)))))
      (:open
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-start
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->= ztm t3)))))
      (:open-end
       (not
	(or (ztime-within-p ztm t1 (nec-<= ztm t2))
	    (ztime-within-p ztm t0 (nec-> ztm t3)))))
      (:closed
       (not
	(or (ztime-within-p ztm t1 (nec-<  ztm t2))
	    (ztime-within-p ztm t0 (nec->  ztm t3)))))
      )))


(defun ORDER-TIMES (ztm time1 time2)
  ;; Both TIME1 and TIME2 are time specs.  Make sure that
  ;; the TIME1 occurs before TIME2 by
  ;; placing appropriate constraints on their endpoints.
  (let ((t2 (find-ztime ztm (i-start time2)))
	(c2 (i-type time2)))
    (ecase c2
      (:point
       (order-before-point ztm time1 t2))
      (:open
       (zset<= ztm (find-ztime ztm (i-end time1)) t2))
      (:open-start
       (zset<= ztm (find-ztime ztm (i-end time1)) t2))
      (:open-end
       (order-before-point ztm time1 t2))
      (:closed
       (order-before-point ztm time1 t2)))))

(defun ORDER-BEFORE-POINT (ztm time1 t2)
  (let ((t1 (find-ztime ztm (i-end time1)))
	(c1 (i-type time1)))
    (ecase c1
      (:point      (zset< ztm t1 t2))
      (:open       (zset<= ztm t1 t2))
      (:open-start (zset< ztm t1 t2))
      (:open-end   (zset<= ztm t1 t2))
      (:closed     (zset< ztm t1 t2)))))

(defun ORDER-OK? (ztm time1 time2)
  ;; Test whether it is possible for TIME1 to precede TIME2.
  ;;
  (let ((t2 (i-start time2))
	(c2 (i-type time2)))
    (ecase c2
      (:point
       (ok-before-point? ztm time1 t2))
      (:open
       (not (ztime-within-p ztm (i-end time1) (nec-> ztm t2))))
      (:open-start
       (not (ztime-within-p ztm (i-end time1) (nec-> ztm t2))))
      (:open-end
       (ok-before-point? ztm time1 t2))
      (:closed
       (ok-before-point? ztm time1 t2)))))

(defun OK-BEFORE-POINT? (ztm time1 t2)
  (let ((t1 (i-end time1))
	(c1 (i-type time1)))
    (ecase c1
      (:point
       (not (ztime-within-p ztm t1 (nec->= ztm t2))))
      (:open
       (not (ztime-within-p ztm t1 (nec-> ztm t2))))
      (:open-start
       (not (ztime-within-p ztm t1 (nec->= ztm t2))))
      (:open-end  
       (not (ztime-within-p ztm t1 (nec-> ztm t2))))
      (:closed
       (not (ztime-within-p ztm t1 (nec->= ztm t2)))))))
