(in-package "ZENO")
(use-package "VARIABLE")
(export '(*verbose* *version*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Data Structures

(defconstant *version* "2.0")

(defvar *verbose* nil)			; Print whole plan?

(defmacro defpropfn (symbol indicator args &body body)
  (let ((name (intern (format nil "PROPERTY-~a-~a"
			      symbol indicator))))
  `(progn
    (defun ,name ,args ,@body)
    (eval-when (load eval)
      (setf (get ',symbol ',indicator) (function ,name)))
    (values ',name))))


(defstruct (PLAN (:constructor make-plan*)
	    (:print-function print-plan))
  steps					; list of steps
  links					; list of causal links
  flaws					; list of UNSAFEs and OPENCs
  ordering				; list of TIMEs
  bindings				; binding constraints
  high-step				; integer # of highest step in plan
  names					; a list of funargs
  constraints				; metric constraints
  gantt					; a gantt chart representation
  (other nil)				; an alist of scr & debug stuff
  )

(defun GET-STEP-WITH-ID (plan id)
  (dolist (s (plan-steps plan))
    (if (eql id (p-step-id s))
	(return s))))

(defun START-TIME-OF-STEP (plan step-id)
    (dolist (step (plan-steps plan))
      (when (eq (p-step-id step) step-id)
	(return-from start-time-of-step (p-step-start step)))))

(defun END-TIME-OF-STEP (plan step-id)
    (dolist (step (plan-steps plan))
      (when (eq (p-step-id step) step-id)
	(return-from end-time-of-step (p-step-end step)))))

;;;  The terse print function
(defun PRINT-PLAN (plan &optional (stream t) depth)
  (declare (ignore depth))
  (if *verbose* (display-plan plan stream)
    (format stream "#plan<S=~a; F=~a>" 
	    (- (length (plan-steps plan)) 1)
	    (length (plan-flaws plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct FLAW (rank nil))  ;; so we can use Tony Barrett's search controller

(defstruct (OPENC
	    (:include flaw)
            (:print-function print-open))
  condition         ;; open precondition {condx}
  time              ;; an interval of time over which condition should be true
  step		    ;; id of step requiring it (for printing reasons)
  marked?	    ;; T if marked -- no decomposition allowed if True.
  (level 0)         ;; split level -- indicates depth of splitting for
                    ;; search control purposes.
  )

 
(defun print-interval (self &optional (stream *standard-output*))
  (let ((type (i-type self))
        (t1 (i-start self))
	(t2 (i-end self))
	)
  (case type
    (:closed (format stream "[~s,~s]" t1 t2))
    (:open (format stream "(~s,~s)" t1 t2))
    (:open-end (format stream "[~s,~s)" t1 t2))
    (:open-start (format stream "(~s,~s]" t1 t2))
    (:point (format stream "~s" t1))
    (otherwise
     (format stream "?")))))

(defun PRINT-OPEN (self stream depth)
  (declare (ignore depth))
  (format stream "<~a," (openc-condition self))
  (print-interval (openc-time self) stream)
  (format stream " by ~a>" (openc-step self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (LINK (:type vector))
  time
  condition
  Si					;should be a list, but NOT.
  Sj
  effect                                ; for threat purposes
  )


(defun print-link (link &optional (stream *standard-output*))
  ;; print links like they appear in the thesis.
  (format stream "S~d->S~d: " (link-si link) (link-sj link)) 
  (write-char #\< stream)
  (print-interval (effect-time (link-effect link)) stream)
  (write-char #\, stream)
  (format stream "~(~s~)," (link-condition link))
  (print-interval (link-time link) stream)
  (write-char #\> stream)
  (terpri stream))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (UNSAFE
	    (:include flaw)
	    (:print-function print-unsafe))
  link                 ;; id of threatened link
  clobber-effect       ;; effect which threatens it
  clobber-condition    ;; added condition which causes the threat
  )

(defun print-unsafe (self stream depth)
  (declare (ignore depth))
  (format stream "#<UNSAFE ~a step~a ~a>"
          (unsafe-link self)
	  (unsafe-clobber-effect self)
	  (unsafe-clobber-condition self)))

(defun display-unsafe (self &optional (stream *debug-io*) (indent 0))
  (indent-stream stream indent)
  (format stream "Unsafe: ~a threatens ~a under ~a"
	  (unsafe-clobber-effect self)
	  (unsafe-link self)
	  (unsafe-clobber-condition self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (P-STEP (:print-function print-p-step))
  ID					; integer step number
  action				; formula such as (puton ?X ?Y)
  precond				; conditions like (clear ?X)
  add					; effects asserted by step
  (cache nil)				; A cache of existing steps
  start					; starting time point
  end					; ending time point
  ca					; step constraints
  parms   ;; get rid of this!!
  res					; list of resource clauses (resource ?x)
  )

(defun PRINT-P-STEP (s &optional (stream t) depth)
  (declare (ignore depth))
  (format stream "<~a>" (car (p-step-action s))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct effect
  (id nil)				; id of step from whence this came
  (influence-p nil)			;d/dt if an influence
  (time nil)				;effect time, an interval
  (post nil)				;postcondition
  (pre  nil)				;precondition
  (forall nil)				;universal vars
  (exists nil)				;exists for the universals
  (ca nil)				;delayed Ca constraints
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;(defstruct SCR
;  name					; The name of the scr
;  when					; The preconditions
;  effect)				; What the scr does


;; Macros

(defmacro push-non-nil (item list)
  (let ((val (gensym)))
    `(let ((,val ,item))
       (if ,val (push ,val ,list) nil))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether one condition affects another.  This happens when one can
;;;  unify with the other or its negation.

(defun affects (effect theta1 theta2 bindings)
  ;; in the future, peel bindings automatically
  (declare (ignore effect))
  (cond ((eq (car theta1) (car theta2))
	 (mgu theta1 theta2 bindings))
	(t
	 (mgu
	  (if (eq :not (car theta1)) (second theta1) theta1)
	  (if (eq :not (car theta2)) (second theta2) theta2)
	  bindings))))

(defvar *forever* nil "All atemporal values stored here")

;;
;;; The table from my thesis.
;;

(defconstant *logical-link-table*
    ;;
    ;; Goal time is (:code t2 t3)
    ;; Effect time is (:code t0 t1)
    ;;
    ;; Entries in this table are (goal-code goal-table)
    ;; Entries in the goal-table are (effect-code &rest specs)
    ;;
    ;; This table tells us how to create the supporting
    ;; links from an effect to a goal, and which temporal
    ;; constraints to add.  We've only done the case for
    ;; one action satisfying a goal... yow.
    ;;
   '(:point
      (:point
       ((:link (:closed t1 t2) :time ((<= t1 t2))))
       :closed
       ((:link (:closed t0 t2) :time ((< t1 t2)))
        (:link (:closed t0 t1) :time ((<= t0 t2) (<= t2 t1))))
       :open-start
       ((:link (:open-start t0 t2) :time ((<= t1 t2)))
	(:link (:open-start t0 t1) :time ((< t0 t2) (< t2 t1))))
       :open-end
       ((:link (:closed t0 t2) :time ((<= t1 t2)))
	(:link (:open-end t0 t1) :time ((<= t0 t2) (< t2 t1))))
       :open
       ((:link (:open-start t0 t2) :time ((<= t1 t2)))
	(:link (:open t0 t1) :time ((< t0 t2) (< t2 t1))))
       )
     ;;
     :closed
      (:point
       ((:link (:closed t1 t3) :time ((<= t1 t2))))
       :closed
       ((:link (:closed t0 t3) :time ((<= t0 t2) (< t1 t3)))
        (:link (:closed t0 t1) :time ((<= t0 t2) (<= t3 t1))))
       :open-start
       ((:link (:open-start t0 t3) :time ((< t0 t2) (< t1 t3)))
	(:link (:open-start t0 t1) :time ((< t0 t2) (<= t3 t1))))
       :open-end
       ((:link (:closed t0 t3) :time ((<= t0 t2) (<= t1 t3)))
	(:link (:open-end t0 t1) :time ((<= t0 t2) (< t3 t1))))
       :open
       ((:link (:open-start t0 t3) :time ((< t0 t2) (<= t1 t3)))
	(:link (:open t0 t1) :time ((< t0 t2) (< t3 t1))))
       )
     ;;
     :open-end
      (:point
       ((:link (:open-end t1 t3) :time ((<= t1 t2))))
       :closed
       ((:link (:open-end t0 t3) :time ((<= t0 t2) (< t1 t3)))
        (:link (:closed t0 t1) :time ((<= t0 t2) (<= t3 t1))))
       :open-start
       ((:link (:open t0 t3) :time ((< t0 t2) (< t1 t3)))
	(:link (:open-start t0 t1) :time ((< t0 t2) (<= t3 t1))))
       :open-end
       ((:link (:open t0 t3) :time ((<= t0 t2) (<= t1 t3)))
	(:link (:open-end t0 t1) :time ((<= t0 t2) (< t3 t1))))
       :open
       ((:link (:open t0 t3) :time ((< t0 t2) (<= t1 t3)))
	(:link (:open t0 t1) :time ((< t0 t2) (< t3 t1))))
       )))

;;
;;; User settable parameters
;;


(defvar *IGNORE-RESOURCES* nil "Set to T to ignore resource constraints.")

(defvar *FIFO-FLAWS* nil "When T, choose flaws first-in-first-out")

(defvar *SHOW-DEAD-ENDS* nil "Set to T to see dead ends in planning.")

(defvar *GREEDY* nil "Set to T to greedily bind all atemporal goals.")

(defvar *DEBUG* nil "Set to T to see stuff")

(defvar *SHOW-COMPLETE-PLANS* nil
  "Set to T when you want all complete plans displayed during search.")

(defvar *RANKER* 
   #(1 1 4 0 0.2)
   "Weights used to evaluate a plan in a 5-dimensional parameter space.
    
    Index 1 = Number of threats in plan.
          2 = Number of temporal goals in plan
          3 = Number of steps in plan
          4 = Number of links in plan
          5 = 10*number of nonlinear equations + 
               2*number of inequalities +
               number of equations
 
    Rank = weighted sum.")

(defvar *RESOURCE-CHECK*
    :late 
  "Either :EARLY or :LATE, depending upon when you want resources checked.")

(defvar *DEADLINE-GOALS* t
  "Set this to T if you want all temporal constraints posted to the Simplex
Tableau, NIL otherwise.")
