;;
;;; Inequality reasoner -- use an incremental Simplex algorithm
;;

(in-package "ZENO")
;(proclaim '(optimize (speed 3) (safety 2) (space 0) (compilation-speed 0)))

(defclass INEQ-SOLVER ()
  ((tableau :initform NIL :accessor ineq-tableau)
   (consistent? :initform T :accessor ineq-consistent?)
   (csolver :accessor eqn-csolver :initarg :csolver)
   (slack-history :initform nil :accessor slack-history)
   ))

(defmethod EQ-TABLEAU ((self ineq-solver))
  (with-slots (csolver) self
    (cs-eqn csolver)))

(defmethod SPAWN ((self ineq-solver) old-ineq-solver)
  ;; Makes this instance a duplicate of OLD-INEQ-SOLVER, minus
  ;; the link to the CSOLVER.
  (with-slots (tableau consistent? slack-history) self
    (cond ((ineq-consistent? old-ineq-solver)
	   (setf tableau (mapcar #'copy-row (ineq-tableau old-ineq-solver)))
	   (setf slack-history (slack-history old-ineq-solver))
	   (setf consistent? t))
	  (t
	    (setf consistent? nil)))
    (values)))

(defmethod RESET ((self ineq-solver))
  (setf (ineq-tableau self) nil)
  (setf (ineq-consistent? self) t)
  (setf (slack-history self) nil)
  (values))

(defmethod BASIC-VAR-P ((self ineq-solver) thing)
  ;; If THING is a basic variable, return a ROW in the tableau
  ;; where thing occurs, NIL otherwise.
  (dolist (row (ineq-tableau self))
    (if (eq (cdar (row-vars row)) thing)
	(return row))))

(defmethod NONBASIC-VAR-P ((self ineq-solver) thing)
  ;; If THING is a nonbasic variable, return a ROW in the tableau
  ;; where thing occurs, NIL otherwise.
  (dolist (row (ineq-tableau self))
    (if (vassoc thing (cdr (row-vars row)))
	(return row))))

(defmethod HAVE-SEEN-VAR-P ((self ineq-solver) thing)
  ;; If THING is a basic or nonbasic variable, return a ROW in the tableau
  ;; where thing occurs, NIL otherwise.
  (dolist (row (ineq-tableau self))
    (if (vassoc thing (row-vars row)) (return T))))

(defun SLACK-VAR-P (thing)
  (and (numberp thing) (< thing 0)))
  
(defun varname (thing)
  (symbol-name
   (cond ((symbolp thing) thing)
	 (t
	  (variable::var-name thing)))))

(defun MORE-JUNIOR-THAN (symbol1 symbol2)
  (let ((sym1 (not (numberp symbol1)))
	(sym2 (not (numberp symbol2))))
    (cond (sym1
	   (if sym2
	       (string> (varname symbol1)
			(varname symbol2))
	       t))
	  (sym2 nil)
	  (t
	   (> (abs symbol1) (abs symbol2))))))
	
(defmethod EQUATION->ROW ((self ineq-solver) form)
  ;; FORM is :inequality <type> <constant> ((coeff . var)*)
  ;; Returns two values:  (1) a row in standard form, (2) T if this was an
  ;; inequality
;  (print form)
  (let ((ineq (second form))
	(const (third form))
	(vars (canonicalize-vars (cdddr form)))
	(slack nil)
	(row nil))
    (setf row
      (ecase ineq
	(<
	 (setf slack (cons 1.0d0 (genslack)))
	 (make-row :constant const
		   :vars (nconc vars (cons slack nil))
		   :eps -1.0d0))
	(> 
	 (setf slack (cons -1.0d0 (genslack)))
	 (make-row :constant const
		   :vars (nconc vars (cons slack nil))
		   :eps 1.0d0))
	(<=
	 (setf slack (cons 1.0d0 (genslack)))
	 (make-row :constant const
		   :vars (nconc vars (cons slack nil))
		   :eps 0))
	(>=
	 (setf slack (cons -1.0d0 (genslack)))
	 (make-row :constant const
		   :vars (nconc vars (cons slack nil))
		   :eps 0))
	(=
	 (make-row :constant const :vars vars :eps 0))))
    (if (approx< const 0) (scale-row row -1))
    (values row slack)))
      
(defmethod ADD-RAW-EQUATION ((self ineq-solver) raw-form)
  ;; FORM is :equation <number> &rest ((coeff . var)*)
;  (show-constraints self)
  (when (ineq-consistent? self)
    (multiple-value-bind (row slack)
	(equation->row self raw-form)
      (if slack
	  (handle-inequality self row slack raw-form)
	(handle-equality self row)))))
	  
(defmethod PICK-NEW-BASIC-VAR ((self ineq-solver) row)
  ;; Pick a NEW legal basic variable from ROW and returns its symbol.
;  (terpri)
;  (condense-row row)
  (setf (row-vars row) (order-variables (row-vars row)))
  (if (slack-var-p (cdar (row-vars row)))
      ;; have to find a non-negative coefficient for the slack vars.
      (do* ((vars (row-vars row) (cdr vars))
	    (v (car vars) (car vars))
	    (chosen nil))
	  ((or chosen (null vars))
	   (if (not chosen)
	       nil
	     (progn (setf (row-vars row)
		      (cons chosen (delete-1 chosen (row-vars row))))
		    (cdr chosen))))
	(when (> (car v) *tolerance*) (setf chosen v)))
    (let ((old (mapcar #'cdr (mapcar #'row-vars (ineq-tableau self)))))
      (dolist (v (row-vars row))
	(unless (eql-member (cdr v) old)
	  (return-from pick-new-basic-var (cdr v)))))))

;    (cdar (row-vars row))))

(defmethod HANDLE-INEQUALITY ((self ineq-solver) row slack raw-form)
  ;; Raw-form will be (:inequality {} const vars)
  (let ((eq-version (cons :equation (cddr raw-form)))
	(bv nil))
    ;; keep around an (= ..) version of the constraint, should it
    ;; become an equality.
    (push (cons (cdr slack) eq-version) (slack-history self))
    (substitute-out-subjects (eq-tableau self) row)
    (substitute-out-basic-vars self row)
;    (condense-row row)
    (if (const<0? row) (scale-row row -1))
    (push row (ineq-tableau self))
    (cond ((eliminable-row? row)
	   (solve self (ineq-tableau self)))
	  ((setf bv (pick-new-basic-var self row))
	   (basicfy-var-in-row row bv)
;	   (values :ok)
	   (solve self (ineq-tableau self))
	   )
	  (t
	   (solve self (ineq-tableau self))))))
	     
(defmethod HANDLE-EQUALITY 
   ((self ineq-solver) row &optional (from-gauss? nil))
  ;; We assume that ROW is an equality.  Test to see if ROW
  ;; contains any basic variables.  If not, apply case b of
  ;; CLP(R).
  (let ((basic-row nil))
    ;; The next line will reduce any constraints from the Gaussian
    ;; table to 0=0, a real no-no.
    (unless from-gauss?
      (substitute-out-subjects (eq-tableau self) row))
;    (condense-row row)
    (if (const<0? row) (scale-row row -1))
    (dolist (var (row-vars row))
      (setf basic-row (basic-var-p self (cdr var)))
      (when basic-row
	(setf (row-vars row) (move-to-front var (row-vars row)))
	(return)))
    (cond (basic-row
	   (simplex-case-c self row basic-row))
	  (t
	   (simplex-case-b self row)))))

(defmethod SIMPLEX-CASE-B ((self ineq-solver) row)
  (let ((nonbasic-row nil)
	(vars (row-vars row))
	(pivot (cdr (row-subject row))))
    (when vars
      (dolist (var vars)
	(setf nonbasic-row (nonbasic-var-p self (cdr var)))
	(when nonbasic-row
	  (setf (row-vars row) (move-to-front var (row-vars row)))
	  (return)))
      (cond (nonbasic-row
	     (scale-row row (/ 1.0d0 (caar (row-vars row))))
	     (dolist (other-row (ineq-tableau self))
	       (unless (not (vassoc pivot (row-vars other-row)))
		 (when (row-substitute-out row other-row)
;		   (condense-row other-row)
		   (if (const<0? other-row)
		       (scale-row other-row -1))))))
	    (t
	     (substitute-out-basic-vars self row)
;	     (condense-row row)
	     (scale-row row (/ 1.0d0 (caar (row-vars row))))
	     (if (const<0? row) (scale-row row -1))
	     (push row (ineq-tableau self)))))
    :ok))

(defmethod SIMPLEX-CASE-C ((self ineq-solver) row basic-row)
  ;; The new row defines a new constraint on an existing
  ;; basic variable, which is seen in BASIC-ROW of the tableau.
;  (show-constraints self)
  (let ((temp nil)
	(vars (row-vars row)))
    (dolist (other (ineq-tableau self))
      (unless (or (eq other basic-row)
		  (not (vassoc (cdr (row-subject other)) vars)))
	(row-substitute-out other row)))
;    (condense-row row)
    (scale-row row (/ 1.0d0 (caar (row-vars row))))
    (when (row-substitute-out row basic-row)
;      (condense-row basic-row)
      (if (const<0? basic-row) (scale-row basic-row -1)))
    (setf (ineq-tableau self)
      (cons basic-row (delete-1 basic-row (ineq-tableau self))))
;  (show-constraints self)
  (cond ((and (not (slack-var-p (cdar (row-vars basic-row))))
	      (setf temp (pick-new-basic-var self basic-row)))
	 (basicfy-var-in-row basic-row temp)
	 :ok)
	((const=0? basic-row)
	 (cond ((eq :inconsistent (solve self (ineq-tableau self)))
		:inconsistent)
	       (t
;		(print "trying negative version")
		(scale-row basic-row -1.0d0)
		(solve self (ineq-tableau self)))))
	(t
	 (solve self (ineq-tableau self))))))

(defmethod SUBSTITUTE-OUT-BASIC-VARS ((self ineq-solver) row)
  (let ((vars (row-vars row)))
    (dolist (entry (ineq-tableau self))
      (unless (not (vassoc (cdr (row-subject entry)) vars))
	(row-substitute-out entry row)))
;  (condense-row row)
    (values row)))

(defun all-minusp? (vars)
  (dolist (v vars T)
    (if (> (car v) *tolerance*) (return nil))))

(defun eliminable-row? (r)
  (let ((vars (row-vars r)))
    (and (const=0? r)
	 (dolist (v vars t)
	   (if (not (slack-var-p (cdr v)))
	       (return nil)))
	 (or (all-minusp? vars)
	     (all-minusp? (row-vars (scale-row r -1)))))))

(defmethod SOLVE ((self ineq-solver) tableau)
  (if tableau
      (let ((result (primary-simplex self tableau)))
	(cond ((eq result :inconsistent)
	       (setf (ineq-consistent? self) nil)
	       :inconsistent)
	      ((or (eq result :implicit)
		   (eliminable-row? (car (ineq-tableau self))))
	       (let ((ie (implicit-equalities self)))
		 (dolist (slack ie)
		   (let ((entry (assoc slack (slack-history self))))
		     (when entry
		       (count-stat .simplex->gauss.)
		       (setf (slack-history self)
			 (remove-1 entry (slack-history self)))
		       (add-raw-equation (eq-tableau self) (cdr entry)))))
		 (cond ((and (eqn-consistent? (eq-tableau self))
			     (ineq-consistent? self))
			(values :ok))
		       (t
			(setf (ineq-consistent? self) nil)
			(values :inconsistent)))))
	      (t
	       (values result))))))

(defmethod PRIMARY-SIMPLEX ((self ineq-solver) tableau)
  (do* ((toprow (car tableau))
          (basic (pick-new-basic-var self toprow)
               (pick-new-basic-var self toprow))
          (entering-row (find-entering-row self basic toprow)
               (find-entering-row self basic toprow)))
         (nil)
;    (when basic
;      (format t "~&Variable ~s enters, ~s leaves in: ~%"
;	      basic (cdar (row-vars entering-row)))
;      (dolist (r tableau)
;	(print-row-internal r t '=) (terpri))
;      (terpri))
    (count-stat .spivot.)
    (cond ((null basic)
	   (return
	     (if (const=0? toprow)
		 :implicit
	         :inconsistent)))
	  (t
	   (basicfy-var-in-row entering-row basic)
	   (let ((pivot (cdr (row-subject entering-row))))
	     (dolist (other tableau)
	       (unless (or (eq other entering-row)
			   (not (vassoc pivot (row-vars other))))
		 (simplex-ero entering-row other))))
;	   (if (eql basic -11) (break))
	   (if (eq toprow entering-row)
	       (return :ok))))))

(defmethod RELEVANT-ROWS ((self ineq-solver) slack-var)
  (let ((rows nil))
    (dolist (row (ineq-tableau self))
      (when (and (const=0? row)
		 (slack-var-p (cdr (car (row-vars row))))
		 (vassoc slack-var (row-vars row)))
	(push row rows)))
    (values rows)))

(defun FIND-ROW-WITH-NEGATIVE-COEFF (rows symbol)
  (if symbol
      (let ((coeff nil))
	(dolist (r rows)
	  (setf coeff (car (vassoc symbol (row-vars r))))
	  (if (and coeff (approx< coeff 0.0d0))
	      (return r))))))

(defun SIMPLEX-ERO (from-row to-row)
  (count-stat .simplex-ero.)
  (when (row-substitute-out from-row to-row)
;    (condense-row to-row)
    (when (const<0? to-row)
      (scale-row to-row -1))))
  

(defmethod IMPLICIT-EQUALITIES ((self ineq-solver))
  ;; The toprow of the tableau now contains c*s1 + c*s2..=0
  ;; We will now rid ourselves of these null slack variables.
  (let ((ie nil)			;implicit equalities
	(s (mapcar #'cdr (row-vars (car (ineq-tableau self))))) ;set of slack vars
	null-var relevant r pivot)
;    (print (list :slacks-are s))
    (pop (ineq-tableau self))
    (do ()
	((null s))
      (setf null-var (pop s))
      (push null-var ie)
      (count-stat .implicits.)
      (setf relevant (relevant-rows self null-var))
      (when (setf r (find-row-with-negative-coeff relevant null-var))
	(setf relevant (move-to-front r (remove-1 r relevant)))
	(basicfy-var-in-row r null-var)
	(setf pivot (cdr (row-subject r)))
	(dolist (other (cdr relevant))
	  (unless (not (vassoc pivot (row-vars other)))
	    (simplex-ero r other)))
	(setf (row-vars r)
	  (delete-1 (vassoc null-var (row-vars r)) (row-vars r)))
	(when (eq :implicit (primary-simplex self relevant))
	  (dolist (var (row-vars r))
	    (push (cdr var) s))
	  (setf (ineq-tableau self) (delete-1 r (ineq-tableau self))))))
    (dolist (row (ineq-tableau self))
      (setf pivot (cdr (row-subject row)))
      (when (const=0? row)
	(dolist (other (ineq-tableau self))
	  (unless (or (const=0? other)
		      (not (vassoc pivot (row-vars other))))
	    (simplex-ero row other)))))
      (values ie)))

(defun BASICFY-VAR-IN-ROW (row name)
  (let ((var (vassoc name (row-vars row))))
    (setf (row-vars row)
      (cons var
	    (delete-1 var (row-vars row))))
    (scale-row row (/ 1.0d0 (caar (row-vars row))))
    (if (const<0? row) (scale-row row -1))))

(defmethod FIND-ENTERING-ROW ((self ineq-solver) basic-var-name toprow)
  (cond ((null basic-var-name) nil)
	((not (slack-var-p basic-var-name))
	 toprow)
	(t
	 (find-entering-row-1 self basic-var-name toprow))))

(defmethod FIND-ENTERING-ROW-1 ((self ineq-solver) basic-var-name toprow)
  (let ((candidates nil)
	(var nil))
    (dolist (row (ineq-tableau self))
;      (when (slack-var-p (cdar (row-vars row)))
	(when (setf var (vassoc basic-var-name (row-vars row)))
	  (when (> (car var) *tolerance*)
	    (push (list (/ (double-float (row-constant row)) (car var))
			(/ (double-float (row-eps row)) (car var))
			row)
		  candidates)))); )
    (cond ((null (cdr candidates))
	   (third (car candidates)))
	  (t
	   (multiple-value-bind (change-in-slack candidates) 
	       (minimize-unit-value candidates nil nil nil)
	     (cond ((< (abs change-in-slack) *simplex-tolerance*)
		    ;; we've minimized slack as much as we can.
		    toprow)
		   ((null (cdr candidates))
		    (car candidates))
		   (t
		    (pick-most-junior-basic-var candidates nil))))))))

(defun MINIMIZE-UNIT-VALUE (candidates min-k min-e new-cands)
  (let ((k (caar candidates))
	(e (cadar candidates))
	(row (third (car candidates))))
    (cond ((null candidates)
	   (values min-k new-cands))
	  ((or (null min-k) (approx< k min-k))
	   (minimize-unit-value (cdr candidates) k e (list row)))
	  ((== k min-k)
	   (cond ((approx< e min-e)
		  (minimize-unit-value (cdr candidates) k e (list row)))
		 (t
		  (minimize-unit-value (cdr candidates) k e (cons row new-cands)))))
	  (t
	   (minimize-unit-value (cdr candidates) min-k min-e new-cands)))))

(defun PICK-MOST-JUNIOR-BASIC-VAR (rows junior)
  (cond ((null junior)
	 (pick-most-junior-basic-var (cdr rows) (car rows)))
	((null rows)
	 junior)
	(t
	 (if (more-junior-than (cdar (row-vars (car rows)))
			       (cdar (row-vars junior)))
	     (pick-most-junior-basic-var (cdr rows) (car rows))
	     (pick-most-junior-basic-var (cdr rows) junior)))))


#||
(setf hard
 '((>=  90 (+ (* -112 x6) (* -147 x5) (* -210 x4) (* -120 x3) (*   2  x2) (* -47  x1)))
   (>= 120 (+ (* 135  x6) (* -127 x5) (* 80   x4) (* 83   x3) (* 206  x2) (* -64  x1)))
   (>=  78 (+ (* -126 x6) (* 187  x5) (* 209  x4) (* -201 x3) (* -190 x2) (* 9    x1)))
   (>= 132 (+ (* 170  x6) (* -135 x5) (* -68  x4) (* 113  x3) (* -130 x2) (* 122  x1)))
   (>= 163 (+ (* 35   x6) (* 101  x5) (* -194 x4) (* -102 x3) (* 209  x2) (* -185 x1)))
   (>= 70  (+ (* -67  x6) (* 194  x5) (* 150  x4) (* -49  x3) (* -188 x2) (* 118  x1)))
   (>= 148 (+ (* -15  x6) (* -6   x5) (* 160  x4) (* 112  x3) (* -28  x2) (* 60   x1)))
   (>= 17  (+ (* -26  x6) (* -143 x5) (* 143  x4) (* -172 x3) (* 48   x2) (* -83  x1)))
   (>= 97  (+ (* -146 x6) (* -204 x5) (* -160 x4) (* 68   x3) (* 180  x2) (* 101  x1))))
  )

(setf red
  '((>= 0 (+ (* -1 x) (* 1 y) (* -1 z)))
    (>= 0 (+ (* 1 x) (* -1 y) (* 2 z)))
    (>= 0 (+ (* -1 x) (* 1 y) (* -3 z)))
    (>= 18 (+ (* 2 x) (* 1 y) (* -4 z)))
    (>= 6 (+ (* -1 x) (* 2 y) (* 1 z)))
    (>= -4 (+ (* -1 x) (* -3 y)))
    (>= 1 (+ (* 1 x) (* -1 y)))
    (>= -2 (* -1 x))
    ))
    
(defun setup ()
  (setf e (make-instance 'eqn-solver))
  (setf i (make-instance 'ineq-solver))
  (setf n (make-instance 'nlin-solver))
  (setf (eq-tableau i) e)
  (setf (eq-solver n) e)
  (setf (ineq-solver n) i)
  (setf (eqn-simplex e) i)
  (setf (eqn-nonlin e) n)
  (reset i))


	     
(defun tloop ()
  (terpri)
  (princ "> ")
  (let ((arg (read)))
    (unless (null arg)
      (nc arg)
      (tloop))))
||#

(defmethod RESTORE-TABLEAU ((self ineq-solver))
  ;; Restore the tableau to normal format!
  (dolist (row (ineq-tableau self))
    (if (const<0? row)
	(scale-row row -1))
    (setf (row-vars row) (order-variables (row-vars row))))
  (dolist (row (ineq-tableau self))
;    (condense-row row)
    (setf (row-vars row) (order-variables (row-vars row)))
    (when (row-vars row)
      (scale-row row (/ 1.0d0 (caar (row-vars row))))
      (dolist (row2 (ineq-tableau self))
	(unless (or (eq row row2)
		    (not (row-substitute-out row row2)))
	  (if (const<0? row2)
	      (scale-row row -1))))))
;  (dolist (row (ineq-tableau self))
;    (condense-row row))
  (setf (ineq-tableau self)
    (remove-if #'null (ineq-tableau self) :key #'row-vars)))

(defmethod SHOW-CONSTRAINTS ((self ineq-solver))
  (dolist (row (ineq-tableau self))
    (when (row-vars row)
	(print-row-internal row *standard-output* '=)
	(terpri))))
