;;
;;; Temporally quantified domains
;;

(in-package "ZENO")

(defparameter *tests* nil)

(defstruct problem name start end domain inits goal (rank-fun 'rank3) )

(defmacro defproblem (name (start end) &body body)
  (unless (evenp (length body))
    (error "Bad problem spec for ~s." name))
  `(progn
     (setf *tests* (remove ',name *tests* :key 'problem-name))
     (push (make-problem :name ',name 
             :start ',start 
             :end ',end ,@body) *tests*)
     ',name))

(defun bf-show (prob &optional (display "okanogan:0.0"))
  (when (symbolp prob)
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (record-vcr)
  (multiple-value-bind (plan stat)
      (plan (problem-inits prob) (problem-goal prob)
            :rank-fun (problem-rank-fun prob))            
    (when plan (display-plan plan))
    (display-stat stat))
  (load-vcr "bf-show")
  (play display))

(defun bf-control (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (let ((plan (plan (problem-start prob) (problem-end prob)
                    (problem-inits prob) (problem-goal prob) 	
		    :rank-fun (problem-rank-fun prob))))
    (when plan (display-plan plan))
    (display-stats)
    (values plan)))

(defun testit (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (let ((plan (plan (problem-start prob) (problem-end prob)
                    (problem-inits prob) (problem-goal prob) 	
		    :rank-fun 'simple-rank)))
    (when plan (display-plan plan))
    (display-stats)
    (values plan)))  

(defun ie-search (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (let ((plan (plan (problem-start prob) (problem-end prob)
                    (problem-inits prob) (problem-goal prob)
		    :search-fun #'call-ie :rank-fun (problem-rank-fun prob))))
    (when plan (display-plan plan))
    (display-stats)
    (values plan)))

(defun set-clause-time (thing time)
 (cond ((eq (car thing) :not)
        `(:not ,(set-clause-time (second thing) time)))
        (t
         `(,(car thing) ,time ,@(cdr thing)))))

(defun initify (things)
  (cons :and
	(mapcar #'(lambda (clause) (set-clause-time clause 0)) things)))

(defun goalify (things)
  (cons :and
	(mapcar #'(lambda (clause) (set-clause-time clause :goal)) things)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This file contains a set of test problems.  Some of them work,
;;;  and others are intractable for ucpop.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun blocks-world-domain ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  ;; Define step for placing one block on another.
  (define (operator puton)
      :resources ((ucpop))
      :at-time (?s ?e)
      :parameters (?x ?y ?z)
      :precondition (:and (:forall (time ?t "[?s,?e)")
			   (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and
       (on ?e ?X ?Y)
       (:not (on ?e ?X ?Z))
       (:when (:neq ?z table) (clear ?e ?z))
       (:when (:neq ?y table) (:not (clear ?e ?y))))))

(defun sipe-blocks-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)
  (add-domain-function 'size)

  ;; Define step for placing one block on another.
  (define (operator fuel-puton)
      :at-time (?s ?e)
      :resources ((robot ?r))
      :parameters (?r ?X ?Y ?Z)
      :precondition (:and (:fact (robot ?r))
			  (:forall (time ?t "[?s,?e)") 
			    (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (> (fuel-level ?s ?r) (size ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and
	(= (fuel-level ?e ?r) (- (fuel-level ?s ?r) (size ?x)))
        (on ?e ?x ?y)
        (:not (on ?e ?x ?z))
        (:when (:neq ?z table) (clear ?e ?z))
        (:when (:neq ?y table) (:not (clear ?e ?y)))))

  (define (operator government-puton)
      ;; consume twice the fuel of PUTON
      :at-time (?s ?e)
      :resources ((robot ?r))
      :parameters (?r ?X ?Y ?Z)
      :precondition (:and (:fact (robot ?r))
			  (:forall (time ?t "[?s,?e)") 
                            (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (> (fuel-level ?s ?r) (size ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and 
	(= (fuel-level ?e ?r) (- (fuel-level ?s ?r) (* 2 (size ?x))))
	(on ?e ?x ?y)
        (:not (on ?e ?x ?z))
        (:when (:neq ?z ?table) (clear ?e ?z))
        (:when (:neq ?y ?table) (:not (clear ?e ?y)))))
  )

(defproblem sipe-1 (0 :goal)
 :domain 'sipe-blocks-world
 :inits '(:and
           (= (fuel-level 0 flakey) 40)
           (:fact (robot flakey))
	   (:fact (block A))
	   (:fact (block Table))
	   (:fact (block B))
	   (:fact (block D))
           (= 15 (size A))
           (= 5 (size B))
           (= 5 (size D))
           (on 0 A B)
           (clear 0 A)
           (clear 0 D)
           (clear 0 Table)
           (on 0 D table)
           (on 0 B table))
 :goal '(:and
         (<= 5 (fuel-level :goal flakey) 50)
	 (on :goal ?x B)
	 (:neq ?x A)))

(defproblem sussman-anomaly (0 :goal)
  :domain 'blocks-world-domain
  :inits '(:and (on 0 C A)
  	        (on 0 A Table)
                (on 0 B Table) 
                (clear 0 C)
                (clear 0 B)
                (clear 0 Table))
  :goal '(:and (on :goal b c)
               (on :goal a b)))

(defproblem tower-invert (0 :goal)
       :domain 'blocks-world-domain
       :inits (initify '((on a b) (on b c) (on c table)
			 (clear a) (clear table)))
       :goal (goalify '((on b c) (on c a))))

(defun briefcase-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator mov-b)
      :at-time (?s ?e)
      :parameters (?m ?l)
      :precondition (:and (:neq ?m ?l)
			  (:forall (time ?t "[?s,?e)") (at ?t b ?m)))
      :effect
      (:forall ?o 
        (:when (:or (:eq ?o b) (in ?s ?o))
          (:and (at ?e ?o ?l) (:not (at ?e ?o ?m))))))

  (define (operator take-out)
      :at-time (?s ?e)
      :parameters (?x)
      :precondition (:neq ?x B)
      :effect (:not (in ?e ?x)))
      
  (define (operator put-in)
      :at-time (?s ?e)
      :parameters (?x ?l)
      :precondition (:neq ?x B)
      :effect (:when (:and (at ?e ?x ?l) (at ?e B ?l)) (in ?e ?x)))
  )


(defproblem get-paid (0 :goal)
       :name 'get-paid
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at B office) (at D office) (at P home))))

(defproblem get-paid2 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at P home) (at D office) (at B home))))

(defproblem get-paid3 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at P bank) (at D office) (at B home))))

(defproblem get-paid4 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at B home) (at D office) (at P bank))))

(defun clown-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator rub-tummy)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (:fact (hand ?x)) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)") 
            (:and (rubbing-tummy ?t ?x) (:not (free ?t ?x))))
           (:not (rubbing-tummy ?e ?x))
           (free ?e ?x)))

  (define (operator pat-head)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (:fact (hand ?x)) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)")
              (:and (:not (free ?t ?x)) (patting-head ?t ?x)))
           (free ?e ?x)
           (:not (patting-head ?e ?x))))

  (define (operator slap-knee)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (hand ?s ?x) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)")
              (:and (:not (free ?t ?x)) (slapping-knee ?t ?x)))
          (:not (slapping-knee ?e ?x))
          (free ?e ?x)))
  )


(defproblem clown1 (0 :end)
      :domain 'clown-world
       :inits (initify '((hand left) (hand right) (free left) (free right)))
       :goal (goalify '((hand ?a) (hand ?b)
			(patting-head ?a) (rubbing-tummy ?b))))

;; This next problem should NEVER return an answer.

(defproblem clown2 (0 :goal)
       :domain 'clown-world
       :inits (initify '((hand left) (hand right) (free left) (free right)))
       :goal (goalify '((hand ?a) (hand ?b) (hand ?c)
			(patting-head ?a) (rubbing-tummy ?b)
			(slapping-knee ?c))))

;;
;;; Now, some metric stuff
;;

(defun jug-world ()
  (reset-domain)
  (add-domain-function 'capacity)
  
  (define (operator fill)
      :at-time (?s ?e)
      :parameters (?jug)
      :precondition (:and (:fact (jug ?jug))
			  (>= (volume ?s ?jug) 0))
      :effect (= (volume ?e ?jug) (capacity ?jug)))

    (define (operator empty)
	:at-time (?s ?e)
	:parameters (?jug)
	:precondition (:and (:fact (jug ?jug))
			    (> (volume ?s ?jug) 0))
	:effect (= (volume ?e ?jug) 0))

    (define (operator pour)
	:at-time (?s ?e)
	:parameters (?jug1 ?jug2)
	:precondition (:and (:neq ?jug1 ?jug2)
			    (:fact (jug ?jug1))
			    (:fact (jug ?jug2)))
	:effect
	(:and
	 (= (volume ?e ?jug2)
	    (min (capacity ?jug2)
		 (+ (volume ?s ?jug1)
		    (volume ?s ?jug2))))
	 (= (volume ?e ?jug1)
	    (max 0
		 (+ (volume ?s ?jug1)
		    (volume ?s ?jug2)
		    (* -1 (capacity ?jug2)))))))
    )


(defproblem jug0 (0 :goal)
  :domain 'jug-world
  :inits '(:and (:fact (jug big))
	   (:fact (jug small))
	   (= 4 (capacity big))
	   (= 3 (capacity small))
	   (= 0 (volume 0 big))
	   (= 0 (volume 0 small)))
  :goal '(:and
	  (= 3 (volume :goal ?jug))))

(defproblem jug0-a (0 :goal)
  :domain 'jug-world
  :inits '(:and (:fact (jug big))
	   (:fact (jug small))
	   (= 4 (capacity big))
	   (= 3 (capacity small))
	   (= 0 (volume 0 big))
	   (= 0 (volume 0 small)))
  :goal '(:and
	  (:neq ?jug ?jugb)
	  (= 3 (volume :goal ?jug))
	  (= 4 (volume :goal ?jugb))))

(defproblem jug0-b (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 3 (volume :goal big))
	       ))

(defproblem jug0-c (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal small))
	       ))

(defproblem jug1 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal ?jug))))

(defproblem jug2 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
		(:fact (jug small))
		(= 4 (capacity big))
		(= 3 (capacity small))
		(= 0 (volume 0 big))
		(= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal big))))

(defproblem jug3 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
		(:fact (jug small))
		(= 4 (capacity big))
		(= 3 (capacity small))
		(= 0 (volume 0 big))
		(= 0 (volume 0 small)))
       :goal '(:and
	       (> (volume :goal big) 1)
	       (> (volume :goal small) 1)
	       (> (volume :goal small)
		  (volume :goal big))))

(defun fly-domain ()
  (reset-domain)
  (add-domain-function 'dist)
  (add-domain-function 'mpg)

  (define (operator fly)
      :resources ((plane ?m))
      :parameters (?m ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?m ?l)
			  (at ?s plane ?m)
			  (:forall (time ?t "[?s,?e]")
				   (> (fuel ?t plane) 0)))
      :effect
      (:and
       (= (- ?e ?s) (/ (dist ?m ?l) 650))
       (:influence ?s ?e (fuel plane) (- (/ 650 (mpg plane))))
       (:forall ?o
		(:when (:or (:eq ?o plane) 
			    (:forall (time ?t "[?s,?e]") (in ?t ?o)))
		  (:and
		   (at ?e ?o ?l)
		   (:forall (time ?t "(?s,?e]") (:not (at ?t ?o ?m))))))
       ))

  (define (operator deplane)
      :parameters (?x ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?x plane)
			  (:forall (time ?t "[?s,?e]")
				   (:and (at ?t plane ?l)
					 (at ?t ?x ?l))))
      :effect (:and (= (- ?e ?s) 0.25) (:not (in ?e ?x))))

  (define (operator board)
      :parameters (?x ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?x plane)
			  (:forall (time ?t "[?s,?e]")
				   (:and (at ?t plane ?l)
					 (at ?t ?x ?l))))
      :effect (:and (= (- ?e ?s) 0.50) (in ?e ?x)))
  )

(defproblem phd (7 13)
  :domain 'fly-domain
  :inits '(:and
	   (at 7 dan seatac)
	   (at 7 scott seatac)
	   (at 7 plane seatac)
	   (in 7 dan)
	   (:not (in 7 scott))
	   (= (fuel 7 plane) 1200)
	   (= (mpg plane) 3)
	   (= (dist seatac jfk) 3000))
  :goal '(:and
	  (at 12 dan seatac)
	  (at 13 scott jfk)
	  (at 13 plane jfk)))

(defun latch-world ()
  ;; From allen's Reasoning about Plans
  ;; This won't work unless we can model simultaneous action.
  (reset-domain)

  (define (operator turn-latch)
     :at-time (?s ?e)
     :effect
     (:and
       (:forall (time ?t "(?s,?e)") (latch ?t open))
       (:not (latch ?e open))))

  (define (operator pull-door)
     :at-time (?s ?e)
     :effect
     (:when (:forall (time ?t "[?s,?e]") (latch ?t open))
       (door ?e open))
     ))

(defproblem door-latch (0 :goal)
       :domain 'latch-world
       :inits '(:and (:not (latch 0 open)) (:not (door 0 open)))
       :goal '(:and (door :goal open)))

;; 
;; Hanoi
;;

(defun hanoi-domain ()
  (reset-domain)
  (define (operator move-disk)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?disk ?below-disk ?new-below-disk)
      :precondition (:and (:fact (disk ?disk))
                          (:fact (smaller ?disk ?new-below-disk))
			  (:neq ?new-below-disk ?below-disk)
			  (:neq ?new-below-disk ?disk)
			  (:neq ?below-disk ?disk)
			  (on ?s ?disk ?below-disk)
			  (clear ?s ?disk)
			  (clear ?s ?new-below-disk))
      :effect 
      (:and (clear ?e ?below-disk)
            (on ?e ?disk ?new-below-disk)
	    (:not (on ?e ?disk ?below-disk))
	    (:not (clear ?e ?new-below-disk)))))

(defun hanoi (n)
  (make-problem
   :start 0 :end :goal
   :name 'hanoi
   :domain #'hanoi-domain
   :inits
   (let* ((disks (subseq '(d1 d2 d3 d4 d5 d6 d7 d8 d9) 0 n))
	  (sizes (nconc (mapcar #'(lambda (d) `(:fact (smaller ,d p1))) disks)
			(mapcar #'(lambda (d) `(:fact (smaller ,d p2))) disks)
			(mapcar #'(lambda (d) `(:fact (smaller ,d p3))) disks)
			(mapcar #'(lambda (d) `(:fact (disk ,d))) disks)
			(mapcon
			 #'(lambda (d)
			     (mapcar #'(lambda (d2)
					 `(:fact (smaller ,(car d) ,d2)))
				     (cdr d)))
			 disks)))
	  (initial (append '((clear 0 p1)(clear 0 p2)(clear 0 d1))
			   (maplist
			    #'(lambda (d)
				(if (cdr d)
				    `(on 0 ,(car d) ,(cadr d))
				  `(on 0 ,(car d) p3)))
			    disks))))
     (cons :and (nconc (nreverse sizes) initial)))
   :goal 
   (let* ((disks (subseq '(d1 d2 d3 d4 d5 d6 d7 d8 d9) 0 n)))
     (cons :and (maplist #'(lambda (d)
			     (if (cdr d)
				 `(on :goal ,(car d) ,(cadr d))
			       `(on :goal ,(car d) p1)))
			 disks)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ferry domain

(defun ferry-domain ()
  (reset-domain)
  (define (operator board)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y)
      :precondition (:and (:fact (auto ?x)) (:fact (place ?y))
                          (at ?s ?x ?y)(at-ferry ?s ?y)(empty-ferry ?s))
      :effects  
      ((:cause (on ?e ?x ferry))
       (:cause (:not (at ?e ?x ?y)))
       (:cause (:not (empty-ferry ?e)))))
  (define (operator sail)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y)
      :precondition (:and (:fact (place ?x)) (:fact (place ?y))
                          (at-ferry ?s ?x) (:neq ?x ?y))
      :effects 
      ((:cause (at-ferry ?e ?y))
       (:cause (:not (at-ferry ?e ?x)))))
  (define (operator debark)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y)
      :precondition (:and (:fact (place ?x)) (:fact (place ?y))
                          (on ?s ?x ferry) (at-ferry ?s ?y))
      :effects  
      ((:cause (:not (on ?e ?x ferry)))
       (:cause (at ?e ?x ?y))
       (:cause (empty-ferry ?e)))))

(defproblem test-ferry (0 :goal)
 :domain 'ferry-domain
 :inits '(:and (:fact (place a)) (:fact (place b))
               (:fact (auto c1)) (:fact (auto c2))
               (at 0 c1 a) (at 0c2 a) (at-ferry 0 a)
               (empty-ferry 0))
 :goal '(:and (at :goal c1 b) (at :goal c2 b)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; molgen domain

(defun molgen-domain ()
  (reset-domain)
  
  ;; steps for building DNA molecules from mRNA
  
  (define (operator reverse-transcribe)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x)
      :precondition (:fact (mRNA ?x))
      :effects 
      ((:cause (connected-cDNA-mRNA ?e ?x))))
  (define (operator separate)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x)
      :precondition (connected-cDNA-mRNA ?s ?x)
      :effects 
      ((:cause (single-strand ?e ?x))
       (:cause (:not (connected-cDNA-mRNA ?e ?x)))))
  (define (operator polymerize)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x)
      :precondition (single-strand ?s ?x)
      :effects 
      ((:cause (hair-pin ?e ?x))
       (:cause (:not (single-strand ?e ?x)))))
  (define (operator digest)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x)
      :precondition (hair-pin ?s ?x)
      :effects 
      ((:cause (double-strand ?e ?x))
       (:cause (:not (hair-pin ?e ?x)))))
  ;; steps for splicing DNA molecules
  (define (operator ligate)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y)
      :precondition (:neq ?x ?y)
      :effects
      ((:cause (cleavable ?e ?y)
	:when (:and (double-strand ?s ?y) (:eq ?x LINKER)))
       (:cause (:and (contains ?e ?x ?y) (cleavable ?e ?y)
	      (:not (cleaved ?e ?x)) (:not (cleaved ?e ?y)))
	:when (:and (cleaved ?s ?x) (cleaved ?s ?y) (:neq ?x LINKER)))))
       
  (define (operator cleave)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x)
      :precondition (cleavable ?s ?x)
      :effects
      ((:cause (cleaved ?e ?x))
       (:cause (:not (cleavable ?e ?x)))))
  
  ;; Step for inserting a molecule into an organism
  (define (operator transform)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y)
      :precondition (:and (:fact (bacterium ?y))
                          (:neq ?x ?y)
			  (cleavable ?s ?x) ; molecule must be whole
			  (:fact (accepts ?x ?y))) ; Is molecule accepted?
      :effects
      ((:cause (contains ?e ?x ?y))
       (:cause (:not (cleavable ?e ?x)))))
  
  ;; purify a culture with an antibiotic
  (define (operator screen)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y ?z)
      :precondition (:and (:fact (bacterium ?x)) (:fact (antibiotic ?z))
                          (:neq ?x ?y) (:neq ?y ?z) (:neq ?x ?z)
			  (:fact (resists ?z ?y)) (contains ?s ?y ?x))
      :effects
      ((:cause (pure ?e ?x)))))


(defproblem rat-insulin (0 :goal)
 :domain 'molgen-domain
 :inits '(:and  (:fact (molecule insulin-gene))
                (:fact (molecule e-coli-exosome))
		(:fact (molecule junk-exosome)) 
		(:fact (molecule linker))
		(:fact (bacterium e-coli)) 
		(:fact (bacterium junk))
		(:fact (antibiotic antibiotic-1))
		(:fact (mRNA insulin-gene))
		(:fact (accepts junk-exosome junk))
		(:fact (accepts e-coli-exosome e-coli))
		(:fact (resists antibiotic-1 e-coli-exosome))
		(cleavable 0 e-coli-exosome) 
		(cleavable 0 junk-exosome))
       :goal '(:exists (bacterium ?y) 
	       (:exists (molecule ?x)
		(:and (contains :goal insulin-gene ?x)
		 (contains :goal ?x ?y)
		 (pure :goal ?y)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dan's fridge domain

(defun fridge-domain ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)
  (define (operator unscrew)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters ((screw ?x) (backplane ?y))
      :precondition (:and (screwed ?s ?X) (:fact (holds ?x ?y)))
      :effect (:not (screwed ?e ?X)))
  (define (operator screw)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters ((screw ?x) (backplane ?y))
      :precondition (:and (:not (screwed ?s ?X)) (:fact (holds ?x ?y)))
      :effect (screwed ?e ?X))
  (define (operator remove-backplane)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters ((backplane ?x) ?f ?a ?b ?c ?d)
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (in-place ?s ?x) 
                          (:fact (part-of ?x ?f))
                          (:not (fridge-on ?s ?f))
			  (:fact (holds ?a ?x))  (:fact (holds ?b ?x))  
			  (:fact (holds ?c ?x))  (:fact (holds ?d ?x))
			  (:not (screwed ?s ?a)) (:not (screwed ?s ?b)) 
			  (:not (screwed ?s ?c)) (:not (screwed ?s ?d)))
      :effect (:not (in-place ?e ?X)))
  (define (operator attach-backplane)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters ((backplane ?x) ?f ?a ?b ?c ?d)
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (:not (in-place ?s ?x))
			  (:fact (part-of ?x ?f))
                          (:not (fridge-on ?s ?f))
			  (:fact (holds ?a ?x))  (:fact (holds ?b ?x)) 
			  (:fact (holds ?c ?x))  (:fact (holds ?d ?x))
			  (:not (screwed ?s ?a)) (:not (screwed ?s ?b))
			  (:not (screwed ?s ?c)) (:not (screwed ?s ?d)))
      :effect (in-place ?e ?X))
  (define (operator start-fridge)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?f ?a ?b ?c ?d (backplane ?x))
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (in-place ?s ?x) (:fact (part-of ?x ?f))
			  (:fact (holds ?a ?x))(:fact (holds ?b ?x))
                          (:fact (holds ?c ?x))(:fact (holds ?d ?x))
			  (screwed ?s ?a) (screwed ?s ?b) 
                          (screwed ?s ?c) (screwed ?s ?d)
			  (:not (fridge-on ?s ?f)))
      :effect (fridge-on ?e ?f))
  (define (operator stop-fridge)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?f)
      :precondition (fridge-on ?s ?f)
      :effect (:not (fridge-on ?e ?f)))
  (define (operator change-compressor)
      :at-time (?s ?e)
      :resources ((ucpop))
      :parameters (?x ?y (backplane ?a))
      :precondition (:and (:neq ?x ?y) (:not (in-place ?s ?a)) 
                          (covers ?s ?a ?x)
			  (:fact (compressor ?x))
			  (:fact (compressor ?y))
			  (attached ?s ?x) (:not (attached ?s ?y)))
      :effect (:and (:not (attached ?e ?X)) (attached ?e ?y)
		    (:not (covers ?e ?a ?x)) (covers ?e ?a ?y))))

(defproblem fixa (0 :goal)
       :domain 'fridge-domain
       :inits '(:and (:fact (screw s1)) (:fact (screw s2))
                     (:fact (screw s3)) (:fact (screw s4))
		(:fact (backplane b1))
		(:fact (compressor c1))
                (:fact (compressor c2)) (:fact (fridge f1))
		(covers 0 b1 c1) (:fact (part-of b1 f1))
		(:fact (holds s1 b1)) (:fact (holds s2 b1))
		(:fact (holds s3 b1)) (:fact (holds s4 b1))
;		(ok 0 c1) (ok 0 c2)
		(fridge-on 0 f1)
		(screwed 0 s1) (screwed 0 s2) (screwed 0 s3) (screwed 0 s4)
		(in-place 0 b1) (attached 0 c1)
		;(:not (attached 0 c2))
		)
       :goal '(:and (attached :goal c2)
	       ;(ok :goal c2)
	       ))


(defun mcd-blocksworld ()
  (reset-domain)
  (define (operator puton)
      :resources ((ucpop))
      :at-time (?s ?e)
      :parameters (?x ?y ?d)
      :precondition (:and (:neq ?x ?y) (:neq ?x table) (:neq ?d ?y) 
			  (on ?s ?x ?d) 
			  (:or (:eq ?x Table)
			       (:forall (block ?b) (:not (on ?s ?b ?x))))
			  (:or (:eq ?y Table)
			       (:forall (block ?b) (:not (on ?s ?b ?y)))))
      :effect
      (:and
       (on ?e ?x ?y) (:not (on ?e ?x ?d))
       (:forall ?c
		(:when (:or (:eq ?y ?c) (above ?s ?y ?c))
		  (above ?e ?x ?c))
		(:when  (:and (above ?s ?x ?c)
			      (:neq ?y ?c)
			      (:not (above ?s ?y ?c)))
		  (:not (above ?e ?x ?c)))))))

(defproblem ysp (0 :goal)
  :domain 'mcd-blocksworld
  :inits '(:and
	   (on 0 a b) (on 0 c table) (on 0 d table) (on 0 b table))
  :goal '(:and (on :goal a d) (above :goal a c)))

;;
;;; Ye old strips world
(defun strips-world ()
  (reset-domain)
  (define (fact (loc-in-room x y room))
      (labels 
	  ((convert-loc-to-room (x y)
	     (cond ((and (<= y 5) (>= y 1))
		    (cond ((and (<= x 4) (>= x 3))   'rpdp)
			  ((and (<= x 9) (>= x 5))   'rclk)
			  ((and (<= x 12) (>= x 10)) 'rril)
			  (t (error "Invalid x room coordinate"))))
		   ((and (<= y 10) (>= y 6))
		    (cond ((and (<= x 2) (>= x 1))   'runi)
			  ((and (<= x 6) (>= x 3))   'rmys)
			  ((and (<= x 9) (>= x 7))   'rram)
			  ((and (<= x 12) (>= x 10)) 'rhal)
			  (t (error "Invalid x room coordinate"))))
		   (t (error "Invalid y room coordinate")))))
	(cond ((or (variable? x) (variable? y))
	       :no-match-attempted)
	      ((variable? room) 
	       (setb room (convert-loc-to-room x y)))
	      ((atom room)
	       (if (eq room (convert-loc-to-room x y)) :ok nil))
	      (t (error "Room is not a variable or an atom")))))
  
  (define (operator GOTO-BOX)
      :at-time (?s ?e)
      :parameters (?box ?room)
      :precondition 
      (:and (:fact (is-type ?box object))	;[6]
	    (in-room ?s ?box ?room)	;[5]
	    (in-room ?s robot ?room))	;[5]
      :effect (:and (:forall (?1 ?2) 
			     (:when (at ?s robot ?1 ?2) 
			       (:not (at ?e robot ?1 ?2))))
		    (:forall (?1)
			     (:when (:and (:neq ?1 ?box) (next-to ?s robot ?1))
			       (:not (next-to ?e robot ?1))))
		    (next-to ?e robot ?box)))

  (define (operator GOTO-DOOR)
      :at-time (?s ?e)
      :parameters (?door ?roomx ?roomy)
      :precondition 
      (:and (:fact (is-type ?door door)) ;[6]
	    (:fact (connects ?door ?roomx ?roomy)) ;[6]
	    (in-room ?s robot ?roomx))	;[5]
      :effect (:and (:forall (?1 ?2) 
			     (:when (at ?s robot ?1 ?2)
			       (:not (at ?e robot ?1 ?2))))
		    (:forall (?1)
			     (:when (:and (:neq ?1 ?door)
					  (next-to ?s robot ?1))
			       (:not (next-to ?e robot ?1))))
		    (next-to ?e robot ?door)))
  
  (define (operator GOTO-LOC)
      :at-time (?s ?e)
      :parameters (?x ?y ?roomx)
      :precondition 
      (:and (loc-in-room ?x ?y ?roomx) ;[6]
	    (in-room ?s robot ?roomx))	;[5]
      :effect (:and (:forall (?1 ?2) 
			     (:when (at ?s robot ?1 ?2)
			       (:not (at ?e robot ?1 ?2))))
		    (:forall (?1)
			     (:when (next-to ?s robot ?1)
			       (:not (next-to ?e robot ?1))))
		    (at ?e robot ?x ?y)))
  
  (define (operator PUSH-BOX)
      :at-time (?s ?e)
      :parameters (?boxx ?boxy ?roomx)
      :precondition 
      (:and (:fact (is-type ?boxy object))	;[6]
	    (:fact (pushable ?boxx))		;[6]
	    (in-room ?s ?boxx ?roomx)	;[5]
	    (in-room ?s ?boxy ?roomx)	;[5]
	    (in-room ?s robot ?roomx)	;[5]
	    (next-to ?s robot ?boxx))	;[1]
      :effect
      (:and (:forall (?1 ?2) 
		     (:and (:when (at ?s robot ?1 ?2)
			     (:not (at ?e robot ?1 ?2)))
			   (:when (at ?s ?boxx ?1 ?2)
			     (:not (at ?e ?boxx ?1 ?2)))))
	    (:forall (?1)
		     (:and (:when (:and (:neq ?1 ?boxx) (next-to ?s robot ?1)) 
			     (:not (next-to ?e robot ?1)))
			   (:when (:and (:neq ?1 ?boxy)  (next-to ?s ?boxx ?1))
			     (:not (next-to ?e ?boxx ?1)))
			   (:when (:and (:neq ?1 robot) (:neq ?1 ?boxy)
					(next-to ?s ?1 ?boxx))
			     (:not (next-to ?e ?1 ?boxx)))))
	    (next-to ?e ?boxy ?boxx)
	    (next-to ?e ?boxx ?boxy)
	    (next-to ?e robot ?boxx)))
  
  (define (operator PUSH-TO-DOOR)
      :at-time (?s ?e)
      :parameters (?box ?door ?roomx ?roomy)
      :precondition
      (:and (:fact (connects ?door ?roomx ?roomy)) ;[6]
	    (:fact (pushable ?box))		;[6]
	    (:fact (is-type ?door door))	;[6]
	    (in-room ?s robot ?roomx)	;[5]
	    (in-room ?s ?box ?roomx)	;[5]
	    (next-to ?s robot ?box))	;[1]
      :effect     
      (:and (:forall (?1 ?2) 
		     (:and (:when (at robot ?s ?1 ?2)
			     (:not (at robot ?e ?1 ?2)))
			   (:when (at ?box ?s ?1 ?2)
			     (:not (at ?box ?e ?1 ?2)))))
	    (:forall (?1)
		     (:and (:when (:and (:neq ?1 ?box) (next-to ?s robot ?1))
			     (:not (next-to ?e robot ?1)))
			   (:when (:and (:neq ?1 ?door) (next-to ?s ?box ?1))
			     (:not (next-to ?e ?box ?1)))
			   (:when (:and (:neq ?1 robot) (next-to ?s ?1 ?box))
			     (:not (next-to ?e ?1 ?box)))))
	    (next-to ?e ?box ?door)
	    (next-to ?e robot ?box)))
  
  
  (define (operator PUSH-TO-LOC)
      :at-time (?s ?e)
      :parameters (?box ?x ?y ?roomx)
      :precondition
      (:and (:fact (pushable ?box))		;[6]
	    (loc-in-room ?x ?y ?roomx)	;[6]
	    (in-room ?s robot ?roomx)	;[5]
	    (in-room ?s ?box ?roomx)	;[5]
	    (next-to ?s robot ?box))	;[1]
      :effect
      (:and (:forall (?1 ?2) 
		     (:and (:when (at ?s robot ?1 ?2)
			     (:not (at ?e robot ?1 ?2)))
			   (:when (:and (:neq ?x ?1) (:neq ?y ?2)
					(at ?s ?box ?1 ?2))
			     (:not (at ?e ?box ?1 ?2)))))
	    (:forall (?1)
		     (:and (:when (:and (:neq ?1 ?box) (next-to ?s robot ?1))
			     (:not (next-to ?e robot ?1)))
			   (:when (next-to ?s ?box ?1) 
			     (:not (next-to ?e ?box ?1)))
			   (:when (:and (:neq ?1 robot) (next-to ?s ?1 ?box))
			     (:not (next-to ?e ?1 ?box)))))
	    (at ?e ?box ?x ?y)
	    (next-to ?e robot ?box)))
  
  (define (operator GO-THRU-DOOR)
      :at-time (?S ?e)
      :parameters (?door ?roomy ?roomx)
      :precondition 
      (:and (:fact (connects ?door ?roomx ?roomy)) ;[6]
	    (:fact (is-type ?door door))	;[6]
	    (:fact (is-type ?roomx room))	;[6]
	    (in-room ?s robot ?roomy)	;[5]
	    (statis ?s ?door open)	;[2]
	    (next-to ?s robot ?door))	;[1]
      :effect   
      (:and (:forall (?1 ?2) 
		     (:when (at ?s robot ?1 ?2)
		       (:not (at ?e robot ?1 ?2))))
	    (:forall (?1)
		     (:when (next-to ?s robot ?1)
		       (:not (next-to ?e robot ?1))))
	    (:when (in-room ?s robot ?roomy)
	      (:not (in-room ?e robot ?roomy)))
	    (in-room ?e robot ?roomx)))
  
  (define (operator PUSH-THRU-DOOR)
      :at-time (?s ?e)
      :parameters (?box ?door ?roomy ?roomx)
      :precondition
      (:and (:fact (connects ?door ?roomy ?roomx)) ;[6]
	    (:fact (pushable ?box))		;[6]
	    (:fact (is-type ?door door))	;[6]
	    (:fact (is-type ?roomx room))	;[6]
	    (in-room ?s robot ?roomy)	;[5]
	    (in-room ?s ?box ?roomy)	;[5]
	    (statis ?s ?door open)		;[2]
	    (next-to ?s ?box ?door)	;[1]
	    (next-to ?s robot ?box))	;[1]
      :effect
      (:and (:forall (?1 ?2) 
		     (:and (:when (at ?s robot ?1 ?2)
			     (:not (at ?e robot ?1 ?2)))
			   (:when (at ?s ?box ?1 ?2)
			     (:not (at ?e ?box ?1 ?2)))))
	    (:forall (?1)
		     (:and (:when (:and (:neq ?1 ?box) (next-to ?s robot ?1))
			     (:not (next-to ?e robot ?1)))
			   (:when (next-to ?s ?box ?1)
			     (:not (next-to ?e ?box ?1)))
			   (:when (:and (:neq ?1 robot) (next-to ?s ?1 ?box))
			     (:not (next-to ?e ?1 ?box)))))
	    (:when (in-room ?s robot ?roomy) (:not (in-room ?e robot ?roomy)))
	    (:when (in-room ?s ?box ?roomy) (:not (in-room ?e ?box ?roomy)))
	    (in-room ?e robot ?roomx)
	    (in-room ?e ?box ?roomx)
	    (next-to ?e robot ?box)))
       
  (define (operator OPEN-DOOR)
      :at-time (?s ?e)
      :parameters (?door)
      :precondition 
      (:and (:fact (is-type ?door door))	;[6]
	    (next-to ?s robot ?door)	;[5]
	    (statis ?s ?door closed))	;[5]
      :effect
      (:and (:when (statis ?s ?door closed) (:not (statis ?s ?door closed)))
	    (statis ?e ?door open)))
  
  (define (operator CLOSE-DOOR)
      :at-time (?s ?e)
      :parameters (?door)
      :precondition 
      (:and (:fact (is-type ?door door))	;[6]
	    (next-to ?s robot ?door)	;[5]
	    (statis ?s ?door open))	;[5]
      :effect
      (:and (:when (statis ?s ?door open) (:not (statis ?e ?door open)))
	    (statis ?e ?door closed))))

(defvar *strips-inits*
    (mapcar #'(lambda (f)
		(list :fact f))
    '((connects dunimys runi rmys)
      (connects dmysram rmys rram)
      (connects dramhal rram rhal)
      (connects dmyspdp rmys rpdp)
      (connects dpdpclk rpdp rclk)
      (connects dmysclk rmys rclk)
      (connects dramclk rram rclk)
      (connects dclkril rclk rril)
      (connects dunimys rmys runi)
      (connects dmysram rram rmys)
      (connects dramhal rhal rram)
      (connects dmyspdp rpdp rmys)
      (connects dpdpclk rclk rpdp)
      (connects dmysclk rclk rmys)
      (connects dramclk rclk rram)
      (connects dclkril rril rclk)
      (statis dunimys open)
      (statis dmysram open)
      (statis dramhal open)
      (statis dmyspdp open)
      (statis dpdpclk open)
      (statis dmysclk open)
      (statis dramclk open)
      (statis dclkril closed)
      (is-type dunimys door)
      (is-type dmysram door)
      (is-type dramhal door)
      (is-type dmyspdp door)
      (is-type dpdpclk door)
      (is-type dmysclk door)
      (is-type dramclk door)
      (is-type dclkril door)
      (is-type runi room)
      (is-type rmys room)
      (is-type rram room)
      (is-type rhal room)
      (is-type rpdp room)
      (is-type rclk room)
      (is-type rril room)
      (is-type box1 object)
      (is-type box2 object)
      (is-type box3 object)
      (pushable box1)
      (pushable box2)
      (pushable box3))))


(defproblem move-boxes (0 :goal)
  :domain 'strips-world
  :inits (cons :and
	       (append *strips-inits*
		 '((in-room 0 robot rril)
		   (in-room 0 box1 rpdp)
		   (in-room 0 box2 rpdp)
		   (in-room 0 box3 rclk))))
  :goal '(:and (next-to :goal box1 box2)(next-to :goal box2 box3)))

