;;
;;; Parsing for metric functions.
;;
(in-package "ZENO")

(proclaim '(optimize (speed 3) (safety 2) (space 0) (compilation-speed 0)))

(defconstant *TOLERANCE* 1.0d-10)
(defvar *SIMPLEX-TOLERANCE* 0.0001)
(defvar *METRIC-COUNT* 0)
(defvar *SLACK-COUNT* 0)

(defmacro == (a b)
  `(< (abs (- ,a ,b)) *tolerance*))

(defmacro approx> (a b)
  `(> (- ,a *tolerance*) ,b))

(defmacro approx< (a b)
  `(< ,a (- ,b *tolerance*)))

(defmacro genmetric ()
  `(incf *metric-count*))

(defmacro genslack ()
  `(decf *slack-count*))

(defmacro ROW-SUBJECT (row)
  `(car (row-vars ,row)))

(defmacro ROW-BASIC-VAR (row)
  `(car (row-vars ,row)))

(defmacro ORDER-VARIABLES (vars)
  `(sort ,vars 'more-junior-than :key #'cdr))

#-allegro(defmacro DOUBLE-FLOAT (x)
  `(coerce ,x 'double-float))


;;
;;; Plan interface to the metric reasoner: 6 functions
;;

(defvar *debug-time* nil "Set to T to debug temporal stuff.")

(defun METRIC-CONSTRAINTS-OK? (plan)
  (let ((pc (plan-constraints plan)))
    (if (consp pc) (setf pc (car pc)))
    (cond ((null pc)
           (and (plan-bindings plan)
		(ztimes-consistent? (plan-ordering plan))))
          (t
   	   (and (cs-consistent? pc)
		(plan-bindings plan)
		(or (not *debug-time*)
		    (ztimes-consistent? (Plan-ordering plan)))
		)))))

(defvar *nonlins-ok* nil "Set to T to allow nonlinear eqns in final sol'n")

(defun METRIC-CONSTRAINTS-SOUND? (plan)
  (let ((pc (plan-constraints plan)))
    (if (consp pc) (setf pc (car pc)))
    (or (null pc)
	(and (cs-consistent? pc)
	     (ztimes-consistent? (plan-ordering plan))
	     (or (and *nonlins-ok* (verify-plan-with-user plan pc))
		 (null (nlin-table (cs-nlin pc))))))))

(defun verify-plan-with-user (plan con)
  (terpri)
  (display-plan plan)
  (show con)
  (terpri)
  (y-or-n-p "Is this plan acceptable? "))

;(defun METRIC-CONSTRAINTS-SOUND? (plan)
;  (let ((pc (plan-constraints plan)))
;    (or (null pc)
;	(cs-consistent? pc))))

(defun INITIALIZE-METRIC-CONSTRAINTS (plan)
  (setf *metric-count* 0
	*slack-count* 0)
  (setf (plan-constraints plan) (make-instance 'csolver))
  (values (plan-constraints plan)))

(defun COPY-METRIC-CONSTRAINTS (plan)
  (let ((old (plan-constraints plan)))
    (cond (old
	   (if (consp old)
	       old
	     (cons old nil)))		;means "copy when you need to"
	  (t
	   (values old)))))

(defun ADD-METRIC-CONSTRAINT (plan thing)
  ;; Add a new constraint THING to PLAN.
  ;;
  (count-stat .constraints.)
  (cond ((eq (car thing) :eq)
         (setf (plan-bindings plan)
           (add-bind (acons (second thing) (third thing) nil)
                     (plan-bindings plan))))
        ((eq (car thing) :neq)
         (setf (plan-bindings plan)
           (add-bind `(,(cons :not (cons (second thing) (third thing))))
                      (plan-bindings plan))))
        (t
	 (let ((c (plan-constraints plan)))
	   (cond ((consp c)
		  ;; need to copy it for sure
		  (setf c (make-instance 'csolver))
		  (spawn c (car (plan-constraints plan)))
		  (setf (plan-constraints plan) c)
		  (new-constraint c thing))
		 (c
		  (new-constraint c thing))
		 (t
		  (new-constraint (initialize-metric-constraints plan)
				  thing))))))
  (values))

(defun GET-METRIC-VALUE (plan variable)
  (let ((cs (plan-constraints plan)))
    (if (consp cs) (setf cs (car cs)))
    (with-slots (nlin) cs
      (or (cdr (assoc variable (nlin-values nlin)))
	  (bind-variable variable (plan-bindings plan))))))

(defun VASSOC (name var-list)
  (declare (optimize (speed 3) (safety 1))
	   (type list var-list))
  (dolist (v var-list)
    (if (eq name (cdr v)) (return v))))

(defun MOVE-TO-FRONT (elt list)
  (cond ((eq elt (car list))
	 list)
	(t
	 (cons elt (delete-1 elt list)))))

(defun PARSE-CONSTRAINT (form)
  (declare (optimize (speed 3) (safety 0)))
  (let ((fn (get (car form) :parse)))
    (cond ((null fn)
	   (error "Cannot parse ~s constraints!"
		  (car form)))
	  (t
	   (funcall fn (cdr form))))))

(defun REDUCE-TO-SYMBOL (form)
  (cond ((atom form)
	 (if (numberp form)
	     form
	     (values (cons 1 form) nil)))
	(t
	 (multiple-value-bind (sym eqns)
	     (parse-constraint form)
	   (cond ((null sym)
		  (error "An argument did not reduce to a symbol:~%~s"
			 form))
		 (t
		  (values sym eqns)))))))

(defun COLLECT-SYMBOLS-AND-EQNS (form)
  (declare (optimize (speed 3) (safety 1)))
  (cond ((null form)
	 (values nil nil))
	(t
	 (multiple-value-bind (sym eqns1)
	     (reduce-to-symbol (car form))
	   (multiple-value-bind (syms eqns2)
	       (collect-symbols-and-eqns (cdr form))
	     (values (cons sym syms)
		     (nconc eqns1 eqns2)))))))

(defun NEGATE-SYM (metric-sym)
  (cons 
   (- (car metric-sym))
   (cdr metric-sym)))

(defun SCALE-SYM (metric-sym scalar)
  (cond ((= 1 scalar)
	 (cons (car metric-sym) (cdr metric-sym)))
	(t
	 (cons 
	  (* (car metric-sym) scalar)
	  (cdr metric-sym)))))

(defun CONSTANT-SPLIT (syms)
  ;; split up SYMS, in order, into an ordered list of constants and numbers
  (cond ((null syms)
	 (values nil nil))
	((numberp (car syms))
	 (multiple-value-bind (numbers syms2)
	     (constant-split (cdr syms))
	   (values (cons (car syms) numbers)
		   syms2)))
	(t
	 (multiple-value-bind (numbers syms2)
	     (constant-split (cdr syms))
	   (values numbers
		   (cons (car syms) syms2))))))

(defpropfn = :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
      (values
       nil
       (nconc (create-pairwise-equality syms)
	      eqns))))

(defun CREATE-PAIRWISE-EQUALITY (syms)
  ;; Returns a list of pairwise equalities, in linear format
  (cond ((null (cdr syms))
	 (error "Bad format for equality!"))
	(t
	 (multiple-value-bind (numbers vars)
	     (constant-split syms)
	   (cond ((null numbers)
		  (values (create-pairwise-equality-1 vars)))
		 ((not (every #'(lambda (elt) (= elt (car numbers)))
			      (cdr numbers)))
		  (throw :inconsistent 'violate-equality))
		 (t
		  (values (mapcar #'(lambda (s)
				      `(:equation ,(car numbers) ,s))
				  vars))))))))

(defun CREATE-PAIRWISE-EQUALITY-1 (syms)
  ;; Returns a list of pairwise equalities, in linear format
  (cond ((null (cdr syms)) nil)
	(t
	 (cons
	  (list ':equation 0 (negate-sym (car syms)) (cadr syms))
	  (create-pairwise-equality-1 (cdr syms))))))

(defpropfn - :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
      (cond ((third syms)
	     (error "Too many arguments for minus (-):~%~s" args))
            ((null (cdr syms))
             (values (negate-sym (car syms)) eqns))
	    (t
	     (multiple-value-bind (numbers vars)
		 (constant-split syms)
	       (cond ((null vars)
		      (cond ((= 0 (cadr numbers))
			     ;; subtract 0, don't generate a new var
                             (values (car numbers) eqns))
			    (t
			     (values (- (car numbers) (cadr numbers))
				     eqns))))
		     ((and (numberp (second syms))  ;; was the number second?
		           (= 0 (second syms)))
 		       ;; subtract 0, don't generate a new var
                       (values (car syms) eqns))
		     (t
		      (let ((eqnval (cons 1 (genmetric))))
			(values eqnval
				(cons `(:equation 0 ,eqnval ,(cadr syms)
					,(scale-sym (car syms) -1))
				      eqns))))))))))

(defpropfn + :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
    (multiple-value-bind (numbers vars)
	(constant-split syms)
      (let ((sum (if numbers (apply #'+ numbers) 0)))
	(cond ((null vars)
	       (values sum nil))
	      (t
	       (let ((eqnval (genmetric)))
		 (values
		  (cons 1 eqnval)
		  (cons `(:equation ,sum (-1 . ,eqnval) ,@vars)
			eqns)))))))))
     
(defpropfn * :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
    (multiple-value-bind (sym *eqns)
	(pairwise-multiplication syms)
      (values
       sym
       (nconc *eqns eqns)))))
	     
(defun PAIRWISE-MULTIPLICATION (syms)
  (multiple-value-bind (numbers vars)
      (constant-split syms)
    (let ((scalar (if numbers (apply #'* numbers) 1)))
      (cond ((null vars)
	     (values scalar nil))
	    ((null (cdr vars))
	     (values (scale-sym (car vars) scalar) nil))
	    ((= 1 scalar)
	     (pair-multiply-1 vars))
	    (t
	     (pair-multiply-1 (cons (scale-sym (car vars) scalar)
				    (cdr vars))))))))

(defun PAIR-MULTIPLY-1 (vars)
  (cond ((null (cdr vars))
	 (values (car vars) nil))
	(t
	 (let ((eqnval (cons 1 (genmetric))))
	   (multiple-value-bind (sym1 eqns)
	       (pair-multiply-1 (cdr vars))
	     (values eqnval
		     (cons (list :nonlinear '*
				 :args (list (car vars) sym1)
				 :value eqnval)
			   eqns)))))))

(defpropfn / :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
      (cond ((third syms)
	     (error "Too many arguments for quotient (/):~%~s" args))
	    (t
	     (multiple-value-bind (numbers vars)
		 (constant-split syms)
	       (cond ((null vars)
		      (cond ((= 0 (cadr numbers))
			     ;; divide by zero, an error
			     (throw :inconsistent 'divide-by-zero))
			    (t
			     (values (/ (double-float (car numbers))
					(double-float (cadr numbers)))
				     eqns))))
		     ((numberp (second syms))       ;; was the number second?
		      (cond ((= 0 (second syms))
			     (throw :inconsistent 'divide-by-zero))
			    (t
			     (values (scale-sym (first vars) 
			     	     (/ 1.0d0 (car numbers)))
				     eqns))))
		     (t
		      (let ((eqnval (cons 1 (genmetric))))
			(values eqnval
				(cons (list :nonlinear '/
					    :args syms
					    :value eqnval)
				      eqns))))))))))

(defpropfn MAX :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
    (if (cddr syms)
	(error "MAX only accepts two args."))
    (multiple-value-bind (nums vars)
	(constant-split syms)
	(if nums
	    (setf nums (cons (apply #'max nums) nil)))
	(cond ((null vars)
	       (values nums eqns))
	      (t
	       (let ((eqnval (cons 1 (genmetric))))
		 (values eqnval
			 (cons (list :nonlinear 'max
				     :args (nconc nums vars)
				     :value eqnval)
			       eqns))))))))



(defpropfn MIN :parse (args)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
    (if (cddr syms)
	(error "MIN only accepts two args."))
    (multiple-value-bind (nums vars)
	(constant-split syms)
      (if nums
	  (setf nums (cons (apply #'min nums) nil)))
      (cond ((null vars)
	     (values nums eqns))
	    (t
	     (let ((eqnval (cons 1 (genmetric))))
	       (values eqnval
		       (cons (list :nonlinear 'min
				   :args (nconc nums vars)
				   :value eqnval)
			     eqns))))))))

(defun TRANSITIVE-INEQ-PARSE (args relation)
  (multiple-value-bind (syms eqns)
      (collect-symbols-and-eqns args)
    (values nil
	    (nconc (create-pairwise-inequality syms relation)
		   eqns))))
    
(defpropfn <= :parse (args)
  (transitive-ineq-parse args '<=))

(defpropfn >= :parse (args)
  (transitive-ineq-parse args '>=))

(defpropfn > :parse (args)
  (transitive-ineq-parse args '>))

(defpropfn < :parse (args)
  (transitive-ineq-parse args '<))

(defun CREATE-PAIRWISE-INEQUALITY (syms relation)
  ;; Returns a list of pairwise inequalities, in linear format
  (cond ((null (cdr syms))
	 (error "Bad format for equality!"))
	(t
	 (multiple-value-bind (numbers vars)
	     (constant-split syms)
	   (declare (ignore vars))
	   (cond ((and (cdr numbers)
		       (do ((a numbers (cdr a)) ;; verify ordering of numbers
			    (b (cdr numbers) (cdr b)))
			   ((null b) NIL)
			 (unless (funcall relation (car a) (car b))
			   (return T))))
		  (throw :inconsistent 'violate-inequality))
		 (t
		  (values (create-pairwise-inequality-1 syms relation))))))))

(defun CREATE-PAIRWISE-INEQUALITY-1 (syms relation)
  ;; Returns a list of pairwise equalities, in linear format
  (cond ((null (cdr syms)) nil)
	((numberp (car syms))
	 (if (numberp (cadr syms))
	     (create-pairwise-inequality-1 (cdr syms) relation)
	   (cons
	    (list :inequality relation (* -1 (car syms)) 
                              (negate-sym (cadr syms)))
	    (create-pairwise-inequality-1 (cdr syms) relation))))
	((numberp (cadr syms))
	 (cons
	  (list :inequality relation (cadr syms) (car syms))
	  (create-pairwise-inequality-1 (cdr syms) relation)))
	(t
	 (cons
	  (list ':inequality relation 0 (car syms) (negate-sym (cadr syms)))
	  (create-pairwise-inequality-1 (cdr syms) relation)))))

;;
;;; Constraint solver -- main routine.
;;

(defclass CSOLVER ()
  ((eqn :initform nil :accessor cs-eqn)
   (ineq :initform nil :accessor cs-ineq)
   (nlin :initform nil :accessor cs-nlin)
   (consistent? :initform T :accessor cs-consistent?)
   ))

(defmethod INITIALIZE-INSTANCE :AFTER ((self csolver) &rest ignore)
  (declare (ignore ignore))
  (with-slots (eqn ineq nlin) self
    (setf eqn (make-instance 'eqn-solver :csolver self))
    (setf ineq (make-instance 'ineq-solver :csolver self))
    (setf nlin (make-instance 'nlin-solver :csolver self))
    ))

(defmethod SPAWN ((self csolver) old-csolver)
  ;; Makes this instance a duplicate of OLD-CSOLVER.
  (with-slots (eqn ineq nlin consistent?) self
    (cond ((cs-consistent? old-csolver)
;	   (setf (cs-foo self) (cs-foo old-csolver))
	   (spawn eqn (cs-eqn old-csolver))
	   (spawn ineq (cs-ineq old-csolver))
	   (spawn nlin (cs-nlin old-csolver))
	   (setf consistent? t))
	  (t
	   (setf consistent? nil)))
    (values)))

(defmethod NEW-CONSTRAINT ((self csolver) form)
  (multiple-value-bind (var entries)
      (parse-constraint form)
    (cond (var
	   (error "Constraint was not an equation or an inequality!"))
	  (t
	   (with-slots (eqn ineq nlin consistent?) self
	     (let ((val nil))
	       (dolist (e entries)
		 (setf val 
		   (ecase (car e)
		     (:equation
		      (count-stat .equations.)
		      (add-raw-equation eqn e))
		     (:nonlinear
		      (count-stat .nonlin.)
		      (add-raw-equation nlin e))
		     (:inequality
		      (count-stat .ineq.)
		      (add-raw-equation ineq e))
		      ))
		 (when (eq val :inconsistent)
		   (setf consistent? nil)
		   (return)))
		   )))))
    (values))

(defmethod show ((self csolver))
  (with-slots (eqn ineq nlin consistent?) self
    (format t "~&Equations:~%")
    (show-constraints eqn)
    (format t "~%Inequalities:~%")
    (show-constraints ineq)
    (format t "~%Nonlinear delays:~%")
    (show-constraints nlin)
    )
  (values))

(defmethod reset ((self csolver))
  (with-slots (eqn ineq nlin consistent?) self
    (reset eqn)
    (reset ineq)
    (reset nlin)
    (setf consistent? t)
    )
  (values))
