;;; Copyright (C) 1994 by Istituto per la Ricerca Scientifica e Tecnologica 
;;; (IRST) (38050 Povo, Trento Italy) and the Trustees of the University 
;;; of Rochester (Rochester, NY 14627, USA).  All rights reserved.
;;; See the COPYRIGHT.TEXT file for more information

;; Some separation of the ui from the engine, and some other cleanups
;; by Brad Miller miller@cs.rochester.edu 12/7/93
;; more heavy duty optimizations and metering by Brad Miller 
;; miller@cs.rochester.edu 12/13/93

#|  Dec. 18 1993

 *****************************************************
 *                                                   *
 *  TimeGraph II (TG-II) - Temporal Reasoning System *
 *                                                   *
 *  written by Alfonso Gerevini                                 *
 *  IRST 38050 Povo Trento Italy                     *
 *  and                                              *
 *  Department of Computer Science                   *
 *  University of Rochester, 14627 Rochester, USA    *
 *                                                   *
 *  email: gerevini@irst.it                          *
 *         gerevini@cs.rochester.edu	             *
 *                                                   *
 *****************************************************

|#

;; TIMEGRAPH-II CODE

(in-package TG-II)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                         ;;
;; FUNCTIONS FOR THE CONSTRUCTION OF THE {<=,<,=/=}-GRAPH  ;;
;;                                                         ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun DFS (transpose)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((l-points *points*))
    (setq *points* nil)
    (dolist (pp l-points)
      (cond ((zerop (time-node-c (get-tg pp)))
	     (if transpose
                 (push pp *roots*))
	     (DFS_Visit pp transpose))))))

;depth-first search 
; See Cormen,Leiserson,Rivest "Introduction to Algorithms"
; During the search only <,<= edges are considered.
; When the search is applied to a DAG, at the end of the search the
; global variable points contains a topological-sort order of the vertices
; of the graph.


(defun DFS_Visit (node transpose)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((tg-entry (get-tg node)))
    (setf (time-node-c tg-entry) 1)
    (cond ((not transpose)
           (dolist (next (time-node-next tg-entry))
             (cond ((zerop (time-node-c (get-tg (time-edge-to next))))
                    (DFS_Visit (time-edge-to next) transpose)))))
          (t
           (dolist (next (time-node-prev tg-entry))
             (cond ((zerop (time-node-c (get-tg (time-edge-from next))))
                    (update-alist node (cons next (cdr (assoc node *trees*))) *trees*)
                    (DFS_Visit (time-edge-from next) transpose))))))
    (setf (time-node-c tg-entry) 2)
    (push node *points*)))
; white >> 0
; greay >> 1
; black >> 2


(defun check_cycles ()
  (setq *trees* nil)
  (setq *roots* nil)
  (init-nodes)
  (DFS nil)
  (init-nodes)
  (DFS t)
  (amalgamate (check_DFS_trees)))
; find the strongly connected components and reports an inconsistency
; if there are < cycles or <= cycles with inequalities


(defun init-nodes ()
  (setq *roots* nil)
  (dolist (pp *points*)
    (setf (time-node-c (get-tg pp)) 0)
    (if (and *trees* (cdr (assoc pp *trees*)))
        (setf (cdr (assoc pp *trees*)) nil))))

(defun get_edge (node1 node2)
  "return the edge starting from node1 and entering in node2 or the inequality 
edge. Nil if such a node does't exist."

  (let ((tg-entry (get-tg node1))
        (found nil))
    (do* ((succ (time-node-next tg-entry) (cdr succ))
          (edge (car succ) (car succ)))
        ((or (null succ) found))
      (cond ((eql (time-edge-to edge) node2)
             (setq found edge))))
    (if (not found)
        (do ((noteq-nodes (time-node-noteq tg-entry)
                          (cdr noteq-nodes)))
            ((or (null noteq-nodes) found))
          (cond ((eql (car noteq-nodes) node2)
                 (setq found (new-time-edge node1 node2 2))))))
    found))


(defun get_next_edge (node1 node2)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do* ((succ (time-node-next (get-tg node1)) (cdr succ))
        (edge (car succ) (car succ)))
      ((null succ) nil)
    (cond ((eql (time-edge-to edge) node2)
           (return-from get_next_edge edge)))))

(defun get_nextchain_edge (node1 node2)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do* ((succ (time-node-next-chain (get-tg node1)) (cdr succ))
        (edge (car succ) (car succ)))
      ((null succ) nil)
    (cond ((eql (time-edge-to edge) node2)
           (return-from get_nextchain_edge edge)))))

(defun get_=/=_edge (node1 node2)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((noteq-nodes (time-node-noteq (get-tg node1))
                    (cdr noteq-nodes)))
      ((null noteq-nodes) nil)
      (cond ((eql (car noteq-nodes) node2)
             (return-from get_=/=_edge (new-time-edge node1 node2 2))))))

(defun check_DFS_trees ()
  (let ((trees nil))
    (dolist (r *roots*)
      (let ((tree (read_tree r)))
        (if (null (car tree))
            (return-from check_DFS_trees (nreverse trees)))
        (push tree trees)))
    (nreverse trees)))

; checks if there are inconsistent cycles in the node of the depht search 
; trees

(defun read_tree (root)
  (let ((nodes (list root))
	(sons (cdr (assoc root *trees*))))
    (cond ((null sons)
           root)
          (t  (do ((edges sons (cdr edges)))
                  ((null edges) nodes)
                (setq nodes (nconc nodes
                                   (read_tree (time-edge-from
                                               (car edges))))))))
    nodes))
; read the elements of a search tree stored in 'trees

(defun intersect-rels (r1 r2)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (cond
   ((or (eql r1 2) (eql r2 2))
    nil)
   ((null r1) 
    r2)
   ((null r2)
    r1)
   ((and (zerop r1) (zerop r2))
    0)
   (t
    1)))

(defun change_edge_prev (node1 node2 node3)
  (let ((type-rel nil)
	(existing-edge (get_next_edge node3 node1))
	(move-edge (get_next_edge node2 node1))
	(new-edge nil))
    (if (not (null existing-edge))
	(setq type-rel (time-edge-kind existing-edge)))
    (setq new-edge (new-time-edge node3 node1 (intersect-rels
                                               type-rel (time-edge-kind move-edge))))
    (if (not (null existing-edge))
        (let ((tg-entry (get-tg node1)))
          (cond ((and (zerop type-rel)
                      (eql (time-edge-kind new-edge) 1))
                 (setf (time-node-prev tg-entry)
                   (remove_prev node1 `(,node3)))
                 (change-prev-edge node1 node2 new-edge))
                (t (setf (time-node-prev tg-entry)
                     (remove_prev node1 `(,node2))))))
      (change-prev-edge node1 node2 new-edge))
    (if (not (null existing-edge))
	(if (and (zerop type-rel)
		 (eql (time-edge-kind new-edge) 1))
	    (change-next-edge node3 node1 new-edge))
      (push new-edge (time-node-next (get-tg node3))))))
; change (node2 -> node1) in (node3 -> node1) in the prev list of node1
; (if such a link is not already present). Add the same link to the next list
; of node3


(defun change_edge_next (node1 node2 node3)
  (let ((new-edge nil)
	(type-rel nil)
	(existing-edge (get_next_edge node1 node3))
	(move-edge (get_next_edge node1 node2)))
    (if (not (null existing-edge))
	(setq type-rel (time-edge-kind existing-edge)))
    (setq new-edge (new-time-edge node1 node3 (intersect-rels
                                               type-rel (time-edge-kind move-edge))))
    (if (not (null existing-edge))
        (let ((tg-entry (get-tg node1)))
          (cond ((and (zerop type-rel)
                      (eql (time-edge-kind new-edge) 1))
                 (setf (time-node-next tg-entry)
                   (remove_next node1 `(,node3)))
                 (change-next-edge node1 node2 new-edge))
                (t (setf (time-node-next tg-entry)
                     (remove_next node1 `(,node2))))))
      (change-next-edge node1 node2 new-edge))
    (if (not (null existing-edge))
	(if (and (eql (time-edge-kind new-edge) 1)
		 (zerop type-rel))
	    (change-prev-edge node3 node1 new-edge))
      (push new-edge (time-node-prev (get-tg node3))))))
; change (node1 -> node2) in (node1 -> node3) in the next list of node1
; (if such a link is not already present). Add the same link to the prev
; list of node3


(defun change-next-edge (v1 v2 new-edge)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (setf (time-node-next (get-tg v1))
	(cons new-edge (remove_next v1 `(,v2)))))


(defun change-prev-edge (v1 v2 new-edge)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (setf (time-node-prev (get-tg v1))
	(cons new-edge (remove_prev v1 `(,v2)))))


(defun change-nextchain-edge (v1 v2 new-edge)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (setf (time-node-next-chain (get-tg v1))
	(cons new-edge (remove_nextchain v1 `(,v2)))))


(defun change-prevchain-edge (v1 v2 new-edge)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (setf (time-node-prev-chain (get-tg v1))
	(cons new-edge (remove_prevchain v1 `(,v2)))))


(defun change_edge_=/= (node1 node2 node3)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((tg-entry (get-tg node1)))
    (cond ((not (get_=/=_edge node1 node3))
           (setf (time-node-noteq tg-entry)
             (cons node3 (remove node2 (time-node-noteq tg-entry))))
           (push node1 (time-node-noteq (get-tg node3))))
          (t
           (setf (time-node-noteq tg-entry)
             (remove node2 (time-node-noteq tg-entry)))))))
; change (node2 =/= node1) in (node3 =/= node1) in the time-node-noteq of node1
; and of node3. 

(defun successor-nodes (next-edges)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (mapcan #'(lambda (edge) (list (time-edge-to edge)))
          next-edges))

(defun previous-nodes (prev-edges)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (mapcan #'(lambda (edge) (list (time-edge-from edge)))
          prev-edges))

(defun remove_nextchain (node l-nodes)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (remove-if #'(lambda (next) (member (time-edge-to next) l-nodes))
             (time-node-next-chain (get-tg node))))
  
(defun remove_prevchain (node l-nodes)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (remove-if #'(lambda (prev) (member (time-edge-from prev) l-nodes))
             (time-node-prev-chain (get-tg node))))

(defun remove_next (node l-nodes)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  
  (remove-if #'(lambda (next) (member (time-edge-to next) l-nodes))
             (time-node-next (get-tg node))))

(defun remove_prev (node l-nodes)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (remove-if #'(lambda (prev) (member (time-edge-from prev) l-nodes))
            (time-node-prev (get-tg node))))
  
(defun joint_nodes (SCC)
   (let ((class-node (car SCC)))
     (dolist (node (cdr SCC))
       (dolist (prev (previous-nodes (time-node-prev (get-tg node))))
           (cond ((not (member prev SCC))
                  (change_edge_next prev node class-node))))
         (dolist (next (successor-nodes (time-node-next (get-tg node))))
           (cond ((not (member next SCC))
                  (change_edge_prev next node class-node))))
         (dolist (ne-node (time-node-noteq (get-tg node)))
           (cond ((not (member ne-node SCC))
                  (change_edge_=/= ne-node node class-node))))
         (setf (get-tg node) nil)
         (setq *points* (delete node *points*)))
     (cond ((not (null (cdr SCC)))
            (let ((cn-entry (get-tg class-node)))
              (setf (time-node-next cn-entry)
                (remove_next class-node (cdr SCC)))
              (setf (time-node-prev cn-entry)
                (remove_prev class-node (cdr SCC))))))))
; change the timegraph collapsing the set of nodes SCC in a single node 
; (the first in SCC) that is chosen as the representative of the equivalence class.	
 
(defun amalgamate (DFS-trees)
  (cond ((not_consistent DFS-trees)
	 #+tg-debug (format t "~% INCONSISTENT SET OF RELATIONS")
         (error 'tg-inconsistancy))
	(t
	 (dolist (SCC DFS-trees)
	   (cond ((not (null (cdr SCC)))
		  (dolist (node (cdr SCC))
		    (setf (get-eq node)
			  (adjoin (car SCC) (get-eq node))))))
	   (joint_nodes SCC)))))
; collapse strong connected components in a single node (one of the SCC)
; the equality of all the other nodes is stored in the hash table equal-classes 


(defun not_consistent (SCC)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (notevery #'check_<_=/= SCC))

(defun check_<_=/= (stronglyCC)
  (let ((consistent t))
    (do ((nodes stronglyCC (cdr nodes)))
	((or (null nodes) (not consistent)) consistent)
      (do* ((nnodes stronglyCC (cdr nnodes))
	    (edge (get_edge (car nodes) (car nnodes))
		  (get_edge (car nodes) (car nnodes))))
	  ((or (null nnodes) (not consistent)))
	(if (not (null edge))
	    ;; if there is a < edge than there is an inconsistency
	    (if (eql (time-edge-kind edge) 1)
		(setq consistent nil)
	      (if (eql (time-edge-kind edge) 2)
		  ;; if there is a =/= edge there is an inconsistency iff the
		  ;; stronglyCC contains more than two nodes
		  (if (not (eql (length stronglyCC) 2))  
		      (setq consistent nil)))))))))
; check if there are < or =/= relations in a strongly connected component.
; If there is a < OR there is a =/= AND the strongly connected componet
; contains more than 2 nodes (the two nodes related by =/= form a "consistent"
; cycle), then there is an inconsistency.

(defun init_timegraph ()
  (setq *start-times* nil)
  (setq *number-chains* 0)
  (init-timegraph)
  (init-equal)
  (clear-chains))

(defun make_DAG ()
  (init_timegraph)
  (dolist (rel *point-relations*)
    (let ((firstpoint (get-tg (first rel)))
	  (secondpoint (get-tg (second rel))))
      (if (null firstpoint)
	  (if (eql (third rel) 2)
	      (update-tg (first rel)
                         :noteq (list (second rel)))
            (update-tg (first rel)
                       :next (list (new-time-edge (first rel)  (second rel) (third rel)))))
	(add_next_node firstpoint rel))
      (if (null secondpoint)
	  (if (eql (third rel) 2)
              (update-tg (second rel)
                         :noteq (list (first rel)))
            (update-tg (second rel)
                       :prev  (list (new-time-edge  (first rel) (second rel) (third rel)))))
	(add_prev_node secondpoint rel)))))

; Build the initial DAG from the point relations stored in *point-relations*
; *point-relations*: ((pt1 pt2 rel) (p3 pt4 rel)....)
; representaion of the rels:
; 0 for <=
; 1 for <
; 2 for =/=
; NB: each equality relation pt1=pt2 is has to be represented by a pair 
; of "less than or equal" relations: pt1 <= pt2, pt2 <= pt1

(defun add_next_node (firstpoint rel)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (when (not (equal (first rel) (second rel)))
    (case (third rel)
      (1
       (push  (new-time-edge  (first rel) (second rel) 1)
              (time-node-next firstpoint)))
      (0
       (setf (time-node-next firstpoint)
         (nconc (time-node-next firstpoint)
                (list (new-time-edge (first rel)  (second rel) 0)))))
      (t
       (setf (time-node-noteq firstpoint)
         (nconc (time-node-noteq firstpoint)
                (list (second rel))))))))
; The outgoing edges in each "next" list are ordered so that < edges
; precedes all <= edges 


(defun add_prev_node (secondpoint rel)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (when (not (equal (first rel) (second rel)))
    (case (third rel)
      (1
       (push (new-time-edge (first rel) (second rel) 1)
             (time-node-prev secondpoint)))
      (0
       (setf (time-node-prev secondpoint)
         (nconc (time-node-prev (get-tg (second rel)))
                (list (new-time-edge (first rel) (second rel) 0)))))
      (t
       (setf (time-node-noteq secondpoint)
         (nconc (time-node-noteq secondpoint)
                (list (first rel))))))))
; the ingoing edges in each "prev" list are ordered so that < edges
; precedes all <= edges 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        FUNCTIONS FOR THE CREATION OF THE CHAINS
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun make_chains (root)
  (init-nodes)
  (assign_ranks root)
  (init_ranks_hash)
  (clear-chains)
  (setq *points* (cons 0 *points*))
  (do ((rank *max-rank* (next-max-rank rank))
       (chain 1 (+ chain 1)))
      ((zerop rank) t)
    (setq *number-chains* chain)
    (m_chain chain rank))
  (make_chain_links)
  (clear-ranks))
; make the chains starting from the nodes with maximum rank
; the rank of each node is equal to 1000*(1+L), where L is
; the longest distance from the root node 


(defun next-max-rank (rank)
  (declare (optimize (speed 3) (safety 0) (debug 0))
           (type pseudotime rank))
  (cond
   ((zerop rank) 0)
   ((not (null (get-rank rank)))
    rank)
   (t (next-max-rank (- rank +pseudotime-step+)))))
; find the next greater rank to start building a new chain


(defun prev_chain_node (rank node)
  (declare (optimize (speed 3) (safety 0) (debug 0))
           (type pseudotime rank))
  (if (zerop rank) '(-1)
    (let ((l-prev (time-node-prev (get-tg node))))
      (if (null l-prev) '(-1)
	(let ((prev (find_prev rank l-prev)))
	  (cond ((not (null prev))
		 (setf (get-rank (time-node-rank (get-tg prev)))
		   (remove prev (get-rank (time-node-rank (get-tg prev)))))
		 `(,prev ,rank))
		(t '(-1))))))))
; find the next node (the previous temporally) of the chain


(defun find_prev (rank edges)
  (declare (type pseudotime rank))
  (if (not (null edges))
      (do* ((l edges (cdr l))
	    (kind (time-edge-kind (car l))
		  (if (not (null l)) (time-edge-kind (car l))))
	    (node (time-edge-from (car l))
		  (if (not (null l)) (time-edge-from (car l))))
	    (rank-n (time-node-rank (get-tg node))
		    (if (not (null node))
			(time-node-rank (get-tg node))))
	    (prev node)
            (prev-entry (get-tg prev) (get-tg prev))
	    (type-prev kind)
	    (found nil))
	  ((or found (null l))
	   (if (null (time-node-chain prev-entry)) prev nil))
        (declare (type pseudotime rank-n))
	(if (null (time-node-chain prev-entry))
	    (cond ((and (eql (time-node-rank prev-entry) rank)
			(zerop type-prev))
		   (setq found t))
		  ((> rank-n (time-node-rank prev-entry))
		   (setq prev node)
		   (setq type-prev kind))
		  ((and (= rank-n (time-node-rank prev-entry))
			(zerop type-prev)
			(not (zerop kind)))
		   (setq prev node)))))
    nil))
; find the first node in the list "l-nodes" that is a vertex that preceeds
; "node" in the timegraph. When there are several predecessors it prefers 
; < edges rather than <= edges.

 

(defun m_chain (chain rank)
  (declare (optimize (speed 3) (safety 0) (debug 0))
           (type pseudotime rank)
           (type chainnumber chain))

  (let ((nodes (get-rank rank)))
    (setf (time-node-chain (get-tg (car nodes))) chain)
    (setf (get-rank rank) (cdr nodes))
    (do ((prev (prev_chain_node rank (car nodes))
	       (prev_chain_node rank (car prev)))
	 (node (car nodes) (car prev)))
	((minusp (car prev)) (setf (get-chain chain)
			       (add_chains_entry node (car nodes))))
      (setf (time-node-chain (get-tg (car prev))) chain)
      (next_in_chain (car prev) node)
      (prev_in_chain node (car prev))
      (setq rank (- (second prev) +pseudotime-step+)))))

; make a chain starting with a node of rank equal or less than the
; rank specified in the the parameter list. The first node of each chains is stored 
; in the hash table "chains". The links corresponding to the next and the previous
; nodes in the same chain are moved as first items in the list of next and previuos
; nodes. 


(defun add_chains_entry (start end)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (vector start end nil nil nil nil))
; The entries in the hash table 'chains are arrays of 6 integers repressenting
; 0, the first node of the chain
; 1, the last node of the chain
; 2, the first cross node in the chain with and incoming cross edge
; 3, the last cross node in the chain with an incomind cross edge
; 4, the first node in the chain with an outcoming cross edge
; 5, the last node in the chain with an outcoming cross edge

(defun prev_in_chain (node prev)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((from (time-node-prev (get-tg node)))
	(new-prev nil))
    (do* ((nodes from (cdr nodes))
	  (pprev `(,(car nodes)) (cons (car nodes) pprev))
	  (found nil))
	((or found (null nodes)))
      (cond ((eql (time-edge-from (car nodes)) prev)
  	     (setq found t)
	     (setq new-prev (cons (car nodes) (append (cdr pprev) (cdr nodes)))))))
    (setf (time-node-prev (get-tg node)) new-prev)))
;    (format t "new prev nodes for ~A ~A ~%" node new-prev)))
; The link (prev -> node) is moded in the first position of the list ; (time-node-prev node)


(defun next_in_chain (node nnext)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((succ (time-node-next (get-tg node)))
	(new-next nil))
    (do* ((nodes succ (cdr nodes))
	  (next `(,(car nodes)) (cons (car nodes) next))
	  (found nil))
	((or found (null nodes)))
      (cond ((eql (time-edge-to (car nodes)) nnext)
	     (setq found t)
	     (setq new-next (cons (car nodes) (append (cdr next) (cdr nodes)))))))
    (setf (time-node-next (get-tg node)) new-next)))
;    (format t "new next nodes for ~A ~A ~%" node new-next)))
;; The link (node -> nnext) is moved in the first position of the list
;; (time-node-next node)

(defun assign_ranks (source)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (DAG_longest_paths source +pseudotime-step+))
;; Assign ranks to each node of the timegraph. The ranks are computed as
;; 1000*(1+L) L is the longest path from the "root" to the node. 


(defun DAG_longest_paths (source step)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (setf (time-node-rank (get-tg source)) step)
  (setq *max-rank* step)
  (DFS nil)
  (dolist (vertex (cons 0 *points*))
    (dolist (succ (time-node-next (get-tg vertex)))
      (relax vertex (time-edge-to succ) step))))
; Find the longest path of each node from the source node.
; See Cormen, Leiserson, Rivest "introduction to Algorithms"


(defun relax (p1 p2 step)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let* ((p2-entry (get-tg p2))
         (v1 (time-node-rank (get-tg p1)))
	 (v2 (time-node-rank p2-entry))
	 (rank-entry (+ v1 step)))
    (cond ((< v2 rank-entry)
	   (setf (time-node-rank p2-entry)
             rank-entry)
           (setq *max-rank* (max *max-rank* rank-entry))))))

(defun make_chain_links ()
  (macrolet ((add_next (next-list new-next)
               `(if (null ,next-list)
                    (setq ,next-list (list ,new-next))
                  (if (< (time-node-rank (get-tg (time-edge-to (car ,next-list))))
                         (time-node-rank (get-tg (time-edge-to ,new-next))))
                      (nconc ,next-list (list ,new-next))
                    (push ,new-next ,next-list))))
             (add_prev (prev-list new-prev)
               `(if (null ,prev-list)
                    (setq ,prev-list (list ,new-prev))
                 (if (> (time-node-rank (get-tg (time-edge-from (car ,prev-list))))
                        (time-node-rank (get-tg (time-edge-from ,new-prev))))
                     (nconc ,prev-list (list ,new-prev))
                   (push ,new-prev ,prev-list)))))

    (dolist (pp *points*)
      (let ((pp-entry (get-tg pp)))
        (do ((next-nodes (time-node-next pp-entry) (cdr next-nodes))
             (next-chains nil)
             (new-next-nodes nil))
            ((null next-nodes)
             (and (setf (time-node-next-chain pp-entry) next-chains)
                  (setf (time-node-next pp-entry) new-next-nodes)))
          (if (not (eql (time-node-chain pp-entry)
                        (time-node-chain (get-tg (time-edge-to (car next-nodes))))))
              (push (car next-nodes) next-chains)
            (add_next new-next-nodes (car next-nodes))))
        (move_first_<_next_transitive_edge pp)
        (do ((prev-nodes (time-node-prev pp-entry) (cdr prev-nodes))
             (prev-chains nil)
             (new-prev-nodes nil))
            ((null prev-nodes)
             (and (setf (time-node-prev-chain pp-entry) prev-chains)
                  (setf (time-node-prev pp-entry) new-prev-nodes)))
          (if (not (eql (time-node-chain pp-entry)
                        (time-node-chain (get-tg (time-edge-from (car prev-nodes))))))
              (setq prev-chains (nconc prev-chains (list (car prev-nodes))))
            (add_prev new-prev-nodes (car prev-nodes))))
        (move_first_<_prev_transitive_edge pp)
        (move_first_noteq_edges pp))))
  (make_in&out_chain_links)
  (make_first&lastin&out)
  (make_prev_less&next_greater))
; Move from the lists of next nodes and previous nodes to the next-chain
; and prev-chain slots all the links to/from nodes belonging to different chains.
; This is done only for <= edges and < edges. =/= edges that are cross
; chain edges are stored in the field time-node-noteq after the =/= edges
; that are "transitive" edges (see documentation of move_first_noteq_edges).
; Then it creates the prev-less and next-greater pointers for each node in 
; each chains. Finally it set the next-out-chain,next-in-chain and prev-in-chain,
; prev-out-chain.


(defun move_first_<_next_transitive_edge (v)
  (if (not (null v))
      (let* ((node (get-tg v))
	     (next (time-node-next node))
	     (n (car next)))
	(if (>= (length next) 2)
	    (do* ((trans (cdr next) (cdr trans))
		  (new-trans
		   (if (eql (time-edge-kind (car trans)) 1)
		       (list (car trans)) nil)))
		((null trans) (setf (time-node-next node)
				    (cons n new-trans)))
	      (if (not (null (cdr trans)))
		  (if (eql (time-edge-kind (cadr trans)) 1)
		      (if (null (car new-trans))
			  (setq  new-trans (list (cadr trans)))
			(if (> (time-node-rank (get-tg (time-edge-to (car new-trans))))
			       (time-node-rank (get-tg (time-edge-to (cadr trans)))))
			    (setq new-trans (list (cadr trans))))))))))))
; remove from v the forward transitive edges of kind <= and store in v the 
; forward transitive edge of kind < with the smallest rank (if any).

(defun move_first_<_prev_transitive_edge (v)
  (if (not (null v))
      (let* ((node (get-tg v))
	     (prev (time-node-prev node))
	     (n (car prev)))
	(if (>= (length prev) 2)
	    (do* ((trans (cdr prev) (cdr trans))
		  (new-trans
		   (if (eql (time-edge-kind (car trans)) 1)
		       (list (car trans)) nil)))
		((null trans) (setf (time-node-prev node)
				    (cons n new-trans)))
	      (if (not (null (cdr trans)))
		  (if (eql (time-edge-kind (cadr trans)) 1)
		      (if (null (car new-trans))
			  (setq new-trans (list (cadr trans)))
			(if (< (time-node-rank (get-tg (time-edge-from (car new-trans))))
			       (time-node-rank (get-tg (time-edge-from (cadr trans)))))
			    (setq new-trans (list (cadr trans))))))))))))
; remove from v the backward transitive edges of kind <= and store in v the 
; backward transitive edge of kind < with the greatest rank (if any).

(defun move_first_noteq_edges (v)
  (if (not (null v))
      (let* ((node (get-tg v))
	     (ne (time-node-noteq node)))
	(do ((trans ne (cdr trans))
             (=/=_trans_forward '())
             (=/=_trans_backward '())
             (=/=_cross '()))
	    ((null trans) (setf (time-node-noteq node)
				(list =/=_trans_forward
                                      =/=_trans_backward
                                      =/=_cross)))
	  (cond ((same_chain v (car trans))
		 (if (> (time-node-rank (get-tg (car trans)))
			(time-node-rank node))
		     (if (null =/=_trans_forward)
			 (setq =/=_trans_forward (car trans))
		       (if (> (time-node-rank (get-tg =/=_trans_forward))
			      (time-node-rank (get-tg (car trans))))
			   (setq =/=_trans_forward (car trans))))
		   (if (< (time-node-rank (get-tg (car trans)))
			  (time-node-rank node))	
		       (if (null =/=_trans_backward)
			   (setq =/=_trans_backward (car trans))
			 (if (< (time-node-rank (get-tg =/=_trans_backward))
				(time-node-rank (get-tg (car trans))))
			     (setq =/=_trans_backward (car trans)))))))
		(t (push (car trans) =/=_cross)))))))
;; The content of time-node-noteq field of each vertex v is modified:
;; it is a list of three elements, the first is tyhe the name of the first not equal vertex
;; on the same chain of v (NIL if it does not exist), the second is the name of the last 
;; not equal vertex on the same chain of v (NIL if it does not exist).  
;; The third element is the list of names of vertices that are not equal to v and that are
;; on a different chain.


(defun same_chain (v w)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (eql (time-node-chain (get-tg v))
       (time-node-chain (get-tg w))))

(defun make_first&lastin&out ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (dotimes (c *number-chains*)
    (cond ((not (equal (first-node-chain (+ 1 c))
		       (last-node-chain (+ 1 c))))
	   (set_first_in (+ 1 c))
	   (set_last_in (+ 1 c))
	   (set_first_out (+ 1 c))
	   (set_last_out (+ 1 c))))))
; if the chain has only one node, firstin=lastin=firstaout=lastout=nil


(defun set_first_in (chain)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((first (first-node-chain chain)
	      (if (out_connection_node (get-tg first))
		  (time-node-next-in-chain (get-tg first))
		(next_node_1 first)))
       (found nil))
      ((or (null first) found)
       (if found (setf (first_in chain) found) nil))
    (if (in_connection_node (get-tg first))
	(setq found first))))
;; compute the first vertex on the chain with a non nil incoming 
;; cross chain edge

(defun set_last_in (chain)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((last (last-node-chain chain)
	     (if (out_connection_node (get-tg last))
		 (time-node-prev-in-chain (get-tg last))
	       (prev_node_1 last)))
       (found nil))
      ((or (null last) found)
       (if found (setf (last_in chain) found) nil))
    (if (in_connection_node (get-tg last))
	(setq found last))))
;; compute the last vertex on the chain with a non nil incoming 
;; cross chain edge

(defun set_first_out (chain)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((first (first-node-chain chain)
	      (if (in_connection_node (get-tg first))
		  (time-node-next-out-chain (get-tg first))
		(next_node_1 first)))
       (found nil))
      ((or (null first) found)
       (if found (setf (first_out chain) found) nil))
    (if (out_connection_node (get-tg first))
	(setq found first))))
;; compute the first vertex on the chain with a non nil outcoming 
;; cross chain edge

(defun set_last_out (chain)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((last (last-node-chain chain)
	     (if (in_connection_node (get-tg last))
		 (time-node-prev-out-chain (get-tg last))
	       (prev_node_1 last)))
       (found nil))
      ((or (null last) found)
       (if found (setf (last_out chain) found) nil))
    (if (out_connection_node (get-tg last))
	(setq found last))))
;; compute the last vertex on the chain with a non nil outcoming 
;; cross chain edge

(defun make_in&out_chain_links ()
  (dolist (node *points*)
    (cond ((is_meta_node node)
	   (mlet (next-in next-out)
               (next_in&out_chain_node node)
             (mlet (prev-in prev-out)
                 (prev_in&out_chain_node node)
               (if (not (null next-in))
                   (setf (time-node-next-in-chain (get-tg node)) next-in))
               (if (not (null next-out))
                   (setf (time-node-next-out-chain (get-tg node)) next-out))
               (if (not (null prev-in))
                   (setf (time-node-prev-in-chain (get-tg node)) prev-in))
               (if (not (null prev-out))
                   (setf (time-node-prev-out-chain (get-tg node)) prev-out))))))))
;; compute for each cross connection vertexthe first and last next verteces 
;; on their chain with a non null incoming and outcoming cross chain edges.

(defun next_in&out_chain_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((next (next_node node)
	     (next_node (time-node-name next)))
       (in nil)
       (out nil))
      ((or (null next) (and in out)) (values in out))
    (if (not in)
	(if (in_connection_node next)
	    (setq in (time-node-name next))))
    (if (not out)
	(if (out_connection_node next)
	    (setq out (time-node-name next))))))

(defun next_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (progfoo (time-node-next (get-tg node))
    (when foo
      (return-from next_node (get-tg (time-edge-to (car foo)))))))

(defun prev_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (progfoo  (time-node-prev (get-tg node))
     (when foo
       (return-from prev_node (get-tg (time-edge-from (car foo)))))))

(defun prev_node_1 (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (progfoo (time-node-prev (get-tg node))
    (when foo
      (return-from prev_node_1 (time-edge-from (car foo))))))

(defun next_node_1 (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (progfoo (time-node-next (get-tg node))
     (when foo      
       (return-from next_node_1 (time-edge-to (car foo))))))

(defun prev_in&out_chain_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((prev (prev_node node)
	     (prev_node (time-node-name prev)))
       (in nil)
       (out nil))
      ((or (null prev) (and in out)) (values in out))
    (if (not in)
	(if (in_connection_node prev)
	    (setq in (time-node-name prev))))
    (if (not out)
	(if (out_connection_node prev)
	    (setq out (time-node-name prev))))))

(defun next-node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((next (car (time-node-next (get-tg node)))))
    (if (not (null next))
	(time-edge-to next)
      nil)))

(defun prev-node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((prev (car (time-node-prev (get-tg node)))))
    (if (not (null prev))
	(time-edge-from prev)
      nil)))

(defun out_connection_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (or (not (null (time-node-next-chain node)))
      (not (null (time-node-less-dsj-instances node)))))

(defun in_connection_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (or (not (null (time-node-prev-chain node)))
      (not (null (time-node-greater-dsj-instances node)))))


(defun is_meta_node (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (when node
    (let ((tg-entry (get-tg node)))
      (or (in_connection_node tg-entry)
          (out_connection_node tg-entry)))))


(defun next_node_in_chain (node kind)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (when node
    (let ((next (car (time-node-next (get-tg node)))))
      (when next
	(if (eql (time-edge-kind next) kind)
	    (time-edge-to next))))))


(defun prev_node_in_chain (node kind)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (when node
    (let* ((tg-entry  (get-tg node))
           (prev (car (time-node-prev tg-entry))))
      (when (and prev
                 (eql (time-node-chain (get-tg (time-edge-to prev)))
                      (time-node-chain tg-entry))
		 (eql (time-edge-kind (car (time-node-prev tg-entry)))
                      kind))
	    (time-edge-from prev)))))




(defun next_out_chain (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((tg-entry  (get-tg node)))
    (if (out_connection_node tg-entry) 
        node
    (do ((next (car (time-node-next tg-entry))
               (car (time-node-next
                     (get-tg (time-edge-to next)))))
         (cross nil))
	((or (null next) cross)
	 (if (not (null cross)) cross nil))
      (if (out_connection_node (get-tg (time-edge-to next)))
	  (setq cross (time-edge-to next)))))))
;; find the first vertex after node that is in its same chain 
;; and is a cross out node. If node is already such a vertex it
;; returns node, if such vertex does not exist, it returns nil.
      

(defun next_=/=_on_chain (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (car (time-node-noteq (get-tg v))))


(defun prev_=/=_on_chain (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (cadr (time-node-noteq (get-tg v))))


(defun cross_=/= (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (third (time-node-noteq (get-tg v))))

(defun earliest_time ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (update-tg 0 :next (next_start_time)))


(defun next_start_time ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let ((next nil)
	(st (find_earliest_times)))
    (dolist (succ st)
      (push (new-time-edge 0 succ 0) next)
      (setf (time-node-prev (get-tg succ))
	    (nconc (time-node-prev (get-tg succ))
		    (list (new-time-edge 0 succ 0)))))
    next))


(defun find_earliest_times ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (do ((vertices *points* (cdr vertices))
       (earliest nil))
      ((null vertices) earliest)
    (if (null (time-node-prev (get-tg (car vertices))))
	(setq earliest (cons (car vertices) earliest)))))
			       

(defun init_ranks_hash ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (reinit-ranks)
  (dolist (node *points*)
    (push node (get-rank (time-node-rank (get-tg node))))))
      

;; Build the Timegraph reporting information about the chains

(defun create-tg ()
  (setq *points* (delete 0 *points*))
  (make_dag)
  (handler-case (check_cycles)
    (tg-inconsistancy (condition) (format t "An inconsistancy was detected: ~S" condition) (return-from create-tg nil)))
  (earliest_time)
  (make_chains 0)
  (delete_redundant_neq)
  (=/=_diamonds_reduction))
;; Build the Timegraph without reporting any information


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  FUNCTIONS FOR THE COMPUTATION AND MANTAINANCE OF THE
;;  NEXTGREATER AND PREVLESS VALUES
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun make_prev_less&next_greater ()
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (dotimes (chain *number-chains*)
    (next_greater_in_chain (+ 1 chain)))
  (refine_next_greaters t)
  (dotimes (chain *number-chains*)
    (make_prev_less (+ 1 chain))))
;; Compute  the previuos less and next greater for all the verteces in 
;; all the chains

(defun next_greater_in_chain (chain)
  (do* ((start-chain (first-node-chain chain))
	(end-chain (last-node-chain chain))
	(node start-chain)
	(next-greater nil))
      ((eql node end-chain))
    (setq next-greater (next_node_in_chain_1 node 1 end-chain))
    (cond ((null next-greater)
	   (setq node end-chain))
	  (t (do ((n node (next-node n)))
                 ()
	       (setf (time-node-next-greater (get-tg n))
                 (cadr next-greater))
	       (if (eql (car next-greater) n)
                   (return)))
	     (setq node (next-node (car next-greater)))))))
;; Compute the next greater vertex for each verteces in a chain considering 
;; only the verteces in the single chain.


(defun make_prev_less (chain)
  (do* ((start-chain (last-node-chain chain))
	(end-chain (first-node-chain chain))
	(node start-chain)
	(prev-less nil))
      ((eql node end-chain) t)
    (setq prev-less (prev_node_in_chain_2 node end-chain))
    (cond ((null prev-less)
	   (setq node end-chain))
	  (t (do ((n node (prev-node n)))
                 ()
	       (setf (time-node-prev-less (get-tg n))
		 (cadr prev-less))
	       (if (eql n (car prev-less))
		   (return)))
	     (setq node (prev-node (car prev-less)))))))
;; Compute the previous less vertex for all the verteces in a chain


(defun next_node_in_chain_1 (node kind end)
  (unless (null node)
    (do ((next (car (time-node-next (get-tg node)))
	       (car (time-node-next
		     (get-tg (time-edge-to next))))))
	((null next) nil)
      (cond
       ((> (time-node-rank (get-tg (time-edge-to next)))
           (time-node-rank (get-tg end)))
        (return-from next_node_in_chain_1 nil))
       ((eql (time-edge-kind next) kind)
        (return-from next_node_in_chain_1 `(,(time-edge-from next) ,(time-edge-to next))))
       (t
        (let ((trans (earliest_vertex (next_=/=_on_chain (time-edge-from next))
                                      (next_<_trans (time-edge-from next))
                                      end)))
          (if (not (null trans))
              (let ((inter-trans (next_node_in_chain_1
                                  (time-edge-to next) kind
                                  trans)))
                (if (not (null inter-trans))
                    (return-from next_node_in_chain_1 inter-trans)
                  (return-from next_node_in_chain_1 `(,(time-edge-from next) ,trans))))))
        (if (eql (time-edge-to next) end)
            (return-from next_node_in_chain_1 nil)))))))
;; Find on a chain the first successor node v of "node" such that its relation with the 
;; first is of the kind specified by the parameter "kind" (0 for =>, 1 for >, 2 for =/=).
;; If this exists it returns the list (node v). Node and v can be also linked with "node" 
;; by a transitive edge or an inequality edge.


(defun prev_node_in_chain_2 (node end)
  (unless (null node)
    (do ((prev (car (time-node-prev (get-tg node)))
	       (car (time-node-prev
		     (get-tg (time-edge-from prev))))))
	((null prev) nil)
      (cond ((< (time-node-rank (get-tg (time-edge-from prev)))
                (time-node-rank (get-tg end)))
             (return-from prev_node_in_chain_2 nil))
	    ((eql (time-edge-kind prev) 1)
             (return-from prev_node_in_chain_2 `(,(time-edge-to prev) ,(time-edge-from prev))))
            (t
             (let ((cross-prev (latest_vertex (prev_=/=_on_chain (time-edge-to prev))
                                              (prev_<_trans (time-edge-to prev))
                                              (prev_less (time-edge-to prev))
                                              end)))
               (if (not (null cross-prev))
                   (let ((inter-cross (prev_node_in_chain_2
                                       (time-edge-from prev)
                                       cross-prev)))
                     (if (not (null inter-cross))
                         (return-from prev_node_in_chain_2 inter-cross)
                       (return-from prev_node_in_chain_2 `(,(time-edge-to prev) ,cross-prev))))))
             (if (eql (time-edge-from prev) end)
                 (return-from prev_node_in_chain_2 nil)))))))
;; Find the previous vertex (v) of a vertex ("node") that is in the same chain and is
;; strictly less. If this exists it returns the list ("node" v). Node and v can be
;; related by a transitive edges


(defun earliest_vertex (v1 v2 v3)
  (let*-non-null ((first-vertex (min_1 v1 v2)))
     (if (<= (time-node-rank (get-tg first-vertex))
             (time-node-rank (get-tg v3)))
         first-vertex)))
;; find the vertex with minimum rank between v1 and v2 and less than the
;; rank of v3. 

(defun latest_vertex (v1 v2 v3 v4)
  (let*-non-null ((last-vertex (max_1 v1 v2 v3)))
      (if (>= (time-node-rank (get-tg last-vertex))
              (time-node-rank (get-tg v4)))
          last-vertex)))
;; Find the vertex with the maximal rank between the vertices v1 v2 v3 that
;; is greater than the rank of v4
    
(defun min_1 (v1 v2)
  (cond
   ((and (null v1) (null v2))
    nil)
   ((null v1)
    (time-edge-to v2))
   ((null v2)
    v1)
   ((<  (time-node-rank (get-tg v1))
        (time-node-rank (get-tg (time-edge-to v2))))
    v1)
   (t
    (time-edge-to v2))))

(defun max_1 (v1 v2 v3)
  (let ((r-v1 (if (not (null v1))
		  (time-node-rank (get-tg v1))))
	(r-v2 (if (not (null v2))
		  (time-node-rank (get-tg (time-edge-from v2)))))
	(r-v3 (if (not (null v3))
		  (time-node-rank (get-tg v3)))))
    (cond
     ((and (null v1) (null v2) (null v3))
      nil)
     ((null v1)
      (cond
       ((null v2) v3)
       ((null v3) (time-edge-from v2))
       ((>= r-v2 r-v3)
        (time-edge-from v2))
       (t
        v3)))
     ((null v2)
      (cond
       ((null v3) v1)
       ((>= r-v1 r-v3) v1)
       (t
        v3)))
     ((null v3)
      (if (>= r-v1 r-v2)
          v1
        (time-edge-from v2)))
     (t
      (let ((r-max (max r-v1 r-v2 r-v3)))
        (cond ((equal r-max r-v1) v1)
              ((equal r-max r-v2) (time-edge-from v2))
              (t v3)))))))
	
(defun next_<_trans (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (cadr (time-node-next (get-tg node))))

(defun prev_<_trans (node)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (cadr (time-node-prev (get-tg node))))

(defun refine_next_greaters (first-time)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (dotimes (c *number-chains*)
    (refine_next_greaters_in_one_chain (+ 1 c) first-time)))

(defun pseudotime (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (if (null v) 
      +pseudotime-max+
    (pseudotime-i (get-tg v))))

(defun pseudotime-i (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (time-node-rank v))

;; updated 10-8-92
(defun next_greater_1 (v)      
  (if (and (not (null (time-node-next v)))
           (eql (time-edge-kind (car (time-node-next v))) 1))
      (time-edge-to (car (time-node-next v)))
    (time-node-next-greater v)))

(defun prev_less (v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (if (not (null v))
      (time-node-prev-less (get-tg v))
    nil))


(defun succ_cross_under_rmax (v rmax)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let (w)
    (dolist (succ (time-node-next-chain v))
      (if (<= (time-node-rank (get-tg (time-edge-to succ))) rmax)
	(push `(,(time-edge-to succ)
                ,(time-edge-kind succ)) w)))
    w))
;; Find the list of successor nodes of v in different chains whose rank is less than or 
;; equal than rmax. It these exist it returns the list (<node name> <relation> ...)
;; otherwise nil.


(defun add_w (w v nextout)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (if (and (not (null (next_greater_1 (get-tg v))))
           (<= (the pseudotime (pseudotime (next_greater_1 (get-tg v))))
               (the pseudotime (pseudotime nextout))))
      (push `(,nextout 1) w)
    (push `(,nextout 0) w)))
;; Add a new item to the list w that is used by refine_next_greaters_in_one_chain.
;; See documentation in the file refine-next-greater.doc

(defun add-open (open v)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (mapl #'(lambda (l) 
            (cond ((equal (car l) v) 
                   (return-from add-open open))
                  ((endp (cdr l))
                   (setf (cdr l) (list v))
                   (return-from add-open open))))
        open)
  v)

(defun update_prev_nextgreater (newval vstart chain)
  (cond ((not (null vstart))
	 (let* ((previn (time-node-prev-out-chain vstart))
		(end-update (if (null previn) (first-node-chain chain)
			      previn)))
	   (do ((prev (prev_node (time-node-name vstart))
		       (prev_node (time-node-name prev))))
	       ((null prev))
	     (cond ((< (the pseudotime (pseudotime newval))
		       (the pseudotime (pseudotime (time-node-next-greater prev))))
		    (setf (time-node-next-greater prev) newval))
		   (t 
		    (setq prev (get-tg end-update))
		    (setq newval nil)))
	     (if (eql (time-node-name prev) end-update)
                 (return-from update_prev_nextgreater newval)))
	   newval))))
; Update the predecessors of a vertex whose "nextgreater" has been refined
; See file refine-next-greater.doc.
; The function returns nil if there is a vertex preceeding vstart and with
; a nextgreater preceeding newval, otherwise it returns newval.


(defun update_prev_nextgreater_1 (newval v)
  (do ((prev (prev_node (time-node-name (get-tg v)))
             (prev_node (time-node-name prev))))
      ((null prev))
    (cond
     ((or (null (time-node-next-greater prev))
          (< (the pseudotime (pseudotime newval))
             (the pseudotime (pseudotime (time-node-next-greater prev)))))
      (setf (time-node-next-greater prev) newval))
     (t
      (return-from update_prev_nextgreater_1 t)))))
; Update the nextgreaters of the chain-ancestors of a vertex 
; whose nextgreater has been refined.


(defun update_next_prevless (newval v)
  (do ((next (next_node newval)
             (next_node (time-node-name next))))
      ((null next))
    (if (or (null (time-node-prev-less next))
            (> (the pseudotime (pseudotime v))
               (the pseudotime (pseudotime (time-node-prev-less next)))))
	(setf (time-node-prev-less next) v)
      (return-from update_next_prevless t))))
; Update the prev-less  of the chain-successors of a vertex 
; whose next-greater has been refined.


(defun find_vstart (chain)
  (let ((vstart (last_out chain)))
    (cond ((null vstart) nil)
	  (t (do* ((vv vstart (time-node-prev-out-chain (get-tg vv)))
		   (nextin (if (not (null vv))
			       (time-node-next-in-chain (get-tg vv))
			     nil)
			   (if (not (null vv))
			       (time-node-next-in-chain (get-tg vv))
			     nil)))
		 ((or (null vv) (not (null nextin)))
		  (if nextin vv nil)))))))
;; Find the first node from which start the refining of the next greaters in
;; a chain. See the documentation in the file refine-next-greater.doc


(defun min_rank (vstart rmax)
  (declare (optimize (speed 3) (safety 0) (debug 0))
           (type pseudotime rmax))
  (if (null vstart) rmax
    (let ((ng (time-node-next-greater vstart))
          (mm rmax))
      (if (not (null ng))
	  (setq mm (min rmax (time-node-rank (get-tg ng)))))
      mm)))

(defun refine_next_greaters_in_one_chain (chain first-time)
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (let*-non-null ((vstart (find_vstart chain)))
    (rng-1 vstart chain first-time)))

(defun rng-1 (vstart chain first-time)
  (let ((rmax 0)
        (maxless nil)
        (maxleq nil)
	(visited nil))
    (flet ((maxless (v)
             (cdr (assoc v maxless)))
           (maxleq (v)
             (cdr (assoc v maxleq)))
           (set-maxless (v psuedo)
             (update-alist v psuedo maxless))
           (set-maxleq (v psuedo)
             (update-alist v psuedo maxleq)))

      (setq vstart (get-tg vstart))
      (setq rmax (time-node-rank (get-tg (last_in chain))))
      (do* ((vstr vstart 
                  (if (not (null (time-node-prev-out-chain vstr)))
                      (get-tg (time-node-prev-out-chain vstr))
                    nil))
            (rmax (min_rank vstr rmax)
                  (min_rank vstr rmax))
            (tt (the pseudotime (pseudotime (time-node-name vstart))))
            (openlist (list (time-node-name vstart))
                      (if (not (null vstr))
                          (list (time-node-name vstr)) nil))
            (newval nil)
            (new nil))
          ((null vstr))
        (declare (type pseudotime rmax))
        (when (and (not (null (time-node-next-in-chain vstr)))
                   (or (null (time-node-next-greater vstr))
                       (> (the pseudotime (pseudotime (time-node-next-greater vstr)))
                          (the pseudotime (pseudotime (time-node-next-in-chain vstr))))))
          (do* ((open openlist (cdr open))
                (v nil)
                (v-entry nil))
              ((null open))
            (setq v (car open))
            (setq v-entry (get-tg v))
            (cond ((and (eql (time-node-chain v-entry) chain)
                        (not (eql v (time-node-name vstr)))
                        (< (the pseudotime (pseudotime v))
                           (the pseudotime (pseudotime (next_greater_1 vstr))))
                        (not (null (maxless v))))
                   (setf (time-node-next-greater vstr) v)
                   (setf (time-node-prev-less v-entry) (time-node-name vstr))
                   (setq newval v)
                   (setq new t))
                  (t
                   (let ((w (succ_cross_under_rmax v-entry rmax))
                         (nextout (time-node-next-out-chain v-entry)))
                     (if (and (not (eql (time-node-chain v-entry) chain))
                              (not (null nextout))
                              (< (time-node-rank (get-tg nextout))
                                 rmax))
                         (setq w (add_w w v nextout)))
                     (dolist (item w)
                       (let ((ww (car item)))
                         (cond
                          ((and (or (eql (second item) 1)
                                    (not (null (maxless v))))
                                (or (null (maxless ww))
                                    (eql (maxleq ww) tt)))
                           (set-maxless ww tt)
                           (if (eql (maxleq ww) tt)
                               (set-maxleq ww nil)))
                          ((and (null (maxless ww))
                                (null (maxleq ww)))
                           (set-maxleq ww tt)))
                         (setq open (add-open open ww))
                         (pushnew ww visited :test #'equal))))))))
        (cond ((not (null newval))
               (if (and new (not first-time))
                   (update_next_prevless newval (time-node-name vstr)))
               ;; if is not the first time that  next-greaters are computed
               ;; we need to update not only the nextgreater of the  
               ;; chain-ancestors of vstr but also the prevless 
               ;; of the chain-successors of vstart belonging to 
               ;; the sub-chain from newval to oldval (the previous 
               ;; nextgreater of vstart).
               (setq newval (update_prev_nextgreater newval vstr chain))
                                ;		    (format t "newval ~A ~%" newval)
               (if (not (null newval)) (setq new nil))))))))
;; See the documentation in the file refine-next-greater.doc



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;                            QUERY FUNCTIONS
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun query_equal (pt1 pt2)
  (if (not (=_v_w pt1 pt2))
      (let ((rr (relation pt1 pt2)))
	(cond ((or (eq (second rr) :<=)
		   (null rr))
	       :UNKNOWN)
	      (t nil)))
    t))


(defun query_not_equal (pt1 pt2)
  (if (not (=/=_v_w pt1 pt2))
      (let ((rr (relation pt1 pt2)))
	(cond ((eq (second rr) :<)
	       t)
	      ((or (eq (second rr) :<=)
		   (null rr))
	       :UNKNOWN)
	      (t nil)))
    t))

(defun leq (v w)
  (let ((w-entry (get-eq w))
        v-entry)
    (cond
     ((or (member v w-entry)
	  (member w (setq v-entry (get-eq v)))))
     ((or (null v-entry)
          (null w-entry))
      nil)
     (t
      (let ((lq (leq_path v w)))
	(if (null lq) nil
	  (if (eq lq :unknown) :UNKNOWN
	    t)))))))
;; check wheter v <= w


(defun less (v w)
  (if (or (null (get-tg v))
	  (null (get-tg w))) nil
    (let ((ls (less_path v w)))
      (if (or (null ls) (eq ls :unknown))
	  (let ((lq (leq_path w v)))
	    (if (and (not (null lq))
		     (not (equal lq :unknown)))
		nil
	      :unknown))
	t))))
;; check wheter v < w


(defun tg-neq (v w)
  (let*-non-null ((v-entry (get-tg v))
                  (w-entry (get-tg w)))
    (if (not (eql (time-node-chain v-entry)
                  (time-node-chain w-entry)))
	(if (not_= v w) t
	  (if (less_path v w) t
	    (if (less_path w v) t
	      :unknown))))))
;; check wheter v =/= w


(defun leq_chainpath (v w)
  (let ((v-entry (get-tg v))
        (w-entry (get-tg w)))
    (if (eql (time-node-chain v-entry)
             (time-node-chain w-entry))
        (<= (the pseudotime (pseudotime-i v-entry))
            (the pseudotime (pseudotime-i w-entry))))))
;; v and w are assumed to be on the same chain


(defun not_= (v w)
  (if (member w (cross_=/= v))
      t nil))
;; check whether there is a  edge between v and w


(defun less_chainpath (v w)
  (when (and v w)
    (let ((v-entry  (get-tg v))
          (w-entry  (get-tg w)))
      (if (eql (time-node-chain v-entry)
               (time-node-chain w-entry))
          (let*-non-null ((ng (time-node-next-greater v-entry)))
             (<= (the pseudotime (pseudotime ng)) (the pseudotime (pseudotime-i w-entry))))))))
;; v,w are assumed to be on the same chain


(defun chainpath (v w)
  (let ((ng (time-node-next-greater (get-tg v))))
    (if (not (null ng)) 
	(if (<= (the pseudotime (pseudotime ng)) (the pseudotime (pseudotime w)))
	    :<
          :<=)
      :<=)))
;; return the relation (, or <=) between two verterces v,w on the 
;; same chain the rank of v is assumed less than the rank of w.


(defun to (edges)
  (let ((l nil))
    (dolist (ed edges)
      (setq l (cons (time-edge-to ed) l)))
    l))


(defun leq_path (v w)
  (cond ((and (not (null v)) (not (null w)))
	 (if (same_chain v w)
	     (leq_chainpath v w)
	   (let* ((x (next_out_chain v))
		  (rmax (time-node-rank (get-tg w)))
		  (found nil)
                  (x-entry))
	     (cond ((or (null x)
			(>= (time-node-rank (setq x-entry (get-tg x))) rmax)) nil)
		   (t
		    ;;modified Jan 15 1994
                    (update-alist x t *old*)
		    (do* ((open (list x))
			  (x (car open) (car open))
			  (y nil)
			  (Sl nil))
			((or (null open) found))
		      (setq y (time-node-next-out-chain x-entry))
		      (setq Sl (to (time-node-next-chain x-entry)))
		      (setq open (cdr open))
		      (if (not (null y))
			  (setq Sl (cons y Sl)))
		      ;		   (format t "Sl: ~a x ~A ~%" Sl x)
		      (do* ((ss Sl (cdr ss))
			    (yy (car ss) (car ss)))
			  ((or (null ss)) found)
			;		     (format t "yy ~A ~%" yy)
			(cond ((or (eql yy w)
				   (and (same_chain yy w)
					(<= (the pseudotime (pseudotime yy))
                                            (the pseudotime (pseudotime w)))))
                               (setq *old* nil)
			       (setq found t))
			      (t
			       (cond ((and (<  (time-node-rank
						(get-tg yy)) rmax)
					   (null (cdr (assoc yy *old*))))
				      (push yy open)
                                      (update-alist yy t *old*)))))))
		    (cond (found t)
			  (t (setq *old* nil))))))))
	(t 
	 (setq *old* nil)
	 :unknown)))
;; Predicate which is true if an only if there is a "<= path" 
;; from v to w in the ranked timegraph

(defun less_path (v w)
  (if (and (not (null v)) (not (null w)))
      (if (same_chain v w)
	  (less_chainpath v w)
	(let* ((x (next_out_chain v))
	       (rmax (time-node-rank (get-tg w)))
	       (found nil)
	       (visited (list x)))
	  (cond ((or (null x)
		     (>= (time-node-rank (get-tg x)) rmax)) nil)
		(t
		 (if (less_chainpath v x)
                     (update-alist x 1 *strict*)
                   (update-alist x 0 *strict*))
		 (do* ((open (list x))
		       (x (car open) (car open))
		       (y nil) 
		       (Sl nil))
		     ((or (null open) found))
		   (setq y (time-node-next-out-chain (get-tg x)))
		   (setq Sl (create_ordered_successors x rmax))
		   (setq open (cdr open))
		   (if (and (not (null y))
			    (not (eql (cdr (assoc y *strict*)) 1))
			    (< (time-node-rank (get-tg y)) rmax))
		       (if (and (zerop (cdr (assoc x *strict*)))
				(null (cdr (assoc y *strict*))))
			   (setq Sl (cons `(,y 0) Sl))
			 (if (or (eql (cdr (assoc x *strict*)) 1)
				 (less_chainpath x y))
			     (setq Sl (append Sl `((,y 1)))))))
;		   (format t "Sl ~a ~%" Sl)
		   (do* ((SS Sl (cdr SS))
			 (y (caar SS) (caar SS))
			 (r (cadar SS) (cadar SS)))
		       ((or (null SS) found))
		     (setq r (logior r (cdr (assoc x *strict*))))
                     (update-alist y r *strict*)
		     (cond ((or (and (eql y w) (eql r 1))
				(or (less_chainpath y w)
				    (and (eql r 1)
					 (leq_chainpath y w))))
			    (setq found r)
			    (dolist (z (cons y visited))
                              (update-alist z nil *strict*)))
			   (t (setq visited (cons y visited))
			      (setq open (cons y open))))))
		 (cond (found t)
		       (t (dolist (z visited)
                            (update-alist z nil *strict*))
			  nil))))))
    'unknown))
;; Boolean function which is true if and only if there is a "< path"
;; from v to w in the ranked timegraph

(defun relation (v w)
  (cond ((or (and (null (get-tg v))
		  (null (get-eq v)))
	     (and (null (get-tg w))
		  (null (get-eq w))))
	 nil)
	((or (member v (get-eq w))
	     (member w (get-eq v)))
	 `(,v := ,w))
	(t
	 (let ((v1 (get-tg v))
	       (w1 (get-tg w)))
	   (if (null v1)
	       (setq v1 (car (get-eq v)))
	     (setq v1 v))
	   (if (null w1)
	       (setq w1 (car (get-eq w)))
	     (setq w1 w))
	   (if (< (time-node-rank (get-tg v1))
		  (time-node-rank (get-tg w1)))
	       (let ((rel (path v1 w1 t)))
		 (if (not (null rel))
		     `(,v ,rel ,w)
		   nil))
	     (let ((rel (path w1 v1 t)))
	       (if (not (null rel))
		   `(,w ,rel ,v)
		 nil)))))))
;; determine the relation between two verteces v and w, in a
;; ranked timegraph, i.e. <, <=, >, >=, =/=, or nil (unknown)  


(defun relation_1 (v w)
  (if (< (time-node-rank (get-tg v))
	 (time-node-rank (get-tg w)))
      (let ((rel (path v w nil)))
	  (if (not (null rel))
	      `(,v ,rel ,w)
	    nil))
      (let ((rel (path w v nil)))
	(if (not (null rel))
	    `(,w ,rel ,v)
	  nil))))
;; determine the relation between two verteces v and w, in a
;; ranked timegraph, without considering =/= relations.


(defun =_v_w (v w)
  (cond ((or (and (null (get-tg v))
		  (null (get-eq v)))
	     (and (null (get-tg w))
		  (null (get-eq w))))
	 nil)
	((or (member v (get-eq w))
	     (member w (get-eq v)))
	 t)
	((and (null (get-tg v))
	      (null (get-tg w)))
	 (eql (car (get-eq v))
	     (car (get-eq w))))
	(t nil)))


(defun =/=_v_w (v w)
 (let ((rel (relation v w)))
   (if (equal (cadr rel) :=/=)
       t
     nil)))


(defun path (v w with-noteq)
  (setq *strict* nil)
  (cond
   ((eql v w) 
    :=)
   ((and (null v) (null w))
    nil)
   ((same_chain v w)  ;; the rank of v is < rank of w
    (chainpath v w))
   (t
    (if with-noteq
        (cond
         ((member v (cross_=/= w))
          (return-from path :=/=))
         ((eql (time-node-rank (get-tg v))
               (time-node-rank (get-tg w)))
          (return-from path nil))))
    (let* ((rmax (time-node-rank (get-tg w)))
           (xx (list_next_out_chain v rmax))
           (found nil))
      (cond ((null xx) 
             nil)
            (t
             (do* ((open xx)
                   (x (car open) (car open))
                   (next-out nil) 
                   (cross-s nil)
                   (Sl nil))
                 ((or (null open) found))
               (setq next-out (time-node-next-out-chain (get-tg x)))
               (setq Sl (create_ordered_successors x rmax))
               (setq open (cdr open))
               (if (not (null next-out))
                   (cond ((and (same_chain x w)
                               (or (eql (cdr (assoc x *strict*)) 1)
                                   (less_chainpath x w)))
                          (setq found :<)
                          (setq *strict* nil))
                         (t
                          (setq cross-s (legal-cross-successors x x rmax))
                          (setq open (nconc open cross-s)))))
               (do* ((SS Sl (cdr SS))
                     (y (caar SS) (caar SS))
                     (r (cadar SS) (cadar SS)))
                   ((or (null SS) found))
                 (setq r (logior r (cdr (assoc x *strict*))))
                 (update-alist y r *strict*)
                 (cond ((or (and (eql y w) (eql r 1))
                            (or (less_chainpath y w)
                                (and (eql r 1)
                                     (leq_chainpath y w))))
                        (setq found :<))
                       ;; the following 2 lines have been added
                       ;; in date 19-8-1992
                       ((and (zerop r) (leq_chainpath y w))
                        (update-alist w 0 *strict*))
                       (t
                        (setq open (cons y open))))))
             (cond (found :<)
                   (t (if (and (not (null (cdr (assoc w *strict*))))
                               (zerop (cdr (assoc w *strict*))))
                          (setq found :<=))
                      (setq *strict* nil)
                      found))))))))
;; returns < if there exists at least one "< path" from v to w,
;;         <= if all the paths from v to w are "<= paths",
;;         =/= if there exists a "=/= cross edge" between v an w
;;         and the function is called with the parameter with-noteq true,
;;         nil if there isn't any path between v and w



(defun legal-cross-successors (start v rmax)
  (let ((next-cross (time-node-next-out-chain (get-tg v)))
        list-next-cross)
    (while next-cross
       (let ((cross-tg (get-tg next-cross)))
         (cond
          ((> (time-node-rank cross-tg) rmax)
           (return-from legal-cross-successors list-next-cross))
          ((and (null (cdr (assoc next-cross *strict*)))
                (and (not (less_chainpath start next-cross))
                     (not (eql (cdr (assoc start *strict*)) 1))))
           (update-alist next-cross 0 *strict*)
           (setq list-next-cross (nconc list-next-cross (list next-cross))))
          ((and (not (eql (cdr (assoc next-cross *strict*)) 1))
                (or (less_chainpath start next-cross)
                    (eql (cdr (assoc start *strict*)) 1)))
           (update-alist next-cross 1 *strict*)
           (setq list-next-cross (nconc list-next-cross (list next-cross)))))
         (setq next-cross (time-node-next-out-chain cross-tg))))
    list-next-cross))
;; give the list of legal successor vertices of vstart along edges with <= label and
;; with rank not greater than the rank of vstart. 


(defun list_next_out_chain (node rmax)
  (let ((first-out (next_out_chain node))
	(list-next nil))
    (cond ((not (null first-out))
	   (setq list-next (cons first-out (legal-cross-successors node first-out rmax)))
	   (if (less_chainpath node first-out)
               (update-alist first-out 1 *strict*)
             (update-alist first-out 0 *strict*)))
	  (t nil))
    list-next))
    
(defun create_ordered_successors (v rmax)
  (let ((outchain (time-node-next-chain (get-tg v)))
	(result nil))
    (do ((ed outchain (cdr ed)))
	((null ed))
      (if (<= (time-node-rank (get-tg (time-edge-to (car ed))))
		   rmax)
	  (if (zerop (time-edge-kind (car ed)))
	      (if (or (null (cdr (assoc (time-edge-to (car ed)) *strict*)))
		      (and (zerop (cdr (assoc (time-edge-to (car ed)) *strict*)))
			   (eql (cdr (assoc v *strict*)) 1)))
		  (setq result (nconc result `((,(time-edge-to (car ed)) 0)))))
	    (if (not (eql (cdr (assoc (time-edge-to (car ed)) *strict*)) 1))
		(setq result (cons `(,(time-edge-to (car ed)) 1) result))))))
    result))
;; create a partially ordered list of successor nodes to be examined
;; during the path searches. The first edges in the list are the "< edges", 
;; then "<= edges". All the edges connect to vertices whose rank is less than 
;; rmax.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;           FUNCTIONS FOR THE NOT EQUAL RELATIONS
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun =/=_diamonds_reduction ()
 (let ((processed (make-hash-table :size (length *points*)))
       (new-refining nil)
       (=/=_cross nil)
       (ncd nil)
       (nca nil))
   (dolist (vv *points*) (setf (time-node-c (get-tg vv)) 0))
   (dolist (v *points*)
     (setq =/=_cross (cross_=/= v))
     (dolist (=/=_vertex =/=_cross)
       (cond ((not (gethash =/=_vertex processed))
	      (setq ncd (nearest_common_descendants v =/=_vertex))
	      (if (not (null ncd))
		  (setq nca (nearest_common_ancestors v =/=_vertex)))
	      (dolist (vv ncd) (setf (time-node-c (get-tg vv)) 0))
	      (dolist (vv nca) (setf (time-node-c (get-tg vv)) 0))
	      (when (and (not (null ncd)) (not (null nca)))
		(setq new-refining t)
		(add_<_rels ncd nca))
	      ;; new-refining is true when is necessary to run the 
	      ;; algorithm for refinng the next-greaters and prev-lesses. 
	      ;; This happens when the new < edge that is added is a cross 
	      ;; chain edge 
	      ;; need only for test:
	      ;; (setq n-ncd*nca (+ n-ncd*nca (* (length ncd) (length nca))))
	      (setf (gethash v processed) t)))))
   (cond (new-refining
	  (make_first&lastin&out)
	  (refine_next_greaters nil))
	 (t nil))))
; Reduce the "smallest" =/= diamonds. Two vetices s,t such that s =/= t
; the smallest =/= diamonds are those obtained by considering as third 
; vertex all the nearest common descendants of s and t, and ad fourth
; vertex all the nearest common ancestor of s and t.
; s =/= t, v nearest common ancestor of s and t, w nearest common 
; descendant of s and t, if v and w are on the same chain and is not
; the case that the timegraph entails v < w, then the nextgreaters and
; prevless of the verices on the chain are updated.
; If v and w are on a different chain a new edge with label '<' can be
; added to the timegraph. If any nextgreater has been chainged or at
; least one new edge has been added to the timegraph then the nextgreaters
; (and prevless) are refined.

;; New (2nd March)
(defun add_<_rels (ncd nca)
  (do ((set1 ncd (cdr set1))
       (refining nil))
      ((null set1) refining)
    (do ((set2 nca (cdr set2)))
	((null set2))
      (cond ((same_chain (car set1) (car set2))
	     (cond ((or (null (time-node-next-greater
			       (get-tg (car set2))))
			(> (the pseudotime (pseudotime (time-node-next-greater
                                                        (get-tg (car set2)))))
			   (the pseudotime (pseudotime (car set1)))))
		    (setf (time-node-next-greater
			   (get-tg (car set2))) (car set1))
		    (setf (time-node-prev-less
			   (get-tg (car set1))) (car set2))
		    (update_prev_nextgreater_1
		     (car set1) (car set2))
		    (update_next_prevless
		     (car set1) (car set2))
		    (setq refining t))))
	    (t (if (not refining)
		   (setq refining t))
	       ; add a new cross chain edge with label '<'
	       ; from a nearest common ancestor to a nearest common
	       ; descendant.
	       (cond (*check-less-for-each-pair-nca-ncd*
		    
		      (when (not (less_path (car set2) (car set1)))
			(setf (time-node-next-chain (get-tg (car set2)))
			      (add_<_cross_checking_already_existing
			       (car set2) (car set1)))
			(setf (time-node-prev-chain (get-tg (car set1)))
			      (add_>_cross_checking_already_existing
			       (car set2) (car set1)))))
		     (t
		   
		      (setf (time-node-next-chain (get-tg (car set2)))
			    (add_<_cross_checking_already_existing  (car set2) (car set1)))
		      (setf (time-node-prev-chain (get-tg (car set1)))
			    (add_>_cross_checking_already_existing  (car set2) (car set1))))))))))
; For each pair of vertices (v,w) with v in nca and w in ncd, if
; v and w are on the same chain and the timegraph does not entail
; v < w then the nextgreater of v is changed to w and the nextgreater,
; prevless of the other vertices on the chain are updated. If v and w
; are on different chain v < w is checked only if the global variable
; *check-less-for-each-pair-nca-ncd* is true. In this case a new edge
; of kind 1 from v to w is added to the Timegraph only if v < w is
; not already entailed by the graph. If *check-less-for-each-pair-nca-ncd*
; is nil the new edge is added without testing v < w.
; The function returns true if at least one nextgreater has been 
; updated or at least one new cross chain edge has been added in the
; Timegraph.


(defun add_<_cross_checking_already_existing  (node1 node2)
  (let ((found nil)
	(new-succ nil))
    (do* ((succ (time-node-next-chain (get-tg node1)) (cdr succ))
	  (edge (car succ) (car succ)))
	((null succ))
      (cond ((and (not found) (eql (time-edge-to edge) node2))
	     (if (not (eql (time-edge-kind edge) 1))
		 (setq new-succ
		       (cons (new-time-edge node1 node2 1)
			     new-succ))
	       (setq new-succ (cons edge new-succ)))
	     (setq found t))
	    (t
	     (setq new-succ (cons edge new-succ)))))
    (when (not found)
      (setq new-succ (cons (new-time-edge node1 node2 1)
			   new-succ)))
    new-succ))



(defun add_>_cross_checking_already_existing  (node1 node2)
  (let ((found nil)
	(new-prev nil))
    (do* ((prev (time-node-prev-chain (get-tg node2)) (cdr prev))
	  (edge (car prev) (car prev)))
	((null prev))
      (cond ((and (not found) (eql (time-edge-from edge) node1))
	     (if (not (eql (time-edge-kind edge) 1))
		 (setq new-prev
		       (cons (new-time-edge node1 node2 1)
			     new-prev))
	       (setq new-prev (cons edge new-prev)))
	     (setq found t))
	    (t
	     (setq new-prev (cons edge new-prev)))))
    (when (not found)
      (setq new-prev (cons (new-time-edge node1 node2 1)
			   new-prev)))
    new-prev))


(defun nearest_common_ancestors (v1 v2)
  (search2_NCA v1 v2))
; find the nearest common ancestors for two vertices on DIFFERENT 
; chains. when this search is used for "=/= diamond" we consider only
; search in the metagraph (=/= edges are only cross chain links).


(defun nearest_common_descendants (v1 v2)
  (search2_NCD v1 v2))
; find the nearest common descendants for two vertices on DIFFERENT 
; chains. when this search is used for "=/= diamond" we consider only
; search in the metagraph (=/= edges are only cross chain links).

(defun search2_NCD (v1 v2)
  (do* ((in-OPEN (init_open_NCD v1 v2))
	(n 2)
	(codes (vector 1 1 0 0))
	(RESULT nil)
	(OPEN in-open (cdr OPEN))
	(next (car OPEN) (car OPEN)))
      ((null OPEN) RESULT)
    (when (eql (second next) 3)
      (when (not (eql (time-node-c (get-tg (car next))) 1))
	(setq RESULT (cons (car next) RESULT))
	;; set the node as ncd
	(setf (time-node-c (get-tg (car next))) 1)))
    (dolist (meta-node (succ_<=_meta_nodes (car next)))
      (incf n)
      (let* ((item (member_1 meta-node OPEN))
	     (new-code (combine_code (second next) (second item))))
	(cond ((not (null item))
	       (setq OPEN (substitute `(,(car item) ,new-code)
				      item OPEN :test #'equal))
	       (incf (svref codes (- new-code 1)))
	       (decf (svref codes (- (second item) 1))))
	      (t (setq OPEN (insert_in_open_NCD
			     meta-node (open_code (second next)) OPEN))
		 (incf (svref codes (- (open_code (second next)) 1)))))))
    (if (stop_search n codes)
        (return-from search2_NCD RESULT))))
; search for NCD specialized for =/= diamonds 

(defun search2_NCA (v1 v2)
  (do* ((in-OPEN (init_open_NCA v1 v2))
	(n 2)
	(codes (vector 1 1 0 0))
	(RESULT nil)
	(OPEN in-open (cdr OPEN))
	(prev (car OPEN) (car OPEN)))
      ((null OPEN) RESULT)
    (when (eql (second prev) 3)
      (when (not (eql (time-node-c (get-tg (car prev))) 2))
       (setq RESULT (cons (car prev) RESULT))
       (setf (time-node-c (get-tg (car prev))) 2)))
    (dolist (meta-node (prev_<=_meta_nodes (car prev)))
      (incf n)
      (let* ((item (member_1 meta-node OPEN))
	     (new-code (combine_code (second prev) (second item))))
	(cond ((not (null item))
	       (setq OPEN (substitute `(,(car item) ,new-code)
				      item OPEN :test #'equal))
	       (incf (svref codes (- new-code 1)))
	       (decf (svref codes (- (second item) 1))))
	      (t (setq OPEN (insert_in_open_NCA
			     meta-node (open_code (second prev)) OPEN))
		 (incf (svref codes (- (open_code (second prev)) 1)))))))
    (if (stop_search n codes)
        (return-from search2_NCA RESULT))))
; search for NCD specialized for =/= diamond 


(defun stop_search (n codes)
  (declare (simple-vector codes))
  (or (eql n (svref codes 3))
      (and (zerop (svref codes 0))
           (zerop (svref codes 2))
           (zerop (svref codes 3)))
      (and (zerop (svref codes 1))
           (zerop (svref codes 2))
           (zerop (svref codes 3)))
      (and (zerop (svref codes 1))
           (zerop (svref codes 2))
           (not (zerop (svref codes 0)))
           (not (zerop (svref codes 3))))
      (and (zerop (svref codes 0))
           (zerop (svref codes 2))
           (not (zerop (svref codes 1)))
           (not (zerop (svref codes 3))))))
; exit conditions for the NCD-NCA algorithms:
; no 3, and either no 1 or no 2" items in the "open" list.


(defun init_open_NCD (v1 v2)
  (when (and v1 v2)
    (if (<= (time-node-rank (get-tg v1))
	    (time-node-rank (get-tg v2)))	  
	`((,(time-node-name (get-tg v1)) 1)
	  (,(time-node-name (get-tg v2)) 2))
      `((,(time-node-name (get-tg v2)) 2)
	(,(time-node-name (get-tg v1)) 1)))))
; initialize the Open list for the NCD search


(defun init_open_NCA (v1 v2)
  (when (and v1 v2)
    (if (>= (time-node-rank (get-tg v1))
	    (time-node-rank (get-tg v2)))	  
	`((,(time-node-name (get-tg v1)) 1)
	  (,(time-node-name (get-tg v2)) 2))
      `((,(time-node-name (get-tg v2)) 2)
	(,(time-node-name (get-tg v1)) 1)))))
; initialize the Open list for the NCA search

	  
(defun first_cross_node (v kind)
  (while-not (is_meta_node v)
    (if (null (setq v (next_node_in_chain v kind)))
        (return-from first_cross_node nil)))
  v)
; find the next node in the chain that is a meta node.


(defun last_cross_node (v kind)
  (while-not (is_meta_node v)
    (if (null (setq v (prev_node_in_chain v kind)))
        (return-from last_cross_node nil)))
  v)
; find the previous node in the chain that is a meta node

(defun succ_crossnode_on_chain (v kind)
  (first_cross_node (next_node_in_chain v kind) kind))

(defun prev_crossnode_on_chain (v kind)
  (last_cross_node (prev_node_in_chain v kind) kind))

(defun succ_<=_meta_nodes (v)
  (if (null v) nil
    (if (is_meta_node v)
	(let ((next-on-chain (succ_crossnode_on_chain v 0)))
	  (if (not (null next-on-chain))
	      (cons next-on-chain
                    (<=_out_chain (time-node-next-chain (get-tg v))))
	    (<=_out_chain (time-node-next-chain (get-tg v)))))
      (progfoo (succ_crossnode_on_chain v 0)
        (if foo (return-from succ_<=_meta_nodes (list foo)))))))
; give the set of cross connection nodes that are children of v. 
; These nodes can belong to different chains


(defun prev_<=_meta_nodes (v)
  (if (null v) nil
    (if (is_meta_node v)
	(let ((prev-on-chain (prev_crossnode_on_chain v 0)))
	  (if (not (null prev-on-chain))
	      (cons prev-on-chain
                    (<=_in_chain (time-node-prev-chain (get-tg v))))
	    (<=_in_chain (time-node-prev-chain (get-tg v)))))
      (progfoo (prev_crossnode_on_chain v 0)
        (if foo (return-from prev_<=_meta_nodes (list foo)))))))
; If v is a cross-chain vertex give the set of meta nodes that are direct ancestor of v. 
; If v is not a cross-chain vertex give the previous vertex on the same chain with a not 
; null list of cross chain oucoming links.


(defun <=_out_chain (edges)
  (if (null edges) nil
    (if (zerop (time-edge-kind (car edges)))
	(cons (time-edge-to (car edges))
	      (<=_out_chain (cdr edges)))
      (<=_out_chain (cdr edges)))))
; select from a list of edges the out nodes for the edges of kind 0 (<=)


(defun <=_in_chain (edges)
  (if (null edges) nil
    (if (zerop (time-edge-kind (car edges)))
	(cons (time-edge-from (car edges))
	      (<=_in_chain (cdr edges)))
      (<=_in_chain (cdr edges)))))
; select from a list of edges the in nodes for the edges of kind 0 (<=)


(defun member_1 (item open-list)
  (if (null open-list) nil
      (if (eql item (caar open-list))
	  (car open-list)
	(member_1 item (cdr open-list)))))

(defun insert_in_open_NCD (v code open)
  (do-open-insert v code open #'<=))
; insert in the OPEN list a new item (v code) (the OPEN list is ordered by the 
; ranks of the nodes it contains. From the lower to the greater)

(defun insert_in_open_NCA (v code open)
  (do-open-insert v code open #'>=))
; insert in the OPEN list a new item (v code) (the OPEN list is ordered by the 
; ranks of the nodes it contains. From the grater to the lower)

(defun do-open-insert (v code open test-fn)
  (do ((new-open nil)
       (old-open open (cdr old-open))
       (done nil))
      (done new-open)
    (if (not (null old-open))
	(cond ((funcall test-fn (time-node-rank (get-tg (caar old-open)))
                        (time-node-rank (get-tg v)))
	       (setq new-open (nconc new-open (list (car old-open)))))
	      (t (setq new-open
		   (nconc new-open `((,v ,code)) old-open))
		 (setq done t)))
      (cond ((not done)
	     (setq new-open (nconc new-open `((,v ,code))))
	     (setq done t))))))

(defun combine_code (cod1 cod2)
  (if (or (null cod1) (null cod2)) nil
    (cond ((eql cod1 1)
	   (cond ((eql cod2 1) 1)
		 ((eql cod2 2) 3)
		 ((eql cod2 3) 3)
		 ((eql cod2 4) 4)))
	  ((eql cod1 2)
	   (cond ((eql cod2 1) 3)
		 ((eql cod2 2) 2)
		 ((eql cod2 3) 3)
	         ((eql cod2 4) 4)))
	  (t 4))))
; combine the codes of two item in the Open list of the NCA-NCD algorithms

(defun open_code (code)
  (if (eql code 3)
      4
    code))
; the code of a node in open list is the same as the one stored there,
; but the code of a NCA-NCD becomes 4 when it is combined with the code
; of children nodes.


(defun delete_redundant_neq ()
  (let ((processed (make-hash-table :size (length *points*)))
	(=/=_cross nil)
	(rel nil)
	(changes nil))
    (dolist (v *points*)
      (setq =/=_cross (cross_=/= v))
      (dolist (=/=_vertex =/=_cross)
	(cond ((not (gethash =/=_vertex processed))
	       (setq rel (type_rel v =/=_vertex (relation_1 v =/=_vertex)))
	       (if (not (null rel))
		   (cond ((eq rel :<=)
			  (replace_=/=_with_< v =/=_vertex)
			  (setq changes t))
			 ((eq rel :>=)
			  (replace_=/=_with_< =/=_vertex v)
			  (setq changes t))
			 ((or (eq rel :>) (eq rel :<))
			  (delete_=/=_cross v =/=_vertex)))))))
      (setf (gethash v processed) t))
    (cond (changes
	   (make_first&lastin&out)
	   (refine_next_greaters nil))
	  (t nil))))
;; the =/= cross chain edges of each vertex are examined and compared with
;; the kind of cross-chain path relating the two not equal vertices (v1 v2) 
;; If there is a <= path between v1 and v2 the =/= edge is removed and 
;; a new < edge beween v1 and v2 is added. If there is a <= path between v2
;; and v1 the =/= edge is removed and a new < edge between v2 and v1 is added.
;; If there is a <-path between v1 v2 or between v2 and v1 the =/= is removed. 
;; It at least one =/= has been reduced the nextgreaters are refined.


(defun replace_=/=_with_< (v1 v2)
  (let ((vertex1 (get-tg v1))
	(vertex2 (get-tg v2)))
    (if (and (null (time-node-next-chain vertex1))
	     (null (time-node-prev-chain vertex1)))
	(set-prev-in&out-next-in&out v1 t)
      (if (null (time-node-next-chain vertex1))
	  (update-prev-out&next-out v1)))
    (if (and (null (time-node-next-chain vertex2))
	     (null (time-node-prev-chain vertex2)))
	(set-prev-in&out-next-in&out v2 nil)
      (if (null (time-node-prev-chain vertex2))
	  (update-prev-in&next-in v2))))
  (delete_=/=_cross v1 v2)
  (add_<_cross v1 v2))
  

(defun get-next-meta (v)
  (let ((next-in (time-node-next-in-chain (get-tg v)))
	(next-out (time-node-next-out-chain (get-tg v))))
    (cond
     ((and (null next-in)
           (null next-out))
      nil)
     ((null next-in)
      next-out)
     ((null next-out)
      next-in)
     ((< (time-node-rank (get-tg next-in))
         (time-node-rank (get-tg next-out)))
      next-in)
     (t
      next-out))))
  
(defun get-prev-meta (v)
  (let ((prev-in (time-node-prev-in-chain (get-tg v)))
	(prev-out (time-node-prev-out-chain (get-tg v))))
    (cond
     ((and (null prev-in)
           (null prev-out))
      nil)
     ((null prev-in)
      prev-out)
     ((null prev-out)
      prev-in)
     ((> (time-node-rank (get-tg prev-in))
         (time-node-rank (get-tg prev-out)))
      prev-in)
     (t
      prev-out))))


(defun set-prev-in&out-next-in&out  (v out)
  (set-in&out-links v)
  (if out
      (update-prev-out&next-out v)
    (update-prev-in&next-in v)))

(defun set-in&out-links (v)
  (let ((succ-meta (get-first-succ-meta v))
        (succ-entry)
        (v-entry (get-tg v))
	(prev-meta (get-first-prev-meta v))
        (prev-entry))
    (if (not (null succ-meta))
	(cond ((and (in_connection_node (setq succ-entry (get-tg succ-meta)))
		    (out_connection_node succ-entry))
	       (setf (time-node-next-in-chain v-entry) succ-meta)
	       (setf (time-node-next-out-chain v-entry) succ-meta))
	      ((and (in_connection_node succ-entry )
		    (not (out_connection_node succ-entry)))
	       (setf (time-node-next-in-chain v-entry) succ-meta)
	       (setf (time-node-next-out-chain v-entry)
		     (time-node-next-out-chain succ-entry)))
	      ((and (out_connection_node succ-entry)
		    (not (in_connection_node succ-entry)))
	       (setf (time-node-next-in-chain v-entry)
		     (time-node-next-in-chain succ-entry))
	       (setf (time-node-next-out-chain v-entry) succ-meta))))
    (if (not (null prev-meta))
	(cond ((and (in_connection_node (setq prev-entry (get-tg prev-meta)))
		    (out_connection_node prev-entry))
	       (setf (time-node-prev-in-chain v-entry) prev-meta)
	       (setf (time-node-prev-out-chain v-entry) prev-meta))
	      ((and (in_connection_node prev-entry)
		    (not (out_connection_node prev-entry)))
	       (setf (time-node-prev-in-chain v-entry) prev-meta)
	       (setf (time-node-prev-out-chain v-entry)
		     (time-node-prev-out-chain prev-entry)))
	      ((and (out_connection_node prev-entry)
		    (not (in_connection_node prev-entry)))
	       (setf (time-node-prev-in-chain v-entry)
		     (time-node-prev-in-chain prev-entry))
	       (setf (time-node-prev-out-chain v-entry) prev-meta))))))
    

(defun get-first-succ-meta (v)
  (while-not (is_meta_node v)
     (unless (and (setq v (car (time-node-next (get-tg v))))
                  (setq v (time-edge-to v)))
       (return-from get-first-succ-meta nil)))
  v)

(defun get-first-prev-meta (v)
  (while-not (is_meta_node v)
     (unless (and (setq v (car (time-node-prev (get-tg v))))
                  (setq v (time-edge-from v)))
       (return-from get-first-prev-meta nil)))
  v)

(defun update-prev-out&next-out (v)
  (update-prev-out-chain v)
  (update-next-out-chain v))

(defun update-prev-in&next-in (v)
  (update-prev-in-chain v)
  (update-next-in-chain v))

(defun update-prev-out-chain (v)
  (do ((next (get-next-meta v)
	     (get-next-meta next))
       (next-entry)
       (vertex (get-tg v)))
      ((null next) t)
    (cond
     ((null (time-node-prev-out-chain (setq next-entry (get-tg next))))
      (setf (time-node-prev-out-chain next-entry) v))
     ((>= (time-node-rank
           (get-tg (time-node-prev-out-chain next-entry)))
          (time-node-rank vertex))
      (return-from update-prev-out-chain t))
     (t
      (setf (time-node-prev-out-chain next-entry) v)))))

(defun update-prev-in-chain (v)
  (do ((next (get-next-meta v)
	     (get-next-meta next))
       (next-entry)
       (vertex (get-tg v)))
      ((null next) t)
    (cond
     ((null (time-node-prev-in-chain (setq next-entry (get-tg next))))
      (setf (time-node-prev-in-chain next-entry) v))
     ((>= (time-node-rank (get-tg (time-node-prev-in-chain next-entry)))
          (time-node-rank vertex))
      (return-from update-prev-in-chain nil))
     (t
      (setf (time-node-prev-in-chain next-entry) v)))))


(defun update-next-out-chain (v)
  (do ((prev (get-prev-meta v)
	     (get-prev-meta prev))
       (prev-entry)
       (vertex (get-tg v)))
      ((null prev))
    (cond 
     ((null (time-node-next-out-chain (setq prev-entry (get-tg prev))))
      (setf (time-node-next-out-chain prev-entry) v))
     ((<= (time-node-rank (get-tg (time-node-next-out-chain prev-entry)))
          (time-node-rank vertex))
      (return-from update-next-out-chain nil))
     (t
      (setf (time-node-next-out-chain prev-entry) v)))))


(defun update-next-in-chain (v)
  (do ((prev (get-prev-meta v)
	     (get-prev-meta prev))
        (prev-entry)
        (vertex (get-tg v)))
      ((null prev))
    (cond
     ((null (time-node-next-in-chain (setq prev-entry (get-tg prev))))
      (setf (time-node-next-in-chain prev-entry) v))
     ((<= (time-node-rank (get-tg (time-node-next-in-chain prev-entry)))
          (time-node-rank vertex))
      (return-from update-next-in-chain nil))
     (t
      (setf (time-node-next-in-chain prev-entry) v)))))

(defun delete_=/=_cross (v1 v2)
  (let* ((ne-v1 (time-node-noteq (get-tg v1)))
	 (ne-v2 (time-node-noteq (get-tg v2))))
    (setf (third ne-v1)
      (delete v2 (third ne-v1)))
    (setf (third ne-v2)
      (delete v1 (third ne-v2)))))
;; remove from v1 and v2 the the name of the other vertex in the slot
;; containing the list of the not equal vetices that are on a different chain.

(defun add_<_cross (v1 v2)
  (let ((existing-edge (get_nextchain_edge v1 v2))
	(pc (time-node-prev-chain (get-tg v2)))
	(nc (time-node-next-chain (get-tg v1)))
	(new-edge (new-time-edge v1 v2 1)))
    (cond ((not (null existing-edge))
	   (cond ((eql (time-edge-kind existing-edge) 0)
		  (change-nextchain-edge v1 v2 new-edge)
		  (change-prevchain-edge v2 v1 new-edge))))
	  (t
	   (setf (time-node-next-chain (get-tg v1))
		 (cons new-edge nc))
	   (setf (time-node-prev-chain (get-tg v2))
		 (cons new-edge pc))))))
      
;; add to v1 an outcoming cross chain edge to v2, and to v2 an incoming
;; cross chain edge from v1. 


(defun type_rel (v1 v2 rel)
  (cond ((eq (cadr rel) :unknown) nil)
	((eq (cadr rel) :=) :=)
	((eq (cadr rel) :=/=) :=/=)
	((and (eql v1 (car rel)) (eq (cadr rel) :<=)) :<=)
	((and (eql v1 (car rel)) (eq (cadr rel) :<)) :<)
	((and (eql v2 (car rel)) (eq (cadr rel) :<=)) :>=)
	((and (eql v2 (car rel)) (eq (cadr rel) :<)) :>)))




