;;; Copyright (C) 1994 by Istituto per la Ricerca Scientifica e Tecnologica 
;;; (IRST) (38050 Povo, Trento Italy) and the Trustees of the University 
;;; of Rochester (Rochester, NY 14627, USA).  All rights reserved.
;;; See the COPYRIGHT.TEXT file for more information

;; separation of the ui from the engine, and some other cleanups 
;; by Brad Miller miller@cs.rochester.edu 12/7/93

#|  Dec. 18 1993

 *****************************************************
 *                                                   *
 *  TimeGraph II (TG-II) - User Point Interface      *
 *                                                   *
 *  Alfonso Gerevini                                 *
 *  IRST 38050 Povo Trento Italy                     *
 *  and                                              *
 *  Department of Computer Science                   *
 *  University of Rochester, 14627 Rochester, USA    *
 *                                                   *
 *  email: gerevini@irst.it or                       *
 *         gerevini@cs.rochester.edu	             *
 *                                                   *
 *****************************************************

|#

(in-package TG-II)


(defmacro assertpr (left rel right)
  "assert a constraint between two time points"
  `(assert-pr ',left ',rel ',right))


(defun assert-pr (left rel right)
  "assert a constraint between two time points"
  (if *p-mode*
      (cond ((eq rel :=)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left ,right 0)
		     (cons `(,right ,left 0)
			   *point-relations*))))
	    ((eq rel :<)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left ,right 1)
		     *point-relations*)))
	    ((eq rel :<=)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left
		       ,right
		       0)
		     *point-relations*)))
	    ((eq rel :>)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left
		       ,right
		       1)
		     *point-relations*)))
	    ((eq rel :>=)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left
		       ,right
		       0)
		     *point-relations*)))
	    ((eq rel :=/=)
	     (setq left (get-maybe-new-point left))
	     (setq right (get-maybe-new-point right))
	     (setq *point-relations*
	       (cons `(,left
		       ,right
		       2)
		     *point-relations*)))
	    (t
	     (warn "~% ~A is not a legal relations ~%" rel)))
    (warn "~% Illegal assertion, system operating in interval mode"))
  (values))


(defun get-maybe-new-point (pt)
  (or (pt->int pt)
      (prog1 *counter*
        (setf (pt->int pt) *counter*)
        (setf (int->pt *counter*) pt)
        (incf *counter*))))

(defun rel-name (rel)
  (cond ((eql rel 0)
	 :<=)
	((eql rel 1)
	 :<)
	((eql rel 2)
	 :=/=)
	(t (error "~% undefined relation name"))))


(defun query-p-1 (rel-code)
  (when (not (null rel-code))
    (list (int->pt (first rel-code)) 
	  (rel-name (third rel-code)) 
	  (int->pt (second rel-code)))))



(defun stipulated-prel ()
  "return the list of the stipulated constraints"
  (dolist (rr *point-relations*)
    (format t "~% ~s " (query-p-1 rr))))


(defun list-points ()
  "return the list of the time points"
  (let ((l-p nil)) 
    (dotimes (pp (- *counter* 1))
      (setq l-p (cons (int->pt (+ pp 1)) l-p)))
    (reverse l-p)))


(defun less-or-equal (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is <=;
   return NIL if it is >; otherwise return UNKNOWN"
  (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
      (error "~% undefined time point")
    (leq (pt->int pt1) (pt->int pt2))))


(defun less-than (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is <;
   return NIL if it is >=; otherwise return UNKNOWN"
  (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
      (error "~% undefined time point")
    (less (pt->int pt1) (pt->int pt2))))


(defun greater-than (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is <=;
   return NIL if it is <=; otherwise return UNKNOWN"
  (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
      (error "~% undefined time point")
    (less (pt->int pt2) (pt->int pt1))))


(defun greater-or-equal (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is >=;
   return NIL if it is <; otherwise return UNKNOWN"
  (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
      (error "~% undefined time point")
    (leq (pt->int pt2) (pt->int pt1))))


(defun equal-to (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is =;
   return NIL if it is >,<., or =/=; otherwise return UNKNOWN"
  (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
       (error "~% undefined time point")
    (query_equal (pt->int pt1) (pt->int pt2))))


(defun not-equal-to (pt1 pt2)
  "return true if the strongest relation between pt1 and pt2 is <,>, or =/=;
   return NIL if it is =; otherwise return UNKNOWN"
   (if (or (null (pt->int pt1))
	  (null (pt->int pt2)))
       (error "~% undefined time point")
     (query_not_equal (pt->int pt1) (pt->int pt2))))


(defun query-pr (&optional pt1 pt2)
  "query the strongest entailed relation between two time points"
  (cond ((and pt1 pt2)
	 (let ((rr (relation (pt->int pt1) (pt->int pt2))))
	   (if (not (null rr))
	       (list (int->pt (first rr))
		     (second rr)
		     (int->pt (third rr)))	       
	     (if (and (not (null (pt->int pt1)))
		      (not (null (pt->int pt2))))
		 (list pt1 :<=> pt2)
	       (error "~% undefined time point")))))
	(pt1 (all-single-prel pt1))
	(t (all-prel))))


(defun query-rel-pt (rel)
  (list (int->pt (first rel))
	(second rel)
	(int->pt (third rel))))	       


(defmacro prel  (&optional pt1 pt2)
  "query the strongest entailed relation between two time points"
  `(query-pr ',pt1 ',pt2))

;; when the timegraph has not been created all-prel prints the list
;; of the relations stipulated by the user
(defun all-prel ()
  "return the list of all the strongest entailed relations"
  (when *p-mode*
    (if (not (zerop *number-chains*)) ;; the timegraph has been created
	(let ((rel nil))
	  (dolist (pt1 *points*)
	    (when (not (eql pt1 0))
	      (format t "~%Point ~A" (int->pt pt1)))
	    (dolist (pt2 *points*)
	      (when (and (not (eql pt1 0))
			 (not (eql pt2 0))
			 (not (eql pt1 pt2)))
		(setq rel (relation pt1 pt2))
		(if (not (null rel))
		    (if (eql (car rel) pt1)
			(format t "~%         ~S"
				(query-rel-pt rel))
		      (format t "~%         ~S"  
			      (query-rel-pt (inv-point-rel rel))))
		  (format t "~%         (~A :<=> ~A)" 
			  (int->pt pt1) (int->pt pt2))))))
	  (list-equal-relations))
      (stipulated-prel) ;; the timegraph has not yet been created
      )))


(defun all-single-prel (tp)
  "return the list of the strongest entailed relations between a point
 and all the others"
  (when *p-mode*
    (if (null (pt->int tp))
	(error "~% undefined time point")
      (let ((rel nil))
	(dolist (pt1 *points*)
	  (when (and (not (eql pt1 0))
		     (not (eql (pt->int tp) 0))
		     (not (eql pt1 (pt->int tp))))
	    (setq rel (relation pt1 (pt->int tp)))
	    (if (not (null rel))
		(if (eql (car rel) pt1)
		    (format t "~%         ~S"  
			    (query-rel-pt rel))
		  (format t "~%         ~S"  
			  (query-rel-pt (inv-point-rel rel))))
	      (format t "~%         (~A :<=> ~A)" 
		      (int->pt pt1) tp))))))
    (list-equal-relations tp)))

(defun not<=>prel ()
  "return the list of all the strongest entailed relations different from <=>"
  (when *p-mode*
    (let ((rel nil))
      (dolist (pt1 *points*)
	(when (not (eql pt1 0))
	  (format t "~%Point ~A" (int->pt pt1)))
	(dolist (pt2 *points*)
	  (when (and (not (eql pt1 0))
		     (not (eql pt2 0))
		     (not (eql pt1 pt2)))
	    (setq rel (relation pt1 pt2))
	    (if (not (null rel))
		(if (eql (car rel) pt1)
		    (format t "~%         ~S" 
			    (query-rel-pt (relation pt1 pt2)))
		  (format t "~%         ~S"
			  (query-rel-pt (inv-point-rel (relation pt1 pt2))))))))))
    (list-equal-relations)))

(defun list-eq ()
  (let ((l nil))
    (dotimes (i +max-eq+)
      (when (get-eq i)
        (push (list i (get-eq i)) l)))
    l))

(defun list-equal-relations (&optional pt)
  (cond  ((null pt)
	  (format t "~%")
	  (dolist (l_= (list-eq))
	    (format t "~%Point ~A equal to: " (int->pt (car l_=)))
	    (dolist (eqp (cadr l_=))
	      (format t "~A " (int->pt eqp)))))
	 (t
	  (let ((eq-p (get-eq (pt->int pt))))
	    (if (not (null eq-p))
		;;	(format t "~% Point ~A equal to: ~A" pt (int->pt (car eq-p)))
		(if (not (eql (pt->int pt) (car eq-p)))
		    (dolist (l_= (list-eq))
		      (if (and (eql (caadr l_=) (car eq-p))
			       (not (eql (car l_=) (pt->int pt))))
			  (format t "~%         (~A ~S ~A)" (int->pt (car l_=)) := pt))))
	      (dolist (l_= (list-eq))
		(if (and (eql (caadr l_=) (pt->int pt))
                         (not (eql (car l_=) (pt->int pt))))
                    (format t "~%         (~A ~S ~A)" (int->pt (car l_=)) := pt))))
	    (format t "~%")))))
  

(defun inv-point-rel (rel)
  (cons (third rel)
	(cons (cond ((eql (second rel) '<=)
		     `>=)
		    ((eql (second rel) `<)
		     `>)
		    (t (second rel)))
	      (list (first rel)))))
	

;; print the chains of the timegraph. time points 
;; corresponding to Metavertices (cross-chain vertices) 
;; are printed within [], while points corresponding 
;; to regular vertices are printed within (). Transitive
;; edges and metaedges (cross-chain edges) are not 
;; reported.
(defun print-chains ()
  "print the chains ot the timegraph"  
  (format t "~%CHAINS: ~%")
  (dotimes (c *number-chains*)
    (print-chain (+ c 1)))
  (format t "~% ~%"))
		 

(defun print-chain (chain)
  (when *p-mode*
    (format t "~%")
    (if (numberp chain)
	(cond ((<= chain *number-chains*)
	       (format t "chain ~A:" chain)
	       (cond ((and (<= chain 999) (> chain 99))
		      (format t " "))
		     ((and (<= chain 99) (> chain 9))
		      (format t "  "))
		     ((<= chain 9)
		      (format t "   ")))
	       (do* ((stop nil)
		     (pos 0)
		     (next (first-node-chain chain)
			   (if (not stop)
			       (time-edge-to 
				(car (time-node-next (get-tg next)))))))
		   (stop)
		 (when (> pos 9) (format t "~%           ") (setq pos 0))
		 (when (not (eql next 0))
		   (setq pos (+ 1 pos))
		   (if (is_meta_node next)
		         ;;	 next is a metanode
		       (format t "[~A]" (int->pt next))
		     ;; next is not a metanode
		     (format t "(~A)" (int->pt next))) 
		   (when (not (equal next (last-node-chain chain)))
		     (format t "---"))
		   (when (eql next (last-node-chain chain))
		     (setq stop t)))))
	      (t (error "~A is not a time-chain")))
      (error "~A is not a time-chain"))))
  


(defun print-tg ()
  "print information about the timegraph"
  (when *p-mode*
    (print-chains-summary)
    (print-less&greater)
    (print-chain-links)))
  

(defun print-less&greater ()
  (when *p-mode*
    (format t "~% ~%** PRELESS AND NEXTGREATER POINTERS: ~%")
    (dotimes (c *number-chains*)
      (format t "~% Chain ~A ~%" (+ c 1))
      (do* ((next (first-node-chain (+ c 1)) (next-node next)))
	  ((null next) t)
	(when (not (zerop next))
	  (if (not (null (time-node-next-greater (get-tg next))))
	      (format t "nextgreater for ~A ~A ~%" (int->pt next)
		      (int->pt (time-node-next-greater (get-tg next)))))
	  (if (not (null (time-node-prev-less (get-tg next))))
	      (format t "prevless for ~A ~A ~%" (int->pt next)
		      (int->pt (time-node-prev-less (get-tg next))))))))))


(defun print-chain-links ()
  (when *p-mode*
    (format t "~%** CHAIN LINKS: ~% ~%")
    (dolist (pp *points*)
      (if (not (null (time-node-next-out-chain (get-tg pp))))
	  (format t "~% Next out link for node ~A ~A ~%"
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)
		  (time-node-next-out-chain (get-tg pp))))
      (if (not (null (time-node-next-in-chain (get-tg pp))))
	  (format t "Next in link for node ~A ~A ~%"
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)
		  (int->pt (time-node-next-in-chain (get-tg pp)))))
      (if (not (null (time-node-prev-in-chain (get-tg pp))))
	  (format t "Prev in link for node ~A  ~A ~%"
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)
		  (int->pt (time-node-prev-in-chain (get-tg pp)))))
      (if (not (null (time-node-prev-out-chain (get-tg pp))))
	  (format t "Prev out link for node ~A  ~A ~%"
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)
		  (int->pt (time-node-prev-out-chain (get-tg pp)))))
      (if (not (null (time-node-next-chain (get-tg pp))))
	  (format t "~%Cross out for node ~A --> ~A ~%" 
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)

		  (list-to (time-node-next-chain (get-tg pp)))))
      (if (not (null (time-node-prev-chain (get-tg pp))))
	  (format t "Cross in for node ~A <-- ~A ~%" 
		  (if (not (zerop pp))
		      (int->pt pp) 
		    '*ROOT*)
		  (list-from (time-node-prev-chain (get-tg pp))))))))

;; print information about the chains of the timegraph.
;; for each chain the lenght, the starting point and 
;; the ending point are printed.
(defun print-chains-summary ()
  "print information about the chains in the timegraph"
  (when *p-mode*
    (format t "~%** CHAINS: ~% ~%")
    (dotimes (c *number-chains*)
      (format t "chain ~A -> " (+ 1 c))
      (format t "length: ~A first point: ~A last point: ~A ~%"
	      (length-chain (+ 1 c))
	      (if (not (zerop (first-node-chain (+ 1 c))))
		  (int->pt (first-node-chain (+ 1 c)))
		'*ROOT*)
	      (int->pt (last-node-chain (+ 1 c)))))))
  

(defun length-chain (c)
  (do ((n 1 (+ 1 n))
       (next (first-node-chain c)
	     (time-edge-to (car (time-node-next (get-tg next))))))
      ((eql next (last-node-chain c)) n)))		   



(defun list-to (edges)
  (mapcar #'(lambda (arc) (int->pt (time-edge-to arc))) edges))

(defun list-from (edges)
  (mapcan #'(lambda (arc) (if (not (zerop (time-edge-from arc)))
                              (list (int->pt (time-edge-from arc)))))
          edges))










