;;; Copyright (C) 1994 by Istituto per la Ricerca Scientifica e Tecnologica 
;;; (IRST) (38050 Povo, Trento Italy) and the Trustees of the University 
;;; of Rochester (Rochester, NY 14627, USA).  All rights reserved.
;;; See the COPYRIGHT.TEXT file for more information

;; separation of the ui from the engine, and some other cleanups 
;; by Brad Miller miller@cs.rochester.edu 12/7/93

#|  Dec. 18 1993

 *****************************************************
 *                                                   *
 *  TimeGraph II (TG-II) - Interval Interface        *
 *                                                   *
 *  Alfonso Gerevini                                 *
 *  IRST, 38050 Povo, Trento, Italy                  *
 *  and                                              *
 *  Department of Computer Science                   *
 *  University of Rochester, 14627 Rochester, USA    *
 *  email: gerevini@irst.it                          *
 *         gerevini@cs.rochester.edu	             *
 *                                                   *
 *  Ed Yampratom                                     *
 *  Department of Computer Science                   *
 *  University of Rochester, 14627 Rochester, USA    *
 *  email: edyamp@cs.rochester.edu                   *
 *                                                   *
 *****************************************************
|#

;;; Interface functions:
;;; (init-tg)              clear the database
;;; (asserta i1 reln i2)   asserta the constraint
;;; (make-tg)              build the timegraph
;;; (arel i1 i2)           query the relationship between i1 i2
;;; The variable *intervals* contains the list of defined intervals

;; the 13 atomic interval relations are represented by 
;; bit-vector of size 24. Each item in the vector corresponds
;; to a different point relations between endpoins of the intervals.
;; Each block of six bits represents the following point relation:
;; =,<,<=,>,>=,<>
;; Bits 1-6 for relations between begin(I1) and begin(I2)
;; Bits 7-12 for begin(I1) and end(I2)
;; Bits 13-18 for end(I1) and end(I2)
;; Bits 19-24 for end(I1) and begin(I2)
;; Meanings:
;;     :=,:E	equals 
;;	:D	during (proper)
;;	:Di	contains (proper)
;;	:S	starts
;;	:Si	started-by
;;	:F	finishes
;;	:Fi	finished-by
;;     :P,:B	precedes (before)
;;     :A,:P-	preceded-by (after)
;;	:M	meets
;;	:Mi	met-by
;;	:O	overlaps
;;	:Oi	overlapped-by

;;; Example:
;;; (asserta i1 :m i2)
;;; (asserta (i2 (:b :m) i3))
;;; (make-tg) 
;;; (arel i1 i3) will returns (i1 (:b) i3)


;;; -------------------------------------------
;;; Interval relations <-> Point relations Code
;;; -------------------------------------------

(in-package TG-II)

;;; --- CREATING THE POINTIZABLE RELATIONS TABLE --- ;;;
;; Insert a relation into the pointizable-relations hash table
(defun insert-table (key I-relation)
    (setf (get-pr key) I-relation))

;; Initialize the pointizable relations table by reading in from file
(defun init-prtab ()
  (with-open-file 
      (file-i user::*table-file*
       :direction :input)
    (let ((item nil))
      (dotimes (pos +number-pointizable-relations+)
	(setq item (read-from-string (read-line file-i)))
	;; (print (+ pos 1))
	;; (print item)
	(insert-table (car item) (cadr item))))))

;;; --- FUNCTIONS FOR LOOKING UP THE POINTIZABLE RELATIONS TABLE --- ;;;

;; Translate bit-vector into corresponding integer which is hashtable key
(defun make-key (bit-vector)
  (let ((result 0))
    (dotimes (pos +vector-dimension+)
      (incf result (* (aref bit-vector pos) (expt 2 pos))))
    result))


;; Translate a list of disjoint standard/allen relations into bit-vectors
(defun relation->bit (interval-relation)
  (if (null interval-relation)
      #*111111111111111111111111
    (bit-and (cdr (assoc (car interval-relation) +names-atomic-allen-rel+))
	     (relation->bit (cdr interval-relation)))))


;; takes an Allen relation and gives in output the Allen relation
;; represented by its bit-vector representation
(defun back-translation (Allen-relation)
  (get-pr (make-key (relation->bit Allen-relation))))

;;; --- THE INTERVAL INTERFACE --- ;;;

(defun init-tg ()
  "initialize the temporal database"
  (setq *stipulated-relations* nil)
  (when (empty-prtab)
    ;; Call these to set up the tables
    (init-prtab))
  (when *p-mode* 
    (clrhash *point-integer-table*)
    (clrhash *integer-point-table*))
  #+tg-debug (format t "New graph is created.~%")
  (init_timegraph)
  (setf *points* nil)
  (setf *point-relations* nil)
  (setf *counter* 1)
  (when (not *p-mode*)
    (setq *intervals* nil)
    (clrhash *itab*))
  (values))

(defun empty-prtab ()
  (null (get-pr (make-key b))))
	 
(defun make-tg ()
  "create the timegraph"
  #+tg-debug (format t "Counter: ~A~%Relns: ~A~%" *counter*  *point-relations*)
  (dotimes (i (- *counter* 1))
    (push (- *counter* i 1) *points*))
  (setq *stipulated-relations* nil)
  (when (not (null *points*))
    (create-tg)))


;;; ---Asserting a relation--- ;;;
(defmacro asserta (left rel right)
  "assert a constraint between two intervals"
  `(assert-a ',left ',rel ',right))


;;; ---Querying about intervals--- ;;;


;; (defmacro rels (interval)
;;  "print the list of all the strongest entailed relations involving a given interval"
;;  `(l-rels ',interval))


(defmacro arel (&optional left right)
  "macro to make querying easier (no quoting necessary)."
  `(query-a ',left ',right))

;; when the timegraph has not been created yet query-a print the list
;; of the relattions stipulated by the user
(defun query-a (&optional left right)
  "query the relation between intervals"
  (cond ((and left right)
	 (query-allen-rel left right))
	(left
	 (l-rels left))
	(t
	 ;; the timegraph has not been created yet
	 (if (zerop *number-chains*)
	     (dolist (ll (reverse *stipulated-relations*))
	       (format t "~%~S" ll))
	   ;; the timegraph has been created
	   (dolist (int *intervals*)
	     (l-rels int))))))


(defun query-allen-rel (left right)
  (if (or (null (start-of left)) (null (start-of right)))
      (warn  "Nothing...Non-existence Intervals")
    (let ((bits (make-array +vector-dimension+ :element-type 'bit :initial-element 0))
	  (ss (second (relation-mod (start-of left) (start-of right))))
	  (se (second (relation-mod (start-of left) (end-of right))))
	  (ee (second (relation-mod (end-of left) (end-of right))))
	  (es (second (relation-mod (end-of left) (start-of right)))))
      (modify-bits bits 0 ss)
      (modify-bits bits 6 se)
      (modify-bits bits 12 ee)
      (modify-bits bits 18 es)
      ;;    (print bits)
      ;; (format t "~A ~%" (make-key bits))
      ;;       (list left (get-pr (make-key bits)) right)
      (list left (get-pr (make-key bits)) right)
      )))  
 
;; print all the strongest relations entailed by the timegraph
(defun l-rels (interval)
  (when (member interval *intervals*)
    (dolist (ii *intervals*)
      (when (not (eq ii interval))
	(format t "~S ~%" (query-a interval ii))))))


(defun start-of (point)
  "Get the starting point of the interval"
  (first (get-i point)))
(defun end-of (point)
  "Get the ending point of the interval"
  (second (get-i point)))



;; we need this since (RELATION 5 1) returned (1 < 5) but
;; (RELATION 5 2) returned (5 < 2)
(defun relation-mod (p1 p2)
  "Modify the answer given by relation to make it more appropriate."
  (let ((ans (relation p1 p2)))
    (if (or (eq (first ans) p1) (null ans))
	ans
      (list (third ans)
	    (inverse-reln (second ans))
	    (first ans)))))

(defun inverse-reln (reln)
  "Get the inverse of a given relation."
  (case reln
    (:> :<)
    (:>= :<=)
    (:< :>)
    (:<= :>=)
    (:=/= :<>)				;inconsistent notation
    (otherwise
     reln)))

#|
= implies >=, <= 
< implies <=, <>
> implies >=, <>
but <=, >=  imply nothing else 
|#


(defun modify-bits (bits start reln)
  "Modify the bitstring according to the given relation."
  ;; Bit orders: =,<,<=,>,>=,<>
  ;; Possible answers from relation-mod:
  ;; <, <=, >, >=, <>, or nil (unknown)  
  (case reln
    (:=
     (setf (sbit bits (+ start 0)) 1)
     (setf (sbit bits (+ start 2)) 1) ;; implies <=
     (setf (sbit bits (+ start 4)) 1) ;; implies >=
     )
    (:<
     (setf (sbit bits (+ start 1)) 1)
     (setf (sbit bits (+ start 2)) 1) ;; implies <=
     (setf (sbit bits (+ start 5)) 1) ;; implies <>
     )
    (:<=
     (setf (sbit bits (+ start 2)) 1)
     )
    (:>
     (setf (sbit bits (+ start 3)) 1)
     (setf (sbit bits (+ start 4)) 1) ;; implies >=
     (setf (sbit bits (+ start 5)) 1) ;; implies <> 
     )
    (:>=
     (setf (sbit bits (+ start 4)) 1)
     )
    (:<>
     (setf (sbit bits (+ start 5)) 1)
     )
    (otherwise				;unknown -> set all bits to 0
     (dotimes (i 6)
       (setf (sbit bits (+ i start)) 0)))))


;; return nil if rel is not an Allen interval relation, otherwise return T
(defun check-relation (rel)
  (if (not (listp rel))
      (if (not (null  (member rel +atomic-allen-rels+)))
	  t 
	nil)
    (do* ((ll rel (cdr ll))
	  (check-r t))
	((or (null ll) (not check-r)) (if (null check-r) nil t))
      (setq check-r (member (car ll) +atomic-allen-rels+)))))


(defun assert-a (left reln right)
  "The real assertion job is done here."
  (cond ((and (not *p-mode*) (not (null reln)))
	 (when (check-relation reln)
	   (setq *stipulated-relations* (cons (list left reln right) *stipulated-relations*))
	   (add-interval-relation left reln right)
	   (values)))
	(t (if *p-mode*
	       (warn "~% Illegal assertion, system operating in point mode")
	     nil))))


(defun add-interval-relation (t1 reln t2)
  "add constraint between t1 and t2, which may be new intervals"
  (maybe-new-interval t1)
  (maybe-new-interval t2)
  (if (listp reln)
      (add-constraint t1 reln t2)
    (add-constraint t1 (list reln) t2)))


(defun maybe-new-interval (t1)
  "Create a new interval if it doesn't exist"
  (unless (get-i t1)
    #+tg-debug (format t "Creating a new interval: ~A -> (~A,~A)."
                       t1 *counter* (1+ *counter*))
    (setq *intervals* (cons t1 *intervals*))
    (setf (get-i t1) (list *counter* (1+ *counter*)))
    ;; also need to asserta start < end
    (push (list *counter* (1+ *counter*) 1) *point-relations*)
    #+tg-debug (format t " so we asserta that ~A < ~A.~%" *counter* (1+ *counter*))
    (setf *counter* (+ *counter* 2))))

 (defun add-constraint (t1 reln t2)
   "Add the constraint between 2 intervals"
  (let ((bits (relation->bit reln)))
    #+tg-debug (format t "Constraining ~A ~A ~A.~%" t1 reln t2)
    (express bits 0 (start-of t1) (start-of t2))
    #+tg-debug (format t "~%")
    (express bits 6 (start-of t1) (end-of t2))
    #+tg-debug (format t "~%")
    (express bits 12 (end-of t1) (end-of t2))
    #+tg-debug (format t "~%")
    (express bits 18 (end-of t1) (start-of t2))
    #+tg-debug (format t "~%")
    )
  )


(defun express (bits start p1 p2)
  "Assert the point relations by pushing it into point-relations"
  ;; =,<,<=,>,>=,<>
  ;; 0 for less than or equal, 1 for less than, 2 for not equal.
  (let ((p= (eq (sbit bits start) 1))
	(p< (eq (sbit bits (+ 1 start)) 1))
	(p<= (eq (sbit bits (+ 2 start)) 1))
	(p> (eq (sbit bits (+ 3 start)) 1))
	(p>= (eq (sbit bits (+ 4 start)) 1))
	(p<> (eq (sbit bits (+ 5 start)) 1)))
    (cond (p=
	   #+tg-debug (format t "~A = ~A; " p1 p2)
	   (push (list p1 p2 0) *point-relations*)
	   (push (list p2 p1 0) *point-relations*))
	  ;; 0 & 1 "<=" & "<" --> "<"
	  ;; 0 & 2 "<=" & "<>" --> "<"
	  ;; 1 & 2 "<" & "<>" --> "<"
	  ;; 0 & 1 & 2 --> "<"
	  ((or (and p<= p<) (and p<= p<>) (and p< p<>) (and p<= p< p<>) p<)
	   (push (list p1 p2 1) *point-relations*)
           #+tg-debug(format t "~A < ~A; " p1 p2))
	  ((or (and p>= p>) (and p>= p<>) (and p> p<>) (and p>= p> p<>) p>)
	   (push (list p2 p1 1) *point-relations*)
           #+tg-debug (format t "~A > ~A; " p1 p2))

	  (p<=
	   (push (list p1 p2 0) *point-relations*)
	   #+tg-debug (format t "~A <= ~A; " p1 p2))
	  (p>=
	   (push (list p2 p1 0) *point-relations*)
	   #+tg-debug (format t "~A >= ~A; " p1 p2))
	  (p<>
	   (push (list p1 p2 2) *point-relations*)
	   #+tg-debug (format t "~A <> ~A; " p1 p2)))))


(defun pointizable (Allen-relation)
  "check if a relation is pointizable"
  (let* ((Allen-pointizable (back-translation Allen-relation))
	 (information-loss (set-difference
			    Allen-pointizable
			    (standard-relation Allen-relation))))
    (if (null information-loss) t nil)))


;; change interface mode from points to intervals or viceversa.
;; WARNING: the timegraph is destroyed
(defun set-point-mode (p-m)
  "set the point or interval mode"
  (init-tg)
  (setq *p-mode* p-m))


;;; -- USEFUL FUNCTIONS FOR DEBUGGING --- ;;; 
(defun list-hash (h)
  (prog ((l nil))
    (maphash #'(lambda (key val) (setq l (cons (list key val) l))) h)
    (return l)))

(defun b2r (bits)
  (showbits bits 0)
  (format t "~%")
  (showbits bits 6)
  (format t "~%")
  (showbits bits 12)
  (format t "~%")
  (showbits bits 18)
  (format t "~%")
)

(defun showbits (bits start )
  ;; =,<,<=,>,>=,<>
  ;; 0 for less than or equal, 1 for less than, 2 for not equal.
  (when (eql (sbit bits start) 1)
    (format t "="))
  (when (eql (sbit bits (+ 1 start)) 1)
    (format t "<"))
  (when (eql (sbit bits (+ 2 start)) 1)
    (format t "<="))
  (when (eql (sbit bits (+ 3 start)) 1)
    (format t ">"))
  (when (eql (sbit bits (+ 4 start)) 1)
    (format t ">="))
  (when (eql (sbit bits (+ 5 start)) 1)
    (format t "<>"))
  )



;; takes in input an allen-relation and returns in output its standard name
(defun standard-relation (allen-relation)
  (do* ((relations '(p oi mi si fi di pi =) (cdr relations))
	(standard-names '(:b :o- :m- :s- :f- :d- :a :e) (cdr standard-names)))
      ((null relations) allen-relation)
    (setq allen-relation (substitute (car standard-names) (car relations)
				     allen-relation))))



