;;; Copyright (C) 1994 by Istituto per la Ricerca Scientifica e Tecnologica 
;;; (IRST) (38050 Povo, Trento Italy) and the Trustees of the University 
;;; of Rochester (Rochester, NY 14627, USA).  All rights reserved.
;;; See the COPYRIGHT.TEXT file for more information

;; separation of the ui from the engine, and some other 
;; cleanups by Brad Miller miller@cs.rochester.edu 12/7/93-12/13/93
#| Dec. 8 1993

 *****************************************************
 *                                                   *
 *  TimeGraph II (TG-II) - Temporal Reasoning System *
 *                                                   *
 *  written by Alfonso Gerevini                      *
 *  IRST 38050 Povo Trento Italy                     *
 *  and                                              *
 *  Department of Computer Science                   *
 *  University of Rochester, 14627 Rochester, USA    *
 *                                                   *
 *  Email: gerevini@irst.it                          *
 *         gerevini@cs.rochester.edu	             *
 *                                                   *
 *****************************************************

|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                            ;; 
;;    DATA STRUCTURES AND GLOBAL VARIABLES    ;;
;;                                            ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; The values of the constants +max-size-timegraph+, +max-eq+, +max-chains+ 
;; can be modified according to the size of of the timegraphs which the
;; user intend to build.

(in-package TG-II)

;; TIMEGRAPH
(defparameter *check-less-for-each-pair-nca-ncd* t)

;; used only within check_cycles
(defvar *trees* nil "alist of trees")
(defvar *roots* ())

(defvar *points* ())

(defvar *point-relations* nil)

(defvar *old* nil)

(defvar *strict* nil)

(defvar *start-times* nil)

(defvar *inconsistency* nil)

;; max number of time points 
(defconstant +max-size-timegraph+ 5000)

(defconstant +max-chains+ 5000)
(defconstant +pseudotime-step+ 100)
(defconstant +pseudotime-max+ (round-to most-positive-fixnum +pseudotime-step+))
(defconstant +max-eq+ 5000)

(deftype pseudotime () `(integer 0 ,+pseudotime-max+))
(deftype chainnumber () `(integer 0 ,+max-chains+))
(deftype tg-entry () `(integer 0 ,+max-size-timegraph+))

(defvar *timegraph* (make-array +max-size-timegraph+))
(declaim (type simple-vector *timegraph*))

(defun init-timegraph ()
  (setq *timegraph* (make-array +max-size-timegraph+)))

(defmacro get-tg (entry)
  `(svref *timegraph*  (the tg-entry ,entry)))

(defvar *equal-classes*  (make-array +max-eq+))
(declaim (type simple-vector *equal-classes*))
(defun init-equal ()
  (setq *equal-classes* (make-array +max-eq+)))

(defmacro get-eq (entry)
  `(svref  *equal-classes*  (the (integer 0 ,+max-eq+) ,entry)))


(defvar *number-chains* 0)

(declaim (type chainnumber *number-chains*))
(defvar *chains* (make-array +max-chains+ :initial-element 0))
(declaim (type simple-vector *chains*))
(defmacro get-chain (entry)
  `(svref  *chains*  (the chainnumber ,entry)))
(defun clear-chains ()
   (setq *chains* (make-array +max-chains+ :initial-element 0)))

(defvar *ranks*  nil)
(defvar *max-rank* 0)

(declaim (type pseudotime *max-rank*))
(declaim (type hash-table *ranks*)) ; a small lie, but it will be initialized before any use.
(defun clear-ranks ()
  (clrhash *ranks*))
(defun reinit-ranks ()
  (setq *ranks* (make-hash-table :size (/ *max-rank* +pseudotime-step+))))
(defmacro get-rank (entry)
  `(gethash (the pseudotime ,entry) *ranks*))

(defstruct time-node
  (name nil)
  (chain nil)
  (next nil)
  (prev nil)
  (prev-less nil)
  (next-greater nil)
  (prev-chain nil)
  (next-chain nil)
  (next-out-chain nil)
  (prev-out-chain nil)
  (prev-in-chain nil)
  (next-in-chain nil)
  (noteq nil)
  (rank 0 :type pseudotime)
  (less-dsj-instances nil)  
  (greater-dsj-instances nil)
  (c 0 :type (integer 0 2)) ;this slot is used only by the check_cycles function, (and diamond reduction, apparently. -miller)
  )       

(defmacro update-tg (name &rest slots)
  `(setf (get-tg ,name) (make-time-node :name ,name ,@slots)))

(defstruct time-edge
  (to nil)
  (from nil)
  (kind 0 :type (integer 0 2)))
;; kind 0 is for <=
;; kind 1 is for <
;; kind 2 is for =/=

(defun new-time-edge (from to kind)
  (make-time-edge :from from :to to :kind kind))
      
(define-condition tg-error (error)
  ()
  (:documentation "All timegraph errors should inherit from this condition"))

(define-condition tg-contradiction (tg-error)
  ()
  (:report "A contradiction has been detected")
  (:documentation "General timegraph contradiction; specific contradiction types inherit from this"))

(define-condition tg-inconsistancy (tg-contradiction)
  ()
  (:report "The graph is inconsistant"))

; The entries in the hash table 'chains are arrays of 6 integers repressenting
; 0, the first node of the chain
; 1, the last node of the chain
; 2, the first cross node in the chain with and incoming cross edge
; 3, the last cross node in the chain with an incomind cross edge
; 4, the first node in the chain with an outcoming cross edge
; 5, the last node in the chain with an outcoming cross edge


(defmacro first-node-chain (chain)
  `(svref (get-chain ,chain) 0))

(defmacro last-node-chain (chain)
  `(svref (get-chain ,chain) 1))

(defmacro first_in (chain)
  `(svref (get-chain ,chain) 2))

(defmacro last_in (chain)
  `(svref (get-chain ,chain) 3))

(defmacro first_out (chain)
  `(svref (get-chain ,chain) 4))

(defmacro last_out (chain)
  `(svref (get-chain ,chain) 5))


