;;; Copyright (C) 1994 by Istituto per la Ricerca Scientifica e Tecnologica 
;;; (IRST) (38050 Povo, Trento Italy) and the Trustees of the University 
;;; of Rochester (Rochester, NY 14627, USA).  All rights reserved.

;;; The following is contributed by miller@cs.rochester.edu

;;; NOTE: the full version of this package is distributed as cl-lib.

(in-package tg-ii)              ; usually in our own package.

;; additions for other versions of lisp are welcome!
(defmacro macro-indent-rule (symbol what)
  #-lep nil
  #+lep                         ;must  be 4.1 or later, have lep
  `(lep::eval-in-emacs ,(concatenate 'string "(put '" (string-downcase (string symbol)) " 'fi:lisp-indent-hook " (string-downcase (format nil "'~S)" what)))))


(defmacro while (test &body body)
  "Keeps invoking the body while the test is true;
   test is tested before each loop."
  (let ((end-test (gensym))
	(loop (gensym)))
    `(block nil
       (tagbody (go ,end-test) 
		,loop
		,@body
		,end-test
		(unless (null ,test) (go ,loop))
		(return)))))

(macro-indent-rule while 1)

(defmacro let*-non-null (bindings &body body)
  "like let*, but if any binding is made to NIL, the let*-non-null immediately returns NIL."
#+symbolics  (declare lt:(arg-template ((repeat let)) declare . body))

  `(block lnn (let* ,(mapcar #'process-let-entry bindings)
                    ,@body)))

(macro-indent-rule let*-non-null (like let))

(defun process-let-entry (entry)
  "if it isn't a list, it's getting a nil binding, so generate a return. Otherwise, wrap with test."
  (declare (optimize (speed 3) (safety 0)))

  (if (atom entry)
      `(,entry (return-from lnn nil))
      `(,(car entry) (or ,@(cdr entry) (return-from lnn nil)))))
;;; Explicit tagbody, with end-test at the end, to be nice to poor
;;; compilers.

(defmacro while-not (test &body body)
  "Keeps invoking the body while the test is false;
   test is tested before each loop."
  (let ((end-test (gensym))
	(loop (gensym)))
    `(block nil
       (tagbody (go ,end-test)
		,loop
		,@body
		,end-test
		(unless ,test (go ,loop))
		(return)))))

(macro-indent-rule while-not 1)

;; OK, how many times have you written code of the form
;;
;; (let ((retval (mumble)))
;;    (setf (slot retval) bletch)
;;    (setf (slot retval) barf)
;;    retval)
;;
;; or things of the sort? More than you care to remember most likely. Enter the utterly useful PROGFOO.
;; Think of it as a PROG1 with the value being bound to FOO. inside it's extent Lexically, of course.

(defmacro progfoo (special-term &body body)
  `(let ((foo ,special-term))
     ,@body
     foo))

(macro-indent-rule progfoo (like prog1))

(defmacro with-rhyme (body)
  "Well, there must be rhyme OR reason, and we now admit there is no reason, so...
Used to flag silly constructs that may need to be rewritten for best effect."
  body)

(macro-indent-rule with-rhyme (like progn))

;; and for common lisp fans of multiple values... FOO is the first value, you can access all the values as MV-FOO.
;; returns the multiple values, like multiple-values-prog1

(defmacro mv-progfoo (special-term &body body)
  `(let* ((mv-foo (multiple-value-list ,special-term))
          (foo (car mv-foo)))
     ,@body
     (values-list mv-foo)))

(macro-indent-rule mv-progfoo (like multiple-value-prog1))

;;; The #'eql has to be quoted, since this is a macro. Also, when
;;; binding variables in a macro, use gensym to be safe.
(defmacro update-alist (item value alist &key (test '#'eql) (key '#'identity))
  "If alist already has a value for Key, it is updated to be Value. 
   Otherwise the passed alist is updated with key-value added as a new pair."
  (let ((entry (gensym)))
    `(let ((,entry (assoc ,item ,alist :test ,test :key ,key)))
       (if ,entry
	   (progn (setf (cdr ,entry) ,value)
		  ,alist)
	   (setf ,alist (acons ,item ,value ,alist))))))

(macro-indent-rule update-alist 1)

(DEFUN ROUND-TO (NUMBER &OPTIONAL (DIVISOR 1))
  "Like Round, but returns the resulting number"
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (* (ROUND NUMBER DIVISOR) DIVISOR))
	  
(defmacro msetq (vars value)
#+lispm  (declare (compiler:do-not-record-macroexpansions)
                  (zwei:indentation 1 1))
 `(multiple-value-setq ,vars ,value))

(macro-indent-rule msetq (like multiple-value-setq))

(defmacro mlet (vars value &body body)
#+lispm  (declare (compiler:do-not-record-macroexpansions)
                  (zwei:indentation 1 3 2 1))
   `(multiple-value-bind ,vars ,value ,@body))

(macro-indent-rule mlet (like multiple-value-bind))

