;;;; Extensions to support transfinite fixnum arithmetic
;;;;
;;;; This package provides some arithmetic functions that
;;;; extend fixnums to include the values 'PLUS-INF, 'MINUS-INF,
;;;; and 'UNKNOWN.

(provide 'transfinite)

(in-package 'transfinite :nicknames '(tf))

(export  '(PLUS-INF MINUS-INF UNKNOWN tf+ tf- tf< tfmin tfmax))

(require 'standard "standard")
(use-package 'standard-extensions)


;;;
;;; These symbols use used to augment the built-in numbers
;;;
(intern 'PLUS-INF)
(intern 'MINUS-INF)
(intern 'UNKNOWN)
(defconstant plus-inf 'plus-inf)
(defconstant minus-inf 'minus-inf)
(defconstant unknown 'unknown)

;;;
;;; (tf+ a1 a2 a3...)
;;;
;;; Add the given numbers and return the result.  If any of the 
;;; numbers are 'UNKNOWN, or if there is one or more 'PLUS-INF and
;;; one or more 'MINUS-INF values, the result will be 'UNKNOWN.  
;;; Otherwise, if any of the values are 'PLUS-INF, the result will be
;;; 'PLUS-INF, and similarly, if any of the values are 'MINUS-INF, the 
;;; result will be 'MINUS-INF.
;;;
(defun tf+ (&rest args)
  (let ((unk (member 'UNKNOWN args))
	(pls (member 'PLUS-INF args))
	(mns (member 'MINUS-INF args)))
    (cond ((or unk (and pls mns))
	   'UNKNOWN)
	  (pls
	   'PLUS-INF)
	  (mns
	   'MINUS-INF)
	  (t
	   (apply #'+ args)))))


;;;
;;; (tf- a1 a2 a3...)
;;;
;;; If there is a single parameter, then return it's additive inverse.
;;; Otherwise, return a1 plus the additive inverse of the sum of a2
;;; through ai.
;;;
(defun tf- (a1 &rest arest)
  (cond (arest (tf+ a1 (tf- (apply #'tf+ arest))))
	((eql a1 'UNKNOWN)
	 'UNKNOWN)
	((eql a1 'PLUS-INF)
	 'MINUS-INF)
	((eql a1 'MINUS-INF)
	 'PLUS-INF)
	(t (- a1))))

;;;
;;; (tf< a1 a2)
;;;
;;; Return T just in case a1 is less than a2.
;;;
(defun tf< (a1 a2)
  (or (and (eql a1 'MINUS-INF) 
	   (or (eql a2 'PLUS-INF) (numberp a2)))
      (and (numberp a1) 
	   (or (eql a2 'plus-inf)
	       (and (numberp a2) (< a1 a2))))))

;;;
;;; (tfmin a1 a2 ...)
;;;
;;; Return the transfinite minimum of the a_i.
;;;
(defun tfmin (&rest a-list)
  (let ((smallest (car a-list)))
    (dolist (a (cdr a-list))
      (if (tf< a smallest)
	  (setf smallest a)))
    smallest))


;;;
;;; (tfmax a1 a2 ...)
;;;
;;; Return the transfinite maximum of the a_i.
;;;
(defun tfmax (&rest a-list)
  (let ((largest (car a-list)))
    (dolist (a (cdr a-list))
      (if (not (tf< a largest))
	  (setf largest a)))
    largest))


