;;;; General Purpose Lisp Extensions

(provide 'standard)

(in-package 'standard-extensions)

(export '(exit-when mvb print-spaces one-of dolist-index singletonp random-elt))


;;;
;;; (exit-when condition &optional value))
;;;
;;; Provides a convenient way to terminate (non-extended)
;;; loops.
;;;
(defmacro exit-when (condition &optional value)
  `(when ,condition (return ,value)))

;;;
;;; (mvb ...)
;;;
;;; Replacement for multiple value bind.
;;;
(defmacro mvb (&rest body)
  `(multiple-value-bind ,@body))


;;;
;;; (print-spaces n [stream])
;;;
;;; Print n spaces on the standard output.  Not efficient, but...
;;;
(defun print-spaces (n &optional (stream t))
  (dotimes (i n)
    (format stream " ")))


;;;
;;; (one-of clause-1 clause-2 ... clause-n)
;;;
;;; Randomly evaluates one of the given clauses, and returns the value
;;; of that clause.
;;;
(defmacro one-of (&rest clauses)
  (let ((len (length clauses)))
    (list* 'case
	   (list 'random len)
	   (labels ((constructor (clauses num so-far)
		      (if clauses
			  (constructor (cdr clauses)
				       (1+ num)
				       (cons (list num (car clauses))
					     so-far))
			  so-far)))
	     (constructor clauses 0 '())))))



;;;
;;; (dolist-index (index element item-list [ret-val]) body)
;;;
;;; Iterates over the elements of item-list, with index set to successive
;;; integers.  Kind of a simultaneous dolist and dotimes.
;;;
(defmacro dolist-index (var-specs &rest body)
  (let ((index (first var-specs))
	(element (second var-specs))
	(item-list (third var-specs))
	(ret-val (fourth var-specs))
	(items (gensym)))
    `(do* ((,index 0 (1+ ,index))
	   (,items ,item-list (cdr ,items))
	   (,element (car ,items) (car ,items)))
      ((null ,items)
       ,ret-val)
      ,@body)))



;;;
;;; (singletonp l) -> boolean
;;;
;;; Returns true if the given argument is a list of exactly one element.
;;;
(defun singletonp (l)
  (and (listp l) l (null (cdr l))))


;;;
;;; (random-elt seq) -> elt
;;;
;;; Returns a random element from the list.
;;;
(defun random-elt (seq)
  (elt seq (random (length seq))))

