;;;;
;;;; range.cl -- Transfinite rational ranges
;;;;
;;;; A "range" (nee interval) is a closed, convex set of rational
;;;; numbers, denoted by its two end points.

(provide 'range)

(in-package 'range)

(export '(make-range range-p range-min range-max
	  make-zero-range make-universal-range make-unknown-range
	  empty-range? reverse-range
	  intersect-ranges convolve-ranges))

(require 'transfinite "transfinite")
(use-package 'transfinite)


;;;
;;; The range structure
;;;
;;; The most primitive constraint structure is the "range".  A range
;;; is simply a structure of two transfinite integers, which supports
;;; the obvious interval operations.
;;;
(defstruct (range
	     (:print-function
	      (lambda  (x outstream dpth)
		(if (<= dpth top-level::*print-level*)
		    (format outstream "[~a ~a]"
			    (range-min x)
			    (range-max x)))))
	     (:constructor make-range (min max)))
  min
  max
  )


(defun make-zero-range ()
  (make-range 0 0))

(defun make-universal-range ()
  (make-range 'minus-inf 'plus-inf))

(defun make-unknown-range ()
  (make-range 'unknown 'unknown))


;;;
;;; Operations on ranges
;;;
;;; Note: These should all be done using transfinite math!
;;;

;;;
;;; (empty-range? range)
;;;
;;; A range is empty if the max value is strictly less than the min
;;; value.
;;;
(defun empty-range? (r)
  (tf<  (range-max r) (range-min r)))


;;; 
;;; (reverse-range range) -> range
;;;
;;; Returns the reverse of a range.
;;;
(defun reverse-range (c)
  (make-range
   (tf- (range-max c))
   (tf- (range-min c))))

;;;
;;; (intersect-ranges r1 r2)
;;;
;;; Return a range that is the intersection of the two given ranges.
;;;
(defun intersect-ranges (r1 r2)
  (make-range
   (tfmax (range-min r1) (range-min r2))
   (tfmin (range-max r1) (range-max r2))))

;;;
;;; (convolve-ranges c1 c2)
;;; 
;;; Return the convolution of ranges c1 and c2.
;;;
(defun convolve-ranges (c1 c2)
  (make-range
   (tf+ (range-min c1) (range-min c2))
   (tf+ (range-max c1) (range-max c2))))


