;;;;
;;;; Toy Temporal Projector
;;;;
;;;; To exercise the temporal database manager, specifically the
;;;; interval reasoning facility.  This is built around a simple
;;;; propositional logic, where any proposition is true, false, or
;;;; indeterminate.  The projector is given a sequence of events,
;;;; where each event can have multiple outcomes, depending on what
;;;; conditions are known to hold at the time.  The projector builds a
;;;; chronicle tree which represents all possible outcomes of the
;;;; sequence of actions.
;;;;

(provide 'projector)

(in-package 'projector)

(export '(project random-tp-pairs make-event make-outcome print-chronicle))
(export '(ab-test ab-project make-ab-plan random-queries))

(require 'hic "hic")
(require 'range "range")
(require 'standard "standard")
(use-package 'standard-extensions)

;;;
;;; A tri-state logic is used in which there are three values:
;;; T, NIL, and INDETERMINATE.
;;;

(defconstant indeterminate 'indeterminate)


;;;
;;; The occurrence structure
;;;
;;; This is what each node in the chron-tree consists of.  The outcome
;;; represents the outcome of the event (in the parent of this node)
;;; that this branch in the tree represents.  The interval is the
;;; simple temporal interval over which this event occurs.  Each event
;;; in the chronicle is part of a segment.  A segment is an ordered
;;; sequence interval representing a non-branching portion of the
;;; chronicle tree.  The root of the tree, and each node after a
;;; branch, will be the head of a new segment.  The last interval in a
;;; segment will either be a parallel action sequence (if the node is
;;; a branch point) or a simple interval (if the node is a leaf).  The
;;; event field holds the event that occurs at this node, or is nil in
;;; the case of a leaf.
;;;
(defstruct (occurrence
	     (:print-function (lambda (chron stream depth)
				(format stream "#<Chronicle node ~a>"
					(event-name (occurrence-event chron))))))
  outcome    ; the outcome of the previous event that got us here
  interval   ; interval over which this outcome occurred
  segment    ; interval holding the chronicle segment containing this interval
  event	     ; the event that occurs here
  parent     ; previous event occurrence
  children   ; following event occurrences
  )

(defun occurrence-condition (occ)
  (outcome-condition (occurrence-outcome occ)))

(defun occurrence-effect (occ)
  (outcome-effect (occurrence-outcome occ)))


;;;
;;; (chronicle-printer occ stream depth [indent])
;;;
;;; Print the chronicle rooted at the given occurrence in a pretty
;;; fashion.
;;;
(defun chronicle-printer (occ stream depth &optional (indent 0))
  (when occ
    (print-spaces indent stream)
    (format stream "[time range: ~a ( ~a )~%"
	    (HIC:interval-range (occurrence-interval occ))
	    (HIC:interval-range (occurrence-segment occ)))
    (print-spaces indent stream)
    (format stream "    outcome: ~a -> ~a"
	    (occurrence-condition occ)
	    (occurrence-effect occ))
    (if (occurrence-event occ)
	(progn
	  (format stream "~%")
	  (print-spaces indent stream)
	  (format stream "      event: ~a]~%" (occurrence-event occ)))
	(format stream "]~%"))
    (mapc #'(lambda (oc) (chronicle-printer oc stream depth (+ indent 5)))
	  (occurrence-children occ)))
  (values))


;;;
;;; (print-chronicle occurrence)
;;;
;;; Print the chronicle tree with the given root on the standard
;;; output.
;;;
(defun print-chronicle (occ)
  (chronicle-printer occ t 0))

;;;
;;; Events
;;;
;;; Besides the simple, primitive events, there are also several types
;;; of "complex" events.  They are denoted by lists.  In the
;;; following, "event" refers to any primitive or complex event.
;;;
;;; Sequential event: A list of events to be performed in sequence.
;;;
;;;      (SEQ event1 event2 ...)
;;;
;;; Conditional event: The true-event will be performed iff the given
;;; condition holds, otherwise the false-event will be performed.  In
;;; cases of indeterminacy, the chronicle tree will branch, and both
;;; events will be performed.
;;;
;;;      (IF condition true-event false-event)
;;;
;;; Looping event: The given event will be performed repeatedly as
;;; long as the stated condition holds.  Beware of loops!
;;;
;;;      (WHILE condition event)


;;;
;;; The primitive event
;;;
;;; A primitive event is just a list of "outcomes".
;;;
(defstruct (event  (:print-function (lambda (ev stream depth)
				      (declare (ignore depth))
				      (format stream "#<Event: ~a>" (event-name ev)))))
  name		; symbolic designator for this event
  outcome-list	; list of possible outcomes of this event
  )


;;;
;;; The outcome structure
;;;
;;; An outcome structure describes one of possibly many outcomes that
;;; may result from executing some event.  It contains the conditions
;;; that must hold for that outcome to entail, the effects of that
;;; outcome, and the temporal range that that outcome may require

(defstruct (outcome)
  condition	; the conditions for this outcome to occur
  effect	; the effects of the occurrence of this outcome
  (min-dur 0)	; the minimum duration for this outcome
  (max-dur 0)	; the maximum duration for this outcome
  )



;;;
;;; (project plan world) -> chron-tree
;;; 
;;; Given a plan and a world description, create a chronicle tree
;;; representing the execution of the plan in the given world.
;;; Returns the root of the chronicle tree created.  The given world
;;; description is entered as the effect of the outcome of a virtual
;;; event prior to the first event in the plan.
;;;
(defun project (plan world)
  (let ((chron-tree (make-occurrence :outcome (make-outcome :effect world
							    :min-dur 0
							    :max-dur 0))))
    (setf (occurrence-interval chron-tree)
	  (HIC:create-interval chron-tree 'initial-outcome
			       :simple 0 0))
    (setf (occurrence-segment chron-tree)
	  (HIC:create-interval chron-tree 'chronicle-tree
			       :ordered (list (occurrence-interval chron-tree))))
    (projector plan chron-tree)
    chron-tree))

;;;
;;; (chronicle-duration ct) -> num
;;;
;;; Returns the duration of the given chronicle tree.
;;;
(defun chronicle-duration (ct)
  (HIC:interval-range (occurrence-segment ct)))


;;;
;;; (projector plan chron-leaf) 
;;;
;;; Project a plan in a chronicle leaf.  The given chronicle leaf has
;;; only the outcome slot supplied.  The plan is a list of events.
;;; The first event in the plan is made to occur in the given
;;; chronicle leaf, which is expanded to have a child for each
;;; possible outcome.  The cdr of the plan is then recursively
;;; projected in each child chronicle leaf.
;;;
(defun projector (plan chron-leaf)
  (if plan
      (let* ((event (car plan))
	     (remaining-plan (cdr plan)))
	; (format t "Projecting ~a~%" event)
	(cond
	  ;; Primitive events
	  ((primitive-event? event)
	   (project-primitive-event event chron-leaf)
	   (incorporate-child-intervals  chron-leaf)
	   (project-in-children remaining-plan chron-leaf))
	  ;; Sequential events
	  ((and (listp event)
		(eql (car event) 'SEQ))
	   (projector (append (cdr event) remaining-plan) chron-leaf))
	  ;; Conditional events
	  ((and (listp event)
		(eql (car event) 'IF))
	   (project-conditional-event (second event) (third event) (fourth event)
				      chron-leaf remaining-plan))
	  (t
	   (error "Something weird in plan!"))
	  ))))

;;;
;;; (project-primitive-event event chron-leaf)
;;;
;;; Expand the given chronicle leaf into a chronicle node by adding a
;;; child for each possible outcome of the given event.
;;;
(defun project-primitive-event (event chron-leaf)
  (setf (occurrence-event chron-leaf) event)
  (dolist (outcome (event-outcome-list event))
    (when (possible? (assess (outcome-condition outcome) chron-leaf))
      (let ((child (make-occurrence
		    :outcome outcome
		    :parent chron-leaf)))
	(setf (occurrence-interval child)
	      (HIC:create-interval child 'outcome
				   :simple (outcome-min-dur outcome) (outcome-max-dur outcome)))
	(push child (occurrence-children chron-leaf))))))


;;;
;;; (incorporate-child-intervals chron-node)
;;;
;;; Incorporate the intervals of the children into the interval
;;; hierarchy.  If there is only one child, then that child will be
;;; added to the current chron-tree segment.  If there is more than
;;; one child, then a new segment will be started for each child,
;;; these segments will be collected together into a selection
;;; interval, and that selection interval will be added as the final
;;; interval of the current segment.
;;;
;;;
(defun incorporate-child-intervals (chron-node)
  (let* ((interval (occurrence-interval chron-node))
	 (segment (occurrence-segment chron-node))
	 (children (occurrence-children chron-node)))
    (cond
      ((singletonp children)
       (setf (occurrence-segment (first children)) segment)
       (add-to-segment segment (occurrence-interval (first children))))
      (t
       (dolist (child children)
	 (setf (occurrence-segment child)
	       (HIC:create-interval child 'segment
				    :ordered (list (occurrence-interval child)))))
       (add-to-segment segment
		       (HIC:create-interval
			chron-node 'branch :selection
			(mapcar #'(lambda (oc) (occurrence-segment oc))
				children)))))))


;;;
;;; (add-to-segment segment interval)
;;;
;;; Add a new interval to the end of a segment (which is an ordered
;;; sequence interval).
;;;
(defun add-to-segment (segment interval)
  (let ((current-contents (HIC:interval-sub-intervals segment)))
    (HIC:revise-interval
     segment :sub-intervals (append current-contents  (list interval)))))


;;;
;;; (project-in-children (plan chron-node)
;;;
;;; Given a plan and a chronicle tree node, project the given plan in
;;; each child of the specified node.  Note that each such child
;;; should be a leaf.
;;;
(defun project-in-children (plan chron-node)
  (mapc #'(lambda (cl) (projector plan cl))
	(occurrence-children chron-node)))
			    

;;;
;;; (project-conditional-event condition true-ev false-ev chron-leaf remaining-plan)
;;;
;;; Determine whether the given condition is true, false, or
;;; indeterminate.  If indeterminate, form a fork in the chron-tree.
;;; Then, project either the true-event or the false-event (as
;;; appropriate) and the remaining plan in the appropriate chron leaves.
;;;
(defun project-conditional-event (condition true-ev false-ev chron-leaf remaining-plan)
  (let ((assessment (assess condition chron-leaf)))
    (if (determinate? assessment)
	(projector (cond ((and assessment true-ev)
			  (cons true-ev remaining-plan))
			 ((and (not assessment) false-ev)
			  (cons false-ev remaining-plan))
			 (t
			  remaining-plan))
		   chron-leaf)
	(let* ((true-outcome (make-outcome :condition condition))
	       (false-outcome (make-outcome :condition (list 'NOT condition)))
	       (test-event (make-event :name 'IF-TEST
				       :outcome-list (list true-outcome false-outcome)))
	       (true-leaf (make-occurrence :outcome true-outcome :parent chron-leaf))
	       (false-leaf (make-occurrence :outcome false-outcome :parent chron-leaf)))
	  (setf (occurrence-interval true-leaf)
		(HIC:create-interval true-leaf 'if-test-outcome :simple 0 0))
	  (setf (occurrence-interval false-leaf)
		(HIC:create-interval true-leaf 'if-test-outcome :simple 0 0))		
	  (setf (occurrence-event chron-leaf) test-event)
	  (setf (occurrence-children chron-leaf) (list true-leaf false-leaf))
	  (incorporate-child-intervals chron-leaf)
	  (projector
	   (if true-ev
	       (cons true-ev remaining-plan)
	       remaining-plan)
	   true-leaf)
	  (projector
	   (if false-ev
	       (cons false-ev remaining-plan)
	       remaining-plan)
	   false-leaf)))))
	  

;;;
;;; (primitive-event? event) -> boolean
;;;
;;; Returns true iff the given event is a simple, primitive event
;;; structure.
;;;
(defun primitive-event? (event)
  (event-p event))


;;;
;;; (assess clause chron-node) -> logic-value
;;;
;;; Determines whether the given clause is true, false, or
;;; indeterminate in the given chronicle.
;;;
(defun assess (clause chron-node)
  (cond
    ((null chron-node)
     INDETERMINATE)
    ((symbolp clause)
     (let ((local-effect (justified? clause (occurrence-effect chron-node))))
       (if (determinate? local-effect)
	   local-effect
	   (let ((local-condition (justified? clause (occurrence-condition chron-node))))
	     (if (determinate? local-condition)
		 local-condition
		 (assess clause (occurrence-parent chron-node)))))))
    ((equal (car clause) 'not)
     (tri-not (assess (second clause) chron-node)))
    ((equal (car clause) 'and)
     (apply #'tri-and (mapcar #'(lambda (cl) (assess cl chron-node)) (cdr clause))))
    ((equal (car clause) 'or)
     (apply #'tri-or (mapcar #'(lambda (cl) (assess cl chron-node)) (cdr clause))))
    (t
     (error "Unrecognized stuff in clause: ~D" clause))))


(defun justified? (symbol clause)
  (cond
    ((member symbol '(t nil indeterminate))
     symbol)
    ((null clause)
     indeterminate)
    ((symbolp clause)
     (if (equal symbol clause)
	 t
	 indeterminate))
    ((equal (car clause) 'and)
     (let* ((justs (mapcar #'(lambda (cl) (justified? symbol cl)) (cdr clause)))
	    (t-just (member t justs))
	    (f-just (member nil justs)))
       (cond
	 ((and t-just (not f-just))
	  t)
	 ((and f-just (not t-just))
	  nil)
	 (t
	  indeterminate))))
    ((equal (car clause) 'not)
     (tri-not (justified? symbol (second clause))))
    (t
     indeterminate)))


;;;;
;;;; A bunch of stuff to support the tri-state logic...
;;;;

;;;
;;; (possible? logic-value) -> boolean
;;;
;;; Returns true if the given tri-state logic value is either
;;; true or indeterminate.
;;;
(defun possible? (value)
  (or (equal value T) (equal value INDETERMINATE)))

(defun indeterminate? (value)
  (equal value INDETERMINATE))

(defun determinate? (value)
  (not (indeterminate? value)))


(defun tri-not (value)
  (if (indeterminate? value)
      value
      (not value)))

(defun tri-and (&rest values)
  (cond
    ((member nil values)
     nil)
    ((member indeterminate values)
     indeterminate)
    (t
     t)))

(defun tri-or (&rest values)
  (cond
    ((member t values)
     t)
    ((member indeterminate values)
     indeterminate)
    (t
     nil)))

    
;;;
;;; (chronicle-leaves occurrence) -> list of occurrences
;;;
;;; Returns a list of all the leaves of the chronicle tree specified
;;; by the given root.
;;;
(defun chronicle-leaves (occurrence)
  (cond
    ((null occurrence) nil)
    ((null (occurrence-children occurrence))
     (list occurrence))
    (t
     (apply #'nconc (mapcar #'chronicle-leaves
			     (occurrence-children occurrence))))))


;;;
;;; (path-to-leaf chron-leaf) -> list
;;;
;;; Given a chronicle leaf, returns a list of the chronicle nodes that
;;; make up the path from the root of the chronicle tree to that leaf.
;;;
(defun path-to-leaf (chron-leaf)
  (do* ((node chron-leaf (occurrence-parent node))
	(path '()))
       ((null node) path)
    (push node path)))


;;;
;;; (random-tp-pairs chron num) -> list
;;;
;;; Returns a list of <num> pairs of timepoints from the given
;;; chronicle.  The two timepoints in each pair will come from a
;;; single path in the chronicle tree.
;;;
(defun random-tp-pairs (chron num)
  (let* ((leaves (chronicle-leaves chron))
	 (paths (mapcar #'path-to-leaf leaves))
	 (pairs '()))
    (dotimes (i num)
      (let* ((path (random-elt paths))
	     (int1 (occurrence-interval (random-elt path)))
	     (int2 (occurrence-interval (random-elt path))))
	(push (list (one-of
		     (HIC:beginning-of int1)
		     (HIC:end-of int1))
		    (one-of
		     (HIC:beginning-of int2)
		     (HIC:end-of int2)))
	      pairs)))
    pairs))


;;;
;;; (random-queries ct n)
;;;
;;; Perform n random queries in the given chronicle tree, and display
;;; timing information.
;;;
(defun random-queries (ct n)
  (format t "HIC Performing ~a random queries...~%" n)
  (let ((pair-list (random-tp-pairs ct n)))
    (excl:gc t)
    (time
     (dolist (pair pair-list)
       (HIC:get-distance (first pair) (second pair))))))

;;;
;;; (chronicle-num-nodes ct) -> int
;;;
;;; Returns the number of occurrence nodes in the chronicle tree
;;; specified by the given root.
;;;
(defun chronicle-num-nodes (ct)
  (cond ((null ct)
	 0)
	(t
	 (apply #'+ 1 (mapcar #'chronicle-num-nodes (occurrence-children ct))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; A simple test domain to allow construction of arbitrarily branching
;;;; chronicle trees.
;;;;

(setf op-a
      (make-event :name 'operation-a
		  :outcome-list
		  (list (make-outcome
			 :condition T
			 :effect 'effect-a
			 :min-dur 5
			 :max-dur 10))))

(setf op-b
      (make-event :name 'operation-b
		  :outcome-list
		  (list (make-outcome
			 :condition indeterminate
			 :effect 'effect-b-1
			 :min-dur 3
			 :max-dur 6)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-b-2
			 :min-dur 4
			 :max-dur 8))))


(setf op-c
      (make-event :name 'operation-c
		  :outcome-list
		  (list (make-outcome
			 :condition indeterminate
			 :effect 'effect-c-1
			 :min-dur 3
			 :max-dur 6)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-c-2
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-c-3
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-c-4
			 :min-dur 4
			 :max-dur 8))))

			
(setf op-d
      (make-event :name 'operation-d
		  :outcome-list
		  (list (make-outcome
			 :condition indeterminate
			 :effect 'effect-d-1
			 :min-dur 3
			 :max-dur 6)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-2
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-3
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-4
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-5
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-6
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-7
			 :min-dur 4
			 :max-dur 8)
			(make-outcome
			 :condition indeterminate
			 :effect 'effect-d-8
			 :min-dur 4
			 :max-dur 8))))


(defun make-ab-plan (ab-list)
  (mapcar #'(lambda (sym)
	      (cond ((eq sym :a) op-a)
		    ((eq sym :b) op-b)
		    ((eq sym :c) op-c)
		    ((eq sym :d) op-d)
		    (t (error "Illegal symbol to make-ab-plan"))))
	  ab-list))


(defun ab-project (ab-list)
  (let ((ab-plan (make-ab-plan ab-list))
	(ct))
    (format t "HIC Projecting the plan ~a...~%" ab-list)
    (time
     (setf ct (project ab-plan '())))
    (format t "The plan has ~a nodes.~%" (chronicle-num-nodes ct))
    (format t "The overall duration is ~a~%" (chronicle-duration ct))
    ct))


(defun ab-test (ab-list num-queries)
  (random-queries (ab-project ab-list) num-queries))

;;;
;;; You can test the projector by saying, e.g., 
;;;    (ab-test '(:a :b :a :a) 10)
;;;