;;;;
;;;; hic.cl -- Hierarchical Interval Constraints for temporal reasoning
;;;;
;;;; A timeline is a graph of timepoints connected by intervals.  Each
;;;; timepoints may be attached to any number of intervals, while each
;;;; interval has at most one timepoint at each end.
;;;;
;;;; Note that this is a slightly unusual graph in that it is possible
;;;; to have edges (intervals) without necessarily having vertices
;;;; (timepoints) at both ends.
;;;;
;;;; Timepoints and intervals are both available as primitive elements
;;;; to the user.  Each element has an associated "contents" and
;;;; "type" The contents may be anything the user likes.  The type
;;;; should be a single symbol, and may be used in some queries to
;;;; improve efficiency.  If typing is not desired, the type can be
;;;; left as nil.
;;;;

(provide 'hic)

(in-package 'hic)

;;;
;;; The main database manipulation functions
;;;
(export '(hic-initialize remove-timeline pp-timeline
	  create-timepoint add-constrained-timepoint add-time-constraint
	  create-interval revise-interval attach-interval interval-range
	  get-distance get-timepoints beginning-of end-of))

(export '(constraint-min constraint-max constraint-from constraint-to))

(export '(timepoint-contents timepoint-type
	  interval-contents interval-type interval-kind interval-sub-intervals))

;;;
;;; Other packages required
;;;

(require 'standard "standard")
(use-package 'standard-extensions)

(require 'transfinite "transfinite")
(use-package 'transfinite)

(require 'range "range")
(use-package 'range)

;;;
;;; The interval structure
;;;
;;; An interval is a heirarchical connection between two timepoints.
;;; There are five types:
;;;   :simple
;;;   :ordered
;;;   :unordered
;;;   :parallel
;;;   :selection
;;;
(defstruct (interval
	     (:print-function
	      (lambda (x outstream dpth)
		(if (<= dpth top-level::*print-level*)
		    (format outstream "#{HIC [~a ~a] from ~a to ~a}"
			    (interval-min x)
			    (interval-max x)
			    (interval-from x)
			    (interval-to x))))))
  contents	  ; contents are anything supplied by the user
  type		  ; type is user-defined; used for efficient access
  (kind :simple)  ; what kind of interval this is
  timeline        ; what timeline this interval is in (only for explicit intervals)
  sub-intervals	  ; if a complex interval, a list of it's children
  super-interval  ; if a sub-interval, the containing parent
  depth           ; max depth of nested sub-intervals; zero for simple interval
  from		  ; beginning timepoint
  to		  ; ending timepoint
  range		  ; range of duration of interval.  this is given for a simple
                  ;    interval, and calculated for a complex interval.
  )

(defun interval-min (interval)
  (range-min (interval-range interval)))
    
(defun interval-max (interval)
  (range-max (interval-range interval)))
    
(intern :simple)
(intern :ordered)
(intern :unordered)
(intern :parallel)
(intern :selection)

(intern :begin)
(intern :end)

;;;
;;; The virtual timepoint data structure
;;;
;;; The beginning and end of each interval comprises a virtual
;;; timepoint which may be used in determining the distance to other
;;; [virtual] timepoints.  This is really just a symbol to indicate
;;; the type (beginning or ending), and a pointer to the interval.
;;;
;;;
(defstruct (virtual-timepoint
	     (:conc-name vtp-)
	     (:constructor make-vtp (type interval)))
  type	     ; either :begin or :end
  interval   ; the associated interval
  )

(defun beginning-of (interval)
  (make-vtp :begin interval))

(defun end-of (interval)
  (make-vtp :end interval))


;;; 
;;; The timepoint data structure
;;; 
;;; Each timepoint contains two fields supplied by the user (contents
;;; and type).  Each timepoint is in a single timeline, and maintains a
;;; pointer to the timeline structure that it is a member of.
;;; 
;;; The timepoints in a timeline are connected together in a graphical
;;; structure via intervals, which each timepoint maintains a list of.
;;; It is expected that most timelines will usually be trees, which can
;;; be evaluated much more quickly, so there is special structure to
;;; handle that case.  The graph is explicitly maintained as an
;;; augmented spanning tree; the arcs at each vertex are represented as
;;; a back edge, a list of forward edges, and a list of cross edges.
;;; If the graph is itself a tree, then all cross-edge lists will be
;;; empty.
;;;
;;; Note that there may be "free intervals" which have a timepoint at
;;; only one end.  These are kept in a separate list for that
;;; timepoint, to easily differentiate them from other forward edges.
;;;
(defstruct (timepoint
	     (:print-function 
	      (lambda (x outstream dpth)
		(if (<= dpth top-level::*print-level*)
		    (format outstream "#{Timepoint: ~a}" (timepoint-contents x))))))
  contents         ; contents are anything supplied by user
  type             ; contents-type is user defined; allows more efficient querying
  timeline         ; which timeline this timepoint is in
  back-edge        ; the back edge in the spanning tree
  forward-edges    ; list of forward edges in the spanning tree
  cross-edges      ; list of cross edges in the graph
  free-edges       ; list of intervals without timepoints at other end
  root-dist        ; physical distance from root of tree (# of edges)
  index		   ; index into timepoint table (used for distance calculation)
  )


;;;
;;; The timeline data structure
;;;
;;; The set of all timepoints consists of some number of disjoint
;;; timelines.  Each timeline is a set of interconstrained timepoints,
;;; and is identified by a single canonical timepoint, the root.
;;;
(defstruct (timeline)
  root		   ; the canonical timepoint
  (num-points 0)   ; the number of timepoints in the timepoint list
  point-list	   ; the list of timepoints in this timeline
  interval-list    ; the list of explicit intervals in this timeline
  type-alist       ; an association list linking types with points of those types
  non-tree-edges   ; a list of constraints that are not part of the spanning tree
  solution	   ; a matrix giving the transitive closure of the graph
  )		    


;;; 
;;; Timeline list
;;;
;;; All timelines extant in the system are kept in this global list
;;;

(defvar *timeline-list* '())

;;;
;;;--------------------------------------------------------
;;; Main HIC interface functions
;;;

;;;
;;; (hic-initialize)
;;;
;;; Initialize the HIC to a pristine condition
;;;
(defun hic-initialize ()
  (setf *timeline-list* '())
  (values))


;;;
;;; (remove-timeline tl-or-tp)
;;;
;;; Remove the specified timeline (or the timeline containing the
;;; specified timepoint) from the database.  It is suggested that the
;;; user make no further references to any of the timepoints in that
;;; timeline...
;;;
(defun remove-timeline (tl)
  (setf *timeline-list*
	(delete (if (timepoint-p tl)
		    (timepoint-timeline tl)
		    tl)
		*timeline-list*))
  (values))


;;;
;;; (create-timepoint contents type) -> timepoint
;;;
;;; Create a new timepoint, and insert it into a new timeline.
;;;
(defun create-timepoint (contents type)
  (let* ((tp (make-timepoint :contents contents 
			     :type type
			     :root-dist 0))
	 (tl (make-timeline :root tp)))
    (add-timepoint-to-timeline tl tp)
    (push tl *timeline-list*)
    tp))


;;;
;;; (add-constrained-timepoint contents type parent-tp mint maxt) -> timepoint
;;;
;;; Create a new timepoint with the given contents and type, and add it to 
;;; the same timeline as the given timepoint.  
;;;
;;; NOTE: The constraint specified is FROM the existing timepoint TO the 
;;; new timepoint!
;;;
(defun add-constrained-timepoint (contents type parent-tp mint maxt)
  (let* ((tl (timepoint-timeline parent-tp))
	 (tp (make-timepoint 
	      :contents contents
	      :type type
	      :root-dist (1+ (timepoint-root-dist parent-tp))))
	 (constraint (make-constraint parent-tp tp mint maxt)))
    (setf (timepoint-back-edge tp) constraint)
    (push constraint (timepoint-forward-edges parent-tp))
    (add-timepoint-to-timeline tl tp)
    tp))


;;;
;;; (add-time-constraint tp1 tp2 mint maxt)
;;;
;;; Add the given constraint between the two timepoints.  If the
;;; points are already in the same timeline, then that timeline will
;;; no longer be a tree.  However, if the two timepoints are not in
;;; the same timeline, then their two timelines are merged into one,
;;; which will still be a tree iff the two constituents are both
;;; trees.
;;;
(defun add-time-constraint (tp1 tp2 mint maxt)
  (let ((tl1 (timepoint-timeline tp1))
	(tl2 (timepoint-timeline tp2))
	(constraint (make-constraint tp1 tp2 mint maxt)))
    (if (eq tl1 tl2)
	(add-constraint-in-timeline tl1 tp1 tp2 constraint)
	(merge-timelines tl1 tl2 tp1 tp2 constraint)))
  (values))


;;;
;;; (get-distance tp1 tp2) --> range
;;;
;;; Determine the tightest constraint from tp1 to tp2 and return it as
;;; a range.  Either or both of tp1 and tp2 may be virtual timepoints.
;;;
(defun get-distance (tp1 tp2)
  (cond
    ((virtual-timepoint-p tp1)
     (get-distance-vtp tp1 tp2))
    ((virtual-timepoint-p tp2)
     (reverse-range (get-distance-vtp tp2 tp1)))
    ((not (eql (timepoint-timeline tp1)
	       (timepoint-timeline tp2)))
     (make-unknown-range))
    ((timeline-non-tree-edges (timepoint-timeline tp1))
     (get-distance-graph tp1 tp2))
    (t
     (get-distance-tree tp1 tp2))))


;;;
;;; (get-timepoints tp min max &optional tp-type) -> list
;;;
;;; Returns a list of all the timepoints of type tp-type that could
;;; potentially lie within the range [min max] from tp.
;;;
(defun get-timepoints (tp min max &optional (tp-type nil tp-type-given))
  (let* ((tl (timepoint-timeline tp))
	 (in-range (make-range min max))
	 (possible-points (if tp-type-given
			      (timeline-points-of-type tl tp-type)
			      (timeline-point-list tl)))
	 (result-points '()))
    (dolist (point possible-points)
      (if (not (empty-range?
		(intersect-ranges in-range (get-distance tp point))))
	  (push point result-points)))
    result-points))


;;;
;;; (pp-timeline tp-or-tl)
;;;
;;; Makes a pretty printout of the timeline, or the timeline
;;; containing the given timepoint.
;;;
(defun pp-timeline (tl)
  (tl-printer (timeline-root
	       (if (timepoint-p tl)
		   (timepoint-timeline tl)
		   tl))
	      (make-zero-range)
	      0)
  (values))


;;;
;;; (create-interval contents type kind subs-or-min [max]) -> interval
;;;
;;; Creates an interval with the given contents, type, and kind.  If
;;; the kind is :simple, then min and max specify the range.
;;; Otherwise, a list of sub-intervals is given.
;;;
(defun create-interval (contents type kind subs-or-min &optional max)
  (let ((sub-intervals subs-or-min)
	(min subs-or-min))
    (if (eql kind :simple)
	(make-interval :contents contents
		       :type type
		       :kind :simple
		       :range (make-range min max)
		       :depth 0)
	(let ((interval (make-interval :contents contents
				       :type type
				       :kind kind
				       :sub-intervals sub-intervals)))
	  (dolist (sub-int sub-intervals)
	    (setf (interval-super-interval sub-int) interval))
	  (update-complex-interval-depth-and-range interval)
	  interval))))


;;;
;;; (revise-interval interval &key contents type kind sub-intervals min max)
;;;
;;; Potentially revises the given interval to have new contents, type,
;;; kind, and/or sub-intervals.  This requires updating the depth and
;;; range of this interval and any super-intervals.
;;;
(defun revise-interval (interval &key (contents nil contents-given)
				      (type nil type-given)
				      (kind nil kind-given)
				      (sub-intervals nil sub-intervals-given)
				      (min nil min-given)
				      (max nil max-given))
  (when (or (and (eql kind :simple)
		 (or (not min-given)
		     (not max-given)
		     sub-intervals-given))
	    (and kind-given
		 (complex-kind? kind)
		 (not sub-intervals-given)
		 (not (interval-sub-intervals interval)))
	    (and sub-intervals-given
		 (not (complex-kind? kind))
		 (not (complex-kind? (interval-kind interval)))))
    (error "Illegal interval revision."))
  (when contents-given (setf (interval-contents interval) contents))
  (when type-given (setf (interval-type interval) type))
  (when kind-given (setf (interval-kind interval) kind))
  (when (eql kind :simple)
    (setf (interval-sub-intervals interval) nil)
    (setf (interval-range interval) (make-range min max)))
  (when sub-intervals-given
    (setf (interval-sub-intervals interval) sub-intervals)
    (dolist (sub-int sub-intervals)
      (setf (interval-super-interval sub-int) interval))
    (update-complex-interval-depth-and-range interval))
  (values))  



;;;
;;; (attach-interval interval from-tp to-tp)
;;;
;;; Given an interval and from zero to two timepoints (nil may be
;;; given for either/both timepoints), connect the interval to the
;;; timepoints.  If it is attatched to a single timepoint, then it
;;; will become a free-edge of that timepoint, whereas if it is
;;; connected to two timepoints, then it will form a constraint
;;; between those two timepoints, possibly merging their timelines.
;;; If at least one timepoint is given, the interval will be added to
;;; the interval list of the associated timeline.
;;;
;;; As a special case, this function can be used to attach the other
;;; end of an interval which has already had one end attached.  In
;;; this case, the same timepoint *must* be given for the previously
;;; attached end.  All we need to do is remove it from the old free
;;; edge list.  It will either remain in the same timeline, or be
;;; placed into a new timeline, in which case the previous timeline
;;; will go away.
;;;
;;; Note: The given interval may not be a sub-interval of some complex
;;; interval.
;;; 
(defun attach-interval (interval from-tp to-tp)
  (when (interval-super-interval interval)
    (error "Cannot attach sub-interval to timepoints."))
  (let ((prev-from (interval-from interval))
	(prev-to (interval-to interval)))
    (when (or (and prev-from (not (eql prev-from from-tp)))
	      (and prev-to (not (eql prev-to to-tp))))
      (error "Cannot reattach/detach interval"))
    
    ;; Remove from free edge lists.
    (when prev-from
      (setf (timepoint-free-edges prev-from)
	    (remove interval (timepoint-free-edges prev-from))))
    (when prev-to
      (setf (timepoint-free-edges prev-to)
	    (remove interval (timepoint-free-edges prev-to)))))

  (setf (interval-from interval) from-tp)
  (setf (interval-to interval) to-tp)
  (let ((two-tps (and from-tp to-tp))
	(single-tp (or from-tp to-tp)))
    (cond
      (two-tps
       (let ((from-tl (timepoint-timeline from-tp))
	     (to-tl (timepoint-timeline to-tp)))
	 (if (eq from-tl to-tl)
	     (add-constraint-in-timeline from-tl from-tp to-tp interval)
	     (merge-timelines from-tl to-tl from-tp to-tp interval))
	 (add-interval-to-timeline from-tl interval)))
      (single-tp
       (push interval (timepoint-free-edges single-tp))
       (add-interval-to-timeline (timepoint-timeline single-tp) interval))))
  (values))

;;;
;;;--------------------------------------------------------
;;; HIC Internal Functions
;;;

;;;
;;; (add-timepoint-to-timeline tl tp)
;;; 
;;; Add the specified timepoint to the specified timeline, iff it is
;;; not already a member thereof.
;;;
(defun add-timepoint-to-timeline (tl tp)
  (when (not (eql tl (timepoint-timeline tp)))
    (setf (timepoint-timeline tp) tl)
    (push tp (timeline-point-list tl))
    (incf (timeline-num-points tl))
    (flush-solution-cache tl)
    (let* ((tp-type (timepoint-type tp))
	   (type-list (assoc tp-type (timeline-type-alist tl))))
      (if type-list
	  (push tp (cdr type-list))
	  (push (list tp-type tp) (timeline-type-alist tl))))
    (values)))


;;;
;;; (add-interval-to-timeline tl interval)
;;;
;;; Add the given interval to the given timeline, iff it is not
;;; already a member thereof.
;;;
(defun add-interval-to-timeline (tl interval)
  (when (not (eql tl (interval-timeline interval)))
    (setf (interval-timeline interval) tl)
    (push interval (timeline-interval-list tl))))


;;;
;;; (timeline-points-of-type tl type) -> list
;;;
;;; Returns a list of all the points in the timeline of the specified type.
;;;
(defun timeline-points-of-type (tl type)
  (cdr (assoc type (timeline-type-alist tl))))


;;;
;;; (add-constraint-in-timeline tl tp1 tp2 constraint)
;;;
;;; Add the given constraint between two timepoints within the
;;; same timeline.
;;;
(defun add-constraint-in-timeline (tl tp1 tp2 constraint)
  (push constraint (timepoint-cross-edges tp1))
  (push constraint (timepoint-cross-edges tp2))
  (push constraint (timeline-non-tree-edges tl))
  (flush-solution-cache tl)
  (values))


;;;
;;; (merge-timelines tl1 tl2 tp1 tp2 constraint)
;;;
;;;  Add the given constraint between two timepoints which are in
;;;  different timelines.  The two timelines are merged into one by
;;;  incorporating the second into the first.  The second timeline
;;;  will thereafter cease to exist.
;;;
(defun merge-timelines (tl1 tl2 tp1 tp2 constraint)
  (setf (timeline-non-tree-edges tl1)
	(nconc (timeline-non-tree-edges tl1)
	       (timeline-non-tree-edges tl2)))
  (remove-timeline tl2)
  (push constraint (timepoint-forward-edges tp1))
  (hang-timepoint tp2 constraint (timepoint-root-dist tp1) tl1)
  (values))


;;;
;;; (hang-timepoint tp from-constraint from-dist new-tl)
;;;
;;; Moves tp and all it's neighbors (except the one at the other end
;;; of from-constraint into a new timeline, with a new root, etc.
;;; From-constraint is the constraint to the (new) parent of this
;;; timepoint, and will be used to set the back-edge of the timepoint.
;;; From-dist is the distance to the root from the (new) parent.
;;;
;;; Note: if tp is the root, then it's back edge will be nil.  We need
;;; to check for that.
;;;
(defun hang-timepoint (tp from-constraint from-dist new-tl)
  (incf from-dist)
  (let ((relatives (if (timepoint-back-edge tp)
		       (cons (timepoint-back-edge tp)
			     (timepoint-forward-edges tp))
		       (timepoint-forward-edges tp)))
	(new-children '()))
    (add-timepoint-to-timeline new-tl tp)
    (dolist (edge relatives)
      (when (and edge
		 (not (eql edge from-constraint)))
	(push edge new-children)
	(hang-timepoint (constraint-to tp edge) edge from-dist new-tl)))
    (setf (timepoint-root-dist tp) from-dist
	  (timepoint-back-edge tp) from-constraint
	  (timepoint-forward-edges tp) new-children))
  (values))
    
  
;;;
;;; (get-distance-tree tp1 tp2) -> range
;;;
;;; tp1 and tp2 are timepoints in the same timeline, which is a tree.
;;; Return the constraint interval from tp1 to tp2.
;;;
(defun get-distance-tree (tp1 tp2)
  (let ((from-tp1 (make-zero-range))
	(from-tp2 (make-zero-range)))
    (loop
     (when (eql tp1 tp2)
       (return (convolve-ranges from-tp1 (reverse-range from-tp2))))
     (if (> (timepoint-root-dist tp1) (timepoint-root-dist tp2))
	 (multiple-value-setq
	  (from-tp1 tp1)
	  (accrue-range from-tp1 tp1 (timepoint-back-edge tp1)))
	 (multiple-value-setq
	  (from-tp2 tp2)
	  (accrue-range from-tp2 tp2 (timepoint-back-edge tp2)))))))
	  

;;;
;;; (get-distance-graph tp1 tp2) -> range
;;;
;;; tp1 and tp2 are timepoints in the same timeline, which is an
;;; arbitrary graph.  Returns the tightest interval constraint from
;;; tp1 to tp2.  Just an expensive but easy n^3 algorithm.  Yuk.
;;;
(defun get-distance-graph (tp1 tp2)
  (let ((tl (timepoint-timeline tp1)))
    (when (not (timeline-solution tl))
      (solve-timeline tl))
    (aref (timeline-solution tl) (timepoint-index tp1) (timepoint-index tp2))))


;;;
;;; (solve-timeline tl)
;;;
;;; Makes an array containing the tightest constraint between each
;;; pair of timepoints in the timeline.
;;;
;;; This is a straight-forward implementation of the Floyd-Warshall
;;; algorithm, and runs in N^3 time.
;;;
(defun solve-timeline (tl)
  (let* ((np (timeline-num-points tl))
	 (tp-table (make-array np))
	 (dist (make-array (list np np)
			   :initial-element (make-universal-range)))
	 (index 0))

    ;; Initialize indices
    (dolist (tp (timeline-point-list tl))
      (setf (timepoint-index tp) index)
      (setf (aref tp-table index) tp)
      (incf index))

    ;; Initialize distance array
    (dotimes (from-index np)
      (let ((from-tp (aref tp-table from-index)))
	(setf (aref dist from-index from-index) (make-zero-range))
	(dolist (edge-to (timepoint-all-edges from-tp))
	  (setf (aref dist from-index (timepoint-index (constraint-to from-tp edge-to)))
		(constraint-range from-tp edge-to)))))

    ;; Do the calculation
    (dotimes (w np)
      (dotimes (u np)
	(dotimes (v np)
	  (setf (aref dist u v)
		(intersect-ranges
		 (aref dist u v)
		 (convolve-ranges
		  (aref dist u w)
		  (aref dist w v)))))))

    ;; Save the solution in the timeline structure
    (setf (timeline-solution tl) dist))
  (values))


;;;
;;; (flush-solution-cache tl)
;;;
;;; Flushes the solution cache for the given timeline.
;;; This mush be done whenever the timepoint graph changes.
;;;
(defun flush-solution-cache (tl)
  (setf (timeline-solution tl) nil))


;;;
;;; ----- Complex interval support -----
;;;


;;;
;;; (update-complex-interval-depth-and-range interval)
;;;
;;; Updates the depth and range of a given (complex) interval according to
;;; the appropriate values in it's sub-intervals.
;;;
(defun update-complex-interval-depth-and-range (interval)
  (setf (interval-depth interval)
	(1+ (apply #'max (mapcar #'interval-depth (interval-sub-intervals interval)))))
  (set-complex-interval-range interval)
  (when (interval-timeline interval)
    (flush-solution-cache (interval-timeline interval)))
  (when (interval-super-interval interval)
    (update-complex-interval-depth-and-range (interval-super-interval interval)))
  (values))


;;;
;;; (set-complex-interval-range interval)
;;;
;;; Given a complex interval, set the range (according to the kind of
;;; the interval), by inspection of the ranges of the sub-intervals.
;;;
(defun set-complex-interval-range (interval)
  (setf (interval-range interval)
	(make-range (complex-interval-min interval)
		    (complex-interval-max interval)))
  (values))


;;;
;;; (complex-interval-min interval) -> number
;;;
;;; Return the minimum range of a complex interval, by inspecting
;;; its sub-intervals.
;;;
(defun complex-interval-min (interval)
  (let ((sub-mins (mapcar #'interval-min
			  (interval-sub-intervals interval))))
    (case (interval-kind interval)
      ((:ordered :unordered)
       (apply #'tf+ sub-mins))
      ((:parallel)
       (apply #'tfmax sub-mins))
      ((:selection)
       (apply #'tfmin sub-mins)))))


;;;
;;; (complex-interval-max interval) -> number
;;;
;;; Return the maximum range of a complex interval, by inspecting
;;; its sub-intervals.
;;;
(defun complex-interval-max (interval)
  (let ((sub-maxs (mapcar #'interval-max
			  (interval-sub-intervals interval))))
    (case (interval-kind interval)
      ((:ordered :unordered)
       (apply #'tf+ sub-maxs))
      ((:parallel :selection)
       (apply #'tfmax sub-maxs)))))


;;;
;;; (get-distance-vtp vtp otp) -> range
;;;
;;; Get the distance from a virtual timepoint vtp to another
;;; timepoint.  The other timepoint may be real or virtual.
;;;
(defun get-distance-vtp (vtp otp)
  (cond
   ((virtual-timepoint-p otp)
    (get-distance-two-vtps vtp otp))
   ((null (interval-super-interval (vtp-interval vtp)))
    (get-distance-realized-vtp vtp otp))
   (t
    (get-distance-vtp-through-super vtp otp))))


;;;
;;; (get-distance-vtp-through-super vtp otp) -> range
;;;
;;; Given a virtual timepoint that is one end of a sub-interval, find
;;; the distance to the other given timepoint by going through both
;;; ends of the super-interval.
;;;
(defun get-distance-vtp-through-super (vtp otp)
  (let ((super (interval-super-interval (vtp-interval vtp))))
    (mvb (super-begin super-end) (get-distance-vtp-to-super-interval vtp)
	 (intersect-ranges
	  (convolve-ranges super-begin (get-distance-vtp (beginning-of super) otp))
	  (convolve-ranges super-end (get-distance-vtp (end-of super) otp))))))

;;;
;;; (get-distance-realized-vtp rvtp otp) -> range
;;;
;;; The interval of the given virtual timepoint has no super-interval,
;;; thus it is immediately related to a real timepoint.  The range
;;; from that real timepoint to the other given (possibly virtual)
;;; timepoint is returned.  Note that it is assumed that the other
;;; timepoint is not a virtual timepoint of a sub-interval of the
;;; interval of the given virutal timepoint.
;;;
(defun get-distance-realized-vtp (rvtp otp)
  (let* ((int (vtp-interval rvtp))
	 (from-tp (interval-from int))
	 (to-tp (interval-to int)))
    (case (vtp-type rvtp)
      ((:begin)
       (cond
	 (from-tp
	  (get-distance from-tp otp))
	 (to-tp
	  (convolve-ranges (interval-range int) (get-distance to-tp otp)))
	 (t
	  (make-universal-range))))
      ((:end)
       (cond
	 (to-tp
	  (get-distance to-tp otp))
	 (from-tp
	  (convolve-ranges (reverse-range (interval-range int)) (get-distance from-tp otp)))
	 (t
	  (make-universal-range)))))))


;;;
;;; (get-distance-two-vtps vtp1 vtp2) -> range
;;;
;;; Returns the distance between two virtual timepoints.
;;;
(defun get-distance-two-vtps (vtp1 vtp2)
  (let* ((int1 (vtp-interval vtp1))
	 (int2 (vtp-interval vtp2))
	 (super1 (interval-super-interval int1))
	 (super2 (interval-super-interval int2))
	 (depth1 (interval-depth int1))
	 (depth2 (interval-depth int2)))
    (cond
      ((eql int1 int2)
       (get-distance-two-vtps-same-interval vtp1 vtp2))
      ((and super1 (eql super1 super2))
       (get-distance-two-vtps-same-super vtp1 vtp2))
      ((< depth1 depth2)
       (if super1
	   (get-distance-vtp-through-super vtp1 vtp2)
	   (get-distance-realized-vtp vtp1 vtp2)))
      (t
       (reverse-range
	(if super2
	    (get-distance-vtp-through-super vtp2 vtp1)
	    (get-distance-realized-vtp vtp2 vtp1)))))))

	    
;;;
;;; (get-distance-vtp-to-super-interval vtp) -> range, range
;;;
;;; Returns two values, the distance from a vtp to the beginning and
;;; the end of the immediately containing super-interval.
;;;
(defun get-distance-vtp-to-super-interval (vtp)
  (let* ((which (vtp-type vtp))
	 (int (vtp-interval vtp))
	 (super (interval-super-interval int))
	 (kind (interval-kind super))
	 (subs (interval-sub-intervals super))
	 )
    (case kind
      ((:ordered)
       (let ((before-range (make-zero-range))
	     (after-range (make-zero-range))
	     (seen nil))
	 (dolist (subint subs)
	   (when (and (eql int subint)
		      (eql which :begin))
	     (setf seen t))
	   (if seen
	       (setf after-range (convolve-ranges after-range (interval-range subint)))
	       (setf before-range (convolve-ranges before-range (interval-range subint))))
	   (when (and (eql int subint)
		      (eql which :end))
	     (setf seen t)))
	 (values (reverse-range before-range) after-range))
       )
      ;; assumption: all sub-interval ranges are non-negative (for next two cases)
      ((:unordered)
       (case which
	 ((:begin)
	  (values (reverse-range (make-range 0  (tf- (interval-max super) (interval-max int))))
		  (make-range (interval-min int) (interval-max super))))
	 ((:end)
	  (values (reverse-range (make-range (interval-min int) (interval-max super)))
		  (make-range 0 (tf- (interval-max super) (interval-max int))))))
       )
      ((:parallel)
       (case which
	 ((:begin)
	  (values (reverse-range (make-range 0 (tf- (interval-max super) (interval-min int))))
		  (make-range (interval-min int) (interval-max super))))
	 ((:end)
	  (values (reverse-range (make-range (interval-min int) (interval-max super)))
		  (make-range 0 (tm- (interval-max super) (interval-min int))))))
       )
      ((:selection)
       (case which
	 ((:begin)
	  (values  (make-zero-range)
		   (interval-range int)))
	 ((:end)
	  (values (reverse-range (interval-range int))
		  (make-zero-range))))
       ))))


;;;
;;; (get-distance-two-vtps-same-interval vtp1 vtp2) -> range
;;;
;;; Given two vtps which are ends of the same interval, return the
;;; range between them.  Erroneous results if the two vtps are NOT, in
;;; fact, endpoints of the same interval.
;;;
(defun get-distance-two-vtps-same-interval (vtp1 vtp2)
  (let* ((int (vtp-interval vtp1))
	 (range (interval-range int))
	 (which1 (vtp-type vtp1))
	 (which2 (vtp-type vtp2)))
    (cond
      ((eql which1 which2)
       (make-zero-range))
      ((eql which1 :begin)
       range)
      (t
       (reverse-range range)))))


;;;
;;; (get-distance-two-vtps-same-super vtp1 vtp2) -> range
;;;
;;; Given two virtual timepoints whose intervals are both
;;; sub-intervals of the same super-interval, determine (based on the
;;; kind of the super-interval) what the range between them must be.
;;;
(defun get-distance-two-vtps-same-super (vtp1 vtp2)
  (let* ((int1 (vtp-interval vtp1))
	 (int2 (vtp-interval vtp2))
	 (which1 (vtp-type vtp1))
	 (which2 (vtp-type vtp2))
	 (super (interval-super-interval  int1))
	 (kind (interval-kind super))
	 (subs (interval-sub-intervals super)))
    (case kind
      ((:ordered)
       (let ((first-found nil)
	     (second-found nil)
	     (range-between (make-zero-range)))
	 (dolist (subint subs)
	   ;; check for beginning vtp's
	   (when (and (eql int1 subint) (eql which1 :begin))
	     (cond ((not first-found) (setf first-found vtp1))
		   ((not second-found) (setf second-found vtp1))))
	   (when (and (eql int2 subint) (eql which2 :begin))
	     (cond ((not first-found) (setf first-found vtp2))
		   ((not second-found) (setf second-found vtp2))))
	   ;; accumulate range between points
	   (when (and first-found (not second-found))
	     (setf range-between (convolve-ranges range-between (interval-range subint))))
	   ;; check for ending vtp's
	   (when (and (eql int1 subint) (eql which1 :end))
	     (cond ((not first-found) (setf first-found vtp1))
		   ((not second-found) (setf second-found vtp1))))
	   (when (and (eql int2 subint) (eql which2 :end))
	     (cond ((not first-found) (setf first-found vtp2))
		   ((not second-found) (setf second-found vtp2)))))
	 (cond
	   ((and (eql first-found vtp1) (eql second-found vtp2))
	    range-between)
	   ((and (eql first-found vtp2) (eql second-found vtp1))
	    (reverse-range range-between))
	   (t
	    (error "Unrecoverable error in GET-DISTANCE-TWO-VTPS-SAME-SUPER."))))
       )
      ;; assumption: all sub-interval ranges are non-negative (for next two cases)
      ((:unordered)
       (cond
	 ((and (eql which1 :begin) (eql which2 :begin))
	  (make-range (tf- (interval-max int1) (interval-max super))
		      (tf- (interval-max super) (interval-max int2)))
	  )
	 ((and (eql which1 :begin) (eql which2 :end))
	  (make-range  (tf- (tf+ (interval-max int1) (interval-max int2)) (interval-max super))
		       (interval-max super))
	  )
	 ((and (eql which1 :end) (eql which2 :begin))
	  (make-range (tf- (interval-max super))
		      (tf- (interval-max super) (tf+ (interval-max int1) (interval-max int2))))
	  )
	 ((and (eql which1 :end) (eql which2 :end))
	  (make-range (tf- (interval-max int2) (interval-max super))
		      (tf- (interval-max super) (interval-max int1)))))
       )
      ((:parallel)
       (cond
	 ((and (eql which1 :begin) (eql which2 :begin))
	  (make-range (tf- (interval-min int1) (interval-max super))
		      (tf- (interval-max super) (interval-min int2)))
	  )
	 ((and (eql which1 :begin) (eql which2 :end))
	  (make-range  (tf- (tf+ (interval-min int1) (interval-min int2)) (interval-max super))
		       (interval-max super))
	  )
	 ((and (eql which1 :end) (eql which2 :begin))
	  (make-range (tf- (interval-max super))
		      (tf- (interval-max super) (tf+ (interval-min int1) (interval-min int2))))
	  )
	 ((and (eql which1 :end) (eql which2 :end))
	  (make-range (tf- (interval-min int2) (interval-max super))
		      (tf- (interval-max super) (interval-min int1)))))
       )
      ((:selection)
       (make-unknown-range))
       )
    ))


;;;
;;; (tl-printer tp range depth)
;;;
;;; Given a timepoint, the range to that timepoint from its parent,
;;; and the current print depth, makes a pretty printout of the
;;; timeline tree rooted at tp
;;;
(defun tl-printer (tp range depth)
  (print-spaces depth)
  (format t "~a ~a~%" tp range)
  (dolist (child-con (timepoint-forward-edges tp))
    (tl-printer (constraint-to tp child-con)
		(constraint-range tp child-con)
		(+ depth 3)))
  (values))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Miscellaneous timepoint functions
;;;

;;;
;;; (timepoint-is-branch? tp) -> boolean
;;;
;;; Returns true iff tp has more than one child, or is the
;;; root of the spanning tree.
;;;
(defun timepoint-is-branch? (tp)
  (or (cdr (timepoint-forward-edges tp))
      (timepoint-is-root? tp)))


;;;
;;; (timepoint-is-root? tp) -> boolean
;;;
;;; Returns true iff tp is the root of the spanning tree.
;;;
(defun timepoint-is-root? (tp)
  (null (timepoint-back-edge tp)))
  

;;;
;;; (timepoint-is-leaf? tp) -> boolean
;;;
;;; Returns true iff tp has no children in the tree.
;;;
(defun timepoint-is-leaf? (tp)
  (null (timepoint-forward-edges tp)))


;;;
;;; (timepoint-all-edges tp &optional include-free-edges?) -> list
;;;
;;; Returns a list of all the edges leading out from tp.  By default,
;;; it only returns edges that actually have timepoints at the other
;;; end.  Optionally, it will also include the free intervals.
;;;
(defun timepoint-all-edges (tp &optional include-free-edges?)
  (let ((edge-list (append (if include-free-edges?
			       (timepoint-free-edges tp)
			       '())
			   (timepoint-cross-edges tp)
			   (timepoint-forward-edges tp))))
    (if (timepoint-back-edge tp)
	(cons (timepoint-back-edge tp) edge-list)
	edge-list)))
	  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Various constraint functions
;;;
;;; A constraint between two timepoints is just a simple interval with
;;; no contents or type.  The timepoint to which the constraint is to,
;;; and the distance thereto, depends on which end of the constraint
;;; you are at.
;;;
(defun make-constraint (from to min max)
  (make-interval :kind :simple :from from :to to
		 :range (make-range min max)))

(defun constraint-range-to (tp constraint)
  (cond
    ((eq tp (interval-from constraint))
     (interval-to constraint))
    ((eq tp (interval-to constraint))
     (interval-from constraint))
    (t
     (error "~S: Timepoint is not endpoint of constraint"))))

(defun constraint-to (tp constraint)
  (cond
    ((eq tp (interval-from constraint))
     (interval-to constraint))
    ((eq tp (interval-to constraint))
     (interval-from constraint))
    (t
     (error "~S: Timepoint is not endpoint of constraint"))))

(defun constraint-range (tp constraint)
  (cond
    ((eq tp (interval-from constraint))
     (interval-range constraint))
    ((eq tp (interval-to constraint))
     (reverse-range  (interval-range constraint)))
    (t
     (error "~S: Timepoint is not endpoint of constraint"))))

(defun constraint-min (tp constraint)
  (range-min (constraint-range tp constraint)))

(defun constraint-max (tp constraint)
  (range-max (constraint-range tp constraint)))

(defun make-convolved-constraint (tp &rest intervals)
  (mvb (range tp-to) (apply #'convolve-intervals (cons tp intervals))
       (make-constraint tp tp-to (range-min range) (range-max range))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Miscellaneous Interval Functions

;;;
;;; (accrue-range range tp interval) -> range
;;;
;;; Given a current range, a timepoint, and an interval returns two
;;; values:
;;;
;;; - the given range convolved with the range obtained by
;;;   following interval away from tp
;;;
;;; - the timepoint at the other end of interval from tp
;;;
(defun accrue-range (range tp interval)
  (values
   (convolve-ranges
    range (constraint-range tp interval))
   (constraint-to tp interval)))


;;;
;;; (convolve-intervals tp int1 int2 ...)
;;;
;;; Starting from the given timepoint, convolves the ranges of the
;;; given intervals, and returns two values:
;;;   - the range of time values of the convolved constraint
;;;   - the timepoint at the end of the chain
;;;
(defun convolve-intervals (tp &rest intervals)
  (let ((range (make-zero-range))
	(int (pop intervals)))
    (loop
     (exit-when (null int) (values range tp))
     (multiple-value-setq
	 (range tp)
       (accrue-range range tp int))
     (setf int (pop intervals)))))


;;;
;;; (complex-kind? kind)
;;;
;;; Returns T iff the given interval kind specifier is one of the
;;; complex types (:ordered, :unordered, :parallel, :selection).
;;;
(defun complex-kind? (kind)
  (member kind '(:ordered :unordered :parallel :selection)))
