;;;;
;;;; HIC Benchmark
;;;;
;;;; Compare the times required to find the distance between points
;;;; when using hierarchical intervals vs. pure timepoint graphs.
;;;;
;;;; After loading this, you might say:
;;;;    (hic-bench:interval-setup '((2 2) (2 2) (2 2)))
;;;;    (hic-bench:interval-random-queries 10)
;;;;    (hic-bench:timepoint-setup '((2 2) (2 2) (2 2)))
;;;;    (hic-bench:timepoint-random-queries 10)
;;;; to do a comparison of the amount of time required to do the 
;;;; ten queries in a HIC-based vs. timepoint-based graph.
;;;;

(in-package 'hic-bench)

(export '(interval-setup interval-random-queries
	  timepoint-setup timepoint-random-queries
	  *no-caching*))

(require 'hic "hic")
(require 'standard "standard")
(use-package 'standard-extensions)

;;;
;;; Timeline description
;;;
;;; The timeline used will consist of an origin timepoint, from which
;;; depend a number of action branches, which are all joined by an
;;; open-ended interval to a final ending timepoint, thus:
;;;
;;;	/*---Action-1---*---[0,infinity]---\
;;;    /   		   		    \
;;;   /	   		   		     \
;;;  O---*---Action-2---*---[0,infinity]------E
;;;   \                                      /
;;;    \		   		    /
;;;	\*---Action-N---*---[0,infinity]---/
;;;
;;;
;;; Each action consists of an ordered set of arbitrarily deeply
;;; nested subintervals.  A description of an action is given as a
;;; list of integers, where the first integer gives the number of
;;; intervals in the action, the next integer gives the number of
;;; sub-intervals in each interval, etc.
;;;


;;;
;;; Besides building the timeline itself, there will also be some
;;; global variables to hold:
;;;
;;;    * the beginning and ending timepoints of the graph
;;;    * a list of the beginning timepoints of each action
;;;    * a list of the ending timepoints of each action
;;;    * list of all the action intervals created (in interval case)
;;;    * list of all the action timepoints created (in timepoint case)
;;;

;;;
;;; Global variables to hold pointers into structures
;;;

(defvar *beginning-timepoint*)
(defvar *ending-timepoint*)
(defvar *action-beginnings*)
(defvar *action-endings*)
(defvar *all-action-intervals*)
(defvar *all-action-timepoints*)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Using heirarchical intervals
;;;

;;;
;;; (interval-setup action-desc-list)
;;;
;;; Create a temporal database as described above, given a list of
;;; action descriptions.
;;;
(defun interval-setup (action-description-list)
  (format t "Setting up INTERVAL-BASED timeline...~%")
  (time
   (progn
     (common-setup)
     (dolist-index (num desc action-description-list)
         (let ((action-begin (HIC:create-timepoint (cat-symbol 'ACTION num 'BEGIN) nil))
	       (action-end (HIC:create-timepoint (cat-symbol 'ACTION num 'END) nil))
	       (action-interval (create-action-interval (cat-symbol 'ACTION num 'BODY) desc)))
	   (push action-begin *action-beginnings*)
	   (push action-end *action-endings*)
	   (HIC:add-time-constraint *beginning-timepoint* action-begin 0 0)
	   (HIC:add-time-constraint action-end *ending-timepoint* 0 'TF:plus-inf)
	   (HIC:attach-interval action-interval action-begin action-end)
	   ))))
  (format t "~%Effective number of timepoints: ~a"
	  (+ 2 (length *action-beginnings*) (length *action-endings*)
	     (* 2 (length *all-action-intervals*))))
  (values))


;;;
;;; (create-action-interval symbolic-name action-description) -> hic-interval
;;;
;;; Creates a hierarchical interval of the type specified by the given
;;; action description.
;;;
(defun create-action-interval (action-name desc)
  (car
   (push
    (cond
      ((null desc)
       (HIC:create-interval action-name nil :simple 5 10))
      (t
       (let ((sub-intervals '()))
	 (dotimes (i (car desc))
	   (push (create-action-interval (cat-symbol action-name i)  (cdr desc))
		 sub-intervals))
	 (HIC:create-interval action-name nil :ordered (nreverse sub-intervals)))))
    *all-action-intervals*)))

;;;
;;; (interval-random-queries num)
;;;
;;; Performs the given number of random queries and prints the timing
;;; information.
;;;
(defun interval-random-queries (num)
  ;; Set up a list of random TP pairs...
  (let ((num-ints (length *all-action-intervals*))
	(random-tp-pair-list '()))
    (dotimes (i num)
      (let ((int1 (nth (random num-ints) *all-action-intervals*))
	    (int2 (nth (random num-ints) *all-action-intervals*)))
	(push (list (one-of
		     (HIC:beginning-of int1)
		     (HIC:end-of int1))
		    (one-of
		     (HIC:beginning-of int2)
		     (HIC:end-of int2)))
	      random-tp-pair-list)))
  (format t "Performing ~a random quer~:@p...~%" num)
  (time
   (dolist (tp-pair random-tp-pair-list)
     (HIC:get-distance (first tp-pair) (second tp-pair))
     (clear-cache)))))
    




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Using timepoints
;;;

;;;
;;; (timepoint-setup action-desc-list)
;;;
;;; Create a temporal database as described above, given a list of
;;; action descriptions.
;;;
(defun timepoint-setup (action-description-list)
  (format t "Setting up TIMEPOINT-BASED timeline...~%")
  (time
   (progn
     (common-setup)
     (dolist-index (num desc action-description-list)
         (let ((action-begin (HIC:create-timepoint (cat-symbol 'ACTION num 'BEGIN) nil))
	       (action-end (HIC:create-timepoint (cat-symbol 'ACTION num 'END) nil))
	       (action-timepoints (create-action-timepoints (cat-symbol 'ACTION num 'BODY) desc)))
	   (push action-begin *action-beginnings*)
	   (push action-end *action-endings*)
	   (HIC:add-time-constraint *beginning-timepoint* action-begin 0 0)
	   (HIC:add-time-constraint action-begin (first action-timepoints) 0 0)
	   (HIC:add-time-constraint (second action-timepoints) action-end 0 0)
	   (HIC:add-time-constraint action-end *ending-timepoint* 0 'TF:plus-inf)
	   ))))
  (format t "~%Effective number of timepoints: ~a"
	  (+ 2 (length *action-beginnings*) (length *action-endings*)
	     (length *all-action-timepoints*)))
  (values))


;;;
;;; (create-action-timepoints symbolic-name action-description) -> list of two timepoints
;;;
;;; Creates a sequence of interconstrained timepoints that represents
;;; the given action description.
;;;
(defun create-action-timepoints (action-name desc)
  (let ((begin-tp (HIC:create-timepoint (cat-symbol action-name 'BEGIN) nil))
	(end-tp (HIC:create-timepoint (cat-symbol action-name 'END) nil)))
    (cond
      ((null desc)
       (HIC:add-time-constraint begin-tp end-tp 5 10))
      (t
       (let ((from-tp begin-tp))
	 (dotimes (i (car desc))
	   (let ((sub-timepoints (create-action-timepoints (cat-symbol action-name i)
							   (cdr desc))))
	     (HIC:add-time-constraint from-tp (first sub-timepoints) 0 0)
	     (setf from-tp (second sub-timepoints))))
	 (HIC:add-time-constraint from-tp end-tp 0 0))))
    (push end-tp *all-action-timepoints*)
    (push begin-tp *all-action-timepoints*)
    (list begin-tp end-tp)))


;;;
;;; (timepoint-random-queries num)
;;;
;;; Performs the given number of random queries and prints the timing
;;; information.
;;;
(defun timepoint-random-queries (num)
  ;; Set up a list of random TP pairs...
  (let ((num-tps (length *all-action-timepoints*))
	(random-tp-pair-list '()))
    (dotimes (i num)
      (let ((tp1 (nth (random num-tps) *all-action-timepoints*))
	    (tp2 (nth (random num-tps) *all-action-timepoints*)))
	(push (list tp1 tp2) random-tp-pair-list)))
    (format t "Performing ~a random quer~:@p...~%" num)
    (time
     (dolist (tp-pair random-tp-pair-list)
       (HIC:get-distance (first tp-pair) (second tp-pair))
       (clear-cache)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Stuff used in both interval and timepoints approaches
;;;


;;;
;;; (common-setup)
;;;
;;; Benchmark setup that is common to both the interval and timepoint
;;; based configurations.
;;;
(defun common-setup ()
  (HIC:hic-initialize)
  (setf *beginning-timepoint* (HIC:create-timepoint 'PLAN-BEGIN nil))
  (setf *ending-timepoint* (HIC:create-timepoint 'PLAN-END nil))
  (setf *action-beginnings* '())
  (setf *action-endings* '())
  (setf *all-action-intervals* '())
  (setf *all-action-timepoints* '())
  (values))


;;;
;;; Clear the solution cache.
;;;
(defun clear-cache ()
  (if *no-caching*
      (HIC::flush-solution-cache  (HIC::timepoint-timeline *beginning-timepoint*))))

(defvar *no-caching* nil)


;;;
;;; (cat-symbol s1 s2 ...) -> symbol
;;;
;;; Concatenates the given symbols together with hyphens to create a
;;; new symbol.
;;;
(defun cat-symbol (&rest syms)
  (intern (apply #'format nil "~@{~a~#[~:;-~]~}" syms)))


;;;
;;; (effective-timepoints action-desc-list) -> number
;;;
;;; Returns that number of effective timepoints that would result from
;;; the given action description.
;;;
(defun effective-timepoints (desc-list)
  (+
   (* 2				  ; two timepoints for each interval
      (apply
       #'+			  ; sum over all action descripions
       (mapcar
	#'action-interval-count   ; get number of intervals for described action
	desc-list)))
   (* 2 (length desc-list))	  ; a beginning and ending timepoint for each interval
   2				  ; the plan beginning and ending timepoints
   ))


;;;
;;; (action-interval-count  (action-desc) -> integer
;;;
;;; Returns the number of intervals entailed by a given action
;;; representation.
;;;
(defun action-interval-count (desc)
  (apply
   #'+			       ; sum over all levels of action hierarchy
   1			       ; the top level action interval
   (maplist		       ; iterate over each intermediate level
    #'(lambda (x) (apply #'* x))
    (reverse desc))))


		     