;;; -*- Syntax: Common-lisp; Package: POS; Mode:Common-Lisp; Base:10  -*-
;;; Copyright 1988 David Throop at the University of Texas at Austin

(in-package 'pos)

#+symbolics
(use-package :scl :pos)

#+symbolics
(import '(self send defmethod))

;;; This file is for graphics extensions which have not been implemented on TI
;;; Explorers.  Some of the code relies upon Symbolics internal code to
;;; calculate the coordinates of some PostScript output.

(defun qplot-cubic-spline (points  &key (alu *black*)(thickness 1)
			   (start-relaxation :relaxed)
			   &allow-other-keys)
  (when (image-to-postscript-p)
    (ps-draw-cubic-spline points :alu alu :thickness thickness
			  :start-relaxation start-relaxation))
  (when (image-to-screen-p)
    (graphics:draw-cubic-spline			; tolerates floatingpt
       points :alu alu :thickness thickness :stream *qplot-output*
       :start-relaxation start-relaxation)))

;;; Unfortunately, PS-DRAW-CUBIC-SPLINE is machine dependent - I just picked up
;;; some of the handy code (used by graphics:draw-cubic-spline) for spline fitting
;;; and adapted it to PostScript output.  [The code for TV:SPLINE has the notation
;;; "Cubic splines from Rogers and Adams, `Mathematical Elements for Computer
;;; Graphics'" if you want to reimplement.]

;;; If you attempt to store too many strokes in a single path, you will get a
;;; stack overflow in the printer.  For this reason, every 50 segments the path is
;;; stroked and then restarted.  I haven't done any testing to check whether this
;;; is an optimal break point.

;;; The list of points (x1 y1 x2 y2 ...) to PS-DRAW-CUBIC-SPLINE is copied into
;;; two arrays [x1 x2 ..], [y1 y2 ...] before TV:SPLINE will do the curve fitting.
;;; PX, PY are the arrays given to TV:SPLINE; CX, CY are the arrays returned.

(defun ps-draw-cubic-spline
       (points &key (alu *black*)(thickness 1)
	(start-relaxation :relaxed) &allow-other-keys)
  (checkgray alu)
  (checkwidth thickness)
  (let ((length (length points)))
    (tv:with-stack-array (px (floor length 2))
      (tv:with-stack-array (py (floor length 2))
	(loop for i below length
	      do (setf (aref (if (oddp i) py px) (floor i 2))
		       (elt points i)))
	(multiple-value-bind (cx cy i)		; cx, cy are arrays of x,y points.
	    (tv:spline px py length
		       nil nil start-relaxation)
	  (format  *postscript-output-file*
		   "~& n ~,1f ~,1f m        % begin spline~&"
		   (aref cx 0)(- yscreen (aref cy 0)))
	  (loop for ctr from 1 below i
		for x = (aref cx ctr)
		for y = (- yscreen (aref cy ctr))
		for oldx = (aref cx (1- ctr))
		for oldy = (- yscreen (aref cy (1- ctr)))
		do (format  *postscript-output-file*
			    "~,1f ~,1f rl " (ps-trim (- x oldx)) (ps-trim (- y oldy)))
		when (and (zerop (mod ctr 50))	; Don't stroke the path if this is the 
			  (not (= ctr (1- i))))	; last pair of points.
		  do (format *postscript-output-file*
			     "s ~,1f ~,1f m " (ps-trim x) (ps-trim y)))
	  (format  *postscript-output-file* "s ")
	  )))))


;;;  QPLOT-STRING-ON-CLEAN is similar to {\tt qplot-string}, but draws a white
;;; rectangle underneath the string before printing the string in black.  It
;;; does not have an {\cf alu} argument -- it always prints a black string on
;;; a white rectangle.
;;;   It does this by printing the string on the screen once using the {\cf
;;; tv:alu-and} alu, which doesn't actually print.  This gives the length of
;;; the string on the screen.  Then it prints the white rectangle and then
;;; finally the string in black.

(defun qplot-string-on-clean (string x y &key (margin 3)(height 8)(font  *plain-font*))
  (let ((right(graphics:draw-string (format nil "~v~a~" font string) x y
				    :alu tv:alu-and :stream *qplot-output*)))
    (qplot-polygon (list (- x margin) (+ y margin)
			 (- x margin) (- y margin height)
			 (+ margin right)(- y margin height)
			 (+ margin right)(+ y margin))
		   :filled t :alu *white*)
    (with-plotting-to-screen-forced 
      (+ margin (qplot-string string x y :font font)))))

;;; For PostScript Output:
;;; When ClockW, "A" is rotated so it points right.  Else it points left
;;; When ClockW, string is pivoted about the "F" in "FooBar".  Else the "r".

;;; THIS IS STILL BUGGY!

(defun qplot-sideways-string (string x y &key
			      (font  *plain-font*)(alu *black*)(ClockW nil))
  (cond ((or (null string)
	     (and (stringp string)(zerop (length string))))
	 nil)
	((> y 700) (qplot-string string x y :alu *black* :font font))	;A hack. D.B.
	(t
	 (setq x (round x))  (setq y (round y))	;D.B. (won't crash now but still doesn't work right.)
	 (when (image-to-postscript-p)
	   (ps-draw-sideways-string  x y string (if ClockW -90 90)
				     :alu alu :font font))
    
	 (when (image-to-screen-p)
	   #+symbolics (progn (send *qplot-output* :set-cursorpos
				    0 0)
			      (graphics:with-room-for-graphics (*qplot-output* y)
				(graphics:draw-string-image (format nil "~v~a~" font string)
							    0 0 :translation (list x 0)
							    :rotation (/ pi (if clockw -2 2))
							    :alu alu :stream *qplot-output*)))
	   #+ti (send *qplot-output* :string-out-explicit
		      (format nil "~a" string) (round x) (round (- y 9)) nil nil
		      font tv:alu-xor 0 (zlc:string-length string) nil)))))



;;; QPLOT-RIGHT-JUSTIFIED-STRING plots a string whose right end is at (xright
;;; y).  The length in pixels of the string is determined by plotting it on
;;; the screen using the tv:alu-and alu, which has no effect on the screen.
;;; Shortcoming:  there is only an approximate match between the postscript
;;; fonts and the symbolic fonts, especially for fonts which are kerned.
;;; Blocks of text which are right justified using this will come out with a
;;; somewhat ragged right edge in the postscript hardcopy.  It is, however,
;;; useful for, inter alia, labeling the Y axis of graphs.

(defun qplot-right-justified-string (string xright y
				     &key (alu *black*)(font *plain-font*))
  (let ((pixellength (graphics:draw-string string 0 y :alu
					   tv:alu-and :character-style font
					   :stream *qplot-output*)))
    (qplot-string string (- xright pixellength) y :alu alu :font font)))

(defun ps-draw-sideways-string (x y string rot  &key (alu *black*)(font *plain-font*))
  (checkgray alu)
  (get-font-for-string string font)
  (format *postscript-output-file* "~&gs ~,1f ~,1f moveto ~,1f rotate (~a) ~
		show gr         % rotated string~&"
	  (ps-trim x) (ps-trim (- yscreen y)) (ps-trim rot)
	  (strip-font-info (ps-string string))))	; Slashify #\\, #\( and #\).

;;; QPLOT-SPLINE is similar to QPLOT-CUBIC-SPLINE; given an (x y x y x
;;; y..) pointlist, it draws a curve that intersects each point.  It
;;; uses graphics:draw-bezier-curve to draw this curve; in order to use
;;; that drawing function, QPLOT-SPLINE has to generate the control
;;; points.


 
(defun qplot-spline (pointlist &key (thickness 1)(alu *black*)(show nil))
  (let* ((length  (length pointlist))
	 (controlpts 
	   (loop for (x1 y1 x2 y2 x3 y3) on pointlist by 'cddr
		 when x3
		   append (let* ((dxa (- x3 x2))
				 (dxb (- x2 x1))
				 (dya (- y3 y2))
				 (dyb (- y2 y1))
				 (theta1 (atan dya dxa))
				 (theta2 (atan dyb dxb))
				 (base (+ theta1 (/ theta2 2) ))
				 (lena (/ (sqrt (+ (* dxa dxa)(* dya dya))) 2))
				 (lenb (/ (sqrt (+ (* dxb dxb)(* dyb dyb))) 2))
				 (nx1 (- x2 (* lenb (cos base))))
				 (nx2 (+ x2 (* lena (cos base))))
				 (ny1 (- y2 (* lenb (sin base))))
				 (ny2 (+ y2 (* lena (sin base)))))
			    (list nx1 ny1 nx2 ny2))))
	 (augments (mapcar 'round
			   `(,@(macrolet ((p (num) `(nth (1- ,num) pointlist)))
				 (intfn (p 5)(p 6)(p 3)(p 4)(p 1)(p 2)))
			     ,@controlpts
			     ,@(apply #'intfn (subseq pointlist (- length 6)))))))
    (loop for (x1 y1 x2 y2) on pointlist by 'cddr
	  for (xc1 yc1 xc2 yc2) on augments by 'cddddr
	  when show
	    do (pnote x1 y1 2) (pnote xc1 yc1 1)(pnote xc2 yc2 1)
	  when x2
	    do (qplot-bezier-curve x1 y1 x2 y2 xc1 yc1 xc2 yc2
				   :thickness thickness :alu alu))))

;;; The idea of QPLOT-FUNCTIONAL-SPLINE is that we are often given (to
;;; plot) a series of (x y x y...)  where we know that y = f(x).  We
;;; want the fitted curve to be a function - that is, a line drawn
;;; perpendicular to the X axis may cut the curve exactly once.  The
;;; other spline fucntions don't guarentee this---consider the following
;;; curve fit to four points (showns as O's) with a sharp stairstep:
;                        O---------O-
;                      /
;                     |
;                      \
;                       |
;                      /
;---O----------------O
;;; QPLOT-FUNCTIONAL-SPLINE assumes that the x's in pointlist are
;;; strictly increasing (or decreasing, it doesn't matter) and chooses
;;; control points such that the x's of the control points are halfway
;;; between the ends of the segment which they control.

(defun qplot-functional-spline (pointlist &key (thickness 1)(alu *black*)(show nil))
    (loop with controlpts = (functional-control-points pointlist)
	  with length = (length  pointlist)
	  for (x1 y1 x2 y2) on pointlist by 'cddr
	  for (xc1 yc1 xc2 yc2) on controlpts by 'cddddr
	  when show
	    do (pnote x1 y1 3) (pnote xc1 yc1 2)(pnote xc2 yc2 2)
	  when x2
	    do (qplot-bezier-curve x1 y1 x2 y2 xc1 yc1 xc2 yc2
				   :thickness thickness :alu alu)
	  finally (when show (pnote (nth (- length 2) pointlist)
				    (nth (- length 1) pointlist) 3))))

(defun functional-control-points  (pointlist)
  (loop for subpolintlist on pointlist by 'cddr
	for (x1 y1 x2 y2 x3 y3) = subpolintlist  
	with lastplace = (nthcdr (- (length pointlist) 6) pointlist)
	when (eq x1 (car pointlist))
	  append (functional-endpoints-interpolation x1 y1 x2 y2 x3 y3)
	when x3
	  append (functional-regular-interpolation x1 y1 x2 y2 x3 y3)
	when (eq subpolintlist lastplace )
	  append (functional-endpoints-interpolation x3 y3 x2 y2 x1 y1)))

;;; Given 3 points, (x1 y1), (x2 y2), (x3 y3)
;;; FUNCTIONAL-REGULAR-INTERPOLATION averages the slope between the
;;; first and second line segments, and draws a line with that avg slope
;;; thru (x2 y2), and finds where that line intersects the halfway x for
;;; each of the intervals.  It returns the coordinates of those two
;;; intercepts as the coordinates of the control points around (x2 y2).

(defun functional-regular-interpolation (x1 y1 x2 y2 x3 y3)
  (let* ((dxa (- x3 x2))
	 (dxb (- x2 x1))
	 (dya (- y3 y2))
	 (dyb (- y2 y1))
	 (halfmavg (+ (/ dya dxa 4.0)(/ dyb dxb 4.0)))
	 (nx1 (+ x1 (/ dxb 2.0)))
	 (nx2 (+ x2 (/ dxa 2.0)))
	 (ny1 (- y2 (* dxb halfmavg)))
	 (ny2 (+ y2 (* dxa halfmavg))))
    (list nx1 ny1 nx2 ny2)))

(defun functional-endpoints-interpolation (x1 y1 x2 y2 x3 y3)
  (let* ((xa (/ (+ x3 x2) 2))
	 (xb (/ (+ x2 x1) 2))
	 (ma (/ (- y3 y2)(- x3 x2)))
	 (mb (/ (- y2 y1)(- x2 x1)))
	 (m1 (- mb (* (- ma mb)(/ (- xb x1)(- xa xb)))))
	 (ycb (+ y1 (* m1 (- xb x1)))))
    (list xb ycb)))

(defun pnote (x y r)
  (qplot-circle x y r :filled t)
;  (qplot-string (format nil "(~a, ~a)" (round x)(round y))
;		x y :font '(:fix :roman :small))
  )

(defun intfn (x1 y1 x2 y2 x3 y3)
  (list (/ (+ x2 x3) 2.0)
	(+ (/ (+ y2 y3) 2.0)
	   (* (- y2 y1) .13 (/ (- x3 x2)(- x2 x1))))))

(defun qplot-bezier-curve (x1 y1 x2 y2 x3 y3 x4 y4 &key (thickness 1)(alu *black*))
  (when (image-to-postscript-p)
    (ps-draw-bezier-curve x1 y1 x2 y2 x3 y3 x4 y4 :thickness thickness :alu alu))
  (when (image-to-screen-p)
    #+symbolics (graphics:draw-bezier-curve x1 y1 x2 y2 x3 y3 x4 y4 :stream *qplot-output*
					    :thickness thickness :alu alu)
    #+ti (error "NO TI IMPLEMENTATION")))

(defun ps-draw-bezier-curve (x1 y1 x2 y2 xc1 yc1 xc2 yc2
			     &key (thickness 1)(alu *black*))
  (checkwidth thickness)
  (checkgray alu)
  (format *postscript-output-file*
	  "~& ~a ~a moveto ~a ~a ~a ~a ~a ~a curveto stroke "
	  (ps-trim x1) (ps-trim (- yscreen y1))
	  (ps-trim xc1) (ps-trim (- yscreen yc1))
	  (ps-trim xc2) (ps-trim (- yscreen yc2))
	  (ps-trim x2) (ps-trim (- yscreen y2))))


;;; Plot the infinity character at coordinates (x,y).  Currently
;;; implemented only in one size.

						
(defun qplot-infinity (x y)
  (qplot-special-char (round (- x 4))
		      (round (+ y *symbol-y-offset*))
		      #\ 165))

;;; QPLOT-STRING-W-INFINITY is similar to {\cf qplot-string}, but a check is
;;; made for the presence of the infinity character (character code 14,
;;; produced by <symbol>-i), and, when present, it is handled correctly in the
;;; postscript output.

(defun qplot-string-w-infinity (string x y &key (alu *black*)(font *plain-font*))
  (when (image-to-screen-p)
    (graphics:draw-string string
			  (round x) (round y) :character-style font
			  :alu alu :stream *qplot-output*))
  (when (image-to-postscript-p)
    (ps-draw-string-w-infinity string x y :alu alu :font font)))

(defun ps-draw-string-w-infinity (string x y &key (alu *black*)(font *plain-font*))
  (unless (zerop (length (format nil "~a" string)))
    (checkgray alu)
    (let* ((*print-case* :upcase)
	   (*readtable* *ps-out-readtable*)
	   (typeface (si:char-style  (char (merge-font-info font #\a) 0)))
	   (font (translate-typeface typeface))
	   (size (size-translation (si:cs-size typeface))) 
	   (pair (list font size))
	   (familiar-font			; will be something like 
	     (assoc pair *used-fonts* :test #'equal))	; ((:|Times-Bold| 18.) TIMES-BOLD-7)
	   (gatom (if familiar-font		
		      (second familiar-font)
		      (gentemp (format nil "~:@(~a-~)" font))))
	   (symbol-pair (list :|Symbol| size))
	   (symbol-font (assoc symbol-pair  *used-fonts* :test #'equal))
	   (satom (if symbol-font 		
		      (second symbol-font)
		      (gentemp "SYMBOL-")))
	   (*standard-output* *postscript-output-file* ))
      (unless symbol-font			; Load the Symbol font, if it hasn't 
	(load-font satom :|Symbol| size symbol-pair))	; already been.
      (unless familiar-font			; Load the font of the rest of the 
	(load-font gatom font size pair))	; string, if it hasn't already been.
	
      (enforce-string font t size gatom pair)	; start off in the character font.
      (format t " ~a ~a m (" (ps-trim x) (ps-trim (- yscreen y)))	; goto coordinates
      ;; Loop thru string char at a time, flipping in and out of Symbol font to print infinities.
      (loop for char being the array-element of (strip-font-info (ps-string string))
	    if (or (equal char #\infinity)
		   (equal char #\Epsilon)
		   (equal char #\Delta))
	      do (print-greek char font size gatom pair satom symbol-pair)
	    else do (prin1 (string char))	; slashify #\( #\) etc.
	    finally (princ ") p ")))))

(defun print-greek (char font size gatom pair satom symbol-pair)
  (format  t ") p ")
  (enforce-string :|Symbol| t size satom symbol-pair)
  (format t "(\\~o) p "
	  (cond ((equal char #\infinity) 165)
		((equal char #\Epsilon) 206)
		((equal char #\Delta) 68)))
  (enforce-string font t size gatom pair)
  (princ "("))

(defun qplot-x-centered-string (string x y &key (alu *black*)(font *plain-font*))
  (setf string (string-conversion string))
  (setf x (- x (/ (- (graphics:draw-string string
		        (round x) (round y) :character-style font
			:alu tv:alu-and :stream *qplot-output*) x)
		  2)))
  (when (image-to-postscript-p)
    (ps-draw-string string x y :alu alu :font font))
  (when (image-to-screen-p)
   (graphics:draw-string string
			 (round x) (round y) :character-style font
			 :alu alu :stream *qplot-output*)))


;;; The RESIZING macro provides a quick way to change the mapping
;;; between a Symbolics fontsize and a PostScript pointsize.  This is
;;; particularly useful when creating documents which use typefaces
;;; larger than 24 point - :very-large can be mapped into a larger
;;; pointsize.  Although the effect will show a smaller than desired
;;; font on the screen, the hardcopy will be correct.

(defmacro resizing (fontsize pointsize &rest body)
  `(let ((*size-translations* (cons (list ,fontsize ,pointsize)
				     *size-translations*)))
     ,@body))


;;; This causes a layout of all of the PostScript characters in the
;;; Symbol font, listed by their decimal ascii character code.  It
;;; prints something similar on the Symbolics screen, but just prints
;;; the character #\x instead of the special character.

(defun symbols (&optional (lim 255) )
  (device-interface
    (qplot-new-behavior)
    (loop for count from 1 to lim
	  with yspace = 20
	  with nrows = (floor (/ (- yscreen 100) yspace))
	  for column = (floor (/ count nrows))
	  for row = (mod count nrows)
	  for x = (+ 20 (* 100 column))
	  for y = (+ 40 (* yspace row))
	  do (qplot-string count x y)
	     (qplot-special-char (+ x 40) y #\x count))
    (qplot-end-display)))

;;; QPLOT-BRACKET plots a bracket from {\cf (x1,y1) to (x2,y2)} The ``width"
;;; of the bracket will be the distance between the points times {\cf
;;; offsetfac}.

(defun qplot-bracket (x1 y1 x2 y2 &key (thickness 1)(offsetfac .05))
   (let ((pairlistpoints (bracket-lists x1 y1 x2 y2 offsetfac)))
    (qplot-cubic-spline (first pairlistpoints) :thickness thickness)
    (qplot-cubic-spline (second pairlistpoints) :thickness thickness)))



(defun bracket-lists (x1 y1 x6 y6 offsetfac)
  (let* ((len (sqrt (+ (expt (- x6 x1) 2)(expt (- y6 y1) 2))))
	 (offset (* offsetfac len))
	 (theta (zl:atan (- y6 y1)(- x6 x1)))
	 (invtheta (zl:atan (- x1 x6)(- y6 y1)))
	 (cos (cos theta))
	 (sin (sin theta))
	 (cosi (cos invtheta))
	 (sini (sin invtheta))
	 (x2 (+ x1 (* len .2 cos)))
	 (x3 (+ x1 (* len .4 cos)))
	 (xm (+ x1 (* len .5 cos)))
	 (x4 (+ x1 (* len .6 cos)))
	 (x5 (+ x1 (* len .8 cos)))
	 (y2 (+ y1 (* len .2 sin)))
	 (y3 (+ y1 (* len .4 sin)))
	 (ym (+ y1 (* len .5 sin)))
	 (y4 (+ y1 (* len .6 sin)))
	 (y5 (+ y1 (* len .8 sin)))
	 (x2p (+ x2 (* offset cosi)))
	 (x3p (+ x3 (* offset cosi)))
	 (xmp (+ xm (* 2 offset cosi)))
	 (x4p (+ x4 (* offset cosi)))
	 (x5p (+ x5 (* offset cosi)))
	 (y2p (+ y2 (* offset sini)))
	 (y3p (+ y3 (* offset sini)))
	 (ymp (+ ym (* 2 offset sini)))
	 (y4p (+ y4 (* offset sini)))
	 (y5p (+ y5 (* offset sini))))
    (list
      (list x1 y1 x2p y2p x3p y3p xmp ymp)
      (list x6 y6 x5p y5p x4p y4p xmp ymp))))








