;;; -*- Mode:Common-Lisp; Package:POS; Base:10 -*-
;;; $Id: qgraph-io.lisp,v 1.6 92/07/15 12:24:46 bert Exp $

(in-package 'pos)


;;; This file is part of the qgraph system.
;;; The qgraph system is composed of the following files :
;;;      qgraph         - The main code for qgraph.
;;;      qgraph-io      - File I/O and plotting functions
;;; In addition, the file following files provide the interface to qsim :
;;;      numeric-plot   - Interface to the qsim display handler
;;;      qsim-to-qgraph - Code for extracting qsim structures.

;;; This file contains the following sections :
;;; 1.  Writing qgraphs to files.
;;; 2.  Reading qgraphs from files.
;;; 3.  Qgraph plotting functions.
;;; 4.  Utilities for creating datasets.


;;; Qgraph structures can be saved to files in an ascii representation.
;;; This permits sharing of qgraphs across machine types as well
;;; as permitting other data analysis tools to use the same datasets
;;; as does qgraph.  For instance, an xgraph file can be plotted by
;;; qgraph with only slight reformatting.

;;; Qgraph file format.
;;; slot value
;;; ...
;;; datasets
;;; ...

;;; We now maintain a list of all qgraphs and datasets
;;;
(defparameter *qgraphs* nil)
(defparameter *datasets* nil)
(defparameter *current-qgraph* nil)


;;; An alist of (qgraph-slot-name file-parameter-name default) entries.
;;;
(defparameter *OLD-QGRAPH-SLOT-XFORMER*
	      `((qgraph-XLimits            XLimits nil)
		(qgraph-YLimits            YLimits nil)
		(qgraph-XQualScaling       XQualScaling nil)
		(qgraph-YQualScaling       YQualScaling nil)
		(qgraph-LnX                LnX nil)
		(qgraph-LnY                LnY nil)
		(qgraph-XUnitText          XUnitText "X")
		(qgraph-YUnitText          YUnitText "Y")
		(qgraph-QualPlottingMethod QualPlottingMethod 'box)
		(qgraph-TickFlag           TickFlag nil)
		(qgraph-ZeroWidth          ZeroWidth 2)
		(qgraph-Gridwidth          Gridwidth 1)
		(qgraph-Colors             Colors nil)
		(qgraph-Styles             Styles nil)
;		(qgraph-RangeStyle         RangeStyle 1)
;		(qgraph-IntervalStyle      IntervalStyle 5)
;		(qgraph-RangeColor         RangeColor "red")
;		(qgraph-IntervalColor      IntervalColor "black")
		))

(defparameter *QGRAPH-SLOT-XFORMER*
	      `((qgraph-Name               Name nil)
		(qgraph-Documentation      Documentation nil)
		(qgraph-Timestamp          Timestamp NIL)
		(qgraph-XLimits            XLimits nil)
		(qgraph-YLimits            YLimits nil)
		(qgraph-XQualScaling       XQualScaling nil)
		(qgraph-YQualScaling       YQualScaling nil)
		(qgraph-LnX                LnX nil)
		(qgraph-LnY                LnY nil)
		(qgraph-XUnitText          XUnitText "X")
		(qgraph-YUnitText          YUnitText "Y")
		(qgraph-QualPlottingMethod QualPlottingMethod 'box)
		(qgraph-TickFlag           TickFlag nil)
		(qgraph-ZeroWidth          ZeroWidth 2)
		(qgraph-Gridwidth          Gridwidth 1)
;		(qgraph-Colors             Colors nil)
;		(qgraph-Styles             Styles nil)
		))



(defparameter lmargin 20.)
(defparameter rmargin 40.)
(defparameter tmargin 100.)
; (defparameter bmargin 50.) ; already defined in machine-params.lisp

(defparameter xsep 30. "X pixels separation between graphs")
(defparameter ysep 25. "Y pixels separation between graphs")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 1.  Writing qgraphs to files.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Write a qgraph structure to an ASCII file.
;;;
(defun qgraph-write-file (s file)
  (let ((old-package *package*))
    (setf *package* (find-package "POS"))
    (unwind-protect
	 (with-open-file (stream file :direction :output 
				 :if-does-not-exist :create
				 :if-exists :supersede)
	   (qgraph-to-stream s stream))
      (setf *package* old-package))
    file))


;;; Send a qgraph to a stream.
;;;
(defun qgraph-to-stream (s stream)
  (dolist (item *QGRAPH-SLOT-XFORMER*)
    (declare (special *QGRAPH-SLOT-XFORMER*))
    (when (not (equal (funcall (first item) s) (third item)))
      (format stream "~%~a ~a" (second item) (funcall (first item) s))))
  (loop for dataset in (qgraph-data s)
	for color   in (qgraph-colors s)
	for style   in (qgraph-styles s)
	do
	(terpri stream)
	(typecase dataset
	  (Line  (line-to-stream dataset color style stream))
	  (TQual (tqual-to-stream dataset color style stream))
	  (TEnv  (tenv-to-stream dataset color style stream))
	  (TQ3   (tq3-to-stream dataset color style stream))    ; rep for Dan Berleant's Q3
      )))


;;; The format for a LINE entry is:
;;; LINE symbol
;;; [TRENDNAME string]
;;; [DOCUMENTATION string]
;;; TIMESTAMP string
;;; COLOR string
;;; STYLE list
;;; number number [delta-or-lo-number] [hi-number]
;;; ...
;;;
(defun line-to-stream (line color style stream)
  (format stream "~%LINE ~a" (Line-Name line))
  (when (Line-Trendname Line)
    (format stream "~%TRENDNAME ~a" (Line-Trendname Line)))
  (when (Line-Documentation Line)
    (format stream "~%DOCUMENTATION ~a" (Line-Documentation Line)))
  (when (Line-Timestamp Line)
    (format stream "~%TIMESTAMP ~a" (Line-Timestamp Line)))
  (format stream "~%COLOR ~a" color)
  (format stream "~%STYLE ~a" style)
  (dolist (point (qlist (Line-Data line)))
    (format stream "~%~{~a  ~}" point))
  (format stream "~%"))


;;; The format for a TQUAL entry is:
;;; TQUAL symbol
;;; [TRENDNAME string]
;;; [DOCUMENTATION string]
;;; [TIMESTAMP string]
;;; COLOR (string string)
;;; STYLE (num num)
;;; TLANDMARKS
;;; symbol num num
;;; ...
;;; YLANDMARKS
;;; symbol num num
;;; ...
;;; DATA
;;; list-or-symbol list-or-symbol symbol
;;; ...
;;;
(defun tqual-to-stream (tqual color style stream)
  (format stream "~%TQUAL ~a" (TQual-Name tqual))
  (when (TQual-Trendname tqual)
    (format stream "~%TRENDNAME ~s" (TQual-TrendName tqual)))
  (when (TQual-Documentation tqual)
    (format stream "~%DOCUMENTATION ~s" (TQual-Documentation tqual)))
  (format stream "~%COLOR ~s" color)
  (format stream "~%STYLE ~a" style)
  (when (TQual-Timestamp tqual)
    (format stream "~%TIMESTAMP ~s" (TQual-Timestamp tqual)))
  (format stream "~%TLANDMARKS")
  (loop for entry in (first (TQual-Lmarks tqual))
	do (format stream "~%~{~a ~}" entry))
  (format stream "~%YLANDMARKS")
  (loop for entry in (second (TQual-Lmarks tqual))
	do (format stream "~%~{~a ~}" entry))
  (format stream "~%DATA")
  (dolist (point (TQual-Data tqual))
    (format stream "~%~{~a ~}" point))
  (format stream "~%"))


;;; The format for a TENV entry is:
;;; TENV symbol
;;; [TRENDNAME string]
;;; [DOCUMENTATION string]
;;; [TIMESTAMP string]
;;; COLOR string
;;; STYLE list
;;; number number number
;;; ...
;;;
(defun tenv-to-stream (tenv color style stream)
  (format stream "~%(TENV ~a" (TEnv-Name tenv))
  (when (TEnv-TrendName tenv)
    (format stream "~%TRENDNAME ~s" (TEnv-TrendName tenv)))
  (when (TEnv-Documentation tenv)
    (format stream "~%DOCUMENTATION ~s" (TEnv-Documentation tenv)))
  (when (TEnv-timestamp tenv)
    (format stream "~%TIMESTAMP ~s" (TEnv-Timestamp tenv)))
  (format stream "~%COLOR ~s" color)
  (format stream "~%STYLE ~a" style)
  (dolist (point (qlist (TEnv-Data tenv)))
    (format stream "~%~{  ~a~} " point))
  (format stream "~%"))


;;; Rep for Dan Berleant's Q3.
;;; This is the same as for TQUAL except that the name is TQ3.
;;;
(defun tq3-to-stream (tqual color style stream)
  (format stream "~%TQUAL ~a" (TQ3-Name tqual))
  (when (TQ3-Trendname tqual)
    (format stream "~%TRENDNAME ~s" (TQ3-TrendName tqual)))
  (when (TQ3-Documentation tqual)
    (format stream "~%DOCUMENTATION ~s" (TQ3-Documentation tqual)))
  (when (TQ3-timestamp tqual)
    (format stream "~%TIMESTAMP ~s" (TQ3-Timestamp tqual)))
  (format stream "~%COLOR ~s" color)
  (format stream "~%STYLE ~a" style)
  (format stream "~%TLANDMARKS")
  (format stream "~{~%~a  ~a  ~a~}" (first (TQ3-Lmarks tqual)))
  (format stream "~%YLANDMARKS")
  (format stream "~{~%~a  ~a  ~a~}" (second (TQ3-Lmarks tqual)))
  (format stream "~%DATA")
  (dolist (point (TQ3-Data tqual))
    (format stream "~%~{~a ~}" point))
  (format stream "~%)"))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 2.  Reading qgraphs from files.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;; Create a qgraph structure from a file.
;;; Implementation:  This is a little tricky because this function
;;;      will normally be called from another package, but slot names
;;;      and such must refer to POS.  To handle this, we switch to
;;;      package POS while reading symbols.  This means that certain
;;;      symbols such as object name slots will be interned in POS.
;;;      To fix this, we explicitly export those symbols that we
;;;      we come across (other than pos slot names).
;;;      We don't have to worry about this on output since we use
;;;      ~a to write symbols, which automatically ignores their package.
;;;      This also solves the problem of writing out symbols (line names)
;;;      defined in another package.
;;;
(defun qgraph-read-file (file &key (old nil))
  (setf *current-qgraph*
	(car (push (let (qg
			 (*old-package* *package*))
		     (declare (special *old-package* *package*))
		     (setf *package* (find-package "POS"))
		     (unwind-protect
			  (with-open-file (stream file)
			    (setf qg
				  (if old
				      (old-stream-to-qgraph stream)
				      (stream-to-qgraph stream))))
		       (setf *package* *old-package*))
		     qg)
		   *qgraphs*))))


(defun stream-to-qgraph (stream)
  (declare (special *old-package*))
  (let ((s (make-qgraph :colors NIL :styles NIL
			:Id (gentemp "QG-" *old-package*)))
	name)
    (loop for line = (read-line stream nil nil)
	  then (read-line stream nil nil)
	  with start = 0
	  with type = nil
	  with slot-entry = nil
	  until (null line)
	  do
	  (multiple-value-setq (type start)
	    (read-from-string line nil nil))
	  (format t "~%Line= ~s   Type = ~a" line type)
	  (cond
	    ((null type))
	    ((setf slot-entry (find type *QGRAPH-SLOT-XFORMER* :key #'second))
	     (let ((item (read-from-string line nil nil :start start)))
	       ;; Make sure that text args are strings.
	       (if (and (member type
				'(xunittext yunittext documentation timestamp))
			(not (null item)))
		   (eval `(setf (,(car slot-entry) ,s)
			   (quote ,(string item))))
		   (progn
		     ;; Make any values of slots that contain symbols
		     ;; local to *old-package* (usually QSIM).
		     (when (symbolp item)
		       (export item)
		       (import item *old-package*))
		     (eval `(setf (,(car slot-entry) ,s)
			     (quote ,item)))))))
	    (T
	     (cond
	       ((member type '(Line TEnv TQual TQ3))
		(setf name (read-from-string line nil nil :start start))
		(when (and (not (null name)) (symbolp name))
		  (export name)
		  (import name *old-package*))
		(case type
		  (Line  (stream-to-line stream name s))
		  (TEnv  (stream-to-tenv stream name s))
		  (TQual (stream-to-tqual stream name s))
		  (TQ3   (stream-to-tq3 stream name s))))
	       (T     (error "Don't know about type ~a" type))))))
    s))


(defun stream-to-line (stream name qg)
  (let (dataset
	(*trendname* nil)
	(*color* "red")
	(*style* '(1 0))
	(*documentation* NIL)
	(*timestamp* NIL)
	datastart)
    (declare (special *start* *color* *style* *trendname*
		      *documentation* *timestamp* *old-package*))
    (setf datastart (qgraph-parse-dataset-header stream 'Line))
    (setf dataset
	  (create-line :name name :trendname *trendname*
		       :documentation *documentation*
		       :Id (gentemp "LI-" *old-package*)
		       :Timestamp *timestamp*
		       :data (qgraph-get-numeric-dataset stream datastart)))
    (push dataset (qgraph-data qg))
    (push *color* (qgraph-colors qg))
    (push *style* (qgraph-styles qg))
    (push dataset *datasets*)
    dataset))


(defun stream-to-tenv (stream name qg)
  (let (dataset
	(*trendname* nil)
	(*color* "red")
	(*style* '(1 0))
	(*documentation* NIL)
	(*timestamp* NIL)
	datastart)
    (declare (special *start* *color* *style* *trendname*
		      *documentation* *timestamp* *old-package*))
    (setf datastart (qgraph-parse-dataset-header stream 'TEnv))
    (setf dataset
	  (make-tenv :name name :trendname *trendname*
		     :documentation *documentation*
		     :timestamp *timestamp*
		     :Id (gentemp "TE-" *old-package*)
		     :data (qgraph-get-numeric-dataset stream datastart)))
    (push dataset (qgraph-data qg))
    (push *color* (qgraph-colors qg))
    (push *style* (qgraph-styles qg))
    (push dataset *datasets*)
    dataset))


(defun stream-to-tqual (stream name qg)
  (let (dataset
	(*trendname* nil)
	(*color* "red")
	(*style* '(1 0))
	(*documentation* NIL)
	(*timestamp* nil)
	lmarks data)
    (declare (special *start* *color* *style* *trendname*
		      *documentation* *timestamp* *old-package*))
    ;; We toss the datastart line since it contains the symbol TLANDMARKS
    (qgraph-parse-dataset-header stream 'TQual)
    (multiple-value-setq (lmarks data)
      (qgraph-get-qual-dataset stream))
    (setf dataset
	  (make-tqual :Id (gentemp "TQ-" *old-package*)
		      :name name :trendname *trendname*
		      :documentation *documentation*
		      :timestamp *timestamp*
		      :lmarks lmarks :data data))
    (push dataset (qgraph-data qg))
    (push *color* (qgraph-colors qg))
    (push *style* (qgraph-styles qg))
    (push dataset *datasets*)
    dataset))


(defun stream-to-tq3 (stream name qg)
  (let (dataset
	(*trendname* nil)
	(*color* "red")
	(*style* '(1 0))
	(*documentation* NIL)
	(*timestamp* NIL)
	lmarks data)
    (declare (special *start* *color* *style* *trendname*
		      *documentation* *timestamp* *old-package*))
    (qgraph-parse-dataset-header stream 'TQ3)
    (multiple-value-setq (lmarks data)
      (qgraph-get-qual-dataset stream))
    (setf dataset
	  (make-tq3 :Id (gentemp "TQ3-" *old-package*)
		    :name name :trendname *trendname*
		    :documentation *documentation*
		    :timestamp *timestamp*
		    :lmarks lmarks :data data))
    (push dataset (qgraph-data qg))
    (push *color* (qgraph-colors qg))
    (push *style* (qgraph-styles qg))
    (push dataset *datasets*)
    dataset))


	    
;;; Get the color, style, trendname, and documentation entries in a
;;; stream.
;;; Inputs:  stream       - A stream.
;;;          dataset-type - The typename of the dataset.
;;; Returns: The line that showed that the header was complete.
;;;
(defun qgraph-parse-dataset-header (stream dataset-type)
  (declare (special *color* *style* *trendname* *documentation*))
  (loop for line = (read-line stream nil nil)
	then (read-line stream nil nil)
	for cmd = (read-from-string line nil nil)
	then (read-from-string line nil nil)
	until (or (null line)
		  (null cmd)
		  (case dataset-type
		    ((Line Tenv) (numberp cmd))
		    ((TQual TQ3) (eq cmd 'TLANDMARKS))
		    (T           (error "Unknown data type ~A" dataset-type))))
	do
	(case cmd
	  (COLOR      (setf *color*
			   (case dataset-type
				  ((Line Tenv) (qgraph-get-string line
								  :start 5))
				  ((TQual TQ3) (read-from-string line nil nil
								 :start 5)))))
	  (STYLE         (setf *style* (read-from-string line nil nil
							 :start 5)))
	  (TRENDNAME     (setf *trendname* (qgraph-get-string line :start 9)))
	  (DOCUMENTATION (setf *documentation* (qgraph-get-string line :start 13))))
	finally (if (or (null line) (null cmd))
		    (error "Syntax error for line ~s" line)
		    (return line))))
	

;;; Treat the remainder of a line as a string and return it with blanks
;;; trimmed.
;;;
(defun qgraph-get-string (str &key (start 0))
  (string-trim '(#\Space #\Tab) (subseq str start)))


;;; Return the lmark and data components of a qual dataset.
;;; Inputs:  stream - A stream.
;;; Returns: #1: The lmark slot for a qual dataset.
;;;          #2: The data slot for a qual dataset.
;;;
(defun qgraph-get-qual-dataset (stream)
  (let (lmarks data)
    (setf lmarks
	  (list
	   (loop for line = (read-line stream nil nil)
		 then (read-line stream nil nil)
		 for entry = (qgraph-read-sexprs line :export t)
		 then (qgraph-read-sexprs line :export t)
		 until (or (null line)
			   (null entry)
			   (and (eq (car entry) 'YLANDMARKS)
				(= (length entry) 1)))
		 collect entry
		 finally (when (or (null line) (null entry))
			   (error "No YLANDMARKS found")))
	   (loop for line = (read-line stream nil nil)
		 then (read-line stream nil nil)
		 for entry = (qgraph-read-sexprs line :export t)
		 then (qgraph-read-sexprs line :export t)
		 until (or (null line)
			   (null entry)
			   (and (eq (car entry) 'DATA)
				(= (length entry) 1)))
		 collect entry
		 finally (when (or (null line) (null entry))
			   (error "No DATA found")))))
    (setf data
	  (loop for line = (read-line stream nil nil)
		then (read-line stream nil nil)
		for entry = (qgraph-read-sexprs line :export t)
		then (qgraph-read-sexprs line :export t)
		until (or (null line)
			  (null entry))
		 collect entry))
    (values lmarks data)))


;;; Get a list of the sexprs in a string.
;;; Inputs:  str - A string or NIL.
;;; Returns: A list of the sexprs in string or NIL.
;;;
(defun qgraph-read-sexprs (str &key (export NIL))
  (let (*start*)
    (declare (special *start* *old-package*))
    (when str
      (loop for val = (qgraph-get-sexpr-from-string str :from-start T)
	    then (qgraph-get-sexpr-from-string str)
	    until (null val)
	    collect (progn
		      (when (and export (symbolp val))
			(export val)
			(import val *old-package*))
		      val)))))


;;; Get a sexpr from a string.
;;; Inputs:  str - A string.
;;;          from-start - If T, sets the reader pointer to the start
;;;                       of the string and otherwise leaves it where it is.
;;; Returns: The next sexpr in str or NIL.
;;;
(defun qgraph-get-sexpr-from-string (str &key (from-start nil))
  (let (sexpr)
    (declare (special *start*))
    (when from-start
      (setf *start* 0))
    (multiple-value-setq (sexpr *start*)
      (read-from-string str nil nil :start *start*))
    sexpr))


#| Under construction   BKay 27Jun92
(defun get-substring-in-string (str &key (start 0))
  (let ((start-of-string (look-for-whitespace str start :option :skip))
	end-of-string)
    (cond
      ((>= start-of-string (length str))
       (values nil atart-of-string))
      (T
       (setf end-of-string (look-for-whitespace str start-of-string :option :find))
       (values (subseq str start-of-string (1- end-of-string))
	       end-of-string)))))


;;; Returns:  The char index of the first non-blank in :skip
;;;           The char index of the first blank in :find
(defun look-for-whitespace (str start &key (whitespace '(#\Tab #\Space))
				(option :skip))
  (loop for i from start upto (length str)
	for c = (elt str i)
	for done = (if (eq option :skip)
		       (not (member c whitespace))
		       (member c whitespace))
	until done
	finally (return i)))

|#

;;; Return a list of number entries from a stream terminated by a
;;; blank line.
;;; Inputs:  stream - A stream.
;;; Returns: A list of items of the form (num num ...) for each line
;;;          of the stream up to a blank line.
;;;
(defun qgraph-get-numeric-dataset (stream firstline)
  (loop for line = firstline
	then (read-line stream nil nil)
	for point = (qgraph-read-nums line)
	then (qgraph-read-nums line)
	until (or (null line) (null point))
	collect point))


;;; Return a list of numbers in str.
;;; Signal an error if there is something other than numbers found.
;;;
(defun qgraph-read-nums (str)
  (let ((nums (qgraph-read-sexprs str)))
    (if (every #'numberp nums)
	nums
	(error "Non-numeric entry found in ~s" str))))





;;; This function is used to retrieve things from the old file format.
;;;

;;; Read a stream and create a qgraph structure.
;;; This uses the old file format.
;;;
(defun old-stream-to-qgraph (stream)
  (let ((s (make-qgraph)))
    (do
     ((item (read stream nil 'DONE) (read stream nil 'DONE))
      type
      slot-entry)
     ((eq item 'DONE) s)
     (setf type (car item))
     (cond
      ((setf slot-entry (find type *OLD-QGRAPH-SLOT-XFORMER* :key #'second))
       ;; Make sure that unit text is a string.
       (if (and (member type '(xunittext yunittext))
		(not (null (cadr item))))
	   (eval `(setf (,(car slot-entry) ,s) (quote ,(string (cadr item)))))
	   (eval `(setf (,(car slot-entry) ,s) (quote ,(cadr item))))))
      ((eq type 'Line)
       (push (old-stream-to-line item) (qgraph-data s))
       (push (first (qgraph-data s)) *datasets*))
      ((eq type 'TEnv)
       (push (old-stream-to-tenv item) (qgraph-data s))
       (push (first (qgraph-data s)) *datasets*))
      ((eq type 'TQual)
       (push (old-stream-to-tqual item) (qgraph-data s))
       (push (first (qgraph-data s)) *datasets*))
      ((eq type 'TQ3)                                 ; Dan Berleant's Q3
       (push (old-stream-to-tq3   item) (qgraph-data s))
       (push (first (qgraph-data s)) *datasets*))
      ;; Handle Range and color tqual entries differently
      ;; (this means ignoring these entries).
      ((member type '(RangeStyle IntervalStyle RangeColor IntervalColor)))
      (T
       (error "Don't know about type ~a" type))))
    ;; Want to preserve the order for color and style orderings
    ;; and insert the proper entries for the TQuals.
    (setf (qgraph-data s) (nreverse (qgraph-data s)))
    (when (< (length (qgraph-colors s)) (length (qgraph-data s)))
      (setf (qgraph-colors s)
	    (append (qgraph-colors s)
		    (loop for i from 0 upto (- (length (qgraph-data s))
					       (length (qgraph-colors s)))
			  collect "red"))))
    (when (< (length (qgraph-colors s)) (length (qgraph-data s)))
      (loop for i from 0 upto (- (length (qgraph-colors s))
				 (length (qgraph-data s)))
	    do (push "red" (qgraph-colors s))))
    (when (< (length (qgraph-styles s)) (length (qgraph-data s)))
      (setf (qgraph-styles s)
	    (append (qgraph-styles s)
		    (loop for i from 0 upto (- (length (qgraph-data s))
					       (length (qgraph-styles s)))
			  collect '(1 0)))))

    ;; All the lists are the same length, so now we just have to clean up the
    ;; Quals
    (loop for d on (qgraph-data s)
	  for c on (qgraph-colors s)
	  for s on (qgraph-styles s)
	  when (or (tqual-p (car d)) (tq3-p (car d)))
	  do
	  (setf (car c) '("red" "black"))
	  (setf (car s) '(1 5)))
    s))

(defun old-stream-to-line (item)
  (let ((line (make-line :TrendName (second item)
			 :Data      (make-q))))
    (dolist (point (cddr item))
      (qpush point (line-data line)))
    line))

(defun old-stream-to-tenv (item)
  (let ((tenv (make-tenv :TrendName (second item)
			 :Data      (make-q))))
    (dolist (point (cddr item))
      (qpush point (tenv-data tenv)))
    tenv))

(defun old-stream-to-tqual (item)
  (make-tqual :Trendname (second item)
	      :Lmarks    (third item)
	      :Data      (cdddr item)))

;;; Dan Berleant's Q3
(defun old-stream-to-tq3 (item)
  (make-tq3   :Trendname (second item)
	      :Lmarks    (third item)
	      :Data      (cdddr item)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 2.  Qgraph plotting functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; This is a the top-level interface for plotting a qgraph.
;;; For lispview, it decides whether or not to actually draw to the screen.
;;; If not then it pushes the request onto the screen queue.
;;; Without lispview, this just passes control to qgraph-structure-internal.
;;;
(defun qgraph-structure (s
			 &key (xsize 600) (ysize 600) (xpos 20) (ypos 20))
  ;; Set the qgraph-window structure if it is undefined.
  ;; This shouldn't be in qgraph-structure-internal because the qgraph-window
  ;; needs to be available to other future qgraph calls (which may execute before
  ;; a damage event is sent to the window).
  (when (null (qgraph-window s))
    (setf (qgraph-window s)
	  (make-QgraphWindow
	   :WindowX xpos
	   :WindowY ypos
	   :XSize   xsize  ; Xscreen is max
	   :YSize   ysize  ; (- Yscreen pos:Bmargin) is max
	   :Font    *Plain-Font*
	   :FontCharHeight #+ti (w:font-char-height (w:font-evaluate *Plain-Font*))
	                   #+symbolics (zl:font-char-height
					(scl:with-character-style
					    (*Plain-Font*)
					  (send *qplot-output* :current-font)))
			   #+unix 15
	   :FontCharWidth #+ti (w:font-char-width (w:font-evaluate *Plain-Font*))
	                  #+symbolics (zl:font-char-width
				       (scl:with-character-style
					   (*Plain-Font*)
					 (send *qplot-output* :current-font)))
			  #+unix 9
			  )))
  (when (image-to-postscript-p)
    (with-plotting-to-screen-inhibited
	(qgraph-structure-internal s)))
  (when (image-to-screen-p)
    #+:pos-lispview  (progn
		       (if (pos::use-lv-call *qplot-output*)
			   (qgraph-structure-internal s)
			   (pos::qpush
			    `(qgraph-structure ,s
			                       :xsize ,xsize :ysize ,ysize
					       :xpos ,xpos :ypos ,ypos)
			    (pos::commands *qplot-output*))))
    #-:pos-lispview  (with-plotting-to-postscript-inhibited
			 (qgraph-structure-internal s))
    ))



;;; These functions determine the size of the plot for use by
;;; pos::determine-plotting-size.
;;;
#+:pos-lispview (push `(qgraph-structure
			,#'(lambda (cmd) (+ (fourth cmd) (eighth cmd))) ; width
			,#'(lambda (cmd) (+ (sixth cmd) (tenth cmd))))
		      pos::*determine-plotting-size-extensions*)
		    

;;; Plot a graph structure.  This is for backwards compatibility.
;;; The real function is "qgraph-structure".
;;;
(defun qgraph-from-struct (s &key (xsize 600) (ysize 600) (xpos 20) (ypos 20))
  (qgraph-structure s :xsize xsize :ysize ysize :xpos xpos :ypos ypos))

(defun qgraph-from-struct-wrapped (s &key (xsize 600) (ysize 600) (xpos 20) (ypos 20))
  (device-interface
   (qplot-new-behavior)
   (qgraph-structure s :xsize xsize :ysize ysize :xpos xpos :ypos ypos)
   (qplot-end-display)))


;;; Plot a qgraph-structure.
;;; Inputs:  s           - A qgraph structure
;;; Returns: nothing
;;;
(defun qgraph-structure-internal (s)
  (qplot-box (qgraphWindow-WindowX (qgraph-window s))
	     (qgraphWindow-WindowY (qgraph-window s))
	     (qgraphWindow-XSize (qgraph-window s))
	     (qgraphWindow-YSize (qgraph-window s))
	     :thickness 2)
  (qgraph s))


;;; Wrap a call to qgraph-structure with a device-interface call.
;;; This function is not used by qsim, but it might be used by other
;;; applications.
;;; style - One of :standard or :bounded.  Bounded style is suitable for
;;;         inserting in a latex file, although it is probably not standalone
;;;         printable because it doesn't translate, rotate, or scale the
;;;         image, so it will not print completely on the paper.
;;; Note also that xpos and ypos offset the image wrt the origin of the window,
;;; but they don't detract from xsize and ysize.
;;;
(defun qgraph-structure-wrapped (s &key (style :standard)
				   (xsize 600) (ysize 600) (xpos 20) (ypos 20))
  ;; In standard mode, the default Xscreen and Yscreen are used so that
  ;; the plot won't fill the display area.  For :bounded, we reset these
  ;; values and compute the bounding box in XPOS coordinates.
  ;; Also, we only compute the bounding box in :bounded mode.
  ;; It is probably a disaster to use anything other than :standard or :bounded as
  ;; styles
  (if (eq style :bounded)
      (let* ((XScreen (+ xsize xpos))
	     (YScreen (+ ysize ypos))
	     ;; We want postscript user coords to be the same as postscript default
	     ;; coords, so we shut off translation, scaling, and rotation.
	     (*x-translation* 0)
	     (*y-translation* 0)
	     (*x-scale* 1.0)
	     (*y-scale* 1.0)
	     (*rotation* 0)
	     ;; I tried using text-format-insert, but this was a disaster.
	     ;; We would like to shut off the scaling so that postscript user coords =
	     ;; postscript default coords, but it can't be done in :Text-format-insert.
	     ;; In fact, this code is really screwed up --
	     ;;   :text-format-insert insists on scaling, plus it decides the correct
	     ;;      scale.  This would be OK, except that it binds it locally in
	     ;;      ps-new-behavior, so other things don't see it.
	     ;; 
	     (*postscript-style* style)
	     (*bounding-box* (list xpos YScreen XScreen ypos)))
	(device-interface 
	 (qplot-new-behavior)
	 (qgraph-structure s :xsize xsize :ysize ysize :xpos xpos :ypos ypos)
	 (qplot-end-display)))
      (device-interface 
	 (qplot-new-behavior)
	 (qgraph-structure s :xsize xsize :ysize ysize :xpos xpos :ypos ypos)
	 (qplot-end-display))))


;;; Add a line (or tenv) structure to an existing qgraph.
;;; If immediate is non-NIL, then draw it immediately as well.
;;; Inputs:  trend     - A line or tenv object.
;;;          qgraph    - The qgraph to add it to.
;;;          immediate - T or NIL.
;;; Returns: The qgraph object.
;;; Notes:   The immediate keyword only works for lispview.
;;;
(defun qgraph-add-trend (trend qgraph &key (color "red") (style '(1 1))
			       (immediate NIL))
  (assert (and (or (tenv-p trend) (line-p trend) (tqual-p trend)) (qgraph-p qgraph)))
  (push trend  (qgraph-data qgraph))
  (push color (qgraph-colors qgraph))
  (push style (qgraph-styles qgraph))
  (when immediate
    #+:pos-lispview (with-plotting-to-lispview-forced 
		     (draw-one-dataset 0 qgraph))
    #-:pos-lispview (format t "~&QGRAPH-ADD-TREND: Immediate mode not ~
                                 implemented without lispview")
    ))


(defun qgraph-add-point (point trend qgraph &key (immediate NIL))
  (assert (and (qgraph-p qgraph) (or (tenv-p trend) (line-p trend))))
  (etypecase trend
    (line (if (<= (length point) 4)
	      (qpush point (line-data trend))
	      (error "Pushing ~a onto a Line not allowed." point)))
    (tenv (if (= (length point) 3)
	      (qpush point (tenv-data trend))
	      (error "Pushing ~a onto a TEnv not allowed." point)))
    )
  (when immediate
    #+:pos-lispview (with-plotting-to-lispview-forced
		     (etypecase trend
				(line (draw-data-line (position trend (qgraph-data qgraph)
								:test #'equal)
						      qgraph
						      :last-segment-only t))
		      ))
    #-:pos-lispview (format t "~&QGRAPH-ADD-POINT: Immediate mode not ~
                               implemented without lispview")
    ))
	  

;;;---------------------------------------------------------------------------
;;; First cut at a general labeled graphing facility for qgraphs.
;;; Ideally, this will take a label set and a layout-style set of qpgraphs and
;;; graph them.
;;; Inputs:  qgraphs - A layout-style list of lists that represent the layout
;;;                    of the qgraph.
;;;          text    - A list of strings.
;;; Returns: nothing.
;;; Notes:  The qgraphs are plotted together with a heading composed of the
;;;         strings.  Only the first 4 lines of the heading are displayed.
;;;---------------------------------------------------------------------------
;;;
(defun qgraph-display (qgraphs &key (text nil))
  (declare (special *xsize* *ysize*))
  (qgraph-compute-layout qgraphs)
  (device-interface
   (qplot-new-behavior)
  
   ;; Plot the display label
   (do ((index 0 (incf index))
	(maxlines 4)
	(xpos lmargin)
	(ypos 25)
	(yadv 15))
       ((or (>= index (min (length text) 4)))
	(if (> (length text) maxlines)
	    (format t
		    "~%Truncating text in call to QGRAPH-DISPLAY to 4 lines")))
     (qplot-string (nth index text) xpos ypos)
     (incf ypos yadv))
   
   ;; Plot the graphs
   ;; *xsize* and *ysize* set by compute-layout call in switch-plot-style
   (do ((rows qgraphs (cdr rows))
	(ypos tmargin (+ ypos *ysize* ysep)))
       ((null rows))
     (do ((qgraph-entries (car rows) (cdr qgraph-entries))
	  (qgraph-entry nil)
	  (xpos lmargin (+ xpos *xsize* xsep)))
	 ((null qgraph-entries))
       (setq qgraph-entry (car qgraph-entries))
       (if qgraph-entry
	   (qgraph-structure qgraph-entry :xpos xpos :ypos ypos
			     :xsize *xsize* :ysize *ysize*))))
   (qplot-end-display)))



(defun qgraph-compute-layout (layout)
  (qgraph-compute-rectangular-layout (length layout)
			      (apply #'max (mapcar #'length layout))))

;;; Compute the width and height of a single paramter plot.

(defun qgraph-compute-rectangular-layout (nrows ncols)
  (declare (special *xsize* *ysize*))
  (setq *xsize* (- (round (/ (+ (- xscreen lmargin rmargin) xsep)
			   ncols))
		 xsep))
  (setq *ysize* (- (round (/ (+ (- yscreen tmargin bmargin) ysep)
			   nrows))
		 ysep)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 3.  Utilities for creating datasets.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Create a line dataset from either a list of points or a
;;; queue.
;;; Modified by BKay2June92 to allow data = NIL (i.e., lines with
;;; no points).
;;;
(defun create-line (&rest keys &key (data NIL) &allow-other-keys)
  (let ((q (cond
	     ((queue-p data)
	      data)
	     ((or (null data) (consp data))
	      (let ((q (make-q)))
		(setf (queue-head q) data)
		(setf (queue-tail q) (last data))
		(setf (queue-len q) (length data))
		q))
	     (T
	      (error "Can't make a line out of ~a" data)))))
    (car (push (apply #'make-line
		      :data q :timestamp (generate-qgraph-timestamp)
		      :allow-other-keys t keys)
	       *datasets*))))



(defun create-qgraph (&rest keys &key &allow-other-keys)
  (setf *current-qgraph*
	(car (push (apply #'make-qgraph :timestamp (generate-qgraph-timestamp)
			  :colors NIL :styles NIL
			  :allow-other-keys t keys)
		   *qgraphs*))))
    



(defun generate-qgraph-timestamp ()
  (multiple-value-bind (sec min hr date month year)
      (get-decoded-time)
    (format nil "~a ~d ~d:~2,'0d:~2,'0d ~d"
	    (case month
	      (1 "Jan") (2 "Feb") (3 "Mar") (4 "Apr")
	      (5 "May") (6 "Jun") (7 "Jul") (8 "Aug")
	      (9 "Sep") (10 "Oct") (11 "Nov") (12 "Dec"))
	    date hr min sec year)))

	    


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 4.  Utilities for operating on qgraphs and datasets
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Set and redisplay a qgraph.
;;;
(defun qgs (option &rest args)
  (let ((selected-qgraph
	 (if (member option
		     '(:Name :Documentation :Xlimits :Ylimits
		       :XQualScaling :YQualScaling :Lnx :Lny
		       :XUnitText :YUnitText :QualPlottingMethod
		       :QualAxisNumLines :QualAxisNumNames
		       :TickFlag :ZeroWidth :GridWidth))
	     (get-qgraph args 1)
	     (case option
	       (:Color        (qgs-colors args))
	       (:Style        (qgs-styles args))
	       (:clear        (qgs-clear args))
	       (:add          (qgs-add args))
	       (:delete       (qgs-delete args))
	       (T             (error "Unknown option ~a" option))))))
    (case option
      (:Name         (setf (qgraph-name selected-qgraph) (first args)))
      (:Documentation      (setf (qgraph-Documentation selected-qgraph)
				 (first args)))
      (:Xlimits      (setf (qgraph-xlimits selected-qgraph) (first args)))
      (:Ylimits      (setf (qgraph-ylimits selected-qgraph) (first args)))
      (:XQualScaling (setf (qgraph-XQualScaling selected-qgraph)
			   (first args)))
      (:YQualScaling (setf (qgraph-YQualScaling selected-qgraph)
			   (first args)))
      (:Lnx          (setf (qgraph-lnx selected-qgraph) (first args)))
      (:Lny          (setf (qgraph-lny selected-qgraph) (first args)))
      (:XUnitText    (setf (qgraph-XUnitText selected-qgraph) (first args)))
      (:YUnitText    (setf (qgraph-YUnitText selected-qgraph) (first args)))
      (:QualPlottingMethod (setf (qgraph-QualPlottingMethod selected-qgraph)
				 (first args)))
      (:QualAxisNumLines   (setf (qgraph-QualAxisNumLines selected-qgraph)
				 (first args)))
      (:QualAxisNumNames   (setf (qgraph-QualAxisNumNames selected-qgraph)
				 (first args)))
      (:Tickflag     (setf (qgraph-Tickflag selected-qgraph) (first args)))
      (:ZeroWidth    (setf (qgraph-ZeroWidth selected-qgraph) (first args)))
      (:GridWidth    (setf (qgraph-GridWidth selected-qgraph) (first args))))
    #+:pos-lispview (when (qgraph-window selected-qgraph)
		      (redraw-qplot *qplot-output*))
    ))



;;; Set option for datasets
(defun qds (option val dataset)
  (case option
    (:Name         (set-dataset-name (get-qgraph-dataset dataset) val))
    (:Documentation      (set-dataset-Documentation (get-qgraph-dataset dataset)
						    val))
    (:Trendname    (set-dataset-trendname (get-qgraph-dataset dataset) val))))


(defun set-dataset-name (d name)
  (typecase d
    (Line  (setf (line-name d) name))
    (TEnv  (setf (TEnv-name d) name))
    (TQual (setf (TQual-name d) name))))


(defun set-dataset-documentation (d doc)
  (typecase d
    (Line  (setf (line-documentation d) doc))
    (TEnv  (setf (TEnv-documentation d) doc))
    (TQual (setf (TQual-documentation d) doc))))


(defun set-dataset-trendname (d trendname)
  (typecase d
    (Line  (setf (line-trendname d) trendname))
    (TEnv  (setf (TEnv-trendname d) trendname))
    (TQual (setf (TQual-trendname d) trendname))))


;;; Select the graph from the argument list or use the current one.
;;; Inputs:  args          - The argument list to qgs.
;;;          num-true-args - The number of required args (the one after
;;;                          that can be a qgraph spec).
;;; Returns: A qgraph structure.
;;;
(defun get-qgraph (args num-true-args)
  (let ((len (length args)))
    (when (> len (1+ num-true-args))
      (error "Too many arguments in ~a" args))
    (if (> len num-true-args)
	(let ((qgspec (first (last args))))
	  (if (qgraph-p qgspec)
	      qgspec
	      (or (find qgspec *qgraphs* :key #'qgraph-id)
		  (error "No qgraph named ~a in *QGRAPHS*" qgspec))))
	*current-qgraph*)))


(defun get-qgraph-dataset (dspec)
  (if (symbolp dspec)
      (or (find dspec
		*datasets*
		:key #'(lambda (d)
			 (case (type-of d)
			   (Line  (line-id d))
			   (TEnv  (tenv-id d))
			   (TQual (tqual-id d))
			   (PEnv  (penv-id d))
			   (PQual (pqual-id d))
			   (T     NIL))))
	  (error "No dataset with id ~a in *DATASETS*" dspec))
      dspec))


;;; Returns: The qgraph.
;;;
(defun qgs-colors (args)
  (let ((len (length args)))
    (when (< len 2)
      (error "Usage: (qgs :color dataset color [qgraph]"))
    (let* ((dataset  (get-qgraph-dataset (first args)))
	   (color    (second args))
	   (qgraph   (get-qgraph args 2))
	   (location (position dataset (qgraph-data qgraph))))
      (when (not (= (length (qgraph-colors qgraph))
		    (length (qgraph-data qgraph))))
	(error "Not a color for every dataset in ~a" qgraph))
      (setf (elt (qgraph-colors qgraph) location) color)
      qgraph)))


;;; Returns: The qgraph.
;;;
(defun qgs-styles (args)
  (let ((len (length args)))
    (when (< len 2)
      (error "Usage: (qgs :style dataset style [qgraph]"))
    (let* ((dataset  (get-qgraph-dataset (first args)))
	   (style    (second args))
	   (qgraph   (get-qgraph args 2))
	   (location (position dataset (qgraph-data qgraph))))
      (when (not (= (length (qgraph-styles qgraph))
		    (length (qgraph-data qgraph))))
	(error "Not a style for every dataset in ~a" qgraph))
      (setf (elt (qgraph-styles qgraph) location) style)
      qgraph)))


;;; Returns: The qgraph.
;;;
(defun qgs-clear (args)
  (let ((qgraph (get-qgraph args 0)))
    (setf (qgraph-data qgraph) nil)
    (setf (qgraph-colors qgraph) nil)
    (setf (qgraph-styles qgraph) nil)
    qgraph
    ))


;;; Returns: The qgraph.
;;;
(defun qgs-add (args)
  (when (null args)
    (error "Usage: (qgs :add object [:color color] [:style style] [qgraph])"))
  (let* ((default-styles (list '(1 0) '(5 1) '(2 2) '(4 2)))
	 (default-colors (list "red" "blue" "green" "black"))
	 (object (get-qgraph-dataset (first args)))
	 (pos-color (position :color args))
	 (pos-style (position :style args))
	 (color  (if pos-color (elt args (1+ pos-color))))
	 (style  (if pos-style (elt args (1+ pos-style))))
	 (qgraph (get-qgraph args (+ 1 (if pos-color 2 0) (if pos-style 2 0)))))
    (push object (qgraph-data qgraph))
    (push (or color (elt default-colors (mod (length (qgraph-data qgraph)) 4)))
	  (qgraph-colors qgraph))
    (push (or style (elt default-styles (mod (length (qgraph-data qgraph)) 4)))
	  (qgraph-styles qgraph))
    qgraph))


;;; Returns: The qgraph.
;;;
(defun qgs-delete (args)
  (when (null args)
    (error "Usage: (qgs :delete object [qgraph])"))
  (let* ((object (get-qgraph-dataset (first args)))
	 (qgraph (get-qgraph args 1))
	 (pos    (position object (qgraph-data qgraph))))
    (format t "~%object = ~a  qgraph = ~a  pos = ~a" object qgraph pos)
    (when (not (= (length (qgraph-styles qgraph))
		    (length (qgraph-data qgraph))))
      (error "Not a style for every dataset in ~a" qgraph))
    (when (not (= (length (qgraph-colors qgraph))
		    (length (qgraph-colors qgraph))))
      (error "Not a color for every dataset in ~a" qgraph))
    (when pos
      (setf (qgraph-data qgraph)
	    (delete-if #'(lambda (obj) (eql (position obj
						      (qgraph-data qgraph))
					    pos))
		       (qgraph-data qgraph)))
      (setf (qgraph-styles qgraph)
	    (delete-if #'(lambda (obj) (eql (position obj
						      (qgraph-styles qgraph))
					    pos))
		 (qgraph-styles qgraph)))
      (setf (qgraph-colors qgraph)
	    (delete-if #'(lambda (obj) (eql (position obj
						      (qgraph-colors qgraph))
					    pos))
		       (qgraph-colors qgraph))))
    qgraph))
	    
    
(defun show-ds (&optional id-or-dataset)
  (cond
    (id-or-dataset
     (show-dataset (if (symbolp id-or-dataset)
		       (find id-or-dataset *datasets*
			     :key #'(lambda (d)
				      (case (type-of d)
					(Line (Line-Id d))
					(TEnv (Tenv-Id d))
					(TQual(TQual-Id d)))))
		       id-or-dataset)))
    (T
     (loop for d in *datasets*
	   do
	   (show-dataset d)))))


(defun show-dataset (d)
  (format t "~%~a~8T~a~25T~a~32T~a~42T~a"
	  (case (type-of d)
	    (Line (Line-Id d))
	    (TEnv (Tenv-Id d))
	    (TQual(TQual-Id d)))
	  (case (type-of d)
	    (Line (Line-name d))
	    (TEnv (Tenv-name d))
	    (TQual(TQual-name d)))
	  (type-of d)
	  (case (type-of d)
	    (Line (Line-timestamp d))
	    (TEnv (Tenv-timestamp d))
	    (TQual(TQual-timestamp d)))
	  NIL))

#|
  (format t "~%Id:~15T~a" (case (type-of d)
			      (Line (Line-id d))
			      (TEnv (Tenv-id d))
			      (TQual(TQual-id d))))
  (format t "~%Name:~15T~a" (case (type-of d)
			      (Line (Line-name d))
			      (TEnv (Tenv-name d))
			      (TQual(TQual-name d))))
  (format t "~%Type:~15T~a" (type-of d))
  (format t "~%Trendname:~15T~s" (case (type-of d)
				   (Line (Line-trendname d))
				   (TEnv (Tenv-trendname d))
				   (TQual(TQual-trendname d))))
  (format t "~%Timestamp:~15T~s" (case (type-of d)
				   (Line (Line-timestamp d))
				   (TEnv (Tenv-timestamp d))
				   (TQual(TQual-timestamp d))))
  (format t "~%Documentation:~15T~s" (case (type-of d)
				       (Line (Line-documentation d))
				       (TEnv (Tenv-documentation d))
				       (TQual(TQual-documentation d))))
  (terpri))
|#

(defun show-qg (&optional id-or-qgraph)
  (cond
    (id-or-qgraph
     (show-qgraph (if (qgraph-p id-or-qgraph)
		      id-or-qgraph
		      (find id-or-qgraph *qgraphs* :key #'qgraph-id))))
     (T
      (loop for q in *qgraphs*
	    do
	    (show-qgraph q))
        (terpri)
	(format t "~%Current qgraph is ~a" (when *current-qgraph*
					     (qgraph-id *current-qgraph*))))))


(defun show-qgraph (q)
  (format t "~%Id:~15T~a" (qgraph-id q))
  (format t "~%Name:~15T~a" (qgraph-name q))
  (format t "~%Documentation:~15T~s" (qgraph-documentation q))
  (format t "~%Timestamp:~15T~s" (qgraph-timestamp q))
  (format t "~%Datasets:")
  (loop for d in (qgraph-data q)
	for c in (qgraph-colors q)
	for s in (qgraph-styles q)
	do
	(format t "~%~a~8T~a~25T~a~32T~a~38T~a~48T~a"
		(case (type-of d)
		  (Line (Line-Id d))
		  (TEnv (Tenv-Id d))
		  (TQual(TQual-Id d)))
		(case (type-of d)
		  (Line (Line-name d))
		  (TEnv (Tenv-name d))
		  (TQual(TQual-name d)))
		(type-of d)
		c
		s
		(case (type-of d)
		  (Line (Line-timestamp d))
		  (TEnv (Tenv-timestamp d))
		  (TQual(TQual-timestamp d)))))
  (terpri))


	
(defun qgraph-cleanup ()
  (loop for d in *datasets*
	do
	(typecase d
	  (Line     (setf (line-data d) nil))
	  (TEnv     (setf (tenv-data d) nil))
	  (TQual    (setf (tqual-data d) nil))
	  (TQ3      (setf (tq3-data d) nil))))
  (loop for qg in (cons *current-qgraph* *qgraphs*)
	do
	(setf (qgraph-window qg) nil)
	(setf (qgraph-data qg) nil)
	(setf (qgraph-colors qg) nil)
	(setf (qgraph-styles qg) nil))
  (setf *qgraphs* nil)
  (setf *datasets* nil)
  (setf *current-qgraph* nil))

	
