;;; -*- Mode:Common-Lisp; Package:Qsim; Syntax:Common-Lisp; Base:10 -*-
;;;  $Id: transitions.lisp,v 1.11 92/06/18 16:47:51 bert Exp $

;;;  Copyright (c) 1987, 1988 by Benjamin Kuipers.

(in-package :qsim)


;;; REGION TRANSITION filter.  Now handles multiple completions
;;;    => This is NOT called as a normal global filter.  It has a separate step
;;;	because the normal global-filter interface didn't like more states 
;;;	coming out than in.  
;;;
;;; The new state resulting from a transition is an alternate description
;;; of the same point in time.

#|
-----------------------------------------------------------------------------
		  A L L	   A B O U T	T R A N S I T I O N S

  General:	    Transitions are somewhat like production rules applied to
		    each state, i.e., if a given predicate on the state is
		    true, then perform the corresponding action.  The action
		    may transform the state or just perform some side-effect.

  Format:	    The qde.transitions slot contains a list of the form:
		      (transitions <transition-spec>+ )
		    where
		      <transition-spec> ::= (<test> [->] {<transition-function>|NIL})
		      [->]   ::= an optional arrow (->)
		      <test> ::= <test-function>  |
				 (<varname> (<landmark> {<qdir>|NIL}))  |
				 (NOT <test>)  |  (AND <test>+)  |  (OR <test>+)
		      {a|b}  ::= a  |  b
		    If <test> applied to state is true, that is,
		      (funcall <test-function> state) returns non-nil, or
		      variable <varname> has value (<landmark> {<qdir>|NIL}), or
		      the boolean <test> returns true,
		    then do
		      (funcall <transition-function> state), or
		      terminate the behavior, if NIL is specified.
		    <Transition-function> is described below.

  Definitions:   -- A "region transition" is a user-specified transition to a
		      new region (i.e., a new QDE); it halts simulation of the
		      triggering state, replacing it with a state in a new
		      QDE.
		   -- A "monitor transition" is performed for side-effect,
		      such as beeping an alarm or printing a message or
		      updating a visual display.  Unlike all other types of
		      transitions, it does not affect further simulation.
		   -- A "halt transition" is a directive to halt further
		      simulation from this state.  It is specified by
		      supplying a user-transition-function of NIL.
		   -- A "boundary transition" occurs when a variable falls off
		      the end of its qspace.  Like the halt transition, it
		      halts further simulation from this state.  This is the
		      only kind of transition that is not user-specified.

  Test Function:  The user-specified test function takes a single argument,
		     the state, and returns two truth values:
		     -- call the associated transition function?
		     -- continue simulating the current state?

		     Thus, [true, false]  = region transition
			   [true, true]	  = monitor transition
			   [false, false] = halt transition
			   [false, true]  = continue normal simulation

		     Actually, [true, true] could also be used for a region
		     transition that activates a new QDE but continues the
		     current one as well (might even be useful...).

		     To simplify the common test of a single variable compared
		     to some landmark value, the test "function" can be a
		     list of the form (<variable name> (<landmark> <qdir>)).
		     If the test is satisfied, [true, false] is returned as
		     for a region transition.

  Transition Function:
		     The user-specified transition function takes a single
		     argument, the current state, and returns either a single
		     state or a list of states or NIL (monitor transitions
		     MUST return NIL).  In all cases, the list of returned
		     states is added to the simulation agenda.  The function
		     create-transition-state makes the transition function
		     trivial to write, in most cases.  It can usually be
		     written as:
		       (lambda (state)
			 (create-transition-state
			   :from-state state
			   :to-qde <whatever>
			   :assert <whatever>
			   :inherit-qmag :rest
			   :inherit-qdir nil))

  Control: 	     How many transitions can be triggered at one time?  This
		     is controlled by the global variable *fire-transitions*
		     (settable from the QSIM control menu) which may have one
		     of these values:
		       :all    -- fire all triggered transitions
		       :first  -- fire only the first triggered transition
		       :unique -- more than one triggered transition is an
				  error
		       :none   -- treat region transitions as halt transitions
				  (useful during testing/debugging).
-----------------------------------------------------------------------------
|#

#| ---------------------------------------------------------------------------
   Function:  (FILTER-FOR-TRANSITIONS  state)
 
   Given:	A state which is to be checked for possible region transitions.
 
   Returns:	A list of states, possibly NIL, for further simulation:
		-- If no transition is triggered then the given state is
		   returned as the only member of the list.
		-- If only a halt/boundary transition is triggered, NIL is
		   returned.
		-- If any region transitions are triggered, their resulting
		   states are accumulated and returned.  If the current state
		   is to be continued and no halt/boundary transition is
		   triggered, then it is included in the returned list.
 
   Notes:	-- If the conditions for a region transition and a
		   boundary/halt transition are concurrently satisfied, the
		   region transition is performed and the boundary/halt
		   transition is ignored (unless the region transition allows
		   continuation of the causing state, in which case the
		   boundary/halt transition IS obeyed).
		-- Monitor transitions have no effect on the returned states.
   ---------------------------------------------------------------------------
|#

(defun FILTER-FOR-TRANSITIONS (state)
  (unless (inconsistent-p state)
    (let ((new-states nil))   ; new states to be returned
      (multiple-value-bind (transition-functions boundary? continue?)
	  (check-for-transitions state)
	
	;; If any kind of transition triggered, then make new landmarks for state
	;; and, because of the new landmarks, re-run the quantitative range filter
	;; on the state.  (Sorta' ugly, but it works).
	(when (and (or transition-functions boundary?)
		   (qpointp (state-time state)))
	  (create-new-landmarks-on-transition state)
	  (quantitative-range-reasoning state))
	(when (or transition-functions boundary? (not continue?))
	  (push 'final (state-status state)))
	
	;; Record all transitions in state status, if requested.
	(when (and *record-all-transitions* (or transition-functions boundary?))
	  (push (cons 'transitions (append transition-functions boundary?))
		(state-status state)))
	(setf new-states (gather-transition-states state transition-functions))
	;; If region transitions, update status/successors of current state.
	(when new-states
	  (pushnew 'transition (state-status state))
	  (if *trace-transitions*
	      (format *QSIM-Trace* "~%REGION TRANSITION(S) identified from ~a to ~a."
		      (state-name state) (mapcar #'state-name new-states))))
	
	(if new-states			   ; Did at least one region transition occur?
	    (if continue?			   ; Yes.  Continue simulation of current state?
		(if boundary?		   ;   Yes.  Did it have a boundary/halt transition?
		    new-states		   ;     Yes.  Return only the region transition results.
		    (cons state new-states)) ;     No.   Return the current AND new states.
		new-states)		   ;   No continuation of current state; return new states.
	    
	    (if continue?			   ; No region transitions (but maybe monitor transitions).
					; Continue current state?
		(if boundary?		   ;   Yes.  Did a boundary/halt transition occur?
		    (halt-transition state)  ;     Yes.  Label transition in current state, return NIL.
		    (list state))		   ;     No.   Return current state for continuation.
		(halt-transition state))))))) ;   No continuation of current state;
					;   treat like halt transition.
  
  
  
;;;-----------------------------------------------------------------------------
;;; GATHER-TRANSITION-STATES loops thru the transition functions and gathers any
;;; transition results.  It installs these states on the Successors list of State.
;;; A list of all transition results is returned - except that incomplete states are
;;; replaced by their transitions.  Tresult may be incomplete.  If it is, its
;;; successors are already generated.
;;;-----------------------------------------------------------------------------

(defun GATHER-TRANSITION-STATES (state transition-functions &aux (new-states))
  ;; Perform all region transitions and monitor transitions.
  (dolist (tran-fn transition-functions)
    ;; Call the transition function and filter the state(s) it returns.
    (let* ((tresult (funcall tran-fn state))
	   (ns (get-list-of-initial-states tresult)))
      (install-as-transition-results state ns)
      (setq ns (mapcan #'apply-global-state-filters (copy-list ns)))
      ;; See if new transition state(s) provoke further transitions.
      (setq ns (mapcan #'filter-for-transitions (copy-list ns)))
      ;; Update list of states for further simulation.
      (setq new-states (nconc new-states ns))))
  new-states)



(defun INSTALL-AS-TRANSITION-RESULTS (ostate nstates)
  (setf (state-successors ostate)
	(if (state-successors ostate)
	    (append (state-successors ostate) (copy-list nstates))
	    (cons 'transition-identity (copy-list nstates)))))

  

(defun HALT-TRANSITION (state)
  "Set status/successor state labels for boundary/halt transition."
  (unless (state-successors state)
    (setf (state-successors state) '(transition-identity)))	; NOT '(transition-identity nil)
  (pushnew 'transition (state-status state))
  (if *trace-transitions*
      (format *QSIM-Trace* "~%BOUNDARY/HALT TRANSITION identified at ~a."
	      (state-name state)))
  nil)



(defmacro TRANSITION-TEST-FN (tran)
  `(car ,tran))


(defmacro TRANSITION-ACTION-FN (tran)
  `(if (eql (cadr ,tran) '->)
       (caddr ,tran)
       (cadr ,tran)))



;;;-----------------------------------------------------------------------------
;;; Create new landmarks for all variables of the pre-transition state that
;;; are not currently equal to landmark values.  Then create corresponding
;;; values if necessary.  (This is called directly by the transition code.)
;;;-----------------------------------------------------------------------------

(defun CREATE-NEW-LANDMARKS-ON-TRANSITION (ostate)
  (when (and *enable-landmark-creation* *new-landmarks-on-transition*)
    (let ((changed nil))
      (mapc #'(lambda (qv-pair qs-pair)
		(let* ((qval (cdr qv-pair))
		       (var  (qval-variable qval)))
		  (when (and (interval-p (qmag qval))
			     (not (no-new-landmarks-p var)))   ; DJC 05/19/92
		    (create-new-lmark ostate qv-pair qs-pair "transition value")
		    (setq changed t))))
	    (cdr (state-qvalues ostate))	; skip over time qval
	    (cdr (state-qspaces ostate)))	; skip over time qspace
      (if changed
	  (make-new-corresponding-values ostate)))))


;;; ------------------------------------------------------------------------------
;;;   CHECK-FOR-TRANSITIONS
;;;   Determines whether a state gives rise to any transitions due to (1) explicit
;;;     transitions or (2) variables running off the ends of their Q-spaces.
;;;   Returns three values:
;;;   - transition-functions: a list of triggered region/monitor transition functions,
;;;   - boundary?: non-nil if a boundary transition or halt transition was triggered,
;;;   - continue?: non-nil if the current state's simulation should continue.
;;;   Modified (RSM 4 Mar 91) to --
;;;   - Treat transition action functions which are keywords as "labeled" halt
;;;     transitions, like nil or t.
;;;   - Push onto boundary? the keyword for all labeled halt transitions and T
;;;     for any other halt transitions.
;;;   - If requested by *record-all-transitions*, check for all boundary
;;;     transitions and for each push (var . qval) onto boundary?.
;;;   ------------------------------------------------------------------------------


(defun CHECK-FOR-TRANSITIONS (state)
  "Determine whether state gives rise to any transitions due to explicit
   transitions or to variables running off the ends of their Q-spaces."
  (let ((transition-functions nil)			; The three return values
	(boundary? nil)
	(continue? t))
    ;; Check for region transitions; collect all that are triggered.
    (dolist (tran (qde-transitions (state-qde state)))
      (let ((action-fn (transition-action-fn tran)))
	(multiple-value-bind (fire? xcontinue?)
	    (funcall (transition-test-fn tran) state)	; Call the transition-test-function.
	  (when fire?					; If a transition was triggered
	    (unless xcontinue? (setq continue? nil))	;   stop if any transition says so.
	    (cond ((member action-fn '(NIL T))		; Regular halt transition
		   (pushnew T boundary?))
		  ((not (fboundp action-fn))            ; Determines if the transition functionm exists  DJC 07/22/91
		   (warn  "~&~% WARNING: Transition function ~a not defined.  ~%          ~
                                         This function will be handled as if the transition ~%          ~
                                         function supplied was T.~%" action-fn)
		   (pushnew T boundary?))
		  ((keywordp action-fn)			; Labeled halt transition
		   (pushnew action-fn boundary?))
		  (t					; Region or monitor transition
		   (case *fire-transitions*
		     (:NONE	(pushnew T boundary?))
		     (:ALL	(push action-fn transition-functions))
		     (:FIRST	(push action-fn transition-functions)
				(return))
		     (:UNIQUE	(if (null transition-functions)
				    (push action-fn transition-functions)
				    (error "Attempt to trigger two transition functions ~
					    (~a and ~a) with transition control = :UNIQUE."
					   (car transition-functions) action-fn))))))))))
    ;; Check for boundary transitions.
    (if *record-all-transitions*
	;; Find all boundary transitions and add them to boundary?.
	(loop
	  for qvalue in (state-qvalues state)
	  for qspace in (state-qspaces state)
	  if (boundary-transition-p qvalue qspace)
	     do (push qvalue boundary?))
	;; Fast boundary transitions check.  Don't check if:
	;; - A halt transition has already been triggered -- halt and boundary transitions
	;;   are treated identically.
	;; - The current state is not to be continued -- due to a region transition, which
	;;   takes precedence over boundary transitions in function filter-for-transitions.
	;; Otherwise check and stop as soon as one is found.
	(when (not (or boundary? (not continue?)))
	  (setq boundary? (some #'boundary-transition-p
				(state-qvalues state)
				(state-qspaces state)))))

    ;; Return three values.
    (values  transition-functions  boundary?  continue?)))


;;;-----------------------------------------------------------------------------
;;;  BOUNDARY-TRANSITION-P returns non-nil if the given qval is falling off
;;;  the edge of its qspace.
;;;-----------------------------------------------------------------------------

(defun BOUNDARY-TRANSITION-P (qvalue qspace-pair)
  (let* ((qval	 (cdr qvalue))
	 (qmag	 (qmag qval))
	 (qdir	 (qdir qval))
	 (qspace (cdr qspace-pair)))

    (and (point-p qmag)
	 (or (and (eql qdir 'dec)
		  (eq  qmag (first qspace)))
	     (and (eql qdir 'inc)
		  (eq  qmag (car (last qspace))))))))



;;;-----------------------------------------------------------------------------
;;;  MAKE-TRANSITION-RESULT is defined for historical compatibility.
;;;  Its job is now performed by the more-general CREATE-TRANSITION-STATE.
;;;-----------------------------------------------------------------------------

(defun MAKE-TRANSITION-RESULT (from-state to-qde values)
  (create-transition-state :from-state	 from-state
			   :to-qde	 to-qde
			   :assert	 values
			   :inherit-qmag nil
			   :inherit-qdir nil))




;;;-----------------------------------------------------------------------------
;;;  Function:  CREATE-TRANSITION-STATE
;;;
;;;  Purpose:	This function is provided to simplify the user's job of writing
;;;		a transition function for a region transition.  This function is
;;;		called from inside a user-defined transition function.  Through
;;;		the keyword arguments, the user can easily declare what should
;;;		happen at the region transition.  For example:
;;;
;;;		(create-transition-state :from-state   ball-state
;;;					 :to-qde       spring-bounce
;;;					 :assert       '((y (0 inc)))
;;;					 :inherit-qmag :rest
;;;					 :inherit-qdir nil
;;;					 :text	       "ball bounced")
;;;
;;;		means that at the initial transition state of spring-bounce,
;;;		the value of Y should be (0 inc), the qmag of all other
;;;		variables should be inherited from ball-state, and none of
;;;		the qdirs should be inherited from ball-state (which means
;;;		they get a value of nil).  The general form is:
;;;
;;;		(create-transition-state
;;;		    :from-state	    <state>
;;;		    :to-qde	    <qde>
;;;		    :assert	    ((<varname> (<qmag> <qdir>)) ...)
;;;		    :inherit-qmag   (<varname> <varname> ...) | :all | :rest | test-fn
;;;		    :inherit-qdir   (<varname> <varname> ...) | :all | :rest | test-fn
;;;                 :assert-ranges  (((<varname> <lmarkname>) (lo hi)) ...)
;;;                 :inherit-ranges (<varname-or-varlist> ...) | :all | testfn
;;;                                 where
;;;                                 <varname-or-varlist> ::=
;;;                                        <varname> | (<varname>  <lmark-or-lmarkname> ...) |
;;;                                        (<varname> :all) | (<varname> testfn)
;;;		    :text	    <string>
;;;		  )
;;;
;;;		A test-fn in :inherit-qmag or :inherit-qdir
;;;             is a user-supplied function that takes a variable name as input
;;;             and returns true/false as to whether inheritance should occur.
;;;
;;;             For :inherit-ranges, :all and testfn select variables as in :inherit-qmag
;;;             and :inherit-qdir, all of whose landmark ranges will be inherited.
;;;             If a variable is listed explicitly, then particular landmarks can
;;;             be inherited without inheriting other landmarks.  If no landmarks are listed
;;;             or if :all is selected then all the lmarks for the variable will be inherited.
;;;             A testfn in <varname-or-varlist> is a user-supplied function that takes a
;;;             variable name and a lmark name and returns true/false as to whether
;;;             inheritance should occur (This paragraph and related code added BKay 16Jun92).
;;;
;;;  Returns:	The initial state of the to-qde; if the state is incomplete,
;;;		it returns the state pointing to the completions.
;;;
;;;  Design:	1.  Make an initial state of the to-qde having the same time
;;;		    as from-state.
;;;		2.  For each variable shared between the from-state and the
;;;		    new state, union their qspaces.
;;;		3.  For identical constraints in the from-state and new state,
;;;		    union their corresponding values.
;;;		4.  Set the qvals of the new state in accordance with :assert,
;;;		    :inherit-qmag, and :inherit-qdir.
;;;             5.  Set the values for Q2 rannges in accordance with
;;;                 :assert-ranges and :inherit-ranges.
;;;		6.  Complete the state and return it.
;;;
;;;		This function sets only properties of the new state(s), not
;;;		of the predecessor.
;;;-----------------------------------------------------------------------------


(defun CREATE-TRANSITION-STATE
       (&key from-state to-qde assert inherit-qmag inherit-qdir
	     inherit-ranges assert-ranges text)
;  (let ((qdes-in-tree (display-block-qdes-in-beh-tree *current-display-block*)))
;    (when (null (member to-qde qdes-in-tree))                        ;;  modified DJC 05/14/91 to reference 
;      (setf (display-block-qdes-in-beh-tree *current-display-block*) ;;  the display block
;	    (cons to-qde qdes-in-tree))))
  ;;the code above is equivalent to the line below
  ;;PF 03 June 1991
  (pushnew to-qde (display-block-qdes-in-beh-tree *current-display-block*))
  ;; STEP 1:  Make a new/initial state of the to-qde, having same state-time.
  (let ((nstate (make-state :qde       to-qde
			    :name      (genname 'S)
			    :text      (format nil "~a: Transition from ~a to ~a"
					       text
					       (qde-name (state-qde from-state))
					       (qde-name to-qde))
			    :justification `(transition-from ,from-state ,to-qde)
			    ;; This next line changed by BKay 3Sept91
			    ;; state.eqn-index and state.assert-ranges
			    ;; should not be carried into the next state.
;			    :other     (copy-tree (state-other from-state))
			    :other     (copy-state-other
					(remove-if #'(lambda (slot)
						       (member (car slot)
							       '(bindings
								 eqn-index
								 assert-ranges)))
						   (state-other from-state)))
			    )))
    (setf (state-predecessors nstate) (list from-state))
    (set (state-name nstate) nstate)

    ;; STEP 2:  Union qspaces of from-state and to-qde for the new state.
    (union-qspaces-at-transition from-state nstate to-qde)

    ;; STEP 3:  Union corresponding values of identical constraints in the
    ;;		from-state and to-qde.
    (union-cvalues-at-transition from-state nstate to-qde)

    ;; STEP 4:  Convert user values now that qspaces have been union'ed.
    (setf (state-qvalues nstate)
	  (get-transition-qvalues from-state nstate assert inherit-qmag inherit-qdir))

    ;; STEP 5:  Add the appropriate Q2 info.  This info will be passed via the
    ;;          state.assert-ranges slot.  Added BKay 15Jun92
    (set-transition-ranges nstate from-state inherit-ranges assert-ranges)
	  
    ;; STEP 6:  Complete the state.
    (cond ((complete-states-from-partial-state nstate))
	  (*prune-uncompletable-transitions*
	   (prune-inconsistent-state from-state "No completions upon transition")) 
	  (t (error "Inconsistent transition result ~a had no completions." nstate)))

    ;; STEP 7:  Return new state.
    (if (member 'Inconsistent (state-status nstate))
	(values nil nstate)			; This VALUES clause is just a hook for
	nstate)))				; tracing the outcome of failed transitions.




;;; Setup the Q2 range info for the first state after the transition.
;;; Inputs:  nstate         - The new state.
;;;          from-state     - The old state.
;;;          inherit-ranges - :all | testfn | (<varname-or-varlist> ...)
;;;                           where <varname-or-varlist> looks like
;;;                           <varname> | (<varname> <lmark> ...) |
;;;                           (<varname> :all) | (<varname> testfn)
;;;          assert-ranges  - A list of ((<varname> <lmname>) (lo hi)) entries
;;; Returns: none, but sets the state.assert-ranges and state.bindings slot for later calls
;;;          by Q2.
;;; Notes:   Q2 is generous about the form of vars and lmarks on the
;;;          state.assert-ranges slot.  Vars can be either varnames or variable
;;;          structs and the same goes for lmarks.
;;; This function added by BKay 16June92.
;;;
(defun set-transition-ranges (nstate from-state inherit-ranges assert-ranges)
  (cond
    ((null (sim-q2-constraints (state-sim nstate))))
    ((member inherit-ranges '(:all :rest))
     ;; This is a shortcut when everything is to be inherited.
     (setf (state-assert-ranges nstate) nil)
     (setf (state-bindings nstate) (copy-tree (state-bindings from-state))))
    (T
     (setf (state-bindings nstate) nil)
     ;; First inherit the ranges that the user asked for.
     (setf (state-assert-ranges nstate)
	   (loop for vlr in (if (functionp inherit-ranges)
				(loop for qspace in (state-qspaces from-state)
				      for var in (first qspace)
				      when (funcall inherit-ranges var)
				      collect var)
				inherit-ranges)
		 nconc
		 (loop with varname = (if (atom vlr) vlr (first vlr))
		       for  lmark in (cond
				       ((or (atom vlr) (eq (second vlr) :all))
					(qspace varname from-state))
				       ((functionp (second vlr))
					(loop for lmark in (qspace varname from-state)
					      when (funcall (second vlr) varname (lmark-name
										  lmark))
					      collect lmark))
				       (T
					(cdr vlr)))
		       for int = (get-q2-interval from-state varname lmark)
		       when int
		       collect (list (list varname (lmark-name lmark)) int))))
     
     ;; Now add the asserted ranges.
     (loop for ((var lmark) range) in assert-ranges
	   do
	   (push (list (list var (lmark-name lmark)) range) (state-assert-ranges nstate))))))


;;; Given a state, a varname or variable struct, and a lmark name or lmark
;;; struct, return the Q2 range for the variable-lmark pair.
;;;
(defun get-q2-interval (from-state varname lmark)
  (let* ((var (if (variable-p varname)
    		  varname
		  (alookup varname (qde-var-alist (state-qde from-state)))))
	 (lm  (if (lmark-p lmark)
		  lmark
		  (lmark-find lmark (lookup-set (variable-name var)
						(state-qspaces from-state)))))
	 (var-entry (assoc var (state-bindings from-state)))
	 (vl-entry (assoc lm (cdr var-entry) :test #'equal)))
    (when vl-entry
      (second vl-entry))))
   



;;; Union qspaces of from-state and to-qde for the new state.

(defun UNION-QSPACES-AT-TRANSITION (from-state nstate to-qde)
  (if (eq to-qde (state-qde from-state))
      (setf (state-qspaces nstate) (copy-alist (state-qspaces from-state)))
      (setf (state-qspaces nstate)
	    (mapcar #'(lambda (qspace-pair)
			(let ((varname (car qspace-pair)))
			  (cons varname
				(qspace-union (cdr qspace-pair)
					      (previous-qspace varname from-state)))))
		    (qde-qspaces to-qde)))))



;;; Union corresponding values of identical constraints in the
;;; from-state and to-qde.

(defun UNION-CVALUES-AT-TRANSITION (from-state nstate to-qde)
  (if (eq to-qde (state-qde from-state))
      (setf (state-cvalues nstate) (copy-alist (state-cvalues from-state)))
      (setf (state-cvalues nstate)
	    (mapcar #'(lambda (con-w-cvs)
			(let* ((con   (car con-w-cvs))
			       (cvs   (cdr con-w-cvs))
			       (cname (constraint-name con)))
			  (cons con
				(cval-union cvs (previous-cvs cname from-state)
					    cname (state-qspaces nstate)))))
		    (qde-cvalues to-qde)))))


;;;-----------------------------------------------------------------------------
;;;  GET-TRANSITION-VALUES creates a qvalues list for the new state.  For each
;;;  variable the value is taken from the assert list (if present), otherwise
;;;  inherited from the from-state (if specified in the inherit list), else
;;;  the value is nil.
;;;-----------------------------------------------------------------------------

(defun GET-TRANSITION-QVALUES (from-state nstate assert inherit-qmag inherit-qdir)
  (declare (special from-state nstate assert inherit-qmag inherit-qdir))
  (mapcar
    #'(lambda (var)
	(cons (variable-name var) (get-transition-qval var)))
    (qde-variables (state-qde nstate))))


(defun GET-TRANSITION-QVAL (var)
  (let ((varname (variable-name var))
	asserted-value)
    (declare (special from-state nstate assert inherit-qmag inherit-qdir))

    (cond
      ;; If user asserted a value, convert it to a qval.
      ((setq asserted-value (assoc varname assert))
       (convert-user-qval asserted-value var (qspace varname nstate)))

      ;; Otherwise, look for inherited qmag and/or qdir.
      (t (make-qval :variable var
		    :qmag     (inherit-or-default-qmag var inherit-qmag from-state nstate)
		    :qdir     (inherit-or-default-qdir var inherit-qdir from-state))))))



(defun INHERIT-OR-DEFAULT-QMAG (var inherit-qmag from-state new-state)
  (let ((varname (variable-name var)))

    ;; If qmag is to be inherited, then get the qmag from
    ;; the from-state and find its equivalent in nstate.
    ;; Independent and discrete variables always inherit old value
    ;; (unless asserted).
    (when (or (variable-independent-p var)
	      (variable-discrete-p var)
	      (eql 'time varname)
	      (inherit? varname inherit-qmag))
      (let ((qspace (qspace varname new-state))
	    (oqmag  (qmag (qval varname from-state)))
	    uqmag)
	(setq uqmag (if (qmag-point-p oqmag)
			(lmark-name oqmag)
			(list (lmark-name (first  oqmag))
			      (lmark-name (second oqmag)))))
	(convert-user-qmag uqmag qspace)))))


(defun INHERIT-OR-DEFAULT-QDIR (var inherit-qdir from-state)
  (let* ((varname (variable-name var))
	 (qvalue (qval varname from-state)))
    (cond
      ;; If ignore qdir for this variable, qdir = ign.
      ((and qvalue (eql (qdir qvalue) 'ign)) 'ign)
      ;; Changed from (variable-ignore-qdir-p var).  BJK (10-8-90)

      ;; If variable is independent or discrete, qdir = std.
      ((or (variable-independent-p var)
	   (variable-discrete-p var))	'std)

      ;; Look for inherited qdir.  Time is to be inherited too.
      ((or (inherit? varname inherit-qdir)
	   (eql 'time varname))
       (qdir qvalue))

      ;; Otherwise, qdir = nil.
      (t nil))))



;;; INHERIT? returns true if inheritance is to occur for the named variable.

(defun INHERIT? (varname inherit-spec)
  (cond ((null inherit-spec)	  nil)
	((listp inherit-spec)	  (member varname inherit-spec))
	((member inherit-spec '(:all :rest)) t)
	((functionp inherit-spec) (funcall inherit-spec varname))
	(t (error "Invalid inheritance specification: ~a" inherit-spec))))


(defun PREVIOUS-QSPACE (varname ostate)
  (cond ((qspace varname ostate))
	((predecessor-of-state ostate)
	 (previous-qspace varname (predecessor-of-state ostate)))
	(t nil)))


(defun PREVIOUS-CVS (cname ostate)
  (cond
    ;; If identical constraint in ostate, then return its cvals.
    ((cdr (assoc cname (state-cvalues ostate)
		 :key #'constraint-name
		 :test #'equal)))
    ;; If ostate has a predecessor, go look for cvals there.
    ((predecessor-of-state ostate)
     (previous-cvs cname (predecessor-of-state ostate)))
    ;; Otherwise, return nil.
    (t nil)))


;;;-----------------------------------------------------------------------------
;;; Make an alist of values taken from the old state, changing the specified values,
;;; leaving the other independent variables alone, and preserving the values, but
;;; not directions of change, of the remaining variables.  For example,
;;;	  (switch-values heater-state '((HFin (0 std))
;;;					(netHF ((minf 0) nil))))
;;;-----------------------------------------------------------------------------

(defun SWITCH-VALUES (old-state changed)
  (declare (ignore old-state changed))
  (error "Use create-transition-state now!"))


