;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-


;;;*****************************************************************************
;;;	 		Q S I M   U S E R   I N T E R F A C E
;;;
;;;		       by Pierre Fouche (fouche@cs.utexas.edu)
;;;
;;;			    First version: February 1990
;;;
;;;
;;;
;;; This is a small user interface for the TI Explorers machines
;;;
;;; Basically it provides a constraint frame, composed of a lisp listener and a 
;;; permanent menu pane containing top level commands, usually available by (q).
;;; The frame is selected by System-q.
;;;*****************************************************************************

(shadowing-import '(ticl:defflavor ticl:defmethod ticl:make-instance
		    ticl:defcommand ticl:build-menu ticl:build-command-table
		    ticl:send))

;;;=============================================================================
;;; 		    Some global variables to begin with
;;;=============================================================================

(defvar *qsim-frame* nil
  "Qsim constraint frame")

(defvar *qsim-menu*  nil
  "Qsim menu pane")

(defvar *qsim-io*  nil
  "Qsim input-output pane")

(defvar *qsim-top-level-commands* nil
  "Qsim top level command table")

(defparameter *qsim-menu-commands*
	      '(control-menu ACC-menu NIC-menu EC-menu Q2-menu PS-menu
			     options catalog QDE simulate)
  "List of Qsim top level commands")

(defparameter *current-catalog* nil
  "Current catalog")

(defparameter *current-structure* nil
  "Current structure")

(defparameter *current-initialization* nil
  "Current initialization")


;;;=============================================================================
;;; Flavors defining the main frame
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; The QSIM listener is simply a ucl command loop
;;; The command menu pane is dynamic and not scrollable
;;;-----------------------------------------------------------------------------

(defflavor qsim-listener () (ucl:command-and-lisp-typein-window))

(defflavor qsim-menu-pane nil (w:menu)
  (:default-init-plist
    :item-list nil
    :scrolling-p nil
    :command-menu t
    :dynamic t))

;;;-----------------------------------------------------------------------------
;;; The Qsim frame flavor links the menu pane and the qsim listener together
;;; It also defines the active command table and the shared input buffer.
;;;-----------------------------------------------------------------------------

(defflavor qsim-frame ()
	   (;; basicaly it includes a command loop and a constraint frame
	    ;; all the panes share the same io buffer
	    ucl:command-loop-mixin
	    tv:bordered-constraint-frame-with-shared-io-buffer
	    ;; must be a window also, otherwise :menu-panes does not work
	    tv:window
	    ;; allow notification to come out
	    tv:notification-mixin
	    ;; disable the selection of inferior windows
	    tv:inferiors-not-in-select-menu-mixin
	    ;; a mouse click produces a complete blip
	    tv:list-mouse-buttons-mixin)
  
  (:default-init-plist
    :active-command-tables '(*qsim-top-level-commands*)
    :all-command-tables '(*qsim-top-level-commands*)
    :typein-handler :handle-typein-input
    :constraints
    '((main (qsim-menu qsim-io)
	    ((qsim-menu 1 :lines))
	    ((qsim-io :even))))
    :panes
    '((qsim-io qsim-listener)
      (qsim-menu qsim-menu-pane))
    :menu-panes '((qsim-menu *permanent-qsim-menu*))
    :prompt "qsim> "
    ))

(defun create-and-select-qsim-frame ()
  (let ((qsim-frame (make-instance 'qsim-frame)))
    (in-package :qsim) ;changed DJC
    (send (send qsim-frame :get-pane 'qsim-io) :select)
    (setf *qplot-output* (send qsim-frame :get-pane 'qsim-io))
    (cover-page)
    (format *qplot-output* "~2%~a" (send qsim-frame :prompt))))

(w:add-system-key #\q 'qsim-listener
		  "qsim frame" '(create-and-select-qsim-frame))


;;;-----------------------------------------------------------------------------
;;; Redefines the method :designate-io-streams so that the io buffer is the 
;;; qsim listener and not the entire frame.
;;;-----------------------------------------------------------------------------

(defmethod (qsim-frame :designate-io-streams) ()
  (setq *terminal-io* (send self :get-pane 'qsim-io)))


;;;=============================================================================
;;; 				Command definitions
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; get-menu-pane returns the menu pane from the selected window, which is
;;; currently the qsim listener.
;;; It is useful to position pop-up menus
;;;-----------------------------------------------------------------------------

(defun get-menu-pane ()
  (send ucl:this-application :get-pane 'qsim-menu))

(defcommand control-menu()
  '(:names
     ("Main")
     :keys (#\meta-shift-c)
     :documentation "QSIM Control Variables")
  (w:choose-variable-values
    qsim:*main-control-choices*
    :label "QSIM options and flags"
    :height 650
    :near-mode `(:window ,(get-menu-pane))))

(defcommand ACC-menu()
  '(:names
     ("ACC")
     :keys (#\meta-shift-a)
     :documentation "ACC controls")
  (w:choose-variable-values
    qsim:*new-acc-control-variables*
    :label "New ACC controls"
    :near-mode `(:window ,(get-menu-pane))))

(defcommand NIC-menu()
  '(:names
     ("NIC")
     :keys (#\meta-shift-n)
     :documentation "NIC controls")
  (w:choose-variable-values
    qsim:*nic-control-variables*
    :label "Non-Intersection Constraint Controls"
    :near-mode `(:window ,(get-menu-pane))))

(defcommand EC-menu()
  '(:names
     ("EC")
     :documentation "Energy Constraint Controls")
  (w:choose-variable-values
    qsim:*ec-control-variables*
    :label "Energy Constraint Controls"
    :near-mode `(:window ,(get-menu-pane)))
  (qsim:untrace-predicate)
  (when qsim:*traced-predicates*
    (eval `(qsim:trace-predicate ,@ qsim:*traced-predicates*))))

(defcommand Q2-menu()
  '(:names
     ("Q2")
     :keys (#\meta-shift-q)
     :documentation "Q2 quantitative inference controls")
  (w:choose-variable-values
    qsim:*q2-control-variables*
    :label "Q2 trace and control switches"
    :near-mode `(:window ,(get-menu-pane))))

(defcommand PS-menu()
  '(:names
     ("PS")
     :keys (#\meta-shift-p)
     :documentation "PS controls")
   (w:choose-variable-values
     pos:*ps-control-variables*
     :label "Put QSIM plots in a PostScript File"
     :near-mode `(:window ,(get-menu-pane))))


(defparameter qsim:*option-variables*
  '("Redefine the variable *option-variables* to customize this menu")
  "This menu entry allows the user to build its private control menu")

(defcommand options ()
  '(:names
     ("Options")
     :keys (#\meta-shift-o)
     :documentation "Control Options")
   (w:choose-variable-values
     qsim:*option-variables*
     :label "Set private/additional control options"
     :height 650
     :near-mode `(:window ,(get-menu-pane))))


;;;-----------------------------------------------------------------------------
;;; The next commands manage catalogs, QDEs and initializations.
;;; Current catalog and current QDE are defined so that the user does not have
;;; to select them every time.
;;;-----------------------------------------------------------------------------

(defun set-current-catalog ()
  (let ((choice (w:menu-choose
		  qsim:*available-catalogs*
		  :label "Set current catalog"
		  :near-mode `(:window ,(get-menu-pane)))))
    (when choice (setq *current-catalog* choice)
	  (set-current-structure))))

(defcommand catalog()
  '(:names
     ("Catalogs")
     :keys (#\meta-shift-d)
     :documentation "Set the current catalog")
  (set-current-catalog))


;;;-----------------------------------------------------------------------------
;;; The code of set-current-structure and simul-current-structure is based on
;;; the function "q"
;;;-----------------------------------------------------------------------------

(defun set-current-structure ()
  (let* ((catalog (eval *current-catalog*))
	 (choice (w:menu-choose
		   (mapcar #'car catalog)
		   :label "Set current structure"
		   :near-mode `(:window ,(get-menu-pane))))
	 (retrieved-record nil))
    (when choice
      (setq *current-structure* choice
	    retrieved-record (assoc choice catalog))
      (mapc 'qsim:load-file-unless-already-loaded
            (cadr retrieved-record))            ; files to load
      (eval (caddr retrieved-record))           ; structure initialization form
      (setq *current-initialization* (cadddr retrieved-record))
      (simul-current-structure))))

(defcommand QDE()
  '(:names
     ("Structures")
     :keys (#\meta-shift-s)
     :documentation "Set the current QDE")
  (set-current-structure))

(defun simul-current-structure ()
  (let* ((choice (w:menu-choose
		   (mapcar #'car *current-initialization*)
		   :label " Select initialization: "
		   :near-mode `(:window ,(get-menu-pane))))
	 (retrieved-record nil))
    (when choice
      (setq retrieved-record (assoc choice *current-initialization*))
      (qsim:load-file-unless-already-loaded (cadr retrieved-record))
;      (send (send ucl:this-application :get-pane 'qsim-io) :select)
      (eval (caddr retrieved-record))
      (format t "~2%~a" (send ucl:this-application :prompt)))))

(defcommand simulate()
  '(:names
     ("Simul")
     :keys (#\meta-shift-i)
     :documentation "Simulate the current QDE")
  (simul-current-structure))


;;;-----------------------------------------------------------------------------
;;; Build the qsim frame command table
;;;-----------------------------------------------------------------------------

(build-command-table
  '*qsim-top-level-commands* 'qsim-frame
  *qsim-menu-commands*)


;;;-----------------------------------------------------------------------------
;;; Build the top level command menu from the *qsim-menu-commands* list
;;;-----------------------------------------------------------------------------

(build-menu '*permanent-qsim-menu* 'qsim-frame     
  :item-list-order *qsim-menu-commands*)

(format t "~2%*** Type <system>-q to use the QSIM User Interface ***")

