;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-
;;; $Id: rk45s.lisp,v 1.6 92/04/02 18:04:20 bert Exp $

(in-package 'qsim)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Fourth-fifth order Runge-Kutta Alogrithm.
;;;
;;; See Forsythe, "Computer Methods for Mathematical Computations"
;;; pp. 129-147 for details.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This follows Forsythe pretty closely.  The major change is that a structure
;;; called 'rk' is defined to hold all the stored data that is passed by
;;; reference in the fortran version.
;;;
;;; Another change is that there is another argument "c" which is a vector
;;; of constants that is passed to the derivative function.
;;;
;;;
;;; HOW TO USE THE SIMULATOR :
;;;
;;; First, define a function f(t, y, y', c) where t is a number and y and y'
;;; are vectors of length n and c is a vector of some length.
;;; F should evaluate y' given t, y, and a vector of constants c and place the
;;; result in the vector y'.
;;;
;;; Now, call rk45-init to set up the rk struct.
;;;
;;; Finally, rk45 can be called repeatedly, with new time values to integrate
;;; the system.
;;;
;;; At the end of this file is a small example of how to use the simulator.
;;; Note also that the function rk45-run can be used rather than rk45.  It provides
;;; automatic error handling so that calling programs don't have to worry about it.
;;;
;;;
;;; IFLAG ERROR CODES :
;;;
;;; Iflag returns the status of a call to rk45 as an integer.  Possible values are :
;;;  
;;;   2 : Integration reached tout.  Simulation step succeeded.
;;;  -2 : A single step in the direction of tout has been taken (see Forsythe). 
;;;   3 : Integration not completed because relerr was too small.  Relerr has been
;;;       automatically increased for continuation.
;;;   4 : Integration not completed because more than 3000 derivative steps were needed.
;;;       This normally means that the system is too stiff to simulate with this Runge-Kutta
;;;       implementation.  You can try to continue, but your progress towards tout will be
;;;       VERY slow.
;;;   5 : Integration was not completed because solution vanished making a pure relative error
;;;       test impossible.  A non-zero abserr is needed to continue.
;;;   6 : Integration was not completed because requested accuracy was unattainable with 
;;;       smallest possible stepsize.  You must increase the error tolerance to continue.
;;;   7 : The algorithm is inefficient for this problem.  Consider using the one-step
;;;       integration mode (see Forsythe).
;;;   8 : Invalid input parameters.

(defstruct rk
  (yp     nil)
  (h      0)
  (f1     nil)
  (f2     nil)
  (f3     nil)
  (f4     nil)
  (f5     nil)
  (savre  0)
  (savae  0)
  (nfe    0)
  (kop    0)
  (init   0)
  (jflag  0)
  (kflag  0)
  (eps    0.0)
  (u26    0.0)

  (relerr 0.0)
  (abserr 0.0)
  (iflag  0)
  (time   0.0)
  (f      nil)
  (neqn   0)

  (cArray nil)    ; An array of constants (this isn't in the Forsythe implementation)
  )


;;;------------------------------------------------------------------------------------
;;; Initialize the rk struct for a simulation.
;;; Inputs:  f      - The function that evals y'.
;;;          neqn   - The length of the y vector (i.e. the order of the system).
;;;          relerr - Desired relative error.
;;;          abserr - Desired absolute error.
;;;          time   - The starting simulation time.
;;;          iflag  - The status flag.
;;;          c      - An array of constants to be used by f (may be nil if unused).
;;; Returns: An rk struct.
;;;------------------------------------------------------------------------------------
;;;
(defun rk45-init (f neqn relerr abserr time iflag &optional (c nil))
  (let (eps u26)
    (setq eps (do* 
	       ((eps   0.5         (/ eps 2.0))
		(epsp1 (+ eps 1.0) (+ eps 1.0)))
	       ((<= epsp1 1.0) eps)))
    (setq u26 (* 26.0 eps))

    (make-rk :yp     (make-array (list neqn) :initial-element 0.0)
	     :h      0.0
	     :f1     (make-array (list neqn) :initial-element 0.0)
	     :f2     (make-array (list neqn) :initial-element 0.0)
	     :f3     (make-array (list neqn) :initial-element 0.0)
	     :f4     (make-array (list neqn) :initial-element 0.0)
	     :f5     (make-array (list neqn) :initial-element 0.0)
	     :savre  0.0
	     :savae  0.0
	     :nfe    0
	     :kop    0
	     :init   0
	     :jflag  0
	     :kflag  0
	     :eps    eps
	     :u26    u26
	     :relerr relerr
	     :abserr abserr
	     :iflag  iflag
	     :time   time
	     :f      f
	     :neqn   neqn
	     :cArray c
	  )))


;;;------------------------------------------------------------------------------------
;;; This writes the simulation status into a rk struct.  The fortran version uses
;;; a set of global vars to do this.  With this method, multiple simulations could
;;; be run simulataneously.
;;;------------------------------------------------------------------------------------
;;;
(defun write-vals (yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		   relerr iflag time w)
  (setf (rk-yp w)    yp)
  (setf (rk-h w)     h)
  (setf (rk-f1 w)    f1)
  (setf (rk-f2 w)    f2)
  (setf (rk-f3 w)    f3)
  (setf (rk-f4 w)    f4)
  (setf (rk-f5 w)    f5)
  (setf (rk-savre w) savre)
  (setf (rk-savae w) savae)
  (setf (rk-nfe w)   nfe)
  (setf (rk-kop w)   kop)
  (setf (rk-init w)  init)
  (setf (rk-jflag w) jflag)
  (setf (rk-kflag w) kflag)
  (setf (rk-relerr w) relerr)
  (setf (rk-iflag w) iflag)
  (setf (rk-time w) time))


;;;------------------------------------------------------------------------------------
;;; Do a simulation step.
;;; Inputs:  y     - The current value of the state vector.
;;;          tout  - The desired time at which y is desired.
;;;          w     - The rk struct for the simulation.
;;;          iflag - The iflag (normally this is unused).
;;; Returns: #1 - iflag, which contains the status of the simulation attempt.
;;;          #2 - time, the time for which the new y is valid (normally, this is tout).
;;; Notes: y is updated as well.  This is really the "output" of the function.
;;;
;;;        This code is a direct translation of the fortran routine, so don't expect
;;;        to understand it without first reading the Forsythe book.
;;;------------------------------------------------------------------------------------
;;;
(defun rk45 (y tout w &optional iflag)
  (when iflag
    (setf (rk-iflag w) iflag))
  (let ((remin   1.e-12)
	(maxnfe  3000)
	mflag rer dt toln tol ypk output scale hfaild hmin eeoet et ee esttol
	ae s a
	(yp      (rk-yp w))
	(h       (rk-h w))
	(f1      (rk-f1 w))
	(f2      (rk-f2 w))
	(f3      (rk-f3 w))
	(f4      (rk-f4 w))
	(f5      (rk-f5 w))
	(savre   (rk-savre w))
	(savae   (rk-savae w))
	(nfe     (rk-nfe w))
	(kop     (rk-kop w))
	(init    (rk-init w))
	(jflag   (rk-jflag w))
	(kflag   (rk-kflag w))
	(eps     (rk-eps w))
	(u26     (rk-u26 w))
	(relerr  (rk-relerr w))
	(abserr  (rk-abserr w))
	(iflag   (rk-iflag w))
	(time    (rk-time w))
	(f       (rk-f w))
	(neqn    (rk-neqn w))
	(cArray  (rk-cArray w)))

    (prog ()
     (if (< neqn 1) (go L10))
     (if (or (< relerr 0.0) (< abserr 0.0)) (go L10))
     (setq mflag (abs iflag))
     (if (not (= mflag 1)) (go L20))
     (go L50)
  L10
     (setf iflag 8)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L20
     (if (and (= time tout) (not (= kflag 3))) (go L10))
     (if (not (= mflag 2)) (go L25))

     (if (or (= kflag 3) (= init 0)) (go L45))
     (if (= kflag 4) (go L40))
     (if (and (= kflag 5) (= abserr 0.0)) (go L30))
     (if (and (= kflag 6) (<= relerr savre) (<= abserr savae)) (go L30))
     (go L50)
  L25
     (if (= iflag 3) (go L45))
     (if (= iflag 4) (go L40))
     (if (and (= iflag 5) (> abserr 0.0)) (go L45))
  L30
     (error "RK45: Improper arguments")
  L40
     (setq nfe 0)
     (if (= mflag 2) (go L50))
  L45
     (setq iflag jflag)
     (if (= kflag 3) (setq mflag (abs iflag)))
  L50
     (setq jflag iflag)
     (setq kflag 0)
     (setq savre relerr)
     (setq savae abserr)
     (setq rer (+ (* 2.0 eps) remin))
     (if (>= relerr rer) (go L55))
     (setq relerr rer)
     (setq iflag 3)
     (setq kflag 3)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L55
     (setq dt (- tout time))
     (if (= mflag 1) (go L60))
     (if (= init 0) (go L65))
     (go L80)
  L60
     (setq init 0)
     (setq kop 0)
     (setq a time)
     (funcall f a y yp cArray)
     (setq nfe 1)
     (if (not (= time tout)) (go L65))
     (setq iflag 2)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L65
     (setq init 1)
     (setq h (abs dt))
     (setq toln 0.0)
     (dotimes (k neqn)
       (setq tol (+ (* relerr (abs (svref y k))) abserr))
       (when (> tol 0.0)
         (setq toln tol)
	 (setq ypk (abs (svref yp k)))
	 (if (> (* ypk (expt h 5)) tol) (setq h (expt (/ tol ypk) 0.2)))))
     (if (<= toln 0.0) (setq h 0.0))
     (setq h (max h (* u26 (max (abs time) (abs dt)))))
     (setq jflag (rksign 2 iflag))
  L80
     (setq h (rksign h dt))
     (if (>= (abs h) (* 2.0 (abs dt))) (incf kop))
     (if (not (= kop 100)) (go L85))
     (setq kop 0)
     (setq iflag 7)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L85
     (if (> (abs dt) (* u26 (abs time))) (go L95))
     (dotimes (k neqn)
       (setf (svref y k) (+ (svref y k) (* dt (svref yp k)))))
     (setq a tout)
     (funcall f a y yp cArray)
     (incf nfe)
     (go L300)
  L95
     (setq output NIL)
     (setq scale (/ 2.0 relerr))
     (setq ae (* scale abserr))
  L100
     (setq hfaild NIL)
     ;; This prevents hmin=0
     (setq hmin (* u26 (max (abs dt) (abs time))))
     (setq dt (- tout time))
     (if (>= (abs dt) (* 2.0 (abs h))) (go L200))
     (if (>  (abs dt) (abs h))         (go L150))
     (setq output T)
     (setq h dt)
     (go L200)
  L150
     (setq h (* 0.5 dt))
  L200
     (if (<= nfe maxnfe) (go L220))
     (setq iflag 4)
     (setq kflag 4)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L220
     (fehl f neqn y time h yp f1 f2 f3 f4 f5 f1 cArray)
     (setq nfe (+ nfe 5))
     (setq eeoet 0.0)
     (dotimes (k neqn)
       (setq et (+ (abs (svref y k)) (abs (svref f1 k)) ae))
       (when (<= et 0.0)
	 (setq iflag 5)
	 (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		     relerr iflag time w)
	 (return (values iflag time)))
       (setq ee 
	     (abs (+ (+ (* -2090.0 (svref yp k)) 
			(- (* 21970.0 (svref f3 k)) (* 15048.0 (svref f4 k))))
		     (- (* 22528.0 (svref f2 k)) (* 27360.0 (svref f5 k))))))
       (setq eeoet (max eeoet (/ ee et))))
     (setq esttol (* (abs h) eeoet (/ scale 752400.0)))
     (if (<= esttol 1.0) (go L260))
     (setq hfaild T)
     (setq output NIL)
     (setq s 0.1)
     (if (< esttol 59049.0) (setq s (/ 0.9 (expt esttol 0.2))))
     (setq h (* s h))
     (if (> (abs h) hmin) (go L200))
     (setq iflag 6)
     (setq kflag 6)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L260
     (setq time (+ time h))
     (dotimes (k neqn)
       (setf (svref y k) (svref f1 k)))
     (setq a time)
     (funcall f a y yp cArray)
     (incf nfe)
     (setq s 5.0)
     (if (> esttol 1.889568e-4) (setq s (/ 0.9 (expt esttol 0.2))))
     (if hfaild (setq s (min s 1.0)))
     (setq h (rksign (max (* s (abs h)) hmin) h))
     
     (if output (go L300))
     (if (> iflag 0) (go L100))
     (setq iflag -2)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
  L300
     (setq time tout)
     (setq iflag 2)
     (write-vals yp h f1 f2 f3 f4 f5 savre savae nfe kop init jflag kflag 
		 relerr iflag time w)
     (return (values iflag time))
     )))

(defun fehl (f neqn y time h yp f1 f2 f3 f4 f5 s cArray)
  (let ((ch (/ h 4.0)))
    (dotimes (k neqn)
      (setf (svref f5 k) (+ (svref y k) (* ch (svref yp k)))))
    (funcall f (+ time ch) f5 f1 cArray)
    (setq ch (/ (* 3.0 h) 32.0))
    (dotimes (k neqn)
      (setf (svref f5 k) (+ (svref y k) 
			    (* ch (+ (svref yp k) (* 3.0 (svref f1 k)))))))
    (funcall f (+ time (/ (* 3.0 h) 8.0)) f5 f2 cArray)
    (setq ch  (/ h 2197.0))
    (dotimes (k neqn)
      (setf (svref f5 k) (+ (svref y k) 
			    (* ch (+ (* 1932.0 (svref yp k))
				     (- (* 7296.0 (svref f2 k))
					(* 7200.0 (svref f1 k))))))))
    (funcall f (+ time (/ (* 12.0 h) 13.0)) f5 f3 cArray)
    (setq ch (/ h 4104.0))
    (dotimes (k neqn)
      (setf (svref f5 k) (+ (svref y k) 
			    (* ch (+ (- (* 8341.0 (svref yp k))
					(* 845.0 (svref f3 k)))
				     (- (* 29440.0 (svref f2 k))
					(* 32832.0 (svref f1 k))))))))
    (funcall f (+ time h) f5 f4 cArray)
    (setq ch (/ h 20520.0))
    (dotimes (k neqn)
      (setf (svref f1 k) (+ (svref y k)
			    (* ch (+ (+ (* -6080.0 (svref yp k))
					(- (* 9295.0 (svref f3 k))
					   (* 5643.0 (svref f4 k))))
				     (- (* 41040.0 (svref f1 k))
					(* 28352.0 (svref f2 k))))))))
    (funcall f (+ time (/ h 2.0)) f1 f5 cArray)
    (setq ch (/ h 7618050.0))
    (dotimes (k neqn)
      (setf (svref s k) (+ (svref y k) 
			   (* ch (+ (+ (* 902880.0 (svref yp k))
				       (- (* 3855735.0 (svref f3 k))
					  (* 1371249.0 (svref f4 k))))
				    (+ (* 3953664.0 (svref f2 k))
				       (* 277020.0 (svref f5 k))))))))
    ))


(defun rksign (a1 a2)
  (if (>= a2 0.0) (abs a1) (- (abs a1))))



;;; This determines how many times an iflag=4 error is permitted to occur
;;; in sequence.  Sometimes rk45 gets hosed for unknown reasons, and this
;;; gives it more than one chance to get it right.
;;;
(defparameter *rk45-max-reset-on-iflag4* 2)


;;;----------------------------------------------------------------------------
;;; Do a simulation step using rk45.
;;; Inputs:  s    - The state vector for the system.
;;;          rk   - The rk struct for the system.
;;;          time - The desired final time.
;;; Returns: The final time and s is set to s(time).
;;; Notes:   This function does the error handling needed on a return from
;;;          rk45.
;;;----------------------------------------------------------------------------
;;;
(defun rk45-run (s rk time)
  (do ((iflag nil)
       (reset-on-iflag4 0)
       tout)
      ((eql iflag 2) tout)
    
    (multiple-value-setq (iflag tout) (rk45 s time rk))
;    (format t "~%~a  ~a   ~a" tout (svref s 0) (svref s 1))
    (cond 
      ((= iflag 2))
      ((= iflag 3) 
       (format t "~%RK45: At time=~,3F, new tolerance (~a) [rel=~a abs=~a]"
	       tout iflag (rk-relerr rk) (rk-abserr rk)))
      ((= iflag 5)
       (setf (rk-abserr rk) 1.e-9)
       (format t "~%RK45: At time=~,3F, new tolerance (~a) [rel=~a abs=~a]"
	       tout iflag (rk-relerr rk) (rk-abserr rk)))
      ((= iflag 6)
       (if (zerop (rk-abserr rk))
	   (setf (rk-abserr rk) (rk-relerr rk)))
;       (setf (rk-relerr rk) (* 10.0 (rk-relerr rk)))
       (setf (rk-iflag rk) 2)
       (format t "~%RK45: At time=~,3F, new tolerance (~a) [rel=~a abs=~a]"
	       tout iflag (rk-relerr rk) (rk-abserr rk)))
      ((= iflag 7)
       (setf (rk-iflag rk) 2)
       (format t "~%RK45: At time=~,3F, iflag = 7; resetting to 2" tout))
      ((= iflag 4)
       (setf (rk-iflag rk) 2)
       (incf reset-on-iflag4)
       (when (>= reset-on-iflag4 *rk45-max-reset-on-iflag4*)
	 (cerror (format nil "Let iflag=4 occur another ~a times"
			 *rk45-max-reset-on-iflag4*)
		 "RK45: At time=~,3F, encountered iflag=4 ~a times.  ~&This means that ~
                  simulation progress is very slow." tout *rk45-max-reset-on-iflag4*)
	 (setf reset-on-iflag4 0)))
      (T
       (error "RK45: iflag = ~a" iflag)))))


;;; Test program

(defun orbit (time y yp)
  (declare (special alfasq) (ignore time))
  (let ((r (+ (* (svref y 0) (svref y 0)) (* (svref y 1) (svref y 1)))))
    (setq r (/ (* r (sqrt r)) alfasq))
    (setf (svref yp 0) (svref y 2))
    (setf (svref yp 1) (svref y 3))
    (setf (svref yp 2) (/ (- (svref y 0)) r))
    (setf (svref yp 3) (/ (- (svref y 1)) r))))


(defun rk45ex ()
  (let* ((neqn 1)
	 (time 0.0)
	 (y   (make-array (list neqn) 
			  :initial-contents (list 0 )))
	 (relerr 1.0e-9)
	 (abserr 0.0)
	 (tfinal 15.0)
	 (tprint 0.1)
	 (iflag 1)
	 (tout  time)
	 (w (rk45-init #'orbit neqn relerr abserr time iflag)))
    (do
     ()
     ((> time tfinal))
     (multiple-value-setq (iflag time) (rk45 y tout w))
     (format t "~%~a   ~a   ~a" time (svref y 0) iflag)
;     (format t "~%~a   ~a   ~a" time (pressure (svref y 0)) iflag)
     (when (= iflag 2)
       (setq tout (+ time tprint)))
     (when (= iflag 7)
       (setf (rk-iflag w) 2))
     (when (member iflag '(3 5 6))
       (format t "  New tolerances ~a ~a" relerr abserr))
     (when (not (member iflag '(2 3 5 6 7)))
       (return)))))
