;;; -*- Mode:Common-Lisp; Package: QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: qutils.lisp,v 1.9 92/07/08 12:56:02 clancy Exp $
;;;
;;; Copyright (c) 1986 by Benjamin Kuipers.

(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;  QUTILS
;;;
;;;  General functions and macros.
;;;-----------------------------------------------------------------------------


(defun reset-trace-switches ()
  (mapc #'(lambda (switch) (set switch nil))
	'(*trace-count* *trace-tuples* *trace-constraint-filter* *trace-mult-constraint*
	  *trace-global-filters* *trace-main-agenda* trace-propagation
	  trace-propagation-result *show-completion-steps*
	  *pause-before-simulating*))
  t)


; Small utilities:  access functions and the like.

(defun insert (x set)
  (error "Use PUSHNEW instead of INSERT.")
  (if (member x set :test #'equal)
      set
      (cons x set)))

; This redefinition of LOOKUP and LOOKUP-SET as macros allows two things:
;   SETF forms will now work on them, and
;   keyword arguments (:test, :key and :test-not) will also work.

(defmacro lookup-set (key alist &rest keyphrases)
  `(cdr (assoc ,key ,alist ,@keyphrases)))

(defmacro lookup (key alist &rest keyphrases)
  `(cadr (assoc ,key ,alist ,@keyphrases)))

(defmacro alookup  (key alist &rest keyphrases)	; lookup in a true alist.
  `(cdr (assoc ,key ,alist ,@keyphrases)))

;;;-----------------------------------------------------------------------------
;;; (PARTITION <elts> :test <test>) builds a partition of <elts>. Two elts are
;;; equivalent if <test> is true when applied on them.
;;; NPARTITION may destroy <elts>.
;;; PFouche 03 June 1991
;;;-----------------------------------------------------------------------------

(defun partition (elts &key (test #'eql))
  (cond ((null elts) nil)
	;((null (cdr elts)) nil)	;must be at least two elts
	(t (let* ((elt1 (first elts))
		  (eq-elts (list elt1))
		  (diff-elts ()))
	     (dolist (elt2 (cdr elts))
	       (if (funcall test elt1 elt2)
		   (push elt2 eq-elts)
		   (push elt2 diff-elts)))
	     ;(format t "~2% EQ: ~a~%DIFF: ~a" eq-elts diff-elts)
	     (cons (nreverse eq-elts)
		   (partition diff-elts :test test))))))

(defun npartition (elts &key (test #'eql))
  (when elts
    (let* ((elt1 (first elts))
           (eqv-elts (list elt1)))
      (setq elts (delete-if #'(lambda (elt2)
                                (when (funcall test elt1 elt2) 
                                  (push elt2 eqv-elts)))
                            (cdr elts)))
      (cons eqv-elts (npartition elts :test test)))))


; Utilities

;;; Return true if xlist has one and only one element.
(defmacro singleton-p (xlist)
  `(and ,xlist (null (cdr ,xlist))))

;;; Return true if the qmag is an interval value.
(defmacro interval-p (qmag)
  `(listp ,qmag))

;;; Return true if the qmag is a point value.
(defmacro point-p (qmag)
  `(atom ,qmag))

;;; Return true if state is a time-interval state     05/16/91  DJC
(defmacro time-interval-state (state)
  `(interval-p (qmag (cdar (state-qvalues ,state)))))

;;; Return true if state is a time-point state        05/16/91  DJC
(defmacro time-point-state (state)
  `(point-p (qmag (cdar (state-qvalues ,state)))))

;;; Pushes elements of first list onto the second list.
(defmacro pushlist (add-list base-list)
  `(setq ,base-list (nconc ,add-list ,base-list)))


;;;-----------------------------------------------------------------------------
;;;  GEN-SYMBOL-NAME returns a string containing the print-name of <prefix>,
;;;  optionally followed by "-", followed by a number, e.g., "level-12". 
;;;  This function is used in place of (format nil "~a-~a" prefix n) because
;;;  format is INCREDIBLY SLOW.
;;;  The dash-p argument controls whether or not "-" is included in name.
;;;-----------------------------------------------------------------------------

(defun gen-symbol-name (prefix n dash-p)
  (let* ((pstring (string prefix))
	 (plength (length pstring))
	 (dlength (if dash-p 1 0))
	 s)
    (if (> n 9999)
	(setq n (rem n 10000)))
    (cond ((< n 10)
	   (setq s (make-string (+ 1 dlength plength))))
	  ((< n 100)
	   (setq s (make-string (+ 2 dlength plength))))
	  ((< n 1000)
	   (setq s (make-string (+ 3 dlength plength))))
	  ((< n 10000)
	   (setq s (make-string (+ 4 dlength plength)))))
    (dotimes (i plength)
      (setf (schar s i) (schar pstring i)))
    (if dash-p
	(setf (schar s plength) #\-))
    (incf plength dlength)
    (cond ((< n 10)
	   (setf (schar s plength) (digit-char n)))
	  ((< n 100)
	   (multiple-value-bind (tens ones) (floor n 10)
	     (setf (schar s plength) (digit-char tens)
		   (schar s (+ 1 plength)) (digit-char ones))))
	  ((< n 1000)
	   (multiple-value-bind (hundreds remainder99) (floor n 100)
	     (setf (schar s plength) (digit-char hundreds))
	     (multiple-value-bind (tens ones) (floor remainder99 10)
	       (setf (schar s (+ 1 plength)) (digit-char tens)
		     (schar s (+ 2 plength)) (digit-char ones)))))
	  ((< n 10000)
	   (multiple-value-bind (thousands remainder999) (floor n 1000)
	     (setf (schar s plength) (digit-char thousands))
	     (multiple-value-bind (hundreds remainder99) (floor remainder999 100)
	       (setf (schar s (+ 1 plength)) (digit-char hundreds))
	       (multiple-value-bind (tens ones) (floor remainder99 10)
		 (setf (schar s (+ 2 plength)) (digit-char tens)
		       (schar s (+ 3 plength)) (digit-char ones)))))))
    s))

(defun initial-time ()
  (intern (gen-symbol-name *Time-stem* 0 nil) 'qsim))

;;;-----------------------------------------------------------------------------
;;;  GENNAME creates and returns a symbol whose print-name is the same as the
;;;  print-name of the given atom, except that a numeric suffix is added,
;;;  such as "-12", where the number is derived from the atom's property list.
;;;
;;;  The returned symbol is either interned or not in package QSIM, depending
;;;  on the value of *intern-gennames*.  For QSIM debugging it is useful to
;;;  have interned symbols, but GENNAME runs about 4 times faster if the
;;;  symbols are not interned.
;;;-----------------------------------------------------------------------------

(defparameter *intern-gennames* t)
(defparameter *interned-symbols* nil)		; list of interned symbols
(defvar *genname-roots* nil)			; Keep a list of root names to
						; be reset in qsim-cleanup.

(defun GENNAME (atom  &key (package-name 'qsim))
  (let ((num (get atom 'gennum))
	name-string)
    (unless num
      (setq num 0)
      (pushnew atom *genname-roots*))
    (setq name-string (gen-symbol-name atom num *dashp*))
    (setf (get atom 'gennum) (1+ num))		
    (if *intern-gennames*
	(let ((name (intern name-string package-name)))
	  (push name *interned-symbols*)
	  ;(set name name)
	  (setf (symbol-plist name) nil)
	  name)
	(make-symbol name-string))))


(defun GET-INDEX-STRING (symbol)
  "Get the index (trailing digits) of a symbol like 'foo-1990"
  (let* ((name (string symbol))
	 (pos (position-if-not #'digit-char-p name :from-end t)))
    (if (null pos)
	name
	(subseq name (1+ pos)))))


;;;-----------------------------------------------------------------------------
;;;  Qspace functions
;;;
;;;  -- A quantity space is a sequence of landmark values:  (L1 L2 ... Ln).
;;;  -- A qualitative value (qval) contains a qualitative magnitude (qmag)
;;;     and a qualitative direction-of-change (qdir).
;;;  -- A qmag is either a landmark L, or two adjacent landmarks (Li Lj).
;;;  -- A qdir is 'inc, 'std, or 'dec (increasing, steady, or decreasing).
;;;-----------------------------------------------------------------------------

;;;  (succ lmark qspace) returns the landmark immediately following
;;;  qmag in the sequence of landmarks.

(defun succ (lmark qspace)
  (or (cadr (member lmark qspace))
      (error "~%Landmark ~a has no successor in ~a." lmark qspace)))

;;; (pred lmark qspace) returns the landmark immediately preceding
;;; lmark in qspace.

(defun pred (lmark qqspace)
  (do ((qspace qqspace (cdr qspace)))
      ((null (cdr qspace))
       (error "~%Landmark ~a has no predecessor in ~a." lmark qqspace))
    (if (eq lmark (second qspace))
	(return (first qspace)))))

;;; Returns non-nil if a < b in sequence of landmarks.
(defun landmark-lt (a b qspace)
  (member b (cdr (member a qspace))))

;;; Returns non-nil if a <= b in sequence of landmarks.
(defun landmark-le (a b qspace)
  (member b (member a qspace)))

;;; Returns non-nil if lower and upper bounds are adjacent in the sequence of landmarks.
(defun adjacent-p (bounds qspace)
  (let ((lower (first bounds))
	(upper (second bounds)))
    (and lower
	 upper
	 (eql upper (cadr (member lower qspace))))))

(defun or* (&rest args)
  (dolist (arg args)
    (when arg (return-from or* arg))))

(defun and* (&rest args)
  (dolist (arg args arg)
    (unless arg (return-from and* nil))))

; EXACTLY-ONE looks for the unique member of list that satisfies F.
; The one non-NIL value of F is returned.
; If there are zero or more than one, NIL is returned.

(defun exactly-one (F list)
  (do ((L list (cdr L))
       (temp nil)
       (val nil))
      ((null L) val)
    (cond ((setq temp (funcall F (car L)))
	   (if val (return nil))
	   (setq val temp)))))

(defun one-nil (L)
  (exactly-one #'null L))


;;;-----------------------------------------------------------------------------
;;;  TIMER UTILITIES
;;;  These functions facilitate the measurement of elapsed clock-time intervals. 
;;;  -- The functions are active only if *timing* is true.
;;;  -- Timer entries in *timers* are of the form:
;;;     (<timer name>  <cumulative time>  <latest starting time>  <# calls>).
;;;  -- To make time measurements, simply surround the thing to be timed with
;;;     (start-timer 'name1) ... (stop-timer 'name1).  Multiple calls will
;;;     record the accumulated time.  When all done, do (print-timers).
;;;-----------------------------------------------------------------------------

(defparameter *timers* nil)
(defparameter *timing* nil)

(defun start-timer (name)
  "Record starting time for a named timer."
  (if *timing*
      (let ((entry (assoc name *timers*)))
	(if entry
	    ;; reset starting time for an existing timer.
	    (progn
	      (setf (third entry) (get-internal-run-time))
	      (incf (fourth entry)))
	    ;; set starting time for a new timer.
	    (push (list name 0 (get-internal-run-time) 1) *timers*)))))

(defun stop-timer (name)
  "Record elapsed time for a named timer."
  (if *timing*
      (let ((entry (assoc name *timers*)))
	(if entry
	    (incf (second entry) (- (get-internal-run-time) (third entry)))
	    (format *qsim-report* "~%Timer ~a was never started!" name)))))

(defun print-timers ()
  "Print elapsed time of all timers."
  (when *timing*
    (format *qsim-report* "~%TIMED ENTITY~35T   Time    Calls")
    (dolist (entry *timers*)
      (format *qsim-report* "~%   ~(~a~)~35T~7d  ~7d"
	      (first entry) (second entry) (fourth entry)))))

(defun reset-timers ()
  "Reset all timers to zero."
  (if *timing*
      (dolist (entry *timers*)
	(setf (second entry) 0
	      (fourth entry) 0))))



;;; The RPT macro is just to cut down on typing.  It sends its report to
;;; T, rather than *qsim-report*, because we generally DO want to see
;;; warning of incorrect syntax when running the comparison utility.
;;; Also, it is preferred over WARN because it binds
;;; *detailed-printing*.

(defmacro rpt (string &rest body)
  `(let ((*detailed-printing* nil))
     (format t ,(concatenate 'string "~&WARNING: " string "~&")
	   ,@body)))

;;;-----------------------------------------------------------------------------
;;;  SETF method for LDB-TEST
;;;
;;;  This setf method for ldb-test allows for setf'ing truth values for forms
;;;  such as (variable-independent-p var).  This method was adapted from the
;;;  example on page 106 of "Common Lisp the Language" by Guy Steele Jr.
;;;
;;;  Note:  The Symbolics and TI Lisp environments defines a similar setf
;;;         method for ldb-test, but since it is not part of the language
;;;         definition, we include this definition so that this will work on
;;;         non-Symbolics systems.
;;;-----------------------------------------------------------------------------

#-lispm
(define-setf-method ldb-test (bytespec word)
  (multiple-value-bind (word-temps word-vals word-stores word-store-form word-access-form)
      (get-setf-method word)			; get setf method for int.
    (let ((btemp (gensym))			; temp var for bytespec.
	  (store (gensym))			; temp var for truth val to store.
	  (vtemp (gensym)))			; temp var new byte value.
      (values (cons btemp word-temps)		; temporary variables.
	      (cons bytespec word-vals)	; value forms.
	      (list store)			; store variables.
	      `(let* ((,vtemp (if ,store 1 0))
		      (,(first word-stores) (dpb ,vtemp ,btemp ,word-access-form)))
		 ,word-store-form)		; Storing form.
	      `(ldb-test ,btemp ,word-access-form)))))	; Accessing form.



;;;-----------------------------------------------------------------------------
;;;  NUMERIC RANGE FORMATTER
;;;
;;;  Given a numeric range, this function returns a string containing the range
;;;  in the form:
;;;                  [<lower bound> <upper bound>]
;;;  such as:
;;;                  [3.51 8.0]  or  [-inf 0.007]  or  [-0.6 +inf]
;;;
;;;  Why not just use the format function, as in (format nil "[~a ~a]" lb ub)?
;;;  The problem is that rationals get printed like 24365/4208 and floats may
;;;  get printed with more digits than you care about.  The ~G format directive
;;;  is the closest to what is desired, but it always has 4 spaces on the right
;;;  in case it needs to include an exponent like "E+03", and of course it 
;;;  doesn't handle '-inf and '+inf.
;;;-----------------------------------------------------------------------------

(defun range-to-string (range)
  (let* ((lb (first  range))
	 (ub (second range))
	 (control1 (control-string lb))
	 (control2 (control-string ub)))
    (format nil "[~? ~?]" control1 (list lb) control2 (list ub))))

(defun control-string (value)
  (etypecase value
    (symbol
      "~A")
    (fixnum
      (let ((absval (abs value)))
		(cond ((< absval 100000)  "~D")
		      (t                  "~,2E"))))
    ((or float rational)
     (let ((absval (abs value)))
		(cond ((< absval 0.001)   "~,2E")
		      ((< absval 0.01)    "~,5F")
		      ((< absval 0.1)     "~,4F")
		      ((< absval 1)       "~,3F")
		      ((< absval 10)      "~,2F")
		      ((< absval 100)     "~,1F")
		      ((< absval 100000)  "~,0F")
		      (t                  "~,2E"))))))

(defmacro lmark-find (lmark-name seq)
  `(car (member ,lmark-name ,seq :key 'lmark-name :test 'equal)))

(defmacro var-find (var-name seq)
  `(car (member ,var-name ,seq :key 'variable-name :test 'equal)))



;;;  Macros used in the Q2 code.  Previously had been in arithetic.lisp
;;;  DJC  06/4/91
;;;  Access macros for intervals: (lo hi).

(defmacro lo (int)
  `(car ,int))

(defmacro hi (int)
  `(cadr ,int))




;;;=============================================================================
;;;		 G R A P H   M A P P I N G   F U N C T I O N S 
;;;
;;; Pierre Fouche
;;; August 1990
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; 			 !  Accumulate result 	| Do not accumulate result |
;;; 		  	 ---------------------------------------------------
;;; Applied on node	 ! 	mapnode		|	  mapn		   |	
;;; Applied on path 	 !	mapgraph	|	  mapg		   |
;;;
;;; Depth-first strategy is always used.
;;;-----------------------------------------------------------------------------

;;;-----------------------------------------------------------------------------
;;; MAPN applies function to node and its successors in the graph. Function
;;; should return nil if mapping should not go deeper than the current node, and
;;; not nil otherwise.
;;;-----------------------------------------------------------------------------

(defun mapn (function node &key (successor-function #'get-successors)
				(max-depth nil))
  (declare (special successor-function max-depth))
  (let ((*visited-nodes* nil))
    (declare (special *visited-nodes*))
    ;; *visited-nodes* contains the list of already visited nodes.
    (mapn-1 function node 0))
  t)

(defun mapn-1 (function node &optional depth)
  (declare (special *visited-nodes* successor-function max-depth))
  (let ((continue-p (funcall function node)))
    (push node *visited-nodes*)
    (unless (or (not continue-p)
		(and max-depth
		     (= depth max-depth)))
      (incf depth)
      (dolist (next-node (funcall successor-function node))
	(unless (member next-node *visited-nodes*)
	  (mapn-1 function next-node depth))))))


;;;-----------------------------------------------------------------------------
;;; MAPG applies function to path and possible paths through the graph.
;;; Apply-when-cycle controls whether function is reapplied when the end of the
;;; path is already a member of the path.
;;;-----------------------------------------------------------------------------

(defun mapg (function path &key (successor-function #'get-successors)
				(max-depth nil)
				(apply-when-cycle nil))
  (declare (special successor-function max-depth apply-when-cycle))
  (let ((*visited-nodes* nil))
    (declare (special *visited-nodes*))
    ;; *visited-nodes* contains the list of already visited nodes.
    (mapg-1 function path 0)))


(defun mapg-1 (function path &optional depth)
  (declare (special *visited-nodes* successor-function max-depth
		    apply-when-cycle))
  (let* ((node (car path))
	 (cycle-p (if *develop-graph*
		      ;; added cdr
		      (member node (cdr path))
		      (member node *visited-nodes*)))
	 (continue-p (funcall function path)))
    (cond (cycle-p
	   (when apply-when-cycle
	     (funcall function node)))
	  (t (push node *visited-nodes*)
	     (unless (or (not continue-p)
			 (and max-depth
			      (= depth max-depth)))
	       (incf depth)
	       (dolist (next-node (funcall successor-function node))
		   (mapg-1 function (cons next-node path) depth)))))))
    

;;;-----------------------------------------------------------------------------
;;; Test functions
;;;-----------------------------------------------------------------------------

;(defun pn (&optional (state *initial-state*))
;  (mapn #'print state))

;(defun pg (&optional (path (list *initial-state*)))
;  (mapg #'print path))

;(defun ln (&optional (node *initial-state*))
;  (mapnode #'(lambda (x) (values x t)) node))

;;;-----------------------------------------------------------------------------
;;; MAPNODE is similar to MAPN but it returns the list of what function returned
;;; on each node. 
;;;-----------------------------------------------------------------------------

(defun mapnode (function node &key (successor-function #'get-successors)
				   (max-depth nil)
				   (apply-when-cycle nil))
  (declare (special successor-function max-depth apply-when-cycle))
  (let ((*visited-nodes* ())
	(*values* ()))
    (declare (special *visited-nodes* *values*))
    ;; *visited-nodes* contains the list of already visited nodes.
    (mapnode-1 function node 0)
    *values*))

(defun mapnode-1 (function node &optional depth)
  (declare (special *visited-nodes* successor-function
		    max-depth *values* apply-when-cycle))
  (if (member node *visited-nodes*)
      (when apply-when-cycle
	(push (funcall function node) *values*))
      (multiple-value-bind (val continue-p)
	  (funcall function node)
	(push node *visited-nodes*)
	(push val *values*)
	(unless (or (not continue-p)
		    (and max-depth
			 (= depth max-depth)))
	  (incf depth)
	  (mapc #'(lambda (next-node)
		    (mapnode-1 function next-node depth))
		(funcall successor-function node))))))


;;;-----------------------------------------------------------------------------
;;; MAPGRAPH is similar to MAPG but it returns a list of lists. Each list
;;; contains the results of each function call, accumulated using nconc.
;;; Function should return two values: the value itself as the first value,
;;; and T or nil to determine if search should continue further from the current
;;; path.
;;;-----------------------------------------------------------------------------

(defun mapgraph (function path &key (successor-function #'get-successors)
				(max-depth nil)
				(apply-when-cycle nil)
				(visited-nodes nil)) ; added DJC so that you can start with a 
                                                     ; with a list of visited nodes.
  (declare (special successor-function max-depth apply-when-cycle))
  (let ((*visited-nodes* visited-nodes))
    (declare (special *visited-nodes*))
    ;; *visited-nodes* contains the list of already visited nodes.
    (mapgraph-1 function path nil 0)))

(defun mapgraph-1 (function path values &optional depth)
  (declare (special *visited-nodes* successor-function max-depth apply-when-cycle))
  (let* ((node (car path))
	 (cycle-p (if *develop-graph*
		      (member node (cdr path))
		      (member node *visited-nodes*)))
	 new-values next-nodes)
    (if cycle-p
	(if apply-when-cycle
	    (list (nconc (funcall function node) values))
	    (list values))
	(multiple-value-bind (val continue-p)
	    (funcall function node)
	  (setq new-values (nconc val values))
	  (setq next-nodes (funcall successor-function node))
	  (push node *visited-nodes*)
	  (cond ((and continue-p next-nodes
		      (or (not (and max-depth
				    (= depth max-depth)))))
		 (incf depth)
		 (mapcan #'(lambda (next-node) 
			     (mapgraph-1 function (cons next-node path)
					 new-values depth))
			 (funcall successor-function node)))
		(t (list new-values)))))))

;(defun lg (&optional (path (list *initial-state*)))
;  (mapgraph #'(lambda (x) (values (list x) t)) path))
