;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: qspace-hierarchy.lisp,v 1.3 92/05/29 09:56:21 bert Exp $

(in-package 'QSIM)

;       Copyright 1987, Benjamin Kuipers.

; The (= X Y) constraint is satisfied if X and Y share a qspace in
; the qspace hierarchy, and if X(t) and Y(t) are compatible.  A shared qspace
; may in general be partially ordered, unlike the base qspaces which must be
; totally ordered.

;     (other
;        (qspace-hierarchy
;            ((X Y Z) ->  (*seq minf 0 (*set Xmax Ymax) Zmax inf))
;            ((P Q R) ->  (*seq minf 0 (*set Pmax (*seq Qmax Rmax)) inf))
;            . . .
;        . . .

(defparameter same-qdir-relation
	      '((inc inc)(std std)(dec dec)))

(declare-qsim-constraint
  (make-contype :name              '=
		:nargs             2
		:propagator        '=-propagator
		:checkfcn          'check-equal-constraint
		:qmag-relation     nil
		:qdir-relation     same-qdir-relation
		:cvals-allowed-p   nil
		:implied-cvals     nil
		:disallowed-lmarks nil))


;;;----------------------------------------------------------------------
;;; A propagator for the = constraint.
;;
;; Written by Adam Farquhar with some consulting from D. Throop.
;; Written on: 14 Feb 1990
;;
;; 24.Oct.90 Adam Farquhar.  Fixed the mysterious deleteion of a paren and "al" in the var cvals.
	      
(defun =-propagator (con cvals state)
  ;; Args: con, a constraint, cvals, always nil, as = does not take
  ;; corresponding values, and state.
  ;;
  ;; Return: a list of vars effected by propagation.
  ;; Side Effect: set the qvalue of x or y.
  ;;
  (when cvals
    (error "The = constraint does not accept corresponding values: ~a ~a ~a"
	   con cvals state))
  (let* ((template (constraint-name con)) ; (= x y)
	 (vars     (constraint-variables con)) ; (<x> <y>)
					;	 (type (first template))    ;used only in tracing  - drt
	 (x-name    (second template))
	 (x         (first  vars))
	 (y-name    (third template))
	 (y         (second vars))
	 (changed nil))
    ;;   Following is already reported elsewhere:	-- RM
    ;;   (and trace-propagation (format t "~&*** Propagating (~a ~a ~a) ***~%" type x-name y-name))
    (cond ((and (consp y-name)
		(eq 'CONSTANT (car y-name)))
	   ;; Handle the special case where y-name = (constant mag).
	   (if
	    (assert-eq x (find (second y-name) (variable--qspace x)
			       :key #'lmark-name)
		       state)
	    (setq changed (list x)))
	   (if  (set-qdir x 'std)
		(pushnew x changed))
	   changed)
	  (t;; The general case where they are both vars.
	   (union (glistify (prop-= x x-name y y-name state))
		  (union 
		   (glistify (prop-= y y-name x x-name state))
		   (glistify (propagate-qdir con cvals state))))))))

(defun glistify (x)
  (and x (list x)))

(defun intervalize (x)
  (if (listp x) x
      (list x x)))

(defun prop-= (x x-name y y-name state)
  ;; prop x to y
  ;; Return Y if successful, else NIL.
  (let* ((xqmag    (qmag (qval x-name state)))
	 (yqmag    (qmag (qval y-name state)))
	 (Xqspace  (variable--qspace x))
	 (HQspace  (get-HQspace (variable-name x)
				(variable-name y))))
    (if (null HQspace)
	(error "No shared qspace for ~a and ~a" x y))
    (when xqmag
      ;; We sometimes use NIL to mean inf.  E.g. (0 nil).  The HQspace
      ;; code doesn't know about this, so we'll patch the xqmag.
      (setq xqmag (patch-qmag xqmag Xqspace))
      (let ((HQvals (map-to-best-HQspace-value Xqmag Xqspace HQspace)))
	(and trace-propagation (format t "~&*** Found value (~a ~a) = ~a~%" x-name y-name HQvals))
	(cond ((null HQvals) nil)
	      ((atom HQvals)
	       (if (and yqmag (atom yqmag))
		   (and trace-propagation
			(format t "~&*** in =prop, ~a already has value ~a, not propagating~%"
				y-name yqmag))
		   (assert-eq y (find HQvals (variable--qspace y)
				      :key #'lmark-name)
			      state)))
	      (T
	       (let ((lower (find (first HQvals) (variable--qspace y)
				  :key #'lmark-name))
		     (upper (find (second HQvals) (variable--qspace y)
				  :key #'lmark-name)))
		 ;; HQval nil or pair of lmark names.
		 (if (or (null upper) (null lower))
		     (error "No HQ mapping for ~a" HQvals)
		     (or* (and lower (assert-lb y lower state))
			  (and upper (assert-ub y upper state)))))))))))

(defun patch-qmag (qmag qspace)
  (cond ((atom qmag) qmag)
	(T
	 (list (or (car qmag)
		   (car qspace))
	       (or (cadr qmag)
		   (car (last qspace)))))))

;;;----------------------------------------------------------------------
;;;
;;
;; Inequality constraints added 23 Jan 1990 by Adam Farquhar.
;;
;; The inequality constraints <, <=, >, >= 
;; The < constraint is the primitive, the others are treated in terms of
;; = and it.
;; (< a b) is true if:
;; a b 
;; p p a       po-lessp b
;; p i a       po-lessp b.upper
;; i p a.lower po-lessp b
;; i i a.lower po-lessp b.upper
;; where p means point, and i interval.
;;
;; Note, this is often weaker than I had expected.  It means that there
;; is some real value, ai, bi in each qualitative value a, b such that
;; ai < bi.  E.g. (0 inf) < (0 inf).
;; 
(declare-qsim-constraint
  (make-contype :name              '<
		:nargs             2
		:propagator        nil
		:checkfcn          'check-inequality-constraint
		:qmag-relation     nil
		:qdir-relation     nil
		:cvals-allowed-p   nil
		:implied-cvals     nil
		:disallowed-lmarks nil))

(declare-qsim-constraint
  (make-contype :name              '>
		:nargs             2
		:propagator        nil
		:checkfcn          'check-inequality-constraint
		:qmag-relation     nil
		:qdir-relation     nil
		:cvals-allowed-p   nil
		:implied-cvals     nil
		:disallowed-lmarks nil))
(declare-qsim-constraint
  (make-contype :name              '<=
		:nargs             2
		:propagator        nil
		:checkfcn          'check-inequality-constraint
		:qmag-relation     nil
		:qdir-relation     nil
		:cvals-allowed-p   nil
		:implied-cvals     nil
		:disallowed-lmarks nil))

(declare-qsim-constraint
  (make-contype :name              '>=
		:nargs             2
		:propagator        nil
		:checkfcn          'check-inequality-constraint
		:qmag-relation     nil
		:qdir-relation     nil
		:cvals-allowed-p   nil
		:implied-cvals     nil
		:disallowed-lmarks nil))

;(declare-qsim-constraints
;  '((=  check-equal-constraint)))		; => generalize to >, <, >=, <=.

(defun check-equal-constraint (tuple constraint)
  (cond ((atom (third (constraint-name constraint)))
	 (check-equal-vv-constraint tuple constraint))
	(t (check-equal-vm-constraint tuple constraint))))

(defun check-inequality-constraint (tuple constraint)
  (let* ((vars     (constraint-variables constraint))
	 (Xqspace  (variable--qspace (first  vars)))
	 (Yqspace  (variable--qspace (second vars)))
	 (HQspace  (get-HQspace (variable-name (first vars))
				(variable-name (second vars))))
	 (Xqmag    (qmag (first tuple)))
	 (Yqmag    (qmag (second tuple))))
    (if (null HQspace)
	(error "Constraint ~a lacks shared quantity space." constraint))
    (inequality-compatible
      (car (constraint-name constraint))
      (map-to-best-HQspace-value Xqmag Xqspace HQspace)
      (map-to-best-HQspace-value Yqmag Yqspace HQspace)
      HQspace)))

(defun qvalue-in-state (var state)
  "Return the qvalue of VAR in STATE.  Var can be either a var or its name."
  (alookup (if (variable-p var)
	       (variable-name var)
	       var)
	   (state-qvalues state)))

(defun check-inequality (state ineq x y)
  ;; {x, y are varnames}
  (let* ((Xqspace  (qspace x state))
	 (Yqspace  (qspace y state))
	 (HQspace (get-HQspace x y))
	 (Xqmag    (qmag (qvalue-in-state x state)))
	 (Yqmag    (qmag (qvalue-in-state y state))))
    (if (not (member ineq '(< > <= >=)))
	(error "Inequality ~a is not one of (< > <= >=)" ineq))
    (if (null HQspace)
	(error "Constraint ~a lacks shared quantity space." (list x y)))
    (let ((*current-qde* (state-qde state)))
      (declare (special *current-qde*))
      (inequality-compatible
	ineq
	(map-to-best-HQspace-value Xqmag Xqspace HQspace)
	(map-to-best-HQspace-value Yqmag Yqspace HQspace)
	HQspace))))

; This function handles the (= <var1> <var2>) version of the constraint.

(defun check-equal-vv-constraint (tuple con)
  (let* ((vars     (constraint-variables con))
	 (Xqspace  (variable--qspace (first  vars)))
	 (Yqspace  (variable--qspace (second vars)))
	 (HQspace  (get-HQspace (variable-name (first vars))
				(variable-name (second vars))))
	 (Xqmag    (qmag (first tuple)))
	 (Yqmag    (qmag (second tuple))))
    (if (null HQspace)
	(error "Constraint ~a lacks shared quantity space." con))
    (and (same-qdirs (qdir (first tuple)) (qdir (second tuple)))
	 (compatible (map-to-best-HQspace-value Xqmag Xqspace HQspace)
		     (map-to-best-HQspace-value Yqmag Yqspace HQspace)
		     HQspace))))


; Unfortunately, this has to use a global variable, *current-qde*, to get access to
; the qspace-hierarchy which is on the qde.other.qspace-hierarchy slot.

(defun get-HQspace (x y &optional (qde *current-qde*))
  (declare (special *current-qde*))
  (do ((HQspaces (alookup 'qspace-hierarchy (qde-other qde))
		 (cdr HQspaces)))
      ((null HQspaces) nil)
    (cond ((and (member x (car (car HQspaces)))
		(member y (car (car HQspaces))))
	   (return (caddr (car HQspaces)))))))

; Compatible takes two qualitative magnitude descriptions and determines whether
; they may be compatible or not.  A qualitative magnitude may be either a landmark P,
; or an open interval (P Q).
; Compatible means that there is some real valued number which could be
; in both A and B (Adam thinks).
; (compatible 'a 'b '(*seq a b)) => nil
; (compatible 'a 'b '(*set a b)) => T
; (compatible 'a 'b '(*eqv a b)) => t
; (compatible 'a 'b '(*set a (*eqv b c))) => T
;
; Modified by Adam Farquhar.  I added the second case under (and (atom
; a)(atom b)).  It is my belief that if a is not less than b, and b is
; not less than a, then a and b are compatible.  Does anyone else
; believe that?
; 
(defun compatible (a b poset)
  (cond ((atom a)
	 (cond ((atom b)
		(or (eql a b)
		    (and (not (po-lessp a b poset))
			 (not (po-lessp b a poset)))))
	       ((member a b) nil)
	       ((po-lessp a (car b) poset) nil)
	       ((po-lessp (cadr b) a poset) nil)
	       (t t)))
	((atom b)
	 (cond ((member b a) nil)
	       ((po-lessp b (car a) poset) nil)
	       ((po-lessp (cadr a) b poset) nil)
	       (t t)))
	(t (cond
	     ((eql (cadr a) (car b)) nil)	; e.g. (minf 0)(0 inf)
	     ((eql (car a) (cadr b)) nil)	; e.g. (0 inf)(minf 0)
	     ((po-lessp (cadr a) (car b) poset) nil)
	     ((po-lessp (cadr b) (car a) poset) nil)
	     (t t)))))

;; inequality-compatible takes an inequality (one of <,>) and two
;; qualitative magnitude descriptions and an HQspace desriptor (poset).
;; The two descriptions are compatible if there could exist some x1 in X
;; and some x1 in Y such that (ineq x1 y1) would be satisfied.
;; con-name is a list (<ineq> var1 var2)
(defun inequality-compatible (ineq-type x y poset)
  (cond
    ((and (member ineq-type '(<= >=))
	  (compatible x y poset))
     T)
    ;; because we have eliminated the = case, we can treate <= as <, and
    ;; >= , > as < of the inverse.
    ((member ineq-type '(> >=))
     (less-compatible y x poset))
    ((member ineq-type '(< <=))
     (less-compatible x y poset))
    (T (error "ineq-type ~a not one of < <= >=." ineq-type))))

;; Ok. A < B is T if (*set A B), because there might be some real value
;; for a and b such that a<b.  This means A < B if A is poless B, or A
;; is compatible with B, but not equal to it.
;; (less-compatible 'a 'b '(*seq a c b)) => t
;; (less-compatible 'a '(b d) '(*seq a c b)) => T
;; (less-compatible 'a '(b d) '(*seq (*set a) (*eqv c b))) => T
;; (less-compatible 'a 'b '(*set a c b)) => T
;; (less-compatible 'a 'b '(*seq b (*set a c))) => NIL
;;
(defun less-compatible (a b poset)
  (cond ((atom a)
	 (if (atom b)
	     (or (po-lessp a b poset)
		 (po-unknown-p a b poset))
	     (or 
	       (po-lessp a (second b) poset)
	       (po-unknown-p a (second b) poset))))
	((atom b)
	 (or
	   (po-lessp (car a) b poset)
	   (po-unknown-p (car a) b poset)))
	(T
	 ;; both a and b are intervals
	 (or (po-lessp (car a) (cadr b) poset)
	     (po-unknown-p (car a) (cadr b) poset)))))

(defun po-unknown-p (a b poset)
  "Return T if a and b have no known relationship to each other."
  (and (not (po-equal-p a b poset))
       (not (po-lessp a b poset))
       (not (po-lessp b a poset))))


; Map a qualitative value to the most restrictive value in the HQspace that
; includes that value.
; qvalue is really a qmag!

(defun map-to-best-HQspace-value (qmag qspace HQspace)
  (cond ((atom qmag)
	 (or (map-landmark-to-HQspace qmag qspace HQspace)
	     (list (map-to-lb qmag qspace HQspace)
		   (map-to-ub qmag qspace HQspace))))
	(t (list (map-to-lb (car qmag) qspace HQspace)
		 (map-to-ub (cadr qmag) qspace HQspace)))))

; Find the next lower (or higher) landmark that has a corresponding value in the HQspace.

(defun map-to-lb (landmark qspace HQspace)
  (let ((next nil))
    (cond ((map-landmark-to-HQspace landmark qspace HQspace))
	  ((setq next (pred landmark qspace))
	   (map-to-lb next qspace HQspace))
	  (t (error "No predecessor for ~a in ~a." landmark qspace)))))

(defun map-to-ub (landmark qspace HQspace)
  (let ((next nil))
    (cond ((map-landmark-to-HQspace landmark qspace HQspace))
	  ((setq next (succ landmark qspace))
	   (map-to-ub next qspace HQspace))
	  (t (error "No successor for ~a in ~a." landmark qspace)))))

; Map-to-HQspace maps a landmark to a corresponding landmark in a HQspace, if possible.
; Returns the landmark symbol in the HQspace, or nil.

(defun map-landmark-to-HQspace (landmark qspace HQspace)
  (declare (ignore qspace))
  (let ((lm (lmark-name landmark)))
    (cond ((po-member lm HQspace) lm)	; the trivial mapping, for now.
	  (t nil))))

(defparameter test1 '(A p B q C E))		; notice new elements p and q.

; A "poset" is a partially ordered set (of atoms).
; A partial order on a set of atoms is stored as structures representing
; equivalence classes, sequences, and sets.  A term Sn refers to a structure
; from the order description, and can be treated as a set, even if it is
; stored as an atomic symbol (which represents the singleton set).
;   (*eqv A B C ... )   means A = B = C = ... in the order relation.
;   (*seq S1 S2 S3 ... ) means forall a in S1, b in S2, c in S3, a < b < c ...
;   (*set S1 S2 S3 ... ) means that order relations are not specified among
;                        the elements of S1, S2, and S3.

(defparameter pox1 '(*seq A B (*set C D) E))	; two versions of one example:  (A B {C D} E)
(defparameter pox2 '(*set (*seq A B)
			  (*seq B C E)
			  (*seq B D E)))


;; IT APPEARS THAT THE follwing are NOT allowed!!!!!!
;; The *eqv only allows symbols!!!
;; (*eqv a (*seq b c (*set d (*seq e f (*eqv g h)))))
;; a is po-equal to each of {b c d e f g h}
;; (*eqv (*seq (*eqv a g) b c) (*eqv d (*seq e f)))))
;; a is o-equal to each of {g d e f}
;;
;; (setq x '(*seq a b (*eqv c1 c2 c3) d e))

(defun po-equal-p (x y poset)
  "X and Y are equal under the partial ordering if there is an *eqv element
   containing both of them."
  (cond ((atom poset) nil)
	((eql (car poset) '*eqv)
	 (and (member x poset)
	      (member y poset)))
	(t (some #'(lambda (po)
		     (po-equal-p x y po))
		 (cdr poset)))))

; po-lessp is the basic order relation on partial orders.  It searches through the
; successors of x until it finds y.

(defun po-lessp (x y poset)
  (let ((next (po-successor x poset)))
    (cond ((null next) nil)
	  ((member y next) T)
	  (t (catch 'lessp
	       (mapc #'(lambda (z) (if (po-lessp z y poset) (throw 'lessp T)))
		     next)
	       nil)))))

; po-member tells whether a given element belongs to a poset.

(defun po-member (x poset)
  (cond ((atom poset) (eql x poset))
	((eql (car poset) '*eqv) (if (member x (cdr poset)) T))
	((member (car poset) '(*seq *set))
	 (catch 'po-member
	   (mapc #'(lambda (s) (if (po-member x s) (throw 'po-member T)))
		 (cdr poset))
	   nil))
	(t (error "Bad prefix ~a in partially ordered set ~a." (car poset) poset))))

; Return a list containing the first element(s) of a poset.

(defun poset-first (poset)
  (cond ((null poset) nil)
	((atom poset) (list poset))
	((eql (car poset) '*eqv) (cdr poset))
	((eql (car poset) '*set) (eliminate-duplicates
				   (mapcan #'poset-first (cdr poset))))
	((eql (car poset) '*seq) (poset-first (cadr poset)))))

; Immediate successors of X.  Returns a list of atoms, or nil.

(defun po-successor (x poset)
  (cond ((atom poset) nil)
	((eql (car poset) '*eqv) nil)		; => fix later
	((eql (car poset) '*seq)
	 (po-successor-in-sequence x (cdr poset)))
	((eql (car poset) '*set)
	 (eliminate-duplicates
	   (mapcan #'(lambda (s) (po-successor x s)) (cdr poset))))
	(t (error "Bad poset"))))

(defun po-successor-in-sequence (x seq)
  (cond ((null seq) nil)
	((po-member x (car seq)) (poset-first (cadr seq)))
	(t (po-successor-in-sequence x (cdr seq)))))

(defun eliminate-duplicates (L)
  (cond ((null L) nil)
	((member (car L) (cdr L)) (eliminate-duplicates (cdr L)))
	(t (cons (car L) (eliminate-duplicates (cdr L))))))

; These are notes from the original discussion from CS 395T on this constraint.

; What are the semantics for the EQUAL constraint?
;   An EQUAL constraint may hold among two variables (EQUAL X Y),
;    -> The quantity spaces for X and Y must have a common ancestor in
;       the global quantity space for that unit.
;    -> The constraint is satisfied if the qualitative values of X and Y are
;       compatible in that common ancestor qspace:  overlapping or non-comparable.
;    -> Should probably consider an equivalence class of variables under EQUAL
;       and MINUS^2.
; How does this relate with MINUS?  Reversed quantity spaces??

; Consider two sequential resistors with the following qspaces:

; (qspaces
;   (R1.I  (minf 0 I1max inf))
;   (R2.I  (minf 0 I2max inf)))

; In QSIM, as we increase I, we can't express knowledge that I1max < I2max, so
; that as we increase current, R1 burns out before R2.  QSIM would branch.
;
; Create a higher quantity space (QR (minf 0 I1max I2max inf)) which expresses
; the inequality.  The constraint (EQUAL R1.I R2.I) is generated by the CC-model,
; and compatibility checks against QR filter out the possibilities:
;    R1.I     I1max      (0 I1max)
;    R2.I     I2max      I2max
; If the higher-quantity space is partially ordered: (QR (minf 0 {I1max I2max} inf)),
; then the branch takes place.

; The (= <var> <qmag-description>) version of the constraint is satisfied 
; if the value of <var> is consistent with <qmag-description>.
; The possible forms for <qmag-description> are:
;        (CONSTANT <qmag>)

(defun check-equal-vm-constraint (tuple con)
  (let* ((qmag-description (third (constraint-name con)))
	 (qspace    (variable--qspace (first (constraint-variables con))))
	 (user-qmag (cadr qmag-description))
	 (qmag      (convert-qmag user-qmag qspace)))
    (cond ((eql (car qmag-description) 'constant)
	   (and (eql (qdir (first tuple)) 'std)
		(subsumes qmag (qmag (car tuple)) qspace))))))

; Test whether one qualitative value subsumes another.

(defun subsumes (x y qspace)
  (cond ((and (atom x) (atom y)) (eql x y))
	((atom x) nil)
	((atom y) (and (landmark-lt (car x) y qspace)
		       (landmark-lt y (cadr x) qspace)))
	(t (and (landmark-le (car x) (car y) qspace)
		(landmark-le (cadr y) (cadr x) qspace)))))


; a couple of tests:
#|
  (define-qde equal-test
	      (quantity-spaces (x (minf 0 x* inf)) (y (minf 0 inf)))
    (constraints ((= x y)))
    (other (qspace-hierarchy ((x y) -> (*seq minf 0 inf)))))

  (setq s (make-new-state :from-qde equal-test :assert-values '((x (x* std)))))

(define-qde ineq-test
	      (quantity-spaces (x (minf 0 x* inf)) (y (minf 0 inf)))
    (constraints ((< x y)))
    (other (qspace-hierarchy ((x y) -> (*seq minf 0 inf)))))

(setq s (make-new-state :from-qde ineq-test :assert-values '((x (x* std)))))
|#
