;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: q2.lisp,v 1.15 92/07/07 11:57:22 clancy Exp $
;
;  Sancho:>xq>experiments>intervals.lisp.21 copied from nq:nq;intervals.lisp.109 (10-16-90).
;  Split into three files (11-5-90)
;    - arithmetic.lisp  - interval arithmetic.
;    - equations.lisp   - generate equations from behavior
;    - q2.lisp          - the Q2 range propagator

;  Copyright (c) 1988, Benjamin Kuipers and Daniel Berleant.
(in-package 'QSIM)

; Modified 3Sept91 by BKay to correct errors in the format of vls.  Now, all vls are
; stored as (<VARIABLE> LMARKEXP) where <VARIABLE> is a QSIM variable structure
; and any landmarks in LMARKEXP are QSIM lmark structures.

; This is the Q2 interval range reasoner, that adds the capability of reasoning
; with incomplete quantitative knowledge (in the form of intervals) about 
; qualitative behaviors (by propagation across constraints).
;
; The code consists of several chapters:
;   1.  Top-level call to Q2 as a global filter on QSIM states.
;   2.  Access functions for variable-landmark pairs (VLPs) and ranges.
;   3.  Propagation loop.
;   4.  Zip-up and Zip-down quantity spaces.
;   5.  Display and trace functions.
;
; File arithmetic.lisp holds the interval-arithmetic routines.


; Information for quantitative range reasoning is stored on m-envelopes and initial-ranges
; slots in the form illustrated by the following examples:
;    (m-envelopes
;      ((M+ amount outflow) (upper-envelope (lambda (x) x))
;                           (upper-inverse  (lambda (y) y))
;			    (lower-envelope (lambda (x) (if (< x 20) (* x .5)
;							             (+ (* x .2) 6))))
;			    (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
;							             (* 5 (- y 6)))))))
;                           (slope (0.2 1.0))
;                           (curvature (-0.1 0.1))
;    (initial-ranges ((inflow if*)  (5 10))
;		    ((amount full) (50 100))
;		    ((time t0)     (0 0))))


; Control parameters.

;      The global data structures for RANGER:
(defparameter *bindings* nil)			; current state of ranges
(defparameter *M-envelopes* nil)		; envelopes for monotonic functions
(defparameter *agenda* nil)			; agenda of equations to propagate

(defparameter *state* nil)			; current state

(defparameter infminf (list *inf-lmark* *minf-lmark*))

(defparameter *trace-Q2-check* nil)		; trace global filter use
(defparameter *trace-q2-agenda* nil)	       	; trace addition of equations to agenda
(defparameter *trace-consider-eqn* nil)		; show when each equation is considered
(defparameter *trace-range-update* nil)		; trace changes to intervals

(defother qde initial-ranges)


; 1.  Top-level call to Q2 as a global filter on QSIM states.
;
;   It is therefore interleaved with qualitative simulation.
;   Runs on time-points (not intervals), if the switch is on, and there is
;   some useful range information to work with.
;     Normally, this is called without the range-init argument, but it can be
;     called with explicit new information.

(defun quantitative-range-reasoning (current-state &key (assert-ranges nil))
  ;; Added the local declaration of *current-qde*
  (let ((*current-qde* (state-qde current-state)))
    (declare (special *current-qde*))
    ;; Added a check of sim.q2-constraints so that a user doesn't have to set
    ;; the *check-quantitative-ranges* flag to run Q2 from outside the
    ;; qsim function.  BKay 25Oct91
    (cond ((and (null *check-quantitative-ranges*)
		(null (sim-q2-constraints (state-sim current-state))))
	   current-state)
	  ((not (qpointp (state-time current-state))) current-state)
	  ((inconsistent-p current-state) current-state)   ; changed so as not to access the slot directly
	  ((incomplete-p current-state) current-state)     ; DJC 09/19/91
	  (t (trace-Q2-global-filter current-state)
	     (setup-equations-from-state current-state)
	     (get-globals-from-state current-state)
	     (setup-agenda current-state)
	     ;; Unfortunately, assimilate-asserted-ranges can cause a throw
	     ;; which needs to be caught, so we call it wrapped in a
	     ;; catch and prune the state if the catch is thrown.
	     ;; BKay 8Jun92
	     (cond
	       ((catch 'contradiction
		  (assimilate-asserted-ranges current-state assert-ranges))
		(cond ;((null *bindings*) current-state)
		  ((propagate-ranges)
		   (store-globals-in-state current-state)
		   current-state)
		  (t (prune-inconsistent-state current-state
					       "Quantitative ranges")
		     (store-globals-in-state current-state)
		     nil)))
	       (T
		(prune-inconsistent-state current-state
					  "Quantitative ranges")
		nil))))))




(defun get-globals-from-state (state)
  (setq *state* state)
  (setq *agenda* nil)
  (setq *bindings* (state-bindings state))
  (setq *m-envelopes* (qde-M-envelopes (state-qde state)))
  t)


(defun store-globals-in-state (state)
  (setf (state-bindings state) *bindings*)
  t)  

;;; Given a varname and a state, get the <var> structure
;;;
(defun var-from-name-and-state (varname state)
  (alookup varname (qde-var-alist (state-qde state))))


;;; This has been modified to just push eqns onto the agenda.
;;; The zips are done immediately.  BKay 27May92.
;;;
(defun setup-agenda (state)
;  (dolist (qspace (state-qspaces state))
;    (let ((var (var-from-name-and-state (car qspace) state)))
;      (add-to-zip-agenda  var)))
  (dolist (var-entry (state-eqn-index state))
    (dolist (vl-entry (cdr var-entry))
      (dolist (eqn (cdr vl-entry))
	(add-to-agenda eqn))))
  t)



;;; Ensure that a vl is really (<VARIABLE> <LMARK>) 
;;; If the vl is not, then it returns a vl that is.
;;; Note that this function permits cases where (vl-lmark vl) is 
;;; either a <LMARK> or (AT S-x) (the latter being used by MIMIC).
;;;
(defun vlize (possible-vl state)
  (let ((v  (vl-var   possible-vl))
	(l  (vl-lmark possible-vl)))
    (cond
      ((and (variable-p v) (lmark-p l))
       possible-vl)
      (T
       (let ((qde (state-qde state)))
	 (list
	   (if (variable-p v)
	       v
	       (cond
		 ((var-from-name-and-state v state))
		 (T
		  (error "No variable found for ~a in qde ~a" v qde))))
	   (if (lmark-p l)
	       l
	       (cond
		 ;; DLD 9/20/91 allow for '(at s-33)
		 ((and (listp l) (eql 'at (first l)))
		  l)
		 ;; This is changed so that if the vl-var is a var struct,
		 ;; and the vl-lmark is a lmark name, everything should work.
		 ;;; BKay 15Jun92.
		 ((lmark-find l
			      (lookup-set (if (variable-p v) (variable-name v) v)
					  (state-qspaces state))))
		 (T
		  (error "No lmark found for ~a in state ~a" l state))))))))))



; This assumes that user-asserted ranges are on state.assert-ranges,
; QDE.initial-ranges, or provided as an explicit argument.
; This is inefficient in that qde.initial-ranges need be called
; only once for each QDE.

(defun assimilate-asserted-ranges (state ranges)
  (let ((from-arg ranges)
	(from-qde (qde-initial-ranges (state-qde state)))
	(from-state (state-assert-ranges state)))
    ;; We don't know what format the vl (car item) will be in, so to be on
    ;; the safe side, we ensure that it is in the form (var lmark) as opposed
    ;; to (varname lmarkname).  Note that this won't allow anything other than
    ;; (var lmark) (e.g., no (var (AT lmark)) specs).
    ;;
    ;; This has been changed to do the updates immediately (rather than
    ;; pushing them onto the agenda).  Since they are the last things
    ;; to be pushed, the effect is the same as to do the updates immediately.
    ;; BKay 27May92
    (dolist (item from-qde)
      (update-range (vlize (car item) state) (cadr item)))
    (dolist (item from-state)
      (update-range (vlize (car item) state) (cadr item)))
    (dolist (item from-arg)
      (update-range (vlize (car item) state) (cadr item)))
    ;; This must return T since it is being used inside a catch whose
    ;; throw returns NIL.  BKay 8Jun92
    T))


#|
;;; THESE FUNCTIONS ARE PRESENTLY UNUSED BKay 3Sept91

;;; In CLEANUP-USER-ASSERTED-RANGES 
;;; Entry := ((var lmark) (lb  ub)) where var and lmark are symbols
;;; Returns list of entry2 where
;;; Entry2 := ((<var> <lmark>)(lb ub))
;;; The previous definition was bombing when an lmark had the same name as the var. -drt 22feb90

(defun cleanup-user-asserted-ranges (entries state)	; ((<var> <lmark-name>) (lb ub))
  (loop with valist = (qde-var-alist (state-qde state))
	for ((varname lmarkname) bounds) in entries
	for var = (alookup varname valist)
	for lmark = (cleanup-landmark-description varname lmarkname state)
	when lmark
	  collect `((,var ,lmark) ,bounds)))

; Clean up a landmark-name by finding the LMARK structure in the qspace.
; Translate (<var> (AT <time>)) into a landmark if possible; else leave be.

(defun cleanup-landmark-description (varname lmdesc state)
  (cond ((symbolp lmdesc)
	 (find lmdesc (lookup-set varname (state-qspaces state)) :key #'lmark-name)
	 ; add (error "No landmark ~a in qspace for ~a in ~a." lmdesc varname state) ?
	 )
	((and (at-expressionp lmdesc)
	      (= (length lmdesc) 2))		; only handling one case
	 (or (lmark-at-time varname (cadr lmdesc) state)
	     lmdesc))
	(t (error "Can't handle landmark description ~a = ~a." varname lmdesc))))


(defun lmark-at-time (varname time-name state)
  (let ((time nil))
    (cond ((null state) nil)
	  ((and (setq time (qval-qmag (cdr (assoc 'time (state-qvalues state)))))
		(atom time)
		(eql time-name (lmark-name time)))
	   (let ((qmag (qval-qmag (cdr (assoc varname (state-qvalues state))))))
	     (if (typep qmag 'lmark) qmag nil)))
	  (t (lmark-at-time varname time-name (predecessor-of-state state))))))
|#

;   2.  Access functions for variable-landmark pairs (VLPs) and ranges.

; The following access functions use find-or-create retrieval to make
; sure that every <vl> starts with the appropriate range (usually (-inf +inf)).  
; Retrieval and updating clobber the actual list cell where the range is stored.

(defun get-range (vl)
  (car (get-range-binding (vl-var vl) (vl-lmark vl))))

; Inputs:  var - a variable structure
;          lm  - a LMARKEXP
;
;; BKay 27May92
;; 1. removed error checks.  This function gets called ALOT, it should
;; not be doing error checking on every call.
;;
;; 2. it is called by zipup/down, and calls zipup/down.  I don't see
;; why, so I commented it out.  This tremondously reduces the number
;; of calls to zipup/down, and the results appeart to be the same.
;;
;; I don't really understand it, but including the zipup/down seems to
;; result in fewer calls to int, thus less consing.  There appears to
;; be an increase in time, however.
;; * My guess is that by not zipping after creation of a range, the
;; * equations are not as effective.  Since the zips are guaranteed to
;; * be run at the end by the new propagate-ranges (I believe that this
;; * is the correct time to run them, to prevent an overlooked zip),
;; * many equations need to be reexamined later on, thus more work,
;; * but less consing (since many of these equations won't really shrink
;; * anything so late in the game).
;;
(defun get-range-binding (var lm)
  (let ((var-entry (or (assoc var *bindings*)
		       (car (push (list var) *bindings*)))))
    (let ((vl-entry (assoc lm (cdr var-entry) :test #'equal)))
      (when (null vl-entry)
	(setq vl-entry (create-vl-entry var lm))
	(nconc var-entry (list vl-entry)))
      (cdr vl-entry))))



; Update-range takes the intersection between the old range and the new one,
; and stores the result, detecting a null intersection.
;
; Inputs:  vl    - a vl
;          range - a pair of (enum enum) (where enum is a number or -inf or +inf)
;
(defun update-range (vl range)
  (let ((binding (get-range-binding (vl-var vl) (vl-lmark vl)))
	(intersection nil))
    (cond ((null range) (error "Invalid range for ~a:  ~a." vl range))
	  ((span>= range (car binding)))	; do nothing
	  ((insignificant-difference range (car binding))
	   (trace-ignore-update vl (car binding) range))
	  ((setq intersection (iintersect range (car binding)))
	   (trace-range-update vl (car binding) intersection)
	   (setf (car binding) intersection)
	   (follow vl))
	  (t (trace-range-update-failure vl (car binding) range)
	     (throw 'contradiction nil)))
    (car binding)))

; A brand-new entry for a landmark is created with range (-inf +inf).
; An entry for (d <l1> <l2>) is created with an initial range (0 +inf), because l1<l2.
; NONONO!  This comment is not true.  (d <l1> <l2>) should NOT be (0 +inf)
; since l1 is not always less than l2.
; Inputs:  var - a VARIABLE (not a varname)
;          lm  - a LMARKEXP (not a lmarkname)
; Returns: (lm range)
;
(defun create-vl-entry (var lm)
  (declare (ignore var))
    (cond ((lmark-p lm)
	   (let ((lmarkname (lmark-name lm)))
	     (if (numberp lmarkname)
		 (list lm (int lmarkname lmarkname))
		 (list lm (int '-inf '+inf)))))
	((d-expressionp lm)      (list lm (int '-inf '+inf))) 
	((at-expressionp lm)     (list lm (int '-inf '+inf)))
	((defint-expressionp lm) (list lm (int '-inf '+inf)))
	(t (error "Impossible case."))))

; Probably the D case should follow the AT case, letting one more step of propagation
; refine the trivial initial value of (0 +inf).  The make+store-d-eqns must go at
; equation creation time, then.

; 3.  Propagation loop.
;
;   *BINDINGS*  ::=  alist of (<var> . alist of (<landmark> <range>))
;        encodes the current state of knowledge about the value of <landmark>
;
;   *AGENDA*    ::=  (priority) queue of <equations> to propagate across.
;
; This is a modified version of the driver for this process (BKay 28May92).

;;; Once all the equations have been run,
;;; do a zip of all variables just to make sure that nothing
;;; has slipped through the cracks.  Previously, there
;;; was a problem where, I surmise, a variable A gets a new
;;; lmark A-5 and all the equations that relate A-5 to something
;;; are also relative to new lmarks, so they don't generate an
;;; update-range call.  If no other lmarks in A change, there
;;; will be no zip of A and so A-5 won't get an interval better
;;; than [-inf +inf].  By forcing the zip at the end of
;;; each cycle thru the equations, this won't happen, although
;;; extra zips may be done (this does not appear to be too costly).
;;; BKay 29May92
;;;
(defun propagate-ranges ()
  (catch 'contradiction
    (loop while *agenda*
	  do
	  (loop for eqn = (pop *agenda*)
		   while eqn
		   do
		   (trace-consider eqn)
		   (ranger* eqn))
	  (loop for var-entry in *bindings*
		do (zip-variable (first var-entry)))
	  )
    T))


;;; original version
;(defun propagate-ranges ()
;  (do ((N 0 (+ N 1))
;       (eqn nil))
;      (nil)
;    (cond ((null *agenda*) (return t))		; done
;	  (t (setq eqn (pop *agenda*))
;	     (or (catch 'contradiction
;		   (trace-consider eqn)
;		   (cond ((member (car eqn) '(zip-up zip-down update-range)) (eval eqn))
;			 (t (ranger* eqn)))
;		   t)				; t = normal termination
;		 (return nil))))))



(defun add-to-agenda (equation)	; => replace with priority queue later.
  (trace-q2-agenda equation)
  ;; :test eq is a big win.
  (pushnew equation *agenda* :test #'eq))



; Once a range is updated, add all expressions that depend on that range
; to the agenda.

;;; Modified to use an eqn and a zip agenda.  BKay 27May92
;;;
(defun follow (vl)
  (declare (special *state*))
  (let ((equations (cdr (assoc (vl-lmark vl)
			       (cdr (assoc (vl-var vl) (state-eqn-index *state*)))
			       :test #'equal))))
    (dolist (eqn equations) (add-to-agenda eqn))
    (zip-variable (vl-var vl))
    equations))


;;; Do a zip on a variable.  Zips are now done immediately by calls to follow.
;;; This removes *zip-agenda* and related code, and should have about the
;;; same effect as pushing them on the stack since they are usually immediately
;;; popped right back off.
;;; BKay 28May92
;;;
(defun zip-variable (var)
  (trace-consider var)
  (zip-up var)
  (zip-down var))


; 4.  Zip-up and Zip-down quantity spaces.
;
; When the range associated with a landmark changes, the implications can 
; quickly be "zipped" up and down the quantity space.
; This shrinks ranges as lower-bounds zip upward and upper-bounds zip downward.
; Contradiction detected when a range upper bound is not greater than the
; upward-propagating lower bound, and vice versa.
; Additionally, a check is made for adjoining landmarks which are both points
; and have the same value (i.e. v-0 is (1 1) and v-1 is also (1 1)).  This case
; raises a contradition.  It is inside zip-up arbitrarily -- it could also be in
; zip-down.  This addition made by BKay 24Jun92
;
; Inputs:  var - a <VARIABLE>
;
(defun zip-up (var)
  (let ((lb '-inf)
	(prev-lmark-is-point NIL))
    (dolist (lmark (cdr (assoc (variable-name var) (state-qspaces *state*))))
      (cond ((member (lmark-name lmark) '(inf minf)))
	    (t (let* ((binding (get-range-binding var lmark))
		      (nlb (lo (car binding)))
		      (cur-lmark-is-point (point-range (car binding))))
		 (cond
		   ;; There is a contradiction if the higher lmark's upper
		   ;; range is below the previous lmark's lower range, or
		   ;; if both lmarks are the same point (there must be space
		   ;; between lmarks).
		   ((or (e<= (hi (car binding)) lb)
			(and cur-lmark-is-point
			     prev-lmark-is-point
			     (insignificant-value-difference nlb lb)))
		    (trace-zip-failure var lmark 'up binding lb)
		    (throw 'contradiction nil)))
		 (setf prev-lmark-is-point cur-lmark-is-point)
		 (cond ((equalp lb nlb))
		       ((or (eql lb '-inf) (and (numberp nlb) (< lb nlb)))
			(setq lb nlb))
		       ((or (eql nlb '-inf) (and (numberp lb) (> lb nlb)))
			;; This cond and the first clause added by  BKay 7Mar91
			;; The problem is that insignificant changes in lower bound 
			;; can lead to an infinite regress of 'follow' calls.
			(cond
			  ((insignificant-value-difference lb nlb)
			   (trace-ignore-update (list var lmark)
						(car binding)
						(list lb (hi (car binding)))))
			  (T
			   (trace-zip var lmark 'up (lo (car binding)) (hi (car binding))
				      lb (hi (car binding)))
			   (setf (lo (car binding)) lb)
			   (follow (list var lmark)))))
		       (t (error "Impossible case.")))))))))


(defun point-range (range)
  (insignificant-value-difference (lo range) (hi range)))


(defun zip-down (var)
  (let ((ub '+inf))
    (dolist (lmark (reverse (cdr (assoc (variable-name var) (state-qspaces *state*)))))
      (cond ((member (lmark-name lmark) '(inf minf)))
	    (t (let* ((binding (get-range-binding var lmark))
		      (nub (hi (car binding))))
		 (cond ((e<= ub (lo (car binding)))
			(trace-zip-failure var lmark 'down binding ub)
			(throw 'contradiction nil)))
		 (cond ((equalp ub nub))
		       ((or (eql ub '+inf) (and (numberp nub) (> ub nub)))
			(setq ub nub))
		       ((or (eql nub '+inf) (and (numberp ub) (< ub nub)))
;;                      This cond and the first clause added by  BKay 7Mar91
;;                      The problem is that insignificant changes in upper bound can lead to an infinite
;;                      regress of 'follow' calls.
			(cond
			  ((insignificant-value-difference ub nub)
			   (trace-ignore-update (list var lmark) (car binding) (list (lo (car binding)) ub)))
			  (T
			   (trace-zip var lmark 'down (lo (car binding)) (hi (car binding))
				      (lo (car binding)) ub)
			   (setf (hi (car binding)) ub)
			   (follow (list var lmark)))))
		       (t (error "Impossible case.")))))))))

; zip-down is more expensive, because of the consing in reversing the qspace.


; 5.  Display and trace functions.

;;; Print out the bindings for a variable (or all vars)
;;; in a state.
;;; Inputs:  state - A qsim state.
;;;          var   - A variable structure or a variable name.
;;; Returns: T, and prints the bindings.
;;; Modified by BKay 24May92
;;;
(defun show-bindings (state &optional (var nil) &aux (*detailed-printing* nil))
  (if var
      (show-bindings-for-variable
        (if (variable-p var)
	    (assoc var (state-bindings state))
	    (assoc var (state-bindings state) :key #'variable-name)))
      (dolist (var-entry (state-bindings state))
	(show-bindings-for-variable var-entry)))
  t)

(defun show-bindings-for-variable (var-entry)
  (format *QSIM-Trace* "~% ~a:" (car var-entry))
  (dolist (vl-entry (cdr var-entry))
    (format *QSIM-Trace* "~%~6t~a: ~16t~a" (car vl-entry) (cadr vl-entry))))

;;; If varname present, SHOW-EQUATIONS prints out only the equations
;;; associated with the variable of that name.  Otherwise, prints out
;;; all the Q2 equations for the state.

(defun show-equations (state &optional (varname))
  (loop with *detailed-printing* = nil
	for (var . var-entry) in (state-eqn-index state)
	when (or (null varname)
		 (equal varname var))
	  do (format *QSIM-Trace* "~% ~a:" var)
	     (dolist (vl-entry var-entry)
	      (format *QSIM-Trace* "~%~6t~a:" (car vl-entry))
	      (dolist (eqn  (cdr vl-entry))
		(format *QSIM-Trace* "~%~12t~a" eqn))))
  (values))

; There is a menu for controlling these variables in front-end.lisp.


(defparameter *q2-trace-line-len* 0)

;;; Break up the output stream after 80 characters have been printed.
;;; This is useful when lisp is run from emacs, as emacs has trouble
;;; with the potentially long lines of trace characters when Q2 is
;;; running.
;;; 
(defun broken-line-format (stream string)
  (when (> *q2-trace-line-len* 80)
    (setf *q2-trace-line-len* 0)
    (format t "~&"))
  (incf *q2-trace-line-len* (length string))
  (format stream string))

     
(defun trace-Q2-global-filter (state)
  (if *trace-Q2-check*
      (format *QSIM-Trace* "~%Starting quantitative check on ~a." state)))

(defun trace-q2-agenda (eqn)
  (if *trace-q2-agenda*
      (format *QSIM-Trace* "~%Add to agenda:  ~a." eqn)))

(defun trace-consider (eqn)
  (if *trace-consider-eqn*
      (format *QSIM-Trace* "~%Considering:  ~a." eqn))
  (if *trace-single-char* (broken-line-format *QSIM-Trace* ".")))

(defun trace-range-update (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Updating:  ~a from ~a -> ~a." vl old new))
  (if *trace-single-char* (broken-line-format *QSIM-Trace* "!")))

(defun trace-ignore-update (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Insignificant update ignored ~a:  ~a ~~ ~a." vl old new))
  (if *trace-single-char* (broken-line-format *QSIM-Trace* "~~")))

(defun trace-range-update-failure (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Contradiction between values for ~a:  ~a vs. ~a." vl old new))
  (if *trace-single-char* (broken-line-format *QSIM-Trace* "#")))

(defun trace-zip (var lm dir olb oub nlb nub)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%ZIP-~a (~a ~a):  (~a ~a) -> (~a ~a)."
	      dir var lm olb oub nlb nub)))

(defun trace-zip-failure (var lm dir range bound)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%ZIP-~a (~a ~a) = ~a.  Contradiction with bound ~a."
	      dir var lm range bound))
  (if *trace-single-char* (broken-line-format *QSIM-Trace* "#")))


; Single character trace output:
;   []      begin/end equation creation from state
;   .       evaluation of an equation
;   !       range update
;   ~       insignificant change to range
;   #       state found inconsistent

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;           Notes about fixes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; => The bug is that transition states are not properly filtered in the
;    current situation, because Q2 can only work after landmarks have been
;    created, and transition-state landmarks are only created after
;    the global-filter-list has been run.  This is fixed by adding code to do 
;    this in filter-for-transitions.
;
; => The interaction with state-transitions is a bit wierd anyway.
;
; => There is a partially-included bug fix that creates (= <lm1> <lm2>) equations
;    for LMARK structures of the same name, but not EQ, across region transitions.

; Need:
;
;  - recognize and handle higher-order derivative combinations:
;      (d/dt x v)  (d/dt v a)
;  - recognize simultaneous equations to solve directly.
;     (I+S minimal complete subset analysis?)

; Inefficiencies:
;  - assimilated-asserted-ranges asserts all user-provided ranges at each state.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;         Changes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; - Make-new-state has &key :assert-ranges argument.
;   This provides a way to assert quantitative range constraints by hand, with either:
;     (quantitative-range-reasoning <state> :assert-ranges <range-specs>)
;     (make-initial-state <state-or-qde> :assert-ranges <range-specs>)
;   Make-initial-state and make-modified-state do not have this feature.
;   However, they are now defined in terms of the more general make-new-state.
;
; - It is now possible to specify a value for (variable (time t2)).
;     e.g. in assert-ranges:   ((<var> (AT <time>)) (lb ub))
;     is equivalent to ((<var> <lmark>) (lb ub)), where <var>(<time>) = <lmark>.
;     If <var>(<time>) is not a landmark, nothing happens!
;   NOTE BY BKAY 3Sept91 : THIS CHANGE DOESN'T APPEAR TO BE IMPLEMENTED
;
; - Envelope functions may now be undefined (returning NIL), in which case
;   they provide no constraint.
;
; - Non-landmark values of the form (V (at T1)) are handled.
;   The MVT forms are handled more generally.
;   Equations are generated more cleanly.
;   Equation generation is not as redundant as before.
;   VLs are (<variable> <lmark>) rather than (<variable-name> <lmark-name>)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;         INCOMPATIBLE Changes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; - replaced unnamed second argument to quantitative-range-reasoning with
;   &key :assert-ranges argument.
;      (I don't believe anyone but me used that second arg anyway.)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;         Changes to speed up Q2.  By BKay and af 27May92
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; An attempt to speedup q2 somewhat.
;;; Results:
;;; On jack's example ~bert/qsim/nq/testsuite/jackcstr1.lisp,
;;; it more than triples the speed of quantitative-range-reasoning
;;; (120s -> 35s)
;;; and reduces the consing ;;; by a factor of 1.5 (1.3Mb -> .9 Mb)
;;; [This is all with monitoring enabled.  Without it, wallclock
;;; runtime improves from 90s -> 25s.].
;;;
;;; The current time bottleneck is get-range-binding.
;;; The current consing bottleneck is int.
;;;
;;; Changes:
;;;
;;; 1. Do updates immediately.  This is not a problem, because in the
;;;    old version, the update-range calls from ranger* are immediately
;;;    executed and the call from assimilate-asserted-ranges are the first
;;;    things pushed onto the agenda, so they happen immediately anyway.
;;;
;;; 1a. Do zips immediately.  This means that Change #2 needs only the
;;;    EQN agenda (now called *agenda*).  In the original code, it seems that
;;;    zips were usually processed immediately after being pushed onto the
;;;    agenda, so there is no real loss doing this, and there is a speed
;;;    by not maintaining the zip agenda.
;;;
;;; 2. break the agenda into a ZIP and an EQUATION (EQN) agenda.
;;;    Because the zip's are indexed by the variable, and that is
;;;    limited by the number of variables in the model, we can simply
;;;    do a pushnew.  The eqn agenda may become quite long (800 on
;;;    jack's model).  Alot of the cost in the old version comes from
;;;    doing a pushnew :test #'equal onto the agenda.  Even after
;;;    splitting the agenda, it is too big to use the equal test.
;;;    Thus, we need another method:
;;;    a. do pushnew eql.  This isn't too bad, but unfortunately there
;;;    are a fair number of equations that are equal but not eql.
;;;    (They are generated when two landmarks are functions of each
;;;    other, and the equations relating them turn out to be
;;;    identical.).  Thus, the test is reasonably fast, but a
;;;    considerable amount of duplicate work may be done (on the
;;;    order of 30% more calls to ranger*).
;;;    b. use an equal hash table to store an in-agenda? flag.  This
;;;    should be ok, but equal hash tables under lucid seem to be
;;;    pretty slow.  The overhead is quite high.
;;;    c. a better solution is to uniquify the equations when they are
;;;    generated.  This would allow an eq test, or even better a mark
;;;    bit on the equations.
;;;    * Actually, the problem was really one of gratuitous copy-trees
;;;    * in various other pieces of QSIM.  Once they were fixed, the
;;;    * equations became eq.
;;;    * We are therefore using method c.
;;;
;;;  3. fold round-interval into int.  Most of the consing is floats.
;;;  This saved 300K words on Jacks' problem.
;;;
;;;  4. get-range-binding no longer does error checking nor does it
;;;  call zipup/down.
;;;
;;;  5. Make *rounding-epsilon* << *epsilon*.  This prevents repeated
;;;  calculations of a quantitity from looking different from the stored
;;;  (and rounded) interval in *bindings*.  This change was a HUGE WIN.
;;;
;;; Further improvements (not yet implemented) :
;;;
;;; 1. Only store slope equations when there is a SLOPE clause in the qde
;;; M-ENVELOPES clause.

