;;; -*- Mode: LISP; Syntax: Common-Lisp; Base: 10; Package: QSIM -*-
;;;  $Id: propagation.lisp,v 1.6 92/05/27 14:23:48 kuipers Exp $

;;; Copyright (c) 1986 by Benjamin Kuipers.
;;; 1988 Dan Dvorak, Adam Farquhar, Benjamin Kuipers
;;; From Southern:>nq>propagation.lisp.7, created 10/02/89


(in-package 'QSIM)


;;; TEMPORARY STUFF TO GET IT TO WORK


(defun VAR-QDIR (var)
  (let ((qval (variable--qval var)))
    (if qval
	(qdir qval)
	nil)))


(defmacro VARIABLE-QDIR (var)
  `(var-qdir ,var))


(defun VARIABLE-QMAG (var)
  (let ((qval (variable--qval var)))
    (if qval
	(qmag qval)
	nil)))


(defmacro STATE-CONSTRAINTS (state)
  `(qde-constraints (state-qde ,state)))


(defmacro STATE-VARIABLES (state)
  `(qde-variables (state-qde ,state)))


(defmacro CONSTRAINT-QDIR-RELATION (con)
  `(contype-qdir-relation (constraint-type ,con)))


(defmacro CONSTRAINT-QMAG-RELATION (con)
  `(contype-qmag-relation (constraint-type ,con)))



;;;-----------------------------------------------------------------------------
;;; Function: (propagation state)
;;;
;;; Given:    -- state, with only partially specified values.
;;;              The goal is to propagate these values as far as possible,
;;;              hopefully completing them.
;;;
;;; Returns:  -- state, with the values as complete as possible.
;;;
;;; Design:   Process an agenda of contraints, attempting to propagate the
;;;           partial value information through it.  This is done by using
;;;           the contype-propagator function for the constraint.
;;;
;;;           After a parameter is changed, add each constraint touching
;;;           it to the agenda, in depth-first fashion.
;;;
;;; Notes:    The propagated state is NOT guaranteed to be consistent!
;;;           It is still necessary to check the values.
;;;
;;;-----------------------------------------------------------------------------


(defun PROPAGATION (state)
  (let ((agenda (copy-list (qde-constraints (state-qde state))))
	(constraint nil))
    (when (or trace-propagation trace-propagation-result)
      (format *Qsim-Trace* "~%~%Propagation"))
    (loop
       (when (null agenda)
	 (return))
       (setq constraint (pop agenda))
       (when (constraint-active-p constraint)
	 (propagation-tracer constraint)
	 (dolist (var (propagate-constraint constraint
					    (constraint--cvals constraint)
					    state))
	   ; (format *Qsim-Report* "~%~a changed in ~a, so ..." var constraint)
	   (dolist (con (alookup (qde-name (state-qde state))
				 (variable-constraints var)))
	     (when (and (constraint-active-p con)	; skip inactives
			(not (eq con constraint)))	; don't redo
	       ; (format *Qsim-Report* "~%  (re)checking ~a" con)
	       (pushnew con agenda)))))))
  (propagation-tracer state)
  state)


(defun PROPAGATION-TRACER (arg)

  "Trace the function Propagation, when required."
  (etypecase arg
    (constraint (when trace-propagation
		  (format *Qsim-Trace* "~&  Constraint ~a" arg)))
    (state (when trace-propagation-result
	     (format *Qsim-Trace* "~%    State after propagation ") (ps arg)))))


(defun PROPAGATE-TRACER (var value &optional landmark)
  
  "Trace the results of propagation, when required."
  (if trace-propagation
      (format *QSIM-Trace* "~%    Variable ~A is ~A~@[ relative to landmark ~A~]."
	      var value landmark)))

  
;;;-----------------------------------------------------------------------------
;;; Function: (copy-a-qval qval )
;;;
;;; Returns: A fresh list of qvalues.  Copy values to eliminate any sharing.  As
;;;          always, copying structures is problematic.  We do a copy-tree of
;;;          the other slot.
;;;-----------------------------------------------------------------------------


(defun COPY-A-QVAL (qval)
  (setq qval (copy-qval qval))
  (when (variable-independent-p (qval-variable qval))
    ;; these should always be std.  Nil => make steady.
    (ecase (qval-qdir qval)
      ((nil) (setf (qval-qdir qval) 'std))
      (std)))
  (setf (qval-qmag qval)
	(copy-list (qval-qmag qval)))
  (setf (qval-other qval)
	(copy-tree (qval-other qval)))
  qval)


;;;-----------------------------------------------------------------------------
;;; Function:  (propagate-constraint constraint cvals state)
;;;
;;; Given:     -- a constraint, its corresponding values list, and the state
;;;               from which it came.
;;;
;;; Returns:   A list of VARIABLES which were affected by propagation.
;;;
;;; Design:   The real work is done by the propagator function on the contype of
;;;           the constraint.  In case this is ever moved to the constraint
;;;           itself, we use an accessor, constraint-propagator.
;;;-----------------------------------------------------------------------------


(defun PROPAGATE-CONSTRAINT (constraint cvals state)
 (let ((propagator (constraint-propagator constraint)))
    (if (and propagator (fboundp propagator))
	(funcall propagator constraint cvals state)
	(propagate-constraint-tracer constraint))))


(defun PROPAGATE-CONSTRAINT-TRACER (constraint)

  "Trace function Propagate-Constraint, when required."
  (if trace-propagation
      (format *Qsim-Trace* "~%  No propagator function for constraint ~a"
	      (contype-name (constraint-type constraint)))))


;;;-----------------------------------------------------------------------------
;;;     C O N S T R A I N T  -  S P E C I F I C    P R O P A G A T O R S
;;;
;;;  Each constraint has an associated propagation function (see the
;;;  |declare-qsim-constraint| declarations in constraints.lisp).
;;;  A propagator function takes three arguments:
;;;
;;;     -- constraint,
;;;     -- corresponding values, and
;;;     -- current partially-defined state.
;;;
;;;  The purpose of a propagator function is to more precisely define the
;;;  partially-specified values of the constraint's variables, if possible.
;;;  For example, if we have the constraint (M+ A B), and if the qdir of A
;;;  is 'inc but the qdir of B is nil (unknown), then the propagator can
;;;  assert that the qdir of B must be 'inc.
;;;-----------------------------------------------------------------------------

;;;  Propagator for D/DT.

(defun DERIV-PROPAGATOR (con cvals state &aux var)
  (when (setq var (deriv-propagation  con cvals state))
    (list var)))

;;;  Propagator for MULT.

(defun MULT-PROPAGATION (con cvals state)
  (let ((changed-vars nil)
	(v1 (propagate-mult-signs con cvals state))	; returns nil or var
	(v2 (propagate-mult-qdirs con cvals state))	; returns nil or var
	(v3 (propagate-mult-cvs con cvals state)))	; returns nil or vars
    (if v1 (push v1 changed-vars))
    (if v2 (pushnew v2 changed-vars))
    (if v3 (dolist (v v3) (pushnew v changed-vars)))
    changed-vars))


;;;  Propagator for ADD, MINUS, M+, and M-.

(defun BASIC-PROPAGATION (con cvals state)
  (let ((vars nil)
	tmp)
    (when (setq tmp (propagate-qdir con cvals state))
      (push tmp vars))
    (when (setq tmp (propagate-qmag con cvals state))
      (pushnew tmp vars))
    vars))


;;;  Propagator for W+ and W-.

(defun PROPAGATE-W (con cvals state)
  (let ((var (propagate-qdir con cvals state)))
    (if var
	(list var))))


;;;  Propagator for CORRESPOND constraint.
(defun CORRESPOND-PROPAGATION (con cvals state)
  (let ((var (propagate-qmag con cvals state)))
    (if var
	(list var))))

;;; Propagation for INCREASING and DECREASING constraints.  If
;;; the QDIR has not been specified in MAKE-INITIAL-STATE or
;;; MAKE-TRANSITION-RESULT, this fills in the value and signals back to
;;; the rest of the propagation that the value is now available.

(defun UNARY-PROPAGATOR (con cvals state)
  (declare (ignore cvals state))
  (let* ((var (car (constraint-variables con)))
	 (qval (variable--qval var)))
    (when (null (qval-qdir qval))
      (setf (qval-qdir qval)
	    (ecase (contype-name (constraint-type con))
	      (increasing 'inc)
	      (decreasing 'dec)))
      (list var))))


;;; Propagation for CONSTANT constraints.  If the QDIR or the QMAG has not
;;; been specified in MAKE-INITIAL-STATE or MAKE-TRANSITION-RESULT, this
;;; fills in the value and signals back to the rest of the propagation that
;;; the value is now available.

(defun CONSTANT-PROPAGATOR (con cvals state)
  (declare (ignore cvals state))
  (let* ((var (car (constraint-variables con)))
	 (qval (variable--qval var))
	 (return-list nil))
    (when (null (qval-qdir qval))
      (setf (qval-qdir qval) 'std)
      (push var return-list))
    (when (and (null (qval-qmag qval)) (car (constraint-bend-points con)))
      ;; Set the QMAG to the landmark specified in the CONSTANT constraint
      (setf (qval-qmag qval)
	    (find (car (constraint-bend-points con)) (variable--qspace var)
		  :key 'lmark-name))
      (pushnew var return-list))
    return-list))


;;;-----------------------------------------------------------------------------
;;; Propagating qdir over a single constraint.
;;; returns the changed parameter if successful; nil otherwise.
;;;-----------------------------------------------------------------------------


(defun PROPAGATE-QDIR (con cvs state)

  (declare (ignore cvs state))
  (let ((qdirs (mapcar #'var-qdir (constraint-variables con))))
    (cond
      ((member 'IGN qdirs) nil)			; don't propagate with IGN
      ((one-nil qdirs)
       (let ((binding
	       (exactly-one #'(lambda (table-entry)
				(match-tuple qdirs table-entry
					     (constraint-variables con)))
			    (contype-qdir-relation (constraint-type con)))))
	 ;; binding is either NIL or (var qdir)
	 (when binding
	   (set-qdir (car binding) (cadr binding))
	   (propagate-tracer (first binding) (second binding))
	   (car binding)))))))


;;;-----------------------------------------------------------------------------
;;; Unless all parameters are completely specified, check the qmags against
;;; all corresponding values. 
;;; Returns: The variable if propagation took place, NIL otherwise.  Sets new
;;; qmag values in the variable.
;;; Note: assume that cvs are already augmented!
;;;-----------------------------------------------------------------------------


(defun PROPAGATE-QMAG (con cvs state &aux propagated?)
  (unless (every #'(lambda (var)
		     (complete-qmag-p var))
		 (constraint-variables con))
    ; (format *Qsim-Report* "~%propagate-qmag ~a: ~a" (constraint-name con) cvs)
    (dolist (cv cvs)
      (setq propagated? (or (propagate-wrt-corr-values con cv state)
			    propagated?)))
    ; (format *Qsim-Report* "~%returning ~a" propagated?)
    propagated?))


;;;-----------------------------------------------------------------------------
;;; propagate-wrt-corr-values returns the changed parameter.
;;;
;;; Returns: a variable if propagation takes place, NIL otherwise.
;;; Given:   -- con, a constraint,
;;;          -- corr-vals, a single set of corr. values,
;;;          -- state.
;;;
;;; Design:  Each constraint has a set of ordering relations, which are
;;; used in propagation.  E.g. ADD has (+ + +).  This means that if we
;;; have a corresponding value set (x1 y1 z1) and the values (x2 y2 nil)
;;; for vars (x y z), with x2 > x1, y2 > y1 {this is the + part}, then
;;; we can conclude that the value for z must be greater than z1.  This
;;; is done by (assert-lb z z1 state).  
;;;
;;; Since we are doing propagation, and trying to form a SINGLE state,
;;; we only use UNAMBIGUOUS information. This is the reason for the
;;; calls to exactly-one.  There can be only one unknown (though in
;;; principle this is not necessary), and only one table entry which
;;; satisfies it.
;;;-----------------------------------------------------------------------------


(defun PROPAGATE-WRT-CORR-VALUES (con corr-values state)
  (let ((orders (mapcar #'(lambda (var val)
			    (qmag-order (variable-qmag var)
					val
					(variable--qspace var)))
			(constraint-variables con)
			corr-values)))
    (when (one-nil orders)
      ;; ensure that there is only one nil
      (let ((landmark (nth (position nil orders) corr-values))
	    ;; this is the landmark in the cvs which matches the nil.
	    ;; we use it as upper,lower,eq bound.
	    (binding
	      (exactly-one #'(lambda (qmag-relation)
			       (match-tuple orders qmag-relation 
					    (constraint-variables con)))
			   (contype-qmag-relation (constraint-type con)))))
	;; binding is either nil (no match)
	;; or a pair (var order) (where order is +,0,-).
	(propagate-tracer (first binding) (second binding) landmark)
	(when binding
	  (case (second binding)
	    (+ (assert-lb (car binding) landmark state))
	    (0 (assert-eq (car binding) landmark state))
	    (- (assert-ub (car binding) landmark state))
	    (otherwise
	      (error "Binding no good:  ~a" binding))))))))


; Derivative propagation needs to be handled specially, because it
; crosses between qmag and qdir.

;(defparameter *quasi-equilibrium-reasoning* nil)	; mode: nil=normal; t=quasi-eq.
; actual definition in qsim-constraints.lisp.

(defun DERIV-PROPAGATION (con cvs state)
  (declare (ignore cvs))
  (cond (*quasi-equilibrium-reasoning*
	 (quasi-equilibrium-deriv-propagation con state))
	(t (normal-deriv-propagation con state))))



;;;----------------------------------------------------------------------
;;; Function: (normal-deriv-propagation con state)
;;;
;;; Given: Is state necessary?
;;;
;;; Returns: NIL if no propagation, otherwise the propagated VAR.
;;;
;;; Design:  If we know the qdir, but not the deriv, the we can assert
;;; the sign of the deriv.  If we know the sign of the deriv, then we
;;; can assert the qdir.  Easy!
;;;
;;;----------------------------------------------------------------------


(defun NORMAL-DERIV-PROPAGATION (con state)

  (let* ((level (first (constraint-variables con)))
	 (qdir  (variable-qdir level))
	 (rate  (second (constraint-variables con)))
	 (sign  (qmag-order (variable-qmag rate) *zero-lmark*
			    (variable--qspace rate))))
    (cond ((eql qdir 'IGN) NIL)
	  ((and sign (null qdir))
	   (set-qdir level (case sign
			     (+ 'INC)
			     (- 'DEC)
			     (0 'STD)))
	   (propagate-tracer level (variable-qdir level))
	   level)
	  ((and qdir (null sign))
	   (assert-sign rate (case qdir
			       (inc '+)
			       (dec '-)
			       (std  0))
			state)
	   (propagate-tracer rate (variable-qmag rate))
	   rate))))


;;;----------------------------------------------------------------------
;;; Function:  (quasi-equilibrium-deriv-propagation con state)
;;;
;;; Given:   is state necessary?
;;;
;;; Returns: NIL if no propagation, otherwise the propagated VAR.
;;;
;;; Design:  In quasi-equilibrium reasoning, the rate parameter must be
;;;          <0,std>, but the level parameter is unconstrained.
;;;
;;;----------------------------------------------------------------------


(defun QUASI-EQUILIBRIUM-DERIV-PROPAGATION (con state)

  (let* ((level (first (constraint-variables con)))
	 (qdir  (variable-qdir level))
	 (rate  (second (constraint-variables con)))
	 (sign  (qmag-order (variable-qmag rate) *zero-lmark* (variable--qspace rate)))
	 (return-value nil))
    (cond ((null sign)
	   (assert-sign rate 0 state)
	   (propagate-tracer rate sign)
	   (setq return-value rate)))
    (cond ((eql qdir 'IGN) nil)
	  ((null qdir)
	   (set-qdir rate qdir)
	   (propagate-tracer rate qdir)
	   (setq return-value rate)))
    return-value))




;;;----------------------------------------------------------------------
;;; Function: propagate-MULT-signs con state
;;;
;;; Returns:  NIL if no propagation, else the VAR propagated.
;;;
;;; Design:   Similar to propagate-qmag.
;;;----------------------------------------------------------------------


(defun PROPAGATE-MULT-SIGNS (con cvs state)

  (declare (ignore cvs))
  (let ((signs (mapcar #'(lambda (var)
			   (qmag-order (variable-qmag var) *zero-lmark*
				       (variable--qspace var)))
		       (constraint-variables con))))
    (when (one-nil signs)
      (let ((binding
	      (exactly-one #'(lambda (entry)
			       (match-tuple signs (car entry)
					    (constraint-variables con)))
			   ;; this is wrong slot
			   (contype-qmag-relation (constraint-type con)))))
	(when binding
	       (assert-sign (car binding) (cadr binding) state)
	       (propagate-tracer (first binding) (second binding))
	       (car binding))))))


;;;----------------------------------------------------------------------
;;; Function: (propagate-MULT-qdirs con state)
;;;
;;; Returns:  NIL or a VAR if propagation has taken place.
;;;
;;; Design:   Similar to propagate-qmag.
;;;----------------------------------------------------------------------	       


(defun PROPAGATE-MULT-QDIRS (con cvs state)

  (declare (ignore cvs state))
  (let ((signs (mapcar #'(lambda (var)
			   (qmag-order (variable-qmag var) *zero-lmark*
				       (variable--qspace var)))
		       (constraint-variables con)))
	(qdirs (mapcar #'var-qdir (constraint-variables con)))
	(qdir-table nil)
	(binding nil))
    (cond ((member 'IGN qdirs) nil)		; don't propagate if any value is IGN.
	  ((and (one-nil qdirs)
		(setq qdir-table
		      (cadr (assoc signs
				   (contype-qmag-relation (constraint-type con))
				   :test #'equal))))
	   (setq binding
		 (exactly-one #'(lambda (tuple)
				  (match-tuple qdirs tuple
					       (constraint-variables con)))
			      qdir-table))
	   (when binding
	     (set-qdir (car binding) (cadr binding))
	     (propagate-tracer (first binding) (second binding))
	     (car binding))))))


;;; Corresponding value propagation returns the list of changed parameters.


(defun PROPAGATE-MULT-CVS (con cvs state)

  (delete nil
	  (mapcar #'(lambda (cv-tuple)
		      (propagate-MULT-cv-tuple con cv-tuple state))
		  cvs)))


(defun PROPAGATE-MULT-CV-TUPLE (con corr-values state)
  
  (declare (special ADD-relation))
  (let ((orders (mapcar #'(lambda (var cv)
			    (MULT-quotient-sign (variable-qmag var)
						cv
						(variable--qspace var)))
			(constraint-variables con)
			corr-values)))
    (when (one-nil orders)
      (let ((landmark (nth (position nil  orders) corr-values))
	    (binding (exactly-one
		       #'(lambda (entry)
			   (match-tuple orders entry
					(constraint-variables con)))
		       ADD-relation))		; interpreted as >1,=1,<1
	    )
	(when (qmag-point-p landmark)		;CVs may contain intervals
						;Pierre Fouche 03/12/90
	  (propagate-tracer (first binding) (second binding) landmark)
	  (cond
	    ((null binding) nil)
	    ((landmark-lt *zero-lmark* landmark (variable--qspace (car binding)))
	     (case (second binding)
	       (+ (assert-lb (car binding) landmark state))
	       (0 (assert-eq (car binding) landmark state))
	       (- (assert-ub (car binding) landmark state))
	       (otherwise (error "Binding no good:  ~a" binding))))
	    ((landmark-lt landmark *zero-lmark* (variable--qspace (car binding)))
	     (case (second binding)
	       (+ (assert-ub (car binding) landmark state))
	       (0 (assert-eq (car binding) landmark state))
	       (- (assert-lb (car binding) landmark state))
	       (otherwise (error "Binding no good:  ~a" binding))))
	    (t (error "Binding no good:  ~a" binding))))))))



;;;----------------------------------------------------------------------
;;; Function: assert-EQ/LB/UB
;;;
;;; Given: VAR, LM, STATE.
;;;
;;; Returns: VAR
;;;
;;; Does:  The ASSERT functions take a LM which is a new val, lower, or
;;; upper bound for the VAR in this STATE.  They do a validity check,
;;; and then use the LM to restrict the value of VAR in the appropriate
;;; fashion.  Instead of a landmark, the qmag of VAR may have the value
;;; NIL, in which case it is (so far) unconstrained.
;;;
;;;----------------------------------------------------------------------
;   Problem:  the use of (L nil) to represent (L inf] is turning into a
;   nuisance of special-case checks.


(defun ASSERT-EQ (var landmark state)

  (let* ((qval      (variable--qval var))
	 (old-val   (qmag qval)))
    (cond ((null old-val)
	   (setf (qmag qval) landmark))
	  ((qmag-point-p old-val)
	   (cond ((equal landmark old-val) nil)
		 ((error "Contradiction ~a(~a):  ~a -> ~a."
			 var state old-val landmark))))
	  ((and (or (null (car old-val))
		    (landmark-lt (car old-val) landmark
				 (variable--qspace var)))
		(or (null (cadr old-val))
		    (landmark-lt landmark (cadr old-val) (variable--qspace var))))
	   (setf (qmag qval) landmark))
	  (t (error "Contradiction ~a(~a):  ~a -> ~a."
		    var state old-val landmark))))
  var)


(defun ASSERT-LB (var lb state)

  (let* ((qval      (variable--qval var))
	 (old-value (qmag qval))
	 (qspace    (variable--qspace var)))
    (cond ((null old-value)
	   (setf (qmag qval) (list lb nil)))
	  ((qmag-point-p old-value)
	   (unless (landmark-lt lb old-value qspace)
	     (error "Contradiction ~a(~a):  ~a -> ~a."
		    var state old-value lb)))
	  ((landmark-le lb (car old-value) qspace)
	   NIL)
	  ((landmark-le (cadr old-value) lb qspace)
	   (error "Contradiction ~a(~a):  ~a -> ~a."
		  var state old-value lb))
	  (t (setf (qmag qval) (list lb (cadr old-value))))))
  var)


(defun ASSERT-UB (var ub state)

  (let* ((qval      (variable--qval var))
	 (old-value (qmag qval))
	 (qspace    (variable--qspace var)))
    (cond ((null old-value)
	   (setf (qmag qval) (list nil ub)))
	  ((qmag-point-p old-value)
	   (unless (landmark-lt old-value ub qspace) 
	     (error "Contradiction ~a(~a):  ~a -> ~a."
		    var state old-value ub)))
	  ((landmark-le (cadr old-value) ub qspace)
	   NIL)
	  ((landmark-le ub (car old-value) qspace)
	   (error "Contradiction ~a(~a):  ~a -> ~a."
		  var state old-value ub))
	  (t (setf (qmag qval) (list (car old-value) ub)))))
  var)


(defun ASSERT-SIGN (var sign state)

  (ecase sign
    (+ (assert-lb var *zero-lmark* state))
    (0 (assert-eq var *zero-lmark* state))
    (- (assert-ub var *zero-lmark* state))))



;;;----------------------------------------------------------------------
;;; Function: (MATCH-TUPLE values tuple params)
;;;
;;; Given:  -- three equal length lists.  One of the elements in values
;;; may be NIL.
;;;
;;; Returns: a binding (param element-of-tuple), or NIL.
;;;
;;; Design: Each element of values must match each element of tuples,
;;; except in the case that one element of values is NIL.  If so, then
;;; the binding to the corresponding element of params is returned.  A
;;; typical call migh be (match-tuples '(+ 0 +) '(+ 0 nil) '(x y z)),
;;; which would return (z +).  It should really return a list of
;;; bindings, one for each nil, but that would result in extra consing,
;;; and the test is already made in current usage.
;;;----------------------------------------------------------------------


(defun MATCH-TUPLE (values tuple params)

  (do ((v values (cdr v))
       (p params (cdr p))
       (tp tuple (cdr tp))
       (binding nil))
      ((null v) binding)
    (cond ((null (car v))
	   (if binding
	       (error "There are two missing values! ~a and ~a."
		      binding (list (car p) (car tp)))
	       (setq binding (list (car p) (car tp)))))
	  ((equal (car v) (car tp)))
	  (t (return nil)))))


;;;----------------------------------------------------------------------
;;; Function: set-qdir var qdir
;;;
;;; Returns: var
;;;
;;; Does: will add info to an empty slot.  
;;;----------------------------------------------------------------------
; REPLACE-QMAG  => (setf (variable-qmag v) val)
      

(defun SET-QDIR (var nqdir)

  (if nqdir
      (let* ((qval (variable--qval var))
	     (qdir (qdir qval)))
	(if qdir
	    (or (eql qdir nqdir)
		(error "Can't reset ~a from ~a to ~a."
		       var qdir nqdir))
	    (setf (qdir qval) nqdir))))
  var)



;;;----------------------------------------------------------------------
;;; Function:  (complete-qval-p  var)   (complete-qmag-p var)
;;;
;;; Returns:   T if the qval/qmag of VAR is complete.
;;;
;;; A qval is complete if the qdir is not null and its qmag is complete.
;;;
;;; A qmag is complete if it is not null, does not contain a nil, and if
;;; it's landmarks are adjacent.  I.e. define a minimal interval.
;;; This criteria is somewhat hidden, oh well...
;;;----------------------------------------------------------------------


(defun COMPLETE-QVAL-P (var)

  (and (variable-qdir var)
       (complete-qmag-p var)))


(defun COMPLETE-QMAG-P (var)

  (let ((qval (variable--qval var)))
    (cond ((null qval) nil)
	  (t (let ((qmag (qmag qval)))
	       (cond ((null qmag) nil)
		     ((qmag-point-p qmag) t)
		     ((member nil qmag) nil)
		     (t (adjacent-p qmag (variable--qspace var)))))))))


;;;======================================================================
;;; M Constraint Propagator                         (AF, JR: 4-13-91)
;;;
;;;
;;; This design is similar to that of the check function.  Please see
;;; M-check in constraints.lisp.
;;;
;;; The M constraint has a slot PARTIALS, which is a tuple each of whose
;;; elements are +, 0, or - depending on the effect that a variable has
;;; on the derivative of the dependent variable.  A trick is used
;;; whereby the dependent variable is included in the list by making the
;;; last element in the list a -.  E.g. partial((M + +)) = (+ + -)
;;;
;;; Much of the code uses plus-minus, which determines the number of
;;; variables with a positive and with a negative effect.  A positive
;;; effect is achieved by either a + that is inc or a - that is dec.
;;;

;;; Todo: make use of the infinite values.

(defun M-propagator (con cvs state)
  "Attempt to propagate some values through the M constraint, CON.  CVS
  is a list of corresponding values (landmark structures).
  Return a list of the variables (variable structures) whose values have
  been changed.
  Side effect: modify the values of those variables in STATE."
  (union (propagate-M-qdir con)
	 (propagate-M-cvs con cvs state)))

;(propagate-M-infinite-values con state)
  
(defun propagate-M-qdir (con &aux orders)
  "Try to propagate qdirs through the constraint CON.
   Return NIL or a list containing the single variable whose qdir
   has been determined.  Side effect: set the variable's qdir in STATE."
  (setq orders (mapcar #'(lambda (var) (qdir-order (var-qdir var)))
		       (constraint-variables con)))
  ;; skip if there is an ignore qdirs *.
  (unless (member '* orders)
    (multiple-value-bind (index plus minus)
	(plus-minus orders (constraint-partials con))
      (when index
	(let ((qdir (if (zerop plus)
			(if (zerop minus) 'std 'inc)
			(if (zerop minus) 'dec nil)))
	      (var (nth index (constraint-variables con))))
	  (when qdir
	    (when (eq (nth index (constraint-partials con)) '-)
	      ;; Need to take the inverse.
	      (setq qdir (qdir-inverse qdir)))
	    (propagate-tracer var qdir)
	    (set-qdir var qdir)
	    (when var (list var))))))))

(defun propagate-M-cvs (con cvs state &aux var)
  "Try to propagate through the constraint CON using the corresponding
values, CVS.  Return NIL or a list containing single unknown variable
whose value was restricted.  Side effect: restrict the qvalue of that
variable in the STATE." 
  (dolist (cv cvs)
    (setq var
	  (or (propagate-m-corr-values con cv state)
	      var)))
  (when var (list var)))
  
(defun propagate-M-corr-values (con corr-values state)
  "Try to propagate through the constraint CON using a single tuple of
corresponding values CORR-VALUES.  Return NIL or the Variable whose
value has been restricted via propagation.  Side effect: restrict the
qvalue of the variable in STATE."
  (let ((orders (mapcar #'(lambda (var val)
			    (qmag-order (variable-qmag var)
					val
					(variable--qspace var)))
			(constraint-variables con)
			corr-values)))
    (multiple-value-bind (index plus minus)
	(plus-minus orders (constraint-partials con))
      (when index
	(let ((order
		(if (zerop plus)
		    (if (zerop minus) 0 '+)
		    (if (zerop minus) '- nil))))
	  (when order
	    (when (eq (nth index (constraint-partials con)) '-)
	      (setq order (opp-sign order)))
	    (let ((var (nth index (constraint-variables con)))
		  (mag (nth index corr-values)))
	      (propagate-tracer var order mag)
	      (ecase order
		(+ (assert-lb var mag state))
		(- (assert-ub var mag state))
		(0 (assert-eq var mag state)))
	      var)))))))



;(defun propagate-M-Infinite-values (con state)
;  (let ((orders (mapcar #'(lambda (var)
;			    (sign-wrt-infinity (variable-qmag var)))
;			(constraint-variables con))))
;    (multiple-value-bind (index plus minus)
;	(plus-minus orders (constraint-partials con))
;      (when index
;	(let ((order
;		(if (zerop plus)
;		    (if (zerop minus) 0 '+)
;		    (if (zerop minus) '- nil))))
;	  (when (eq (nth index orders) '-)
;	    (setq order (opp-sign order)))
;	  (when order
;	    (let ((var (nth index (constraint-variables con))))
;	      (ecase order
;		(+ (assert-eq var *inf-lmark*  state))
;		(- (assert-eq var *minf-lmark* state))
;		(0 (assert-lb var *minf-lmark* state)
;		   (assert-ub var *inf-lmark*  state)))
;	      var)))))))

(defun plus-minus (orders partials)
  "ORDERS and PARTIALS are lists of equal length containing +,-,0, or
nil.  Returns NIL if there is more than one NIL ORDER, otherwise
several values:  the Index of the single unknown Order, Number of
Pluses, Number of Minuses." 
  (let ((Index-of-unknown nil)
	(Pluses 0)
	(Minuses 0)
	(i 0))
    (mapc #'(lambda (o p)
	      (cond ((eql P 0))
		    ((eql O 0))
		    ((eql O nil)
		     (if Index-of-unknown
			 ;; more than one unknown, so exit
			 (return-from plus-minus nil)
			 ;; else save the index (doubles as test)
			 (setq Index-of-unknown i)))
		    ((eq P O) (incf Pluses))
		    (t (incf Minuses)))
	      (incf i))
	  orders partials)
    (values  Index-of-unknown Pluses Minuses)))

(defun qdir-order (qdir)
  (ecase qdir
    (inc   '+)
    (std    0)
    (dec   '-)
    ((nil)  nil)
    (ign   '*)))

(defun qdir-inverse (qdir)
  (ecase qdir
    (std 'std)
    (inc 'dec)
    (dec 'inc)
    (ign 'ign)
    ((nil) 'nil)))

;;;======================================================================
;;;
;;; Propagate the CORRESPONDENCE constraint.
;;;
;;; Author: Adam Farquhar
;;; Date:  20 March 1991
;;; 
;;; Purpose: to provide a very weak correspondence constraint.  See the
;;; definition of the correspondence constraint in constraints.lisp.
;;;

(defun propagate-correspondence-constraint (con cvals state)
  "If there is a NIL qvalue and a cval which matches it, then we have
  one."
  (let ((vals (mapcar #'variable-qmag (constraint-variables con))))
    (when (one-nil vals)
      (dolist (cv cvals)
	(let ((mismatch (mismatched_ (constraint-variables con)
				      vals cv)))
	  (when (variable-p mismatch)
	    (assert-eq mismatch
		       (nth (position mismatch (constraint-variables con))
			    cv)
		       state)
	    (return-from propagate-correspondence-constraint
	      (list mismatch))))))))


;; the below is almost identical to mismatched, but the tuple is of
;; qmags, and one of them may be nil.
;;
(defun mismatched_ (vars tuple cv-tuple &aux (mismatch nil))
  "Tuple1 and Tuple2 are lists of qmags for the VARS.  Return NIL if the
tuples match, var if there is ONE mismatch or T if there is more than
one mismatch."
  (map nil #'(lambda (var mag1 mag2)
	       (let ((order (qmag-order  mag1 mag2
					 (variable--qspace var))))
		 (if (or (null mag1) (member order '(+ -)))
		     ;; a mismatch
		     (if mismatch
			 (RETURN-FROM mismatched_ t)
			 (setq mismatch var)))))
       vars tuple cv-tuple)
  mismatch)


; Additions needed:
;   (1)  Add S+/S- and U+/U-.
;   (2)  Special checking for inf and minf.
 




