;;; -*- Mode:Common-Lisp; Package: QSIM; Base:10 -*-
;;;  $Id: prolog.lisp,v 1.5 91/09/05 12:23:21 clancy Exp $

(in-package :qsim) ;changed DJC

;;;*****************************************************************************
;;;	     A   S I M P L E   P R O L O G   I N T E R P R E T E R
;;;
;;;  Pierre Fouche (fouche@cs.utexas.edu)
;;;  August 1990
;;;*****************************************************************************
;
; Three fixes  (BJK:  10-9-90)
;  - move to package QSIM;  PROLOG may be a conflict.
;  - rename MATCH to PF-MATCH, to avoid conflict with a MATCH in QSIM.
;  - change calls to GENSYM to take strings, rather than atoms, as args.

;;;=============================================================================
;;; Pattern matching first.
;;;
;;; A term (or a pattern) can be any Lisp object.
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; A VARIABLE is a symbol beginning with a ?
;;;-----------------------------------------------------------------------------

(defun var-p (term)
  (and (symbolp term)
       (eql (aref (symbol-name term) 0) #\?)))

(defmacro make-var ()
  `(gensym "?"))


;;;-----------------------------------------------------------------------------
;;; A Prolog CONSTANT is a Lisp atom (not a cons) which is not a variable.
;;; Here integers, structures, etc... are Prolog constants.
;;;-----------------------------------------------------------------------------

(defun constant-p (term)
  (and (atom term)
       (not (var-p term))))


;;;-----------------------------------------------------------------------------
;;; A Prolog ATOM is a Lisp symbol which is not a variable.
;;;-----------------------------------------------------------------------------

(defun atom-p (term)
  (and (symbolp term)
       (not (eql (aref (symbol-name term) 0) #\?))))


;;;-----------------------------------------------------------------------------
;;; A FUNCTOR is a list that begins with a Prolog atom.
;;; Here, atoms are not considered as functors of arity zero.
;;;-----------------------------------------------------------------------------

(defun functor-p (term)
  (and (consp term)
       (atom-p (car term))))


;;;-----------------------------------------------------------------------------
;;; A BINDING is a pair (<var> . <value>).
;;; <value> can be another variable as well.
;;; ADD-BINDING pushes a new binding onto a list of bindings.
;;;-----------------------------------------------------------------------------

(defmacro add-binding (var term bindings)
  `(cons (cons ,var ,term) ,bindings))


;;;-----------------------------------------------------------------------------
;;; GET-BINDING return a pair (<var> . <value>) or nil if no binding.
;;;-----------------------------------------------------------------------------

(defmacro get-binding (var bindings)
  `(assoc ,var ,bindings))


;;;-----------------------------------------------------------------------------
;;; VARIABLES-IN-TERM returns the list of the variables in a clause.
;;;-----------------------------------------------------------------------------

(defun variables-in-term (clause)
  (let ((*variables* nil))
    ;; *variables* stores variables
    (declare (special *variables*))
    (variables-in-term-1 clause)
    *variables*))

(defun variables-in-term-1 (clause)
  (declare (special *variables*))
  (cond ((var-p clause) (pushnew clause *variables*))
	((atom clause))
	(t (variables-in-term-1 (car clause))
	   ;; recursive calls
	   (variables-in-term-1 (cdr clause)))))


;;;-----------------------------------------------------------------------------
;;; (MATCH term1 term2) is the top-level function. It returns 'fail
;;; if matching fails, a list of bindings if it succeeds.
;;; For instance (match '(a ?x (b ?y)) '(?z ?y (b ?t))) returns
;;; ((?y . ?t) (?x . ?y) (?z . a))
;;;-----------------------------------------------------------------------------
;;;   Renamed PF-MATCH to avoid conflict with another MATCH in the QSIM package.
;;;     - BJK (10-9-90)

(defun pf-match (term1 term2)
  (match-with-bindings term1 term2 nil))

(defun match-with-bindings (term1 term2 bindings)
  (cond ((var-p term1)
	 ;; check if terms are variables first
	 (variable-match term1 term2 bindings))
	((var-p term2)
	 (variable-match term2 term1 bindings))
	((atom term1)
	 (if (eql term1 term2) bindings 'fail))
	((atom term2) 'fail)
	;; general case: both terms are lists 
	(t (let ((binding (match-with-bindings (car term1) (car term2) bindings)))
	     (cond ((eq binding 'fail) 'fail)
		   ;; second call, using bindings from the first call
		   (t (match-with-bindings (cdr term1) (cdr term2) binding)))))))


;;;-----------------------------------------------------------------------------
;;; VARIABLE-MATCH is called when the first term is a variable.
;;; This function eliminates pathological cases, like ?x against f(?x), if
;;; *occur-check* is T.
;;;-----------------------------------------------------------------------------

(defparameter *occur-check* nil)

(defun variable-match (var term bindings)
  (if (eql var term)
      ;;variables eql -> ok but no new bindings
      bindings			
      (let ((var-binding (get-binding var bindings)))
	(cond (var-binding
	       ;; if the variable is bound, then recursive call.
	       (match-with-bindings (cdr var-binding) term bindings))
	      ((and (var-p term) (eql (cdr (get-binding term bindings)) var))
	       ;; term is a variable bound to var. Ok but no new bindings
	       bindings)
	      ((and *occur-check* (occur-check var term bindings)) 'fail)
	      ;; If occur check fails, add new bindings
	      (t (add-binding var term bindings))))))


;;;-----------------------------------------------------------------------------
;;; OCCUR-CHECK performs occur check. It checks if a variable appears inside a
;;; term. Rather costly in term of computation time !
;;;-----------------------------------------------------------------------------

(defun occur-check (var term bindings)
  (cond ((var-p term)
	 (or (eql var term)
	     (occur-check var
			  (cdr (get-binding term bindings))
			  bindings)))
	((atom term) nil)
	(t (or (occur-check var (car term) bindings)
	       (occur-check var (cdr term) bindings)))))


;;;=============================================================================
;;; CLAUSE AND PREDICATE REPRESENTATION
;;;
;;; A DATA BASE is a collection of CLAUSEs.
;;; A CLAUSE is made of a HEAD and a BODY.
;;; The HEAD of a clause is a functor, its BODY a (possibly empty) list of
;;; terms.
;;;
;;; A clause with no body is a FACT, otherwise it's a RULE.
;;; Terms in the body of a rule are GOALS or SUB-GOALS.
;;; A rule with only one sub-goal is an ITERATIVE RULE.

;;; A PREDICATE is defined by a set of clauses whose names are all the same.
;;; Predicates are defined by the macro defpredicate. Its syntax is the
;;; following:

;;; (defpredicate <predicate-name> [<doc-string>] <clause>*)
;;; <predicate-name> :== <name> | (<name> <data-base>)
;;; <clause> :== (<functor> [<-] <term>*)
;;; <functor> :== (<name> <term>) 

;;; All the clauses in a defpredicate must have the same name.

;;; Facts and rules are stored in a data-base. A data-base is a Lisp symbol, which
;;; stores predicates in its property :predicates. The data base is indexed by the
;;; predicates, using the symbol table. A predicate (Lisp symbol) has a property
;;; <data-base> which stores an instance of the structure 'predicate. That 
;;; strutures stores various informations about a predicate, such as its lists of
;;; clauses.
;;; This allows you to have several data base at the same time.
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; The structure PREDICATE stores useful infromation about a predicate.
;;;-----------------------------------------------------------------------------

(defstruct (predicate (:print-function predicate-printer))
  name						;predicate's symbol
  (documentation "")				;doc string
  clauses					;list of clauses
  (variables nil)				;list of lists of variables
  (cut-p nil)					;flag: T if a cut appears
  (traced-p nil))				;flag: T if traced

(defun predicate-printer (predicate stream ignore)
  (declare (ignore ignore))  ;;  added DJC porting to the Sun
  (format stream "!~a" (predicate-name predicate)))


;;;-----------------------------------------------------------------------------
;;; Predefined predicates
;;;-----------------------------------------------------------------------------

(defparameter
  *predefined-predicates*
  `((cut  . ,(make-predicate
	       :name 'cut
	       :documentation "Prolog cut"
	       ;; stop-this is catched by the interpreter
	       :clauses '(((cut ?goal))
			  ((cut ?goal) (stop-this)))
	       :variables '(() (?goal))))
    (fail . ,(make-predicate
	       :name 'fail
	       :documentation "Always fails"
	       :clauses nil))))


;;;-----------------------------------------------------------------------------
;;; *DEFAULT-DB* holds a list of predicate names in its property :predicates
;;;-----------------------------------------------------------------------------

(defparameter *default-db* 'my-db)


;;;-----------------------------------------------------------------------------
;;; RESET-DB clears the property :predicates of the db and the property 'db
;;; of its predicates.
;;; It then initializes the db with the current list of *predefined-predicates*
;;;-----------------------------------------------------------------------------

(defun reset-db (&optional (db *default-db*))
  (let ((predicates (get db :predicates)))
    (dolist (predicate predicates)
      (remprop predicate db))
    (setf (get db :predicates) (mapcar #'car *predefined-predicates*))
    (dolist (predicate-pair *predefined-predicates*)
      (setf (get (car predicate-pair) db)
	    (cdr predicate-pair)))))

(reset-db)


;;;-----------------------------------------------------------------------------
;;; DATA-BASE-P checks whether the property :predicates is present
;;;-----------------------------------------------------------------------------

(defun data-base-p (&optional (db *default-db*))
  (get db :predicates))


;;;-----------------------------------------------------------------------------
;;; Macros to access properties of a predicate.
;;;-----------------------------------------------------------------------------

(defmacro get-clauses (predicate &optional (db '*default-db*))
  `(predicate-clauses (get ,predicate ,db)))

(defmacro get-variables (predicate &optional (db '*default-db*))
  `(predicate-variables (get ,predicate ,db)))

(defmacro cut-p (predicate &optional (db '*default-db*))
  `(predicate-cut-p (get ,predicate ,db)))

(defmacro traced-p (predicate &optional (db '*default-db*))
  `(predicate-traced-p (get ,predicate ,db)))

(defmacro find-predicate (predicate &optional (db '*default-db*))
  `(get ,predicate ,db))


;;;-----------------------------------------------------------------------------
;;; Macros to access clauses
;;;-----------------------------------------------------------------------------

(defmacro head (clause)
  `(car ,clause))

(defmacro body (clause)
  `(cdr ,clause))

(defmacro predicate (functor)
  `(car ,functor))


;;;-----------------------------------------------------------------------------
;;; Functions to test the syntax of a predicate
;;;-----------------------------------------------------------------------------

;;;-----------------------------------------------------------------------------
;;; A predicate name is either a predicate (a symbol) or a pair
;;; (<predicate> <data-base>).
;;;
;;; EXPLODE-PREDICATE-NAME checks syntax and returns two values: the
;;; predicate and the symbol to be evaluated as a data-base.
;;;-----------------------------------------------------------------------------

(defun explode-predicate-name (predicate-name)
  (let (predicate db)
    (if (consp predicate-name)
	(setq predicate (first predicate-name)
	      db `',(second predicate-name))
	(setq predicate predicate-name
	      db '*default-db*))
    (unless (symbolp predicate)
      (error "Bad name for a predicate: ~a" predicate))
    (unless (and (symbolp (eval db)) (data-base-p (eval db)))
      (error "Bad data base: ~a" db))
    (values predicate db)))


;;;-----------------------------------------------------------------------------
;;; ANALYZE-CLAUSE checks the syntax of a clause and process it so it can be
;;; stored in the slot clauses of the structure predicate.
;;;-----------------------------------------------------------------------------

(defun analyze-clause (clause predicate)
  (unless (consp clause)
    (error "A clause must be a list: ~a" clause))
  (let* ((head (head clause))
	 (body (body clause))
	 (predicate-2
	   (if (functor-p head) (predicate head)
	       (error "The head of a clause must be a functor: ~a" head))))
    (unless (eq predicate-2 predicate)
      (error "The clause: ~a should not appear in the definition of: ~a"
	     clause predicate))
    (cons head (analyze-body body))))

(defun analyze-body (body)
  (if (eq (car body) '<-) (setq body (cdr body)))
  (mapcar #'(lambda (goal)
	      (if (eq goal '!) (setq goal '(cut ?goal))
		  (if (constant-p goal)
		      (error "A subgoal cannot be a constant: ~a" goal)
		      goal)))
	  body))


;;;-----------------------------------------------------------------------------
;;; Predicates are defined using the macro DEFPREDICATE.
;;;-----------------------------------------------------------------------------

(defmacro defpredicate (predicate-name &rest doc-n-clauses)
  (multiple-value-bind (name db)
      (explode-predicate-name predicate-name)
    `(build-predicate ',name ',doc-n-clauses ,db)))


;;;-----------------------------------------------------------------------------
;;; BUILD-PREDICATE is the function that creates predicate instances, checking
;;; the correct syntax first.
;;;-----------------------------------------------------------------------------

(defmacro cut-in-clause-p (clause)
  `(member '(cut ?goal) ,clause :test #'equal)) 


(defun build-predicate (name doc-n-clauses db)
  (when (member name *predefined-predicates* :key #'car)
    (error "It's a *very* bad idea to redefine the predicate ~a !" name))
  (let ((predicate (make-predicate :name name))
	(doc (car doc-n-clauses))
	(clauses doc-n-clauses))
    (when (stringp doc) (setf (predicate-documentation predicate) doc
			       clauses (cdr clauses)))
    (setf (predicate-clauses predicate)
	  (mapcar #'(lambda (clause) (analyze-clause clause name)) clauses))
    (setf (predicate-variables predicate)
	  (mapcar #'variables-in-term clauses))
    (dolist (clause (predicate-clauses predicate))
      (when (cut-in-clause-p (body clause))
	(setf (predicate-cut-p predicate) t) (return)))
    (pushnew name (get db :predicates))
    (setf (get name db) predicate)))


;;;-----------------------------------------------------------------------------
;;; UNDEFPREDICATE removes a predicate definition from the data base.
;;;-----------------------------------------------------------------------------

(defmacro undefpredicate (predicate-name)
  (multiple-value-bind (name db)
      (explode-predicate-name predicate-name)
    `(progn
       (when (member ',name *predefined-predicates* :key #'car)
	 (error "It's a *very* bad idea to undefine the predicate ~a !" ',name))
       (setf (get ,db :predicates)
	     (delete ',name (get ,db :predicates)))
       (remprop ',name ,db)
       ',name)))


;;;-----------------------------------------------------------------------------
;;; TRACE-PREDICATE is a macro that sets the traced-p flags of its arguments.
;;; If no arguments are given, all the predicates are traced.
;;;-----------------------------------------------------------------------------

(defun set-predicate-trace-flags (predicates value)
  (mapcan #'(lambda (name)
	      (unless (eq value (traced-p name))
		(setf (traced-p name) value)
		(list name)))
	   predicates))

(defparameter *trace-execution* nil
  "Internal use only; set by trace-predicate")

(defmacro trace-predicate (&rest predicates)
  `(and (setf *trace-execution* t)
	(if ',predicates
	    (set-predicate-trace-flags ',predicates t)
	    (set-predicate-trace-flags (set-difference (get *default-db* :predicates)
						       (mapcar #'car *predefined-predicates*))
				       t))))


;;;-----------------------------------------------------------------------------
;;; UNTRACE-PREDICATE unsets the traced-p flags of its arguments.
;;; If no arguments are given, all the predicates are untraced.
;;;-----------------------------------------------------------------------------

(defmacro untrace-predicate (&rest predicates)
  `(if (null ',predicates)
       (or (setf *trace-execution* nil)
	   (set-predicate-trace-flags (get *default-db* :predicates) nil))
       (set-predicate-trace-flags ',predicates nil)))


;;;=============================================================================
;;; UTILITITIES FOR THE INTERPRETER
;;;
;;; Warning: the variable *default-db* must be bound to the correct data base.
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; RENAME-VARIABLES renames all the variables in a clause. It returns a copy
;;; of clause with new variables.
;;;-----------------------------------------------------------------------------

(defun rename-variables (clause &optional (*bindings* nil))
  ;; *bindings* stores (old-var . new-var) pairs
  (declare (special *bindings*))
  (values (rename-variables-with-bindings clause) *bindings*))

(defun rename-variables-with-bindings (clause)
  (declare (special *bindings*))
  (cond ((var-p clause)
	 ;; if the clause is a variable...
	 (let ((var-binding (get-binding clause *bindings*)))
	   (cond (var-binding
		  ;; and the variable is already renamed...
		  (cdr var-binding))		; ... return the new name.
		 (t (let ((new-var (make-var)))
		      ;; otherwise, rename it.
		      (setq *bindings* (add-binding clause new-var *bindings*))
		      new-var)))))
	((atom clause) clause)
	(t (cons (rename-variables-with-bindings (car clause))
		 ;; recursive calls
		 (rename-variables-with-bindings (cdr clause))))))


;;;-----------------------------------------------------------------------------
;;; SUBSTITUTE-VARS returns a new clause with variables substituted with their
;;; values from bindings. The clause is copied.
;;; NSUBSTITUTE-VARS is the destructive version. The clause is not copied.
;;; Bindings is altered as well.
;;;-----------------------------------------------------------------------------

(defun substitute-vars (clause bindings)
  (cond ((var-p clause)
	 (let ((binding (get-binding clause bindings)))
	   (cond (binding (substitute-vars (cdr binding) bindings))
		 (t clause))))
	((atom clause) clause)
	(t (cons (substitute-vars (car clause) bindings)
		 (substitute-vars (cdr clause) bindings)))))

;(defun nsubstitute-vars (clause bindings)
;  (cond ((var-p clause)
;	 (let ((binding (get-binding clause bindings)))
;	   (cond (binding (nsubstitute-vars (cdr binding) bindings))
;		 (t clause))))
;	((atom clause) clause)
;	((var-p (car clause))
;	 (let ((binding (get-binding (car clause) bindings)))
;	   (when binding
;	     (setf (car clause) 
;		   (nsubstitute-vars (cdr binding) bindings)))
;	   (nsubstitute-vars (cdr clause) bindings)
;	   clause))
;	(t (nsubstitute-vars (car clause) bindings)
;	   (nsubstitute-vars (cdr clause) bindings)
;	   clause)))

(defun nsubstitute-vars (clause bindings)
  (when (consp clause) 
    (cond ((var-p (car clause))
	   (let ((binding (get-binding (car clause) bindings)))
	     (when binding (setf (car clause) (cdr binding)))))
	  ((consp (car clause)) (nsubstitute-vars (car clause) bindings)))
    (cond ((var-p (cdr clause))
	   (let ((binding (get-binding (cdr clause) bindings)))
	     (when binding (setf (cdr clause) (cdr binding)))))
	  ((consp (cdr clause)) (nsubstitute-vars (cdr clause) bindings)))))

(defun smart-substitute-vars (goals bindings)
  (labels ((find-variables (goal bindings)
	     (cond ((var-p goal) (member goal bindings :key #'car))
		   ((atom goal) nil)
		   (t (or (find-variables (car goal) bindings)
			  (find-variables (cdr goal) bindings))))))
    (mapcar #'(lambda (goal)
	      (if (find-variables goal bindings)
		  (substitute-vars goal bindings)
		  goal))
	    goals)))


;;;-----------------------------------------------------------------------------
;;; NRENAME-VARIABLES substitute the variables in clause with new variables. The
;;; initial clause is not copied.
;;;-----------------------------------------------------------------------------

(defun nrename-variables (clause)
  (let* ((vars (get-variables clause))
	 (old-new (mapcar #'(lambda (old) (cons old (make-var))) vars)))
    (nsubstitute-vars clause old-new)))


;;;-----------------------------------------------------------------------------
;;; FILTER-BINDINGS returns the bindings of variables from the binding list.
;;; It also simplifies the bindings so that only elementary substitutions
;;; remain.
;;;-----------------------------------------------------------------------------

(defun filter-bindings (bindings variables)
  (mapcan #'(lambda (variable)
	      (let* ((binding (get-binding variable bindings))
		     (value (cdr binding)))
		(when binding
		  (list (if (var-p value)
			    (cons variable
				  (substitute-vars (cdr binding) bindings))
			    binding)))))
	  variables))


;;;-----------------------------------------------------------------------------
;;; APPEND-BINDINGS appends the list of local bindings to all the solutions,
;;; destructively.
;;;-----------------------------------------------------------------------------

(defun append-bindings (local-bindings solutions)
  (when solutions
    (setf (car solutions) (nconc (copy-alist local-bindings) (car solutions)))
    (append-bindings local-bindings (cdr solutions))
    solutions))


;;;=============================================================================
;;; THE INTERPRETER
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; ? is the macro that handles the queries.
;;; It returns a list of initial goals, with their variables substituted with
;;; their bindings.
;;;-----------------------------------------------------------------------------

(defparameter *goals* nil)

(defmacro ? (&rest goals)
  `(let ((new-goals (analyze-body ',goals)))
     (setq *goals* ',goals)
     ;;(solve ',goals '() 0)))
     (mapcar #'(lambda (bindings)
		 (substitute-vars *goals* bindings))
	     (solve new-goals '() 0))))

;(defmacro ? (&rest goals)
;  (let ((vars (variables-in-term goals)))
;    `(let (solutions)
;     (defpredicate query
;       ((query .,vars) <- ,.goals))
;     (setq solutions (solve '((query .,vars)) '() 0))
;     (mapcar #'(lambda (bindings)
;		 (substitute-vars ',goals bindings))
;	     solutions))))

;(? (append ?x ?y (a)))


;;;-----------------------------------------------------------------------------
;;; (SOLVE goals bindings level) is the main function that
;;; handles the list of goals.
;;;
;;; goals is the current list of goals to satisfy; bindings are bindings
;;; inherited from previous goals; level is used in tracing predicates
;;; (for indentation).
;;;
;;; It takes the first goal in the list, finds clauses that may match with that
;;; goal, renames variables in those clauses, and calls REDUCE-GOAL on the
;;; current goal and each renamed clause.
;;;
;;; The second value returned by reduce-goal is a flag to test if a cut has been
;;; called twice. In that case, all the other applicable clauses are not used.
;;; This flag remains set until we come back to the initial clause where the
;;; cut was in. 
;;;
;;; Solve returns a list of lists of bindings, or nil.
;;;-----------------------------------------------------------------------------

;;; Modified by P. Fouche.  Change inserted by DJC 09/14/91

(defun solve (goals bindings level)
  (let* ((first-goal  (car goals))
         (other-goals (cdr goals))
         (predicate (predicate first-goal))
         clauses cut-p cut-index traced-p)
    (when (find-predicate predicate)
      (setf clauses (get-clauses predicate)
            cut-p (cut-p predicate)
            cut-index (when cut-p (gensym "cut"))
            traced-p (and *trace-execution* (traced-p predicate)))

      (when traced-p
        (incf level)
        (trace-goal first-goal other-goals bindings level))
      (loop
          for clause in clauses
          for new-clause = (rename-variables clause (when cut-p `((?goal . ,cut-index))))
          with (sol stop-p)
          do (multiple-value-setq (sol stop-p)
	       (reduce-goal first-goal new-clause other-goals bindings level traced-p))
          nconc sol into solutions
          until stop-p
          finally (progn
                    (when traced-p (trace-solutions first-goal solutions level))
                    (return (values solutions (if (eq cut-index stop-p) nil stop-p))))))))

;; Version before P. Fouche's change  08/14/91
;(defun solve (goals bindings level)
;  (let* ((first-goal  (car goals))
;	 (other-goals (cdr goals))
;	 (predicate (predicate first-goal))
;	 (clauses (get-clauses predicate))
;	 (cut-p (cut-p predicate))
;	 (cut-index (when cut-p (gensym "cut")))
;	 (traced-p (and *trace-execution* (traced-p predicate)))
;	 )
;    (when traced-p
;      (incf level)
;      (trace-goal first-goal other-goals bindings level))
;    (loop
;      for clause in clauses
;      for new-clause = (rename-variables clause (when cut-p `((?goal . ,cut-index))))
;      with (sol stop-p)
;      do (multiple-value-setq (sol stop-p)
;	   (reduce-goal first-goal new-clause other-goals bindings level traced-p))
;      nconc sol into solutions
;      until stop-p 
;      finally (progn
;		(when traced-p (trace-solutions first-goal solutions level))
;		(return (values solutions (if (eq cut-index stop-p) nil stop-p)))))))


;;;-----------------------------------------------------------------------------
;;; REDUCE-GOAL tries to unify the head of a clause and a goal.
;;;
;;; If matching fails, it returns nil.
;;; If matching succeeds,
;;;   + if the clause corresponds to the second cut clause, it fails and returns 
;;;     a flag which will be catched by SOLVE.
;;;   + if the clause is a fact and the resolvent is empty, we got it ! It returns
;;;     the list of bindings.
;;;   + general case: It substitutes the variables in the body of the clause with
;;;     their bindings, nconcs that body with the resolvent and calls SOLVE again.
;;;-----------------------------------------------------------------------------

(defun reduce-goal (goal clause other-goals bindings level traced-p)
  (let* ((head (car clause))
	 (body (cdr clause))
	 (local-bindings (pf-match (cdr head) (cdr goal))))
    ;;(print local-bindings)
    (cond ((eq local-bindings 'fail) nil)		;fail -> return nil
	  ((equal body '((stop-this))) (values nil (second goal)))
	  ((and (null other-goals) (null body))
	   (when traced-p
	     (trace-clause head body local-bindings level))
	   (setq bindings (nconc local-bindings bindings))
	   (trace-success bindings level)
	   (list bindings))
	  (t (when traced-p (trace-clause head body local-bindings level))
	     (nsubstitute-vars body local-bindings)
	     (setq other-goals (smart-substitute-vars other-goals local-bindings))
	     (solve (nconc body other-goals) (nconc local-bindings bindings) level)))))



;;;=============================================================================
;;; Trace facility
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; Trace variables
;;;-----------------------------------------------------------------------------

(defparameter *trace-goals* nil
  "Trace goals to be solved")

(defparameter *trace-clauses* nil
  "Trace which clauses are applied")

(defparameter *trace-resolvent* nil
  "Trace the list of remaining goals")

(defparameter *trace-bindings* nil
  "Trace current bindings")

(defparameter *trace-success* nil
  "Trace when the resolvent becomes empty")

(defparameter *trace-results* nil
  "Trace results of a given goal")

(defparameter *trace-solutions* nil
  "Trace the solutions for a given goal")


;;;-----------------------------------------------------------------------------
;;; INDENT-STRING returns a string that is to be used as an argument for format.
;;; (indent-string 3) returns "~%~3t3".
;;;-----------------------------------------------------------------------------

(defun indent-string (level)
  (format nil "~~%~~~at~a" level level))


;;;-----------------------------------------------------------------------------
;;; TRACE-PROLOG is used by all the trace functions, to display a string with
;;; a correct indentation.
;;;-----------------------------------------------------------------------------

(defmacro trace-prolog (trace-variable level format-string &rest args)
  `(when ,trace-variable
     (format t (indent-string ,level))
     (format t ,format-string ,@args)))


;;;-----------------------------------------------------------------------------
;;; TRACE-SUCCESS prints the solution whenever the resolvent becomes empty
;;;-----------------------------------------------------------------------------

(defun trace-success (bindings level)
  (trace-prolog *trace-success* level "<= SUCCESS: ~a" (substitute-vars *goals* bindings)))


;;;-----------------------------------------------------------------------------
;;; TRACE-GOAL traces the current goal, inherited bindings and the list of
;;; remaining subgoals.
;;;-----------------------------------------------------------------------------

(defun trace-goal (goal goals bindings level)
  (format t "~%")
  (trace-prolog *trace-goals* level "=> Solving goal ~a" goal)
  (trace-prolog *trace-bindings*   level "   Bindings: ~a" bindings)
  (trace-prolog *trace-resolvent*  level "   Resolvent: ~a" goals))


;;;-----------------------------------------------------------------------------
;;; TRACE-CLAUSE traces the clauses used to solve a goal, and its reduction
;;; when it is a rule.
;;;-----------------------------------------------------------------------------

(defun trace-clause (head body binding level)
  (trace-prolog *trace-clauses*    level
		"-> Using clause ~a <- ~a" head body)
  (trace-prolog (and *trace-clauses* body) level
		"   Reduction: ~a" (substitute-vars body binding)))


;;;-----------------------------------------------------------------------------
;;; TRACE-SOLUTIONS traces success of failure for the current goal.
;;;-----------------------------------------------------------------------------

(defun trace-solutions (goal solutions level)
  (cond (solutions
	 (trace-prolog *trace-results* level "<= SUCCESS (~a) of: ~a"
		       (length solutions) goal)
	 (dolist (sol solutions)
	   (trace-prolog *trace-solutions* level "  + ~a" (substitute-vars goal sol))))
	(t (trace-prolog *trace-results* level "<= FAILURE of: ~a" goal)))) 
