;;; -*- Mode:Common-Lisp; Package:qsim; Base:10 -*-
;;;  $Id: nic.lisp,v 1.9 92/07/23 13:24:32 clancy Exp $

(in-package 'QSIM)

;;  The code in this file implements the Non-Intersection Constraint as described by Lee and
;;  Kuipers in AAAI-88.  It prunes behaviors of second order systems having intersecting
;;  phase space trajectories.

;;  The filter can be invoked by setting *apply-nic-p* to T or specifying a Phase-Planes
;;  clause in the Other clause of the QDE.  In the first case you will be prompted for the
;;  phase planes you want to detect intersections in if a Phase-Planes clause is not already
;;  present in the Other clause of the QDE.  A Phase-Planes clause is of the form:
;;  (phase-planes (V1 V2) . . . ) where Vn is a variable of the system simulated.  Either
;;  case, you have to determine the set of phase planes to check.  This is expected to be
;;  automated in the future.

;;  In specifying phase planes to be checked for trajectory intersection, care should be
;;  taken to avoid using any phase plane whose phase variables are monotonically related.
;;  To ensure that all trajectory intersections are detected, pick a maximal set of linearly
;;  independent variables for the system being simulated (where no pair of variables are
;;  monotonically related).  Check all possible phase planes between this set of variables
;;  for trajectory intersection.

; This definition has moved to qdefs.lisp
; (defvar *intersection-count* '((repeated-phase-point 0) phase-planes)); for counting intersections

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;                                    Re-implemented 9/90
;;
;;  Previous implementations checks each individual rectangle independently (formed by one or more
;;  trajectory segments) for intersection.  This is unsatisfactory.  For example:
;;
;;     A                        B     A trajectory evolves through AC, AB, CD, DG, GH, and EH.
;;       *____________________*       Consider AB through GH.  Previous implementations treat
;;       |   / \              |       these segemnts in one of two ways:  Use one big rectangle to
;;       | /    \             |       represent them, or use three small rectangles to represent
;;   --> /       \            |       them.  Either way, each rectangle is checked independently
;;       |        \           |       for intersection.
;;       |         \          |       
;;       |          \         |       In the first case, if the trajectory then enters FG, reaches 
;;       |           \        |                            D, and exits GH, an intersection would 
;;       |            \       | D                    E     have occurred, but not detected.  In
;;     C *-------------\------*--------------------*       the second case, if the trajectory 
;;       |               \    |                    |       enters FG and exits DE, an intersection 
;;       |                 \  |                    |       would have occurred, but not detected.
;;       |                   \|                    --->
;;       |                    |\                /  |       The re-implementation incorporates the
;;       |                    |  \            /    |       left-right idea in [Struss 88] and
;;       |                    |    \        /      |       uses small rectangles as building
;;       |                    |      \    /        |       blocks.  For AB thru GH, the left edges
;;       |____________________|________\/__________|       are BD, DE and the right edges CF, FG.
;;       *                    *                    *       Both of the above intersections would
;;     F                      G                      H     have been detected.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;  A Portrait (of a state) contains information to determine whether the trajectory in the
;;  corresponding phase-plane has self-intersected.

(defstruct seg-seq				; sequence of trajectory segments
  nums						; positions of trajectory segments sequence
  (status 'null)				; status of sequence with regard to self-intersection
  entry-pt					; point at which trajectory sequence area is entered
  exit-pt)					; self-intersection exit point

(defstruct traj-seg				; a trajectory segment
  corners					; the two ends of the segment (phase points)
  corner-states					; the two ends of the segment (QSIM states)
  subsumed-pts					; phase points subsumed (for composite segment)
  left-edges					; left side edges of segment
  right-edges					; right side edges of segment
  ranges)					; ranges of segment (a rectangle)

(defstruct portrait				; phase portrait info
  vars						; phase variables
  in-corner					; end of last segment (state or adjusted phase point)
  in-corner-state				; end state of last segment
  (seg-seqs (list (make-seg-seq)))		; segment sequences (last is current)
  (traj-pts (list 'traj-pts))			; real points in trajectory
  (traj-segs (list 'traj-segs)))		; trajectory segments (regular and composite)



;  Macros for various elements of a line (an edge)

(defmacro pos-var (line) `(caar ,line))
(defmacro line-pos (line) `(cadar ,line))
(defmacro seg-var (line) `(caadr ,line))
(defmacro line-seg (line) `(cadadr ,line))

;  Crossing-trajectory-filter is applied at time points.  It is applied if
;  *apply-nic-p* is true or if there is a Phase-planes clause in the other clause
;  of the QDE.  See "examples;nic-simple-spring" or "examples;nic-damped-spring"
;  for how the clause is specified.
;
;  *check-nic* added 06/11/90 by Pierre Fouche, to disable NIC even when a phase plane 
;  clause is present.
;
;  (BJK:  10-10-90)  Simplified to eliminate user interaction from a core QSIM filter.

(defun crossing-trajectory-filter (state)	; returns state or nil
  (if (and *check-nic*
	   (qpointp (state-time state)))
      (cond ((assoc 'portraits (state-other state))
	     (if (predecessor-of-state state) (filter-for-intersection state) state))
	    ((sim-phase-planes *current-sim*)
	     (create-portraits-for-state (sim-phase-planes *current-sim*) state))))
      state)

;(defun crossing-trajectory-filter (state)	; returns state or nil
;  (if (and *check-nic*
;	   (qpointp (state-time state)))
;      (if *apply-nic-p*
;	  (cond ((assoc 'portraits (state-other state))
;		 (if (predecessor-of-state state) (filter-for-intersection state) state))
;		((assoc 'phase-planes (qde-other (state-qde state)))
;		 (create-portraits-for-state (get-phase-planes-from-qde state) state))
;		(t
;		 (create-portraits-for-state (get-phase-planes-from-prompt state) state)))
;	  (cond ((assoc 'portraits (state-other state))
;		 (if (predecessor-of-state state) (filter-for-intersection state) state))
;		((assoc 'phase-planes (qde-other (state-qde state)))
;		 (if *confirm-when-present-p*
;		     (if (y-or-n-p "Apply Non-Intersection Constraint? ")
;			 (create-portraits-for-state (get-phase-planes-from-qde state) state)
;			 state)
;		     (create-portraits-for-state (get-phase-planes-from-qde state) state)))
;		(t state)))
;      state))

;  Establish criteria for intersection:
;       1.  A state matches a previous state, but their predecessors are incompatible.
;       2.  Intersection in criterial rectangles or at real points.
;       3.  Formation of new criterial rectangles.



(defun filter-for-intersection (state)		; returns state or nil
  (if (or (member 'cycle (state-status state))
	  (do ((pred (predecessor-of-state state)
		     (predecessor-of-state pred)))
	      ((null pred) nil)
	    (when (and (qpointp (state-time pred)) (match-states state pred))
	      (setf (state-successors state) `(cycle-identity ,pred))
	      (pushnew 'cycle (state-status state))
	      (return t))))
      ; checks to see if it is a real cycle     ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
      (if (compatible-predecessors state (cadr (filtered-successor-slot-contents state)))
	  (progn
	    (when (and *query-user*
		       (eq *nic-trace-mode* 'along-trajectory) (eq 'portrait-also *nic-trace-display*))
	      (format *qsim-report* "~&Enter any character to continue ...")
	      (read-char)
	      (plot-phase-diagrams-for-state state)
	      (format *qsim-report* "~&  * * *   A   C Y C L I C   B E H A V I O R   * * *~2%"))
	    state)				; yes => no intersection
	  (progn				; no => intersection
	    (when *intersection-count-p*
	      (rplaca (cdr (assoc 'repeated-phase-point *intersection-count*))
		      (1+ (cadr (assoc 'repeated-phase-point *intersection-count*)))))
	    (intersection-occurence state `(repeated-phase-point ,(cadr (state-successors state))))))
      (progn
	(make-new-portraits-data-for-state state)	; copies trajectories data from prev state
	(when (not (qval-equal (state-time state) (state-time (predecessor-of-state state))))
	  (when (and *query-user*
		     (eq *nic-trace-mode* 'along-trajectory) (eq 'portrait-also *nic-trace-display*))
	    (format *qsim-report* "~&Enter any character to continue ...")
	    (read-char)
	    (plot-phase-diagrams-for-state state))
	  (when (and  *query-user* (eq *nic-trace-mode* 'along-trajectory))
	    (format *qsim-report* "~&Trajectory evolves from ~a to ~a ..." 
		    (predecessor-of-state (predecessor-of-state state)) state))
	  (let ((portraits (cdr (assoc 'portraits (state-other state))))
		intersection-info
		(returned-value state))
	    ; When a quantity moving thru an interval, say (lmk1 lmk2), reaches a steady value, say lmk*,
	    ; all immediately preceeding (lmk1 lmk2) intervals should be changed to (lmk1 lmk*) or
	    ; (lmk* lmk2) accordingly.
	    (do ((list portraits (cdr list)))
		((null list))
	      (when (and (car list)		; make sure pertinent variables are active
			 (assoc (car (portrait-vars (car list))) (state-qvalues state))
			 (assoc (cadr (portrait-vars (car list))) (state-qvalues state)))
		(when (steady-state (car (portrait-vars (car list))) state)
		  (check-to-adjust-prev-traj-segs (car (portrait-vars (car list))) (car list) state))
		(when (steady-state (cadr (portrait-vars (car list))) state)
		  (check-to-adjust-prev-traj-segs (cadr (portrait-vars (car list))) (car list) state))))
	    (do ((list portraits (cdr list)))
		((or (null list) intersection-info)
		 returned-value)
	      (when (and (car list)		; make sure pertinent variables are active
			 (assoc (car (portrait-vars (car list))) (state-qvalues state))
			 (assoc (cadr (portrait-vars (car list))) (state-qvalues state)))
		(when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		  (let ((vars (portrait-vars (car list))))
		    (format *qsim-report* "~&  In ~a-~a phase-plane:" (car vars) (cadr vars))))
		(if (setq intersection-info
			  ;  Check intersection, either at points or in criterial rectangles.
			  (let ((phase-point (get-phase-point (car list) state))
				repeated-point)
			    (if (and (pointp phase-point)
				     (setq repeated-point
					   (check-repeated-trajectory-point phase-point (car list) state)))
				`(repeated-phase-point ,repeated-point)
				(progn
				  (when (pointp phase-point)
				    (nconc (portrait-traj-pts (car list)) (list state)))
				  (check-traj-intersection phase-point (car list) state)))))
		    (progn (setq returned-value (intersection-occurence state intersection-info))
			   (when *intersection-count-p*
			     (rplaca (cdr (assoc (portrait-vars (car list)) *intersection-count*
						 :test #'equal))
				     (1+ (cadr (assoc (portrait-vars (car list)) *intersection-count*
						      :test #'equal))))))
		    (add-new-traj-seg-info-when-appropriate (car list) state)))))))))





(defun check-to-adjust-prev-traj-segs (var portrait state)	; returned value does not matter
  (let ((in-state (portrait-in-corner-state portrait)))
    (when (assoc var (state-qvalues in-state))
      (let* ((prev-qval (cdr (assoc var (state-qvalues in-state))))
	     (prev-qmag (qval-qmag prev-qval))
	     (prev-qdir (qval-qdir prev-qval))
	     (new-interval-bound (qval-qmag (cdr (assoc var (state-qvalues state)))))
	     (qspace (cdr (assoc var (state-qspaces state))))
	     (vars (portrait-vars portrait))
	     new-qmag)
	(when (and (listp prev-qmag)
		   (landmark-lt (car prev-qmag) new-interval-bound qspace)
		   (landmark-lt new-interval-bound (cadr prev-qmag) qspace)
		   (or (when (eq 'inc prev-qdir)
			 (setq new-qmag (list (car prev-qmag) new-interval-bound)))
		       (when (eq 'dec prev-qdir)
			 (setq new-qmag (list new-interval-bound (cadr prev-qmag))))))
	  (setf (portrait-in-corner portrait)
		(if (eq var (car vars))
		    (list new-qmag
			  (qval-qmag (cdr (assoc (cadr vars) (state-qvalues in-state)))))
		    (list (qval-qmag (cdr (assoc (car vars) (state-qvalues in-state))))
			  new-qmag)))
	  (let ((traj-segs (portrait-traj-segs portrait)))
	    (do ((seqs (reverse (portrait-seg-seqs portrait)) (cdr seqs))
		 (n 1 (1+ n))
		 (no-adjustment-for-prev-seq-p nil))
		((or no-adjustment-for-prev-seq-p (null seqs)) nil)
	      (setq no-adjustment-for-prev-seq-p t)
	      (do ((nums (reverse (seg-seq-nums (car seqs))) (cdr nums)))
		  ((null nums) nil)
		(let* ((seg (nth (car nums) traj-segs))
		       (seg-corners (traj-seg-corners seg))
		       (seg-states (traj-seg-corner-states seg))
		       (subsumed-pts (traj-seg-subsumed-pts seg)))
		  (when (equal prev-qval
			       (cdr (assoc var (state-qvalues (cadr seg-states)))))
		    (let* ((pt1 (if (equal prev-qval
					   (cdr (assoc var (state-qvalues (car seg-states)))))
				    (if (eq var (car vars))
					(list new-qmag (cadar seg-corners))
					(list (caar seg-corners) new-qmag))
				    (car seg-corners)))
			   (pt2 (if (eq var (car vars))
				    (list new-qmag (cadadr seg-corners))
				    (list (caadr seg-corners) new-qmag)))
			   (xqspace (cdr (assoc (car vars) (state-qspaces state))))
			   (yqspace (cdr (assoc (cadr vars) (state-qspaces state))))
			   adjusted)
		      (when (not (phase-pts-okay-for-forming-seg-p pt1 pt2))
			(do ((l subsumed-pts (cdr l)))
			    ((null l))
			  (when (phase-pts-okay-for-forming-seg-p (caar l) pt2)
			    (setq pt1 (caar l)
				  seg-states (list (cadar l) (cadr seg-states))
				  subsumed-pts (cdr l))
			    (return nil))))
		      (setq adjusted (form-seg-with-edges-n-ranges vars pt1 pt2 xqspace yqspace))
		      (setf (traj-seg-corners adjusted) (list pt1 pt2))
		      (setf (traj-seg-corner-states adjusted) seg-states)
		      (when subsumed-pts
			(setf (traj-seg-subsumed-pts adjusted) subsumed-pts))
		      (nsubst adjusted seg traj-segs)
		      (setq no-adjustment-for-prev-seq-p nil))))))))))))



(defun add-new-traj-seg-info-when-appropriate (portrait state)	; returned value does not matter
  (let* ((x (car (portrait-vars portrait)))
	 (y (cadr (portrait-vars portrait)))
	 (in-corner (portrait-in-corner portrait)))
    (when (or (listp in-corner)
	      (and (assoc x (state-qvalues in-corner)) (assoc y (state-qvalues in-corner))))
      (when (and (null (member x (sim-ignore-qdirs *current-sim*)))
		 (null (member y (sim-ignore-qdirs *current-sim*))))
	(when (or (atom (qval-qmag (cdr (assoc x (state-qvalues state)))))
		  (atom (qval-qmag (cdr (assoc y (state-qvalues state))))))
	  (when (or (listp in-corner)
		    (atom (qval-qmag (cdr (assoc x (state-qvalues in-corner)))))
		    (atom (qval-qmag (cdr (assoc y (state-qvalues in-corner))))))
	    (add-new-traj-seg-info (get-phase-point portrait in-corner)
				   (get-phase-point portrait state) portrait state)
	    (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	      (format *qsim-report* "~&    Trajectory segment ~a just formed."
		      (format-rect (traj-seg-corners (car (last (portrait-traj-segs portrait))))))))
	  (setf (portrait-in-corner portrait) state
		(portrait-in-corner-state portrait) state))))))

(defun new-seg-corners-within-seg-ranges-p (new-seg seg xqspace yqspace)
  (let ((corners (traj-seg-corners new-seg))
	(ranges (traj-seg-ranges seg)))
    (and (within-ranges (car corners) ranges xqspace yqspace)
	 (within-ranges (cadr corners) ranges xqspace yqspace))))

(defun new-seg-corners-within-seg-seq-ranges-p (new-seg seg-seq-nums traj-segs xqspace yqspace)
  (do ((l seg-seq-nums (cdr l)))
      ((null l) nil)
    (when (new-seg-corners-within-seg-ranges-p new-seg (nth (car l) traj-segs) xqspace yqspace)
      (return t))))

(defun phase-pts-okay-for-forming-seg-p (pt1 pt2)
  (if (or (pointp pt1) (pointp pt2))
      (and (not (equal (car pt1) (car pt2))) (not (equal (cadr pt1) (cadr pt2))))
      (if (atom (car pt1))
	  (if (atom (car pt2))
	      (not (equal (car pt1) (car pt2)))
	      (or (and (equal (car pt1) (caar pt2)) (equal (caadr pt1) (cadr pt2)))
		  (and (equal (car pt1) (cadar pt2)) (equal (caadr pt1) (cadr pt2)))
		  (and (equal (car pt1) (caar pt2)) (equal (cadadr pt1) (cadr pt2)))
		  (and (equal (car pt1) (cadar pt2)) (equal (cadadr pt1) (cadr pt2)))))
	  (if (atom (car pt2))
	      (or (and (equal (car pt2) (caar pt1)) (equal (caadr pt2) (cadr pt1)))
		  (and (equal (car pt2) (cadar pt1)) (equal (caadr pt2) (cadr pt1)))
		  (and (equal (car pt2) (caar pt1)) (equal (cadadr pt2) (cadr pt1)))
		  (and (equal (car pt2) (cadar pt1)) (equal (cadadr pt2) (cadr pt1))))
	      (not (equal (cadr pt1) (cadr pt2)))))))

(defun make-composite-seg (vars new-seg last-seg xqspace yqspace)
  (let* ((pt1 (car (traj-seg-corners last-seg)))
	 (pt2 (cadr (traj-seg-corners new-seg)))
	 (composite-seg (form-seg-with-edges-n-ranges vars pt1 pt2 xqspace yqspace new-seg)))
    (setf (traj-seg-corners composite-seg) (list pt1 pt2)
	  (traj-seg-corner-states composite-seg)
	  (list (car (traj-seg-corner-states last-seg)) (cadr (traj-seg-corner-states new-seg)))
	  (traj-seg-subsumed-pts composite-seg)
	  (append (copy-tree (traj-seg-subsumed-pts last-seg))
		  `((,(car (traj-seg-corners new-seg)) ,(car (traj-seg-corner-states new-seg))))))
    composite-seg))

(defun adjust-to-form-composite-seg (vars new-seg last-seg xqspace yqspace)
  (let ((pt2 (cadr (traj-seg-corners new-seg)))
	pt1 pt1-state still-subsumed-pts composite-seg)
    (do ((l (traj-seg-subsumed-pts last-seg) (cdr l)))
	((null l))
      (when (phase-pts-okay-for-forming-seg-p (caar l) pt2)
	(setq pt1 (caar l) pt1-state (cadar l) still-subsumed-pts (cdr l)) (return nil)))
    (cond ((null pt1) new-seg)
	  (t
	   (setq composite-seg (form-seg-with-edges-n-ranges vars pt1 pt2 xqspace yqspace))
	   (setf (traj-seg-corners composite-seg) (list pt1 pt2)
		 (traj-seg-corner-states composite-seg)
		 (list pt1-state (cadr (traj-seg-corner-states new-seg)))
		 (traj-seg-subsumed-pts composite-seg)
		 (append (copy-tree still-subsumed-pts)
			 `((,(car (traj-seg-corners new-seg)) ,(car (traj-seg-corner-states new-seg))))))
	   composite-seg))))

;;
;;  Adding a new trajectory segment to existing trajectory information can be tricky because
;;  of the possible interaction between trajectory segments.  The problem is one of how to
;;  record trajectory segment information in order not to miss possible trajectory intersections.
;;  The example in Figures 1 illustrates the problem.
;;
;;                                  In Figure 1, a trajectory first evolves from GH to EG and
;;       E +--------/-------+ F     then to EF.  If we record the the two segments separately,
;;         |      /         |       we'll miss an intersection when the trajectory later evolves
;;         |    /           |       through EG and then FH.  In this case, if we record GH, EG,
;;         |  /             |       then EF as GH then EF, we will capture a later EG then FH.
;;         |/               |       On the other hand, we'll miss both a G then EF, and a E then
;;         \                |       GH.  A solution is to keep both basic segments, GH-EG and
;;         | \              |       EG-EF, as well as the composite of the two, GH-EF.
;;         |   \            |       
;;         |     \          |       Further complications arise when trying to form a composite
;;       G +-------\--------+ H     segment.  For example, if the trajectory in Figure 1 now
;;                                  evolves to H, a composite of GH then H is of no value.  The
;;              Figure 1            strategy used to deal with these complications is basically:
;;                                  Form composite segments when possible and keep all basic
;;                                  segments.
;;

(defun add-new-traj-seg-info (phase-pt1 phase-pt2 portrait state)
  (let* ((traj-segs (portrait-traj-segs portrait))
	 (vars (portrait-vars portrait))
	 (xqspace (cdr (assoc (car vars) (state-qspaces state))))
	 (yqspace (cdr (assoc (cadr vars) (state-qspaces state))))
	 (new-seg (form-seg-with-edges-n-ranges vars phase-pt1 phase-pt2 xqspace yqspace))
	 (seg-seqs (portrait-seg-seqs portrait))
	 (current-seg-seq (car (last seg-seqs)))
	 (current-seg-seq-nums (seg-seq-nums current-seg-seq))
	 (last-seg (and current-seg-seq-nums
			(nth (car (last current-seg-seq-nums)) traj-segs))))
    (when new-seg
      (setf (traj-seg-corners new-seg) (list phase-pt1 phase-pt2)
	    (traj-seg-corner-states new-seg) (list (portrait-in-corner-state portrait) state))
      (cond ((and current-seg-seq-nums
		  (new-seg-corners-within-seg-ranges-p new-seg last-seg xqspace yqspace))
	     (let (composite-seg new-seg-seq)
	       (cond ((phase-pts-okay-for-forming-seg-p
			(car (traj-seg-corners last-seg)) (cadr (traj-seg-corners new-seg)))
		      (setq composite-seg
			    (make-composite-seg vars new-seg last-seg xqspace yqspace)
			    new-seg-seq
			    (make-seg-seq :nums (append (copy-list (butlast current-seg-seq-nums))
							(list (1+ (length traj-segs)))))))
		     (t
		      (setq composite-seg
			    (adjust-to-form-composite-seg vars new-seg last-seg xqspace yqspace)
			    new-seg-seq
			    (make-seg-seq :nums (list (1+ (length traj-segs)))))))
	       (nconc seg-seqs
		      (list (make-seg-seq :nums (list (length traj-segs))) new-seg-seq))
	       (nconc traj-segs
		      (list new-seg composite-seg))))
	    ((and current-seg-seq-nums
		  (new-seg-corners-within-seg-seq-ranges-p new-seg current-seg-seq-nums traj-segs
							   xqspace yqspace))
	     (nconc seg-seqs (list (make-seg-seq :nums (list (length traj-segs)))))
	     (nconc traj-segs (list new-seg)))
	    (t
	     (cond (current-seg-seq-nums
		    (nconc (seg-seq-nums current-seg-seq) (list (length traj-segs))))
		   (t (setf (seg-seq-nums current-seg-seq) (list (length traj-segs)))))
	     (nconc traj-segs (list new-seg))))
      (setf (portrait-seg-seqs portrait) (remove-redundant-seg-seqs (cdr seg-seqs) (car seg-seqs))))))

(defun remove-redundant-seg-seqs (seg-seqs prev-seq)
  (cond ((null seg-seqs) (list prev-seq))
	((equal (seg-seq-nums prev-seq) (seg-seq-nums (car seg-seqs)))
	 (remove-redundant-seg-seqs (cdr seg-seqs) prev-seq))
	(t (cons prev-seq (remove-redundant-seg-seqs (cdr seg-seqs) (car seg-seqs))))))

(defun form-seg-with-edges-n-ranges (vars pt1 pt2 xqspace yqspace &optional (seg-flag nil))
  (if (pointp pt1)
      (if (pointp pt2)
	  (form-pt-pt-seg vars pt1 pt2 xqspace yqspace seg-flag)
	  (form-pt-int-seg vars pt1 pt2 xqspace yqspace))
      (if (pointp pt2)
	  (form-int-pt-seg vars pt1 pt2 xqspace yqspace)
	  (form-int-int-seg vars pt1 pt2 xqspace yqspace))))

(defun form-int-int-seg (vars int1 int2 xqspace yqspace)
  (let* ((x (car vars))
	 (y (cadr vars))
	 (x1 (car int1))
	 (y1 (cadr int1))
	 (x2 (car int2))
	 (y2 (cadr int2)))
    (if (equal int1 int2)
	(when *nic-notifications-p*
	  (format *qsim-report* "~&  *** Identical intervals, ~a, do not form a rectangle" int1)
	  nil)
	(if (atom x1)
	    (if (atom x2)
		(if (equal y1 y2)
		    (let* ((dx (calc-dist x1 x2 xqspace))
			   (ranges (list dx y1)))
		      (if (landmark-lt x1 x2 xqspace)
			  (make-traj-seg :left-edges `(((,y ,(cadr y1)) (,x ,dx)))
					 :right-edges `(((,y ,(car y1)) (,x ,dx)))
					 :ranges ranges)
			  (make-traj-seg :left-edges `(((,y ,(car y1)) (,x ,dx)))
					 :right-edges `(((,y ,(cadr y1)) (,x ,dx)))
					 :ranges ranges)))
		    (let ((dx (calc-dist x1 x2 xqspace)))
		      (if (equal (car y1) (car y2))
			  (if (landmark-lt (cadr y1) (cadr y2) yqspace)
			      (if (landmark-lt x1 x2 xqspace)
				  (make-traj-seg :left-edges `(((,x ,x1) (,y (,(cadr y1) ,(cadr y2))))
							       ((,y ,(cadr y2)) (,x ,dx)))
						 :right-edges `(((,y ,(car y1)) (,x ,dx)))
						 :ranges (list dx y2))
				  (make-traj-seg :left-edges `(((,y ,(car y1)) (,x ,dx)))
						 :right-edges `(((,x ,x1) (,y (,(cadr y1) ,(cadr y2))))
								((,y ,(cadr y2)) (,x ,dx)))
						 :ranges (list dx y2)))
			      (if (landmark-lt x1 x2 xqspace)
				  (make-traj-seg :left-edges `(((,x ,x2) (,y (,(cadr y2) ,(cadr y1))))
							       ((,y ,(cadr y1)) (,x ,dx)))
						 :right-edges `(((,y ,(car y1)) (,x ,dx)))
						 :ranges (list dx y1))
				  (make-traj-seg :left-edges `(((,y ,(car y1)) (,x ,dx)))
						 :right-edges `(((,x ,x2) (,y (,(cadr y2) ,(cadr y1))))
								((,y ,(cadr y1)) (,x ,dx)))
						 :ranges (list dx y1))))
			  (if (equal (cadr y1) (cadr y2))
			      (if (landmark-lt (car y1) (car y2) yqspace)
				  (if (landmark-lt x1 x2 xqspace)
				      (make-traj-seg :left-edges `(((,y ,(cadr y2)) (,x ,dx)))
						     :right-edges `(((,x ,x2) (,y (,(car y1) ,(car y2))))
								    ((,y ,(car y1)) (,x ,dx)))
						     :ranges (list dx y1))
				      (make-traj-seg :left-edges `(((,x ,x2) (,y (,(car y1) ,(car y2))))
								   ((,y ,(car y1)) (,x ,dx)))
						     :right-edges `(((,y ,(cadr y2)) (,x ,dx)))
						     :ranges (list dx y1)))
				  (if (landmark-lt x1 x2 xqspace)
				      (make-traj-seg :left-edges `(((,y ,(cadr y1)) (,x ,dx)))
						     :right-edges `(((,x ,x1) (,y (,(car y2) ,(car y1))))
								    ((,y ,(car y2)) (,x ,dx)))
						     :ranges (list dx y2))
				      (make-traj-seg :left-edges `(((,x ,x1) (,y (,(car y2) ,(car y1))))
								   ((,y ,(car y2)) (,x ,dx)))
						     :right-edges `(((,y ,(cadr y1)) (,x ,dx)))
						     :ranges (list dx y2))))
			      (when *nic-notifications-p*
				(format *qsim-report* "~&  *** Parallel intervals, distinct ends: ~a ~a."
					int1 int2)
				nil)))))
		(if (equal x1 (car x2))
		    (if (equal (car y1) y2)
			(make-traj-seg :left-edges `(((,y ,(cadr y1)) (,x ,x2))
						     ((,x ,(cadr x2)) (,y ,y1)))
				       :right-edges (list (list x1 y2))
				       :ranges (list x2 y1))
			(make-traj-seg :left-edges (list (list x1 y2))
				       :right-edges `(((,y ,(car y1)) (,x ,x2))
						      ((,x ,(cadr x2)) (,y ,y1)))
				       :ranges (list x2 y1)))
		
		    (if (equal (car y1) y2)
			(make-traj-seg :left-edges (list (list x1 y2))
				       :right-edges `(((,y ,(cadr y1)) (,x ,x2))
						      ((,x ,(car x2)) (,y ,y1)))
				       :ranges (list x2 y1))
			(make-traj-seg :left-edges `(((,y ,(car y1)) (,x ,x2))
						     ((,x ,(car x2)) (,y ,y1)))
				       :right-edges (list (list x1 y2))
				       :ranges (list x2 y1)))))
	    (if (atom y2)
		(if (equal x1 x2)
		    (let* ((dy (calc-dist y1 y2 yqspace))
			   (ranges (list x1 dy)))
		      (if (landmark-lt y1 y2 yqspace)
			  (make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy)))
					 :right-edges `(((,x ,(cadr x1)) (,y ,dy)))
					 :ranges ranges)
			  (make-traj-seg :left-edges `(((,x ,(cadr x1)) (,y ,dy)))
					 :right-edges `(((,x ,(car x1)) (,y ,dy)))
					 :ranges ranges)))
		    (let ((dy (calc-dist y1 y2 yqspace)))
		      (if (equal (car x1) (car x2))
			  (if (landmark-lt (cadr x1) (cadr x2) xqspace)
			      (if (landmark-lt y1 y2 yqspace)
				  (make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy)))
						 :right-edges `(((,x ,(cadr x2)) (,y ,dy))
								((,y ,y1) (,x (,(cadr x1) ,(cadr x2)))))
						 :ranges (list x2 dy))
				  (make-traj-seg :left-edges `(((,y ,y1) (,x (,(cadr x1) ,(cadr x2))))
							       ((,x ,(cadr x2)) (,y ,dy)))
						 :right-edges `(((,x ,(car x1)) (,y ,dy)))
						 :ranges (list x2 dy)))
			      (if (landmark-lt y1 y2 yqspace)
				  (make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy)))
						 :right-edges `(((,x ,(cadr x1)) (,y ,dy))
								((,y ,y2) (,x (,(cadr x2) ,(cadr x1)))))
						 :ranges (list x1 dy))
				  (make-traj-seg :left-edges `(((,y ,y2) (,x (,(cadr x2) ,(cadr x1))))
							       ((,x ,(cadr x1)) (,y ,dy)))
						 :right-edges `(((,x ,(car x1)) (,y ,dy)))
						 :ranges (list x1 dy))))
			  (if (equal (cadr x1) (cadr x2))
			      (if (landmark-lt (car x1) (car x2) xqspace)
				  (if (landmark-lt y1 y2 yqspace)
				      (make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy))
								   ((,y ,y2) (,x (,(car x1) ,(car x2)))))
						     :right-edges `(((,x ,(cadr x1)) (,y ,dy)))
						     :ranges (list x1 dy))
				      (make-traj-seg :left-edges `(((,x ,(cadr x1)) (,y ,dy)))
						     :right-edges `(((,x ,(car x1)) (,y ,dy))
								    ((,y ,y2) (,x (,(car x1) ,(car x2)))))
						     :ranges (list x1 dy)))
				  (if (landmark-lt y1 y2 yqspace)
				      (make-traj-seg :left-edges `(((,x ,(car x2)) (,y ,dy))
								   ((,y ,y1) (,x (,(car x2) ,(car x1)))))
						     :right-edges `(((,x ,(cadr x1)) (,y ,dy)))
						     :ranges (list x2 dy))
				      (make-traj-seg :left-edges `(((,x ,(cadr x1)) (,y ,dy)))
						     :right-edges `(((,x ,(car x2)) (,y ,dy))
								    ((,y ,y1) (,x (,(car x2) ,(car x1)))))
						     :ranges (list x2 dy))))
			      (when *nic-notifications-p*
				(format *qsim-report* "~&  *** Parallel intervals, distinct ends: ~a ~a."
					int1 int2)
				nil)))))
		(if (equal y1 (car y2))
		    (if (equal (car x1) x2)
			(make-traj-seg :left-edges (list (list x2 y1))
				       :right-edges `(((,y ,(cadr y2)) (,x ,x1))
						      ((,x ,(cadr x1)) (,y ,y2)))
				       :ranges (list x1 y2))
			(make-traj-seg :left-edges `(((,y ,(cadr y2)) (,x ,x1))
						     ((,x ,(car x1)) (,y ,y2)))
				       :right-edges (list (list x2 y1))
				       :ranges (list x1 y2)))
		    (if (equal (car x1) x2)
			(make-traj-seg :left-edges `(((,y ,(car y2)) (,x ,x1))
						     ((,x ,(cadr x1)) (,y ,y2)))
				       :right-edges (list (list x2 y1))
				       :ranges (list x1 y2))
			(make-traj-seg :left-edges (list (list x2 y1))
				       :right-edges `(((,y ,(car y2)) (,x ,x1))
						      ((,x ,(car x1)) (,y ,y2)))
				       :ranges (list x1 y2)))))))))

(defun form-int-pt-seg (vars int pt xqspace yqspace)
  (let* ((x (car vars))
	 (y (cadr vars))
	 (x1 (car int))
	 (y1 (cadr int))
	 (x2 (car pt))
	 (y2 (cadr pt))
	 (dx (calc-dist x1 x2 xqspace))
	 (dy (calc-dist y1 y2 yqspace))
	 (ranges (if (atom x1)
		     (list dx y1)
		     (list x1 dy))))
    (if (atom x1)
	(if (equal x1 x2)
	    (when *nic-notifications-p*
	      (format *qsim-report* "~&  *** Point ~a along the same line with interval ~a does not ~
                                     form a rectangle." pt int)
	      nil)
	    (if (landmark-lt x1 x2 xqspace)
		(if (equal (car y1) y2)
		    (make-traj-seg :left-edges `(((,x ,x2) (,y ,y1)) ((,y ,(cadr y1)) (,x ,dx)))
				   :right-edges `(((,y ,y2) (,x ,dx)))
				   :ranges ranges)
		    (if (equal (cadr y1) y2)
			(make-traj-seg :left-edges `(((,y ,y2) (,x ,dx)))
				       :right-edges `(((,x ,x2) (,y ,y1)) ((,y ,(car y1)) (,x ,dx)))
				       :ranges ranges)
			(make-traj-seg :left-edges `(((,y ,(cadr y1)) (,x ,dx))
						     ((,x ,x2) (,y (,y2 ,(cadr y1)))))
				       :right-edges `(((,y ,(car y1)) (,x ,dx))
						      ((,x ,x2) (,y (,(car y1) ,y2))))
				       :ranges ranges)))
		(if (equal (car y1) y2)
		    (make-traj-seg :left-edges `(((,y ,y2) (,x ,dx)))
				   :right-edges `(((,x ,x2) (,y ,y1)) ((,y ,(cadr y1)) (,x ,dx)))
				   :ranges ranges)
		    (if (equal (cadr y1) y2)
			(make-traj-seg :left-edges `(((,x ,x2) (,y ,y1)) ((,y ,(car y1)) (,x ,dx)))
				       :right-edges `(((,y ,y2) (,x ,dx)))
				       :ranges ranges)
			(make-traj-seg :left-edges `(((,y ,(car y1)) (,x ,dx))
						     ((,x ,x2) (,y (,(car y1) ,y2))))
				       :right-edges `(((,y ,(cadr y1)) (,x ,dx))
						      ((,x ,x2) (,y (,y2 ,(cadr y1)))))
				       :ranges ranges)))))
	(if (equal y1 y2)
	    (when *nic-notifications-p*
	      (format *qsim-report* "~&  *** Point ~a along the same line with interval ~a does not ~
                                     form a rectangle." pt int)
	      nil)
	    (if (landmark-lt y1 y2 yqspace)
		(if (equal (car x1) x2)
		    (make-traj-seg :left-edges `(((,x ,x2) (,y ,dy)))
				   :right-edges `(((,x ,(cadr x1)) (,y ,dy)) ((,y ,y2) (,x ,x1)))
				   :ranges ranges)
		    (if (equal (cadr x1) x2)
			(make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy)) ((,y ,y2) (,x ,x1)))
				       :right-edges `(((,x ,x2) (,y ,dy)))
				       :ranges ranges)
			(make-traj-seg :left-edges `(((,x ,(car x1)) (,y ,dy))
						     ((,y ,y2) (,x (,(car x1) ,x2))))
				       :right-edges `(((,x ,(cadr x1)) (,y ,dy))
						      ((,y ,y2) (,x (,x2 ,(cadr x1)))))
				       :ranges ranges)))
		(if (equal (car x1) x2)
		    (make-traj-seg :left-edges `(((,x ,(cadr x1)) (,y ,dy)) ((,y ,y2) (,x ,x1)))
				   :right-edges `(((,x ,x2) (,y ,dy)))
				   :ranges ranges)
		    (if (equal (cadr x1) x2)
			(make-traj-seg :left-edges `(((,x ,x2) (,y ,dy)))
				       :right-edges `(((,x ,(car x1)) (,y ,dy)) ((,y ,y2) (,x ,x1)))
				       :ranges ranges)
			(make-traj-seg :left-edges `(((,x ,(cadr x1)) (,y ,dy))
						     ((,y ,y2) (,x (,(cadr x1) ,x2))))
				       :right-edges `(((,x ,(car x1)) (,y ,dy))
						      ((,y ,y2) (,x (,x2 ,(car x1)))))
				       :ranges ranges))))))))

(defun form-pt-int-seg (vars pt int xqspace yqspace)
  (let* ((x (car vars))
	 (y (cadr vars))
	 (x1 (car pt))
	 (y1 (cadr pt))
	 (x2 (car int))
	 (y2 (cadr int))
	 (dx (calc-dist x1 x2 xqspace))
	 (dy (calc-dist y1 y2 yqspace))
	 (ranges (if (atom x2)
		     (list dx y2)
		     (list x2 dy))))
    (if (atom x2)
	(if (equal x1 x2)
	    (when *nic-notifications-p*
	      (format *qsim-report* "~&  *** Point ~a along the same line with interval ~a does not ~
                                     form a rectangle." pt int)
	      nil)
	    (if (landmark-lt x1 x2 xqspace)
		(if (equal y1 (car y2))
		    (make-traj-seg :left-edges `(((,x ,x1) (,y ,y2)) ((,y ,(cadr y2)) (,x ,dx)))
				   :right-edges `(((,y ,y1) (,x ,dx)))
				   :ranges ranges)
		    (make-traj-seg :left-edges `(((,y ,y1) (,x ,dx)))
				   :right-edges `(((,x ,x1) (,y ,y2)) ((,y ,(car y2)) (,x ,dx)))
				   :ranges ranges))
		(if (equal y1 (car y2))
		    (make-traj-seg :left-edges `(((,y ,y1) (,x ,dx)))
				   :right-edges `(((,x ,x1) (,y ,y2)) ((,y ,(cadr y2)) (,x ,dx)))
				   :ranges ranges)
		    (make-traj-seg :left-edges `(((,x ,x1) (,y ,y2)) ((,y ,(car y2)) (,x ,dx)))
				   :right-edges `(((,y ,y1) (,x ,dx)))
				   :ranges ranges))))
	(if (equal y1 y2)
	    (when *nic-notifications-p*
	      (format *qsim-report* "~&  *** Point ~a along the same line with interval ~a does not ~
                                     form a rectangle." pt int)
	      nil)
	    (if (landmark-lt y1 y2 yqspace)
		(if (equal x1 (car x2))
		    (make-traj-seg :left-edges `(((,x ,x1) (,y ,dy)))
				   :right-edges `(((,x ,(cadr x2)) (,y ,dy)) ((,y ,y1) (,x ,x2)))
				   :ranges ranges)
		    (make-traj-seg :left-edges `(((,x ,(car x2)) (,y ,dy)) ((,y ,y1) (,x ,x2)))
				   :right-edges `(((,x ,x1) (,y ,dy)))
				   :ranges ranges))
		(if (equal x1 (car x2))
		    (make-traj-seg :left-edges `(((,x ,(cadr x2)) (,y ,dy)) ((,y ,y1) (,x ,x2)))
				   :right-edges `(((,x ,x1) (,y ,dy)))
				   :ranges ranges)
		    (make-traj-seg :left-edges `(((,x ,x1) (,y ,dy)))
				   :right-edges `(((,x ,(car x2)) (,y ,dy)) ((,y ,y1) (,x ,x2)))
				   :ranges ranges)))))))

(defun form-pt-pt-seg (vars pt1 pt2 xqspace yqspace seg-flag)
  (let* ((x (car vars))
	 (y (cadr vars))
	 (x1 (car pt1))
	 (y1 (cadr pt1))
	 (x2 (car pt2))
	 (y2 (cadr pt2))
	 (x-range (calc-lmk-lmk-dist x1 x2 xqspace))
	 (y-range (calc-lmk-lmk-dist y1 y2 yqspace))
	 (ranges (list x-range y-range)))
    (if (or (equal x1 x2) (equal y1 y2))
	(if seg-flag
	    (let* ((o-ranges (traj-seg-ranges seg-flag))
		   (ox-range (car o-ranges))
		   (oy-range (cadr o-ranges))
		   (ox1 (car ox-range))
		   (ox2 (cadr ox-range))
		   (oy1 (car oy-range))
		   (oy2 (cadr oy-range)))
	      (if (equal x1 x2)
		  (if (equal x1 ox1)
		      (if (equal y1 oy1)
			  (make-traj-seg :left-edges `(((,x ,ox1) (,y ,y-range)))
					 :right-edges `(((,x ,ox2) (,y ,y-range))
							((,y ,y1) (,x ,ox-range))
							((,y ,y2) (,x ,ox-range)))
					 :ranges o-ranges)
			  (if (equal y1 oy2)
			      (make-traj-seg :left-edges `(((,x ,ox2) (,y ,y-range))
							   ((,y ,y1) (,x ,ox-range))
							   ((,y ,y2) (,x ,ox-range)))
					     :right-edges `(((,x ,ox1) (,y ,y-range)))
					     :ranges o-ranges)
			      (when *nic-notifications-p*
				(format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                                       for composite-seg: ~a ~a" pt1 pt2)
				nil)))
		      (if (equal x1 ox2)
			  (if (equal y1 oy1)
			      (make-traj-seg :left-edges `(((,x ,ox1) (,y ,y-range))
							   ((,y ,y1) (,x ,ox-range))
							   ((,y ,y2) (,x ,ox-range)))
					     :right-edges `(((,x ,ox2) (,y ,y-range)))
					     :ranges o-ranges)
			      (if (equal y1 oy2)
				  (make-traj-seg :left-edges `(((,x ,ox2) (,y ,y-range)))
						 :right-edges `(((,x ,ox1) (,y ,y-range))
								((,y ,y1) (,x ,ox-range))
								((,y ,y2) (,x ,ox-range)))
						 :ranges o-ranges)
			      (when *nic-notifications-p*
				(format *qsim-report* "~&  *** Unanticipated pt-pt segment 
                                                       for composite-seg: ~a ~a" pt1 pt2)
				nil)))
			  (when *nic-notifications-p*
			    (format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                                   for composite-seg: ~a ~a" pt1 pt2)
			    nil)))
		  (if (equal y1 y2)
		      (if (equal x1 ox1)
			  (if (equal y1 oy1)
			      (make-traj-seg :left-edges `(((,y ,oy2) (,x ,x-range))
							   ((,x ,x1) (,y ,oy-range))
							   ((,x ,x2) (,y ,oy-range)))
					     :right-edges `(((,y ,oy1) (,x ,x-range)))
					     :ranges o-ranges)
			      (if (equal y1 oy2)
				  (make-traj-seg :left-edges `(((,y ,oy2) (,x ,x-range)))
						 :right-edges `(((,y ,oy1) (,x ,x-range))
								((,x ,x1) (,y ,oy-range))
								((,x ,x2) (,y ,oy-range)))
						 :ranges o-ranges)
				  (when *nic-notifications-p*
				    (format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                                           for composite-seg: ~a ~a" pt1 pt2)
				    nil)))
			  (if (equal x1 ox2)
			      (if (equal y1 oy1)
				  (make-traj-seg :left-edges `(((,y ,oy1) (,x ,x-range)))
						 :right-edges `(((,y ,oy2) (,x ,x-range))
								((,x ,x1) (,y ,oy-range))
								((,x ,x2) (,y ,oy-range)))
						 :ranges o-ranges)
				  (if (equal y1 oy2)
				      (make-traj-seg :left-edges `(((,y ,oy1) (,x ,x-range))
								   ((,x ,x1) (,y ,oy-range))
								   ((,x ,x2) (,y ,oy-range)))
						     :right-edges `(((,y ,oy2) (,x ,x-range)))
						     :ranges o-ranges)
				      (when *nic-notifications-p*
					(format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                                               for composite-seg: ~a ~a" pt1 pt2)
					nil)))
			      (when *nic-notifications-p*
				(format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                                       for composite-seg: ~a ~a" pt1 pt2)
				nil)))
		      (when *nic-notifications-p*
			(format *qsim-report* "~&  *** Unanticipated pt-pt segment ~
                                               for composite-seg: ~a ~a" pt1 pt2)
			nil))))
	    (when *nic-notifications-p*
	      (format *qsim-report* "~&  *** These form a line, not a rectangle: ~a ~a" pt1 pt2)
	      nil))
	(if (landmark-lt x1 x2 xqspace)
	    (if (landmark-lt y1 y2 yqspace)
		(make-traj-seg :left-edges `(((,x ,x1) (,y ,y-range)) ((,y ,y2) (,x ,x-range)))
			       :right-edges `(((,x ,x2) (,y ,y-range)) ((,y ,y1) (,x ,x-range)))
			       :ranges ranges)
		(make-traj-seg :left-edges `(((,x ,x2) (,y ,y-range)) ((,y ,y1) (,x ,x-range)))
			       :right-edges `(((,x ,x1) (,y ,y-range)) ((,y ,y2) (,x ,x-range)))
			       :ranges ranges))
	    (if (landmark-lt y1 y2 yqspace)
		(make-traj-seg :left-edges `(((,x ,x2) (,y ,y-range)) ((,y ,y1) (,x ,x-range)))
			       :right-edges `(((,x ,x1) (,y ,y-range)) ((,y ,y2) (,x ,x-range)))
			       :ranges ranges)
		(make-traj-seg :left-edges `(((,x ,x1) (,y ,y-range)) ((,y ,y2) (,x ,x-range)))
			       :right-edges `(((,x ,x2) (,y ,y-range)) ((,y ,y1) (,x ,x-range)))
			       :ranges ranges))))))

(defun check-traj-intersection (phase-pt portrait state)
  (do ((l (portrait-seg-seqs portrait) (cdr l)))
      ((null l) nil)
    (let ((intersect-p (check-each-seg-seq-for-intersection (car l) phase-pt portrait state)))
      (cond ((eq intersect-p 'entered)
	     (when (eq (car l) (car (last (portrait-seg-seqs portrait))))
	       (nconc (portrait-seg-seqs portrait)
		      (let ((seg-seq (car (make-copy-of-seg-seq (car l)))))
			(setf (seg-seq-status seg-seq) 'null)
			(setf (seg-seq-entry-pt seg-seq) nil)
			(list seg-seq)))
	       (return nil)))
	    (intersect-p (return intersect-p))))))

(defun traj-evolved-backwards-p (phase-pt seg-seq portrait state)
  (let* ((x (car (portrait-vars portrait)))
	 (y (cadr (portrait-vars portrait)))
	 (xqspace (cdr (assoc x (state-qspaces state))))
	 (yqspace (cdr (assoc y (state-qspaces state))))
	 (last-seg (nth (car (last (seg-seq-nums seg-seq))) (portrait-traj-segs portrait))))
    (when (within-ranges phase-pt (traj-seg-ranges last-seg) xqspace yqspace)
      (let* ((bef-state (predecessor-of-state (cadr (traj-seg-corner-states last-seg))))
	     (aft-state (predecessor-of-state state))
	     (b-xdir (qval-qdir (cdr (assoc x (state-qvalues bef-state)))))
	     (b-ydir (qval-qdir (cdr (assoc y (state-qvalues bef-state)))))
	     (a-xdir (qval-qdir (cdr (assoc x (state-qvalues aft-state)))))
	     (a-ydir (qval-qdir (cdr (assoc y (state-qvalues aft-state))))))
	(if (eq b-xdir 'inc) (if (eq a-xdir 'inc) (if (eq b-ydir 'inc) (if (eq a-ydir 'dec)
									   'evolved-to-right
									   nil)
						      (if (eq b-ydir 'dec) (if (eq a-ydir 'inc)
									       'evolved-to-left
									       nil)
							  nil))
				 (if (eq a-xdir 'dec) (if (eq b-ydir 'inc) (if (eq a-ydir 'inc)
									       'evolved-to-left
									       nil)
							  (if (eq b-ydir 'dec) (if (eq a-ydir 'dec)
										   'evolved-to-right
										   nil)
							      nil))
				     nil))
	    (if (eq b-xdir 'dec) (if (eq a-xdir 'inc) (if (eq b-ydir 'inc) (if (eq a-ydir 'inc)
									       'evolved-to-right
									       nil)
							  (if (eq b-ydir 'dec) (if (eq a-ydir 'dec)
										   'evolved-to-left
										   nil)
							      nil))
				     (if (eq a-xdir 'dec) (if (eq b-ydir 'inc) (if (eq a-ydir 'dec)
										   'evolved-to-left
										   nil)
							      (if (eq b-ydir 'dec) (if (eq a-ydir 'inc)
										       'evolved-to-right
										       nil)
								  nil))
					 nil))
		nil))))))

(defun check-each-seg-seq-for-intersection (seg-seq phase-pt portrait state)
  (let ((status (in-traj-area-p seg-seq phase-pt portrait state)))
    (case (seg-seq-status seg-seq)
      (null
	(case status
	  (on-left
	    (setf (seg-seq-status seg-seq) 'entered-on-left
		  (seg-seq-entry-pt seg-seq) phase-pt)
	    (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	      (format *qsim-report* "~&    Trajectory evolved to a left edge of segment sequence ~a."
		      (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))))
	    'entered)
	  (on-right
	    (setf (seg-seq-status seg-seq) 'entered-on-right
		  (seg-seq-entry-pt seg-seq) phase-pt)
	    (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	      (format *qsim-report* "~&    Trajectory evolved to a right edge of segment sequence ~a."
		      (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))))
	    'entered)
	  (inside
	    (let ((evolved-back-p (when (eq seg-seq (car (last (portrait-seg-seqs portrait))))
				    (traj-evolved-backwards-p phase-pt seg-seq portrait state))))
	      (when evolved-back-p
		(setf (seg-seq-status seg-seq) evolved-back-p
		      (seg-seq-entry-pt seg-seq) phase-pt)
		(when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		  (format *qsim-report* "~&    Trajectory evolved to the ~a of segment sequence ~a."
			  (if (eq evolved-back-p 'evolved-to-left) 'left 'right)
			  (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))))
		'entered)))
	  (null
	    (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory) (seg-seq-nums seg-seq))
	      (format *qsim-report* "~&    Trajectory not in segment sequence ~a."
		      (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))))
	    nil)))
      ((entered-on-left evolved-to-left)
       (case status
	 (on-left
	   (cond ((not (prev-state-in-traj-area-p seg-seq portrait state))
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    After entering segment sequence ~a on the left at ~a, ~
                                           trajectory exits and reenters."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))
		  (setf (seg-seq-entry-pt seg-seq) phase-pt))
		 (t
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    Trajectory remains in segment sequence ~a after ~
                                           entering on the left at ~a."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))))
	   nil)
	 (on-right
	   (cond ((prev-state-in-traj-area-p seg-seq portrait state)
		  (setf (seg-seq-exit-pt seg-seq) phase-pt)
		  `(crossed-trajectory-area ,(portrait-vars portrait) ,seg-seq	; intersection !!
					    ,(format-seq (seg-seq-nums seg-seq)
							 (portrait-traj-segs portrait))))
		 (t
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    After entering segment sequence ~a on the left at ~a, ~
                                           trajectory exits and reenters on the right."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))
		  (setf (seg-seq-status seg-seq) 'entered-on-right
			(seg-seq-entry-pt seg-seq) phase-pt)
		  nil)))
	 (inside
	   (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	     (format *qsim-report* "~&    Trajectory remains in segment sequence ~a after ~
                                    entering on the left at ~a."
		     (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
		     (format-point (seg-seq-entry-pt seg-seq))))
	   nil)
	 (null
	   (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	     (format *qsim-report* "~&    Trajectory exits segment sequence ~a after entering on the ~
                                  left at ~a."
		     (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
		     (format-point (seg-seq-entry-pt seg-seq))))
	   (setf (seg-seq-status seg-seq) 'null)
	   nil)))
      ((entered-on-right evolved-to-right)
       (case status
	 (on-left
	   (cond ((prev-state-in-traj-area-p seg-seq portrait state)
		  (setf (seg-seq-exit-pt seg-seq) phase-pt)
		  `(crossed-trajectory-area ,(portrait-vars portrait) ,seg-seq	; intersection !!
					    ,(format-seq (seg-seq-nums seg-seq)
							 (portrait-traj-segs portrait))))
		 (t
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    After entering segment sequence ~a on the right at ~a, ~
                                           trajectory exits and reenters on the left."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))
		  (setf (seg-seq-status seg-seq) 'entered-on-left
			(seg-seq-entry-pt seg-seq) phase-pt)
		  nil)))
	 (on-right
	   (cond ((not (prev-state-in-traj-area-p seg-seq portrait state))
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    After entering segment sequence ~a on the right at ~a, ~
                                           trajectory exits and reenters."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))
		  (setf (seg-seq-entry-pt seg-seq) phase-pt))
		 (t
		  (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
		    (format *qsim-report* "~&    Trajectory remains in segment sequence ~a after ~
                                           entering on the right at ~a."
			    (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
			    (format-point (seg-seq-entry-pt seg-seq))))))
	   nil)
	 (inside
	   (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	     (format *qsim-report* "~&    Trajectory remains in segment sequence ~a after ~
                                    entering on the right at ~a."
		     (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
		     (format-point (seg-seq-entry-pt seg-seq))))
	   nil)
	 (null
	   (when (and *query-user* (eq *nic-trace-mode* 'along-trajectory))
	     (format *qsim-report* "~&    Trajectory exits segment sequence ~a after entering on the ~
                                  right at ~a."
		     (format-seq (seg-seq-nums seg-seq) (portrait-traj-segs portrait))
		     (format-point (seg-seq-entry-pt seg-seq))))
	   (setf (seg-seq-status seg-seq) 'null)
	   nil))))))

(defun in-traj-area-p (seg-seq phase-pt portrait state)
  (let* ((seg-nums (seg-seq-nums seg-seq))
	 (traj-segs (portrait-traj-segs portrait))
	 (qvalues (state-qvalues state))
	 (qspaces (state-qspaces state))
	 (vars (portrait-vars portrait))
	 (xqspace (cdr (assoc (car vars) (state-qspaces state))))
	 (yqspace (cdr (assoc (cadr vars) (state-qspaces state)))))
    (cond ((on-edges-p #'traj-seg-left-edges seg-nums traj-segs qvalues qspaces portrait state)
	   'on-left)
	  ((on-edges-p #'traj-seg-right-edges seg-nums traj-segs qvalues qspaces portrait state)
	   'on-right)
	  ((in-area-p phase-pt seg-nums traj-segs xqspace yqspace) 'inside)
	  (t 'null))))

(defun on-edges-p (edges seg-nums traj-segs qvalues qspaces portrait state)
  (do ((l seg-nums (cdr l)))
      ((null l) nil)
    (when (on-an-edge (apply edges (list (nth (car l) traj-segs))) qvalues qspaces portrait state)
      (return t))))

(defun in-area-p (phase-pt seg-nums traj-segs xqspace yqspace)
  (do ((l seg-nums (cdr l)))
      ((null l) nil)
    (when (within-ranges phase-pt (traj-seg-ranges (nth (car l) traj-segs)) xqspace yqspace)
      (return t))))

(defun prev-state-in-traj-area-p (seg-seq portrait state)
  (let ((pstate (predecessor-of-state state)))
    (when pstate
      (let* ((vars (portrait-vars portrait))
	     (xqspace (cdr (assoc (car vars) (state-qspaces state))))
	     (yqspace (cdr (assoc (cadr vars) (state-qspaces state))))
	     (phase-pt (get-phase-point portrait pstate))
	     (traj-segs (portrait-traj-segs portrait)))
	(do ((l (seg-seq-nums seg-seq) (cdr l)))
	    ((null l) nil)
	  (when (within-ranges phase-pt (traj-seg-ranges (nth (car l) traj-segs)) xqspace yqspace)
	    (return t)))))))



;  Creates 'portraits clause for initial states.  Initial states do not go thru the regular filter
;  since no intersection is possible in the initial state.

(defun create-portraits-for-state (phase-planes state)
  (if (state-other state)
      (nconc (state-other state)
	     (list (cons 'portraits
			 (mapcan #'(lambda (pp) (when pp
						  (check-for-initial-point
						    (make-portrait :vars (get-portrait-vars pp)
								   :in-corner state
								   :in-corner-state state)
						    state)))
				 phase-planes))))
      (setf (state-other state)
	    (list (cons 'portraits
			(mapcan #'(lambda (pp) (when pp
						 (check-for-initial-point
						   (make-portrait :vars (get-portrait-vars pp)
								  :in-corner state
								  :in-corner-state state)
						   state)))
				phase-planes)))))
  (rplaca (cdar *intersection-count*) 0)
  (rplacd *intersection-count* (mapcar #'(lambda (pp) (list (get-portrait-vars pp) 0)) phase-planes))
  state)

;  Takes care of real points in the initial state.

(defun check-for-initial-point (portrait state)
  (when (and (assoc (car (portrait-vars portrait)) (state-qvalues state))
	     (assoc (cadr (portrait-vars portrait)) (state-qvalues state)))
    (when (pointp (get-phase-point portrait state))
      (nconc (portrait-traj-pts portrait) (list state))))
  (list portrait))




;;  CHECK-FOR-CYCLE may draw incorrect conclusions.
;;
;;  If predecessors of matched states are incompatible, a trajectory intersection occurs
;;  rather than a cycle.

;  Checks to see if predecessors of matched states are compatible.

(defun compatible-predecessors (state pred)
  (do ((p1 (predecessor-of-state pred)
	   (predecessor-of-state p1))
       (p2 (predecessor-of-state state)
	   (predecessor-of-state p2)))
      ((null p1) t)
    (when (incompatible-states p1 p2 (state-qspaces p2))
      (return nil))))

;  Checks to see if two states are incompatible.  They are if either the qmags of any
;  parameter are distinct, or the qdirs are different.

(defun incompatible-states (s1 s2 qspaces)
  (do ((vals1 (state-qvalues s1) (cdr vals1))
       (vals2 (state-qvalues s2)))
      ((null vals1) nil)
    (let* ((parm (caar vals1))
	   (qval1 (cdar vals1))
	   (qval2 (cdr (assoc parm vals2))))
      (when (not (eq parm 'time))
	(when (or (distinct-qmags (qval-qmag qval1) (qval-qmag qval2) (cdr (assoc parm qspaces)))
		  (not (eq (qval-qdir qval1) (qval-qdir qval2))))
	  (return t))))))

;  Two qmags are distinct if their intersection is empty.

(defun distinct-qmags (qmag1 qmag2 qspace)
  (if (atom qmag1)
      (if (atom qmag2)
	  (not (eq qmag1 qmag2))
	  (or (landmark-le qmag1 (car qmag2) qspace)
	      (landmark-le (cadr qmag2) qmag1 qspace)))
      (if (atom qmag2)
	  (or (landmark-le qmag2 (car qmag1) qspace)
	      (landmark-le (cadr qmag1) qmag2 qspace))
	  (or (landmark-le (cadr qmag1) (car qmag2) qspace)
	      (landmark-le (cadr qmag2) (car qmag1) qspace)))))



;  A new set of portraits for each state to include info pertaining to current state.

(defun make-new-portraits-data-for-state (state)
  (setf (state-other state) (copy-state-other (state-other state))) ; BKay 26May92
  (rplacd (assoc 'portraits (state-other state))
	  (mapcan #'(lambda (p) (copy-lists-over-for-new-portrait
				  p
				  (make-portrait :vars (portrait-vars p)
						 :in-corner (portrait-in-corner p)
						 :in-corner-state (portrait-in-corner-state p))))
		  (cdr (assoc 'portraits (state-other state))))))

(defun copy-lists-over-for-new-portrait (old new)
  (setf (portrait-seg-seqs new) (mapcan #'(lambda (r) (make-copy-of-seg-seq r)) (portrait-seg-seqs old)))
  (setf (portrait-traj-pts new) (copy-tree (portrait-traj-pts old)))
  (setf (portrait-traj-segs new)
	(cons 'traj-segs
	      (mapcan #'(lambda (r) (make-copy-of-traj-seg r)) (cdr (portrait-traj-segs old)))))
  (list new))

(defun make-copy-of-seg-seq (seg-seq)
  (list (make-seg-seq :nums (copy-list (seg-seq-nums seg-seq))
		      :status (seg-seq-status seg-seq)
		      :entry-pt (copy-tree (seg-seq-entry-pt seg-seq)))))

(defun make-copy-of-traj-seg (traj-seg)
  (list (make-traj-seg :corners (copy-tree (traj-seg-corners traj-seg))
		       :corner-states (copy-list (traj-seg-corner-states traj-seg))
		       :subsumed-pts (copy-tree (traj-seg-subsumed-pts traj-seg))
		       :left-edges (copy-tree (traj-seg-left-edges traj-seg))
		       :right-edges (copy-tree (traj-seg-right-edges traj-seg))
		       :ranges (copy-tree (traj-seg-ranges traj-seg)))))

;  Intersection occurs.

(defun intersection-occurence (state info)	; returns state or nil
  (setf (state-successors state) `(trajectory-intersection))
  (pushnew `(self-intersection ,info) (state-status state) :test #'equal)
  (when (and  *query-user*
	      (or (and *nic-trace-mode* (eq *nic-trace-display* 'portrait-on-intersection))
		  (and (eq *nic-trace-mode* 'on-intersection-only)
		       (eq *nic-trace-display* 'portrait-also))
		  (and (eq *nic-trace-mode* 'along-trajectory) (eq 'portrait-also *nic-trace-display*))))
    (cond ((and (not (eq (car info) 'repeated-phase-point))
		(eq *nic-trace-mode* 'along-trajectory)
		(eq 'portrait-also *nic-trace-display*))
	   nil)
	  (t
	   (format *qsim-report* "~&Enter any character to continue ...")
	   (read-char)))
    (plot-phase-diagrams-for-state state)
    (when (and (not (eq (car info) 'repeated-phase-point))
	       *query-user*
	       (eq *nic-trace-mode* 'along-trajectory)
	       (eq 'portrait-also *nic-trace-display*))
      (format *qsim-report* "~&***** Plot updated to reflect trajectory intersection. *****~2%"))
    (when (not (eq (car info) 'repeated-phase-point))
      (format *qsim-report* "~&Trajectory evolves from ~a to ~a ..."
	      (predecessor-of-state (predecessor-of-state state)) state)
      (format *qsim-report* "~&  In ~a-~a phase-plane:" (caadr info) (cadadr info))))
  (when *nic-trace-mode* 
    (when (and (not (eq (car info) 'repeated-phase-point)) (eq *nic-trace-display* 'text-only)
	       (eq *nic-trace-mode* 'on-intersection-only))
      (format *qsim-report* "~&Trajectory evolves from ~a to ~a ..."
	      (predecessor-of-state (predecessor-of-state state)) state)
      (format *qsim-report* "~&  In ~a-~a phase-plane:" (caadr info) (cadadr info)))
    (display-intersection-message state info))
  (cond (*prune-intersections-p*
	 (prune-inconsistent-state state "Trajectory self-intersection")
	 nil)
	(t state)))



; Plots a phase diagrams for behavior up to State (during trace).

(defun plot-phase-diagrams-for-state (state)
  (let ((beh (get-behavior-up-to-state state))
	(portraits (cdr (assoc 'portraits (state-other state)))))
    (compute-rectangular-layout *phase-slice-rows* *phase-slice-cols*)
    (qplot-new-behavior)
    #+(or ti symbolics) (set-cursor-pos 0 0)
    (format *qsim-report* "~%Tracing evolution of trajectories of behaviors:")
    (format *qsim-report* "~%  Simulation time currently at ~a." (qval-qmag (state-time state)))
    (format *qsim-report* "~%  Looking at trajectory ending in state ~a." state)
    (format *qsim-report* "~%  Sequence of states: ")
    (mapc #'(lambda (s) (prin1 s) (format *qsim-report* " ")) beh)
    (do* ((ps portraits (cdr ps))
	  (p (car ps) (car ps))	    
	  (xpos lmargin (+ xpos *xsize* xsep)))
	 ((null ps))
      (let* ((parms (portrait-vars p))
	     (label (format nil "~a vs ~a" (car parms) (cadr parms))))
	(plot-one-phase-diagram parms beh xpos tmargin *xsize* *ysize* label)))
    (let* ((initials (get-list-of-initial-states (car beh) :complete-partial-state nil))
	   (behaviors (mapcar #'get-behaviors initials))
	   (bnum (get-bnum (apply #'append behaviors) beh))
	   (btotal (apply #'+ (mapcar #'length behaviors)))
	   (bmax (apply #'max (mapcan #'(lambda (L) (mapcar #'length L)) behaviors))))
      (plot-state-tree initials bmax btotal (- xscreen rmargin xbsize)
		       tmargin xbsize ybsize bnum))
    #+(or ti symbolics) (set-cursor-pos 0 (+ tmargin *ysize* ysep))))

(defun get-behavior-up-to-state (state)
  (cond ((null (predecessor-of-state state)) (list state))
	(t (append (get-behavior-up-to-state (predecessor-of-state state)) (list state)))))

(defun get-bnum (behs beh)
  (cond ((null behs) 1)
	((eq (car (last (car behs))) (car (last beh))) 1)
	(t (1+ (get-bnum (cdr behs) beh)))))

; Trace messages for intersection.

(defun display-intersection-message (state info)
  (case (car info)
    (repeated-phase-point
      (format *qsim-report* "~&Trajectory evolves from ~a to ~a ..."
	      (predecessor-of-state (predecessor-of-state state)) state)
      (format *qsim-report* "~&  Trajectory self-intersects:  ~a = ~a  with distinct predecessors."
	      state (cadr info)))
    (crossed-trajectory-area
      (let (in out)
	(case (seg-seq-status (caddr info))
	  ((entered-on-left evolved-to-left) (setq in "left" out "right"))
	  (t (setq in "right" out "left")))
	(format *qsim-report* "~&    Trajectory self-intersects:  Entering segment sequence ~a ~
                               on the ~a at ~a and exiting on the ~a at ~a."
		(cadddr info) in (format-point (seg-seq-entry-pt (caddr info)))
		out (format-point (seg-seq-exit-pt (caddr info)))))))
  (format *qsim-report* "~&  Simulation from this state is inhibited (state marked inconsistent). ~
                         ~%Continuing simulation of remaining branches ...~2%"))

(defun format-seq (nums segs)
  (if nums
      (do ((l nums (cdr l))
	   (string "{"))
	  ((null l) (format nil "~a~a}" string
			    (format-point (cadr (traj-seg-corners (nth (car (last nums)) segs))))))
	(setq string
	      (format nil "~a~a," string (format-point (car (traj-seg-corners (nth (car l) segs)))))))
      "{}"))

(defun format-rect (corners)
  (format nil "~a~a" (format-point (car corners)) (format-point (cadr corners))))

(defun format-point (point)
  (format nil "[~a ~a]" (car point) (cadr point)))



;; Various supporting functions in the next two pages.

(defun get-portrait-vars (portrait)
  (do ((p portrait (portrait-vars p)))
      ((listp p) p)))

(defun check-repeated-trajectory-point (phase-point portrait state)
  (do ((points (cdr (portrait-traj-pts portrait)) (cdr points)))
      ((null points) nil)
    (when (and (equal (get-phase-point portrait (car points)) phase-point)
	       (not (compatible-predecessors state (car points))))
      (return t))))

(defun within-ranges (phase-point ranges xqspace yqspace)
  (and (within-closed-int (car phase-point) (car ranges) xqspace)
       (within-closed-int (cadr phase-point) (cadr ranges) yqspace)))

(defun on-an-edge (edges values qspaces portrait state)
  (do* ((e edges (cdr e))
	(line (car e) (car e)))
       ((null e) nil)
    (cond ((pointp line) (when (equal line (get-phase-point portrait state)) (return t)))
	  ((on-line line values qspaces) (return t)))))

(defun on-line (line values qspaces)
  (if (or (null line) (not (listp line)))
      nil
      (and (equal (line-pos line) (qval-qmag (cdr (assoc (pos-var line) values))))
	   (within-closed-int (qval-qmag (cdr (assoc (seg-var line) values)))
			      (line-seg line)
			      (cdr (assoc (seg-var line) qspaces))))))

(defun within-closed-int (qval int qspace)
  (if (or (null qval) (null int))
      nil
      (if (atom qval)
	  (or (eql (car int) qval) (eql (cadr int) qval)
	      (and (landmark-lt (car int) qval qspace) (landmark-lt qval (cadr int) qspace)))
	  (and (or (eql (car int) (car qval)) (landmark-lt (car int) (car qval) qspace))
	       (or (eql (cadr int) (cadr qval)) (landmark-lt (cadr qval) (cadr int) qspace))))))



(defun get-phase-point (portrait state)
  (if (listp state)
      state
      (list (qval-qmag (cdr (assoc (car (portrait-vars portrait)) (state-qvalues state))))
	    (qval-qmag (cdr (assoc (cadr (portrait-vars portrait)) (state-qvalues state)))))))

(defun form-edge (segment vars)
  (if (atom (car segment))
      `((,(car vars) ,(car segment)) (,(cadr vars) ,(cadr segment)))
      `((,(cadr vars) ,(cadr segment)) (,(car vars) ,(car segment)))))

(defun pointp (phase-point)
  (and (atom (car phase-point)) (atom (cadr phase-point))))

(defun steady-state (var state)
  (eq 'std (qval-qdir (cdr (assoc var (state-qvalues state))))))

(defun calc-dist (qval1 qval2 qspace)
  (if (atom qval1)
      (if (atom qval2)
	  (calc-lmk-lmk-dist qval1 qval2 qspace)
	  (calc-lmk-int-dist qval1 qval2 qspace))
      (if (atom qval2)
	  (calc-lmk-int-dist qval2 qval1 qspace)
	  (if (landmark-lt (cadr qval1) (car qval2) qspace)
	      (list (cadr qval1) (car qval2))
	      (if (landmark-lt (cadr qval2) (car qval1) qspace)
		  (list (cadr qval2) (car qval1))
		  nil)))))

(defun calc-lmk-lmk-dist (l1 l2 qspace)
  (if (eql l1 l2)
      nil
      (if (landmark-lt l1 l2 qspace)
	  (list l1 l2)
	  (list l2 l1))))

(defun calc-lmk-int-dist (lmk int qspace)
  (if (landmark-lt lmk (car int) qspace)
      (list lmk (car int))
      (if (landmark-lt (cadr int) lmk qspace)
	  (list (cadr int) lmk)
	  nil)))
