;;; -*- Syntax: Common-Lisp; Package: QSIM -*-
;;;  $Id: nhod-deriviation.lisp,v 1.1 91/03/26 21:38:00 clancy Exp $
; Copyright 1989 Benjamin Kuipers

; QDE.other.chattering-variables is a slot for the user to assert variables that chatter.
; QDE.other.derived-chvars holds the variables for which curvature constraints will be
;     derived.  They are the union of user-asserted and automatically identified ones.
; QDE.other.curvature-at-steady hold user-asserted expressions for (sd2 <var>).
; QDE.other.derived-sd2-expressions hold the union of the user-asserted and derived
;     expressions for (sd2 <var>).
; QDE.other.sd3-constraint holds user-asserted (sd3 <var>) expressions.
;
;             These defother forms are in structures.lisp:
;(defother qde chattering-variables)		; user assert chattering variables for QDE
;(defother qde derived-chvars)			; merge w results of automatic identification
;(defother qde curvature-at-steady)		; alist of (<var> <sd2-exp>)
;(defother qde derived-sd2-expressions)		;   derived alist of (<var> <sd2-exp>)
;(defother qde sd3-constraint)     		; alist of (<var> <sd3-exp>)

; Eventually, the curvature expressions should be stored in QDE.curvatures:
;     ((= (sd2 <var>) <exp>)
;     ((= (sd3 <var>) <exp>)

; Control switches.

(defparameter *derive-curvature-constraints* t)	; automatically derive HOD constraint
(defparameter *chiu-acc-derivation* nil)	; use Charles Chiu's ACC derivation method
(defparameter *search-state-limit* 30)		; resource limit for transformation search
(defparameter *search-type* 'best-first)	; depth-first | breadth-first | best-first

(defparameter *trace-successful-rules* nil)
(defparameter *trace-children* nil)
(defparameter *trace-sd2-derivation* t)
(defparameter *trace-chatter-vars* t)		; trace identification of chatter vars
(defparameter *trace-acc-filter-application* nil)
(defparameter *trace-AOnode-creation* nil)	; Trace node creation in expression search


; Find-or-create retrieval of sd2 expressions.
;  - if they are stored in QDE.derived-sd2-expressions, derivation has already been done.
;  - otherwise translate and assert user-provided expressions in QDE.curvature-at-steady.
;  - then try automatic derivation of expressions.

(defun get-curvature-expressions (qde)
  (or (qde-derived-sd2-expressions qde)
      (let ((temp (qde-curvature-at-steady qde)))
	(if temp (setf (qde-derived-sd2-expressions qde)
		       (sublis (qde-var-alist qde) temp)))
	(if *derive-curvature-constraints*
	    (derive-curvature-constraints qde))))
  (qde-derived-sd2-expressions qde))


; For each <var> in QDE.derived-chvars, derive an expression for (sd2 <var>).
; Store the result in QDE.derived-sd2-constraints.

(defun derive-curvature-constraints (qde)
  (if *derive-curvature-constraints*
      (let ((chvars (get-chattering-variables qde)))
	(cond ((eql chvars 'none) nil)
	      (t (dolist (chvar chvars)
		   (curvature-constraint qde chvar))
		 t)))))

; Compute the curvature constraint for a given variable, from a given QDE.
;  (unless there is already an expression stored for it)

(defun curvature-constraint (qde chvar)
  (check-var chvar)
  (or (lookup chvar (qde-derived-sd2-expressions qde))
      (let ((*constraints* (qde-constraints qde))
	    (*independents* (sublis (qde-var-alist qde) (qde-independent qde)))
	    (*qde* qde)
	    (*chatter-var* chvar))
	(declare (special *qde* *constraints* *independents* *chatter-var*))
	(let ((curv-con nil))
	  (trace-sd2-derivation-start chvar curv-con)
;	  (setq curv-con (AO-simplify (list 'sd2 chvar)))
	  (setq curv-con (AO-search (list 'sd2 chvar)))	; testing new method
	  (trace-sd2-derivation-end chvar curv-con)
	  (cond (t ;curv-con
		 (pushnew (list chvar curv-con)
			  (qde-derived-sd2-expressions qde)
			  :test #'equal) ))))))


; The user may specify variables alleged to chatter in QDE.other.chattering-variables.
; The program adds variables it believes may chatter, in QDE.other.derived-chvars.
;   - put variables into the same equivalence class if linked by M+ or M- or equivalent.
;   - delete an equivalence class if it contains a constant.
;   - delete an equivalance class if it contains a variable with an explicit derivative.
;   - select a representative from each remaining equivalence class.

(defun get-chattering-variables (qde)
  (or (qde-derived-chvars qde)
      (identify-chatter qde)))

(defun identify-chatter (qde)
  (let ((eqc (classes-of-chattering-variables qde))
	(constraints (qde-constraints qde))
	(known-chvars (sublis (qde-var-alist qde) (qde-chattering-variables qde))))
    (setf (qde-derived-chvars qde)
	  (or (union (mapcar #'(lambda (class)
				 (representative-from-eqc class constraints known-chvars))
			     eqc)
		     known-chvars)
	      'none))
    (trace-chatter-vars qde)
    (qde-derived-chvars qde)))

; The equivalence-class algorithm for identifying variables likely to chatter.
; This is a function, without external side-effect.

(defun classes-of-chattering-variables (qde)			; => equivalence classes
  (let ((eqc (mapcar #'list (cdr (qde-variables qde))))	; cdr to eliminate variable TIME
	(constraints (qde-constraints qde)))
    ;
    (dolist (con constraints)
      (setq eqc (merge-on-single-constraint con qde eqc)))
    ;
    (dolist (con constraints)
      (setq eqc (delete-eqc-on-constraint con eqc)))
    ;
    (dolist (var (qde-variables qde))
      (if (member (variable-name var) (qde-independent qde))
	  (setq eqc (delete-class-of var eqc))))
    ;
    eqc))

; Return the set of equivalence classes after merging, if possible.

(defun merge-on-single-constraint (con qde eqc)
  (cond ((member (contype-name (constraint-type con)) '(M+ M- minus))
	 (merge-classes-of (first (constraint-variables con))
			   (second (constraint-variables con))
			   eqc))
	((member (contype-name (constraint-type con)) '(add mult))
	 (let ((realvars (remove-if #'(lambda (v) (constant-var v qde))
				    (constraint-variables con))))
	   (cond ((= (length realvars) 2)
		  (merge-classes-of (first realvars) (second realvars) eqc))
		 (t eqc))))
	(t eqc)))

(defun constant-var (v qde)
  (or (member (variable-name v) (qde-independent qde))
      (dolist (con (qde-constraints qde))
	(if (and (eql 'constant (contype-name (constraint-type con)))
		 (eql v (first (constraint-variables con))))
	    (return t)))))

; Delete equivalence class if the constraint says so.

(defun delete-eqc-on-constraint (con eqc)
  (cond ((eql (contype-name (constraint-type con)) 'd/dt)
	 (delete-class-of (first (constraint-variables con)) eqc))
	((eql (contype-name (constraint-type con)) 'constant)
	 (delete-class-of (first (constraint-variables con)) eqc))
	(t eqc)))

; Assert the equivalence of two elements, and merge their classes, if necessary.

(defun merge-classes-of (a b eqc)
  (let ((ac (car (member a eqc :test #'member)))
	(bc (car (member b eqc :test #'member))))
    (cond ((eql ac bc) eqc)
	  (t (nconc ac bc)
	     (delete bc eqc)))))

(defun delete-class-of (a eqc)
  (let ((ac (car (member a eqc :test #'member))))
    (delete ac eqc)))


; Select a representative from an equivalence class
;   - the single element if there is only one.
;   - if an element from this class is already designated, stick with it.
;   - if some element is a derivative, that element.
;   - otherwise the first element.

(defun representative-from-eqc (eqc constraints &optional (designated nil))
  (cond ((= (length eqc) 1) (first eqc))
	((dolist (var eqc)
	   (if (member var designated) (return var))))
	((dolist (var eqc)
	   (if (dolist (con constraints)
		 (if (and (eql (contype-name (constraint-type con)) 'd/dt)
			  (eql var (second (constraint-variables con))))
		     (return t)))
	       (return var))))
	(t (first eqc))))

; Trace functions.

(defun trace-successful-rule (rule exp result)
  (if *trace-successful-rules*
      (format *qsim-trace* "~%Transform ~a ~%  to ~a ~%  by rule ~a." exp result rule)))

(defun trace-AO-successors (AOnode)
  (when *trace-children*
    (format *qsim-trace* "~%The successors of ~a:" (AOnode-expression AOnode))
    (dolist (entry (AOnode-successors AOnode))
      (format *qsim-trace* "~%  = ~a" (car entry)))))

(defun trace-sd2-derivation-start (chvar exp)
  (declare (ignore exp))
  (if *trace-sd2-derivation*
      (format *qsim-trace* "~%Curvature constraint (SD2 ~a) = " chvar)))

(defun trace-sd2-derivation-end (chvar exp)
  (declare (ignore chvar))
  (if *trace-sd2-derivation*
      (format *qsim-trace* " ~a." exp)))

(defun trace-chatter-vars (qde)
  (if *trace-chatter-vars*
      (format *qsim-trace* "~%Variables in ~a likely to chatter are ~a." qde (qde-derived-chvars qde))))



(defun trace-AOnode-creation-A (parent)
  (if *trace-AOnode-creation*
      (format *QSIM-trace* "~%AO successor to ~a:  " parent)))

(defun trace-AOnode-creation-B (child foundp)
  (if *trace-AOnode-creation*
      (format *QSIM-trace* "~a  (~a):  " child foundp)))





; Experimenting with AO-search [Nilsson, 1980, pp.104-105].

; The intention is that AOnode.expression should only be of the form (SD2 <var>).

(defstruct (AOnode (:print-function AO-printer))
  (expression nil)				; expression of the form (SD2 <var>)
  (solved nil)					; NIL | T  --  has this node been solved?
  (solution nil)				; if so, the solution.
  (successors nil)				; alist of (<exp> <alist of (<subexp> <node>)>)
  (parents nil)					; set of nodes (parent pointers may be cyclic)
  )

(defparameter *AOnode-index* nil)		; alist of (<exp> <node>)

(defun AO-printer (AOnode stream ignore)
  (format stream "#<AOnode: ~a>" (AOnode-expression AOnode)))

;   Create a node representing the initial (SD2 <var>) expression.
;   Repeat:
;       Select an unexpanded node.
;       Expand it.  (Solutions will propagate upward.)
;       Check to see whether the initial node is now solved.

(defun AO-search (exp)
  (setq *AOnode-index* nil)
  (let* ((top-node (make-AOnode :expression exp))
	 (queue (list top-node)))		; queue of nodes
    (do ((resources *search-state-limit* (- resources 1))
	 (focus (car queue) (car queue)))
	((AOnode-solved top-node) (AOnode-solution top-node))
     ;(format *QSIM-trace* ".")
      (cond ((null queue) (return nil))
	    ((AOnode-successors focus) (setq queue (cdr queue)))
	    ((AOnode-solved focus)     (setq queue (cdr queue)))
	    (t (AO-expand focus)
	       (trace-AO-successors focus)
	       (setq queue (extend-AO-queue focus (cdr queue))))))))

; Search of the AO-graph is exhaustive.  No effort is yet made to manage the queue sensibly.

(defun extend-AO-queue (focus queue)
  (union (mapcan #'(lambda (entry)
		     (mapcan #'(lambda (pair) (unless (or (AOnode-solved (cadr pair))
							  (AOnode-successors (cadr pair)))
						(list (cadr pair))))
			     (cadr entry)))
		 (AOnode-successors focus))
	 queue))

; The successors to a node exist at two levels:
;  - the OR-links are the alternate equivalent expressions produced by QA-TRANSFORM.
;      * if an expression is free of SD2 terms, the current node is solved.
;      * in that case, propagate the solved label upward to parents.
;  - each expression is an AND-link of its (SD2 <var>) subexpressions.
;  - create a new node for each (SD2 <var>) subexpression.

(defun AO-expand (AOnode)
  (let ((*transforms* nil)
	(exp (AOnode-expression AOnode)))
    (declare (special *transforms*))
    (qa-transform exp exp)
    (setq *transforms* (sort *transforms*
			     #'(lambda (a b) (< (size* a) (size* b)))))

    (dolist (new-exp *transforms*)
      (when (not (member* 'sd2 new-exp))
	(assert-solved AOnode new-exp)))

    (setf (AOnode-successors AOnode)
	  (mapcar #'(lambda (exp) (list exp (AOnodes-from-exp exp AOnode)))
		  *transforms*))

    (length (AOnode-successors AOnode))))

(defun find-or-create-AOnode-expression (exp parent)
  (cond ((atom exp) nil)
	((eql (car exp) 'SD2)
	 (let ((node (lookup exp *AOnode-index* :test #'equal)))
	   (trace-AOnode-creation-A parent)
	   (cond (node (setf (AOnode-parents node) (cons parent (AOnode-parents node)))
		       (trace-AOnode-creation-B node "found"))
		 (t (setq node (make-AOnode :expression exp
					    :parents (list parent)))
		    (trace-AOnode-creation-B node "created")
		    (push (list exp node) *AOnode-index*)))
	   node))
	(t (error "Bad AOnode expression:  ~a" exp))))

(defun AOnodes-from-exp (exp parent)			; => alist of (<subexp> <node>)
  (cond ((atom exp) nil)
	((eql (car exp) 'SD2)
	 (list (list exp (find-or-create-AOnode-expression exp parent))))
 	(t (mapcan #'(lambda (e) (AOnodes-from-exp e parent))
		   exp))))

; Once a node is solved, propagate the solution upward to its parents,
; substituting in the expression and simplifying.

(defun assert-solved (node exp)
  (setf (AOnode-solved node) T)
  (setf (AOnode-solution node) exp)
  (dolist (parent (AOnode-parents node))
    (propagate-solved parent))
  t)

(defun propagate-solved (node)
  (unless (AOnode-solved node)
    (dolist (or-link (AOnode-successors node))
      (when (every #'(lambda (pair) (AOnode-solved (cadr pair)))
		   (cadr or-link))
	(assert-solved node
		       (qa-simplify (subst-solutions (car or-link) (cadr or-link))))
	(return node)))))

(defun subst-solutions (exp alist)
  (cond ((atom exp) exp)
	((eql (car exp) 'SD2)
	 (let ((soln (lookup exp alist :test #'equal)))
	   (cond (soln (AOnode-solution soln))
		 (t exp))))
	(t (mapcar #'(lambda (e) (subst-solutions e alist)) exp))))



; There are significant problems with this approach to deriving curvature constraints.
;
;  1.  The heuristic search is poorly guided.       (FIXED)
;      - It would be improved by being organized as an And-Or search, so that
;        transformations to subexpressions would not be duplicated.
;      - Try viewing it as a Means-Ends analysis problem.
;
;  2.  The match is not indexed, but simply iterates through the list of rules.
;      The match and transformations does a great deal of consing.

; Utilities

(defun member* (x S)				; does x appear anywhere in S?
  (cond ((eql x S) T)
	((atom S) nil)
	((member* x (car S)) t)
	((member* x (cdr S)) t)))

(defun size* (S)
  (cond ((atom S) 1)
	(t (let ((N (length S)))
	     (dolist (item S N)
	       (setq N (+ N (size* item))))))))

(defun count-atoms (x S)
  (cond ((atom S) (if (eql x S) 1 0))
	(t (+ (count-atoms x (car S))
	      (count-atoms x (cdr S))))))
