;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: ncurvature.lisp,v 1.3 92/07/07 11:57:19 clancy Exp $

(in-package 'QSIM)

;;; Application of Higher-Order Derivatives (Curvature-at-Steady Constraints).
;;; [Refs: Kuipers and Chiu,  IJCAI-87.
;;;        Chiu, Higher order derivative constraints and a QSIM-based total simulation scheme.
;;;             U-Texas, AITR88-65, 1988.]              

;;; Author: Charles Chiu.
;;; Completion date:  August 1987.  Revised November 1988.
;;; Substantially revised by Ben Kuipers, 9-21-89.
;;;      Replaced the derivation of curvature constraints with a different package.
;;;      in HOD-derivation.lisp.

;;; This file together with the accompanying files listed below derives and applies 
;;; the curvature-at-steady-filter, or the cast-filter as one of the global filters in QSIM algorithm.
;;; 
;;; It consists of following files:

;;; I.   Curvature.lisp: The present file. It calls the derivation of 
;;;      algebraic expressions of the cast-filter and applies this filter to QSIM algorithm.
;;; II.  HOD-derivation.lisp:  Uses AO search to derive the SD2 expression.
;;; III. Hod system property.  
;;; IV.  In front-end.lisp: Specifications of menus.
;;;  V.  Catalogs of examples:
;;;      a. Chatter1.lisp: Models with two tanks illustrating the applications of ignore-qdirs
;;;         and curvature constraints in taming the explosive QSIM branching behaviors 
;;;      b. Chatter2.lisp: Models with three and more tanks to illustrate the application
;;;         of curvature constraints and third order derivative constraints. 
;;;      c. Chatter3.lisp Examples illustrating several technical points in the derivation and
;;;         the application of curvature constraints. 
;;;      d. Chatter4.lisp: Examples of W-tube model and 3t-tank model where the hod-system-property
;;;         helps to resolve the chattering behaviors. 
;;;        
;;; To run acc-models, it is necessary to leave the "perform acc analysis" switch on. 
;;; Also switch to "accept analytic function only".
;;; Trace acc-filter at work: Swith on "ACC filter details".

;;; CURVATURE-AT-STEADY filter.
;;; Curvature filter is specified through qde-other clause in define-QDE.
;;; Format: For automatic calculation, do nothing.  NOT (other (curvature-at-steady)).
;;;         For manual input, use (other (curvature-at-steady (HOD1 exp1)(HOD2 exp2))).


;;; This filter applies to alists of parameter values proposed by CFILTER.

(defun curvature-at-steady-filter (state param-values)
  (cond ((null *perform-acc-analysis*) param-values)
        (t (let* ((qde (state-qde state))
		  (cast-clauses (get-curvature-expressions qde)))
	     (trace-curvature-at-steady-filter state)
	     (cast-filter-on-param-list cast-clauses param-values param-values state)))))

(defun trace-curvature-at-steady-filter (state)
  (let ((qde (state-qde state)))
    (when (and *trace-acc-filter-application*
	       (eql state *initial-state*))
      (format *QSIM-trace* "~%Curvatures: ~a" (get-curvature-expressions qde))
      (format *QSIM-trace* "~%Sd3-constraints:  ~a~%" (qde-sd3-constraint qde)))))


;;; 1. Initialization of curvature constraint.

;;; Trace functions.

(defun trace-acc (param exp cast-val sd3-list present next pvalues state)
  (when *qsim-report*
    (format *trace-acc*
	    "~%~%Acc-filter for HOD: ~a   State: ~a ~%Curvature at steady: ~a = ~a    present-qdir:~a    next-qdir:~a"
	    param state exp cast-val present next)
    (unless (or (null sd3-list) (null (car sd3-list)))
      (format *trace-acc* "~%sd3-list: ~a " sd3-list)) pvalues))



;;; A debugging check, to catch cases where <var> comes through as a symbol.

(defun check-var (var)
  (or (eql (type-of var) 'variable)
      (error "Variable ~a of type ~a, rather than VARIABLE." var (type-of var)))
  var)



;;; 2. Curvature constraint filtering. 

(defun cast-filter-on-param-list (cast-clauses pvs param-values state)
  (declare (ignore pvs))
  (dolist (clause cast-clauses param-values)
    (or (cast-filter-on-param (cadr clause)
			      (assoc (variable-name (car clause)) param-values)
			      param-values
			      state)
	(return nil))))

;;; The cast-value is the sign of cast-expression, which is evaluated at the steady point
;;; of HOD = (car cast-expression). 
;;; Exclusion of adjacent steady point: Filtering effect is on only when curvature is finite,
;;; i.e. when cast-value = + or -. When this is the case, it excludes an adjacent steady point.

(defun cast-filter-on-param (cast-expression param-value param-values state)
  (let* ((param (car param-value))
	 (present-qdir (qdir (alookup param (state-qvalues state))))
	 (next-qdir    (qdir (cdr param-value))) 
	 (cast-value1  (when (eql 'std present-qdir)
			 (eval-cast cast-expression (state-qvalues state) state)))
	 (cast-value2  (when (eql 'std next-qdir)
			 (eval-cast cast-expression param-values state)))
	 (sd3-list1    (when (eql cast-value1 0)
			 (get-sd3-list param (state-qvalues state) state)))
	 (sd3-list2    (when (eql cast-value2 0)
			 (get-sd3-list param param-values state)))
	 (temp nil))
    (cond ((and (eql 'std present-qdir)
	       (eql 'std next-qdir))
	  (setq temp (not (or (eql cast-value1 '+)
			      (eql cast-value1 '-)
			      (eql cast-value2 '+)
			      (eql cast-value2 '-))))
	  (if (null temp)
	      (trace-HOD-filter param-value state 'constant-interval))
	  temp)
	 ((eql 'std present-qdir)
	  (setq temp (post-steady-filter next-qdir cast-value1 sd3-list1))
	  (if (null temp)
	      (trace-HOD-filter param-value state 'post-steady-filter))
;	  (trace-acc param cast-expression cast-value1 sd3-list1
;		     present-qdir next-qdir (state-qvalues state) state)
	  temp)   
	 ((eql 'std next-qdir)
	  (setq temp (pre-steady-filter present-qdir cast-value2 (car sd3-list2)))
	  (if (null temp)
	      (trace-HOD-filter param-value state 'pre-steady-filter))
;	  (trace-acc param cast-expression cast-value2 sd3-list2
;		     present-qdir next-qdir param-values state)
	  temp)
	 (t t))))

(defun trace-HOD-filter (param-value state why)
  (if *trace-ACC-filter-application*
      (format *QSIM-trace* "~%HOD ~a eliminated ~a following ~a."
	      why param-value state)))

;;; This trace-acc call may be unnecessary or unhelpful.


;Following cases will survive after the HOD pre-steady-filter: i.e. return t.
;;; a. curvature = nil
;;; b. curvature = 0, but sd3-value = 0 and nil.
;;; (  Note here the sd3-value = + or - cases have been discarded. We assume these
;;;    cases are qualitatively similar to the corresponding cases with qdir = + and qdir = -.)
;;; c. curvature = -, increasing toward steady and curvature + decreasing toward steady. 

(defun pre-steady-filter (present-slope curvature sd3-value)
  (cond ((null curvature) t)
	((eql curvature 0)
	 (not (member sd3-value '(+ -)))) 
	(t (or (and (eql '- curvature)			
		    (eql 'inc present-slope))
	       (and (eql '+ curvature)			
		    (eql 'dec present-slope))))))

;Following cases will survive after the HOD post-steady-filter: i.e. return t.   
;;; a. curvature = nil  
;;; b. curvature= 0, and whenever sd3-value is untagged, i.e. (cadr sd3-list) = nil, call sd3-filter. 
;;; ( "Tagged case": For a coupled QDEs, when curvature = 0, sd3-value = 0
;;;    and (cadr sd3-list) = t -- the "tagged case", the filter returns nil.
;;;    This is due to the theorem: for coupled QDEs, the location where sd1 HOD = 0
;;;    and sd2 HOD = 0 can only occur at the quiescent point. So a post-steady
;;;    state is inadmissible here.)
;;; c. curvature = -, decreasing after steady and curvature + increasing after steady.

(defun post-steady-filter (next-slope curvature sd3-list)
  (let ((sd3-value (car sd3-list)))
    (cond((null curvature)t)
	 ((eql curvature 0)
	  (unless (and (eql sd3-value 0)
		       (cadr sd3-list))
	    (inflection-filter next-slope sd3-value)))
	 (t (or (and (eql '- curvature)			
		     (eql 'dec next-slope))
		(and (eql '+ curvature)
		     (eql 'inc next-slope)))))))

(defun inflection-filter (slope sd3-value)
  (or (and (not (eql '+ sd3-value))
	   (not (eql '- sd3-value)))
      (and (eql '- sd3-value)			
	   (eql 'dec slope))
      (and (eql '+ sd3-value)
	   (eql 'inc slope))))

;;;  GET-SD3-LIST is redefined at the end of this file using
;;;  David Dalle Molle's extended version.



;;; 3. Evaluate curvature at steady point:
;;; We assume curvature expression has a general form which involves parameters and 6 operators:

;;; (m+ x) is the Monotonically increasing function of x
;;; (add x y) = x + y
;;; (minus x) = - x
;;; (mult x y) = x*y
;;; (divide x y) = x/y
;;; (sd1 x) = (sign dx/dt)    (For the derivative operator, also use the symbols sd, or deriv.)

;;; See cast.lisp file for derivation of curvature expressions. There the divide operator has not been considered.
;;; Thus far the divide operator is used only for manual input. 

;;; Modified (9-19-89, BJK) to use a CASE statement, and to accept the syntax produced
;;; by my ACC algebra package.

(defun eval-cast(expression param-values state)
  (cond((null expression)nil)
       ((numberp expression)(sign-num expression))
       ((atom expression)(param-sign expression param-values state))
       (t(let ((operator (car expression)))
	   (case operator
	     ((sd sd1 deriv) (deriv-sign (cadr expression) param-values))
	     ((m+)           (eval-cast (cadr expression) param-values state))
	     ((minus)        (opposite-sign (eval-cast (cadr expression) param-values state)))
	     ((+ add)        (qadd (eval-cast (cadr expression) param-values state)
				   (eval-cast (caddr expression) param-values state)))
	     ((-)            (qadd (eval-cast (cadr expression) param-values state)
				   (opposite-sign (eval-cast (caddr expression) param-values state))))
	     ((* mult)       (qmult (eval-cast (cadr expression) param-values state)
				    (eval-cast (caddr expression) param-values state)))

	     ((/ divide)
	      (let ((denom (eval-cast (caddr expression) param-values state)))
		(cond ((eql denom 0)
		       (error "~%Curvature constraint:  Eval-cast denom = 0"))
		      (t (qmult (eval-cast (cadr expression) param-values state) denom)))))

	     ((^)            (let ((arg (eval-cast  (cadr expression) param-values state)))
			       (cond ((= (caddr expression) 2) (qmult arg arg))
				     ((= (caddr expression) 3) arg)
				     (t (error "Can't handle ~a." expression)))))
	     (t (error "new operator:~a in cast-expression:~a"operator expression)))))))

;;; Utilities for evaluating signs.

(defun sign-num(number)
  (cond ((= number 0)    0)
	((plusp number) '+)
	(t              '-)))

(defun param-sign (param param-values state)
  (let* ((param-name (variable-name param))
	 (qmag   (qmag (alookup param-name param-values)))
	 (qspace (alookup param-name (state-qspaces state))))
    (qmag-order qmag *zero-lmark* qspace)))
       
(defun deriv-sign (expression param-values)
  (cond ((atom expression)
	 (check-var expression)			; catch non-variable
	 (cond ((typep expression 'variable)
		(qdir-sign (qval-qdir (alookup (variable-name expression) param-values))))
	       (t (qdir-sign (qdir (alookup expression param-values))))))	; dead?
	(t (error "falling through"))))

(defun qadd(sign1 sign2)
  (cond((or (null sign1)(null sign2)) nil)
       ((and (eql sign1 '+)(eql sign2 '+)) '+)
       ((and (eql sign1 '+)(eql sign2  0)) '+)
       ((and (eql sign1 '+)(eql sign2 '-)) nil)
       ((and (eql sign1  0)(eql sign2 '+)) '+)
       ((and (eql sign1  0)(eql sign2  0))  0)
       ((and (eql sign1  0)(eql sign2 '-)) '-)
       ((and (eql sign1 '-)(eql sign2 '+)) nil)
       ((and (eql sign1 '-)(eql sign2  0)) '-)
       ((and (eql sign1 '-)(eql sign2 '-)) '-)
       (t (error "qadd- sign1:~a sign2:~a" sign1 sign2))))

(defun qmult(sign1 sign2)
  (cond((or (eql sign1 0) (eql sign2 0)) 0)
       ((or (null sign1)(null sign2)) nil)
       ((and (eql sign1 '+)(eql sign2 '+)) '+)
       ((and (eql sign1 '+)(eql sign2  0))  0)
       ((and (eql sign1 '+)(eql sign2 '-)) '-)
       ((and (eql sign1  0)(eql sign2 '+))  0)
       ((and (eql sign1  0)(eql sign2  0))  0)
       ((and (eql sign1  0)(eql sign2 '-))  0)
       ((and (eql sign1 '-)(eql sign2 '+)) '-)
       ((and (eql sign1 '-)(eql sign2  0))  0)
       ((and (eql sign1 '-)(eql sign2 '-)) '+)
       (t (error "qadd- sign1:~a sign2:~a" sign1 sign2))))
	   
(defun qdir-sign (qdir)
  (cond((eql qdir 'inc) '+)
       ((eql qdir 'std)  0)
       ((eql qdir 'dec) '-)))

(defun opposite-sign (sign)
  (cond ((null sign) nil)
	((eql sign '+) '-)
	((eql sign '-) '+)
	((eql sign 0)   0)
	(t (error "opposite-sign:~a" sign))))

(defun remove-indicator (lists)
  (mapcar #'(lambda (list)
	      (list (car list) (cadr list)))
	  lists))


;David T. Dalle Molle

;November 15, 1988

;This file contains code to allow the value of the sign of the 3rd order derivative 
;of a parameter (that has a curvature-at-steady expression) to be evaluated (sd3 param) 
;in some cases from the curvature-at-steady expressions already derived by the ACC 
;program or supplied in the model.

;When a curvature-at-steady clause evaluates to 0, then every term in the expression
;must have evaluated to 0.  Within eval-cast, a value of +, -, or 0 is eventually 
;returned by evaluating the param-sign or deriv-sign of a parameter.  To determine the 
;curvature of this expression (which has a value of 0), the curvature of each term is 
;evaluated.  For a parameter, deriv-sign will return the curvature of that parameter.  
;For a (sd1 param) term, the value of the param's derivative parameter is returned if it
;exists in the model.  If not,the curvature-at-steady clause for that param is evaluated
;if one exists.  If a curvature-at-steady expression does not exist, the sd3-value is 0, 
;and no additional filtering has been done.

;The function get-sd3-list is the only function in curvature.lisp that needs to be
;replaced to use the automatic evaluation of the sign of the 3rd derivative of a
;parameter (sd3 x).

;
(defun get-sd3-list (param param-values state)
  (let* ((clause (sublis (qde-var-alist (state-qde state))	; varnames -> vars
			 (qde-sd3-constraint (state-qde state))))
	 (param-clause (cdr (assoc param clause)))
	 (sd3-value nil))
    (cond ((and *auto-sd3-evaluation* (null (car param-clause)))
	   (let* ((cast-clauses (get-curvature-expressions (state-qde state)))
		  (param-cast-clause (assoc param cast-clauses
					    :test #'(lambda (p v) (eql p (variable-name v)))
					    )))
	     (cond ((null param-cast-clause) nil) ;not sure this could ever occur
		   (t (setq sd3-value (eval-cast-sd3 (cadr param-cast-clause)
						     param-values state))
		      (trace-sd3-value param sd3-value state)
		      (list sd3-value)))))
	  (t (cons (eval-cast (car param-clause) param-values state) (cdr param-clause))))))

(defun trace-sd3-value (param sd3-value state)
  (if *trace-auto-sd3-evaluation*
      (format *qsim-trace* "~%Automatically evaluated sd3(~a) = ~a at ~a."
	      param sd3-value state)))


;;; The following function evaluates the derivative of the curvature at the steady point.  
;;; The curvature expression is assumed to have a general form which involves parameters 
;;; and the 6 operators used in the ACC program.  The curvature of expressions based on 
;;; these operators is evaluated as follows:

;;; d/dt (add x y) = dx/dt + dy/dt
;;; d/dt (minus x) = - dx/dt
;;; d/dt (mult x y) = x*dy/dt + y*dx/dt
;;; d/dt (divide x y) = [y*dx/dt - x*dy/dt]/y*y
;;; d/dt (sd1 x) = (sign (or (sign-value of x's derivative parameter)
;;;                          (eval-cast (curvature-constraint-of dx/dt))))
;(For the derivative operator, also use the symbols sd, or deriv.)

;;; M+/M- constraints are treated the same way as in the ACC program
;;; d/dt (m+ x) dx/dt

;Caution:  Nested mult expressions may not evaluated properly.  The implementation may
;;;          need revising depending on how (deriv-sign (mult expression)) evaluates.  
;;;          At the moment, I'm not sure how it works.

;
(defun eval-cast-sd3 (expression param-values state)
  (cond ((null expression) nil)
	((numberp expression) 0)
	((atom expression) (param-sign-sd3 expression param-values))
	(t (let ((operator (car expression)))
	     (case operator
	       ((sd sd1 deriv) (deriv-sign-sd3 (cadr expression) param-values state))
	       ((m+)           (eval-cast-sd3 (cadr expression) param-values state))
	       ((minus)	       (opposite-sign (eval-cast-sd3 (cadr expression)
							     param-values state)))
	       ((add +)        (qadd (eval-cast-sd3 (cadr expression) param-values state)
				     (eval-cast-sd3 (caddr expression) param-values state)))
	       ((-)            (qadd (eval-cast-sd3 (cadr expression) param-values state)
				     (opposite-sign
				       (eval-cast-sd3 (caddr expression) param-values state))))
	       ((* mult)       (qadd (qmult (eval-cast-sd3 (cadr expression)
							   param-values state)
					    (eval-cast (caddr expression) param-values state))
				     (qmult (eval-cast (cadr expression) param-values state)
					    (eval-cast-sd3 (caddr expression)
							   param-values state))))
;;; No need to test for divide by zero since it's been done in eval-cast previously
	       ((/ divide)     (qadd (qmult (eval-cast-sd3 (cadr expression)
							   param-values state)
					    (eval-cast (caddr expression) param-values state))
				     (opposite-sign
				       (qmult (eval-cast (cadr expression) param-values state)
					      (eval-cast-sd3 (caddr expression)
							     param-values state)))))
	       ; d u^2 = 2 u u' ;    d u^3 = 3 u^2 u'
	       ((^)            (let ((arg (eval-cast-sd3 (cadr expression) param-values state)))
				 (cond ((= (caddr expression) 2)
					(qmult arg (eval-cast (cadr expression) param-values state)))
				       ((= (caddr expression) 3) arg)
				       (t (error "Can't handle ~a." expression)))))
	       (t (error "New operator: ~a in cast-sd3-expression:~a"
			 operator expression)))))))

;Return the qdir of a parameter (instead of its qmag).

(defun param-sign-sd3 (param param-values)
  (deriv-sign param param-values))


(defun deriv-sign-sd3 (expression param-values state)
  (cond ((atom expression)
	 (check-var expression)
	 (cond ((typep expression 'variable)
		(qdir-sign-sd3 (qval-qdir (alookup (variable-name expression) param-values))
			       expression
			       param-values
			       state))
	       (t (qdir-sign-sd3 (qdir (alookup expression param-values))     ;; -- DD
				 expression param-values state))))
	(t (let ((head (car expression))
		 (tail (cdr expression)))
	   (cond ((eql head 'minus)
		  (opposite-sign (deriv-sign-sd3 (car tail) param-values state)))
		 ((eql head 'add) (qadd (deriv-sign-sd3 (car tail) param-values state)
					(deriv-sign-sd3 (cadr tail) param-values state)))
		 ((eql head 'mult) (qmult (deriv-sign-sd3 (car tail) param-values state)
					  (deriv-sign-sd3 (cadr tail) param-values state)))

;Not sure that this clause can be used in sd3 evaluation.  Ask Charles Chiu.
;		 ((member head '(sd sd1 deriv))
;		  (caddr (cadr (assoc (car tail) param-values))))

		 (t (error "new operator:~a in deriv-expression-sd3:~a"
			   head expression)))))))

;If the qdir of a parameter is std, then evalutate 
;;; 1) qdir of param's deriv param if it exists, or
;;; 2) its curvature-at-steady expression, or
;;; 3) return nil.

(defun qdir-sign-sd3 (qdir param param-values state)
  (cond ((eql qdir 'inc) '+)
	((eql qdir 'dec) '-)
	((eql qdir 'std)
	 (let* ((cast-clauses (get-curvature-expressions (state-qde state)))
		(param-cast-clause (assoc param cast-clauses))
		param-deriv-qdir)
	   (cond ((setq param-deriv-qdir
			(get-and-eval-param-deriv-qdir param param-values state))
		  param-deriv-qdir)		;qdir of param's deriv param if it exists
		 ((null param-cast-clause) 0)   ;no cast clause for this param
		 (t (eval-cast (cadr param-cast-clause) param-values state)))
	   ))
	))



(defun GET-AND-EVAL-PARAM-DERIV-QDIR (var param-values state)
  (let* ((qde   (state-qde state))
	 (deriv (get-param-deriv var qde)))
    (cond ((null deriv) nil)
	  (t (deriv-sign deriv param-values)))))


(defun GET-PARAM-DERIV (var qde)
  (some #'(lambda (con)
	    (if (and (eql (first  (constraint-name con))   'd/dt)
		     (eql (first (constraint-variables con)) var))
		(second (constraint-variables con))))
	(alookup (qde-name qde)
		 (variable-constraints var))))
