;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: hq-plot.lisp,v 1.1 91/03/26 21:37:58 clancy Exp $

(in-package 'QSIM)

; Copyright (c) 1987, Benjamin Kuipers

(proclaim '(special Xbloc Ybloc Xbsize Ybsize
		   Lmargin Rmargin Xsep Ysep *Xsize* *Ysize*))

; Does two types of displays, starting from the root state of a tsa behavior:
;  - scan through, doing full displays of each behavior at each time-scale.
;  - do a slice-display of selected variables at all time-scales, along one behavior.


(defun HQD-display (state &optional slice-vars)
  (device-interface
    (when (y-or-n-p "Full plot of all behaviors? ")
      (HQD-full state))
    (when (and slice-vars
	       (y-or-n-p "Hierarchical slice display? "))
      (HQD-slices state slice-vars))))

; Print hierarchical behavior map.  (model for recursive scan)

(defun HQD (state &optional (label 'start))
  (do ((behaviors (get-behaviors state) (cdr behaviors)))
      ((null behaviors) t)
    (format *QSIM-Trace* "~%~a:  ~a has behavior ~a."
	    label (qde-name (state-qde state)) (mapcar #'state-name (car behaviors)))
    (let* ((final (car (last (car behaviors))))
	   (successors (filtered-successor-slot-contents final)))      ;;  used to call state-successor 02/14/91 DJC
      (cond ((not (eql (car successors) 'tsa-id)) nil)
	    (t (mapc #'(lambda (s) (HQD s 'faster))
		     (cdr (assoc 'faster-states (cdr successors))))
	       (mapc #'(lambda (s) (HQD s 'slower))
		     (cdr (assoc 'slower-states (cdr successors)))))))))

; Given a state and a partially-complete tsa-behavior, return a list of tsa-behaviors
; that are extensions of that one.
;      A tsa-behavior is a list of flat-behaviors.
;      A flat-behavior is a list of states.

(defun HQD-list (state &optional (prior-tsa-behavior nil))
  (do ((behaviors (get-behaviors state) (cdr behaviors))
       (L nil)
       (intermediate-behaviors nil)
       (current nil))
      ((null behaviors) L)
    (setq current (append prior-tsa-behavior
			  (list (car behaviors))))

    (let* ((final (car (last (car behaviors))))
	   (successors (filtered-successor-slot-contents final)))       ;;  used to call state-successor 02/14/91 DJC

      (cond ((and (eql (car successors) 'tsa-id)
		  (cdr (assoc 'faster-states (cdr successors))))
	     ; if there are tsa-faster states . . .
	     ; treat each like a one-state behavior
	     (setq intermediate-behaviors
		   (extend-tsa-behaviors-w-states
		     (cdr (assoc 'faster-states (cdr successors)))
		     (list current))))
	    (t (setq intermediate-behaviors (list current))))

      (cond ((and (eql (car successors) 'tsa-id)
		  (cdr (assoc 'slower-states (cdr successors))))
	     ; if there are tsa-slower states . . .
	     (setq L (append L (extend-tsa-behaviors-w-states
				 (cdr (assoc 'slower-states (cdr successors)))
				 intermediate-behaviors))))
	    (t (setq L (append L intermediate-behaviors))))
		   )))

;  => HQD-list has gotten crufty and awful (but it works!).  Fix it.

(defun extend-tsa-behaviors-w-states (states tsa-behaviors)
  (cond ((null states) tsa-behaviors)
	(t (extend-tsa-behaviors-w-states
	     (cdr states)
	     (mapcan #'(lambda (tsa-behavior)
			 (HQD-list (car states) tsa-behavior))
		     tsa-behaviors)))))


; HQD-full recurses through the TSA behavior tree, doing a full display of
; the behavior at each time-scale level.

(defun HQD-full (state &optional (label 'start))
  (do ((behaviors (get-behaviors state) (cdr behaviors)))
      ((null behaviors) t)
    (format *QSIM-Trace* "~%~a:  ~a has behavior ~a."
	    label (qde-name (state-qde state)) (car behaviors))
    (qplot-behavior (car behaviors))
    (let* ((final (car (last (car behaviors))))
	   (successors (filtered-successor-slot-contents final)))    ;;  used to call state-successor 02/14/91 DJC
      (cond ((not (eql (car successors) 'tsa-id)) nil)
	    (t (mapc #'(lambda (s) (HQD-full s 'faster))
		     (cdr (assoc 'faster-states (cdr successors))))
	       (mapc #'(lambda (s) (HQD-full s 'slower))
		     (cdr (assoc 'slower-states (cdr successors)))))))))

; QPLOT-BEHAVIOR is a slightly modified version of QSIM-DISPLAY, for displaying
; a single selected behavior.

(defun qplot-behavior (beh)
  (let* ((initial (car beh))
	 (qde (state-qde initial))		
	 (refpoints `((normal ,(get-normal-state qde))))
	 (layout (qde-layout qde))
	 (qspaces (qde-qspaces qde))
	 (behaviors (behavior-context initial))
	 (bpoint (member beh behaviors :test #'equal))
	 (btotal (length behaviors))
	 (bmax (apply #'max (mapcar #'length behaviors)))
	 (bnum (+ 1 (- btotal (length bpoint)))))
    (setq *initial-state* initial)
    (setq *reference-states* refpoints)
    (compute-layout layout)
    (cond ((y-or-n-p (format nil "~%Display behavior (~a/~a) for ~a?  "
			     bnum btotal (qde-name qde)))
	   (qplot-new-behavior)
	   (qplot-label beh bnum btotal)
	   (plot-state-tree (behavior-context-roots initial) bmax btotal
			     (- xscreen rmargin xbsize) ybloc xbsize ybsize bnum)
	   (display-behavior refpoints beh layout qspaces)
	   (qplot-end-display)))))

; Cycle through the possible slices, displaying one screen (selected variables) per slice.

(defun HQD-slices (state vars)
  (do ((tsa-behaviors (HQD-list state) (cdr tsa-behaviors))
       (n 1 (+ n 1)))
      ((null tsa-behaviors) t)
    (if (y-or-n-p "Display Qslice behavior ~a? " n)
	(HQD-slice vars (car tsa-behaviors)))))

; QSLICE:  displays a slice of each behavior in the hierarchy, on a single screen

(defvar hq-lmargin 100.)
(defvar hq-rmargin 100.)
(defvar hq-xsep 100.)

(defun HQD-slice (vars tsa-behavior)
  (let ((lmargin-save lmargin)
	(rmargin-save rmargin)
	(xsep-save xsep))
    (unwind-protect				; save and restore margins
	(progn
	  (setq lmargin hq-lmargin
		rmargin hq-rmargin
		xsep hq-xsep)
	  (compute-rectangular-layout (+ 1 (length vars)) (length tsa-behavior))
	  (qplot-new-behavior)
	  (do ((xpos lmargin (+ xpos *xsize* xsep))
	       (L tsa-behavior (cdr L))
	       (behavior nil)
	       (state-names nil)
	       (mech nil))
	      ((null L) t)
	    (setq behavior (car L))
	    (setq state-names (mapcar #'state-name behavior))
	    (setq mech (qde-name (state-qde (car behavior))))
	    (qslice-behavior vars behavior state-names mech xpos *xsize* *ysize*))
	  (qplot-end-display))
      (setq lmargin lmargin-save
	    rmargin rmargin-save
	    xsep xsep-save))))

; Plot one column of the qslice display.

(defun qslice-behavior (vars beh states name xpos xsize ysize)
  (let* ((initial (car beh))
	 (qde (state-qde initial))		
	 (refpoints `((normal ,(get-normal-state qde))))
	 (xalloc (allocate-h-axis refpoints beh xsize))
	 (qspaces (qde-qspaces qde))
	 (behaviors (behavior-context initial))
	 (bpoint (member beh behaviors :test #'equal))
	 (btotal (length behaviors))
	 (bmax (apply #'max (mapcar #'length behaviors)))
	 (bnum (+ 1 (- btotal (length bpoint)))))
    (setq *initial-state* initial)
    (setq *reference-states* refpoints)
    (qplot-box-label xpos 20. name)
    (qplot-box-label xpos 35. states)
    (plot-state-tree (behavior-context-roots initial) bmax btotal xpos 60. xsize 40. bnum)
    (do ((L vars (cdr L))
	 (ypos 100. (+ ypos ysize ysep)))
	((null L))
      (or (not (assoc (car L) qspaces))
	  (plot-parameter (car L) refpoints beh xpos ypos xsize ysize xalloc)))
    ))

; When dealing with completions of an incomplete state, get the right context,
; and the roots from which to plot the state tree.

(defun behavior-context (initial)
  (if (eql (car (state-justification initial))
	   'one-of-several-completions-of)
      (get-behaviors (cadr (state-justification initial)))
      (get-behaviors initial)))

(defun behavior-context-roots (initial)
  (if (eql (car (state-justification initial))
	   'one-of-several-completions-of)
                              ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
      (successor-states (cadr (state-justification initial)))
      (list initial)))

