;;; -*- Mode:Common-Lisp; Package:qsim; Syntax:common-lisp; Base:10 -*-
;;;  $Id: hod-derivation.lisp,v 1.3 91/07/08 15:51:30 clancy Exp Locker: farquhar $

; Copyright 1989 Benjamin Kuipers

(in-package :qsim)

;;; MODIFICATIONS
;;;
;;; 16 April 1990.  Adam Farquhar.
;;;
;;; This file has been updated 16 April 1990 to take into account the =
;;; constraint, the (= x (constant y)) syntax, as well as having some
;;; minor bugs in the rules fixed, and some new transformation rules
;;; added.
;;; 
;;; 4 January 1990.  Ben Kuipers.
;;;    Updated derivation of equivalence classes of chattering variables.
;;; 

;
; QDE.other.chattering-variables is a slot for the user to assert variables that chatter.
; QDE.other.derived-chvars holds the variables for which curvature constraints will be
;     derived.  They are the union of user-asserted and automatically identified ones.
; QDE.other.curvature-at-steady hold user-asserted expressions for (sd2 <var>).
; QDE.other.derived-sd2-expressions hold the union of the user-asserted and derived
;     expressions for (sd2 <var>).
; QDE.other.sd3-constraint holds user-asserted (sd3 <var>) expressions.
;
;             These defother forms are in defothers.lisp:
;(defother qde chattering-variables)		; user assert chattering variables for QDE
;(defother qde derived-chvars)			; merge w results of automatic identification
;(defother qde derived-chvar-classes)		; Equivalence classes of chattering vars.
;(defother qde curvature-at-steady)		; alist of (<var> <sd2-exp>)
;(defother qde derived-sd2-expressions)		;   derived alist of (<var> <sd2-exp>)
;(defother qde sd3-constraint)     		; alist of (<var> <sd3-exp>)

(defun delete-hod-clauses (qde)
  "Remove or set to nil all of the clauses used for HOD derivation.
   This will force rederivation of the clauses, otherwise the HOD
   will reuse the existing ones."
  (setf (qde-chattering-variables qde) nil)
  (setf (qde-derived-chvars qde) nil)
  (setf (qde-curvature-at-steady qde) nil)
  (setf (qde-derived-sd2-expressions qde) nil)
  (setf (qde-sd3-constraint qde) nil))

; Eventually, the curvature expressions should be stored in QDE.curvatures:
;     ((= (sd2 <var>) <exp>)
;     ((= (sd3 <var>) <exp>)

; Control switches.

(defparameter *derive-curvature-constraints* t)	; automatically derive HOD constraint
(defparameter *chiu-acc-derivation* nil)	; use Charles Chiu's ACC derivation method
(defparameter *search-state-limit* 30)		; resource limit for transformation search
(defparameter *search-type* 'best-first)	; depth-first | breadth-first | best-first

(defparameter *trace-successful-rules* nil)
(defparameter *trace-children* nil)
(defparameter *trace-sd2-derivation* t)
(defparameter *trace-chatter-vars* t)		; trace identification of chatter vars
(defparameter *trace-eqc* nil)			; trace equivalence class creation
; (defparameter *trace-acc-filter-application* nil) ; Moved to Qdefs.lisp
(defparameter *trace-AOnode-creation* nil)	; Trace node creation in expression search


; Find-or-create retrieval of sd2 expressions.
;  - if they are stored in QDE.derived-sd2-expressions, derivation has already been done.
;  - otherwise translate and assert user-provided expressions in QDE.curvature-at-steady.
;  - then try automatic derivation of expressions.

(defun get-curvature-expressions (qde)
  (or (qde-derived-sd2-expressions qde)
      (let ((temp (qde-curvature-at-steady qde)))
	(if temp (setf (qde-derived-sd2-expressions qde)
		       (sublis (qde-var-alist qde) temp)))
	(if *derive-curvature-constraints*
	    (derive-curvature-constraints qde))))
  (qde-derived-sd2-expressions qde))


; For each <var> in QDE.derived-chvars, derive an expression for (sd2 <var>).
; Store the result in QDE.derived-sd2-constraints.

(defun derive-curvature-constraints (qde)
  (if *derive-curvature-constraints*
      (let ((chvars (get-chattering-variables qde)))
	(cond ((eql chvars 'none) nil)
	      (t (dolist (chvar chvars)
		   (curvature-constraint qde chvar))
		 t)))))

; Compute the curvature constraint for a given variable, from a given QDE.
;  (unless there is already an expression stored for it)

(defun curvature-constraint (qde chvar)
  (check-var chvar)
  (or (lookup chvar (qde-derived-sd2-expressions qde))
      (let ((*constraints* (qde-constraints qde))
	    (*independents* (sublis (qde-var-alist qde) (qde-independent qde)))
	    (*qde* qde)
	    (*chatter-var* chvar))
	(declare (special *qde* *constraints* *independents* *chatter-var*))
	(let ((curv-con nil))
	  (trace-sd2-derivation-start chvar curv-con)
;	  (setq curv-con (AO-simplify (list 'sd2 chvar)))
	  (setq curv-con (AO-search (list 'sd2 chvar)))	; testing new method
	  (trace-sd2-derivation-end chvar curv-con)
	  (cond (t ;curv-con
		 (pushnew (list chvar curv-con)
			  (qde-derived-sd2-expressions qde)
			  :test #'equal) ))))))


; The following improved identification of chattering variables (4 pages)
; was added by BJK, 1-4-91.

; The user may specify variables alleged to chatter in 
;      ->  QDE.chattering-variables   =  set of variable-names
; The program determined equivalence classes of potentially chattering variables
;      ->  QDE.derived-chvar-classes  =  set of sets of variables
; The program adds a representative of each equivalence class to the user-asserted variables.
;      ->  QDE.derived-chvars.
; HOD constraints are then derived for these variables.

(defun get-chattering-variables (qde)
  (or (qde-derived-chvars qde)
      (identify-chatter qde)))

(defun identify-chatter (qde)
  (let* ((eqc (chatter-equivalence-classes qde))
	 (constraints (qde-constraints qde))
	 (known-chvars (sublis (qde-var-alist qde) (qde-chattering-variables qde))))
    (setf (qde-derived-chvar-classes qde) eqc)
    (setf (qde-derived-chvars qde)
	  (or (union (mapcar #'(lambda (class)
				 (representative-from-eqc class constraints known-chvars))
			     eqc)
		     known-chvars)
	      'none))
    (trace-chatter-vars qde)
    (qde-derived-chvars qde)))

; Select a representative from an equivalence class
;   - the single element if there is only one.
;   - if an element from this class is already designated, stick with it.
;   - if some element is a derivative, that element.
;   - otherwise the first element.

(defun representative-from-eqc (eqc constraints &optional (designated nil))
  (cond ((= (length eqc) 1) (first eqc))
	((dolist (var eqc)
	   (if (member var designated) (return var))))
	((dolist (var eqc)
	   (if (dolist (con constraints)
		 (if (and (eql (contype-name (constraint-type con)) 'd/dt)
			  (eql var (second (constraint-variables con))))
		     (return t)))
	       (return var))))
	(t (first eqc))))

; Return equivalence classes of potentially chattering variables.
;  - Initialize equivalence classes, sets of constants and non-chattering variables.
;  - A whole equivalence class is constant, or non-chattering, if one element is.
;  - Two classes are merged if linked by a 3-arg constraint with one constraint constant.
;  - Repeat the cycle until nothing changes.

(defun chatter-equivalence-classes (qde)
  (let ((eqc (sort-classes (initial-eqc-classes qde)
			   (make-initial-constant-list qde)
			   (make-initial-nonchat-list qde)
			   qde)))
    (trace-eqc-final eqc qde)
    eqc))

(defun sort-classes (eqc constants nonchat qde)
  (let ((ne (length eqc))
	(nc (length constants))
	(nn (length nonchat)))

    ;; First, identify classes containing constant or non-chattering variables.

    (dolist (class eqc)
      (when (intersection class constants)
	(trace-eqc-new-constants class constants)
	(setq constants (union class constants)))
      (when (intersection class nonchat)
	(trace-eqc-new-nonchats class nonchat)
	(setq nonchat (union class nonchat))))

    ;; Second, merge classes if linked by a constraint with only two non-constant args.
    ;; Assert a constant if a constraint has only one non-constant arg.

    (dolist (con (qde-constraints qde))
      (case (contype-name (constraint-type con))

	((add mult sum-zero)
	 (let ((realvars (remove-if #'(lambda (v) (member v constants))
				    (constraint-variables con))))
	   (cond ((= (length realvars) 1)
		  (pushnew (car realvars) constants)
		  (trace-eqc-con3a con realvars))

		 ((= (length realvars) 2)
		  (setq eqc (merge-classes (eqv-class-of (first realvars) eqc)
					   (eqv-class-of (second realvars) eqc)
					   eqc))
		  (trace-eqc-con3 con realvars)))))))

    (unless (and (= ne (length eqc))
		 (= nc (length constants))
		 (= nn (length nonchat)))
      (sort-classes eqc constants nonchat qde))		

    (remove-if #'(lambda (c) (or (intersection c nonchat)
				 (intersection c constants)))
	       eqc)))

; Given a set of classes, return the class holding a given element.

(defun eqv-class-of (x eqc)
  (car (member x eqc :test #'member)))

; Merge two classes by destructively modifying the data structure.

(defun merge-classes (a b eqc)
  (cond ((eql a b) eqc)
	(t (nconc a b)
	   (delete b eqc))))

; Initialize:
;   - The initial equivalence classes are the variables, linked by 2-arg constraints.
;   - The initial non-chattering variables are those constrained by a d/dt constraint.
;   - The initial constant variables are those explicitly constrained to be constant.

(defun initial-eqc-classes (qde)		; initial set of equivalence classes
  (let ((eqc (mapcar #'list (cdr (qde-variables qde)))))
    (dolist (con (qde-constraints qde))
      (case (contype-name (constraint-type con))

	((= M+ M- minus equal)
	 (setq eqc (merge-classes (eqv-class-of (first (constraint-variables con)) eqc)
				  (eqv-class-of (second (constraint-variables con)) eqc)
				  eqc)))))
    (trace-initial-eqcs eqc)
    eqc))

(defun make-initial-nonchat-list (qde)		; initial set of nonchattering vars
  (let ((nonchats nil))
    (dolist (con (qde-constraints qde))
      (cond ((eql (contype-name (constraint-type con)) 'd/dt)
	     (pushnew (first (constraint-variables con)) nonchats)
	     )))
    (trace-eqc-initial-nonchats nonchats)
    nonchats))

(defun make-initial-constant-list (qde)		; initial set of constant vars
  (let ((constants nil))
    (dolist (var (qde-variables qde))
      (when (or (variable-independent-p var)
                (variable-discrete-p var))     ;; treats discrete variables as constants.  Made by DJC
        (pushnew var constants)))              ;; for Dan Dvorak 07/08/91
    (dolist (con (qde-constraints qde))
      (when (constant-constraint-p con)
	(pushnew (first (constraint-variables con)) constants)))
    (trace-eqc-initial-constants constants)
    constants))

(defun constant-constraint-p (con)
  (let ((cname (contype-name (constraint-type con))))
    (or (member cname '(constant zero-std positive-std negative-std))
	(and (eql '= cname)
	     (consp (third (constraint-name con)))
	     (eql 'constant (first (third (constraint-name con))))))))

; Trace functions for EQC:  equivalence classes of chattering variables.

(defun trace-eqc-initial-constants (vars)
  (when *trace-eqc*
    (format *QSIM-trace* "~%Initial list of constants:  ~a." vars)))

(defun trace-eqc-initial-nonchats (vars)
  (when *trace-eqc*
    (format *QSIM-trace* "~%Initial list of nonchats:  ~a." vars)))

(defun trace-initial-eqcs (eqc)
  (when *trace-eqc*
    (format *QSIM-trace* "~2%Initial equivalence classes: ")
    (dolist (c eqc)
      (format *QSIM-trace* "~%    ~a " c))))

(defun trace-eqc-con3a (con var)
  (when *trace-eqc*
    (format *QSIM-trace* "~2%Identify new constant ~a from ~a."
	    var (constraint-name con))))

(defun trace-eqc-con3 (con realvars)
  (when *trace-eqc*
    (format *QSIM-trace* "~2%Merging classes of ~a because of ~a."
	    realvars (constraint-name con))))

(defun trace-eqc-new-constants (class constants)
  (when (and *trace-eqc*
	     (not (subsetp class constants)))
    (format *QSIM-trace* "~2%Merging constants ~a into ~a."
	    class constants)))

(defun trace-eqc-new-nonchats (class nonchats)
  (when (and *trace-eqc*
	     (not (subsetp class nonchats)))
    (format *QSIM-trace* "~2%Merging nonchats ~a into ~a."
	    class nonchats)))

(defun trace-eqc-final (eqc qde)
  (when *trace-eqc*
    (format *QSIM-trace* "~2%~a has ~a variables and ~a equivalence classes including ~a possibly-chattering variables."
	    qde (length (cdr (qde-variables qde))) (length eqc) (apply '+ (mapcar #'length eqc)))
    (dolist (c eqc)
      (format *QSIM-trace* "~%    ~a" c))))


; Experimenting with AO-search [Nilsson, 1980, pp.104-105].

; The intention is that AOnode.expression should only be of the form (SD2 <var>).

(defstruct (AOnode (:print-function AO-printer))
  (expression nil)				; expression of the form (SD2 <var>)
  (solved nil)					; NIL | T  --  has this node been solved?
  (solution nil)				; if so, the solution.
  (successors nil)				; alist of (<exp> <alist of (<subexp> <node>)>)
  (parents nil)					; set of nodes (parent pointers may be cyclic)
  (name (genname 'AO))
  )

(defun build-AOnode (exp &optional parent)
  (let ((node (make-AOnode :expression exp
			   :parents (if parent (list parent)))))
    (set (AOnode-name node) node)
    node))

(defparameter *AOnode-index* nil)		; alist of (<exp> <node>)

(defun AO-printer (AOnode stream ignore)
  (declare (ignore ignore))
  (format stream "#<~a: ~a>" (AOnode-name AOnode) (AOnode-expression AOnode)))


; Trace functions.

(defun trace-successful-rule (rule exp result)
  (if *trace-successful-rules*
      (format *qsim-trace* "~%Transform ~a ~%  to ~a ~%  by rule ~a." exp result rule)))

(defun trace-AO-successors (AOnode)
  (when *trace-children*
    (format *qsim-trace* "~2%The successors of ~a = ~a:"
	    (AOnode-name AOnode) (AOnode-expression AOnode))
    (dolist (entry (AOnode-successors AOnode))
      (format *qsim-trace* "~%  = ~a  ~a"
	      (car entry)
	      (mapcar #'(lambda (pair) (AOnode-name (cadr pair)))
		      (cadr entry))))))

(defun trace-sd2-derivation-start (chvar exp)
  (declare (ignore exp))
  (if *trace-sd2-derivation*
      (format *qsim-trace* "~%Curvature constraint (SD2 ~a) = " chvar)))

(defun trace-sd2-derivation-end (chvar exp)
  (declare (ignore chvar))
  (if *trace-sd2-derivation*
      (format *qsim-trace* " ~a." exp)))

(defun trace-chatter-vars (qde)
  (if *trace-chatter-vars*
      (format *qsim-trace* "~%Variables in ~a likely to chatter are ~a." qde (qde-derived-chvars qde))))

(defun trace-AOnode-creation-A (parent)
  (if *trace-AOnode-creation*
      (format *QSIM-trace* "~%AO successor to ~a:  " parent)))

(defun trace-AOnode-creation-B (child foundp)
  (if *trace-AOnode-creation*
      (format *QSIM-trace* "~a  (~a):  " child foundp)))




;   Create a node representing the initial (SD2 <var>) expression.
;   Repeat:
;       Select an unexpanded node.
;       Expand it.  (Solutions will propagate upward.)
;       Check to see whether the initial node is now solved.

; AO-search

(defun AO-search (exp)
  (setq *AOnode-index* nil)
  (let* ((top-node (build-AOnode exp))
	 (queue (list top-node)))		; queue of nodes
    (do ((resources *search-state-limit* (- resources 1))
	 (focus (car queue) (car queue)))
	((AOnode-solved top-node) (AOnode-solution top-node))
     ;(format *QSIM-trace* ".")
      (cond ((null queue) (return nil))
	    ((AOnode-successors focus) (setq queue (cdr queue)))
	    ((AOnode-solved focus)     (setq queue (cdr queue)))
	    (t (AO-expand focus)
	       (trace-AO-successors focus)
	       (setq queue (extend-AO-queue focus (cdr queue))))))))

; Search of the AO-graph is exhaustive.  No effort is yet made to manage the queue sensibly.

(defun extend-AO-queue (focus queue)
  (union (mapcan #'(lambda (entry)
		     (mapcan #'(lambda (pair) (unless (or (AOnode-solved (cadr pair))
							  (AOnode-successors (cadr pair)))
						(list (cadr pair))))
			     (cadr entry)))
		 (AOnode-successors focus))
	 queue))

; The successors to a node exist at two levels:
;  - the OR-links are the alternate equivalent expressions produced by QA-TRANSFORM.
;      * if an expression is free of SD2 terms, the current node is solved.
;      * in that case, propagate the solved label upward to parents.
;  - each expression is an AND-link of its (SD2 <var>) subexpressions.
;  - create a new node for each (SD2 <var>) subexpression.

(defun AO-expand (AOnode)
  (let ((*transforms* nil)
	(exp (AOnode-expression AOnode)))
    (declare (special *transforms*))
    (qa-transform exp exp)
    (setq *transforms* (sort *transforms*
			     #'(lambda (a b) (< (size* a) (size* b)))))

    (dolist (new-exp *transforms*)
      (when (not (member* 'sd2 new-exp))
	(assert-solved AOnode new-exp)))

    (setf (AOnode-successors AOnode)
	  (mapcar #'(lambda (exp) (list exp (AOnodes-from-exp exp AOnode)))
		  *transforms*))
    (propagate-solved AOnode)			; check for new link to solved node
						; BJK:  1-4-91
    (length (AOnode-successors AOnode))))

(defun AOnodes-from-exp (exp parent)			; => alist of (<subexp> <node>)
  (cond ((atom exp) nil)
	((eql (car exp) 'SD2)
	 (list (list exp (find-or-create-AOnode-expression exp parent))))
 	(t (mapcan #'(lambda (e) (AOnodes-from-exp e parent))
		   exp))))

(defun find-or-create-AOnode-expression (exp parent)
  (cond ((atom exp) nil)
	((eql (car exp) 'SD2)
	 (let ((node (lookup exp *AOnode-index* :test #'equal)))
	   (trace-AOnode-creation-A parent)
	   (cond (node (setf (AOnode-parents node) (cons parent (AOnode-parents node)))
		       (trace-AOnode-creation-B node "found"))
		 (t (setq node (build-AOnode exp parent))
		    (trace-AOnode-creation-B node "created")
		    (push (list exp node) *AOnode-index*)))
	   node))
	(t (error "Bad AOnode expression:  ~a" exp))))

; Once a node is solved, propagate the solution upward to its parents,
; substituting in the expression and simplifying.

(defun assert-solved (node exp)
  (setf (AOnode-solved node) T)
  (setf (AOnode-solution node) exp)
  (dolist (parent (AOnode-parents node))
    (propagate-solved parent))
  t)

(defun propagate-solved (node)
  (unless (AOnode-solved node)
    (dolist (or-link (AOnode-successors node))
      (when (every #'(lambda (pair) (AOnode-solved (cadr pair)))
		   (cadr or-link))
	(assert-solved node
		       (qa-simplify (subst-solutions (car or-link) (cadr or-link))))
	(return node)))))

(defun subst-solutions (exp alist)
  (cond ((atom exp) exp)
	((eql (car exp) 'SD2)
	 (let ((soln (lookup exp alist :test #'equal)))
	   (cond (soln (AOnode-solution soln))
		 (t exp))))
	(t (mapcar #'(lambda (e) (subst-solutions e alist)) exp))))

; Transform generates all children after a single transformation:

(defun qa-transform (fullexp part)
  (declare (special *transforms* *transformation-rules*))
  (dolist (rule *transformation-rules*)
    (let ((result (apply-crule rule part)))
      (if result
	  (pushnew (qa-simplify (subst result part fullexp))
		   *transforms* :test #'equalp))))
  (if (listp part)
      (dolist (piece part)
	(qa-transform fullexp piece)))
  (length *transforms*))

; Apply obligatory, identity-based simplification rules, first top-down, then bottom-up.

(defun qa-simplify (exp)
  (declare (special *identity-rules*))
  (cond ((atom exp) exp)
	((variable? exp) exp)
	(t (dolist (rule *identity-rules*)
	     (setq exp (or (apply-crule rule exp)
			   exp)))
	   (if (listp exp)
	       (setq exp (mapcar #'qa-simplify exp)))
	   (dolist (rule *identity-rules*)
	     (if (listp exp)
		 (setq exp (or (apply-crule rule exp)
			       exp))))
	   exp)))

; Apply a rule to transform old-exp -> new-exp.  Failure returns nil.
;  - first clause matches the expression
;  - subsequent clauses match constraints, or independent, or HOD, or are Lisp-evalled.

(defun apply-crule (rule exp)
  (declare (special *qde* *constraints* *independents* *chatter-var*))
  (loop with alist = (match (first rule) exp)	; Does the rule bind?
	while (not (eq alist 'failed))
        for (clause . restclauses) on (rest rule)
	when (eql clause '->)
	  do (let ((result (substitute-bindings (first restclauses) alist)))
		(trace-successful-rule rule exp result)
		(return-from apply-crule result))
	unless (ecase (first clause)
		 ((= M+ M- add mult minus equal d/dt constant)
		  (setq alist (find-matching-constraint alist clause *qde*)))
		 (=-constant
		   (dolist (con (qde-constraints *qde*))
		     (when (and (constant-constraint-p con)
				(eql (first (constraint-variables con))
				     (lookup* (second clause) alist)))
		       (return t)))) 
		 (independent
		   (member (lookup* (second clause) alist) *independents*))
		 (chatter
		   (eql (lookup* (second clause) alist) *chatter-var*)))
	  do (return-from apply-crule nil)))

;;; FIND-MATCHING-CONSTRAINT returns the bindings of rule-variables
;;; [e.g., ?x, from the transform rules] bound to the variables from the
;;; first constraint which matches the clause of the transform rule's
;;; pattern.  For example: if APPLY-CRULE were given the exp (sd2 AMOUNT) to
;;; transform, it would match on ((sd2 ?x) (M+ ?x ?y) -> (sd2 ?y)).
;;; Then ?x would be bound to AMOUNT (by the MATCH in APPLY-CRULE). The
;;; alist passed to FIND-MATCHING-CONSTRAINT would be ((?x . AMOUNT)),
;;; and clause would be (M+ ?x ?y).  

;;; Then, if FIND-MATCHING-CONSTRAINT found a constraint of the form (M+ AMOUNT
;;; SIGNAL), it would return a resulting alist of ((?x AMOUNT)(?y SIGNAL)).  If
;;; there were another constraint (M+ AMOUNT PRESSURE), it would not be found in
;;; the current implementation.
 
(defun find-matching-constraint (alist clause qde)
  (dolist (con (qde-constraints qde))		; bind variables if possible
    (when (eql (first clause)			; with first constraint which will 
	       (contype-name (constraint-type con)))	; satisfy test
      (let ((result (match (rest clause)
			   (constraint-variables con)
			   alist)))
	(unless (eql result 'failed)
	  (return result))))))


; These are the transformation rules for deriving Higher-Order Derivative constraints
; (sometimes called Curvature-at-Steady Constraints).
;   - The first clause in the rule is matched against the sd2 expression.
;   - Additional clauses before "->" are matched against QDE constraints, after substitutions.
;   - The clause after the "->" has bindings substituted, and is returned.

(defparameter *transformation-rules*
	      '(((sd2 ?x) (M+ ?x ?y) -> (sd2 ?y))
		((sd2 ?y) (M+ ?x ?y) -> (sd2 ?x))
		
		((sd2 ?x) (= ?x ?y) -> (sd2 ?y))
		((sd2 ?y) (= ?x ?y) -> (sd2 ?x))
		
		((sd2 ?x) (equal ?x ?y) -> (sd2 ?y))	; BJK:  10-25-90
		((sd2 ?y) (equal ?x ?y) -> (sd2 ?x))
		;
		((sd2 ?x) (M- ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (M- ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?z) (add ?x ?y ?z) -> (+ (sd2 ?x) (sd2 ?y)))
		((sd2 ?x) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?y)))
		((sd2 ?y) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?x)))
		;
		((sd2 ?z) (mult ?x ?y ?z) -> (+ (* ?y (sd2 ?x))
						(+ (* ?x (sd2 ?y))
						   (* 2 (* (sd1 ?x) (sd1 ?y))))))
		((sd2 ?x) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?y)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?y) (^ ?y 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?y) 2)
								    (^ ?y 3))))
						      (* ?z (/ (sd2 ?z) (^ ?y 2)))))))
		((sd2 ?y) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?x)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?x) (^ ?x 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?x) 2)
								    (^ ?x 3))))
						      (* ?z (/ (sd2 ?z) (^ ?x 2)))))))
		;
		((sd2 ?x) (minus ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (minus ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?x) (d/dt ?x ?y) -> (sd1 ?y))
		))
; => extend to make these state-dependent, and handle S+/S-.

(defparameter *identity-rules*
	      '(((+ 0 ?x) -> ?x)		; additive and multiplicative identities
		((+ ?x 0) -> ?x)
		((- ?x 0) -> ?x)
		((- ?x ?x) -> 0)
		((* 0 ?x) -> 0)
		((* ?x 0) -> 0)
		((/ 0 ?x) -> 0)
		((^ 0 ?x) -> 0)
		((* 1 ?x) -> ?x)
		((* ?x 1) -> ?x)
		((/ ?x 1) -> ?x)
		((^ ?x 0) -> 1)
		((^ ?x 1) -> ?x)

		((- 0 (- 0 ?x)) -> ?x)		; strict simplifications
		((- 0 (- ?x ?y)) -> (- ?y ?x))
		((- 0 (+ (- 0 ?x) ?y)) -> (- ?x ?y))
		((- ?x (- 0 ?y)) -> (+ ?x ?y))
		((- ?x (- ?x ?y)) -> ?y)
		((- ?x (+ ?x ?y)) -> (- 0 ?y))
		((- ?x (+ ?y ?x)) -> (- 0 ?y))
		((- (+ ?x ?y) ?x) -> ?y)
		((- (+ ?y ?x) ?x) -> ?y)
		((+ ?x (- ?y ?x)) -> ?y)
		((+ (- ?y ?x) ?x) -> ?y)
		((- (- ?x ?y) ?x) -> (- 0 ?y))

		((+ ?x ?x) -> (* 2 ?x))		; preferable form
		((+ (+ ?x ?y) ?z) -> (+ ?x (+ ?y ?z)))
		((- (- ?x ?y) ?z) -> (- ?x (+ ?y ?z)))
		((* (* ?x ?y) ?z) -> (* ?x (* ?y ?z)))
		;
		((sd2 ?x) (constant ?x) -> 0)
		((sd1 ?x) (constant ?x) -> 0)	        ; BJK:  1-4-90
		((sd2 ?x) (independent ?x) -> 0)	; final simplifications
		((sd1 ?x) (chatter ?x) -> 0)
		;; Hack for (= x (constant m))
		((sd2 ?x) (=-constant ?x) -> 0)
		))

; There are significant problems with this approach to deriving curvature constraints.
;
;  1.  The heuristic search is poorly guided.       (FIXED)
;      - It would be improved by being organized as an And-Or search, so that
;        transformations to subexpressions would not be duplicated.
;      - Try viewing it as a Means-Ends analysis problem.
;
;  2.  The match is not indexed, but simply iterates through the list of rules.
;      The match and transformations does a great deal of consing.

; Utilities

(defun member* (x S)				; does x appear anywhere in S?
  (cond ((eql x S) T)
	((atom S) nil)
	((member* x (car S)) t)
	((member* x (cdr S)) t)))

(defun size* (S)
  (cond ((atom S) 1)
	(t (let ((N (length S)))
	     (dolist (item S N)
	       (setq N (+ N (size* item))))))))

(defun count-atoms (x S)
  (cond ((atom S) (if (eql x S) 1 0))
	(t (+ (count-atoms x (car S))
	      (count-atoms x (cdr S))))))
