;;; -*- Mode:Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: global-filters.lisp,v 1.9 92/07/08 16:38:30 clancy Exp $
;;;  Copyright (c) 1987, 1988 by Benjamin Kuipers.

(in-package 'QSIM)


;;;=============================================================================
;;;
;;;                        G L O B A L    F I L T E R S
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  Global filters are of three types:
;;;
;;;  1.  "Global alist filters" eliminate alists before states are built
;;;      (e.g. NO-CHANGE).
;;;      ---> called on STATE and ALIST (ALIST is a possible successor to STATE).
;;;      ===> returns [ALIST | nil]
;;;      
;;;  2.  "Global state analysis" transforms the current state description
;;;      (e.g. NEW-LANDMARKS).
;;;      ---> called on a STATE (for side-effects on the state).
;;;      ===> returns STATE
;;;
;;;  3.  "Global state filters" identify important properties of the state
;;;      (e.g. QUIESCENCE, INCONSISTENT).
;;;      ---> called on STATE. They should return nil if the state is
;;;           somehow inconsistent, or can be absolutely eliminated.
;;;      ===> returns [STATE | nil]
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  Notes:  -- QSIM examines state.status to decide what goes on the agenda.
;;;
;;;          -- ALIST refers to a set of value completions of the form 
;;;             ((<var1> . <qval1>)  (<var2> . <qval2>)  ...).
;;;
;;;          -- A good way to add an experimental filter to Q is to push a new
;;;             filter onto the beginning or end of one of the lists below.
;;;
;;;  Warning:   Currently, global filters are called while the constraint
;;;             network is still "set up" from the parent state.  In particular,
;;;             the constraint-active-p value is assumed to be correctly set
;;;             for the current filtering operations.  If, in the future,
;;;             someone wants to call a global filter on a state *after* it has
;;;             been popped from the qsim agenda (rather than before it was
;;;             pushed onto the agenda), then that caller will have to ensure
;;;             that the constraint-active-p values are set.
;;;
;;;=============================================================================


(defparameter *global-alist-filters*
	      '(no-change-filter1		; no change (t0 t1) -> t1
		 analytic-function-filter	; only constants may be steady on interval
		 inconsistent-ign-filter	; detect unsatisfiable uses of the IGN qdir
		 curvature-at-steady-filter	; apply curvature constraints
		 ))

(defparameter *global-state-analysis*
	      '(make-new-landmarks		; create new landmarks (3 types)
		make-new-corresponding-values	; identify and record new corr. vals.
		check-for-cycle		; identify cyclic behavior
		quiescence-marker		; identify quiescent states
		 ;;;;;	 check-for-abstracted-corresponding-values
						; corr. vals on abstracted constraints
		stability-label		        ; test stability of quiescent states
		                                ; SHOULD BE THE LAST FILTER
		 ))

(defparameter *global-state-filters*
	      '(label-infinite-times		; identify when we know t=inf or t<inf
		 apply-time-label		; filter on inconsistent t=inf label
		 filter-unreachable-values
		 energy-global-filter		; Kinetic Energy Theorem constraint
	;;;;	 check-abstracted-constraints	; filter states inconsistent w abstracted c.
		 quantitative-range-reasoning	; Q2 quantitative range reasoning
		 crossing-trajectory-filter	; filter self-intersecting trajectories
						; ***** it needs to go after cycle
						; ***** detection and before quiescence
						; ***** check
		 ))
;;;  Note:  filter-for-transitions is called explicitly after all other
;;;         filters to perform transitions (region, monitor, boundary,
;;;         and halt transitions).



;;;  The following functions apply these filters.  They are called by the
;;;  functions QSIM and SUCCESSORS-OF-STATE using mapcan to filter a list
;;;  of candidates.  
;;;  The selection of which states should go onto the agenda for further
;;;  simulation is now made explicitly based on state.status.

(defun apply-global-alist-filters (state alist)	; => (list alist) | nil
  (start-timer 'global-alist-filters)
  (dolist (filter *global-alist-filters*)
    (unless (setq alist (funcall filter state alist))
      (return)))
  (stop-timer 'global-alist-filters)
  (if alist
      (list alist)
      nil))
 
(defun apply-global-state-analysis (state)	; => t
  (start-timer 'global-state-analysis)
  (dolist (analysis-fn *global-state-analysis*)
    (funcall analysis-fn state))
  (stop-timer 'global-state-analysis)
  t)

(defun apply-global-state-filters (state)	; => (list state) | nil
  (start-timer 'global-state-filters)
  (cond ((member 'inconsistent (state-status state))	; BJK:  10-23-90
	 (setq state nil))
	(t (let ((*current-qde* (state-qde state)))
	     (declare (special *current-qde*))
	     (dolist (filter *global-state-filters*)
	       (unless (setq state (funcall filter state))
		 (return))))))
  (stop-timer 'global-state-filters)
  (if state
      (list state)
      nil))

;;;-----------------------------------------------------------------------------
;;; NO CHANGE filter -- on alists:
;;;  - its predecessor is a time-interval
;;;  - all its values are equal to its predecessors values.
;;;  - qdir(p)=std  =>  context(p).
;;;-----------------------------------------------------------------------------

(defun no-change-filter1 (state alist)
  (cond ((no-change-test1 state alist)
	 (if *trace-no-change*
	     (format *QSIM-Trace* "~%NO CHANGE(1) filters out ~a as successor to ~a."
		     alist state))
	 nil)
	((no-change-test2 state alist)
	 (if *trace-no-change*
	     (format *QSIM-Trace* "~%NO CHANGE(2) filters out ~a as successor to ~a."
		     alist state))
	 nil)
	(t alist)))


;;;-----------------------------------------------------------------------------
;;;  Efficiency Note:
;;;         The two functions "no-change-test1" and "no-change-test2" take
;;;         advantage of the fact that a previous state usually has the same
;;;         qde (unless there has been a transition) and therefore its state
;;;         values are in the same order as the given list of values.  When this
;;;         is true, these functions "walk down" the two lists in parallel,
;;;         eliminating the need to search for the matching variable.
;;;-----------------------------------------------------------------------------

;;;  Returns T if previous state was at a time interval and the new qvals match
;;;  the previous state's qvals.  A T result causes the state to be filtered.

(defun no-change-test1 (state candidate-qvalues)
  (declare (optimize speed) (special *current-qde*))
  (when (qintervalp (state-time state))
    (if (eq *current-qde* (state-qde state))

	;; Same qdes, so qvals are in same order.  See if no change between
	;; old values and new values.
	(every #'(lambda (ovalue nvalue &aux oval nval)
		   (setq oval (cdr ovalue)
			 nval (cdr nvalue))
		   (qval-equal oval nval))
	       (cdr (state-qvalues state))	; skip over time value
	       (cdr candidate-qvalues))		; skip over time value

	;; Different qdes (must have had a transition).
	(every #'(lambda (ovalue)
		   (let* ((name (car ovalue))
			  (oval (cdr ovalue))
			  (nval (alookup name candidate-qvalues)))
		     (qval-equal oval nval)))
	       (cdr (state-qvalues state))))))	; skip over time value


;;; The more general NO-CHANGE filter only looks after we have reached the
;;; next interval.

(defun no-change-test2 (state candidate-qvalues)
  (declare (optimize speed) (special *current-qde*))
  (cond ((qpointp (state-time state))
	 (let* ((previous (predecessor-of-state state))
		(svalues  (state-qvalues state))
		(pvalues  (if previous (state-qvalues previous))))
	   (cond ((null previous) nil)
		 ((qpointp (state-time previous)) nil)
		 (t (if (and (eq *current-qde* (state-qde state))
			     (eq *current-qde* (state-qde previous)))
			
			;; All using same qde, so qvals are in same order.
			(every #'(lambda (pvalue svalue nvalue)
				   (and (qval-equal (cdr nvalue) (cdr svalue))
					(qval-equal (cdr nvalue) (cdr pvalue))))
			       (cdr pvalues)	; skip over time value
			       (cdr svalues)	; skip over time value
			       (cdr candidate-qvalues))	; skip over time value

			;; Different qdes (must have had a transition).
			(every #'(lambda (nvalue)
				   (let* ((name (car nvalue))
					  (nval (cdr nvalue)))
				     (and (qval-equal nval (alookup name svalues))
					  (qval-equal nval (alookup name pvalues)))))
			       (cdr candidate-qvalues)))))))))	; skip over time value

;;;-----------------------------------------------------------------------------
;;; ANALYTIC-FUNCTION filter -- on alists:
;;;   - permit only analytic functions, i.e.
;;;   - if a variable is constant over an interval, it must be constant always.
;;;-----------------------------------------------------------------------------

(defun analytic-function-filter (state alist)
  (cond ((and *analytic-functions-only*
	      (qpointp (state-time state))
	      (not (constants-OK state alist)))
	 (if *trace-analytic-functions*
	     (format *QSIM-Trace* "~%ANALYTIC FUNCTIONS ONLY filters out ~a as successor to ~a."
		     alist state))
	 nil)
	(t alist)))

; Check this by rejecting any alist with a variable which is proposed
; to be constant over the next interval, but is not equal to its value
; over the preceding interval.  Return NIL to reject.

(defun constants-OK (state candidate-qvalues)
  (declare (optimize speed) (special *current-qde*))
  (let ((previous (predecessor-of-state state)))
    (cond ((null previous) t)			      ; don't filter at the beginning
	  ((qpointp (state-time previous)) t)    ; or at transitions
	  ((eq *current-qde* (state-qde state))

	   ;; Qdes are same, so walk down lists in parallel.
	   (notany #'(lambda (pvalue nvalue &aux nqval)
		       (setq nqval (cdr nvalue))
		       (and  (eql 'std (qdir nqval))
			     (not (qval-equal nqval (cdr pvalue)))))
		   (state-qvalues previous)
		   candidate-qvalues))

	  (t
	   ;; Qdes different, so must find matching variable in previous state.
	   (notany #'(lambda (nvalue &aux name nqval)
		       (setq name  (car nvalue)
			     nqval (cdr nvalue))
		       (if (eql 'std (qdir nqval))
			   (not (qval-equal nqval (qval name previous)))))
		   candidate-qvalues)))))


;;;-----------------------------------------------------------------------------
;;; INCONSISTENT-IGN filter -- on alists:
;;;   If the qvalues include an IGN value, make sure it's consistent
;;;   by making a copy with IGN changed to NIL, and calling CFILTER
;;;   to create consistent states.
;;;
;;; The problem is not that it can't be completed, but that it's not consistent
;;; with the predecessor.
;;;-----------------------------------------------------------------------------

(defun inconsistent-ign-filter (state qvalues)
  (if (not *check-for-inconsistent-IGN*)
      (return-from inconsistent-ign-filter qvalues))

  (let ((nqvals nil)
	(changed nil)
	(current-state state)
	completions
	nqval)
    (declare (special current-state))		; make visible to cfilter, etc.

    ;; First, see if there are any IGN qdirs.  If not, return.
    (dolist (qvalue qvalues)
      (let ((qval (cdr qvalue)))
	(if (eql 'ign (qdir qval))
	    (progn
	      (setq nqval (copy-qval qval))
	      (setf (qdir nqval) nil)
	      (push nqval nqvals)
	      (setq changed t))
	    (push qval nqvals))))

    (unless changed
      (return-from inconsistent-ign-filter qvalues))

    ;; Compute possible values for each variable.  Since the time value
    ;; is fully specified, it is returned unchanged by all-qvalues.
    (dolist (qval nqvals)
      (setf (variable--pvals (qval-variable qval)) (all-qvalues qval)))


    ;; Normally, all constraint--cvals slots must be initialized from the
    ;; "parent" state before calling cfilter, but that isn't necessary here
    ;; because this function is a global-alist-filter, which is applied
    ;; immediately after the alist was created (by a call to cfilter).
    ;; Thus, the cvals are still correct in the constraint network.

    (setq completions (cfilter (state-qde state) #'check-qsim-constraint))

    (cond ((null completions)
	   (if *trace-ignore-qdir*
	       (format *QSIM-Trace* "~%INCONSISTENT-IGN filters out ~a at ~a ~
                                           because it can't be completed."
		       qvalues (state-time state)))
	   nil)

	  ((null
	     (mapcan #'(lambda (completion)
			 (if (inconsistent-continuity state completion)
			     nil
			     (list state)))
		     completions))
	   (if *trace-ignore-qdir*
	       (format *QSIM-Trace* "~%INCONSISTENT-IGN filters out ~a at ~a ~
                                           because it can't follow ~a."
		       qvalues (state-time state) (state-name state)))
	   nil)
	  (t qvalues))))


;;;  This checks all the qualitative values in the two alists, and returns T
;;;  if the transition is NOT consistent.
;;;  dd-- This new code assumes that the old and new values are from the same qde.
;;;  If not true, then this won't give correct answer!!!!

(defun inconsistent-continuity (state candidate-qvalues)
  (let ((check-function (if (qpointp (state-time state))
			    #'test-P-successor
			    #'test-I-successor)))

    ;; This code expects the three lists to be in the same order, by variable.
    (notevery #'(lambda (oqvalue nqvalue qspace-pair)
		  (funcall check-function oqvalue nqvalue (cdr qspace-pair)))
	      (cdr (state-qvalues state))	; skip over time value
	      (cdr candidate-qvalues)		; skip over time value
	      (cdr (state-qspaces state)))))	; skip over time qspace

;;;-----------------------------------------------------------------------------
;;; QUIESCENCE filter  -- on states:
;;;     all landmarks STD.
;;;-----------------------------------------------------------------------------

(defun quiescence-marker (state)
  (quiescent state))

(defun quiescent (state)
  (cond ((member 'quiescent (state-status state)) t)
	((test-quiescence state)
	 (if *trace-quiescence*
	     (format *QSIM-Trace* "~%QUIESCENT state ~a identified." (state-name state)))
	 (pushnew 'quiescent (state-status state))
	 t)
	(t nil)))

(defun test-quiescence (state)
  (and (qpointp (state-time state))
       (every #'(lambda (qvalue &aux qval)
		  (setq qval (cdr qvalue))
		  (or (eql 'std (qdir qval))
		      (eql 'ign (qdir qval))))
	      (cdr (state-qvalues state)))))	; skip over time value


;;;-----------------------------------------------------------------------------
;;;  UNREACHABLE-VALUES filter is now a filter on states.
;;;    - a state is inconsistent if a variable is equal to an unreachable value.
;;;-----------------------------------------------------------------------------

; A global state filter for unreachable values.

(defun filter-unreachable-values (state)
  (cond ((null *check-for-unreachable-values*) state)
	(t (let ((unreachables (sim-unreachable-values *current-sim*)))
	     (dolist (qvalue (state-qvalues state) state)	; return state if OK
	       (let* ((varname (car qvalue))
		      (qval (or (cdr qvalue)
				(error "Variable ~a has no value at ~a." varname state)))
		      (qmag (or (qval-qmag qval)
				(error "Variable ~a has no qmag at ~a." varname state))))
		 (cond ((and (point-p qmag)
			     (or (member qmag
					 (variable-unreachable-values
					   (qval-variable (cdr qvalue))))
				 (member (lmark-name qmag)
					 (cdr (assoc varname unreachables)))))
			(if *trace-unreachable-values-filter*
			    (format *QSIM-trace* "~%UNREACHABLE VALUE: ~a(~a) = ~a."
				    varname state qmag))
			(prune-inconsistent-state
			  state (format nil "Unreachable ~a = ~a." varname qmag))
			(return nil)))))))))	; return NIL if filtered


;;;-----------------------------------------------------------------------------
;;; NEW LANDMARK transformation:
;;;   - time value: ((t5 inf) inc) --> ((t5 t6) inc)
;;;   - critical value:  ((L1 L2) std) --> (L* std)
;;;   - map across M+ function:  M+(X,Y), X(t)=L, Y(t)=(L1,L2) --> Y(t) = L*
;;;   - values at region transitions:  ((L1 L2) any) --> (L* any)
;;;  Global variables associated with these are defined in QDEFS.LISP
;;;-----------------------------------------------------------------------------



;;;-----------------------------------------------------------------------------
;;;  Function:  (make-new-landmarks  state)
;;;
;;;  Tasks:  1. If time is of the form (t5 inf), then make landmark for t6.
;;;          2. If the state is at a time point, then every variable whose qval
;;;             is ((L1 L2) std) has a "critical value", and so the qmag (L1 L2)
;;;             should be made into a new landmark.
;;;          3. Propagate landmarks across constraints, i.e., find constraints
;;;             having exactly 1 interval value in its value tuple, and create
;;;             a new landmark for that value.
;;;
;;;  Returns:   The same state, possibly containing new landmarks.
;;;-----------------------------------------------------------------------------

(defun make-new-landmarks (state)		; => state with new lmarks

  ;; STEP 1: Make new landmark for time.
  (create-new-landmark-for-time state)

  ;; STEP 2: Make new landmarks for critical values.
  (when (and *enable-landmark-creation*
	     (qpointp (state-time state)))
    (mapc #'(lambda (qv-pair qs-pair)
	      (let ((qval (cdr qv-pair)))
		(unless (typep qval 'qval)
		  (error "Qval ~a in ~a of wrong type at state ~a." (cdr qv-pair) qv-pair state))
		(when (and (qmag-interval-p (qmag qval))
			   (if *always-create-landmarks-at-critical-values*
			       t
			       (not (no-new-landmarks-p (qval-variable qval)))))
		  (cond (*always-create-landmarks*
			 (create-new-lmark state qv-pair qs-pair "always created"))
			((and *create-landmarks-at-critical-values*
			      (eql 'std (qdir qval)))
			 (create-new-lmark state qv-pair qs-pair "critical value"))))))
	  (cdr (state-qvalues state))		; skip over time qval
	  (cdr (state-qspaces state)))		; skip over time qspace

    ;; STEP 3: Propagate landmarks across constraints.
    (propagate-landmarks-across-constraints state))
  state)

;;; NO-NEW-LANDMARKS-P encapsulates old and new ways to check for suppression of
;;; new landmarks for a variable.  To work at an initial state, the global *current-sim*
;;; must be bound before calling make-sim.

(defun no-new-landmarks-p (var)			; BK:  10-11-90
  (or (variable-no-new-landmarks-p var)
      (if *current-sim*
	  (member (variable-name var)
		  (sim-no-new-landmarks *current-sim*)))))

;;;-----------------------------------------------------------------------------
;;;  If this state is at a time interval and the upper bound of the interval
;;;  is infinity, then make a new landmark.  For example:
;;;  ((t5 inf) inc) ---> ((t5 t6) inc)
;;;-----------------------------------------------------------------------------

(defun create-new-landmark-for-time (state)	; broken into two functions 5 may 90 -drt
  (let ((time (state-time state)))
    (when (and (qintervalp time)
	       (eq (second (qmag time)) *inf-lmark*))
      (let* ((lower-lmark  (first (qmag time)))
	     (lower-symbol (lmark-name lower-lmark ))
	     (upper-lmark  (time-successor lower-symbol state))
	     (new-time     (copy-qval time))
	     (new-qspace   (copy-list (time-qspace state)))
	     (qspace-point (member lower-symbol new-qspace :key #'lmark-name)))

	;; Insert new landmark into time's qspace and update time value.
	(setf (cdr qspace-point)  (cons upper-lmark (cdr qspace-point))
	      (cdar (state-qspaces state)) new-qspace
	      (qmag new-time)     (list lower-lmark upper-lmark)
	      (state-time state)  new-time)

	(if *trace-new-landmarks*
	    (format *QSIM-Trace* "~%NEW LANDMARK ~a added to ~a in ~a"
		    upper-lmark (time-variable state) (state-name state)))))))


;;; Given the name of a time, return a lmark whose name is the successor
;;; of the given time.
(defun time-successor (earlier-symbol state)
  (let* ((earlier-string (symbol-name earlier-symbol))
	 (num          (parse-integer earlier-string :start 1))
	 (later-string (gen-symbol-name *Time-stem* (1+ num) nil))
	 (later-symbol (if *intern-gennames*	;Dan Berleant 3/9/89.
			   (intern later-string :qsim)
			   (make-symbol later-string))))
    (make-lmark :name later-symbol :when-defined state)))


; Identify variables needing new landmarks because they correspond to 
; landmarks across M+/M- constraints.  Repeat this process until none are found.

;;;-----------------------------------------------------------------------------
;;;  Function:  (propagate-landmarks-across-constraints  qvals  state)
;;;
;;;  Purpose:   If we are at a time point and all but one of the arguments of a
;;;             constraint are landmarks (and therefore the other argument is
;;;             an interval), and if the constraint can have corresponding
;;;             values, then make a new landmark to replace the interval.
;;;
;;;  Design:    The "qvals" input argument indicates which variables have just
;;;             had a new landmark created, so that on the first pass, the
;;;             only constraints that need to be checked are the ones attached
;;;             to these variables.  If this pass results in any new
;;;             landmarks, then the affected variables are used as guidance
;;;             for the second pass, and so on.
;;;-----------------------------------------------------------------------------

(defun propagate-landmarks-across-constraints (state)
  (when (and *new-landmarks-across-M-constraints*
	     (qpointp (state-time state)))
    (let ((constraints (qde-constraints (state-qde state))))
      (propagate-landmarks constraints nil state))))


(defun PROPAGATE-LANDMARKS (constraints constraints-propagated state)
  (let ((constraints-to-check nil)
	qval)
    ;; Examine each constraint of that variable.
    (dolist (con constraints)
      ;; Ignore constraints that are inactive or have already been propagated
      ;; or that shouldn't have propagation done.
      (when (and (constraint-active-p con)
		 (contype-cvals-allowed-p (constraint-type con))
		 (not (member con constraints-propagated))
		 (setq qval (constraint-has-1-interval con state)))
	(let ((var (qval-variable qval)))
	  (push con constraints-propagated)
	  (unless (no-new-landmarks-p var)
	    (create-new-landmark qval state "propagated landmark")
	    (dolist (ncon (alookup (qde-name (state-qde state))
				   (variable-constraints var)))
	      (unless (eq ncon con)
		(push ncon constraints-to-check)))))))

    ;; Call recursively until all propagation has died down.
    (when constraints-to-check
      (propagate-landmarks constraints-to-check constraints-propagated state))))


;;;-----------------------------------------------------------------------------
;;;  CONSTRAINT-HAS-1-INTERVAL checks the given constraint and if its variables
;;;  have exactly one interval-value (the rest being point-values), then it
;;;  returns that qval.
;;;-----------------------------------------------------------------------------

(defun constraint-has-1-interval (con state)
  (let ((num-intervals 0)
	iqval)   ;; a saved interval-valued qval

    ;; For each variable of the constraint ...
    (dolist (var (constraint-variables con))

      ;; Get the variable's qval from the state ...
      (let ((nqval (qval (variable-name var) state)))

	;; And count the number having an interval value.
	(when (qmag-interval-p (qmag nqval))
	  (setq iqval nqval)
	  (incf num-intervals)
	  (if (> num-intervals 1)
	      (return-from constraint-has-1-interval nil)))))

    (if (= 1 num-intervals)
	iqval
	nil)))

	

;;;-----------------------------------------------------------------------------
;;; If some variables have non-landmark values at the initial state,
;;; create landmarks for those values and then make new corresponding values.
;;; (This also gets called when filling in incomplete states formed at region
;;; transitions.)
;;;-----------------------------------------------------------------------------

(defun create-new-landmarks-at-initial-state (ostate)
  (when (and *enable-landmark-creation* *new-landmarks-at-initial-state*)
    (let ((changed nil)
	  (reason (cond ((eq (state-justification ostate)
			     'one-of-several-completions-of)
			 (concatenate "One-Of-Several-Completions-Of "
					(string ostate)))
			((eq (state-justification ostate) 'transtion-result)
			 "Transition Result")
			(t "initial state"))))
      (mapc #'(lambda (qv-pair qs-pair)
		(let* ((qval (cdr qv-pair))
		       (var  (qval-variable qval)))
		  (when (and (interval-p (qmag qval))
			     (not (no-new-landmarks-p var)))
		    (create-new-lmark ostate qv-pair qs-pair reason)
		    (setq changed t))))
	    (cdr (state-qvalues ostate))	; skip over time qval
	    (cdr (state-qspaces ostate)))	; skip over time qspace
      (if changed
	  (make-new-corresponding-values ostate)))))



;;;-----------------------------------------------------------------------------
;;; create-new-LANDMARK actually creates and inserts the new landmark.
;;;   The calling function copies the information in the state.
;;;-----------------------------------------------------------------------------

;(defun create-new-landmarks-for-vars (qvals state why)
;  (dolist (qval qvals)
;    (let ((var (qval-variable qval)))
;      (unless (variable-no-new-landmarks-p var)
;	(create-new-landmark qval state why)))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (create-new-landmark  qval state why)
;;;
;;;  Purpose:   Given a qval whose qmag is the interval (L1 L2) in the qspace
;;;             (L0 L1 L2 L3), this function creates a new landmark (say L*),
;;;             inserts it into the qspace as (L0 L1 L* L2 L3), and changes the
;;;             qmag to L*.  It is assumed that the given qval is an element of
;;;             (state-qvalues state) so that modifying the qmag of that qval
;;;             actually modifies the state value.
;;;
;;;             Information about when and why the new landmark was created are
;;;             stored in the LMARK structure:
;;;                 when-defined:     <state>
;;;                 where-defined:    (<landmark> <landmark>)
;;;                 why-defined:      <string>
;;;-----------------------------------------------------------------------------
; Changes the qmag slot of the qval of a model parameter (found in
; the qvalues slot of state, accessible via the state-qvalues
; function) from a qualitative interval to a newly-created landmark. The
; qspace of the model parameter (in the qspaces slot of state)
; is augmented to include the new landmark.

(defun create-new-landmark (qval state &optional why)
  (let* ((var     (qval-variable qval))
	 (varname (variable-name var))
	 (qv-pair (assoc varname (state-qvalues state)))	; qv-pair = (<varname> . <qval>)
	 (qs-pair (assoc varname (state-qspaces state))))	; qs-pair = (<varname> . (listof <lmarks>))
    (unless qv-pair (error "Can't find qval ~a in ~a" qval state))
    (unless qs-pair (error "Can't find landmarks for ~a in ~a" var state))
    (create-new-lmark state qv-pair qs-pair why)))

;;; qv-pair := (y . #<qval>)  from y's current value
;;; qs-pair := (y <lmark> <lmark> <lmark>...) from y's qspace

;;; The loop for lname is necessitated by an obscure bug.  If the user
;;; supplies an initial qspace for VAR including a landmark named VAR-0,
;;; the first call to GENNAME can return a symbol that matches the name
;;; of an lmark already in the qspace.  Normally, however, this loop
;;; will return on the first cycle. -drt

;;; LMARK creation is separated into cleaner modules to allow landmark translation. 
;;;      (BJK:  11-20-90)

(defun create-new-lmark (state qv-pair qs-pair why)
  (let* ((qval (cdr qv-pair))
	 (nqval (copy-qval qval))		; newly created for the state's qval for var.
	 (var  (qval-variable qval))
	 (oqmag (qmag qval))
	 (new-lmark (create-and-insert-lmark-at-point oqmag qs-pair var state why)))
    (setf (qmag nqval)       new-lmark)		; Set the qmag of the new qval to the new lmark
    (setf (cdr qv-pair) nqval)			; Set the qval in the state to the new qval
    new-lmark))

;;; QSPACE-POINT := (<lmark1> <lmark2> ...) 
;;;    where NEW-LMARK is to be insterted between <lmark1> & <lmark2>.

(defun create-and-insert-lmark-at-point (oqmag qs-pair var state why)
  ;; Make a new copy of the list of landmarks before modifying it.
  (setf (cdr qs-pair) (copy-list (cdr qs-pair)))
  (let* ((qspace (cdr qs-pair))
	 (qspace-point (member (car oqmag) qspace))
	 (lname (loop for name = (genname (or (variable-prefix var)	; on all calls here.
					      (variable-name var)))
		      unless (member name qspace :key #'lmark-name)
			return name))
	 (new-lmark (make-lmark :name lname 
				:when-defined  state
				:where-defined oqmag
				:why-defined   why))
	 )
    (push new-lmark (cdr qspace-point))		; splice the new lmark to the qspace
    (when *trace-new-landmarks*
      (format *QSIM-Trace* "~%NEW LANDMARK ~a added to ~a in ~a, because ~a."
	      new-lmark var (state-name state)
	      (lmark-why-defined new-lmark)))
    new-lmark))

;;; (These could be improved to narrow the interfaces.)



;;;-----------------------------------------------------------------------------
;;;  Function:  (make-new-corresponding-values  state)
;;;
;;;  Purpose:   If the state values of all the variables of a given constraint
;;;             are landmark values, and if this set of landmarks does not
;;;             already exist as a set of corresponding values, and if it is
;;;             valid to make this tuple into corresponding values given the
;;;             type of constraint, then add the tuple to the list of
;;;             corresponding values for the constraint.
;;;
;;;
;;;  Note:      The new tuple of corresponding values is added to the FRONT of
;;;             <cvals> list.  This means that there is no need to copy the
;;;             <cvals> list for each new state; pointers into the older/smaller
;;;             list of <cvals> remain valid since the list is grown from the
;;;             head end rather than the tail end.
;;;
;;;		Extended 03/12/90 by Pierre Fouche so that intervals may appear
;;;		in CV tuples.
;;;-----------------------------------------------------------------------------


(defun make-new-corresponding-values (state)
  (dolist (pair (state-cvalues state))
    (let* ((con   (car pair))
	   (cvals (cdr pair))
	   tuple)
      (when (and (constraint-active-p con)
		 (setq tuple
		       (if *enable-intervals-in-cvalues*
			   (get-qmag-tuple (constraint-variables con) state)   
			   (get-landmark-tuple
			     (constraint-variables con) state)))
		 (acceptable-cvals con tuple)
		 (not (member tuple cvals :test #'equal)))
	(setf (cdr pair) (cons tuple cvals))
	(when *trace-corresponding-values*
	  (format *QSIM-Trace*
		  "~%NEW CORRESPONDING VALUES ~a added in ~5a for constraint ~a."
		  tuple (state-name state) con)))))
  state)


;;;-----------------------------------------------------------------------------
;;;  GET-LANDMARK-TUPLE takes a tuple of variables and returns the tuple
;;;  of landmark values they take on in the given state, or NIL if any are
;;;  intervals.
;;;  GET-QMAG-TUPLE does the same, but accepts interval values at time points.
;;;-----------------------------------------------------------------------------

(defun get-landmark-tuple (vars state)
  (mapcar #'(lambda (var &aux (qmag (qmag (qval (variable-name var) state))))
	      (if (point-p qmag)
		  qmag
		  (return-from get-landmark-tuple nil)))
	  vars))


(defun get-qmag-tuple (vars state)
  (if (qpointp (state-time state)) 
      (mapcar #'(lambda (var)
		  (qmag (qval (variable-name var) state)))
	      vars)
      (mapcar #'(lambda (var &aux (qmag (qmag (qval (variable-name var) state))))
	      (if (point-p qmag)
		  qmag
		  (return-from get-qmag-tuple nil)))
	  vars)))


;;;-----------------------------------------------------------------------------
;;;  ACCEPTABLE-CVALS returns T if the tuple is acceptable to become a set of
;;;  corresponding values for the given constraint.  Corresponding values are
;;;  not allowed for derivative or integral constraints, and are not meaningful
;;;  if they include inf or minf for ADD or inf, minf, or 0 for MULT.
;;;-----------------------------------------------------------------------------

(defun acceptable-cvals (con tuple)
  (let ((contype (constraint-type con)))
    (when (contype-cvals-allowed-p contype)
      (notany #'(lambda (x)
		  (member x (contype-disallowed-lmarks contype)))
	      tuple))))


;;;-----------------------------------------------------------------------------
;;; INFINITY check.
;;;
;;; Labeling rules:
;;;  (1)  f(t)=inf  and  f'(t)<0   -->  inconsistent
;;;  (2)  f(t)<inf  and  f'(t)<>0  -->  t<inf
;;;  (3)  f(t)=inf  and  f'(t)<inf -->  t=inf
;;; Checking rules:
;;;  (4)  t=inf  -->  [ f(t)<inf --> f'(t)=0 ]
;;;  (5)  t<inf  -->  [ f(t)=inf --> f'(t)=inf ]
;;;
;;; If a state is labeled inconsistently, it is inconsistent.
;;;-----------------------------------------------------------------------------

(defun label-infinite-times (state)
  (dolist (qvalue (cdr (state-qvalues state)) state)	; skip over time qval
    (let ((qval (cdr qvalue)))
      (or (check-for-finite-times qval state)	        ; rule (2)
	  (return nil))
      (or (check-for-infinite-times qval state)    	; rule (3)
	  (return nil)))))
    

(defun check-for-finite-times (qval state)
  (let ((tlabel (state-time-label state)))
    (cond ((and (finite (qmag qval))
		(member (qdir qval) '(inc dec)))
	   (if *trace-time-label*
	       (format *QSIM-Trace* "~%FINITE TIME deduced for ~a from ~a." (state-name state) qval))
	   (cond ((null tlabel) (setf (state-time-label state) 't<inf) t)
		 ((eql tlabel 't<inf) t)
		 (t (if *trace-time-label*
			(format *QSIM-Trace* "~%TIME LABEL (finite) is inconsistent with ~a." (state-name state)))
		    (prune-inconsistent-state state "Time labels")
		    nil)))
	  (t t))))

(defparameter *not-finite* (list *inf-lmark* *minf-lmark* nil))
(defparameter *infinite-lmarks* (list *inf-lmark* *minf-lmark*))

(defun finite (qmag)				; known to be finite
  (not (member qmag *not-finite*)))

(defun infinite (qmag)				; known to be infinite
  (member qmag *infinite-lmarks*))

(defun check-for-infinite-times (qval state)
  (let ((tlabel (state-time-label state)))
    (cond ((not (infinite (qmag qval))) t)	; pass only if infinite(qmag)
	  ((or (eql (qdir qval) 'std)
	       (finite (explicit-derivative-qmag (qval-variable qval) state)))
	   (if *trace-time-label*
	       (format *QSIM-Trace* "~%INFINITE TIME deduced for ~a from ~a."
		       (state-name state) qval))
	   (cond ((null tlabel) (setf (state-time-label state) 't=inf) t)
		 ((eql tlabel 't=inf) t)
		 (t (if *trace-time-label*
			(format *QSIM-Trace* "~%TIME LABEL (infinite) is inconsistent with ~a."
				(state-name state)))
		    (prune-inconsistent-state state "Time labels")
		    nil)))
	  (t t))))

(defun EXPLICIT-DERIVATIVE-QMAG (var state)	; => nil | the qmag of the deriv of var
  (dolist (con (alookup (qde-name (state-qde state))
				 (variable-constraints var)) nil)
    (when (and (constraint-active-p con)
	       (eql var (first (constraint-variables con)))
	       (eql 'd/dt (constraint-typename con)))
      (let* ((deriv-var (second (constraint-variables con)))
	     (dvarname  (variable-name deriv-var)))
	(return (qmag (qval dvarname state)))))))


;;;-----------------------------------------------------------------------------
;;; APPLY TIME LABELS.
;;;-----------------------------------------------------------------------------

(defun apply-time-label (state)
  (let ((tlabel (state-time-label state)))
    (cond ((null tlabel) state)
	  ((eql tlabel 't=inf) (apply-time-is-inf state))	        ; rule (4)
	  ((eql tlabel 't<inf) (apply-time-is-finite state))	        ; rule (5)
	  (t (error "Unknown time label ~a on state ~a." tlabel state))
	  )))

(defun apply-time-is-inf (state)
  (cond ((every #'(lambda (qvalue)
		    (let ((qval (cdr qvalue)))
		      (or (not (finite (qmag qval)))
			  (member (qdir qval) '(std ign)))))
		(cdr (state-qvalues state)))	; skip over time qval
	 state)
	(t (if *trace-time-label*
	       (format *QSIM-Trace* "~%TIME LABEL (infinite) is inconsistent with ~a." (state-name state)))
	   (prune-inconsistent-state state "with infinite time")	
	   nil)))

(defun apply-time-is-finite (state)
  (cond ((every	#'(lambda (qvalue)
		    (let* ((qval (cdr qvalue))
			   (var  (qval-variable qval)))
		      (cond ((eq  (qmag qval) *inf-lmark*)
			     (let ((explicit-deriv (explicit-derivative-qmag var state)))
			       (or (null explicit-deriv)
				   (eql explicit-deriv *inf-lmark*))))
			    ((eq  (qmag qval) *minf-lmark*)
			     (let ((explicit-deriv (explicit-derivative-qmag var state)))
			       (or (null explicit-deriv)
				   (eql explicit-deriv *minf-lmark*))))
			    (t t))))
		(cdr (state-qvalues state)))	; skip over time qval
	 state)
	(t (if *trace-time-label*
	       (format *QSIM-Trace* "~%TIME LABEL (finite) is inconsistent with ~a."
		       (state-name state)))
	   (prune-inconsistent-state state "with finite time")
	   nil)))

; fixed bug:  if explicit derivative does not exist, state is not (yet) inconsistent. (-BJK)


;;;-----------------------------------------------------------------------------
;;; These functions pick up the previous qspaces and corresponding
;;; values, even if they are not the immediate predecessors of the
;;; current one.
;;;-----------------------------------------------------------------------------
; QSPACE-UNION takes the ordered union of two quantity spaces.

(defun qspace-union (Q1 Q2)
  (cond ((null Q1) Q2)
	((null Q2) Q1)
	((lmark-equal (car Q1) (car Q2))
	 (cons (car Q1) (qspace-union (cdr Q1) (cdr Q2))))
	((and (member (car Q1) Q2 :test #'lmark-equal)
	      (not (member (car Q2) Q1 :test #'lmark-equal)))
	 (cons (car Q2) (qspace-union Q1 (cdr Q2))))
	((and (member (car Q2) Q1 :test #'lmark-equal)
	      (not (member (car Q1) Q2 :test #'lmark-equal)))
	 (cons (car Q1) (qspace-union  (cdr Q1) Q2)))
	(t (format t "~%Warning: Can't take qspace-union of ~s and ~s." Q1 Q2)
	   (cons (car Q1) (cons (car q2)(qspace-union (cdr Q1) (cdr Q2)))))))

;;;-----------------------------------------------------------------------------
;;;  CVAL-UNION generates the union of two sets of corresponding values,
;;;  preferring cv-tuples from the first set whenever there is a match in the
;;;  second set.  Two cv-tuples are considered equal if their corresponding
;;;  landmarks have the same name (e.g., 'minf, 'inf, 0, 'full, ...).
;;;-----------------------------------------------------------------------------

(defun cval-union (cvset1 cvset2 cname qspaces)
  "Create union of two sets of corresponding values"
  (let ((ncvset cvset1))
    ;; Test each cv-tuple in set 2 for a match in set 1.
    (dolist (cv-tuple cvset2)
      ;; When no match is found, then add the cv-tuple to the new union'ed set.
      (when (not (member cv-tuple cvset1 :test #'(lambda (cv1 cv2)
						   (every #'lmark-equal
							  cv1
							  cv2))))
	;; Ensure that all landmarks are from the new union'ed qspace.
	(let* ((varnames (cdr cname))
	       (ncv (mapcar #'(lambda (lmark varname)
				(let ((qspace (alookup varname qspaces)))
				  (find (lmark-name lmark) qspace :key #'lmark-name)))
			    cv-tuple varnames)))
	  (push ncv ncvset))))
    ncvset))


;;;-----------------------------------------------------------------------------
;;; CHECK-FOR-CYCLE determines whether a state matches a direct predecessor.
;;; If so, CYCLE and (CYCLE (<match>)) are pushnew'ed onto the state's status;
;;; <match> is the degree of match between state and first matching predecessor.
;;; The state is not put on the agenda, because FILTER-FOR-AGENDA will catch it. 
;;; CHECK-FOR-CYCLE is NOT a filter; it is called only for its side effect.
;;;-----------------------------------------------------------------------------

(defun CHECK-FOR-CYCLE (state)
  (when (and *check-for-cycles*
	     (not *envisionment*)
	     (or (eq *check-for-cycles* :weak)
		 (qpointp (state-time state))))
    (do ((pred (state-predecessor state)
	       (state-predecessor pred))
	 matching)
	 ((null pred) state)
      (when (setq matching (match-states state pred))
	(setf (state-successors state) `(cycle-identity ,pred))
	(pushnew 'cycle (state-status state))
	(pushnew `(cycle (,matching)) (state-status state) :test #'equal)
	(if *trace-cycle-detection*
	    (format *QSIM-Trace*
		    "~%~a Cycle identified:  ~a = ~a." matching state pred))
	(return state))))
  state)


;;; Get the predecessor of a state from its state.justification slot.

(defun PREDECESSOR-OF-STATE (state)
  (let ((just (state-justification state)))
    (ecase (car just)
      ((unique-successor-of one-of-several-successors-of successor-of
			    perturbation-of transition-from)
       (second just))
      (one-of-several-completions-of
	(predecessor-of-state (second just)))
      ((initialized-with copy-of) nil))))



;;;-----------------------------------------------------------------------------
;;; STABILITY is determined by attempting to simulate past a quiescent state.
;;;    - It is stable if simulation is impossible, unstable otherwise.
;;;      All flow vectors in the neighborhood of a stable state point toward it.
;;;    - Requires disabling the quiescent state filter.
;;;                                   BJK:  10-26-90
;;;-----------------------------------------------------------------------------

(defun stability-label (state)
  (cond ((member 'stable (state-status state)) t)
	((member 'unstable (state-status state)) nil)
	((and *compute-stability-label*
	      (quiescent state))
	 (cond ((stability-test state)
		(pushnew 'stable (state-status state))
		t)
	       (t (pushnew 'unstable (state-status state))
		  nil)))
	(t nil)))

; Test stability of a quiescent state by attempting to simulate past it.
;   stable equilibrium    =>  T
;   unstable equilibrium  =>  NIL

(defun stability-test (state)
  (if (and *compute-stability-label*
	   (member 'quiescent (state-status state)))
      (let* ((*compute-stability-label* nil)	; prevent infinite regress (?)
	     (*quiescent-ok-for-agenda* t)
	     (*print-timing* nil)		; prevent useless output
	     (*trace-simulation-stop* nil)
	     (*global-alist-filters* (cons #'no-change-filter3 *global-alist-filters*))
	     (sim (make-sim :state-limit *stability-state-limit*
			    :Q2-constraints nil))
	     (ns (make-new-state :from-state state
				 :sim sim
				 :assert-values (all-variable-values state)
				 :text (format nil "Stability test for ~a in ~a."
					       state sim)))
	     (*initial-state* ns))
	(setf (state-predecessors ns) nil)
	; (setf (state-other ns) nil)
	(trace-stability state ns)
	(setf (state-stability-test state) sim)
	(qsim ns)
	(cond ((inconsistent-p ns)
	       (trace-stability-report 'stable)
	       t)
	      (t (trace-stability-report 'unstable)
		 nil)))))

(defun name-of-qmag (qmag)
  (typecase qmag
    (lmark (lmark-name qmag))
    (cons  (list (name-of-qmag (car qmag))
		 (name-of-qmag (cadr qmag))))
    (t (error "Can't translate malformed qmag ~a" qmag))))

(defun all-variable-values (state)
  (mapcar #'(lambda (pair)
	      (let* ((var (car pair))
		     (qval (cdr pair))
		     (qmag (qval-qmag qval))
		     (qdir (qval-qdir qval)))
		(list var (list (name-of-qmag qmag) qdir))))
	  (cdr (state-qvalues state))))

; Should this require FINAL-STATE(state)?
; Trace functions.

(defun trace-stability (s ns)
  (when *trace-stability-test*
    (format *QSIM-trace* "~%Stability test of quiescent state ~a via ~a  ... " s ns)))

(defun trace-stability-report (tag)
  (when *trace-stability-test*
    (format *QSIM-trace* " ~a." tag)))



;;;-----------------------------------------------------------------------------
;;; For the stability test, we need a special, stronger, No-Change filter,
;;; that filters out a successor if it stays at the same fixed-point.
;;;-----------------------------------------------------------------------------

(defun no-change-filter3 (state alist)
  (cond ((no-change-test3 state alist)
	 (if *trace-no-change*
	     (format *QSIM-Trace* "~%NO CHANGE(3) filters out ~a as successor to ~a."
		     alist state))
	 nil)
	(t alist)))

(defun no-change-test3 (state candidate-qvalues)
  (declare (optimize speed) (special *current-qde*))
  (when (qpointp (state-time state))
    (if (eq *current-qde* (state-qde state))

	;; Same qdes, so qvals are in same order.  See if no change between
	;; old values and new values.
	(every #'(lambda (ovalue nvalue &aux oval nval)
		   (setq oval (cdr ovalue)
			 nval (cdr nvalue))
		   (qval-equal oval nval))
	       (cdr (state-qvalues state))	; skip over time value
	       (cdr candidate-qvalues))		; skip over time value
	)))



;;;-----------------------------------------------------------------------------
;;; MATCH-STATES returns -
;;; - nil if the qvalues of the two states do not match to the extent required
;;;   by *check-for-cycles*, or
;;; - the level of match achieved by the two states, which corresponds to the
;;;   possible values for *check-for-cycles*:
;;;   - :strong - states are time point states and qmags are points.
;;;   - :standard - same, except variables whose qdirs are being ignored need
;;;		    not have qmags which are points.
;;;   - :weak - no requirements beyond ...
;;;   In all cases, qvals must be equal.
;;;-----------------------------------------------------------------------------

(defun MATCH-STATES (s1 s2 &aux (matching-type :strong))
  (and (eq (state-qde s1) (state-qde s2))
       (eq (qpointp (state-time s1))		; S1 and s2 are both time points
	   (qpointp (state-time s2)))		;   or both time intervals.
       (if (qpointp (state-time s1))		; Time must be a point
	   t					;   unless
	   (if (eq *check-for-cycles* :weak)	;   *check-for-cycles* is :weak.
	       (setq matching-type :weak)
	       nil))
       (every #'(lambda (qvalue1 qvalue2)
		  (let ((qval1 (cdr qvalue1))
			(qval2 (cdr qvalue2)))
		    (and (qval-equal qval1 qval2)
			 (or
			   (point-p (qmag qval1))			; strong match
			   (and (or (eq *check-for-cycles* :standard)	; standard match
				    (eq *check-for-cycles* :weak))
				(eql (qdir qval1) 'ign)
				(or (eq matching-type :weak)
				    (setq matching-type :standard)))
			   (and (eq *check-for-cycles* :weak)		; weak match
				(setq matching-type :weak))))))
	      (cdr (state-qvalues s1))		; skip over time qval
	      (cdr (state-qvalues s2)))		; skip over time qval
       matching-type))

; Partial-match succeeds may succeed if two values are between landmarks,
; as long as the QDIR = IGN.  (This is an experiment for David Dalle Molle.)

;(defun partial-match-states (s1 s2)
;  (and (eq (state-qde s1) (state-qde s2))
;       (every #'(lambda (qvalue1 qvalue2)
;		  (let ((qval1 (cdr qvalue1))
;			(qval2 (cdr qvalue2)))
;		    (cond ((and (point-p (qmag qval1))
;				(qval-equal qval1 qval2)))
;			  ((and (eql (qdir qval1) 'ign)
;				(equal (qmag qval1) (qmag qval2)))))))
;	      (cdr (state-qvalues s1))		; skip over time qval
;	      (cdr (state-qvalues s2)))))	; skip over time qval





;;;=============================================================================
;;;
;;;                 S I M I L A R I T Y    F I L T E R I N G
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  When QSIM generates the successors of a given parent state, it creates
;;;  every possible distinct successor, thus revealing all possible behaviors
;;;  of a mechanism.  This desirable property, however, can yield a large bushy
;;;  behavior tree full of "minor" distinctions, making it difficult for a
;;;  human to see if there are just a few "fundamental" behaviors.
;;;
;;;  "Similarity filtering" allows the user to suppress certain distinctions
;;;  during the generation of successor states, thus reducing the size of
;;;  the behavior tree, revealing only the distinctions that the user cares
;;;  about.  [It also reduces the simulation time.]  As described below, the
;;;  user specifies which distinctions are "unimportant" and can therefore be
;;;  filtered out.  Clearly, this is a "USE AT YOUR OWN RISK!" feature since
;;;  you may inadvertantly suppress certain behaviors that you would want to
;;;  see.
;;;
;;;  There are two types of distinctions that a user may suppress:
;;;  occurrence branching and irrelevant-variable branching.  Each is 
;;;  described below, separately.
;;;
;;;  "Occurrence branching" is branching on the temporal order of "events".
;;;  An "event" is some qualitative distinction that occurs at a time point,
;;;  such as a variable passing through a landmark.  For example, suppose
;;;  that in some state S-7 at a time interval, variables A and B are both
;;;  approaching landmarks, and suppose that S-7 has 3 successor states
;;;  representing the 3 possible outcomes: only A reaches a landmark, only B
;;;  reaches a landmark, or A and B both reach a landmark.  If the user
;;;  doesn't care to see all three possibilities, then he/she can specify
;;;  that, for variables A and B, the branching that would normally occur
;;;  on the temporal ordering of the reaching-a-landmark event is to be
;;;  suppressed.  The simple language for specifying events is described
;;;  below by example:
;;;
;;;  1.  (amount (A* std))
;;;      The event when amount = (A* std).
;;;
;;;  2.  (amount (A* nil))
;;;      The event when amount's qmag = A*, regardless of its qdir.
;;;
;;;  3.  (amount (nil std))
;;;      The event when amount becomes std at any landmark.
;;;
;;;  4.  (amount (nil nil))
;;;      The event when amount reaches any landmark, regardless of its qdir.
;;;
;;;  Thus, if the user specifies ((A (nil nil))  (B (nil nil))) for the
;;;  above scenario, then S-7's successors will be filtered down to the
;;;  one "most eventful" successor, the one where A and B reach landmarks
;;;  simultaneously.
;;;
;;;  
;;;  The second type of branching is "irrelevant variable branching".
;;;  An "irrelevant variable" is any variable that the user does not wish to
;;;  use in characterizing a model's behavior, usually because it would
;;;  reveal unimportant distinctions (derivative variables are often a source
;;;  of branching that the user may wish to ignore).  Filtering on
;;;  irrelevant variable branching is stronger than filtering on occurrence
;;;  branching because it suppresses *all* distinctions involving that
;;;  variable.  A variable can be declared "irrelevant" by adding its name
;;;  to the global *ignore-values*, as in:
;;;
;;;      (defparameter *ignore-values* '(netflow1 netflow2))
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  The basic result of similarity filtering is best explained by example.
;;;  Suppose parent state A has successor states B, C, D, and E, and suppose
;;;  that the similarity filter determines that C, D, and E are "similar". 
;;;  In this case, A will effectively have only two successors: B and a
;;;  "representative" (say C) chosen from the similarity set {C D E}.
;;;  Thus, simulation will proceed from states B and C, but not D or E
;;;  (except as noted below).
;;;
;;;  States D and E will be attached to C as "twins" of C, thus permitting
;;;  future programs that might want to know C's range of coverage.
;;;  If simulation from C always leads to inconsistent states, then C would
;;;  ordinarily be pruned.  However, if C has any twins attached, the twin's
;;;  values will be substituted for C's values, and simulation will proceed
;;;  again from C.  This will continue either until simulation leads to a
;;;  consistent final state descended from C, or until all of C's twins have
;;;  been used.  Thus, this ensures that if there is *any* valid behavior from
;;;  a "similarity set", then one will be found.
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  The similarity filtering algorithm is applied when there are multiple
;;;  successors to a state (called the "parent"), where the state is at a time
;;;  interval:
;;;
;;;  1.  States whose status is 'cycle or 'inconsistent always are retained
;;;      as distinct successor states of the parent.  The other states are,
;;;      for the moment, put in the list of other-successors.
;;;
;;;  2.  For each state in other-successors, tag each of its qvals as to
;;;      whether or not the qval signifies an event.
;;;
;;;  3.  Of all the states in other-successors, take the one having the most
;;;      events (call it X) and make it a distinct successor of the parent.
;;;      Also, find all the other states in other-successors which are "similar"
;;;      to X, and attach them as "twins" of X.  In effect, X will be the
;;;      "representative" state, subsuming all of its twins.
;;;      Delete X and its twins from other successors.
;;;
;;;  4.  If other-successors is not empty, repeat step 3.  Otherwise, update
;;;      the parent's |successors| slot and each distinct successor's
;;;      |justification| slot.  Return the list of distinct successors.
;;;-----------------------------------------------------------------------------

(defparameter *event-values*  nil)
(defparameter *ignore-values* nil)


(defun filter-for-similarity (parent successors)
  (let ((distinct-successors  nil)
	(other-successors     nil))

    (dolist (state successors)
      (setf (state-twins state) nil))

    (unless (and *check-for-similarity*
		 parent
		 (cdr successors)
		 (or *event-values* *ignore-values*)
		 (qpointp (state-time (car successors)))
		 )
      (return-from filter-for-similarity successors))

    ;; Separate the successors into two categories:
    ;; -- normal (meaning the state contains some relevant distinction(s)
    ;;     other than occurrence branching).
    ;; -- occurrence (meaning the only distinction is in the occurrence variables.

    ;; STEP 1:  Divide successor states into two sets -- states that should
    ;;          remain as unique successors (e.g., cycle and inconsistent),
    ;;          and states-to-be-checked-for-twins.
    (dolist (state successors)
      (setf (state-twins state) nil)
      (if (intersection (state-status state) '(cycle inconsistent))
	  (push state distinct-successors)
	  (push state other-successors)))

    ;; STEP 2:  For each state in the second set, tag each variable as to
    ;;          whether or not its value is an "event".
    (dolist (state other-successors)
      (let ((num-events 0))
	(dolist (qvalue (state-qvalues state))    	; (car qvalue) = varname
	  (unless (member (car qvalue) *ignore-values*)	; (cdr qvalue) = qval
	    (let ((e  (event-match-p qvalue *event-values*)))
	      (setf (qval-event-p (cdr qvalue)) e)
	      (when e (incf num-events)))))
	(setf (state-event-count state) num-events)))

    ;; STEP 3:  For each state in the second set, compare it with its siblings
    ;;          and if a sibling is completely similar (with respect to event
    ;;          variables and irrelevant variables), then take the sibling as
    ;;          a "twin".

    ;; Of all the other-successors, select the one having the most values
    ;; at landmarks, i.e., the one with the most events.
    (loop
      (if (null other-successors)
	  (return))
      (let ((max-num-events -1)
	    (best-state     nil))
	(dolist (state other-successors)
	  (let ((num-events (state-event-count state)))
	    ;;(format *qsim-report* "~%~a has ~a events." alist n)
	    (if (> num-events max-num-events)
		(setq max-num-events  num-events
		      best-state      state))))

	;; OK, we've found the next most eventful state.  Make it a distinct
	;; successor and see if it subsumes any siblings.
	(push best-state distinct-successors)
	(setq other-successors (delete best-state other-successors :count 1))
	;;(when (and *trace-occurrence-branching* other-successors)
	;;  (format *qsim-report* "~%Selected:~%   ~a~%Subsuming:" best-state))
	
	;; Delete the subsumed siblings of best-alist.
	;; Sibling is subsumed if, for all vars other than ignore-vars,
	;; if best-state has an event then state is equal or is approaching the same event,
	;; or the qvals are equal.

	(dolist (state other-successors)
	  (when (similar best-state state)
	    ;;(when *trace-occurrence-branching*
	    ;;  (format *qsim-report* " ~a" state))
	    (push state (state-twins best-state))
	    (setq other-successors (delete state other-successors :count 1))))
	))

    ;; STEP 4:  Update parent state's |successors| slot and successor states'
    ;;          |justification| slot.
    (unless (= (length successors) (length distinct-successors))
;;      (format *qsim-report* "~%    similarity filter: ~d --> ~d"
;;	      (length successors) (length distinct-successors))
      (setf (cdr (state-successors parent)) distinct-successors)
      (if (singleton-p distinct-successors)
	  (setf (state-justification (car distinct-successors))
		(if (eql 'completions (state-successors parent))
		    `(one-of-several-completions-of ,parent)
		    `(unique-successor-of ,parent)))
	  (dolist (state distinct-successors)
	    (setf (state-justification state)
		  (if (eql 'completions (car (state-successors parent)))	; fixed minor bug
		      `(one-of-several-completions-of ,parent)	; here on 29Jun90 -drt.
		      `(one-of-several-successors-of ,parent ,distinct-successors))))))

    distinct-successors
    ))


(defun similar (best-state state)
  "Return true if the two states are similar wrt events and ignored values."
  (every #'(lambda (qvalue1 qvalue2)
	     (let* ((varname (car qvalue1))
		    (qval1   (cdr qvalue1))
		    (qval2   (cdr qvalue2))
		    (event   (qval-event-p qval1)))
	       (cond ((member varname *ignore-values*) t)
		     (event (event-equal-p event qval1 qval2))
		     (t (qval-equivalent qval1 qval2)))))
	 (cdr (state-qvalues best-state))
	 (cdr (state-qvalues state))))


(defun event-equal-p (event qval1 qval2)
  "Return true if qval1 and qval2 are at or approaching the given event."
  (let ((eqdir (cadr event)))
    (and
      ;; test for compatible qmags
      (or (qmag-equivalent (qmag qval1) (qmag qval2))
	  (let ((qmag1 (qmag qval1))
		(qmag2 (qmag qval2)))
	    (and (qmag-point-p qmag1)
		 (qmag-interval-p qmag2)
		 (or (and (eql 'inc (qdir qval2))
			  (eq  qmag1 (cadr qmag2)))
		     (and (eql 'dec (qdir qval2))
			  (eq  qmag1 (car qmag2)))))))
      ;; check for compatible qdirs
      (or (null eqdir)
	  (eql (qdir qval1) (qdir qval2))))))


(defun event-match-p (qvalue events)
  "Return the matched event-pattern if qvalue matches any event."
  (let ((varname (car qvalue))
	(qval    (cdr qvalue)))
    (dolist (event events nil)
      (let ((evarname (car event))
	    (eqmag    (caadr event))
	    (eqdir    (cadadr event)))
	(when (and (eql varname evarname)
		   (compatible-qdir eqdir (qdir qval))
		   (compatible-qmag eqmag (qmag qval) (qdir qval)))
	  (return (cdr event)))))))

(defun compatible-qdir (eqdir qdir)
  "Return non-nil if qdir is compatible with event qdir spec."
  (if (null eqdir)
      t
      (eql qdir eqdir)))

(defun compatible-qmag (eqmag qmag qdir)
  "Return non-nil if qmag is compatible with event qmag spec."
  (if (null eqmag)
      (or (qmag-point-p qmag)
	  (eql 'std qdir))
      (and (qmag-point-p qmag)
	   (eql eqmag (lmark-name qmag)))))
	

;;; two qmags are equivalent if defined around same landmarks.
(defun qval-equivalent (qv1 qv2)
  "True if two qvals have equal qdirs and equivalent qmags."
  (and (eql (qdir qv1) (qdir qv2))
       (qmag-equivalent (qmag qv1) (qmag qv2))))

;;;  This recursive function returns true if two qmags are equivalent.
;;;  "Equivalence" is determined by the meaning of the landmarks.
;;;  For example, given the two qspaces (0 A1 A* INF) and (0 A2 A* INF),
;;;  A1 and A2 are equivalent iff they were both created to split the
;;;  interval (0 A*).

(defun qmag-equivalent (qm1 qm2)
  (and qm1
       qm2
       (or (equal qm1 qm2)
	   (and (qmag-point-p qm1)
		(qmag-point-p qm2)
		(qmag-equivalent (car  (lmark-where-defined qm1))
				 (car  (lmark-where-defined qm2)))
		(qmag-equivalent (cadr (lmark-where-defined qm1))
				 (cadr (lmark-where-defined qm2))))
	   (and (qmag-interval-p qm1)
		(qmag-interval-p qm2)
		(qmag-equivalent (car  qm1) (car  qm2))
		(qmag-equivalent (cadr qm1) (cadr qm2))))))



;;;=============================================================================
;;;		     E N V I S I O N M E N T   F I L T E R
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; A new filter is added to the list of global filters. It must be performed 
;;; after all the others
;;;-----------------------------------------------------------------------------

(unless (member 'envisionment-filter *global-state-filters*)
  (setq *global-state-filters*
	(nconc *global-state-filters* `(envisionment-filter))))
	

;;;-----------------------------------------------------------------------------
;;; The filter that does the work
;;;-----------------------------------------------------------------------------

(defun envisionment-filter (state)
  (if *envisionment*
    (let ((*check-for-cycles* :weak)
	  matching)
      (dolist (candidate (sim-states *current-sim*) state)
	(when (and (good-matching-candidate-p state candidate)
		   (setq matching (match-states state candidate)))
	  ;; set successors and status of the current state, but this is not
	  ;; very useful; that state won't be part of the graph
	  (setf (state-successors state) `(Cycle-identity ,candidate))
	  (pushnew 'cycle (state-status state))
	  (pushnew `(cycle (,matching)) (state-status state) :test #'equal)

	  (substitute-successor
	    (second (state-justification state)) state candidate)
	  (when *trace-cycle-detection*
	    (format *QSIM-Trace*
		    "~%~a Cycle identified:  ~6a = ~6@a which now has ~a as predecessors"
		    matching state 
		    candidate (state-predecessors candidate)))
	  (return-from envisionment-filter nil))))
    state))


;;;-----------------------------------------------------------------------------
;;; (SUBSTITUTE-SUCCESSOR pred state succ) substitutes state with succ in the
;;; list of pred's successors and pushes pred onto the list of succ's
;;; predecessors.
;;;-----------------------------------------------------------------------------

(defun SUBSTITUTE-SUCCESSOR (pred state succ &aux junk)
  ;; push pred onto succ's list of predecessors
  (push pred (state-predecessors succ))
  ;; Nsubstitute -> substitute, PF 04 June 1991
  (setf (state-successors pred) (substitute succ state (state-successors pred)))
  ;;(nsubstitute succ state (state-successors pred))
  (case (car (state-justification succ))
    (one-of-several-completions-of
     ;; succ was one completion -> remove it from the completions of its
     ;; initial predecessor and set links with its new pred.
     ;; No (setf (state-successors ...) ...) is required because
     ;; (car (state-successors ...)) is always 'successors.
     ;; added junk to get rid of compilation warning: the value of (del.. is not used
     (setq junk (delete succ (state-successors (second (state-justification succ)))))
     (setf (state-justification succ)
	   (if (third (state-successors pred))	;several state ?
	       `(one-of-several-successors-of
		  ,pred ,(cdr (state-successors pred)))
	       `(unique-successor-of ,pred)))
     (setf (state-predecessors succ) (list pred)))
    (t nil)))


;;;-----------------------------------------------------------------------------
;;; When trying to match two states, the two states must not be eq, and its
;;; status must not be one of (cycle incomplete transition)
;;;-----------------------------------------------------------------------------

(defun good-matching-candidate-p (state candidate)
  (and (not (eq state candidate))
       (not (intersection (state-status candidate) '(cycle incomplete transition)))))
