;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)

#|
*******************************************************************************
		  F O C U S I N G   T E C H N I Q U E S

 Pierre Fouche <fouche@frutc51.bitnet>
 First version: June 1990
 Revised: November 1990, May 1991

This file contains function to analyze an envisionment graph and aggregate 
similar states / behaviors. Techniques implemented here are described in detail 
in "Abstracting Irrelevant Distinctions in Qualitative Simulation" by Pierre 
Fouche and Benjamin Kuipers, proceedings of QR'91, Austin, TX, May 1991.

The file is organized as follow:
  1.Partitionning
  2.Collecting and gathering differences
  3.Putting everything together
  4.Top-level functions
*****************************************************************************
|# 

; Moved to qdefs.lisp 
;(defparameter *trace-aggregation* t 
; "Turn on/off tracing of the chatter detection algorithm")


;;;=============================================================================
;;; Partitionning
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; (ALL-STATES <state>) returns the list of all the states attainable from
;;; <state>.
;;;-----------------------------------------------------------------------------

(defun all-states (&optional (initial-state *initial-state*) &aux states)
  (mapn #'(lambda (state) (push state states)) initial-state)
  (nreverse states))


;;;-----------------------------------------------------------------------------
;;;(QVAL--QMAG-EQUIVALENT-P <qv1> <qv2>) is true iff <qv1>'s and <qv2>'s
;;;qmags are equal or undefined.
;;;(QVAL--VAR-EQUIVALENT-P  <qv1> <qv2>) is true iff:
;;;  - <qv1>'s variable is not an *interesting-variables*, or
;;;  - <qv1> and <qv2> are qval-compatible.
;;;-----------------------------------------------------------------------------

(defun qval--qmag-equivalent-p (qv1 qv2)
  "True if two qvals have equivalent qmags"
  (if (and (qval-p qv1) (qval-p qv2))
      ;; Both arguments are qvals.
      ;; See if qmags are equivalent.
      (or (equal (qval-qmag qv1) (qval-qmag qv2))
	  (member (qmag qv1) *unspecified-value*)
	  (member (qmag qv2) *unspecified-value*))
      ;; Error.
      (error "~a or ~a is not a qval!" qv1 qv2)))

(defun qval--var-equivalent-p (qv1 qv2 &aux (var (variable-name (qval-variable qv1))))
  "True if two qvals have equivalent qmags"
  (declare (special *interesting-variables*))
  (if (member var *interesting-variables*)
      (qval-compatible qv1 qv2)
      t))


;;;-----------------------------------------------------------------------------
;;; (STATE-EQUIVALENT <state1> <state2> :test #'<equivalent-p>) is T iff:
;;;    - <state1> and <state2> are built from the same qde,
;;;    - all the qvals of <state1> and <state2> are <equivalent-p>.
;;; STATE-QMAG-EQUIVALENT and STATE-VAR-EQUIVALENT are defined for convenience.
;;;-----------------------------------------------------------------------------

(defun state-equivalent (s1 s2 &key (test #'qval-equal))
  (and (eq (state-qde s1) (state-qde s2))
       (every #'(lambda (var-qval1 var-qval2) 
                  (funcall test (cdr var-qval1) (cdr var-qval2)))
              (cdr (state-qvalues s1)) (cdr (state-qvalues s2)))))

(defun state-qmag-equivalent (s1 s2)
  (state-equivalent s1 s2 :test #'qval--qmag-equivalent-p))

(defun state-variable-equivalent (s1 s2)
  (state-equivalent s1 s2 :test #'qval--var-equivalent-p))

;(defun state-variable-equivalent (s1 s2)
;  (declare (special *interesting-variables*))
;  (and (eq (state-qde s1) (state-qde s2))
;       (every #'(lambda (var-qval1 var-qval2 &aux (var (car var-qval1))) 
;                  (if (or (not *interesting-variables*)
;                          (member var *interesting-variables*))
;                      (qval-compatible (cdr var-qval1) (cdr var-qval2))
;                      t))
;              (cdr (state-qvalues s1)) (cdr (state-qvalues s2)))))


;;;-----------------------------------------------------------------------------
;;; (PARTITION-STATES <states> <relation> <relation-name>) partitions <states>
;;; into <relation> equivalent states and returns the partition. <relation-name>
;;; is used for tracing
;;;-----------------------------------------------------------------------------

(defun partition-states (states relation relation-name &aux partition)
  (when *trace-aggregation*
    (format *qsim-trace* "~%  ~a states to partition into ~a classes.~%  Partitioning..."
	    (length states) relation-name))
  (setq partition (npartition states :test relation))
  (when *trace-aggregation* 
    (format *qsim-trace* "~%  Found a total of ~a classe(s), ~a containing more than one element."
	    (length partition) (length (remove-if-not #'cdr partition))))
  partition)


;;;=============================================================================
;;; Collecting and gathering differences
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;;ALL-QDIRS returns the list of possible qdirs for all the state variables,
;;;given a list of states.
;;;It returns a list like ((X std) (Y inc std dec) (Z dec))
;;;ALL-QDIRS-AND-QMAGS does the same, but returns a list like
;;; ((X 0) (Y (0 inf)) (Z (0 inf) 0 (minf 0))) as second value.
;;;-----------------------------------------------------------------------------

(defun all-qdirs (states)
  (let ((var-qdir-mask
            ;; var-qdir-mask is something like ((X) (Y) (Z))
            (mapcar #'(lambda (var-qval) (list (car var-qval)))
                    (cdr (state-qvalues (first states))))))	;skip over Time
    (dolist (state states)
      (mapc #'(lambda (var-qval var-qdir)
		(pushnew (qval-qdir (cdr var-qval)) (cdr var-qdir)))
	    (cdr (state-qvalues state)) ;skip over TIME
	    var-qdir-mask))
    var-qdir-mask))


(defun all-qdirs-and-qmags (states)
  (let* ((var-qdir-mask
            ;; var-qdir-mask is something like ((X) (Y) (Z))
            (mapcar #'(lambda (var-qval) (list (car var-qval)))
                    (cdr (state-qvalues (first states)))))
         (var-qmag-mask (copy-tree var-qdir-mask)))
    (dolist (state states)
      (mapc #'(lambda (var-qval var-qdir var-qmag)
		(pushnew (qval-qdir (cdr var-qval)) (cdr var-qdir))
                (pushnew (qval-qmag (cdr var-qval)) (cdr var-qmag)))
	    (cdr (state-qvalues state)) ;skip over TIME
	    var-qdir-mask
            var-qmag-mask))
    (values var-qdir-mask var-qmag-mask)))


;;;-----------------------------------------------------------------------------
;;; QMAG< is an ordering relation tha can be used as an argument to SORT.
;;;-----------------------------------------------------------------------------

(defun qmag< (qmag1 qmag2)
  (declare (special *qspace*))
  (ecase (qmag-order qmag1 qmag2 *qspace*)
    (- t)
    ((0 + nil) nil)))


;;;-----------------------------------------------------------------------------
;;; Given a set of qmags, MAXIMIZE-QMAG returns the smallest qmag containing
;;; all the qmags.
;;; That code is not quite efficient, but was so easy to write...
;;;-----------------------------------------------------------------------------

(defun maximize-qmag (qmags *qspace*)
  (declare (special *qspace*))
  (let* ((sorted-qmags (sort qmags #'qmag<))
         (lowest (first sorted-qmags))
         (greatest (car (last sorted-qmags))))
    (list (if (consp lowest) (car lowest) lowest)
          (if (consp greatest) (cadr greatest) greatest ))))

        

;;;-----------------------------------------------------------------------------
;;;(BUILD-NEW-STATE <state> <qvalues>) returns a new state identical to
;;;<state> with new qvalues.  This function is called when building more
;;;abstract states.
;;;-----------------------------------------------------------------------------

(defun build-new-state (state qvalues)
  (let* ((nstate
	  (make-state :qde       (state-qde state)
		      :qvalues   qvalues
		      :qspaces   (copy-alist (state-qspaces state))
		      :cvalues   (copy-alist (state-cvalues state))
                      ;;:other     (copy-tree  (state-other   state))
		      :name      (genname 'S)
		      )))
    (set (state-name nstate) nstate)    ;; Note: SET, not SETF
    (setf (state-sim nstate) (state-sim state))
    (setf (state-display-block nstate) (state-display-block state))
    nstate))


;;;-----------------------------------------------------------------------------
;;; MAKE-DUMMY-QVAL returns a new qval of the specified type. 
;;; Currently, only used to create new qvals for Time.
;;;-----------------------------------------------------------------------------

(defun make-dummy-qval (variable &key (type :interval) (qdir 'inc)
                                 (name "Dummy") 
                                 (where-defined "make-dummy-qval")
                                 when-defined why-defined)
  (let* ((lmark (make-lmark :name name 
                            :when-defined when-defined
                            :where-defined where-defined
                            :why-defined why-defined)))
    (make-qval :variable variable
               :qmag (ecase type
                       (:interval (list lmark (copy-lmark lmark)))
                       (:point lmark))
               :qdir qdir)))               


;;;-----------------------------------------------------------------------------
;;; (GROUP-QMAG-EQUIVALENT-STATES <states>) returns:
;;;   - (car <states>) if <states> contains only one element (in that case, no
;;;     need to group any states)
;;;   - a new state in which qdirs of variables having more than one qdir are
;;;     ignored.  It returns the list of those variables as second value.
;;;-----------------------------------------------------------------------------

(defun group-qmag-equivalent-states (states)
  ;; special case when there is only one state in states
  (unless (cdr states) 
    (cerror "It will return ~a" 
            "GROUP-QMAG-EQUIVALENT-STATES called on a list of only one state: ~a"
            (car states))
    (return-from group-qmag-equivalent-states (car states)))
  ;; general case: some qdirs must be ignored.
  (let* ((qdirs (all-qdirs states))
	 (state (first states))
         (time-variable (first (state-variables state)))
         (qvalues (copy-alist (state-qvalues state)))
	 chattering-variables)
    ;; set time qval in the new qvalues first
    (setf (cdar qvalues) (make-dummy-qval time-variable :name "T*" :qdir 'inc))
    ;; set qdirs of other qvals to 'ign when appropriate
    (mapc #'(lambda (var-qval var-qdir &aux qval)
	      (when (cddr var-qdir)		      
                ;; This variable has several qdirs in the qmag-equivalent states
                ;; -> ignore that qdir.
		(setq qval (copy-qval (cdr var-qval)))  ;copy a qval,
                (setf (qval-qdir qval) 'ign)            ;ign its qdir,
		(setf (cdr var-qval) qval)              ;put it in the qvalues list,
		(push (car var-qval) chattering-variables))) ;and declare the variable as chattering.
	  (cdr qvalues)  ;skip over time
          qdirs)
    (values (build-new-state state qvalues) ;; build a new state with appropriate qvalues
           chattering-variables)))


;;;-----------------------------------------------------------------------------
;;; (GROUP-VARIABLE-EQUIVALENT-STATES <states>) returns:
;;;   - (car <states>) if <states> contains only one element (in that case, no
;;;     need to group any states)
;;;
;;;   - a new state in which qvals of non-interesting variables are aggregated.
;;;     It returns the list of those variables as second value.
;;;     If a non-interesting variable has several possible qdirs, its qdir is
;;;     ignored. Its qmag is the minimal interval containing all its possible 
;;;     qmags.
;;;
;;;     Note: this can be a problem sometimes, for that minimal interval can be
;;;     a closed interval, and there is no way currently in QSIM to distinguish
;;;     between closed and opened intervals.
;;;-----------------------------------------------------------------------------

(defun group-variable-equivalent-states (states)
  (declare (special *interesting-variables*))
  ;; special case when there is only one state in states
  (unless (cdr states) 
    (cerror "It will return ~a" 
            "GROUP-VARIABLE-EQUIVALENT-STATES called on a list of only one state: ~a"
            (car states))
    (return-from group-variable-equivalent-states (car states)))
  ;; general case: some qdirs and qmags must be ignored.
  (let* ((state (first states))
         (time-variable (first (state-variables state)))
         (qvalues (copy-alist (state-qvalues state)))
         multiple-value-variables qdirs qmags)
    (multiple-value-setq (qdirs qmags) (all-qdirs-and-qmags states))
    ;; set time qval in the new qvalues first
    (setf (cdar qvalues) (make-dummy-qval time-variable :name "T*" :qdir 'inc))
    ;; set qdirs of other qvals when appropriate
    (mapc #'(lambda (var-qval var-qdir var-qmag &aux (var (car var-qval)) qval qspace)
	      (when (cddr var-qdir)		      
                ;; This variable has several qdirs in the qmag-equivalent states
                ;; -> ignore that qdir.
		(setq qval (copy-qval (cdr var-qval)))  ;copy a qval,
                (setf (qval-qdir qval) 'ign)            ;ign its qdir,
		(setf (cdr var-qval) qval)              ;put it in the qvalues list,
		(push (car var-qval) multiple-value-variables)) ;and declare the variable as multiple-value.
              (when (cddr var-qmag)
                (setq qspace (qspace var state))
		(setq qval (copy-qval (cdr var-qval)))  ;copy a qval,
                (setf (qval-qmag qval) (maximize-qmag (cdr var-qmag) qspace))            ;maximize its qmag,
		(setf (cdr var-qval) qval)              ;put it in the qvalues list,
		(pushnew (car var-qval) multiple-value-variables))) ;and declare the variable as multiple-value.
          (cdr qvalues)  ;skip over time
          qdirs qmags)
    (values (build-new-state state qvalues) ;; build a new state with appropriate qvalues
            multiple-value-variables)))


;;;=============================================================================
;;;Putting everything together
;;;=============================================================================

;;;--------------------------------------------------------------------------------
;;;(UPDATE-SUCCESSORS <state> <old-level> <new-level>) computes the successors of 
;;;<state> from its successors in <old-level> and their coarsenings in <new-level>.
;;; Similarly for UPDATE-PREDECESSORS.
;;;--------------------------------------------------------------------------------

(defun update-successors (state old-level new-level)
  (let ((*check-level* nil)
        (succs ()))
    (dolist (succ (get-successors state old-level))
      (pushnew (or (get-coarsening succ new-level) succ) succs))
    (nreverse succs)))

(defun update-predecessors (state old-level new-level)
  (let ((*check-level* nil)
        (preds ()))
    (dolist (pred (get-predecessors state old-level)) 
      (pushnew (or (get-coarsening pred new-level) pred) preds))
    preds))


;;;--------------------------------------------------------------------------------
;;;(CREATE-NEW-ABSTRACTION-LEVEL <states> <old-level> <new-level>) sets successors 
;;;and predecessors of states in <states> in level <new-level>.
;;;--------------------------------------------------------------------------------

(defun create-new-abstraction-level (states display-block old-level new-level)
  (or old-level (setq old-level (get-basic-level display-block)))
  (nconc (display-block-levels display-block) (list new-level))
  (push new-level (level-upper-levels old-level))
  (setf (level-lower-level new-level) old-level)
  (dolist (state states)
     (pushnew new-level (state-levels state))
     (set-successors state (update-successors state old-level new-level) new-level)
     (set-predecessors state (update-predecessors state old-level new-level) 
        new-level)))


;;;-----------------------------------------------------------------------------
;;; CONNECT-ABSTRACTION connects a coarser state (i.e. a state in which some
;;; qdirs are ignored) with original states.
;;;-----------------------------------------------------------------------------

(defun connect-abstraction (i-state finer-states status level new-level)
  (let ((coarser-states (list i-state))
        (time-variable (first (state-variables i-state)))
        i-succs i-preds p-succs p-preds p-state)
    ;; i-succs are successors of finer-states, not member of finer-states themselves,
    ;; lasting over an interval of time. Similar meaning for i-preds, p-succs, p-preds.
    ;; Set i/p - succs/preds first
    (dolist (finer-state finer-states) 
      ;;(dolist (s (successor-states finer-state))
      (dolist (s (get-successors finer-state level))
        (unless (member s finer-states)
          (if (i-state-p s) (pushnew s i-succs) (pushnew s p-succs))))
      ;;(dolist (s (state-predecessors finer-state))
      (dolist (s (get-predecessors finer-state level))
        (unless (member s finer-states)
          (if (i-state-p s) (pushnew s i-preds) (pushnew s p-preds)))))
    (setf (state-levels i-state) (list new-level))
    (setf (state-status i-state) `(,status))
    (cond ;;special case when only one abstracted state is necessary
      ((and (not i-succs) (not i-preds))
       (set-state-neighbours i-state p-preds p-succs level))
      ((and (not p-succs) (not p-preds))
       (setf p-state i-state)
       (setf (state-time p-state) (make-dummy-qval time-variable :type :point :name "T*" :qdir 'inc))
       ;;(qmag (state-time i-state)) t)
       (set-state-neighbours p-state i-preds i-succs level))
      (t 
       ;; general case: need to build another state
       (setq p-state (build-new-state i-state (copy-alist (state-qvalues i-state))))
       (setf (state-levels p-state) (list new-level))
       (push p-state coarser-states)
       (setf (state-status p-state) `(,status))
       (setf (state-time p-state) (make-dummy-qval time-variable :type :point :name "T*" :qdir 'inc))
       (set-state-neighbours p-state i-preds (cons i-state i-succs) level)
       (set-state-neighbours i-state (cons p-state p-preds) p-succs level)))
    ;; connect coarser states with finer states
    (dolist (finer-state finer-states)
      (cond ((and (p-state-p finer-state) p-state)
             (push finer-state (state-refinings p-state))
             (set-coarsening finer-state p-state new-level))
            (t (push finer-state (state-refinings i-state))
               (set-coarsening finer-state i-state new-level))))
    ;;(pop (state-levels i-state))
    ;;(when p-state (pop (state-levels p-state)))
    coarser-states))


;;;-----------------------------------------------------------------------------
;;; AGGREGATE-STATES is the main function called by the top-level functions
;;; focus-on...
;;; Given a partition of the set of states, it collects differences between
;;; states of the same class, then build a new abstract state and connects it
;;; with existing states.
;;;-----------------------------------------------------------------------------

(defun aggregate-states (states partition display-block level new-level aggregation-function)
  (let ((new-level-states states))
    (dolist (eqv-states partition)
      (when (cdr eqv-states)			;several qmag-equivalent states
	(multiple-value-bind (abstracted-state chattering-vars)
	    (funcall aggregation-function eqv-states)
	  (when chattering-vars
	    (setq abstracted-state (connect-abstraction abstracted-state eqv-states 'chatter level new-level))
	    (setq new-level-states (set-difference new-level-states eqv-states))
	    (nconc new-level-states abstracted-state)
	    (when *trace-aggregation*
	      (dolist (s abstracted-state)
		(format *qsim-trace*
			"~%  Qdirs and/or qmags of ~a aggregated in ~6a from ~a"
			chattering-vars s (state-refinings s))))))))
    (unless (equal new-level-states states)
      ;; At least one new state was created
      (create-new-abstraction-level new-level-states display-block level new-level))
    (cond ((equal new-level-states states)
	   (when *trace-aggregation*
	     (format *qsim-trace* "~%No change!"))
	   level)
	  (t (format *qsim-trace* "~%Done. Level ~a created." new-level)
	     new-level))))


;;;=============================================================================
;;; Top Level Functions
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;;(FOCUS-ON-VARIABLES <variables> :initial-state <state> :level <level>)
;;;partitions the set of attainable states from <state> at level <level> into
;;;<variable> equivalent classes.  It returns the new abstraction level, or the
;;;current level if no new level was created.
;;;-----------------------------------------------------------------------------

(defun focus-on-variables (*interesting-variables* 
			   &key (initial-state *initial-state*) (level *abstraction-level*))
  (declare (special *interesting-variables*))
  (when *trace-aggregation*
    (format *qsim-trace* "~2%Starting focusing on ~a at level ~a..." *interesting-variables* level))
  (let* ((*abstraction-level* level)
         (display-block (state-display-block initial-state))
         (states (all-states initial-state))
         (new-level (make-new-level level (format nil "Var eqv ~a" *interesting-variables*)))
         (partition (partition-states states #'state-variable-equivalent "variable-equivalent")))
    (aggregate-states states partition display-block level new-level #'group-variable-equivalent-states)))
    

;;;-----------------------------------------------------------------------------
;;;(FOCUS-ON-QMAGS :initial-state <state> :level <level>) partitions the set of
;;;attainable states from <state> at level <level> into qmag equivalent
;;;classes.  It returns the new abstraction level, or the current level if no
;;;new level was created.
;;;-----------------------------------------------------------------------------

(defun focus-on-qmags  (&key (initial-state *initial-state*) (level *abstraction-level*))
  (when *trace-aggregation*
    (format *qsim-trace* "~2%Starting focusing on qmags at level ~a..." level))
  (let* ((*abstraction-level* level)
         (display-block (state-display-block initial-state))
         (states (all-states initial-state))
         (new-level (make-new-level level "Qmag eqv"))
         (partition (partition-states states #'state-qmag-equivalent "qmag-equivalent")))
    (aggregate-states states partition display-block level new-level #'group-qmag-equivalent-states)))


;;;-----------------------------------------------------------------------------
;;; (RESET-ABSTRACTIONS <state> resets the abstraction-related slots of every 
;;; state attainable from <state>, so only the basic level remains.
;;;-----------------------------------------------------------------------------

(defun reset-abstractions (&optional (state *initial-state*))
  (let ((*abstraction-level* nil)
        (level (get-basic-level state)))
    (mapn #'(lambda (s)
              (setf (state-coarsening s) nil)
              (setf (state-refinings s) nil)
              (setf (state-new-successors s) nil)
              (setf (state-new-predecessors s) nil)
              (setf (state-levels s) (list level))
              (setf (display-block-levels (state-display-block state)) (list level)) t)
          state)))
