;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: dynamic-slots.lisp,v 1.8 91/09/05 13:13:01 clancy Exp $

(in-package :qsim) ;changed DJC

;; (in-package 'dynamic-slots)

;;; We output to the *qsim-report* stream.
;; (import '(qsim::*qsim-report*) 'DYNAMIC-SLOTS)

;(export
;  '(
;    DYNAMIC-SLOTS
;    DEFOTHER
;    DEFSLOT
;    UNDEFSLOT
;    DESCRIBE-SLOT
;    FIND-IN-PLIST
;    STRING-TO-SYMBOL
;    LOOKUPX
;    SET-TABLE
;    )
;  'DYNAMIC-SLOTS)

;;; 
;;; PURPOSE: to allow the user to define new slots for a structure.
;;; These normally are allocated on the <structure>-OTHER slot as a
;;; table (what Ben calls an alist), which is a list of (key value)
;;; pairs.  
;;;
;;; USAGE: (defother lmark real-value :type number) would define an
;;; accessor and setf function: lmark-real-value which, when applied to
;;; an lmark return or set the real-value slot.  This slot will be
;;; stored on the lmark-other slot of the lmark.
;;; One can now pretend that real-value was defined as a slot in the
;;; original defstruct. In fact, if it seems useful (frequent access),
;;; the defstruct could be changed at some point, and no other code
;;; would have to be changed. It would, of course, have to be
;;; recompiled, as the structure accessors are defined as macros or
;;; defsubsts.  I use functions for ease of redefinition, etc.
;;;
;;; Also: (describe-slot 'lmark-real-value) would provide some default
;;; info. Call (undefslot 'lmark 'real-value) to remove the associated
;;; definitions and information.
;;;
;;; Note: the same "slot-name" may be on several structures.  The
;;; accessors are differentiated by the structure name.
;;;

(defvar *dynamic-slot-definitions* nil
	"A plist (<structure> (<slot-name> <defn> ...) ...)")

(defstruct dynamic-slots
  "Define a structure to be used with  defother.  Include this
in a structure definition to get the dynamic other slots.  E.g~;
       (defstruct (foo :include dynamic-slots))."

  other
  )

;;;  The IGNORE keyword is a dummy keyword added so that this would compile for LUCID.
;;;  It does not allow the use of $allow-other-keys without a key specified.
(defmacro defother (structure slot &rest defslot-keywords &key ignore &allow-other-keys )
  "Define a new slot for a qsim-like structure with an OTHER slot."
  (declare (ignore ignore))
  `(defslot ,structure ,slot
     :location ,(string-to-symbol "~s-OTHER" structure)
     ,@defslot-keywords))

;;;
;;; To change defslot to use plists or alist for storing, just change lookupx and set-table
;;; to the appropriate functions/macros
;;;

(defmacro defslot (structure slot-name
		   &key
		   type
		   (name (string-to-symbol "~a-~a" structure slot-name))
		   location
		   description)
  "Define a dynamic slot with its own setf and accessor functions.  Use undefslot
to undefine the various functions which are created by defslot."
  (unless location (error "A location must be provided, e.g. foo-extra-slots if ~
                           the extra slots are going to be stuck on the extra-slots ~
                           slot of FOOs."))
  (let ((locator `(,location ,structure))
	(setf-function (string-to-symbol "set-~a" name)))
    (unless description 
      (setq description (format () "The ~a slot for ~a, called ~a."
				slot-name structure	
				name)))
    (when type (setq description 
		     (format nil "~a  Values for this slot must satisfy the type specifier:~%  ~a."
			     description type)))
    `(progn 
       ;; define the accessor
       (defun ,name (,structure)
	 (lookupx ,locator ',slot-name))

       ;; define the setf-function
       (defun ,setf-function (,structure val)
	 ,@(when type
	     (let ((error-msg 
		     (format nil "The value, ~~s, for the slot ~a does not satisfy the type specifier: ~a"
			     name type)))
					   
	       `((unless (typep val ',type)
		   (error ,error-msg val)))))
	 (set-table ,locator ',slot-name val))

       ;; define the setf method
       (defsetf ,name ,setf-function)
       (eval-when (load eval)
	 (setf (getf (getf *dynamic-slot-definitions* ',structure)
		     ',slot-name)
	       (list :definitions '(,name ,setf-function)
		     :slot-name ',slot-name
		     :name ',name
		     ,@(when description `(:description ,description))
		     ,@(when type `(:type ',type))
		     ))))))

;;;----------------------------------------------------------------------
;;;
(defun undefslot  (structure slot-name)
  "Undefine the accessor and settor functions defined by defslot."
  (let ((defs (getf (getf (getf *dynamic-slot-definitions* structure)
			  slot-name)
		    :definitions)))
    (cond ((null defs)
	   (cerror "Do nothing" "Slot ~a is not defined." slot-name))
	  (t (mapc #'fmakunbound
		   defs)
	     (remf  (getf *dynamic-slot-definitions* structure)
		    slot-name)))))


;;;----------------------------------------------------------------------
;;;

(defun describe-slot (name &optional structure)
  "Describe a dynamic slot called NAME.  If the optional STRUCTURE is~:
   provided, then describe STRUCTURE-NAME."
	 (print-slot-definition
	   (if structure
	       (getf (getf *dynamic-slot-definitions* structure) name)
	       (find-slot-def-from-name name *dynamic-slot-definitions*))
	   *qsim-report*))

;; this function is really simple, but because we dont have a good plist
;; iteration function it looks sloppy.  I guess that I should have used alists. 

(defun find-slot-def-from-name (name plist)
  (cond ((null plist)
	 (format *qsim-report* "There is no dynamic slot named ~a defined for any structure." name))
	(T (or (find-in-plist (second plist)
		 #'(lambda (slot-defn)
		     (eql (getf  slot-defn :name) name)))
	       (find-slot-def-from-name name (cddr plist))))))

;; this doesn't need to be fast, and a good compiler should make it ok.
(defun find-in-plist (plist  test)
  "Find the value in plist which satisfies the test."
  (if (null plist) 
      nil
      (if (funcall test (second plist))
	  (second plist)
	  (find-in-plist (cddr plist) test))))

(defun print-slot-definition (defn &optional (str *standard-output*))
  (fresh-line str)
  (format str (getf defn :description))
  (values))


;;;======================================================================
;;;			       UTILITIES
;;;

(defun string-to-symbol (string &rest format-args)
  "Actually more general than it sounds.  Will return the sexpr
which results from reading the formated string."
  (read-from-string
    (format nil "~?" string format-args)))


;;;----------------------------------------------------------------------
;;;
;;;  Table functions.  A table is a list  of (key value) pairs.
;;;
(defmacro lookupx (table key)
  `(cadr (assoc ,key ,table)))

(defmacro set-table (loc key val)
  `(let ((item (assoc ,key ,loc)))
     (if item
	 (setf (second item)  ,val)
	 (setf ,loc
	       (cons (list ,key ,val) ,loc)))
     ,val))

;;;======================================================================
;;; examples
#|
(defstruct (s1 
	     (:include dynamic-slots))
  bar)
(defslot s1 foo :location s1-other :type integer)
(setq a (make-s1))
(setf (s1-foo a) 10)
(s1-foo a)
(setf (s1-foo a) "a string")
(s1-foo a)

(defother s1 bar)
(setf (s1-bar a) "a string")
(s1-bar a)
|#
