;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-
;;;  $Id: define-qde.lisp,v 1.7 92/05/27 12:17:34 bert Exp $
;;; Benjamin Kuipers, 1988.  Daniel Dvorak, 1989.

(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;  The clauses specified by the user in 'define-qde' are in a form for easy
;;;  visual inspection and editing.  The purpose of the 'define-qde' macro is
;;;  to convert that information into an efficient internal representation.
;;;  The end result of 'define-qde' is that each piece of user-specified
;;;  information gets placed somewhere in one of the following structures:
;;;  QDE, VARIABLE, CONSTRAINT, or LMARK.
;;;-----------------------------------------------------------------------------


;;; DEFINE-QDE is the user's way to define a QDE.
;;; It's a PROG instead of a LET to allow the QDE name to become special.

(defmacro DEFINE-QDE (name &rest clauses)
  `(prog ((qde (make-it-plain ',name ',clauses)))
	 (check-syntax qde ',clauses)
	 (initialize-constraint-network qde ',clauses)
	 (setf (get ',name 'qde-definition)
	       ',clauses)
	 (defparameter ,name qde)
	 (critique-qde qde)))


;;; MAKE-IT-PLAIN actually builds the <QDE>.
;;; It installs some of the slots from the user supplied specs.
;;; The keyword BUILD-Fn allows for building defstructs which inherit from QDE,
;;; as well as qde's.

(defun MAKE-IT-PLAIN (name clauses &key (build-fn 'make-QDE))
  (funcall build-fn
	   :name          name
	   :transitions   nil
	   :independent   (alookup 'independent clauses)
	   :history       (alookup 'history clauses)
	   :text          (alookup 'text clauses)
	   :layout        (alookup 'layout clauses)
	   :other         (alookup 'other clauses)))


; SYNTAX:  a PARAM is represented as (param-name (L1 L2 ... Ln))
; where (L1 L2 ... Ln) is its quantity space.


; SYNTAX:  a CONSTRAINT is represented as 
;   ((ctype pn1 pn2 pn3 (x1 y1) (x2 y2)) (cv1 cv2 cv3) (cv4 cv5 cv6) ... )
;         parameters = pn1 pn2 pn3
;         bend-points = (x1 y1) (x2 y2)      ; for S+, etc.
;         corresponding values = (cv1 cv2 cv3) . . . 

; The param-list now consists of the atomic elements in (cdar constraint).
; Non-atomic elements (e.g. bend-points) are returned by (constraint-descriptions constraint).


(defun SHARE-QSPACE-WITH (varname qde)  ; qde arg no longer optional BKay 27May92
  (remove varname (car (hqspace-clause varname qde))))


(defun HQSPACE-CLAUSE (varname qde)
  (dolist (clause (alookup 'qspace-hierarchy (qde-other qde)))
    (if (member varname (car clause))
	(return clause))))


; Still needs:
;   - more consistent usage of syntax access functions and variable names.


;;; Macros and functions for retrieving the VALUE OF A PARAMETER with a form 
;;; similar to P(t):  (qvalue= (param state)).
;;; A value perturbed from the current value is produced by qvalue+ and qvalue-.
;;; Param is not evaluated; state is.
;;; Modified 26 Nov 90 by Mallory so macros call functions containing the code.
;;;  The functions are simpler to call in programs and create less garbage.


(defmacro QVALUE= ((varname state))
  `(qmag= ',varname ,state))


(defun QMAG= (varname state)
  (let* ((xstate (if (state-p state) state (eval state)))
	 (qval   (qval varname xstate))
	 (qmag   (qmag qval)))
    (if (qmag-point-p qmag)
	(lmark-name qmag)
	(list (lmark-name (first qmag)) (lmark-name (second qmag))))))

;;; Old definition:  `(qmag (lookup ',variable (state-values (eval ,state)))))


(defmacro QVALUE+ ((varname state))
  `(qmag+ ',varname ,state))


(defun QMAG+ (varname state)
  (let* ((xstate (if (state-p state) state (eval state)))
	 (qspace (qspace varname xstate))
	 (qval   (qval varname xstate))
	 (qmag   (qmag qval)))
    (if (qmag-interval-p qmag)
	(list (lmark-name (first qmag))
	      (lmark-name (second qmag)))
	(list (lmark-name qmag)
	      (lmark-name (succ qmag qspace))))))


(defmacro QVALUE- ((varname state))
  `(qmag- ',varname ,state))


(defun QMAG- (varname state)
  (let* ((xstate (if (state-p state) state (eval state)))
	 (qspace (qspace varname xstate))
	 (qval   (qval varname xstate))
	 (qmag   (qmag qval)))
    (if (qmag-interval-p qmag)
	(list (lmark-name (first qmag))
	      (lmark-name (second qmag)))
	(list (lmark-name (pred qmag qspace))
	      (lmark-name qmag)))))


(defun QDIR= (varname state)
  (let* ((xstate (if (state-p state) state (eval state))))
    (qval-qdir (qval varname xstate))))




; Need:
;  - check that bend points in S+/S- and U+/U- are ordered appropriately.
;  - redo access functions for variable-list and bend-points to separate 
;    atoms (= variables) from lists (= descriptions).


;;;-----------------------------------------------------------------------------
;;;  Function:  (INITIALIZE-CONSTRAINT-NETWORK  qde clauses)
;;;
;;;  Given:     -- qde, the qde structure instance to be initialized
;;;             -- clauses, the literal clauses of the define-qde form.
;;;
;;;  Returns:   nothing of significance.
;;;
;;;  Purpose:   This function transforms the QDE-style quantity-spaces and
;;;             constraints (which is in a nice human-readable form) into a
;;;             set of interconnected structures that is better suited to the
;;;             computations done in cfilter.  The overhead of this conversion
;;;             is performed only once, at the time of define-qde.
;;; Specificly: This processes the following clauses: QUANTITY-SPACES  OTHER
;;;		PRINT-NAMES CONSTRAINTS TRANSITIONS and DISCRETE-VARIABLES
;;; 		Within the OTHER clause, it process IGNORE-QDIRS IGNORE-QVALS 
;;;		NO-NEW-LANDMARKS and UNREACHABLE-VALUES 
;;; Changes: eval-initial-ranges call added by BKay 3Sept91 so that
;;;          the initial-ranges slot ranges are evaluated.  This lets
;;;          initial ranges be arbitrary lisp forms rather than just numbers.
;;;-----------------------------------------------------------------------------


(defun INITIALIZE-CONSTRAINT-NETWORK (qde clauses)

  ;; Save the variables, variable-name cross-reference list, and
  ;; initial landmark values in the qde.
   
  (add-defother-clauses qde clauses)		; These must be done before BUILD-VARIABLES-LISTS 
  (eval-initial-ranges qde)                     ; BKay 3Sept91
  (compile-q2-envelopes qde)                    ; BKay 23May92
  (add-other-alists qde clauses)		;  accesses the OTHER slot.
  (build-variables-lists qde (alookup 'quantity-spaces clauses)
			 (alookup 'discrete-variables clauses))
  (mark-unreachable-values qde)
  (set-titles-and-prefixes qde (alookup 'print-names clauses)
			   (alookup 'quantity-spaces clauses))
  (mark-constraint-variables qde (alookup 'constraints clauses))
  (fill-neighboring-constraints qde)
  ;; Convert the "transitions" list.
  (convert-transitions-list qde (alookup 'transitions clauses))
  ;; Call user's initialization function, if supplied.  This allows for
  ;; private initialization/setup.
  (when (fboundp 'qde-initialize)
    (funcall 'qde-initialize qde clauses)))


;;; Replace the ranges in the initial-ranges slot with their eval'ed value
;;; Example: (initial-ranges ((foo f1) ((+ 3 2) 6))) ->
;;;                 (initial-ranges ((foo f1) (5 6)))
(defun eval-initial-ranges (qde)
  (let ((initial-ranges (qde-initial-ranges qde)))
    (mapc #'(lambda (var&qval-bounds)
	      (setf (caadr var&qval-bounds)
		    (eval (caadr var&qval-bounds)))
	      (setf (cadadr var&qval-bounds)
		    (eval (cadadr var&qval-bounds))))
	      initial-ranges)))


;;; Compile the q2 envelopes functions in the m-envelopes clause.
;;; This will make them run faster and cons less.
;;; BKay 23May92
;;;
(defun compile-q2-envelopes (qde)
  (loop for env-for-constraint in (qde-m-envelopes qde)
	do (loop for env-clause in (cdr env-for-constraint)
		 do
		 ;; There may be clauses in the envelope set that
		 ;; we don't want to compile here.
		 ;; Note that someday we may even want to preserve
		 ;; the uncompiled form for symbolic manipulation.
		 ;; This clause will clobber the original s-expr.
		 (when (and (member (first env-clause)
				    '(upper-envelope upper-inverse
				      lower-envelope lower-inverse))
			    (not (and (symbolp (second env-clause))
				      (fboundp (second env-clause))
				      (compiled-function-p
				       (symbol-function(second env-clause))))))
		   (let ((fname (gentemp "F")))
		     (when *trace-compilation-of-functions*
		       (format *qsim-report*
			       "~%Compiling ~a function of ~a as ~a"
			       (first env-clause) (first env-for-constraint)
			       fname))
		     (setf (second env-clause)
			   (compile fname
				    (typecase (second env-clause)
				      (symbol
				       (if (fboundp (second env-clause))
					   (symbol-function (second env-clause))
					   (error "No function defined for ~a~
                                                   in ~a of ~a"
						  (second env-clause)
						  (first env-clause)
						  (first env-for-constraint))))
				      (T
				       (second env-clause))))))))))


;;;-----------------------------------------------------------------------------
;;; At the end of BUILD-VARIABLES-LISTS, 
;;;  (qde-qspaces qde)   := ((TIME <lmark 0> <lmark inf>) (VAR-SYMBOL . (list of lmarks)) ...)
;;;  (qde-var-alist qde) := ((VAR-SYMBOL . <var>)...)
;;;  (qde-variables qde) := (<var> <var> ...) 
;;;-----------------------------------------------------------------------------


(defun BUILD-VARIABLES-LISTS (qde  qspace-clause discrete-variables-clause
			      &key (existing-variables)(existing-lmarks))
  (when (alookup 'ignore-qvals (qde-other qde))
    (warn "Ignore-Qvals is being ignored since it has bugs."))
  ;; clear out anything that shouldn't be there
  (setf (qde-var-alist qde) nil
	(qde-qspaces qde) nil
	;; establish LMARKS-ALIST for scratch storage for shared qspaces
	(qde-lmarks-alist qde) nil)
  (install-time qde (copy-list (alookup 'time existing-lmarks))
		(find 'time existing-variables :key 'lmark-name))
  ;; Generate cross-reference alist of variables.
  (cross-reference-variables qspace-clause qde nil
			     existing-variables existing-lmarks)
  (cross-reference-variables discrete-variables-clause qde t
			     existing-variables existing-lmarks)
  (setf (qde-variables qde) (nreverse (qde-variables qde))
	(qde-qspaces qde)   (nreverse (qde-qspaces qde))
	;; erase scratch storage.
	(qde-other qde)(remove 'lmarks-alist (qde-other qde) :key #'car)))


;;; INSTALL-TIME creates "time" as the first variable and first qspace.
;;; Other QSIM functions rely on the fact that "time" will always be the
;;; first variable or first qspace.

(defun INSTALL-TIME (qde existing-qspace existing-var)
  (let* ((time-qspace
	   (or existing-qspace
	       `(,(make-lmark :name (initial-time)
			      :when-defined
			      (format nil "INSTALL-TIME for ~s" (qde-name qde)))
		 ,*inf-lmark*)))
	   (time-var  (or existing-var
			   (make-variable :name   'time
				       :-qspace time-qspace))))
	 (push time-var (qde-variables qde))
	 (push (cons 'time time-qspace) (qde-qspaces qde))
	 (push (cons 'time time-var) (qde-var-alist qde)) ))


(defun CROSS-REFERENCE-VARIABLES (variables qde discrete existing-variables existing-lmarks)
  (loop for ovariable in variables
	for ovname  = (car ovariable)
	for exist-v = (var-find ovname existing-variables)
	for exist-q = (alookup ovname existing-lmarks)
	do (create-a-variable ovariable qde discrete
			      :existing-variable exist-v :existing-qspace exist-q)))


;;; For every qde slot that that has been defined with DEFOTHER, find
;;; any such entry in clauses and put it and its associated value into
;;; the QDE-OTHER slot.
 
(defun ADD-DEFOTHER-CLAUSES (qde clauses)
  (loop with qde-dyn-others = (dynamic-qde-others)
	for (uclausename . val) in clauses
	for memb-p = (member uclausename qde-dyn-others)
	when (and memb-p (assoc uclausename (qde-other qde)))
	  do (rpt "OTHER-SLOT ~s seems to be specified twice in definition of ~s"
	       uclausename qde)
	when memb-p
	  do (push (list uclausename val) (qde-other qde))))


(defun DYNAMIC-QDE-OTHERS ()
  (loop for (clausename doc)
		    on (getf *dynamic-slot-definitions* 'qde) by #'cddr
		collect clausename))


;;; ADD-OTHER-ALISTS is similar to ADD-DEFOTHER-CLAUSES.
;;; However, the OTHER slot syntax isn't entirely standardized.
;;; Things defined with the DEFOTHER form store their value in
;;;   (cadr (assoc slotname (qde-other qde))) 
;;; while the slots listed in  *OTHER-ALIST* store their value in
;;;   (cdr (assoc slotname (qde-other qde)))
;;; So we need two functions to store the values separately.
;;; Notice however, as a toplevel clause in the DEFINE-QDE form,
;;; everything is specified as
;;;   (slotname . val)

(defun ADD-OTHER-ALISTS (qde clauses)
  (loop for (uclausename . val) in clauses
	for memb-p = (member uclausename *other-alist*)
	when (and memb-p (assoc uclausename (qde-other qde)))
	  do (rpt "Other-Slot ~s seems to be specified twice in definition of ~s"
	       uclausename qde)
	when memb-p
	  do (push (cons uclausename val) (qde-other qde))))


;;;-----------------------------------------------------------------------------
;;; CREATE-A-VARIABLE modified by and CREATE-A-LANDMARK created by Pierre Fouche
;;; 03/20/90 so that landmarks with the same name are eq when the quantity 
;;; space is shared.
;;;
;;; Create-a-landmark returns an instance of lmark:
;;;	- predefined lmarks if minf, 0 or inf, 
;;;     - an already existing lmark if two variables share a qspace and the
;;;       lmark has already been created,
;;;	- a new instance of landmark otherwise.
;;;-----------------------------------------------------------------------------

;;; Given a varname, CREATE-A-VARIABLE checks to see if a variable by
;;; this name already exists, (say, in an associated qde), adds the
;;; varname to the QDE-Variables slot, and builds the list of lmarks for
;;; the qde-qspaces list. 

(defun CREATE-A-VARIABLE (ovariable qde discrete &key (existing-variable) (existing-qspace))
  (let* ((varname (first  ovariable))
	 (oqspace (second ovariable))
	 ;; Create the variable.
	 (variable (or existing-variable
		       (make-variable :name varname)))
	 ;; Create the qspace as a list of landmarks.
	 (qspace (mapcar #'(lambda (l)
			     (create-a-landmark l varname qde existing-qspace))
			 oqspace)))
    (push variable (qde-variables qde))
    ;; (qde-var-alist qde) is the cross-reference alist.
    (push (cons varname variable) (qde-var-alist qde))
    ;; The VARIABLE--QSPACE is used by the error-checking code before simulation
    (setf (variable--qspace variable) qspace)	
    (push (cons varname qspace) (qde-qspaces qde))
    (set-variable-switches variable qde discrete)
    ))
	

(defun CREATE-A-LANDMARK (lmark-name varname qde existing-qspace)
  ;; (qde-lmarks-alist qde): alist of (<lmark-name> . (<lmark> . (<varname>+)))
  ;; it is used to store the lmarks shared by several variables.
  (unless (or (and lmark-name (symbolp lmark-name))		; NIL is a symbol
	      (numberp lmark-name))
    (error "Bad object to become landmark: ~s" lmark-name))
  (or (lmark-find lmark-name existing-qspace)
      (case lmark-name
	(0     *zero-lmark*)
	(inf   *inf-lmark*)
	(minf  *minf-lmark*)
	(t (let* ((lmark-varnames (alookup lmark-name (qde-lmarks-alist qde)))
		  ;; looking for a tuple (<lmark> . (<varname>+))

		  (mates (share-qspace-with varname qde))	; mates is a list of varnames
		  ;; looking for variables sharing the same qspace
		  lmark)
	 
	     (cond ((intersection mates (cdr lmark-varnames))
		    ;; Found a common landmark !
		    (setq lmark (car lmark-varnames))
		    (push varname (cdr (alookup lmark-name (qde-lmarks-alist qde))))
		    ;; update the list of shared landmarks
		    lmark)
		   (t (setq lmark
			    (make-lmark :name lmark-name
					:why-defined
					(format nil "Definition of ~d in ~d"
						varname (qde-name qde))))
		      (push (cons lmark-name (list lmark varname))
			    (qde-lmarks-alist qde))
		      lmark)))))))


(defun SET-VARIABLE-SWITCHES (variable qde discrete)
  (when discrete
    (setf (variable-discrete-p variable) t))
  (when (member (variable-name variable)
		(qde-history qde))
    (setf (variable-history-p variable) t))
  (when (member (variable-name variable) (qde-independent qde))
    (setf (variable-independent-p variable) t))
  (when (member (variable-name variable) (alookup 'ignore-qdirs (qde-other qde)))
    (setf (variable-ignore-qdir-p variable) t))
  (when (member (variable-name variable)
		(alookup 'ignore-qvals (qde-other qde)))
    (format *qsim-report* "~%ignore-qval on ~a" (variable-name variable))
    (setf (variable-ignore-qval-p variable) t))
  (when (member (variable-name variable) (alookup 'no-new-landmarks (qde-other qde)))
    (setf (variable-no-new-landmarks-p variable) t)))


(defun SET-TITLES-AND-PREFIXES (qde  print-names qspaces)
  ;; Run through the print-names to set titles and prefixes.
  (loop for (varname qspace title1 abbrev1) in qspaces
	for (var title2 abbrev2) = (assoc varname print-names)
	for variable = (alookup varname (qde-var-alist qde) :test #'equal)
	for title = (or title1 title2)
	for abbrev = (abbreviation  abbrev1 abbrev2 varname)
	if variable
	  do (setf (variable-title variable)  title
		   (variable-prefix variable) abbrev)
	else do
	       (format *qsim-report* "~%Print-name for ~s in ~s has no qspace!"
		       varname (qde-name qde))))


(defun ABBREVIATION (a1 a2 varname)
  (or a1 a2
      (when *short-lmark-abbrev*
	(let ((stub (subseq (symbol-name varname) 0 1)))
	  (when (and (or (equal stub "S") (equal stub "T"))
		     (> (length (symbol-name varname)) 1))	; keep lmarks from using "S-" form,
	    (setf stub (subseq (symbol-name varname) 0 2)))	; which is preferred for states.
	  (intern stub :qsim)))))


(defun MARK-UNREACHABLE-VALUES (qde)
  ;; Attach unreachable-value lists to appropriate variables.
  ;; Example: "(unreachable-values (nfpu 0) (ss 0))"
  (let ((uvlists (alookup 'unreachable-values (qde-other qde))))
    (dolist (uvlist uvlists)
      (let* ((varname (car uvlist))
	     (var (alookup varname (qde-var-alist qde))))
	(when var
	  (setf (variable-unreachable-values var)
		(nconc (variable-unreachable-values var)           ; handles unreachable values for same var
		       (convert-to-lmarks (cdr uvlist) var)))))))) ; in sep. clauses  DJC  07/19/91


;;; Collect the constraint's variables.  The inner do-loop handles three
;;; different forms, as exemplified by (M+ A B), (S+ A B (C D) (E F)),
;;; and (SUM-ZERO A B C ...).  Basically, it collects variables from
;;; the constraint's argument list until it reaches the end of the
;;; list or reaches the limit on number of arguments, whichever comes
;;; first.  Thus, bend-points for S+/S-/U+/U- are not accidentally
;;; picked up as variables.  A special test is made for the "="
;;; constraint since its second argument can be a form such as
;;; (CONSTANT <qmag>).
;;; NOTE/BUG: This function has problems in that constraints within a mode don't get
;;; checked with equivfctn nor are they checked for paranthesis errors.  Are these things errors?
;;;
(defun MARK-CONSTRAINT-VARIABLES (qde con-clause &key (existing-constraints))
  ;; For each constraint, save a list of the variables it constrains.
  (setf (qde-constraints qde) nil)		; erase any bookkeeping cached here by interface
  (loop for  ocon in  con-clause
	do (cond ((not (consp ocon))
		  (error "~s is Garbage in Constraint clause ~s for~s"
			 ocon con-clause qde))
		 ((listp (car ocon))
		  (let* ((contype (if (atom (caar ocon))
				      (contype-from-name (caar ocon))
				      (contype-from-name (caaar ocon))))	; BJK: 10-26-90
			 (buildfcn (contype-buildfcn contype))
			 (equivfcn (or (contype-equivfcn contype)
				       'constraint-name)))
		    (or (find (car ocon) existing-constraints :key equivfcn :test 'equal)
			(funcall buildfcn ocon qde nil))))
		 ((eql 'mode (car ocon))
		  (mode-spec-constraint ocon qde))
		 ((member (car ocon) *known-constraint-types*
			  :key #'contype-name)
		  (error "Parenthesis error in ~s constraint.  ~&Expected ~
		    (constraint-name cval cval...); found constraint-name." ocon))
		 (t (error "Don't recognize constraint form ~s ~&  for ~s.  Expected ~
		    (constraint-name cval cval...);"
			   ocon qde))))
  ;; Save constraints and initial cvals in qde.
  (setf (qde-constraints qde) (reverse (qde-constraints qde))
	(qde-cvalues     qde) (reverse (qde-cvalues qde))))


;;; Modified 9Sept91 by BKay to add full boolean expressions for testing modes.
;;; Previously, only disjunctions were allowed.
;;;
(defun MODE-SPEC-CONSTRAINT (ocon qde)
  (let* ((modes (cadr ocon))
	 (mode-expr
	   ;; Walk along the expression replacing var names and qmag names with
	   ;; the actual vars and qmags.  This will save us time in cfilter.
	   (transform-mode-designators modes qde)))
    (dolist (mcon (cddr ocon))
      ;; moved so that buildfcn is set for each constraint.
      (let ((buildfcn (contype-buildfcn (contype-from-name (caar mcon))))) 
	(funcall buildfcn  mcon qde mode-expr)))))

;;; OLD DEFINITION
;;;(defun MODE-SPEC-CONSTRAINT (ocon qde)
;;;  (let* ((modes (cadr ocon))
;;;	 (mode-specs
;;;	   (mapcar #'(lambda (mode)
;;;		       (let* ((varname (first mode))
;;;			      (var     (alookup varname (qde-var-alist qde)))
;;;			      (qspace  (alookup varname (qde-qspaces qde)))
;;;			      (qmag    (convert-user-qmag (second mode) qspace)))
;;;			 (cons var qmag)))
;;;		   modes))
;;;	 (buildfcn (contype-buildfcn (contype-from-name (caar (third ocon))))))
;;;    (dolist (mcon (cddr ocon))
;;;      (funcall buildfcn  mcon qde mode-specs))))


;;; Go thru the mode argument and turn each mode-designator into a (var qmag) 
;;; pair.  Also, convert old mode syntax to new mode syntax.
;;;
(defun transform-mode-designators (modes qde)
  (declare (special syntax-warnings))
  (when (listp (car modes))
    ;; Old syntax was (mode-des1 mode-des2 ...)
    ;; New syntax is  (OR mode-des1 mode-des2 ...)
    (setq syntax-warnings t)
    (wrn "Old mode expression~%     ~a~%         This will be interpreted as~%     ~a"
         modes (cons 'OR modes))
    (push 'OR modes)
    (setq syntax-warnings nil))
  (replace-mode-designators modes qde))

(defun replace-mode-designators (mode-expr qde)
  (cond
   ((null mode-expr) nil)
   ((mode-designator-name mode-expr)
    (list (alookup (first mode-expr) (qde-var-alist qde)) ;var
	  (convert-user-qmag (second mode-expr) (alookup (first mode-expr) (qde-qspaces qde)))))
   ((eq (car mode-expr) 'NOT)
    (list 'NOT (replace-mode-designators (second mode-expr) qde)))
   ((member (car mode-expr) '(OR AND))
    (cons (car mode-expr) (mapcar #'(lambda (arg) (replace-mode-designators arg qde))
				  (cdr mode-expr))))
   (t
    (error "Unknown mode expression ~a" mode-expr))))


(defun mode-designator-name (mode-expr)
  (and (listp mode-expr) (atom (first mode-expr)) (atom (second mode-expr))))


;;; BUILD-CONSTRAINT (or any other contype-buildfcn) is responsible
;;; (along with any functions called by it) for
;;;    creating the constraint,
;;;    adding the constaint on the qde-constraints slot,
;;;    marking the constraint as active
;;;    putting mode specs on the constraint,
;;;    marking the bend points on the constraint (if applicable)
;;;    checking the syntax,
;;;    building the corresponding values and putting them in the qde-cvalues slot.
;;;    adding the constraint to the variable-constraints slots,

(defun BUILD-CONSTRAINT (ocon qde mode-specs)
  (let* ((oname (if (atom (caar ocon))		; a symbol
		    (caar ocon) (caaar ocon)))
	 (ovariables (cdar ocon))		; list of symbols
	 ;; ocvals is alist of cval lists, where cval list is a list of symbols of landmarks
	 (ocvals     (cdr ocon))
	 (contype    (contype-from-name oname))
	 (cvariables				; cvariables := (#<var> #<var>...)
	   (cvariables-for-ocon ocon contype qde))
	 con)
    ;; Build an instance of the constraint structure.
    (setq con (make-constraint :name       (car ocon)
			       :type       contype
			       :variables  cvariables))
    (push con (qde-constraints qde))
    (setf (constraint-active-p con) t)
    (when mode-specs
      (push con (qde-constraints-within-modes qde))
      (update-mode mode-specs con))

    ;; If bend-points present, convert them to <lmark>s.
    (when (contype-bend-points-p contype)
      (mark-bend-point-constraints
	con contype qde cvariables ovariables))
    ;; If  con = ((M s1 ... sn) v1 ... vn vn1), set partials = (s1 ... sn)   ; BJK: 10-26-90
    (when (eq oname 'M)
	(setf (constraint-partials con) (append (cdr (caar ocon)) (list '-))))
    ;; If con = ((CONSTANT v val)), record value (on bend-points list)
    (when (and (eq oname 'CONSTANT) (> (length ovariables) 1))
      (push (second ovariables) (constraint-bend-points con)))
    ;; Build list of corresponding values.
    (when (contype-cvals-allowed-p contype)
      (push (build-cvalues-list con contype ocvals  cvariables)
	    (qde-cvalues qde)))
    (notify-vars-of-constraint con cvariables qde)	; 3rd arg here is is the only difference
    (check-for-malformed-constraint con contype ocvals oname cvariables)))


(defun CHECK-FOR-MALFORMED-CONSTRAINT (con contype ocvals oname cvariables)
  (when (and  ocvals (not (contype-cvals-allowed-p contype)))
    (format *qsim-report* "~%Warning: ignoring cvals ~a for ~a"
	    ocvals (constraint-name con)))

  ;; Every variable of a MULT must have 0 in its qspace.
  (when (eql 'MULT oname)
    (dolist (var cvariables)
      (unless (member *zero-lmark* (variable--qspace var))
	(error "~a requires ~a to have 0 in its qspace."
	       (constraint-name con) (variable-name var)))))

  ;; The rate variable of a D/DT must have 0 in its qspace.
  (when (eql 'D/DT oname)
    (unless (member *zero-lmark* (variable--qspace (second cvariables)))
      (error "~a requires ~a to have 0 in its qspace."
	     (constraint-name con) (variable-name (second cvariables))))))
 

(defun NOTIFY-VARS-OF-CONSTRAINT (con cvariables qde)
  ;; For each variable of this constraint, add the constraint to
  ;; the variable's list of attached constraints.  Pushnew is used
  ;; in case same variable appears twice in same constraint,
  ;; as in (MULT V V V-SQUARED).
 
  (loop for variable in  cvariables
	for place = (assoc (qde-name qde)
			   (variable-constraints variable))
	unless place
	  do (setf place
		   (car (pushnew (list (qde-name qde))
				 (variable-constraints variable))))
	do (pushnew con (cdr place))))
     

;;; Given ocon, the form of the constraint in DEFINE-QDE, and the
;;; contype of the constraint, build up the corresponding list of
;;; (#<var> #<var>...) corresponding to its arguments.
 
(defun CVARIABLES-FOR-OCON (ocon contype qde)
  (let ((cvariables nil)
	(oname      (caar ocon))
	(ovariables (cdar ocon)))
    (do ((ovars ovariables (cdr ovars))
	 (n     0          (1+ n)))
	((or (endp ovars) (>= n (contype-nargs contype))))
      (let ((varname (car ovars)))
	(if (symbolp varname)
	    (let ((var (alookup varname (qde-var-alist qde))))
	      (if (null var)
		  (error "Variable ~a in ~a has no quantity space."
			 varname ocon))
	      (push var cvariables))
	    (unless (eql '= oname)
	      (error "Bad constraint syntax: ~a" (car ocon))))))
    (nreverse cvariables)))



;;; This function redefined by BKay 9Sept91 to allow full boolean expressions for modes.
;;; We now store a mode expression on the individual constraint slot constraint.mode-expressions.
;;; Note that a constraint may have more than one expression.
;;;
(defother constraint mode-expressions)

(defun UPDATE-MODE (mode-expr constraint)
  (push mode-expr (constraint-mode-expressions constraint)))

;;; OLD DEFINITION -- stores the constraint mode on each affected variable.
;;;
;(defun UPDATE-MODE (mode-specs constraint)
;  (mapc #'(lambda (mode-spec)
;	    (let* ((mode-var  (car mode-spec))
;		   (mode-qmag (cdr mode-spec))
;		   (pair (assoc mode-qmag (variable-modes mode-var))))
;	      (if pair
;		  (push constraint (cdr pair))
;		  (push (cons mode-qmag (list constraint)) (variable-modes mode-var)))))
;	mode-specs))


;;;  MARK-BEND-POINT-CONSTRAINTS is called only on constraints that
;;;  have bend-points, such as S+ and S-.  It simply converts the
;;;  names of the bend-point values to the appropriate LMARKs by
;;;  looking up each name in its quantity space.

(defun MARK-BEND-POINT-CONSTRAINTS (con contype qde cvariables ovariables)
  (let* ((bptemp (nthcdr (contype-nargs contype) ovariables))
	 (bplist (copy-tree bptemp)))
    (dolist (pair bplist)
      (do ((blist pair (cdr blist))
	   (cvars cvariables (cdr cvars)))
	  ((endp blist))
	(let* ((lm-name (car blist))
	       (var     (car cvars))
	       (lmark (or (find lm-name (alookup (variable-name var)
						 (qde-qspaces qde))
				:key 'lmark-name)
			  (error "Couldn't find bendpoint ~a" lm-name))))
	  (rplaca blist lmark))))
    (setf (constraint-bend-points con) bplist)))


;;; Build list of corresponding values. build-cvalues-list is called for value, not
;;; side effect.  The value this function returns := (constr cvals cvals cvals...)
;;; where cval := (<lmark> <lmark>...)

(defun BUILD-CVALUES-LIST (con contype ocvals cvariables)
  (let* ((user-cvals (augmented-cvals contype ocvals cvariables))
	 (ncvals (convert-cvals user-cvals cvariables)))
    ;; Bend-points are also corresponding values.
    (if (constraint-bend-points con)
	(setq ncvals (append (constraint-bend-points con) ncvals)))
    (cons con ncvals)))


(defun FILL-NEIGHBORING-CONSTRAINTS (qde)
  ;; For each constraint, fill in its list of neighboring constraints.
  (dolist (con (qde-constraints qde))
    (dolist (variable (constraint-variables con))
      (dolist (con2 (alookup (qde-name qde)
			     (variable-constraints variable)))
	(if (not (eq con con2))
	    (pushnew con2 (constraint-neighbors con)))))))



;;;-----------------------------------------------------------------------------
;;;-----------------------------------------------------------------------------

(defun CONVERT-TRANSITIONS-LIST (qde tran-clause)
  (setf (qde-transitions qde)
	(mapcar
	  #'(lambda (otran)
	      (let* ((condition (first otran))
		     (arrowp (eq (second otran) '->))
		     (trans-fn  (if arrowp (third otran)(second otran))))

		(cond
		  ;; If it is a function spec, then we're done.
		  ((and (listp condition)
			(or (assoc (car condition) (qde-var-alist qde))
			    (member (car condition) '(and or not))))
		   (list #'(lambda (state)
			     (values
			       (basic-transition-test condition state)
			       nil))
			 trans-fn))
		  ;; otherwise it had better be something acceptable to
		  ;; basic-transition-state.
		  ((or (symbolp condition)
		       (functionp condition))
		   (list condition trans-fn))
		  ;; Anything else is an error.
		  (t (warn "Bad syntax in transition test: ~a" condition)))))
	  tran-clause)))


;;;-----------------------------------------------------------------------------
;;;  Function:  (BASIC-TRANSITION-TEST  condition  state)
;;;
;;;  Purpose:   This function tests a user-specified transition specification
;;;             against the current state values to determine if the associated
;;;             transition function should be called.
;;;
;;;  Syntax:    The user-supplied `condition' (transition specification) can be
;;;             a test of a single value or an arbitrary nesting of AND, OR, and
;;;             NOT on values tests.  For examples:
;;;             1.  '(Temp (Hi inc))
;;;             2.  '(OR (Temp (Hi inc)) (Pressure (Max inc)))
;;;             3.  '(AND (Temp (Hi inc)) (NOT (Relief-valve (open std))))
;;;
;;;  Note:      This code currently assumes that qmags in a `condition' will be
;;;             landmark values, never an interval between landmarks.  This can
;;;             be changed easily, if needed.
;;;-----------------------------------------------------------------------------

(defun BASIC-TRANSITION-TEST (condition state)
  (cond ((eql 'OR (car condition))
	 (some #'(lambda (test)
		   (basic-transition-test test state))
	       (cdr condition)))

	((eql 'AND (car condition))
	 (every #'(lambda (test)
		    (basic-transition-test test state))
		(cdr condition)))

	((eql 'NOT (car condition))
	 (not (basic-transition-test (cadr condition) state)))

	((symbolp (car condition))
	 (let* ((varname (car condition))
		(value   (cadr condition))
		(qval    (qval varname state)))
	   (and (or (eql (cadr value) (qdir qval))
		    (null (cadr value)))
		(or (equal (car value) (if (qmag-point-p (qmag qval))
					   (lmark-name (qmag qval))
					   ; added DJC 01/09/92 to provide for intervals in
					   ; a transition condition
					   (list (lmark-name (car (qmag qval)))  
						 (lmark-name (cadr (qmag qval))))))
		    (null (car value))))))
	
	(t (error "Invalid transition condition: ~a" condition))))




    
;;;-----------------------------------------------------------------------------
;;;  Function:  (AUGMENTED-CVALS  cname  cvals  cvariables)
;;;
;;;  Purpose:   Some constraints have implied corresponding value tuples, 
;;;             whether or not the user has explicitly specified them.  This
;;;             function adds any of these missing-but-implied cval tuples,
;;;             consistent with the quantity spaces of the constraint's variables.
;;;             For example:
;;;                ADD       gets  (0 0 0)
;;;                MINUS     gets  (minf inf) (0 0) (inf minf)
;;;                SUM-ZERO  gets  (0 0 ... 0)
;;;-----------------------------------------------------------------------------

(defun AUGMENTED-CVALS (contype cvals cvariables)
  (let ((icvals (case (contype-name contype)
		  (sum-zero (list (make-list (length cvariables) :initial-element '0)))
		  (t (contype-implied-cvals contype)))))
    (dolist (cv-tuple icvals cvals)
      ;; Every landmark in the implied cv-tuple will appear in the appropriate
      ;;  quantity space except when the implied-cvals go to inf (as with MINUS)
      ;;  and the qspace doesn't.
      (when (every #'(lambda (userval var)
		       (member userval (variable--qspace var)
			       :key #'lmark-name))
	       cv-tuple cvariables)
	  (pushnew cv-tuple cvals :test #'equal)))))


;;; Convert cvals from form like '(A* B*) to '(#<LMARK A*> #<LMARK B*>)

(defun CONVERT-CVALS (cv-tuple-list cvariables)
  (mapcar #'(lambda (cv-tuple)
	      (mapcar #'(lambda (userval var)
			  (cond ((find userval (variable--qspace var) :key #'lmark-name))
				(t (error "Corresponding value ~a missing in qspace of ~a."
					  userval (variable-name var)))))
		      cv-tuple cvariables))
	  cv-tuple-list))


;;; Convert a list of atomic qmags to a list of landmarks.

(defun CONVERT-TO-LMARKS (qmaglist var)
  (let ((qspace (variable--qspace var)))
    (mapcar #'(lambda (qmag)
		(find qmag qspace :key #'lmark-name))
	    qmaglist)))


(defun CONVERT-TO-LMARK (qmag var)
  (find qmag (variable--qspace var) :key #'lmark-name))



