;;; -*- Mode:Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: constraints.lisp,v 1.10 92/05/27 14:23:14 kuipers Exp $
;;; Copyright (c) 1986 by Benjamin Kuipers.


(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;                   Q S I M    C O N S T R A I N T S
;;;
;;;  This file contains all the functions and tables needed for checking and
;;;  propagating constraints.  The file is organized in three sections:
;;;  1.  Basic functions.
;;;  2.  Constraint-specific subsections containing the tables, structures,
;;;      and functions for each constraint.  (This should make it easier to
;;;      see what to add when creating a new constraint type).
;;;  3.  P-successor and I-successor functions (previously called P-transitions
;;;      and I-transitions).
;;;-----------------------------------------------------------------------------


; QSIM propagation and prediction are both done by calling cfilter.
;  - Propagation takes any given values, and all possible other values,
;     and determining all possible states consistent with them.
;  - Prediction takes a given set of possible qualitative states, 
;     derived from the qualitative transitions, and determines all
;     qualitative states consistent with them.

; Customization of CFILTER takes place with the specialized function:
;  - CHECK-QSIM-CONSTRAINT checks a proposed qualitative state against
;     the semantics of the constraint, and any corresponding values.



;;;-----------------------------------------------------------------------------
;;; CHECK-QSIM-CONSTRAINT checks a proposed tuple of values against the
;;; semantics of a constraint, and any corresponding values.
;;;  This code is justified by Appendix B in [Kuipers, 1986].
;;;-----------------------------------------------------------------------------

(defun check-qsim-constraint (tuple constraint)
  (let ((checker (constraint-checkfcn constraint)))
    (if checker
	(funcall checker tuple constraint)
	(progn
	  (format *qsim-report* "~%NO CONSTRAINT CHECK FUNCTION FOR ~a" constraint)
	  nil))))

;;;-----------------------------------------------------------------------------
;;;  ALL-QVALUES takes a variable and its quantity space, and returns
;;;  all possible qualitative values consistent with a given pattern.
;;;  ALL-QVALUES enumerates all the qualitative values of the perhaps
;;;  partially-specified qualitative value.  A qval may be partially
;;;  specified if its qdir is nil or if its qmag is nil or if its qmag
;;;  is an interval of lower and upper bounds where one of the bounds is
;;;  nil or the bounds are not adjacent landmarks.
;;;-----------------------------------------------------------------------------

(defun all-qvalues (qval)
  (let* ((var     (qval-variable qval))
	 (bounds  (qmag qval))
	 (qdir    (qdir qval))
	 (qspace  (variable--qspace var)))

    ;; If the given qval is fully-specified (which is often the case),
    ;; or if only the qdir is unspecified, then we can quickly create
    ;; the list of qvals to return, and thereby avoid the overhead in
    ;; the remainder of this function.  Thus, we make a special test
    ;; here for a quick return.
    (if bounds
	(if (or (qmag-point-p bounds) (adjacent-p bounds qspace))
	    (if qdir
		(return-from all-qvalues (list qval))
		(return-from all-qvalues (list
					   (make-qval :variable var :qmag bounds :qdir 'inc)
					   (make-qval :variable var :qmag bounds :qdir 'std)
					   (make-qval :variable var :qmag bounds :qdir 'dec))))))

    ;; The qmag is not fully-specified, so we must do the general algorithm.
    (let ((qdirs  (if qdir (list qdir) '(dec std inc)))
	  (*all-qvals* nil))
      (declare (special *all-qvals*))

      (cond ((null bounds)          (gen-qvals nil nil qspace qdirs var))
	    ((qmag-point-p bounds)  (gen-qvals bounds bounds qspace qdirs var))  ; not needed
	    (t                      (gen-qvals (first bounds) (second bounds) qspace qdirs var)))
      *all-qvals*)))



(defun gen-qvals (lb ub qspace qdirs var)
  (let (lmark1)
    (cond ((null qspace)
	   ;; No remaining landmarks, so we're all done.
	   nil)
	  ((eql ub (setq lmark1 (first qspace)))
	   ;; We've reached the upper bound, so we're all done.
	   nil)
	  ((null (cdr qspace))
	   ;; We've reached the last landmark.
	   (if (null ub)
	       (create-qvals lmark1 qdirs var)
	       (error "Upper bound ~a not in qspace ~a." ub qspace)))

	  ((null lb)
	   ;; Lower-bound = nil, so create create qvals for point and interval.
	   (let ((lmark2 (second qspace)))
	     (create-qvals lmark1 qdirs var)
	     (create-qvals (list lmark1 lmark2) qdirs var)
	     (gen-qvals lb ub (cdr qspace) qdirs var)))

	  ((eql lb lmark1)
	   ;; Lower-bound = first landmark in remaining landmarks.
	   (let ((lmark2 (second qspace)))
	     (create-qvals (list lmark1 lmark2) qdirs var)
	     (gen-qvals nil ub (cdr qspace) qdirs var)))

	  (t
	   ;; Still looking for the lower-bound in qspace.
	   (gen-qvals lb ub (cdr qspace) qdirs var)))))


(defun create-qvals (qmag qdirs var)
  (declare (special *all-qvals*))
  (dolist (qdir qdirs)
    (push (make-qval :variable var
		     :qmag qmag
		     :qdir qdir)
	  *all-qvals*)))


;;;-----------------------------------------------------------------------------
;;;  Miscellaneous comparison functions.
;;;-----------------------------------------------------------------------------

(defun same-qdirs (x y)
  (or (eql x y)					; Allows (inc inc), (std std),
      (eql x 'ign)				; (dec dec), (ign *), (* ign).
      (eql y 'ign)))

(defun opposite-qdirs (x y)
  (or (and (eql x 'inc) (eql y 'dec))
      (and (eql x 'std) (eql y 'std))
      (and (eql x 'dec) (eql y 'inc))
      (eql x 'ign)
      (eql y 'ign)))

(defun same-order (x y)
  (or (not x)
      (not y)
      (and (eql x '+) (eql y '+))  
      (and (eql x '0) (eql y '0))
      (and (eql x '-) (eql y '-))))

(defun opposite-order (x y)
  (or (not x)
      (not y)
      (and (eql x '+) (eql y '-))
      (and (eql x '0) (eql y '0))
      (and (eql x '-) (eql y '+))))

(defmacro bad-sign (sign)
  `(error "~% ~a is not a valid sign !" ,sign))

(defun opp-sign (sign)
  "Returns the opposite sign of sign"
  (cond ((not sign) nil)
	((eq sign '-) '+)
	((eql sign 0) 0)
	((eq sign '+) '-)
	(t (bad-sign sign))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (qmag-order  qmag  landmark  qspace)
;;;
;;;  Purpose:   Given a qmag and a landmark, this function returns +/0/-
;;;             according to whether the qmag is greater, equal, or less than
;;;             the landmark, and NIL if it is unknown.  
;;;             Note:  a qmag of (nil L) means [minf L).
;;;
;;;  Note:      This is a *heavily used* function, so it is important that it
;;;             be as efficient as possible for the most common cases.
;;;
;;;		Modified 03/12/90 by Pierre Fouche so that it can compare two
;;;		arbitrary qmags. Qmag-order-1 compares a qmag with a landmark.
;;;-----------------------------------------------------------------------------

(defun qmag-order-1 (qmag landmark qspace)
  (declare (optimize speed))
  (cond
    ;; qmag is NIL, so can't determine order.
    ((null qmag) NIL)

    ;; qmag is an atom/landmark, so compare it against "landmark".
    ((qmag-point-p qmag)
     (cond ((eq qmag landmark)                            0)
	   ((member qmag (cdr (member landmark qspace))) '+)
	   
	   ((not (member landmark qspace)) nil)
	   ((member qmag qspace)                         '-)
	   (t nil)))

    ;; qmag is an interval, so check lower and upper bounds against landmark.
    ((member landmark (member (cadr qmag) qspace)) '-)
    ((member (car qmag) (member landmark  qspace)) '+)

    ;; All that remains is to recognize the three cases where the order
    ;; cannot be determined.  Anything else is an error situation.

    ;; Case 1:  landmark is between lower and upper bounds of qmag.
    ((member (cadr qmag)
	     (cdr (member landmark
			  (cdr (member (car qmag) qspace)))))   NIL)

    ;; Case 2:  qmag is (nil X) and landmark < X.
    ((and (null (car qmag))
          (member (cadr qmag) (cdr (member landmark qspace))))  NIL)

    ;; Case 3:  qmag is (X nil) and X < landmark.
    ((and (null (cadr qmag))
          (member landmark (cdr (member (car qmag) qspace))))   NIL)

    ;; Unknown ordering.
    (t nil)))


(defun qmag-order (qmag1 qmag2 qspace)
  (declare (optimize speed))
  (cond
    ;; qmag2 is a landmark -> call directly qmag-order-1
    ((qmag-point-p qmag2) (qmag-order-1 qmag1 qmag2 qspace))

    ;; qmag1 is a landmark -> call qmag-order-1 and return the opposite-sign 
    ((qmag-point-p qmag1) (opp-sign (qmag-order-1 qmag2 qmag1 qspace)))

    ;; both qmags are OPEN interval. The only cases where the order can be 
    ;; determined are:
    
    ;; qmag1   ................|........|................
    
    ;; qmag2   ......|.....|.............................
    ;; or      ......|.........|.........................
    ;; or      .........................|.......|........
    ;; or      ............................|....|........

    ((eq '+ (qmag-order-1 qmag1 (second qmag2) qspace)) '+)
    ((eq '- (qmag-order-1 qmag1 (first  qmag2) qspace)) '-)
    (t nil)))



;;;                   Q S I M    C O N S T R A I N T S
;;;
;;;
;;;  So you want to add a new type of constraint.  What do you do?
;;;  There are only 3 things to do:
;;;
;;;  1.  Create a declare-qsim-constraint with an appropriately filled-in
;;;      contype structure (the ADD constraint is a good example to examine).
;;;      The slots of contype that you absolutely must supply a non-NIL value
;;;      for are name, nargs, and checkfcn.  Note that the contype slots
;;;      qmag-relation and qdir-relation do not require values unless your
;;;      propagator or check function accesses those slots. The reason that
;;;      these slots exist is because some constraints (such as M+, M-, MINUS,
;;;      and ADD) share a common propagator, and these slots supply all the
;;;      constraint-specific data that is needed.
;;;
;;;  2.  Write a check function.  This function will be called with two 
;;;      arguments (tuple and constraint), and must return non-NIL if the
;;;      given tuple satisfies the constraint (see check-M+-constraint for a
;;;      simple example).
;;;
;;;  3.  Write a propagator function, if appropriate.  (Constraints of just one
;;;      argument need propagtors, too, see the propagtor function UNARY-PROPAGATOR) 
;;;      The propagator is a function of 3 arguments (constraint, corresponding
;;;      values, and state) and its job is to further specify the qmag and qdir
;;;      of any partially specified variables of the constraint, when possible.
;;;      A propagator is not required to return anything; it is called solely
;;;      for its side-effect on the qvals of variables in the QDE.  The state
;;;      argument is used only in error messages; it serves no purpose during
;;;      propagation.  See basic-propagation as an example.
;;;
;;;  Helpful hint: if your new constraint is similar to some existing constraint,
;;;  then copy or reuse as much code as you can.
;;;-----------------------------------------------------------------------------

;;; *Known-Constraint-Types*'s definition should be kept here, rather than in
;;; file Qdefs, so that it is always loaded and reloaded along with the contype
;;; definitions.  -drt
;;; Use defparameter, not defvar, so variable is set to nil on each reloading.  -RSM

(defparameter *KNOWN-CONSTRAINT-TYPES* nil)


(defun DECLARE-QSIM-CONSTRAINT (contype)
  (setq *known-constraint-types*
	(cons contype				; Add new definition.
	      (delete (contype-name contype)	; Delete old definition, if any,
		      *known-constraint-types*	; so contype may be redefined.
		      :key #'contype-name))))	; RSM 12-27-91


;;;-----------------------------------------------------------------------------
;;;  M+ Constraint
;;;-----------------------------------------------------------------------------

(defparameter M+-relation '((+ +)
			    (0 0)
			    (- -)))

(defparameter M+-qdir-relation '((inc inc)
				 (std std)
				 (dec dec)))

(declare-qsim-constraint
  (make-contype :name           'M+
		:nargs          2
		:propagator     'basic-propagation
		:checkfcn       'check-M+-constraint
		:qmag-relation  M+-relation
		:qdir-relation  M+-qdir-relation
		:cvals-allowed-p t))


(defun check-M+-constraint (tuple con)
  (declare (optimize speed))
  (and (same-qdirs (qdir (first  tuple))
		   (qdir (second tuple)))
       (check-M+-cvals tuple con)))

(defun check-M+-cvals (tuple con)
  (declare (optimize speed))
  (let* ((vars    (constraint-variables con))
	 (Xqspace (variable--qspace (first  vars)))
	 (Yqspace (variable--qspace (second vars)))
	 (Xqmag   (qmag (first  tuple)))
	 (Yqmag   (qmag (second tuple))))
    ;; Ensure that tuple's relation to every cv-tuple is OK.
    (dolist (cv (constraint--cvals con) T)
      (if (not (same-order (qmag-order Xqmag (first  cv) Xqspace)
			   (qmag-order Yqmag (second cv) Yqspace)))
	  (return nil)))))


;;;-----------------------------------------------------------------------------
;;;  M- Constraint
;;;-----------------------------------------------------------------------------

(defparameter M--relation '((+ -)
			    (0 0)
			    (- +)))

(defparameter M--qdir-relation '((inc dec)
				 (std std)
				 (dec inc)))


(declare-qsim-constraint
  (make-contype :name           'M-
		:nargs          2
		:propagator     'basic-propagation
		:checkfcn       'check-M--constraint
		:qmag-relation  M--relation
		:qdir-relation  M--qdir-relation
		:cvals-allowed-p t))


(defun check-M--constraint (tuple con)
  (declare (optimize speed))
  (and (opposite-qdirs (qdir (first tuple))
		       (qdir (second tuple)))
       (check-M--cvals tuple con)))

(defun check-M--cvals (tuple con)
  (declare (optimize speed))
  (let* ((vars    (constraint-variables con))
	 (Xqspace (variable--qspace (first  vars)))
	 (Yqspace (variable--qspace (second vars)))
	 (Xqmag   (qmag (first  tuple)))
	 (Yqmag   (qmag (second tuple))))
    ;; Ensure that tuple's relation to every cv-tuple is OK.
    (dolist (cv (constraint--cvals con) T)
      (if (not (opposite-order (qmag-order Xqmag (first  cv) Xqspace)
			       (qmag-order Yqmag (second cv) Yqspace)))
	  (return nil)))))


;;;-----------------------------------------------------------------------------
;;;  MINUS Constraint
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'MINUS
		:nargs           2
		:propagator      'basic-propagation
		:checkfcn        'check-MINUS-constraint
		:qmag-relation   M--relation
		:qdir-relation   M--qdir-relation
		:cvals-allowed-p t
		:implied-cvals   '((0 0))))

(defun check-MINUS-constraint (tuple con)	; same as M-
  (and (check-M--constraint tuple con)
       (check-MINUS-infinities tuple)))

(defun check-MINUS-infinities (tuple)
  (let ((x (qmag (car  tuple)))
	(y (qmag (cadr tuple))))
    (and (eq (eql x *inf-lmark*)  (eql y *minf-lmark*))
	 (eq (eql x *minf-lmark*) (eql y *inf-lmark*)))))

;;;-----------------------------------------------------------------------------
;;;  EQUAL Constraint:  (EQUAL X Y)  <=>  X(t)=Y(t) forall t.
;;;-----------------------------------------------------------------------------

; We need the EQUAL constraint, in order to identify distinct terminal variables of
; a component.  It behaves like M+, but will be treated specially by Q2.

(declare-qsim-constraint
  (make-contype :name            'EQUAL
		:nargs           2
		:propagator      'basic-propagation
		:checkfcn        'check-SIMPLE-EQUAL-constraint
		:qmag-relation   M+-relation
		:qdir-relation   M+-qdir-relation
		:cvals-allowed-p t
		:implied-cvals   '((0 0))))

(defun check-SIMPLE-EQUAL-constraint (tuple con)	; same as M+
  (and (check-M+-constraint tuple con)
       (check-EQUAL-infinities tuple)))

(defun check-EQUAL-infinities (tuple)
  (let ((x (qmag (car  tuple)))
	(y (qmag (cadr tuple))))
    (and (eq (eql x *inf-lmark*)  (eql y *inf-lmark*))
	 (eq (eql x *minf-lmark*) (eql y *minf-lmark*)))))


;;;-----------------------------------------------------------------------------
;;;  ABS-VALUE Constraint
;;;-----------------------------------------------------------------------------

;;; (sign xqmag xqspace) returns the sign of xqmag: +, 0 or -.

(defmacro sign (qmag qspace)
  `(qmag-order ,qmag *zero-lmark* ,qspace))


;;; (abs x y) means that y is the absolute value of x

(declare-qsim-constraint
  (make-contype :name           'abs-value
		:nargs          2
		:checkfcn       'check-abs-value-constraint
		:cvals-allowed-p t
		:implied-cvals '((0 0) (inf inf) (minf inf))))

(defun check-abs-value-constraint (tuple con)
  (declare (optimize speed))
  (let* ((vars    (constraint-variables con))
	 (Xqspace (variable--qspace (first  vars)))
	 (Yqspace (variable--qspace (second vars)))
	 (Xqmag   (qmag (first  tuple)))
	 (Yqmag   (qmag (second tuple))))
    (cond ((eql (sign xqmag xqspace) 0)
	   ;; x' absolute value must be 0 and its qdir std
	   (and (eql (sign yqmag yqspace) 0)
		(eql (qdir (second tuple)) 'std)))
	  
	  ((eql (sign xqmag xqspace) '+)
	   ;; x and its absolute value have the same qdir 
	   (and (same-qdirs (qdir (first  tuple))
			    (qdir (second tuple)))
		(check-abs-pos-cvals xqmag xqspace yqmag yqspace con)))
	  
	  ((eql (sign xqmag xqspace) '-)
	   ;; x and its absolute value have an opposite qdir
	   (and (opposite-qdirs (qdir (first  tuple))
				(qdir (second tuple)))
		(check-abs-neg-cvals xqmag xqspace yqmag yqspace con))))))


;;; When checking a tuple against cvalues, xqmag and the first element of the
;;; cv tuple must have the sign.
	  
(defun check-abs-pos-cvals (xqmag xqspace yqmag yqspace con)
  (dolist (cv (constraint--cvals con) t)
    (if (and (not (eq (sign (first cv) xqspace) '-))
	     (not (same-order (qmag-order Xqmag (first  cv) Xqspace)
			      (qmag-order Yqmag (second cv) Yqspace))))
	(return nil))))

(defun check-abs-neg-cvals (xqmag xqspace yqmag yqspace con)
  (dolist (cv (constraint--cvals con) t)
    (if (and (not (eq (sign (first cv) xqspace) '+))
	     (not (opposite-order (qmag-order Xqmag (first  cv) Xqspace)
				  (qmag-order Yqmag (second cv) Yqspace))))
	(return nil))))





;;;-----------------------------------------------------------------------------
;;;  ADD Constraint
;;;-----------------------------------------------------------------------------

(defparameter ADD-relation
	     '((+ + +) (0 + +) (- + +)
	       (+ 0 +) (0 0 0) (- + 0)
	       (+ - +) (0 - -) (- + -)
	       (+ - 0)	       (- 0 -)
	       (+ - -)	       (- - -)))

(defparameter ADD-qdir-relation
	     '((inc inc inc) (std inc inc) (dec inc inc)
	       (inc std inc) (std std std) (dec inc std)
	       (inc dec inc) (std dec dec) (dec inc dec)
	       (inc dec std)               (dec std dec)
	       (inc dec dec)	           (dec dec dec)))

(declare-qsim-constraint
  (make-contype :name              'ADD
		:nargs             3
		:propagator        'basic-propagation
		:checkfcn          'check-ADD-constraint
		:qmag-relation     ADD-relation
		:qdir-relation     ADD-qdir-relation
		:cvals-allowed-p   t
		:implied-cvals     '((0 0 0))
		:disallowed-lmarks (list *minf-lmark* *inf-lmark*)))

(defun check-ADD-constraint (tuple constraint)
  (declare (optimize speed))
  (and (check-ADD-qdirs tuple)
       (check-ADD-cvals tuple constraint)
       (check-ADD-infinite-values tuple)))

; Testing qdirs

;;;  This is the older and easier-to-understand function, but it's
;;;  noticeably slower than the new one, below.
;(defun check-ADD-qdirs (tuple)
;  (let ((qdirs (mapcar #'qval-qdir tuple)))
;    (or (member 'ign qdirs)
;	(member qdirs ADD-qdir-relation :test #'equal))))

(defun check-ADD-qdirs (tuple)
  (declare (optimize speed))
  (case (qdir (first tuple))
    (ign  t)
    (std  (let ((Yqdir (qdir (second tuple)))
		(Zqdir (qdir (third  tuple))))
	    (or (eql Yqdir Zqdir)
		(eql Yqdir 'ign)
		(eql Zqdir 'ign))))
    (inc  (case (qdir (second tuple))
	    ((dec ign)  t)
	    (t (let ((Zqdir (qdir (third tuple))))
		 (or (eql Zqdir 'inc)
		     (eql Zqdir 'ign))))))
    (dec  (case (qdir (second tuple))
	    ((inc ign)  t)
	    (t (let ((Zqdir (qdir (third tuple))))
		 (or (eql Zqdir 'dec)
		     (eql Zqdir 'ign))))))))

;;;  This is the older and easier-to-understand function, but it's
;;;  noticeably slower than the new one, below.
;(defun check-ADD-cvals (tuple con)		; (ADD x y z)
;  (let* ((vars     (constraint-variables con))
;	 (Xqspace  (variable--qspace (first  vars)))
;	 (Yqspace  (variable--qspace (second vars)))
;	 (Zqspace  (variable--qspace (third  vars)))
;	 (Xqmag    (qmag (first  tuple)))
;	 (Yqmag    (qmag (second tuple)))
;	 (Zqmag    (qmag (third  tuple))))
;
;    (every #'(lambda (cv)
;	       (member (list (qmag-order Xqmag (first  cv) Xqspace)
;			     (qmag-order Yqmag (second cv) Yqspace)
;			     (qmag-order Zqmag (third  cv) Zqspace))
;		       ADD-relation
;		       :test #'equal))
;	   (constraint--cvals con))))

(defun check-ADD-cvals (tuple con)		; (ADD x y z)
  (declare (optimize speed))
  (let* ((vars     (constraint-variables con))
	 (Xqspace  (variable--qspace (first  vars)))
	 (Yqspace  (variable--qspace (second vars)))
	 (Zqspace  (variable--qspace (third  vars)))
	 (Xqmag    (qmag (first  tuple)))
	 (Yqmag    (qmag (second tuple)))
	 (Zqmag    (qmag (third  tuple))))
    
    (dolist (cv (constraint--cvals con) T)
      (let ((qmag-orderX (qmag-order Xqmag (first cv) Xqspace))
	    (qmag-orderY (qmag-order Yqmag (second cv) Yqspace))
	    (qmag-orderZ (qmag-order Zqmag (third cv) Zqspace)))
	(when (and qmag-orderX qmag-orderY qmag-orderZ)
	  (if (not (case qmag-orderX
		     (0  (eql qmag-orderY
			      qmag-orderZ))
		     (+  (case qmag-orderY
			   (-  t)
			   (t  (eql '+ qmag-orderZ))))
		     (-  (case qmag-orderY
			   (+  t)
			   (t  (eql '- qmag-orderZ))))))
	      (return nil))
	  )))))


; Check for infinite values, and trace their deletion.


;;;  This is the older and easier-to-understand function, but it's
;;;  noticeably slower than the new one, below.
;(defun check-ADD-infinite-values (tuple)
;  (let ((qmags (mapcar #'(lambda (qval)
;			   (cond ((eq (qmag qval) *inf-lmark*)  '+)
;				 ((eq (qmag qval) *minf-lmark*) '-)
;				 (t                              0)))
;		       tuple)))
;    (cond ((member qmags ADD-relation :test #'equal) t)
;	  (*trace-deletion-of-infinite-values*
;	   (format *QSIM-Trace* "~%(ADD ~a) deleting infinite values." tuple)
;	   nil)
;	  (t nil))))

(defmacro infinity (qmag)
  `(cond ((eq ,qmag *inf-lmark*)   '+)
	 ((eq ,qmag *minf-lmark*)  '-)
	 (t                         0)))

(defun check-ADD-infinite-values (tuple)
  (declare (optimize speed))
  (let ((Xqmag  (qmag (first  tuple)))
	(Yqmag  (qmag (second tuple)))
	(Zqmag  (qmag (third  tuple))))
    (cond ((case (infinity Xqmag)
	     (0  (eql (infinity Yqmag) (infinity Zqmag)))
	     (+  (if (eql '- (infinity Yqmag))
		     t
		     (eql '+ (infinity Zqmag))))
	     (-  (if (eql '+ (infinity Yqmag))
		     t
		     (eql '- (infinity Zqmag)))))     t)
	  (*trace-deletion-of-infinite-values*
	   (format *QSIM-Trace* "~%(ADD ~a) deleting infinite values." tuple)
	   nil)
	  (t nil))))


;;;-----------------------------------------------------------------------------
;;;  D/DT Constraint:  (d/dt level rate)
;;;
;;;  The derivative constraint is checked in two distinct modes:
;;;  1.  Under normal circumstances, the direction of change of the level
;;;      variable is controlled by the sign of the rate variable.
;;;  2.  Under quasi-equilibrium reasoning, the rate variable must be <0,std>,
;;;      while the level variable can be anything.
;;;-----------------------------------------------------------------------------

(defparameter D/DT-relation '((inc +)
			      (std 0)
			      (dec -)))

(declare-qsim-constraint
  (make-contype :name            'D/DT
		:nargs           2
		:propagator      'deriv-propagator
		:checkfcn        'check-d/dt-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p nil))


(defun check-d/dt-constraint (tuple con)
  (declare (optimize speed))
  (cond (*quasi-equilibrium-reasoning*
	 (and (eq *zero-lmark* (qmag (second tuple)))
	      (member (qdir (second tuple)) '(std ign))))
	(t (let ((qdir1 (qdir (first tuple))))
	     (or (eq qdir1 'ign)
		 (eql (lookup qdir1 D/DT-relation)
		      (qmag-order (qmag (second tuple))
				  *zero-lmark*
				  (variable--qspace
				   (second (constraint-variables con))))))))))



;;;-----------------------------------------------------------------------------
;;;  INTEGRAL Constraint
;;;
;;;  This is like check-d/dt-constraint except that the two variables are
;;;  reversed.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'INTEGRAL
		:nargs           2
		:propagator      'propagate-INTEGRAL
		:checkfcn        'check-integral-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p nil))

(defun check-integral-constraint (tuple con)
  (cond (*quasi-equilibrium-reasoning*
	 (and (eq *zero-lmark* (qmag (first tuple)))
	      (member (qdir (first tuple)) '(std ign))))
	(t (eql (lookup (qdir (second tuple)) D/DT-relation)
		(qmag-order (qmag (first tuple))
			    *zero-lmark*
			    (variable--qspace (first (constraint-variables con))))))))

;;;-----------------------------------------------------------------------------
;;;  MULT Constraint
;;;-----------------------------------------------------------------------------

(defparameter MULT-fminus-relation
  '((inc inc inc)  (std inc dec)  (dec inc dec)
    (inc inc std)  (std std std)  (dec std dec)
    (inc inc dec)  (std dec inc)  (dec dec inc)
    (inc std inc)                 (dec dec std)
    (inc dec inc)                 (dec dec dec)))

(defparameter MULT-gminus-relation
  '((inc inc inc)  (std inc inc)  (dec inc inc)
    (inc inc std)  (std std std)  (dec std inc)
    (inc inc dec)  (std dec dec)  (dec dec inc)
    (inc std dec)                 (dec dec std)
    (inc dec dec)                 (dec dec dec)))

(defparameter MULT-fgminus-relation
  '((inc inc dec)  (std inc dec)  (dec inc inc)
    (inc std dec)  (std std std)  (dec inc std)
    (inc dec inc)  (std dec inc)  (dec inc dec)
    (inc dec std)                 (dec std inc)
    (inc dec dec)                 (dec dec inc)))

(defparameter MULT-fzero-relation
  '((inc inc inc)   (std inc std)   (dec inc dec)
    (inc std inc)   (std std std)   (dec std dec)
    (inc dec inc)   (std dec std)   (dec dec dec)))

(defparameter MULT-fzero-gminus-relation
  '((inc inc dec)   (std inc std)   (dec inc inc)
    (inc std dec)   (std std std)   (dec std inc)
    (inc dec dec)   (std dec std)   (dec dec inc)))

(defparameter MULT-gzero-relation
  '((inc inc inc)  (inc std std)  (inc dec dec)
    (std inc inc)  (std std std)  (std dec dec)
    (dec inc inc)  (dec std std)  (dec dec dec)))

(defparameter MULT-gzero-fminus-relation
  '((inc inc dec)  (inc std std)  (inc dec inc)
    (std inc dec)  (std std std)  (std dec inc)
    (dec inc dec)  (dec std std)  (dec dec inc)))

(defparameter MULT-fghzero-relation
  '((inc inc std)  (std inc std)  (dec inc std)
    (inc std std)  (std std std)  (dec std std)
    (inc dec std)  (std dec std)  (dec dec std)))


; This table checks the signs of the values, and also indexes the
; appropriate table for checking the derivatives.

(defparameter MULT-relation
  `(((+ + +) ,ADD-qdir-relation)
    ((+ 0 0) ,MULT-gzero-relation)
    ((+ - -) ,MULT-gminus-relation)
    ((0 + 0) ,MULT-fzero-relation)
    ((0 0 0) ,MULT-fghzero-relation)
    ((0 - 0) ,MULT-fzero-gminus-relation)
    ((- + -) ,MULT-fminus-relation)
    ((- 0 0) ,MULT-gzero-fminus-relation)
    ((- - +) ,MULT-fgminus-relation)
     ))

;;; This table selects the proper sign for A'B when B'A=0 (or B'A when A'B=0)
;;; Added by BKay 23Sept91
;;;
(defparameter MULT-ign-relation
  '(((+ inc) inc)
    ((+ std) std)
    ((+ dec) dec)
    ((+ ign) ign)
    ((0 inc) std)
    ((0 std) std)
    ((0 dec) std)
    ((0 ign) std)
    ((- inc) dec)
    ((- std) std)
    ((- dec) inc)
    ((- ign) ign)))


(declare-qsim-constraint
  (make-contype :name              'MULT
		:nargs             3
		:propagator        'mult-propagation
		:checkfcn          'check-MULT-constraint
		:qmag-relation     MULT-relation
		:qdir-relation     nil
		:cvals-allowed-p   t
		:disallowed-lmarks (list *minf-lmark* *inf-lmark* *zero-lmark*)))


; The multiplication constraint is the most complex to check.

(defun check-MULT-constraint (tuple constraint)
  (and (mult-value-checker tuple (constraint-variables constraint))
       (check-MULT-infinite-values tuple)
       (mult-corr-vals-checker
	 (mapcar #'qval-qmag tuple)		; tuple of qmags only
	 (constraint-variables constraint)	; tuple of variables only
	 (constraint--cvals constraint))))

; A tuple of values is OK for MULT if its sign-tuple is acceptable, and
; if the qdir-tuple is acceptable to the table indexed by the sign-tuple.

(defun mult-value-checker (qval-tuple var-list)
  (let* ((sign-tuple (mapcar 
		       #'(lambda (var qval)
			   (qmag-order (qmag qval) *zero-lmark* (variable--qspace var)))
		       var-list
		       qval-tuple))
	 (qdir-relation (cadr (assoc sign-tuple MULT-relation :test #'equal)))
	 (qdir-tuple nil))
    (cond ((null qdir-relation)
	   (if *trace-mult-constraint*
	       (format *QSIM-Trace* "~%MULT~a rejecting sign tuple ~a." var-list sign-tuple))
	   nil)
	  (t (setq qdir-tuple (mapcar #'qval-qdir qval-tuple))
	     (cond ;; Replaced the next clause - BKay 23Sept91.  The old
	           ;; method was not restrictive enough.
	           ;; We now check the sign of B'A if A'=ign and B=0 and
	           ;; the sign of A'B if B'=ign and A=0.
	           ;; ((member 'ign qdir-tuple) t)
	           ((member 'ign qdir-tuple)
		    (cond
		      ((eq (third qdir-tuple) 'ign)
		       T)
		      ((and (eq (first qdir-tuple) 'ign)
			    (eq (second sign-tuple) 0))
		       (eq (third qdir-tuple)
			   (second (assoc (list (first sign-tuple)
						(second qdir-tuple))
					  MULT-ign-relation :test #'equal))))
		      ((and (eq (second qdir-tuple) 'ign)
			    (eq (first sign-tuple) 0))
		       (eq (third qdir-tuple)
			   (second (assoc  (list (second sign-tuple)
						 (first qdir-tuple))
					   MULT-ign-relation :test #'equal))))
		      (T
		       T)))
		   ((member qdir-tuple qdir-relation :test #'equal) t)
		   (t (if *trace-mult-constraint*
			  (format *QSIM-Trace* "~%MULT~a, with signs ~a, rejecting qdir tuple ~a."
				  var-list sign-tuple qdir-tuple))
		      nil))))))


(defun check-MULT-infinite-values (tuple)
  (let ((qmags (mapcar #'(lambda (qval) (cond ((eql (qmag qval) *inf-lmark*)  '+)
					      ((eql (qmag qval) *minf-lmark*) '+)
					      ((eql (qmag qval) *zero-lmark*) '-)
					      (t 0)))
		       tuple)))
    (cond ((member qmags ADD-relation :test #'equal) t)
	  (*trace-deletion-of-infinite-values*
	   (format *QSIM-Trace* "~%(MULT ~a) deleting infinite values." tuple)
	   nil)
	  (t nil))))

; Corresponding values are checked using the formula x*y=z is consistent
; with corresponding values p*q=r if the formula (x/p) * (y/q) = (z/r) is 
; acceptable (cf. [Kuipers, 1986, AIJ]).

(defun mult-corr-vals-checker (qmag-tuple vars cvlist)
  (declare (optimize speed))
  (dolist (cvtuple cvlist T)
    (if (not (mult-cv-check qmag-tuple vars cvtuple))
	(return nil))))

(defun mult-cv-check (qmag-tuple vars cvtuple)
  (declare (optimize speed))
  (let ((order-tuple
	  (mapcar #'(lambda (qmag var cv)
		      (mult-quotient-sign qmag cv (variable--qspace var)))
		  qmag-tuple
		  vars
		  cvtuple)))
    (cond ((member nil order-tuple) t)
	  ((member order-tuple ADD-relation :test #'equal) t)
	  (t (if *trace-mult-constraint*
		 (format *QSIM-Trace* "~%MULT~a rejecting ~a against corresponding values ~a."
			 vars qmag-tuple cvtuple))
	     nil))))

; This returns +,0,- according to whether the quotient f(t)/L is >1,=1,<1.
; NIL if zero or negative.

(defun MULT-quotient-sign (qmag cv qspace)
  (declare (optimize speed))
  (cond ((or (eql qmag *zero-lmark*) (eql cv *zero-lmark*))  nil)
	((and (qmag-point-p cv) (eql qmag cv))                0)
	((and (landmark-lt *zero-lmark* cv qspace)
	      (eql (qmag-order qmag cv qspace) '+))          '+)
	((and (eql (qmag-order qmag *zero-lmark* qspace) '+)
	      (eql (qmag-order qmag cv qspace) '-))          '-)
	((and (landmark-lt cv *zero-lmark* qspace)
	      (eql (qmag-order qmag cv qspace) '-))          '+)
	((and (eql (qmag-order qmag *zero-lmark* qspace) '-)
	      (eql (qmag-order qmag cv qspace) '+))          '-)
	(t nil)))


;;;-----------------------------------------------------------------------------
;;;  S+/S- Constraints
;;;
;;;  These constraints are constant on the two ends; monotonic over an interval.
;;;
;;;  Syntax:  ((S+ X Y (a b) (c d))  (x1 y1) (x2 y2) . . . )
;;;  Semantics:
;;;   1.  x <= a
;;;           qdir(y) = std
;;;           y = b
;;;
;;;   2.  a < x < c
;;;           qdir(x) = qdir(y)
;;;           forall corresponding values (p,q),  a <= p <= c
;;;              ord(x,p) = ord(y,q)
;;;
;;;   3.  x >= c
;;;           qdir(y) = std
;;;           y = d
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'S+
		:nargs           2
		:propagator      'propagate-S+
		:checkfcn        'check-S+-constraint
		:qmag-relation   M+-relation
		:qdir-relation   M+-qdir-relation
		:cvals-allowed-p t
		:bend-points-p   t))

(declare-qsim-constraint
  (make-contype :name            'S-
		:nargs           2
		:propagator      'propagate-S-
		:checkfcn        'check-S--constraint
		:qmag-relation   M--relation
		:qdir-relation   M--qdir-relation
		:cvals-allowed-p t
		:bend-points-p   t))


;;; BKay 3Sept91 - Changed requirement that qmag of the second tuple be
;;; std.  Now it must be std or ign.  This permits S+/- to work with
;;; vars with ignore-qdirs.  This change is also made in check-S--constraints.

(defun check-S+-constraint (tuple con)
  (let ((Xspace (variable--qspace (first (constraint-variables con))))
	(lb (first  (constraint-bend-points con)))
	(ub (second (constraint-bend-points con))))
    (cond ((member (qmag-order (qmag (car tuple)) (car lb) Xspace) '(- 0))
	   (and (or (eql (qdir (cadr tuple)) 'std)
		    (eql (qdir (cadr tuple)) 'ign))
		(eql (qmag (cadr tuple)) (cadr lb))))
	  ((member (qmag-order (qmag (car tuple)) (car ub) Xspace) '(0 +))
	   (and (or (eql (qdir (cadr tuple)) 'std)
		    (eql (qdir (cadr tuple)) 'ign))
		(eql (qmag (cadr tuple)) (cadr ub))))
	  (t (and (same-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		  (check-M+-cvals tuple con))))))

(defun check-S--constraint (tuple con)
  (let ((Xspace (variable--qspace (first (constraint-variables con))))
	(ub (first  (constraint-bend-points con)))
	(lb (second (constraint-bend-points con))))
    (cond ((member (qmag-order (qmag (car tuple)) (car ub) Xspace) '(- 0))
	   (and (or (eql (qdir (cadr tuple)) 'std)
		    (eql (qdir (cadr tuple)) 'ign))
		(eql (qmag (cadr tuple)) (cadr ub))))
	  ((member (qmag-order (qmag (car tuple)) (car lb) Xspace) '(0 +))
	   (and (or (eql (qdir (cadr tuple)) 'std)
		    (eql (qdir (cadr tuple)) 'ign))
		(eql (qmag (cadr tuple)) (cadr lb))))
	  (t (and (opposite-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		  (check-M--cvals tuple con))))))

;;;-----------------------------------------------------------------------------
;;;  U+/U- Constraints
;;;
;;;  Working on U+/U- constraints.
;;;
;;;  Syntax:  ((U+ X Y (a b))  (x1 y1) (x2 y2) . . . )
;;;  Semantics:
;;;   1.  x = a
;;;           qdir(y) = std
;;;           y = b
;;;
;;;   2.  x < a                               - 
;;;           acts like M-
;;;           check only corresponding values (p,q), p <= a.
;;;
;;;   3.  x > a                               - 
;;;           acts like M+
;;;           check only corresponding values (p,q), p >= a.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'U+
		:nargs           2
		:propagator      'propagate-U+
		:checkfcn        'check-U+-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p t
		:bend-points-p   t))

(declare-qsim-constraint
  (make-contype :name            'U-
		:nargs           2
		:propagator      nil		; need 'propagate-U-
		:checkfcn        'check-U--constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p t
		:bend-points-p   t))

(defun check-U+-constraint (tuple con)
  (let ((Xspace (variable--qspace (first  (constraint-variables con))))
	(Yspace (variable--qspace (second (constraint-variables con))))
	(cvals  (constraint--cvals con))
	(cusp   (first  (constraint-bend-points con))))

    (cond ((eql (qmag (car tuple)) (car cusp))
	   (and (eql (qdir (cadr tuple)) 'std)
		(eql (qmag (cadr tuple)) (cadr cusp))))

	  ; for x < a, behave like M-
	  ((eql (qmag-order (qmag (car tuple)) (car cusp) Xspace) '-)
	   (and (opposite-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		(dolist (cv-tuple cvals t)
		  (or (eql (qmag-order (car cv-tuple) (car cusp) Xspace) '+)
		      (opposite-order (qmag-order (qmag (car tuple)) (car cv-tuple) Xspace)
				      (qmag-order (qmag (cadr tuple)) (cadr cv-tuple) Yspace))
		      (return nil)))))

	  ; for x > a, behave like M+
	  ((eql (qmag-order (qmag (car tuple)) (car cusp) Xspace) '+)
	   (and (same-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		(dolist (cv-tuple cvals t)
		  (or (eql (qmag-order (car cv-tuple) (car cusp) Xspace) '-)
		      (same-order (qmag-order (qmag (car tuple)) (car cv-tuple) Xspace)
				  (qmag-order (qmag (cadr tuple)) (cadr cv-tuple) Yspace))
		      (return nil))))))))

(defun check-U--constraint (tuple con)
  (let ((Xspace (variable--qspace (first  (constraint-variables con))))
	(Yspace (variable--qspace (second (constraint-variables con))))
	(cvals  (constraint--cvals con))
	(cusp   (first  (constraint-bend-points con))))

    (cond ((eql (qmag (car tuple)) (car cusp))
	   (and (eql (qdir (cadr tuple)) 'std)
		(eql (qmag (cadr tuple)) (cadr cusp))))

	  ; for x < a, behave like M+
	  ((eql (qmag-order (qmag (car tuple)) (car cusp) Xspace) '-)
	   (and (same-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		(dolist (cv-tuple cvals t)
		  (or (eql (qmag-order (car cv-tuple) (car cusp) Xspace) '+)
		      (same-order (qmag-order (qmag (car tuple)) (car cv-tuple) Xspace)
				  (qmag-order (qmag (cadr tuple)) (cadr cv-tuple) Yspace))
		      (return nil)))))

	  ; for x > a, behave like M-
	  ((eql (qmag-order (qmag (car tuple)) (car cusp) Xspace) '+)
	   (and (opposite-qdirs (qdir (car tuple)) (qdir (cadr tuple)))
		(dolist (cv-tuple cvals t)
		  (or (eql (qmag-order (car cv-tuple) (car cusp) Xspace) '-)
		      (opposite-order (qmag-order (qmag (car tuple)) (car cv-tuple) Xspace)
				      (qmag-order (qmag (cadr tuple)) (cadr cv-tuple) Yspace))
		      (return nil))))))))


;;;-----------------------------------------------------------------------------
;;;  ZERO-STD Constraint
;;;
;;;  This constraint requires its single variable to have the qval '(0 std).
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'zero-std
		:nargs           1
		:propagator      'propagate-zero-std
		:checkfcn        'check-zero-std-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p nil))


(defun check-zero-std-constraint (tuple constraint)
  (declare (ignore constraint))
  ;; The ign clause was added by BKay 2Jul91 so that it works for
  ;; vars which have qdir = ign.  Should the propagator be changed likewise?
  (and (or (eql 'std (qdir (first tuple)))
	   (eql 'ign (qdir (first tuple))))
       (eq  *zero-lmark* (qmag (first tuple)))))

(defun propagate-zero-std (con cvals state)
  (declare (ignore cvals))
  (let* ((var  (first (constraint-variables con)))
	(qmag (qmag (variable--qval var)))
	(qdir (qdir (variable--qval var))))
    (unless qmag
      (assert-EQ var *zero-lmark* state))
    (unless qdir
      (set-qdir var 'std))
    (when (or (null qmag)(null qdir))
      (list var))))				; return list of changed variables


;;;-----------------------------------------------------------------------------
;;;  POSITIVE-STD Constraint
;;;
;;;  This constraint requires its single variable to have the qval '(0 std).
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'positive-std
		:nargs           1
		:propagator      'propagate-positive-std
		:checkfcn        'check-positive-std-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p nil))


(defun check-positive-std-constraint (tuple constraint)
  (let* ((qmag (qmag (first tuple)))
	 (lowend (if (consp qmag) (first qmag) qmag)))
    (and (eql 'std (qdir (first tuple)))
	 (not (eq *inf-lmark* (qmag (first tuple))))
	 (member lowend
		 (cdr (member *zero-lmark*
			      (variable--qspace
				(first (constraint-variables
					 constraint)))))))))

(defun propagate-positive-std (con cvals state)
  (declare (ignore cvals))
  (let* ((var  (first (constraint-variables con)))
	(qmag (qmag (variable--qval var)))
	(qdir (qdir (variable--qval var))))
    (unless qmag
      (assert-lb var *zero-lmark* state)
      (assert-ub var (car (last (variable--qspace var))) state))
    (unless qdir
      (set-qdir var 'std))
    (when (or (null qmag)(null qdir))
      (list var))))				; return list of changed variables

;;;-----------------------------------------------------------------------------
;;;  NEGATIVE-STD Constraint
;;;
;;;  This constraint requires its single variable to have the qval '(0 std).
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'negative-std
		:nargs           1
		:propagator      'propagate-negative-std
		:checkfcn        'check-negative-std-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p nil))


(defun check-negative-std-constraint (tuple constraint)
  (and (eql 'std (qdir (first tuple)))
       (not (eq *minf-lmark* (qmag (first tuple))))
       (let* ((qmag (qmag (first tuple)))
	      (lowend (if (consp qmag) (first qmag) qmag)))
	 (member *zero-lmark*
		 (cdr (member lowend
			      (variable--qspace
				(first (constraint-variables
					 constraint)))))))))

(defun propagate-negative-std (con cvals state)
  (declare (ignore cvals))
  (let* ((var  (first (constraint-variables con)))
	 (qspace (variable--qspace var))
	 (qmag (qmag (variable--qval var)))
	 (qdir (qdir (variable--qval var))))
    (unless qmag
      (assert-ub var *zero-lmark* state)
      (assert-lb var (car qspace) state))
    (unless qdir
      (set-qdir var 'std))
    (when (or (null qmag)(null qdir))
      (list var))))				; return list of changed variables


;;;-----------------------------------------------------------------------------
;;;  SUM-ZERO Constraint
;;;
;;;  The SUM-ZERO constraint takes any number of variables and asserts that
;;;  their sum is zero.  Either all signs are zero, or there are both + and -
;;;  values.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'sum-zero
		:nargs           99
		:propagator      'propagate-sum-zero
		:checkfcn        'check-sum-zero-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p t))


(defun check-sum-zero-constraint (tuple constraint)
  (and (check-sum-zero-qdirs tuple)
       (check-sum-zero-infinite-values tuple)
       (check-sum-zero-cvals tuple constraint)))

(defun sum-zero-test (fn sequence)
  (let ((last-sign nil)
	x)
    (dolist (item sequence)
      (case (setq x (funcall fn item))
	(+    (case last-sign
		(-    (return-from sum-zero-test T))
		((nil)  (setq last-sign '+))))
	(-    (case last-sign
		(+    (return-from sum-zero-test T))
		((nil)  (setq last-sign '-))))
	((NIL)  (return-from sum-zero-test T))      
	(0    nil)
	(t    (error "sum-zero-test: ~a invalid" x))))
    (null last-sign)))

(defun iff (x y)				; T iff truth(x)=truth(y)
  (eql (null x) (null y)))

(defun sum-zero-tuple-test (tuple)
  (or (member nil tuple)
      (iff (member '+ tuple) (member '- tuple))))

;;; New version (almost twice as fast as old version).
;;; 18.Oct.90 Adam Farquhar.  Added the NIL case.  I think that a Nil should
;;; result in an OK check.
(defun check-sum-zero-qdirs (tuple)
  (declare (optimize speed))
  (let ((last-sign nil))
    (dolist (qval tuple)
      (case (qdir qval)
	(inc  (case last-sign
		(-     (return-from check-sum-zero-qdirs T))
		((nil) (setq last-sign '+))))
	(dec  (case last-sign
		(+     (return-from check-sum-zero-qdirs T))
		((nil) (setq last-sign '-))))
	(ign  (return-from check-sum-zero-qdirs T))
	((NIL)   (return-from check-sum-zero-qdirs T))
	(std  nil)
	(t    (error "sum-zero-test: ~a invalid" (qdir qval)))))
    (null last-sign)))
  

;;; Old Version (slower, but maybe easier to understand).
;(defun check-sum-zero-qdirs (tuple)
;  (sum-zero-tuple-test
;    (mapcar #'(lambda (qval) (case (qdir qval)
;			       (inc '+)
;			       (std '0)
;			       (dec '-)
;			       (ign nil)))
;	    tuple)))

;;; New Version (over twice as fast as old version, below).
(defun check-sum-zero-infinite-values (tuple)
  (declare (optimize speed))
  (let ((last-sign nil)
	qmag)
    (dolist (qval tuple)
      (setq qmag (qmag qval))
      (if (point-p qmag)
	  (cond ((eq qmag *inf-lmark*)
		 (case last-sign
		   (-     (return-from check-sum-zero-infinite-values T))
		   ((nil) (setq last-sign '+))))
		((eq qmag *minf-lmark*)
		 (case last-sign
		   (+     (return-from check-sum-zero-infinite-values T))
		   ((nil) (setq last-sign '-)))))))
    (null last-sign)))

;;; Old Version (slower, but maybe easier to understand).
;(defun check-sum-zero-infinite-values (tuple)
;  (sum-zero-tuple-test
;    (mapcar #'(lambda (qval)
;		(let ((qmag (qmag qval)))
;		  (if (interval-p qmag)
;		      0
;		      (case (lmark-name qmag)
;			(inf  '+)
;			(minf '-)
;			(t    0)))))
;	    tuple)))

;;; New Version (faster).
(defun check-sum-zero-cvals (tuple constraint)
  (declare (special tuple))
  (let* ((vars (constraint-variables constraint)))
    (declare (special vars))
    ;; Return T only if all cv-tuples are OK.
    (dolist (cv-tuple (constraint--cvals constraint) t)
      (if (not (check-sum-zero-cv-tuple cv-tuple))
	  (return nil)))))

;;    (every #'check-sum-zero-cv-tuple (constraint--cvals constraint))))

(defun check-sum-zero-cv-tuple (cv-tuple)
  (declare (optimize speed))
  (let ((last-sign nil))
    (declare (special tuple vars))
    (mapc #'(lambda (qval cval var)
	      (case (qmag-order (qmag qval) cval (variable--qspace var))
		(+    (case last-sign
			(-     (return-from check-sum-zero-cv-tuple T))
			((nil) (setq last-sign '+))))
		(-    (case last-sign
			(+     (return-from check-sum-zero-cv-tuple T))
			((nil) (setq last-sign '-))))
		(0    nil)
		((nil)  (return-from check-sum-zero-cv-tuple T))
		(t    (error "sum-zero-test: ~a invalid"
			     (qmag-order (qmag qval) cval (variable--qspace var))))))
	  tuple
	  cv-tuple
	  vars)
    (null last-sign)))
		
;;; Old Version (slower but easier to understand).
;(defun check-sum-zero-cvals (tuple constraint)
;  (let* ((vars (constraint-variables constraint)))
;    (every #'(lambda (cv-tuple)
;	       (sum-zero-tuple-test
;		 (mapcar #'(lambda (qval cval var)
;			     (qmag-order (qmag qval) cval (variable--qspace var)))
;			 tuple
;			 cv-tuple
;			 vars)))
;	   (constraint--cvals constraint))))

;;;-----------------------------------------------------------------------------
;;;  (M s1 ... sn) Constraints                                   (BJK:  10-26-90)
;;;
;;;  The M constraint represents a multivariable function relation y = f(x1 ... xn).
;;;  It takes any number of variables and a tuple of signs representing the
;;;  partial derivatives, si = dy/dxi (partial derivs).
;;;  The product of corresponding signs and partial are tested like SUM-ZERO:
;;;  Either all signs are zero, or there are both + and - values.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'M
		:nargs           99
		:propagator      'M-propagator	; doesn't exist yet
		:checkfcn        'check-M-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p t))

; When the constraint structure is build for an instance of the M constraint
; (M s1 ... sn), the sign tuple (s1 ... sn -) must be put into constraint.partials.

(defun check-M-constraint (tuple constraint)
  (let ((partials (constraint-partials constraint)))
    (and (check-M-qdirs tuple partials)
	 (check-M-infinite-values tuple partials)
	 (check-M-cvals tuple constraint partials))))

; Test whether a tuple of qdirs, multiplied by a tuple of partials, is consistent.

(defun check-M-qdirs (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null P) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  ((eql (qval-qdir (car L)) 'ign) (return-from check-M-qdirs t))
	  ((eql (qval-qdir (car L)) 'std))
          ((sign-qdir-match (car P) (qval-qdir (car L))) (incf pluses))
          (t (incf minuses)))))

(defun sign-qdir-match (sign qdir)
  (case sign
    (+ (eql qdir 'inc))
    (0 (eql qdir 'std))
    (- (eql qdir 'dec))))

; Infinity check depends on [x]_inf = +,0,- for infinity,finite,neg-infinity, respectively.

(defun check-M-infinite-values (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null L) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  (t (let ((s (sign-wrt-infinity (qval-qmag (car L)))))
	       (cond ((eql s 0))
		     ((eql s (car P)) (incf pluses))
		     (t (incf minuses))))))))

(defun sign-wrt-infinity (qmag)
  (cond ((listp qmag) 0)
	((eql qmag *inf-lmark*) '+)
	((eql qmag *minf-lmark*) '-)
	(t 0)))

; Check wrt corresponding values computes a sign tuple wrt each cval tuple.
;   This can be speeded up by replacing the (M-check (mapcar ... )) with a DO.

(defun check-M-cvals (tuple constraint partials)
  (let ((qspaces (mapcar #'variable--qspace (constraint-variables constraint)))
	(cv-tuples (constraint--cvals constraint)))
    (dolist (cv-tuple cv-tuples t)
      (or (M-check (mapcar #'(lambda (qval cval qspace)
			       (qmag-order (qmag qval) cval qspace))
			   tuple
			   cv-tuple
			   qspaces)
		   partials)
	  (return-from check-M-cvals nil)))))

(defun M-check (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null P) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  ((eql (car L) 0))
          ((eql (car L) (car P)) (incf pluses))
          (t (incf minuses)))))

; Note that the ADD check can easily be redefined in terms of M-check.
;
;(defun ADD-tuple-check (tuple)
;  (M-check tuple '(+ + -)))

;;;-----------------------------------------------------------------------------
;;;  These 1-argument constraints simply assert that its argument has a certain
;;;  direction of change.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'constant
		:nargs           1
		:propagator      'constant-propagator 
		:checkfcn        'check-constant-constraint
		:cvals-allowed-p nil))

(defun check-constant-constraint (tuple constraint)
  (and (or (eql (qdir (first tuple)) 'std)
	   (eql (qdir (first tuple)) 'ign))
       (if (car (constraint-bend-points constraint))
	   (and (lmark-p (qmag (first tuple)))
		(eql (car (constraint-bend-points constraint))
		     (lmark-name (qmag (first tuple)))))
	 t)))


(declare-qsim-constraint
  (make-contype :name            'non-constant
		:nargs           1
		:propagator      nil
		:checkfcn        'check-non-constant-constraint
		:cvals-allowed-p nil))

(defun check-non-constant-constraint (tuple constraint)
  (declare (ignore constraint))
  (not (eql (qdir (first tuple)) 'std)))


(declare-qsim-constraint
  (make-contype :name            'increasing
		:nargs           1
		:propagator      'unary-propagator 
		:checkfcn        'check-increase-constraint
		:cvals-allowed-p nil))

(defun check-increase-constraint (tuple constraint)
  (declare (ignore constraint))
  (eql (qdir (first tuple)) 'inc))


(declare-qsim-constraint
  (make-contype :name            'decreasing
		:nargs           1
		:propagator      'unary-propagator 
		:checkfcn        'check-decrease-constraint
		:cvals-allowed-p nil))

(defun check-decrease-constraint (tuple constraint)
  (declare (ignore constraint))
  (eql (qdir (first tuple)) 'dec))


(declare-qsim-constraint
  (make-contype :name            'non-increasing
		:nargs           1
		:propagator      nil
		:checkfcn        'check-non-increase-constraint
		:cvals-allowed-p nil))

(defun check-non-increase-constraint (tuple constraint)
  (declare (ignore constraint))
  (not (eql (qdir (first tuple)) 'inc)))


(declare-qsim-constraint
  (make-contype :name            'non-decreasing
		:nargs           1
		:propagator      nil
		:checkfcn        'check-non-decrease-constraint
		:cvals-allowed-p nil))

(defun check-non-decrease-constraint (tuple constraint)
  (declare (ignore constraint))
  (not (eql (qdir (first tuple)) 'dec)))


;;; Unreachable values are properly handled as unary constraints;
;;;   (UNREACHABLE <var>  (VALUES ... <values> ... )),  e.g.
;;;   (UNREACHABLE volume (VALUES 0 inf))
;;; The test is #`equal in case there is a reason to use non-landmark values.

(declare-qsim-constraint
  (make-contype :name            'unreachable
		:nargs           1
		:propagator      nil
		:checkfcn        'check-unreachable-constraint
		:cvals-allowed-p nil))

(defun check-unreachable-constraint (tuple constraint)
  (let ((qval (first tuple))
	(var  (first (constraint-variables constraint))))
    (not (member (qmag qval) (variable-unreachable-values var)))))

;;;-----------------------------------------------------------------------------
;;;  W+/W- Constraints
;;;
;;;  Weak monotonic constraint W+ (or W-) simply check that the qdirs are
;;;  identical (or opposite).  Therefore, W+(X,Y) is essentially the same as
;;;  the confluence dX = dY.  Of course, these constraints cannot have
;;;   corresponding values.
;;;
;;;  This was required by the Heart model where a monotonic function constraint 
;;;  needs to be put in parallel with a quasi-static feedback mechanism in order
;;;  to assert that the gain < 1, so that the response of the system with
;;;  feedback cannot be opposite to the unregulated system.
;;;
;;;  It captures a very incomplete knowledge:  the directions of change are
;;;  the same without a functional relationship.  The functional relationship
;;;  is screwed up because of possible changes to some of the parameters of the
;;;  feedback path.  Probably this should more legitimately be handled by an
;;;  abstraction relation where the M+ could have corresponding values
;;;  conditional on the values of the parameters on the feedback path.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'W+
		:nargs           2
		:propagator      'propagate-W
		:checkfcn        'check-W+-constraint
		:qmag-relation   nil
		:qdir-relation   M+-qdir-relation
		:cvals-allowed-p nil))

(defun check-W+-constraint (tuple constraint)
  (declare (ignore constraint))
  (same-qdirs (qdir (first tuple)) (qdir (second tuple))))


(declare-qsim-constraint
  (make-contype :name            'W-
		:nargs           2
		:propagator      'propagate-W
		:checkfcn        'check-W--constraint
		:qmag-relation   nil
		:qdir-relation   M--qdir-relation
		:cvals-allowed-p nil))

(defun check-W--constraint (tuple constraint)
  (declare (ignore constraint))
  (opposite-qdirs (qdir (first tuple)) (qdir (second tuple))))


;;;-----------------------------------------------------------------------------
;;;  CORRESPOND+, CORRESPOND- Constraints
;;;  
;;;  This constraint enforces corresponding values, and that's all.
;;;  CORRESPOND+ and CORRESPOND- are like M+ and M- except that they don't
;;;  enforce qdirs.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name           'CORRESPOND+
		:nargs          2
		:propagator     'correspond-propagation
		:checkfcn       'check-CORRESPOND+-constraint
		:qmag-relation  nil
		:qdir-relation  nil
		:cvals-allowed-p t))


(defun check-CORRESPOND+-constraint (tuple con)
  (declare (optimize speed))
  (let* ((vars    (constraint-variables con))
	 (Xqspace (variable--qspace (first  vars)))
	 (Yqspace (variable--qspace (second vars)))
	 (Xqmag   (qmag (first  tuple)))
	 (Yqmag   (qmag (second tuple))))
    ;; Ensure that tuple's relation to every cv-tuple is OK.
    (dolist (cv (constraint--cvals con) T)
      (if (not (same-order (qmag-order Xqmag (first  cv) Xqspace)
			   (qmag-order Yqmag (second cv) Yqspace)))
	  (return nil)))))


(declare-qsim-constraint
  (make-contype :name           'CORRESPOND-
		:nargs          2
		:propagator     'correspond-propagation
		:checkfcn       'check-CORRESPOND--constraint
		:qmag-relation  nil
		:qdir-relation  nil
		:cvals-allowed-p t))


(defun check-CORRESPOND--constraint (tuple con)
  (declare (optimize speed))
  (let* ((vars    (constraint-variables con))
	 (Xqspace (variable--qspace (first  vars)))
	 (Yqspace (variable--qspace (second vars)))
	 (Xqmag   (qmag (first  tuple)))
	 (Yqmag   (qmag (second tuple))))
    ;; Ensure that tuple's relation to every cv-tuple is OK.
    (dolist (cv (constraint--cvals con) T)
      (if (not (opposite-order (qmag-order Xqmag (first  cv) Xqspace)
			       (qmag-order Yqmag (second cv) Yqspace)))
	  (return nil)))))


;;;----------------------------------------------------------------------
;;;
;;; CORRESPONDENCE constraint                         (AF: 20 March 1991)
;;;
;;; 
;;; This constraint is weaker than correspond+/-, as it ONLY checks
;;; for specific tuples.
;;;
;;; Given a cvalue: (m1 ... mn), and a tuple of variables: (v1 ... vn),
;;; Then, the correspondence constraint is satisfied if:
;;; 1. forall i, vi=mi, or
;;; 2. vi <> mi and vj <> mj (i<>j).
;;;
;;; Purpose: to provide a VERY weak constraint.  QPC sometimes can infer
;;; this information.
;;;

(declare-qsim-constraint
  (make-contype :name            'correspondence
		:nargs           99
		:propagator      'propagate-correspondence-constraint
		:checkfcn        'check-correspondence-constraint
		:cvals-allowed-p t))

(defun check-correspondence-constraint (tuple con)
  (declare (optimize speed))
  (dolist (corresponding-values (constraint--cvals con) T)
    (let ((mismatch? (mismatched (constraint-variables con)
			   tuple corresponding-values)))
      (cond
	((null mismatch?) 
	 (return-from check-correspondence-constraint T))
	((not (eq mismatch? T))
	 (return-from check-correspondence-constraint NIL))))))

(defun mismatched (vars tuple cv-tuple &aux (mismatch nil))
  "Return NIL if the tuples match, var if there is ONE mismatch or T if
there is more than one mismatch." 
  (map nil #'(lambda (var val mag)
	       (let ((order (qmag-order  (qmag val) mag
					 (variable--qspace var))))
		 (if (member order '(+ -))
		     ;; a mismatch
		     (if mismatch
			 (RETURN-FROM mismatched t)
			 (setq mismatch var)))))
       vars tuple cv-tuple)
  mismatch)

  


;;;-----------------------------------------------------------------------------
;;;  Function:  (P-successors  qval  qspace  var)
;;;             (I-successors  qval  qspace  var)
;;;
;;;  Purpose:   Given a qualitative value from the current state, create and
;;;             return all possible next values of the variable.  This code is
;;;             justified by Appendix A in [Kuipers, 1986].
;;;
;;;  Terminology change:  P- & I-transitions ====> P- & I-successors
;;;             [Kuipers, 1986] refers to this step as P- and I-transitions,
;;;             but the word "transitions" is also used to refer to region
;;;             transitions, which is a jump from one QDE to another.  Thus,
;;;             to avoid confusion, the word "transitions" will now refer only
;;;             to region transitions, and we'll use the more suggestive word
;;;             "successors" here.  After all, the successor values generated
;;;             here give rise to successor states.  -- DD
;;;
;;;  P-successors are possible next values when moving from a time-point
;;;  to a time-interval:  t0 --> (t0 t1).
;;;  11-21:  experimentally added a new qdir:  IGN.
;;;-----------------------------------------------------------------------------

(defun P-successors (qval qspace var)
  (let ((qmag    (qmag qval))
	(qdir    (qdir qval)))
    (cond ((qmag-point-p qmag)
	   ;; qmag is a point value
	   (cond ((eql qdir 'inc) (list (make-qval :variable var
						   :qmag (list qmag (succ qmag qspace))
						   :qdir 'inc)))
		 ((eql qdir 'dec) (list (make-qval :variable var
						   :qmag (list (pred qmag qspace) qmag)
						   :qdir 'dec)))
		 ((eql qdir 'std) (delete nil
					  (list (if (not (eql qmag (car (last qspace))))
						    (make-qval :variable var
							       :qmag (list qmag (succ qmag qspace))
							       :qdir 'inc))
						(if (not (eql qmag (first qspace)))
						    (make-qval :variable var
							       :qmag (list (pred qmag qspace) qmag)
							       :qdir 'dec))
						(make-qval :variable var
							   :qmag qmag
							   :qdir 'std))))
		 ((eql qdir 'ign) (delete nil
					  (list (if (not (eql qmag (car (last qspace))))
						    (make-qval :variable var
							       :qmag (list qmag (succ qmag qspace))
							       :qdir 'ign))
						(if (not (eql qmag (first qspace)))
						    (make-qval :variable var
							       :qmag (list (pred qmag qspace) qmag)
							       :qdir 'ign))
						(make-qval :variable var
							   :qmag qmag
							   :qdir 'ign))))))
	  ;; qmag is an interval
	  (t (cond ((eql qdir 'std) (list (make-qval :variable var :qmag qmag :qdir 'inc)
					  (make-qval :variable var :qmag qmag :qdir 'dec)
					  (make-qval :variable var :qmag qmag :qdir 'std)))
		   (t  (list qval)))))))


;;;-----------------------------------------------------------------------------
;;;  I-successors are possible next values when moving from a time-interval
;;;  to a time-point:  (t0 t1) --> t1.
;;;-----------------------------------------------------------------------------

(defun I-successors (qval qspace var)
  (declare (ignore qspace))
  (let ((qmag (qmag qval))
	(qdir (qdir qval)))
    (cond ((qmag-point-p qmag)
	   ;; qmag is a point value
	   (cond ((eql qdir 'std) (list qval))
		 ((eql qdir 'ign) (list (make-qval :variable var :qmag qmag :qdir 'ign)))
		 (t  (error "Can't be (~a, ~a) over an interval." qmag qdir))))
	  ;; qmag is an interval
	  (t (cond ((eql qdir 'inc) (list (make-qval :variable var :qmag (cadr qmag) :qdir 'std)
					  (make-qval :variable var :qmag (cadr qmag) :qdir 'inc)
					  (make-qval :variable var :qmag qmag :qdir 'std)
					  qval))
		   ((eql qdir 'dec) (list (make-qval :variable var :qmag (car qmag) :qdir 'std)
					  (make-qval :variable var :qmag (car qmag) :qdir 'dec)
					  (make-qval :variable var :qmag qmag :qdir 'std)
					  qval))
		   ((eql qdir 'std) (list qval))
		   ((eql qdir 'ign) (list (make-qval :variable var :qmag (cadr qmag) :qdir 'ign)
					  (make-qval :variable var :qmag (car qmag)  :qdir 'ign)
					  qval)))))))



;;;-----------------------------------------------------------------------------
;;;  Test for P- and I-successors:  Is a pair of qvalues consistent with continuity?
;;;  P-successors go from a time-point to a time-interval:  t0 -> (t0,t1).
;;;-----------------------------------------------------------------------------

(defun test-P-successor (qvalue1 qvalue2 qspace)
  (declare (ignore qspace))
  (let* ((qval1 (cdr qvalue1))
	 (qval2 (cdr qvalue2))
	 (qmag1 (qmag qval1))
	 (qmag2 (qmag qval2))
	 (qdir1 (qdir qval1))
	 (qdir2 (qdir qval2)))
    (cond ((qmag-point-p qmag1)
	   (cond ((eql qdir1 'inc) (and (member qdir2 '(inc ign))
					(qmag-interval-p qmag2)
					(eql qmag1 (car qmag2))))
		 ((eql qdir1 'dec) (and (member qdir2 '(dec ign))
					(qmag-interval-p qmag2)
					(eql qmag1 (cadr qmag2))))
		 ((eql qdir1 'std) (cond ((eql qmag1 qmag2)
					  (member qdir2 '(std ign)))
					 ((qmag-point-p qmag2) nil)
					 ((eql qmag1 (car qmag2))
					  (member qdir2 '(inc ign)))
					 ((eql qmag1 (cadr qmag2))
					  (member qdir2 '(dec ign)))))
		 ((eql qdir1 'ign) (cond ((eql qmag1 qmag2)
					  (member qdir2 '(std ign)))
					 ((qmag-point-p qmag2) nil)
					 ((eql qmag1 (car qmag2))
					  (member qdir2 '(inc ign)))
					 ((eql qmag1 (cadr qmag2))
					  (member qdir2 '(dec ign)))))))
	  (t (cond ((member qdir1 '(std ign)) (equal qmag1 qmag2))
		   ((eql qdir1 'inc) (and (equal qmag1 qmag2)
					  (member qdir2 '(inc ign))))
		   ((eql qdir1 'dec) (and (equal qmag1 qmag2)
					  (member qdir2 '(dec ign)))))))))


;;;-----------------------------------------------------------------------------
;;;  I-successors go from a time-interval to a time-point:  (t0,t1) -> t1.
;;;-----------------------------------------------------------------------------

(defun test-I-successor (qvalue1 qvalue2 qspace)
  (let* ((qval1 (cdr qvalue1))
	 (qval2 (cdr qvalue2))
	 (qmag1 (qmag qval1))
	 (qmag2 (qmag qval2))
	 (qdir1 (qdir qval1))
	 (qdir2 (qdir qval2)))
    (cond ((qmag-point-p qmag1)
	   (cond ((eql qdir1 'std) (and (equal qmag1 qmag2)
					(member qdir2 '(std ign))))
		 ((eql qdir1 'ign) (and (equal qmag1 qmag2)
					(member qdir2 '(std ign))))))
	  (t (cond ((eql qdir1 'inc) (cond ((eql qmag2 (cadr qmag1))
					    (member qdir2 '(inc std ign)))
					   ((equal qmag2 qmag1)
					    (member qdir2 '(inc std ign)))
					   ((eql qmag2 (succ (car qmag1) qspace))
					    (member qdir2 '(std ign)))))
		   ((eql qdir1 'dec) (cond ((eql qmag2 (car qmag1))
					    (member qdir2 '(dec std ign)))
					   ((equal qmag2 qmag1)
					    (member qdir2 '(dec std ign)))
					   ((eql qmag2 (pred (cadr qmag1) qspace))
					    (member qdir2 '(std ign)))))
		   ((eql qdir1 'std) (and (equal qmag1 qmag2)
					  (member qdir2 '(std ign))))
		   ((eql qdir1 'ign) (cond ((equal qmag2 qmag1) t)
					   ((qmag-interval-p qmag2) nil)
					   ((eql qmag2 (car qmag1))
					    (member qdir2 '(dec std ign)))
					   ((eql qmag2 (cadr qmag1))
					    (member qdir2 '(inc std ign)))
					   ((eql qmag2 (pred (cadr qmag1) qspace))
					    (member qdir2 '(std ign)))
					   ((eql qmag2 (succ (car qmag1) qspace))
					    (member qdir2 '(std ign))))))))))



