;;; -*- Mode: Common-Lisp; Package: QSIM; Syntax: COMMON-LISP; Base: 10 -*-

(in-package 'QSIM)

;;; This file contains the majority of the vanilla Common Lisp code for VAR-SLICE-VIEWER and
;;; PHASE-SLICE-VIEWER.  It depends on the code in Viewers-Misc which is not Common Lisp
;;; and the code in Viewers-Clisp-2 which is.  Refer to Read-Me-About-Viewers for details.

;;; VAR-SLICE-VIEWER is executed by entering the form (VAR-SLICE-VIEWER).  It assumes that QSIM
;;; has run so that *initial-state* is bound (thus state.successors has pointers leading
;;; to the behaviors).  It also assumes that *reference-states* is bound.  It prompts for a
;;; parameter and displays behaviors of the paramter
;;; across branches of the behavior tree.  It also plots to PostScript files according to
;;; switches set in PS control.

;;;  Phase Diagram Plotter uses the exact same structure as Var Slice Viewer.  They differ only
;;;  in the individual plots.  It is executed with the form (phase-slice-viewer).

;;;  This file also contains the code for the Behavior Comparison Viewer.  This is contained
;;;  at the end of this file and it is used to compare the behavior plots of
;;;  particular behaviors for specific variables.   DJC  05Feb92


(defvar *viewer-mode*)				; time plot or phase plot
(defvar viewer-rows 4)				; layout parameters --- rows of plots per screen
(defvar viewer-cols 4)				;                   --- plots per row
(defvar *time-points-only* t)			; whether to plot time intervals or not in phase plots

;; Default layouts.
(defvar *var-slice-rows* 4)
(defvar *var-slice-cols* 3)
(defvar *phase-slice-rows* 2)
(defvar *phase-slice-cols* 4)

(defun var-slice-viewer (initials)       ; added the argument so that this can be used without accessing 
  (setq *viewer-mode* 'var-slice         ; the special variable *initial-state*  DJC  07/20/91
	viewer-rows *var-slice-rows*
	viewer-cols *var-slice-cols*)
  (viewer initials))



(defun phase-slice-viewer (initials)      ; added the argument so that this can be used without accessing 
  (setq *viewer-mode* 'phase-slice        ; the special variable *initial-state*  DJC  07/20/91
	viewer-rows *phase-slice-rows*
	viewer-cols *phase-slice-cols*)
  (viewer initials))

;; Getting parameter(s) for Viewer.

(defun viewer (initials)  ;  changed so that it uses initials as opposed to initial state  DJC  07/20/91
  (let* ((layout (or (qde-layout (state-qde (car initials)))
		     (find-some-layout
		       (apply #'append (mapcar #'get-behaviors
					       initials))
		       *reference-states*)))
	 next)
    (compute-rectangular-layout viewer-rows viewer-cols)
    (do ((parms (get-parms (car initials))
	        (case next
		  ((#\p #\P #\v #\V) (get-parms (car initials)))
		  ((#\q #\Q) nil))))
	((null parms))
      (setq next (viewer-plot parms initials)))
    (compute-layout layout)))



;; Viewer Command selection.

(defun viewer-plot (parm initials)
  (let* ((beh-lists (mapcar #'get-behaviors initials))
	 (behaviors (apply #'append beh-lists))
	 (lengths (mapcar #'length beh-lists))
	 (max-length (apply #'max (mapcan #'(lambda (L) (mapcar #'length L)) beh-lists)))
	 (btotal (length behaviors))
	 (click-p nil))
    (when (eq *viewer-mode* 'phase-slice)
      (format *Qsim-Report* "~&Time intervals are currently~a plotted.  ~
                   Enter I to change Mode. " (if *time-points-only* " not" "")))
    (loop
      for cmd = (get-viewer-cmd btotal click-p)
      until (member cmd '(#\q #\Q #\p #\P #\v #\V))
      do
      (setq click-p nil)
      (case cmd
	((#\C #\c)
	 #+Symbolics
	 (let ((list (select-behs-from-beh-tree initials max-length btotal nil)))
	   (qplot-new-behavior)
	   (viewer-header parm initials btotal)
	   (format *Qsim-Trace* "~&Plotting ...")
	   (view-behs 'given initials parm behaviors btotal lengths list)
	   (format *Qsim-Trace* "  Done. ")
	   )
	 #-Symbolics
	 (format *Qsim-Report* "~&Command ~a not available on non-Symbolics machines." cmd))
	((#\l #\L) (change-viewer-layout))
	((#\t #\T)
	 (qplot-behavior-tree initials max-length btotal 0)
	 (setq click-p t))
	((#\a #\A)
	 (qplot-new-behavior)
	 (viewer-header parm initials btotal)
	 (format *Qsim-Trace* "~&Plotting ...")
	 (view-behs 'all initials parm behaviors btotal lengths)
	 (format *Qsim-Trace* "  Done. ")
	 ; Line commented out because it was causing an additional page to be printed.
	 ;(when (image-to-postscript-p) (ps-end-image))
	 )
	((#\s #\S)
	 (let ((list (get-list-of-bnums initials max-length btotal)))
	   (qplot-new-behavior)
	   (viewer-header parm initials btotal)
	   (format *Qsim-Trace* "~&Plotting ...")
	   (view-behs 'given initials parm behaviors btotal lengths list)
	   (format *Qsim-Trace* "  Done. ")
	   ; Line commented out because it was causing an additional page to be printed.
	   ;(when (image-to-postscript-p) (ps-end-image))
	   ))
	((#\i #\I)
	 (case *viewer-mode*
	   (var-slice
	    (format *Qsim-Report* "~&The ~a command is not applicable in the Var-Slice-Viewer." cmd))
	   (phase-slice
	    (setq *time-points-only* (not *time-points-only*))
	    (format *Qsim-Report* "~&Time intervals are currently~a plotted."
		    (if *time-points-only* " not" ""))))))
      finally (return cmd))))

(defconstant viewer-cmd-list-1 '(#\A #\a #\S #\s #\P #\p #\L #\l #\T #\t #\Q #\q #\I #\i))
(defconstant viewer-cmd-list-2 '(#\A #\a #\S #\s #\P #\p #\L #\l #\C #\c #\Q #\q #\I #\i))
(defconstant viewer-cmd-list-3 '(#\A #\a #\S #\s #\V #\v #\L #\l #\T #\t #\Q #\q #\I #\i))
(defconstant viewer-cmd-list-4 '(#\A #\a #\S #\s #\V #\v #\L #\l #\C #\c #\Q #\q #\I #\i))

(defun get-viewer-cmd (btotal click-p)
  (do ((cmd (prompt-for-viewer-cmd btotal click-p) (prompt-for-viewer-cmd btotal click-p)))
      ((case *viewer-mode*
	 (phase-slice (if click-p
			  (member cmd viewer-cmd-list-2)
			  (member cmd viewer-cmd-list-1)))
	 (var-slice (if click-p
			(member cmd viewer-cmd-list-4)
			(member cmd viewer-cmd-list-3))))
       cmd)
    (format *qsim-report* "~&~a is an invalid command." cmd)))


(defun PROMPT-FOR-VIEWER-CMD (btotal click-p)
  (format *Qsim-Report*
	  "~&Plot: A=All ~A behaviors, S=Specific behaviors,~A~A L=change Layout, Q=Quit: "
	  (if click-p "" btotal)
	  (case *viewer-mode*
	    (phase-slice " P=another Phase plane,")
	    (var-slice   " V=another Variable,"))
	  (if click-p
	      #+Symbolics " C=Click for behavior," #-Symbolics ""
	      " T=behavior Tree,"))
  (clear-input)
  #-:ccl
  (read-char)
  #+:ccl
  (convert-display-input (read)))


;; Changes layout of Viewer plots.

(defun change-viewer-layout ()
  #+:lispm (change-viewer-layout-parms)
  #+:lispm (compute-rectangular-layout viewer-rows viewer-cols)
  #-:lispm (format *Qsim-Trace* "~%This capability not implemented on a non-lisp machine."))

(defun viewer-header (parm initials btotal)
  (let ((text (qde-text (state-qde (car initials))))
	(plot-type (case *viewer-mode*
		     (var-slice "behaviors of parameter")
		     (phase-slice "phase diagrams of parameters"))))
    (loop for string in (list (format nil "Plotting ~a ~a for (structure): ~a."
	       			      plot-type parm text)
			      (format nil "Simulation from ~a complete initializations."
				      (length initials))
			      (format nil "A total of ~a behaviors."
				      btotal))
	  for y from 25 by 14
	  do (qplot-string string 15 y))))



;; Plot behaviors.  Both `all' and `given' option.

(defun VIEW-BEHS (option initials parm behaviors btotal lengths &optional bnum-list)
  (let* ((ps-new-page-offset 0)
	 (normal-exit t)
	 ;; added DJC 5/1/92 to truncate the behavior if this is set
	 (truncate? (display-block-truncate? (find-display-block (car behaviors))))
	 (behs (if truncate? (truncate-behaviors behaviors truncate?)
		   behaviors)))
    (do ((ypos tmargin (+ ypos *ysize* ysep))
	 (rows 0 (1+ rows))
	 (remainder (case option
		      (all 1)
		      (given bnum-list)))
	 row-of-bnums)
	((case option
	   (all (> remainder btotal))
	   (given (null remainder))))
      (when (and (> rows 0) (zerop (mod rows viewer-rows)))
	(qplot-end-display)
	(unless (continue-plotting?)
	  (setf normal-exit nil)
	  (return #\q))
	(when (image-to-postscript-p)
	  (ps-new-behavior)
	  (with-plotting-to-screen-inhibited (viewer-header parm initials btotal)))
	; Symbolics has dynamic Lisp Listener window. Others default to static window.
	#+symbolics (let ((oldpos (- yscreen bmargin)))
		      (set-cursor-pos 0 oldpos)
		      (clear-line)
		      (set-viewport 0 (- ypos tmargin))
		      (set-cursor-pos 0 (- (+ ypos (- yscreen bmargin)) tmargin))
		      (setq oldpos (- (+ ypos (- yscreen bmargin)) tmargin))
		      (format *QSIM-Trace* "~%Plotting ..."))
	#-symbolics (with-plotting-to-postscript-inhibited
		      (qplot-new-behavior)
		      (viewer-header parm initials btotal)
		      (format *QSIM-Trace* "~&Plotting ..."))
	(setq ps-new-page-offset (- ypos tmargin)))
      (case option
	(all (setq row-of-bnums (make-row-of-bnums remainder btotal viewer-cols)
		   remainder (+ remainder viewer-cols)))
	(given (setq row-of-bnums (firstn viewer-cols remainder)
		     remainder (nthcdr viewer-cols remainder))))
      ; Only the Symbolics screen (with dynamic Lisp Listener window) is plotted on
      ; continuously.  Other plots are broken up into a screenful at a time.
      (with-plotting-to-postscript-inhibited
	(view-a-row parm behs row-of-bnums lengths
		    #+symbolics ypos 
		    #-symbolics (- ypos ps-new-page-offset)))
      (with-plotting-to-screen-inhibited
	(view-a-row parm behs row-of-bnums lengths (- ypos ps-new-page-offset))))
    (when normal-exit (qplot-end-display))))                                                ;;; added DJC porting to the suns to get
                                                                                            ;;; the variable slice viewer to work


(defun continue-plotting? ()
  (format *QSIM-Trace*
	  " More plots to be displayed.  C=Continue Plotting, Q=Quit: ")
  (clear-input)                                                    
  ;; Modified because Ilisp does not handle read-char correctly  DJC 7jun92
  (loop for cmd = (read)
	do (case cmd
	     ((Q) (return nil))
	     ((C) (return t))
	     (otherwise
	      (format *Qsim-Trace* "~&Invalid entry. Try Again: ")))))




;; Plots one row.

(defun view-a-row (parm behaviors row-of-behs lengths ypos)
  (do ((row row-of-behs (cdr row))
       (xpos lmargin (+ xpos *xsize* xsep)))
      ((null row))
    (when (car row)
      (let* ((beh (nth (1- (car row)) behaviors))
	     (init-num (determine-init-num (car row) lengths))
	     (label (format nil "beh ~a (init ~a) " (car row) init-num)))
	(case *viewer-mode*
	  (var-slice (plot-parameter parm *reference-states* beh xpos ypos *xsize* *ysize*
			(allocate-h-axis *reference-states* beh *xsize*) :label label))
	  (phase-slice (plot-one-phase-diagram parm beh xpos ypos *xsize* *ysize* label)))))))



;; Plots one phase diagram.

(defun plot-one-phase-diagram (parms beh xpos ypos xsize ysize label)
  (let* ((x (car parms))
	 (y (cadr parms))
	 (size 3)
	 (*detailed-printing* nil)
	 (x-lists (collect-intervals-and-plot-list x beh))
	 (x-plot-list (car x-lists))
	 (x-intervals (cadr x-lists))
	 (y-lists (collect-intervals-and-plot-list y beh))
	 (y-plot-list (car y-lists))
	 (y-intervals (cadr y-lists))
	 (xalloc (allocate-x-axis x-intervals (qspace-from-behavior x beh) xsize))
	 (yalloc (allocate-y-axis y-intervals (qspace-from-behavior y beh) ysize))
	 (xcoords (determine-plot-coords xalloc x-plot-list))
	 (ycoords (determine-plot-coords yalloc y-plot-list))
	 (plot-p nil)
	 stars)
    (do* ((xlist xcoords (cdr xlist))
	  (ylist ycoords (cdr ylist))
	  (ox (+ xpos (car xlist)) (+ xpos (car xlist)))
	  (oy (+ ypos (car ylist)) (+ ypos (car ylist)))
	  (prev-circle-x nil)
	  (prev-circle-y nil))
	 ((or (null (cdr xlist)) (null (cdr ylist)))
	  (when (and (not (eql (car xlist) -1000)) (not (eql (car ylist) -1000)))
	    (setq plot-p t)
	    (qplot-state (car (last beh)) ox oy))
	  (setq stars (list ox oy)))
      (when (and (not (eql (car xlist) -1000)) (not (eql (car ylist) -1000)))
	(setq plot-p t)
	(when prev-circle-x (connect-with-dots prev-circle-x prev-circle-y ox oy))
	(qplot-circle ox oy size :filled t)
	(cond ((and (not (eql (cadr xlist) -1000)) (not (eql (cadr ylist) -1000)))
	       (draw-arrows ox oy (+ xpos (cadr xlist)) (+ ypos (cadr ylist)) 10)
	       (setq prev-circle-x nil) (setq prev-circle-y nil))
	      (t (setq prev-circle-x ox) (setq prev-circle-y oy)))))
    (when plot-p
      (qplot-box xpos ypos xsize (- ysize labelspace) :thickness 2)
      (plot-x-axis xalloc xpos ypos ysize)
      (plot-y-axis yalloc xpos ypos xsize)
      (plot-grid xpos ypos xalloc yalloc)
      (qplot-box-label xpos (+ ypos (- ysize labelspace) 15) label)
      (when (quiescent-p  (car (last beh)))
	(qplot-star-lines xpos ypos xsize ysize (car stars) (cadr stars))))))

;; Plotting dash lines to indicate quiescence.

(defun qplot-star-lines (xloc yloc xsize ysize xstar ystar)
  (qplot-dashed-line xstar
		     (+ yloc (* tshare *VshareSize*))
		     xstar
		     (+ (+ yloc (* tshare *VshareSize*))
			(- ysize labelspace (* *VshareSize* (+ tshare bshare)))))
  (qplot-dashed-line (+ xloc (* lshare Hshare))
		     ystar
		     (+ (+ xloc (* lshare Hshare)) 
			(- xsize (* Hshare (+ lshare rshare))))
		     ystar))



;; Create two lists, Intervals (list of intervals w/ count of consecutive occurrences) for
;; determining axis allocations and Plot-List (list of qvals to plot) for
;; creating list of plot coordinates for plotting.  Spacing between adjacent landmarks
;; may not be the same for all pairs of adjacent landmarks.  This depends on what the longest
;; time (in terms of the number of consecutive time points) spent in an interval is. (And
;; in turn on which pair of adjacent landmarks the interval is mapped to.)

(defun collect-intervals-and-plot-list (parm beh)
  (let ((qspace (qspace-from-behavior parm beh))
	(plot-list nil)
	(intervals nil))
    (do ((L beh (cdr L))
	 (prev-qval nil)
	 (n 0)
	 qval
	 plot-pos
	 occurred)
	((null L)
	 (when (and plot-pos prev-qval (listp (qmag prev-qval)))
	   (setq plot-list (append plot-list (list (list plot-pos (qdir prev-qval) n))))
	   (if (null (setq occurred (assoc plot-pos intervals :test #'equal)))
	       (setq intervals (append intervals (list (list plot-pos (qdir prev-qval) n))))
	       (when (> n (caddr occurred))
		 (rplacd (cdr occurred) (list n)))))
	 (list plot-list intervals))
      (when (or (qpointp (state-time (car L))) (not *time-points-only*))
	(if (null (setq qval (alookup parm (state-qvalues (car L)))))
	    (setq plot-list (append plot-list (list nil)))
	    (cond ((or (equal qval prev-qval) (and (> n 0) (listp (qmag qval))))
		   (if (atom (qmag qval))
		       (setq plot-list (append plot-list (list (qmag qval))))
		       (case (qdir prev-qval)
			 (ign (case (qdir qval)
				(ign (setq n (1+ n)))
				(t (format *QSIM-Trace*
					   "Collect-Intervals-And-Plot-List error"))))
			 ((inc dec)
			  (if (eq (qdir prev-qval) (qdir qval))
			      (setq n (1+ n))
			      (case (qdir qval)
				(std (setq n (1+ n))
				     (setq plot-list
					   (append plot-list
						   (list (list plot-pos (qdir prev-qval) n))))
				     (if (null (setq occurred
						     (assoc plot-pos intervals :test #'equal)))
					 (setq intervals
					       (append intervals
						       (list (list plot-pos (qdir prev-qval) n))))
					 (when (> n (caddr occurred))
					   (rplacd (cdr occurred) (list n))))
				     (setq n 1))
				(t (format *QSIM-Trace*
					   "Collect-Intervals-And-Plot-List error")))))
			 (std (case (qdir qval)
				(std (setq n (1+ n)))
				((inc dec)
				 (setq plot-list (append plot-list (list (list plot-pos (qdir prev-qval) n))))
				 (when (null (assoc plot-pos intervals :test #'equal))
				   (setq intervals
					 (append intervals (list (list plot-pos (qdir prev-qval) 1)))))
				 (setq n 2))
				(t (format *QSIM-Trace*
					   "Collect-Intervals-And-Plot-List error"))))))
		   (setq prev-qval qval))
		  (t
		   (cond ((atom (qmag qval))
			  (when (and plot-pos prev-qval (listp (qmag prev-qval)))
			    (setq plot-list (append plot-list (list (list plot-pos (qdir prev-qval) n))))
			    (if (null (setq occurred (assoc plot-pos intervals :test #'equal)))
				(setq intervals (append intervals (list (list plot-pos (qdir prev-qval) n))))
				(when (> n (caddr occurred))
				  (rplacd (cdr occurred) (list n))))
			    (setq n 0))
			  (setq plot-list (append plot-list (list (qmag qval)))))
			 (t (setq plot-pos (determine-plot-pos qval qspace))
			    (setq n 1)))
		   (setq prev-qval qval))))))))

;; For an interval qmag whose end points are non-adjacent landmarks, determine between which
;; pair of landmarks to plot the qmag.

(defun determine-plot-pos (qval qspace)
  (let ((qmag (qmag qval))
	(qdir (qdir qval)))
    (case qdir
      (inc (let ((beg-lmk (car qmag)))
	     (do ((L qspace (cdr L)))
		 ((null (cdr L)) (format *QSIM-Trace* "~%Error in qmag."))
	       (when (eql (car L) beg-lmk)
		 (return (list (car L) (cadr L)))))))
      (dec (let ((end-lmk (cadr qmag)))
	     (do ((L qspace (cdr L)))
		 ((null (cdr L)) (format *QSIM-Trace* "~%Error in qmag."))
	       (when (lmark-equal (cadr L) end-lmk)
		 (return (list (car L) (cadr L)))))))
      (t (let ((pos (truncate (+ (position (car qmag) qspace)
				 (position (cadr qmag) qspace))
			      2)))
	   (list (nth pos qspace) (nth (1+ pos) qspace)))))))



;; Determine positions of landmarks.

(defun allocate-x-axis (x-intervals xqspace xsize)
  (let ((coords nil)
	(interval-widths (determine-interval-widths x-intervals xqspace)))
    (setq Hshare (/ xsize (+ lshare (float (* 2 (apply #'+ interval-widths))) rshare)))
    (setq vtickpoint (- xsize (* htickprop Hshare)))
    (setq vlabelpoint (- xsize (* hlabelprop Hshare)))
    (do* ((L xqspace (cdr L))
	  (W (append '(0) interval-widths '(0)) (cdr W))
	  (x (* Hshare lshare)
	     (+ x (* 2.0 Hshare (car W)))))
	 ((null L))
      (setq coords (cons (list (car L) (round x))
			 coords)))
    coords))

(defun allocate-y-axis (y-intervals yqspace ysize)
  (let ((coords nil)
	(interval-widths (determine-interval-widths y-intervals yqspace)))
    (setq *VshareSize* (/ (- ysize labelspace) (+ tshare (float (* 2 (apply #'+ interval-widths))) bshare)))
    (setq htickpoint (- ysize labelspace (* htickprop *VshareSize*)))
    (setq hlabelpoint (- ysize labelspace (* hlabelprop *VshareSize*)))
    (do* ((L (reverse yqspace) (cdr L))
	  (W (append '(0) (reverse interval-widths) '(0)) (cdr W))
	  (y (* *VshareSize* tshare)
	     (+ y (* 2.0 *VshareSize* (car W)))))
	 ((null L))
      (setq coords (cons (list (car L) (round y))
			 coords)))
    (reverse coords)))

;; Determine spacing between intervals.

(defun determine-interval-widths (intervals qspace)
  (let ((widths nil))
    (do* ((L qspace (cdr L))
	  (int (list (car L) (cadr L))
	       (list (car L) (cadr L)))
	  occurred)
	 ((null (cdr L)) widths)
      (if (setq occurred (assoc int intervals :test #'equal))
	  (if (or (eq 'ign (cadr occurred)) (eq 'std (cadr occurred)))
	      (setq widths (append widths '(1)))
	      (setq widths (append widths (list (caddr occurred)))))
	  (setq widths (append widths '(1)))))))

;; Return list of coordinates for plotting.

(defun determine-plot-coords (alloc plot-list)
  (let ((coords nil) (offset 0))
    (do ((L plot-list (cdr L)))
	((null L) coords)
      (if (car L)
	  (if (atom (car L))
	      (setq coords (append coords (cdr (assoc (car L) alloc :test #'lmark-equal))))
	      (let* ((pts (caddar L))
		     (beg-coord (cadr (assoc (caaar L) alloc :test #'lmark-equal)))
		     (end-coord (cadr (assoc (cadaar L) alloc :test #'lmark-equal)))
		     (pt-width (/ (- end-coord beg-coord) (1+ pts))))
		(case (cadar L)
		  (inc (do ((N (- pts offset) (1- N))
			    (pt-coord (+ beg-coord (* (1+ offset) pt-width)) (+ pt-coord pt-width)))
			   ((zerop N))
			 (setq coords (append coords (list (round pt-coord)))))
		       (setq offset 0)
		       (when (and (cdr L) (listp (cadr L)) (equal (caar L) (caadr L)) (eq (cadadr L) 'std))
			 (setq offset 1)))
		  (dec (do ((N (- pts offset) (1- N))
			    (pt-coord (- end-coord (* (1+ offset) pt-width)) (- pt-coord pt-width)))
			   ((zerop N))
			 (setq coords (append coords (list (round pt-coord)))))
		       (setq offset 0)
		       (when (and (cdr L) (listp (cadr L)) (equal (caar L) (caadr L)) (eq (cadadr L) 'std))
			 (setq offset 1)))
		  (t (do ((N (- pts offset) (1- N))
			  (pt-coord (round (/ (+ beg-coord end-coord) 2))))
			 ((zerop N))
		       (setq coords (append coords (list pt-coord))))
		     (setq offset 0)
		     (when (and (eq (cadar L) 'std) (cdr L) (listp (cadr L)) (equal (caar L) (caadr L))
				(or (eq (cadadr L) 'inc) (eq (cadadr L) 'dec)))
		       (setq offset 1))))))
	  (setq coords (append coords (list -1000)))))))



(defun plot-x-axis (xalloc xloc yloc ysize)
  (let ((zeropos (lookup 0 xalloc)))
    (when zeropos
      (qplot-vline (+ xloc zeropos)
		   (+ yloc (* tshare *VshareSize*))
		   (- ysize labelspace (* *VshareSize* (+ tshare bshare))))))
  (do ((ytick (+ yloc htickpoint))
       (ylm (+ yloc hlabelpoint))
       (pairs xalloc (cdr pairs)))
      ((null pairs))
    (let ((xmark (+ xloc (cadr (car pairs)))))
      (qplot-vline xmark ytick tick-length)
      (qplot-vline  xmark yloc tick-length)
      (qplot-axis-label (- xmark 5) (+ 5 ylm) (car (car pairs))))))

(defun plot-y-axis (yalloc xloc yloc xsize)
  (let ((zeropos (lookup 0 yalloc)))
    (when zeropos
      (qplot-hline (+ xloc (* lshare Hshare))
		   (+ yloc zeropos)
		   (- xsize (* Hshare (+ lshare rshare))))))
  (do ((xtick (+ xloc vtickpoint))
       (xlm (+ xloc vlabelpoint))
       (pairs yalloc (cdr pairs)))
      ((null pairs))
    (let ((ymark (+ yloc (cadr (car pairs)))))
      (qplot-hline xtick ymark tick-length)
      (qplot-hline xloc ymark tick-length)
      (qplot-axis-label (+ 5 xlm) (+ ymark 5) (car (car pairs))))))

(defun plot-grid (xpos ypos xalloc yalloc)
  (mapc #'(lambda (x)
	    (mapc #'(lambda (y) (qplot-dot (+ (cadr x) xpos) (+ (cadr y) ypos))) yalloc))
	xalloc))

;; Connect two phase points with arrows.

(defun draw-arrows (ox oy x y len)
  (let* ((n (round (/ (+ (sqrt (+ (* (- ox x) (- ox x)) (* (- oy y) (- oy y)))) len) len 2)))
	 (dx (/ (- x ox) (- (+ n n) 1)))
	 (dy (/ (- y oy) (- (+ n n) 1))))
    (do ((arrows n (- arrows 1))
	 (bx ox (+ bx dx dx))
	 (by oy (+ by dy dy)))
	((= arrows 0))
      (qplot-vector (round bx) (round by) (round (+ bx dx)) (round (+ by dy))
		    :arrow-head-length 5
		    :arrow-base-width 4
		    :filled t))))



;;; The rest are supporting functions for both viewers.

;; Gets a list of behaviors specified by the user.
;; Selecting from the beh tree available only for the Symbolics.

(defun get-list-of-bnums (inits max-len btotal)
  #+symbolics
  (let ((option (get-specifying-option)))
    (if (or (eql option #\n) (eql option #\N))
	(make-list-of-bnums (specify-behs-by-bnum btotal))
	(select-behs-from-beh-tree inits max-len btotal)))
  #-symbolics (declare (ignore inits max-len))
  #-symbolics
  (make-list-of-bnums (specify-behs-by-bnum btotal)))

;; Makes a flat list of behavior numbers out of prompted list.

(defun make-list-of-bnums (input-list)
  (cond ((null input-list) nil)
	((listp (car input-list))
	 (append (return-num-sequence (caar input-list) (cadar input-list))
		 (make-list-of-bnums (cdr input-list))))
	(t
	 (cons (car input-list) (make-list-of-bnums (cdr input-list))))))

;; Makes a list of behavior numbers for plotting one row.

(defun make-row-of-bnums (bnum btotal per-row)
  (let ((list nil) (maxnum (+ bnum (1- per-row))))
    (do ((num bnum (1+ num)))
	((or (> num maxnum) (> num btotal)) list)
      (setq list (append list (list num))))))

(defun beh-number-p (num btotal)
  (and (numberp num)
       (>= num 1)
       (<= num btotal)))

(defun determine-init-num (beh-num lengths)
  (do* ((num 1 (1+ num))
	(offset beh-num (- offset (car list)))
	(list lengths (cdr list)))
       ((or (<= offset (car list)) (null list)) num)))

(defun firstn (n list)
  (cond ((or (<= n 0) (null list)) nil)
	(t (cons (car list) (firstn (1- n) (cdr list))))))



(defun get-parms (initial)
  (case *viewer-mode*
    (var-slice (get-var-slice-parm initial))
    (phase-slice (get-phase-slice-parms initial))))

(defun get-var-slice-parm (initial)
  (declare (ignore initial))
  (let ((var-list (get-var-list-from-qdes)))
    (do ((parm (prompt-for-var-slice-parm) (prompt-for-var-slice-parm)))
	((or (member parm var-list) (null parm)) parm)
      (format *QSIM-Trace* "~&~a is not a valid parameter.  The list of parameters is: ~a."
	      parm var-list))))

(defun prompt-for-var-slice-parm ()
  (format *qsim-report* "~&Parameter to var-slice (NIL to quit), ending with <space> or <CR>: ")
  (read))

(defun get-phase-slice-parms (initial)
  (declare (ignore initial))
  (let ((var-list (get-var-list-from-qdes)))
    (do ((parms (prompt-for-phase-slice-parms) (prompt-for-phase-slice-parms)))
	((or (null parms)
	     (cond ((or (not (listp parms)) (not (eql 2 (length parms))))
		    (format *qsim-report* "~&Input is not a list of length two.")
		    nil)
		   ((and (member (car parms) var-list) (member (cadr parms) var-list))
		    t)
		   (t
		    (format *qsim-report* "~&Invalid parameter in ~a.  The list of parameters is: ~a."
			    parms var-list)
		    nil)))
	 parms))))

(defun prompt-for-phase-slice-parms ()
  (format *qsim-report* "~&Phase plane (list of two parameters) to plot (NIL to quit): ")
  (read))

(defun get-specifying-option ()
  (do ((option (prompt-for-specifying-option) (prompt-for-specifying-option)))
      ((member option (list #\n #\N #\t #\T)) option)
    (format *qsim-report* "~a is not a valid option." option)))

(defun prompt-for-specifying-option ()
  (format *qsim-report* "~&Specify behaviors by numbers (N) or through the tree (T)? ")
  (read-char))

(defun specify-behs-by-bnum (btotal)
  (do ((list (prompt-for-beh-list btotal) (prompt-for-beh-list btotal)))
      ((cond ((not (listp list)) (format *qsim-report* "~&Input is not a list.") nil)
	     ((do ((L list (cdr L)))
		  ((cond ((null L) t)
			 ((cond ((or (null (car L)) (beh-number-p (car L) btotal)) nil)
			  	((listp (car L))
				 (cond ((and (eql (length (car L)) 2)
					     (beh-number-p (caar L) btotal)
					     (beh-number-p (cadar L) btotal)
					     (>= (cadar L) (caar L)))
					nil)
				       (t
					(format *qsim-report* "~&Sublist ~a does not specify a valid ~
                                                               sequence of behavior numbers." (car L))
					(return nil))))
				(t
				 (format *qsim-report* "~&~a is neither NIL, a valid behavior number, ~
                                                        nor a sublist." (car L))
				 (return nil)))))
		   t))
	      t))
       list)))

(defun prompt-for-beh-list (btotal)
  (format *qsim-report* "~&~%Specify behaviors in the form of a list of behavior numbers. ~
    ~%Valid behavior numbers are from 1 thru ~a. ~
    ~%Specify a sequence in the form of a sublist: (beginning-beh-num ending-beh-num). ~
    ~%For example, ((4 6) 2) displays behaviors 4, 5, 6 and 2. ~
    ~%You can use NILs in the list to format the display the way you like it. ~
    ~%Enter list: " btotal)
  (read))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;     COMPARISON-VIEWER CODE
;;;
;;;  The following code allows the user to compare the time plots of various
;;;  behaviors simultaneously on the screen.  The user identifies the behaviors and
;;;  the variables of interest.  The rows are used for multiple behaviors while the
;;;  columns on the display are for the different variables.
;;;
;;;  E.G.  If the user requests behaviors 1 3 and 4 are compared for variables A and B
;;;        they will be diplayed in the following format:
;;;
;;;        Beh 1 Var A        Beh 1 Var B
;;;        Beh 2 Var A        Beh 2 Var B
;;;        Beh 3 Var A        Beh 3 Var A
;;;
;;;  The user will be required to identify the behaviors of interest.  He will then have 
;;;  the option of letting the variables of interest be selected automatically.  If this
;;;  selection is made, then the behaviors will be analyzed and the states following the
;;;  various branch points will be identified.  These states will be compared to determine
;;;  which variables are different at these branch points.  Thus, these variables are the
;;;  ones that led to the branch.
;;; 
;;;  DJC  09/12/91

;; Default layouts.
(defvar *compare-view-rows* 4)
(defvar *compare-view-cols* 3)
(defparameter *comparison-branch-time-points* nil)

;;; REturns t if all of the elements in the list are equal.

(defmacro all-equal (list &key (test #'equal))
  `(every #'(lambda (elem)
	      (funcall ,test elem (car ,list)))
	  (cdr ,list)))

(defun cp (&optional (init-state *initial-state*))
  (device-interface
    (comparison-viewer (get-list-of-initial-states init-state :complete-partial-state nil))))


(defun comparison-viewer (initials)
  (let* ((display-block (find-display-block initials))
	 (old-truncate? (display-block-truncate? display-block))
	 (selection-method (get-beh-selection-method)))
    (multiple-value-bind (selected-behs bnums-or-path)
	(select-behaviors initials selection-method)
      (when selected-behs
	(let ((*compare-view-rows* (case (length selected-behs)
				     (2 2)
				     ((3 5 6 9) 3)
				     (otherwise 4)))
	      (params (get-compare-params initials selected-behs
					  selection-method display-block)))
	  (when params
	    (compare-plot params selected-behs bnums-or-path
			  selection-method initials
			  (display-block-truncate? display-block))))))
    (setf (display-block-truncate? display-block)
	  old-truncate?)))


;;; This function is the plotting function.  It will receive a list of parameters,
;;; a set of behaviors numbers and a set of behaviors.  It will plot
;;; the behaviors specified for the parameters in PARAMS. 

(defun compare-plot (params behaviors bnums-or-path selection-method initials truncate?)
  (let* ((num-vars (length params))
	 (num-behs (length behaviors))
	 (total-num-rows (* (ceiling num-vars *compare-view-cols*) num-behs))
	 (*xsize* (- (round (/ (+ (- xscreen lmargin rmargin) xsep)
			       *compare-view-cols*))
		     xsep))
	 (*ysize* (- (round (/ (+ (- yscreen tmargin bmargin) ysep)
			       *compare-view-rows*))
		     ysep)))
    (qplot-new-behavior)
    (comparison-header params bnums-or-path selection-method initials)
    (do* ((ypos tmargin (+ ypos *ysize* ysep))
	  (total-rows-displayed 0 (1+ total-rows-displayed))
	  (display-row 0 (mod (1+ display-row) *compare-view-rows*))
	  (beh-counter 0 (mod (1+ beh-counter) num-behs))
	  (beh-num-or-path (if (equal selection-method :bnums)
			       (nth beh-counter bnums-or-path)
			       (append bnums-or-path (list beh-counter)))
			   (if (equal selection-method :bnums)
			       (nth beh-counter bnums-or-path)
			       (append bnums-or-path (list beh-counter))))
	  (cur-var-num 0 (if (zerop beh-counter)
			     (+ cur-var-num *compare-view-cols*)
			     cur-var-num)))
	 ((equal total-num-rows total-rows-displayed))
      (when (and (> total-rows-displayed 0) (or (zerop beh-counter) (zerop display-row)))
	(qplot-end-display)
	(when (image-to-screen-p)
	  (unless (continue-plotting?)
	    (return #\q)))
	(qplot-new-behavior)
	(comparison-header params bnums-or-path selection-method initials)
	(setq ypos tmargin)
	(setq display-row 0))
      (loop as var-num-plotting from cur-var-num to num-vars
	    for xpos = lmargin then (+ xpos *xsize* xsep)
	    for full-beh  = (nth beh-counter behaviors)
	    for beh = (if truncate? (truncate-behavior full-beh truncate?) full-beh)
	    for var = (nth var-num-plotting params)
	    for label = (if (equal selection-method :bnums)
			    (format nil "Beh ~a  Var ~a" beh-num-or-path (get-box-label var beh))
			    (format nil "Path ~a Var ~a" beh-num-or-path (get-box-label var beh)))
	    when (eq var-num-plotting (+ cur-var-num *compare-view-cols*))
	    do (return)
	    do (plot-parameter var *reference-states* beh
			       xpos ypos *xsize* *ysize*
			       (allocate-h-axis *reference-states* beh *xsize*) :label label)))
    (qplot-end-display)))

;;;  Prints the header for the comparison viewer.

(defun comparison-header (params bnums-or-path selection-method initials)
  (let ((text (qde-text (state-qde (car initials)))))
    (loop for string in (list (format nil "Plotting a behvaior comparison for (structure): ~a."
	       			      text)
			      (if (equal selection-method :bnums)
				  (format nil "The variables ~a are compared in behaviors ~a"
					  params bnums-or-path)
				  (format nil "The variables ~a are compared in the path ~a"
					  params bnums-or-path))
			      (format nil "Simulation from ~a complete initializations."
				      (length initials)))
	  for y from 25 by 14
	  do (qplot-string string 15 y))))



;;;  This will get the parameters to be compared.  If auto is chosen then these
;;;  parameters will be automatically determined and the user will be asked
;;;  to confirm their selection.

(defun get-compare-params (initials behaviors selection-method
				    &optional (display-block (find-display-block (car initials))))
  (let ((var-list (get-var-list-from-qdes))
	(*comparison-branch-time-points* nil)) ; time points of branches
    (when (equal selection-method :branch-path)
      (format *qsim-report* "~%Time plots are automatically truncated at the branch point.~%"))
    (let ((menu (append '((Auto "Automatically Determine Parameters")
			  (S    "Specify Parameters")
			  (All  "Select All Parameters"))
			(when (equal selection-method :bnums)
			  '((TR   "Select Truncation Point")
			    (L    "Truncate at latest interesting time point")))
			'((D    "Display Comparison"))))
	  params)
      (loop for cmd = (general-menu menu :skip-lines t)
	    do (case (car cmd)
		 ((Auto)
		  (setf params (auto-calc-params behaviors))
		  (format *qsim-trace* "~%The parameters with different qvalues are: ~%   ~a"
			  params)
		  (format *qsim-trace* "~%The interesting time-points are: ~a~%"
			  *comparison-branch-time-points*))
		 ((S) (setf params (get-param-list var-list))
		  (format *qsim-report* "~%"))
		 ((All) (setf params var-list)
		  (format *qsim-report* "~%The parameters chosen are: ~%  ~a~%" params))
		 ((TR) (setf (display-block-truncate? display-block)
			     (get-trunc-point))
		  (format *qsim-report* "~%The display will be truncated at ~a~%"
			  (display-block-truncate? display-block)))
		 ((L) (setf (display-block-truncate? display-block)
			    (loop for time in *comparison-branch-time-points*
				  for max = (car *comparison-branch-time-points*) 
				  do (if (later-time? time max)
					 (return time)
					 (return  max))))
		  (format *qsim-report* "~%The display will be truncated at ~a~%"
			  (display-block-truncate? display-block)))
		 ((D) (return params))
		 (otherwise (return nil)))))))
		  
(defun get-param-list (valid-params)
  (loop for params = (prompt-param-list)
	if (loop for param in params
		 do (unless (member param valid-params)
		      (format *qsim-report* "~&~a is an invalid parameter.  The list of ~
                               paramters is: ~% ~a" param valid-params)
		      (return nil))
		 finally (return t))
	return params))

(defun prompt-param-list ()
  (format *qsim-report* "~%Enter the list of parameters (NIL to quit): ")
  (read))
  

;;;  This function will receive an initial state and the behaviors to be compared.  It
;;;  compare these behaviors and determine which variables cause the branching
;;;  in the selected behaviors and will return this list of variables.

(defun auto-calc-params (selected-behs)
  (return-diff-vars selected-behs))


;;;   Returns the parameters which cause the branching in the behaviors.

(defun return-diff-vars (behs)
  (remove-duplicates
    (mapcan #'compare-states-for-diff-vars
	    (partition-by-diff-states behs))
    :test #'equal))


;;; This function receives a list of behaviors.  It returns a list of sets of states.
;;; The states which are included in each list are the states which follow branching
;;; points in these behaviors.  The function will step through the behaviors
;;; in unison.  When the behaviors are no longer in agreement (i.e. a branch occurs)
;;; then it will select the different states that occur at this branch point,
;;; place these states in a set to be returned, and then group the behaviors
;;; by these states.  The function will then be called recursively.

(defun partition-by-diff-states (behs)
  (when (and behs (cdr behs))
    (let ((first-states (mapcar #'car behs)))
      (cond ((every #'(lambda (elem)            ; if all of the elements in first-states are equal
			(equal elem (car first-states)))
		    (cdr first-states))
	     (partition-by-diff-states (mapcar #'cdr behs)))
	    (t (let* ((diff-states (remove-duplicates first-states :test #'equal))
		      (branch-time-point (qmag (state-time (car diff-states)))))
		 (push branch-time-point *comparison-branch-time-points*)
		 (cons diff-states
		       (mapcan #'partition-by-diff-states
			       (mapcar #'(lambda (state)
					   (remove-if-not #'(lambda (beh)
							      (equal state (car beh)))
							  behs))
				       diff-states)))))))))




(defun compare-states-for-diff-vars (states)
  "Receives a list of states and will return the vars which are different"
  (let ((qvals (mapcar #'(lambda (state)
			   (cdr (state-qvalues state)))
		       states))
	diff-vars)
    (do* ((rem-qvals qvals (mapcar #'cdr rem-qvals)))
	 ((null (car rem-qvals)))
      (push
	(compare-qvals (mapcar #'car rem-qvals))
	diff-vars))
    (remove-nil (nreverse diff-vars))))


(defun compare-qvals (qvals)
  "Receives a list of qvals and will return t if they are equivalent"
  (unless (every #'(lambda (qval)
		     (qval-equivalent (cdar qvals) (cdr qval)))
		 (cdr qvals))
    (caar qvals)))


(defun prompt-for-compare-params ()
  (format *qsim-report* "~&Enter the paramters that you are interested in comparing in list format~% ~
                         (NIL to quit ~% ALL for all of the paramters ~% AUTO for automatic selection): ")
  (read))

;;;
;;;  BRANCH PATH CODE
;;;
;;;  This section of code is used to identify a state or a set of behaviors in
;;;  a behavior tree through selecting which branch number should be
;;;  traversed in the tree.  The user enters a list of numbers.  Each
;;;  number identifies which branch should be traversed at the next
;;;  branch point in the tree.  This technique can be used to identify
;;;  either a state or a set of behaviors.  When a set of behaviors
;;;  are identified, they can eitehr be truncated after the last
;;;  identified branching point, or they can be complete behaviors in
;;;  which case all of the behaviors which follow are selected.  Thus,
;;;  this can also be used to select a subtree.
;;;
;;;  Below is a more detailed description of this method of
;;;  identifying states.
;;;
;;;		IDENTIFYING STATES AND BEHAVIORS VIA A BRANCH PATH
;;;
;;;
;;; Recently I have been developing portions of code that require a user
;;; to identify either a state within the tree or a subtree of the tree.
;;; I would like to be able to identify this without requiring the user
;;; to manually find the state name in the tree.  Below is a mechanism 
;;; which I am using to identify either a state, a subtree or a set of
;;; behaviors.
;;;
;;; This mechanism entail selecting branches from within a tree by 
;;; identifying the branch number.  The branches extending from a state
;;; are numbered starting with 1 and increasing in a downward direction.
;;; The user identifies a state by selecting which branch should be traversed 
;;; at each succesive branch state.  Thus the sequence
;;;
;;;		(1 3 2)
;;;
;;; would take the top branch at the first branch point in the tree, the third
;;; branch at the next branch point and the second point at the following branch 
;;; point.  This branch will be traversed until the next branch point is reached.
;;; Whatever state lies at this branch point is the selected state.
;;; 
;;; For example, suppose the tree is as follows:
;;; 
;;; 
;;; 
;;; 
;;; 		      	           |--S13	
;;; 		       |--S5--S10--|
;;; 		       |	   |--S14
;;; 	    |--S2--S4--|--S6
;;; 	    |	       |
;;; 	    |	       |--S7
;;; 	    |  
;;; 	    |
;;; 	S1--|		   |--S11
;;; 	    |--S3--S8--S9--|
;;; 			   |--S12
;;; 
;;; 
;;; The path (1) would select the first branch path and tracerse it to the
;;; next branching state.  This would return S4.  This denotation can also
;;; be used to identify the set of behaviors extending from S4 or the
;;; behavior prefix path which starts at S1 and terminates at S4.
;;; Below are some other examples:
;;; 
;;; 	(2 1)		S11
;;; 	()		S1
;;; 	(1 1)		S10
;;; 	(1 3)		S7
;;; 

(defun get-branch-prefix (state-or-states branch-path)
  "This function will receive a state(s) and a list of numbers.  The
list of numbers will be sequence of branch numbers which have been
selected.  It will return the behavior prefix which leads up to
the next branching state along the identified path."
  (cond ((and (null state-or-states) branch-path)
	 (format *Qsim-Report* "~%WARNING: Invalid branch path.  Remaining path ~a not used." branch-path))
	;; Hanldes the normal case in which a single initial state is passed in
	((and state-or-states
	      (atom state-or-states))
	 (let* ((state state-or-states)
		(successors (successor-states state)))
	   (cond ((and (null branch-path)
		       (cdr successors))
		  (list state))
		 ((or (null (cdr successors)))
		      (cons state
			    (get-branch-prefix (car successors) branch-path)))
		 (t (let ((selected-state (nth (1- (car branch-path))  successors)))
		      (if selected-state
			  (cons state (get-branch-prefix selected-state
							 (cdr branch-path)))
			  (progn
			    (format *Qsim-Report* "~%WARNING: Invalid branch path.  ~
                                             Remaining path ~a not used." branch-path)
			    (list state))))))))
	;; When multiple initial states are passed in and the branch path is nil, these
	;; states should be compared.  Passes back the predecssor of one of these states.
	((and (null branch-path)
	      (cdr state-or-states))
	 (list (state-predecessor (car state-or-states))))
	;; A list of initial states is passed with a non-nil branch path.  Will
	;; call this function recursively selecting the initial state identified by
	;; the branch path.
	((cdr state-or-states)
	 (let ((selected-state (if branch-path
				   (nth (1- (car branch-path)) state-or-states)
				   (car state-or-states))))
				   
	   (if selected-state
	       (get-branch-prefix selected-state
				  (cdr branch-path))
	       (format *Qsim-Report* "~%WARNING: Invalid branch path.  Remaining path ~a ~
                                      not used." branch-path))))
	;; A single state is passed in, but it is in a list format
	(state-or-states 
	 (get-branch-prefix (car state-or-states) branch-path))
	(t nil)))


(defun select-state (initials)
  "Will prompt the user for a branch path and return the last state in the path."
  (let ((path (get-branch-path)))
    (car (last (get-branch-prefix initials path)))))


(defun select-behaviors (initials selection-method &optional (complete-behs nil))
  "Allows the user to identify a set of behaviors from the behavior tree.  This
can be done either by entering a list of behavior numbers or a a branch path.  The
  desired entry type is idenitifed by the parameter SELECTION-METHOD.  This
parameter can either be :BNUMS or :BRANCH-PATH. If
the optional argument COMPLETE-BEHS is true, then when a branch path is used the
complete set of behaviors which extends from the selected branch is returned."
  (cond ((equal selection-method :bnums)
	 (let* ((behaviors (get-behaviors initials))
		(btotal (length behaviors))
		(bnums (make-list-of-bnums (specify-behs-by-bnum btotal)))
		(selected-behs (mapcar #'(lambda (bnum)
					   (nth (1- bnum) behaviors))
				       bnums)))
	   (values selected-behs bnums)))
	((equal selection-method :branch-path)
	 (let* ((branch-path (get-branch-path))
		(beh-prefix (get-branch-prefix initials branch-path))
		(successors (successor-states (car (last beh-prefix))))
		(behs-to-ret (cond ((null successors) (list beh-prefix))
				   (complete-behs
				    (let ((behaviors (get-behaviors (car (last beh-prefix)))))
				      (mapcar #'(lambda (beh)
						  (append beh-prefix beh))
					      behaviors)))
				   (t (mapcar #'(lambda (state)
						  (append beh-prefix (list state)))
					      successors)))))
	   (if (cdr behs-to-ret)
	       (values behs-to-ret branch-path)
	       (format *qsim-report* "~%Branch path too long. Only one behavior identified. ~
                                       Cannot perform behavior comparison"))))))

(defun get-beh-selection-method ()
  (let ((options `((N "Specify Behavior Numbers")
		   (B "Specify Sequence of Branch Numbers"))))
    (case (car (general-menu options :skip-lines t))
      ((N) :bnums)
      ((B) :branch-path))))

(defun get-branch-path ()
  (loop for path = (get-number-list)
	do (if (every #'(lambda (numb)
			  (and (numberp numb)
			       (not (zerop numb))))
		      path)
	       (return path)
	       (format *Qsim-Report* "~%Invalid list of branch numbers.  Each must be a ~
                                        number greater than zero."))))
(defun get-number-list ()
  (format *Qsim-Report* "~%Enter a list of numbers identifying which branch should ~
                         ~%should be chosen at each branch point. ~
                         ~%(NIL selects the first branch point): ")
  (read))


