;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-

;;; $Id: qsim-to-qgraph.lisp,v 1.11 92/06/30 16:04:35 bert Exp $

(in-package 'qsim)

;;; Change Log:
;;;  20Jan90 - Stable version to Symbolics.
;;;  18Jan91 - Cleaned up TQ3 stuff.
;;;  21Jan91 - qgraph-from-struct takes key args for sizing,
;;;            added q[23]-behavior-to-qgraph
;;;  22Jan91 - fixed bugs in q[23]-behavior-to-tqual
;;;  18Jun91 - Changed Line and TEnv to queues
;;;  26Sep91 - Revamped for folding into QSIM


;;; This file is part of the qgraph system.
;;; The qgraph system is composed of the following files :
;;;      qgraph         - The main code for qgraph.
;;;      qgraph-io      - File I/O and plotting functions
;;; In addition, the file following files provide the interface to qsim :
;;;      numeric-plot   - Interface to the qsim display handler
;;;      qsim-to-qgraph - Code for extracting qsim structures.


;;; TEMPORARY

(defother display-block numeric-time-plot-layout)  ; just like layout
(defother display-block numeric-time-plot-options) ; (var (opt val) ...) list
(defother display-block numeric-time-plot-defaults) ; (var (opt val) ...) list

;;; This file contains the code for pulling qgraph structures out of
;;; QSIM.


;;; Given a behavior, make a qgraph of it, including Q2 and Nsim info.
;;; If neither type of info exists for a behavior then return NIL.
;;; Inputs:  beh     - A Qsim behavior.
;;;          var     - A variable name.
;;;          options - An alist of (slot-name val).
;;; Returns: A qgraph structure or NIL.
;;;
(defun make-qgraph-from-behavior (beh var options)
  (let ((q2-dataset     (if (not (member 'q2
					 (find-option 'ignore options)))
			    (behavior-to-tqual beh var)
			    nil))
	(nsim-dataset   (if (not (member 'nsim
					 (find-option 'ignore options)))
			    (behavior-to-tenv beh var)
			    nil))
	(numsim-dataset (if (not (member 'numsim
					 (find-option 'ignore options)))
			    (behavior-to-line beh var)
			    nil))
	(qg nil))
    (when (or q2-dataset nsim-dataset numsim-dataset)
      (setf qg (make-qgraph :data (nconc (if q2-dataset 
					     (list q2-dataset)
					     nil)
					 (if nsim-dataset
					     (list nsim-dataset)
					     nil)
					 (if numsim-dataset
					     (list numsim-dataset)
					     nil))
			    :XUnitText  "T"
			    :YUnitText  var
;			    :RangeStyle    (find-option 'RangeStyle options
;							:default 1)
;			    :IntervalStyle (find-option 'IntervalStyle options
;							:default 5)
			    ;; There is a separate color for and style entry for each
			    ;; dataset now.  This screws up the default mechanism.
			    ;; BKay 27Jun92
			    :Colors        (nconc (if q2-dataset
						      (list '("red" "black"))
						      nil)
						  (if nsim-dataset
						      (list "red")
						      nil)
						  (if numsim-dataset
						      (list "blue")))
			    :Styles        (nconc (if q2-dataset
						      (list '(1 5))
						      nil)
						  (if nsim-dataset
						      (list '(5 2))
						      nil)
						  (if numsim-dataset
						      (list '(1 0))))
			    :LnX           (find-option 'LnX options
							:default NIL)
			    :LnY           (find-option 'LnY options
							:default NIL)
			    :XLimits       (find-option 'XLimits options
							:default NIL)
			    :YLimits       (find-option 'YLimits options
							:default NIL)
			    :QualPlottingMethod (find-option 'QualPlottingMethod 
							     options :default 'PLAIN-BOX)
			    )))
    qg))


;;; Find an option in a list.
;;; Return it or return default if not found.
;;;
(defun find-option (item options &key (default nil))
  (let ((option-entry (assoc item options)))
    (cond
      (option-entry (second option-entry))
      (T default))))




;;; Generate a TQual structure from a behavior.
;;; Inputs:  beh     - A Qsim behavior.
;;;          var     - A variable name.
;;; Returns: A TQual structure or NIL if one couldn't be made.
;;; Method:  Normally, the final state of the behavior contains the best info about
;;;          Lmark ranges.  However, if there has been a region transition, this might not
;;;          be the case.  The function best-state finds the state with the most detailed 
;;;          qspace for the var.  We go to this state and extract all the lmarks
;;;          for time and the variable of interest.  Then we extract 
;;;          records relating time and var value for each state in the behavior.
;;;          Normally, only the values at the time-points should matter, but
;;;          for vars with ign qdirs, changes of interest may occur between
;;;          time-points.  We therefore extract both time points and time intervals
;;;          from the behavior.
;;;
(defun behavior-to-TQual (beh var)
  (cond
    ;; This can happen if we ran numsim on a dummy initial state.
    ((member 'dummy (state-status (first beh)))
     NIL)
    (T
     (let* ((best-state (best-state-for-var var beh))
	    ;; This is a cheap trick to see if Q2 was run on the behavior.
	    ;; If it wasn't then extract-lmark-entries will have returned NIL.
	    (time-lmarks (extract-lmark-entries 'TIME best-state)))
       (if time-lmarks
	   (make-TQual :Lmarks (list time-lmarks
				     (extract-lmark-entries var   best-state))
		       :Data (tqual-records beh var best-state))
	   nil)
       ))))


;;; Form time records for the behavior.
;;; 
;;;
(defun tqual-records (beh var best-state)
  (loop for s in beh
	with best-qspace = (lookup-set var (state-qspaces best-state))
	collect (list (qmag-to-sexp (qmag (qval 'TIME s)))
		      (qmag-to-sexp (best-qmag var s beh best-qspace))
		      (qdir (qval var s)))))


;;; Find the state that most precisely defines var.
;;; Inputs:  var - A varname.
;;;          beh - A behavior.
;;; Returns: A qspace.
;;; Method:  Start at the end of the behavior and return the first state
;;;          that mentions var.
;;; Note:    This won't work if a transition has occurred that changes the
;;;          qspace of var such that old landmarks are no longer included
;;;          [IS THAT EVEN POSSIBLE?]
;;;
(defun best-state-for-var (var beh)
  (do ((n (1- (length beh)) (1- n))
       state)
      ((< n 0) (error "No state contains variable ~A" var))
    (setf state (nth n beh))
    (when (assoc var (state-qspaces state))
      (return state))))


;;; Find the best (i.e. most restrictive) qmag for a var in a state.
;;; Note that this may not be (qmag var state) if the qmag is an interval.
;;; Inputs:  var    - A variable.
;;;          s      - The state to look in.
;;;          beh    - The behavior to look in.
;;;          qspace - The best (largest) qspace for the var.
;;; Returns: A qmag at least as big as (qmag (qval var s))
;;;
(defun best-qmag (var s beh qspace)
  (let* ((qval (qval var s))
	 (qmag (qmag qval))
	 (n    (position s beh)))
    (cond
      ;; An atom is always best
      ((atom qmag)
       qmag)
      (T
       (let ((qdir (qdir qval)))
	 (when (eq qdir 'IGN)
	   ;; Determine which way the var is changing and use the correct
	   ;; lmark set.
	   (setf qdir (implied-qdir qmag qspace var n beh)))
	 (cond 
	   ;; If the var is increasing then the upper bound will be the next
	   ;; lmark bigger than (car qmag)
	   ((eq qdir 'INC)
	    (list (car qmag) (next-lmark-in-qspace (car qmag) qspace)))
	   ;; If the var is decreasing then the lower bound will be the next
	   ;; lmark smaller than (cadr qmag)
	   ((eq qdir 'DEC)
	    (list (prev-lmark-in-qspace (cadr qmag) qspace) (cadr qmag)))
	   ;; A std interval means that no-new-landmarks is in effect, so 
	   ;; there aren't any intervening lmarks.
	   ((eq qdir 'STD)
	    qmag)       
	   ;; Otherwise, we can't slice the interval any finer.
	   (T
	    qmag)))))))


;;; Get the "implied" qdir for a var.  This is done by comparing its
;;; previous and next lmark values and seeing how they compare to the
;;; current qmag.
;;; Inputs:  qmag   - The qmag at the state of interest.
;;;          qspace - The qspace for var.
;;;          var    - The var name.
;;;          n      - The position of the state of interest in the behavior.
;;;          beh    - The behavior.
;;; Returns: UNKNOWN, INC, DEC, INFLECTION (the latter implying that the qmag
;;;          is an inflection point (both prev and next lmark are on the
;;;          same side of the qmag)).
;;;
(defun implied-qdir (qmag qspace var n beh)
  (let ((prev-lmark (find-prev-lmark var n beh))
	(next-lmark (find-next-lmark var n beh)))
    (cond
      ((and prev-lmark next-lmark)
       (let ((prev-rel-pos (qmag-order prev-lmark qmag qspace))
	     (next-rel-pos (qmag-order next-lmark qmag qspace)))
	 (cond
	   ((and (eq prev-rel-pos '-) (eq next-rel-pos '+))
	    'INC)
	   ((and (eq prev-rel-pos '+) (eq next-rel-pos '-))
	    'DEC)
	   (T
	    'INFLECTION))))
      (T
       'UNKNOWN))))


;;; Find the next state in the behavior that has a lmark qmag for var
;;; and return that lmark.
;;; Inputs:  var - The var name of interest.
;;;          n   - The position of the state of interest (seach starts
;;;                after this).
;;;          beh - The behavior.
;;; Returns: A lmark or NIL.
;;;
(defun find-next-lmark (var n beh)
  (let (qmag)
    (dolist (s (nthcdr (1+ n) beh) NIL)
      (setf qmag (qmag (qval var s)))
      (when (and qmag (atom qmag))
	(return qmag)))))

(defun find-prev-lmark (var n beh)
  (let (qmag)
    (do ((c (1- n) (1- c))
	 s)
	((< c 0) NIL)
      (setf s (nth c beh))
      (setf qmag (qmag (qval var s)))
      (when (and qmag (atom qmag))
	(return qmag)))))


;;; Get the next landmark after lmark in a qspace.
;;; Inputs:  lmark  - a lmark structure.
;;;          qspace - a qspace.
;;; Returns: lmark.
;;;
(defun next-lmark-in-qspace (lmark qspace)
  (let ((lm-pos (position lmark qspace :test #'eq)))
    (if (and lm-pos (< lm-pos (1- (length qspace))))
	(nth (1+ lm-pos) qspace)
	(error "No landmark after ~a in qspace ~a" lmark qspace))))

(defun prev-lmark-in-qspace (lmark qspace)
  (let ((lm-pos (position lmark qspace :test #'eq)))
    (if (and lm-pos (> lm-pos 0))
	(nth (1- lm-pos) qspace)
	(error "No landmark before ~a in qspace ~a" lmark qspace))))


;;; Given a qmag, expand out the structure to use lmark names rather
;;; than lmark structures.
;;;
(defun qmag-to-sexp (qmag)
  (if (listp qmag)
      (list (lmark-name (car qmag)) (lmark-name (cadr qmag)))
      (lmark-name qmag)))

;;; Get the list of lmark entries and their ranges.
;;; Inputs:  var   - A var name.
;;;          state - A state.
;;; Returns: A list of (Lmarkname Loval Hival) or NIL if there were no Q2 ranges.
;;; 
(defun extract-lmark-entries (var state)
  (let ((entries (mapcar #'(lambda (lmark)
			     (cons (lmark-name lmark) 
				   (get-q2-range var lmark state)))
			 (lookup-set var (state-qspaces state)))))
    (if (null (second (first entries)))
	NIL
	entries)))

;;; Get the q2 range associated with a variable in a state.
;;; Inputs:  var   - a variable struct
;;;          lmark - a lmark struct
;;;          state - a state
;;; Returns: a list of (lo hi)
;;;
(defun get-q2-range (var lmark state)
  (cond
    ((eq lmark *INF-LMARK*)
     `(+inf +inf))
    ((eq lmark *MINF-LMARK*)
     `(-inf -inf))
    (T
     (cadr (assoc lmark 
		  (cdr (assoc var (state-bindings state) :key #'variable-name))
		  :test #'equal)))))


 
;;; Generate a TEnv structure from a behavior.
;;; The TEnv is derived from an NSIM run.
;;; Inputs:  beh - A behavior.
;;;          var - The variable of interest.
;;; Returns: A TEnv struct or nil.
;;;
(defun behavior-to-tenv (beh var)
  (let* ((state (first beh))
	 (struct (state-nsim-struct state)))
    (if struct
	(let ((rs (position-rspointer var
				      (nsim-struct-start struct)
				      struct)))
	  (do ((entry (retrieve-nsim-record-var rs '(LB UB) :increment T)
		      (retrieve-nsim-record-var rs '(LB UB) :increment T))
	       (time (nsim-struct-start struct)
		     (incf time (nsim-struct-simStep struct)))
	       (data (make-q)))
	      ((null entry)
	       (make-tenv :TrendName NIL
			  :data data))
	    (qpush (cons time entry)
		   data)))
	NIL)))


;;; Generate a Line structure from a behavior.
;;; Inputs:  beh - A behavior.
;;;          var - The variable of interest.
;;; Returns: A Line struct or nil.
;;;
(defun behavior-to-line (beh var)
  (let* ((state (first beh))
	 (ns (state-numsim-struct state))
	 (varpos nil))
    (when ns
      (cond
	((setf varpos (position var (numsim-svars ns)))
	 (make-line :trendname NIL
		    :data      (nth varpos (first (numsim-results ns)))))
	((setf varpos (position var (numsim-ovars ns)))
	 (make-line :trendname NIL
		    :data      (nth varpos (second (numsim-results ns)))))
	((setf varpos (position var (numsim-cvars ns)))
	 (make-line :trendname NIL
		    :data      (qpush `(,(numsim-time ns) ,(svref (numsim-cv ns) varpos))
				      (make-q :init `(,(numsim-start ns) 
						      ,(svref (numsim-cv ns) varpos))))))))))
						   


;;; This is for Dan Berleant's Q3.
;;; It will soon be obsolete.
#|
(defun q3-behavior-to-qgraph (beh var)
  (make-qgraph :data (list (behavior-to-tq3 beh var))))


(defun behavior-to-TQ3 (beh var)
  (let* ((final-state (car (last beh)))
	 (ranges      (numeric-database final-state)))
    (make-TQ3   :Lmarks (list (extract-lmark-entries-q3 'TIME ranges final-state)
			      (extract-lmark-entries-q3 var   ranges final-state))
		:Data   (mapcan #'(lambda (state)
				    (when (qpointp (state-time state))
				      (list (list (qmag-to-sexp (qmag (qval 'TIME state)))
						  (q3-ranges-to-qgraph-ranges
						    (get-interval ranges :qvar var :state state))
						  (qdir (qval var   state))))))

				beh))))

(defun extract-lmark-entries-q3 (var ranges state)
  (mapcar #'(lambda (lmark)
	      (cons (lmark-name lmark) 
		    (q3-ranges-to-qgraph-ranges
		      (get-interval ranges :qvalu (cons var (lmark-name lmark))
				         :state state))))
	  (lookup-set var (state-qspaces state))))

(defun q3-ranges-to-qgraph-ranges (range)
  (mapcar #'(lambda (x)
	      (case x
		(0+ 0)
		(0- 0)
		(inf '+inf)
		(minf '-inf)
		(T    x)))
	  range))
|#

