;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-

;;; $Id: qplot.lisp,v 1.53 92/07/21 13:18:59 clancy Exp $

(in-package 'QSIM)

; Copyright (c) 1986, 1987, 1988, 1989, 1990, Benjamin Kuipers.

; THE QUALITATIVE PLOTTER.
;     This tries to be device-independent, with the device-dependent code
;     segregated into (e.g.) plot-on-symbolics-screen.lisp.  
;     However, the coordinates are currently in symbolics screen dots!

; UPDATES.
; RSM.  28 Jan 91.
;	Removed old display version code to boneyard>old-displays.lisp
;	Changed functions other than Qsim-Display, Qsim-Bounded-Display, and New-Displays
;	  to use a parameter instead of *reference-states*.

; DJC   14 MAY 91
;       Added the binding of the variable *current-display-block* which
;       maintaines information available in the tree which is needed
;       during display.
;
; DJC   23 Oct 91
;       Added the ability to pass it a list of states as opposed to a
;       single state.  Each of the states in the list may in turn have
;       a number of completions.  This was done at the request of A. Farquhar     

(defun QSIM-DISPLAY (&optional (initial-state-or-SIM *initial-state*)
		     &key 
		     (reference-states nil)
		     (trees *plot-small-state-tree*)
		     (plot-mode 'time-plot)
		     (tree-ystep nil)  ; A number will fix the ystep in
		                       ; behavior plots.  This makes them
		                       ; long, but readable.  Use the unix
		                       ; command btprint to print them. BKay 30Jan92
		     (layout)
		     (show-inconsistent-successors *show-inconsistent-successors*))     ;  28 Feb DJC
  (unless (or (state-p initial-state-or-SIM)
	      (sim-p initial-state-or-SIM))
      (error "Argument passed to Qsim-Display, ~a, is neither a state nor a SIM."
	     initial-state-or-sim))
  (let* ((initials (cond ((listp initial-state-or-SIM)
			  initial-state-or-SIM)
			 ((state-p initial-state-or-SIM)
			  (list initial-state-or-SIM))
			 (t (list (sim-state initial-state-or-SIM)))))

	 ;(initial (if (typep initial-state-or-SIM 'state)    ; added for backward compatibility in case
	 ;	      initial-state-or-SIM                   ; someone has a model which passes the value
	 ;	      (sim-state initial-state-or-SIM)))     ; returned by QSIM to qsim-display.  qsim now returns
                                                             ; a SIM as opposed to a state  DJC 11 Aug 1991
	 (*current-display-block* (find-display-block initials)))  ; 19 Jul 91 DJC
    (declare (special *current-display-block*))
    (unless (string-equal *qsim-display-version* 'new)		; Retain warning for now
      (error "The old display version is no longer supported.~@
	      Specify the new display version with:   (setq *Qsim-Display-Version* 'new)"))
    (setq *reference-states* reference-states)			; Poor, but retain for now
    ;; Added DJC 04/14/91 to record the most recent layout used in the display-block
    (if layout  
	(setf (display-block-layout *current-display-block*) layout)
	(setq layout (display-block-layout *current-display-block*)))
    (if *allow-plotting*
	(with-bindings-from-sim (state-sim (car initials))  ; added DJC 05/25/92
          (device-interface (displays
			      :states initials
			      :reference-states reference-states
			      :layout layout :trees trees :plot-mode plot-mode
			      :tree-ystep tree-ystep
			      :show-inconsistent-successors show-inconsistent-successors))))))
  
;;; Retain warning for now:

(defun New-Displays (&rest ignore)
  (declare (ignore ignore))
  (error "The function New-Displays has been renamed Displays."))


;;; QSIM-BOUNDED-DISPLAY calls Qsim-Display within bindings set up so that
;;; the output can be included in LaTeX output using the psfig macros.

(defun QSIM-BOUNDED-DISPLAY (&key (initial *initial-state*)
			     (reference-states nil)
			     (trees *plot-small-state-tree*)
			     (layout)
			     (plot-mode 'time-plot)               ; BKay 23Oct91
			     (show-inconsistent-successors nil))  ;  28 Feb DJC
  (let* ((*x-translation* 1)
	 (*y-translation* 1)
	 (*rotation* 0)
	 (*bounding-box* (list lmargin (- yscreen bmargin) (- xscreen rmargin) 0))
	 (*postscript-style* :bounded)
	 (*x-scale* (/ (* 72.0 6) xscreen))
	 (*y-scale* *x-scale*))
    (qsim-display initial :reference-states reference-states :trees trees
		  :plot-mode plot-mode
		  :layout layout :show-inconsistent-successors show-inconsistent-successors)))

; Plot the behavior of a single parameter.


(defun PERTURB-LINK (x1 y1 x2 y2)
  (flet ((i (a b)
	   (+ a (* .15 (- b a)))))
    (qplot-vector (i x1 x2)(i y1 y2)
		  (i x2 x1)(i y2 y1) :shaftthick *perturb-arrow-thickness*)
    (qplot-string "p" (+ (/ (+ x1 x2) 2)
			 (if (< y1 y2) 1 -4))
		  (- (/ (+ y1 y2) 2) 4)
		  :font #+symbolics '(:fix :roman :very-small)
		  #-symbolics axis-font)))


;;; QSPACE-FROM-BEHAVIOR is designed to produce a suitable qspace for labeling
;;; the vertical axis, even if it has to dig through the behavior looking for
;;; an appropriate state just before a region transition.

;;; Modified to handle elimination of occurrence branching
;;; 15 May 1991  DJC

(defun QSPACE-FROM-BEHAVIOR (varname behavior)
  (let* ((last-state (car (last behavior)))
	 (qspace     (qspace varname last-state)))
    (dolist (state behavior)
      (if (and (typep state 'state)          ;;  added because if occurrence branching is eliminated 
	                                     ;;  a behavior will contain aggregate intervals  DJC
	       (eql (car (filtered-successor-slot-contents state)) 'transition-identity))
	  (setq qspace (qspace-union (qspace varname state) qspace))))
    qspace))



(defun RETRIEVE-LMARK (lmark alist) ;1/25/90 by D.B.
  (second (assoc (lmark-name lmark) alist :key #'lmark-name)))
 
                                                       ; Modified 1/25/90 by D.B.
(defun MUST-RETRIEVE-LMARK (lmark alist)		; lookup with error on failure.
  (cond ((retrieve-lmark lmark alist))
	(t (error "No value for ~a in ~a." lmark alist))))


;;; If a parameter is in an interval, GUESS-VLOC will guess at the best
;;; location to plot it.  This can be tricky - if the parameter is in an
;;; interval which a later state will split with a new landmark, we must aviod
;;; the appearance of plotting the current value at that lmark.

(defun GUESS-VLOC (qval yalloc param n behavior)
  (let ((qmag (qval-qmag qval)))
    (cond
      ((eql (qdir qval) 'inc)
       (- (must-retrieve-lmark (car qmag) yalloc)
	  (/ *VshareSize* 1.0)))
      ((eql (qdir qval) 'dec)
       (+ (must-retrieve-lmark (second qmag) yalloc)
	  (/ *VshareSize* 1.0)))
      ((and (eql (qdir qval) 'ign) param)
       (let (lmark-qmag)
	 (if (setq lmark-qmag (or (get-prev-lmark-qmag param n behavior)
				  (get-next-lmark-qmag param n behavior)))
	     (if (lmark-equal lmark-qmag (car qmag))
		 (- (must-retrieve-lmark (car qmag) yalloc)
		    (/ *VshareSize* 1.0))
		 (+ (must-retrieve-lmark (second qmag) yalloc)
		    (/ *VshareSize* 1.0)))
	     (/ (+ (must-retrieve-lmark (car qmag) yalloc)
		   (must-retrieve-lmark (second qmag) yalloc))
		2.))))
      (t (let ((prev-dir (get-prev-direction param n behavior)))  ; added because behavior plots were inaccurately
	   (cond ((eql prev-dir 'inc)                             ; plotting std when there was no landmark
		  (- (must-retrieve-lmark (car qmag) yalloc)      ; DJC  08/13/91
		     (/ *VshareSize* 1.0)))
		 ((eql prev-dir 'dec)
		  (+ (must-retrieve-lmark (second qmag) yalloc)
		     (/ *VshareSize* 1.0)))
		 (t (/ (+ (must-retrieve-lmark (car qmag) yalloc)
			  (must-retrieve-lmark (second qmag) yalloc))
		       2.))))))))


;;; This function is called when the qdir of the PARAM in the current state is
;;; std.  It will search previous states in the behavior to find the previous
;;; qdir which is not std or ign (i.e. it is looking for inc or dec).  It
;;; will return this direction of change or else nil.
;;; DJC  08/13/91

(defun GET-PREV-DIRECTION (param n behavior)
  (when n
    (do ((pos (1- n) (1- pos)))
	((< pos 0) nil)
      (when (and (not (agg-interval-p (nth pos behavior)))
		 (alookup param (state-qvalues (nth pos behavior)))
		 (or 
		   (eql 'inc
			(qdir (alookup param (state-qvalues (nth pos behavior)))))
		   (eql 'dec
			(qdir (alookup param (state-qvalues (nth pos behavior)))))))
	(return (qdir (alookup param (state-qvalues (nth pos behavior)))))))))



;;; The (AND ...) check in the next two functions is necessary because a
;;; parameter may not even exist in the quantity space of a previous /
;;; succeeding state, if there is a region transition.

(defun GET-PREV-LMARK-QMAG (param n behavior)
  (do ((pos (1- n) (1- pos)))
      ((< pos 0) nil)
    (when (and (not (agg-interval-p (nth pos behavior)))          ;;  added DJC 18 May 1991
	       (alookup param (state-qvalues (nth pos behavior)))
	       (atom (qmag (alookup param (state-qvalues (nth pos behavior))))))
      (return (qmag (alookup param (state-qvalues (nth pos behavior))))))))


(defun GET-NEXT-LMARK-QMAG (param n behavior)
  (do ((pos (1+ n) (1+ pos)))
      ((>= pos (length behavior)) nil)
    (when (and (not (agg-interval-p (nth pos behavior)))          ;;  added DJC 18 May 1991
	       (alookup param (state-qvalues (nth pos behavior)))
	       (atom (qmag (alookup param (state-qvalues (nth pos behavior))))))
      (return (qmag (alookup param (state-qvalues (nth pos behavior))))))))


;;;  FIND-DISPLAY-BLOCK will return the display block for this state.  A change has
;;;  been made so that whenever a state is created it points to the SIM 
;;;  under which it was created.  DJC
;;;  Modified to handle a list of states.  If a list of states is passed, then it will
;;;  return the display block for the first state in the list.  This change was
;;;  made in conjunction with a change which allows qsim-display to receive a list
;;;  of states.   DJC  23Oct91

(defun find-display-block (state-or-states)
  "Returns the display block for this state"
  (let ((state (if (listp state-or-states)   ; added DJC
		   (car state-or-states)
		   state-or-states)))
    (if (state-sim state)
	(cond ((sim-display-block (state-sim state)))
	      (t (format *Qsim-Report* "~&~%Warning: ~a does not have a display block attatched.  ~
                     A default one is being added." 
			 (state-sim state))
		 (setf (sim-display-block (state-sim state))
		       (initialize-display-block state))))
	(error "State ~a does not have a SIM attatched to it.  This is a bug in the QSIM code."))))
  


;;;-----------------------------------------------------------------------------
;;;  QPLOT-STATE  - Plot and perhaps number a state in a state tree diagram.
;;;  Modified by DLD, per Ben's suggestion, to display time points as filled
;;;    circles (discs) and time intervals as empty circles.  Unfinished states
;;;    are followed by a short dashed line to indicate that the state's
;;;    successors have not been generated.
;;;  Modified by RSM 13 Feb 91 to include in the state index, for states that
;;;    end cycles, the number of the state that starts the cycle.
;;;-----------------------------------------------------------------------------

(defun QPLOT-STATE (state x y &optional status)
  "Plot and perhaps number a state in a state tree diagram."
  ;; Status is optional, so it may be overwritten.
  (unless status (setq status (state-status state)))
  (let* ((size 3)				        	; BJK:  10-25-90
	 (index (cond ((member 'cycle status)                   ; DJC:  03-15-92
		       ;; If state ends a cycle, also record in the state index
		       ;; the state that begins the cycle.
		       (concatenate 'string
				    (get-index-string (state-name state))
				    " = "
				    (get-index-string (state-name (cadr (state-successors state))))))
		      ((member 'cross-edge (state-status state))
		       (concatenate 'string
				    (get-index-string (state-name state))
				    " = "
				    (get-index-string (state-name (state-cross-edge-identity state)))))
		      (t
		       (get-index-string (state-name state)))))
	 (fill   (time-point-p (state-time state))))
    ;; Print state index (name stripped of leading "s-") near state symbol.
    (case *plot-state-indices*
      (:right-of-node
       (qplot-string index (+ x 5) (- y 1) :font axis-font))
      (:above-node
       (qplot-string index
		     (- x 5) (- y 5) :font axis-font))
      (:at-node
       (qplot-string index
		     (- x 5) (+ y 4) :font axis-font)))
    ;; Plot state symbol, unless index takes its place.
    (unless (eq *plot-state-indices* :at-node)
      (cond
	;; Unfinished state displayed with trailing dashed line.
	((null status)
	 (qplot-circle x y size :filled fill)
	 (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3)))
	;; Final state displayed as a small disc inside a larger circle.
	((intersection status '(QUIESCENT FINAL-STATE))
	 (qplot-circle x y (+ size 1))
	 (qplot-circle x y (- size 1) :alu *black* :filled t))
	;; Normal intermediate state.
	((member 'OK status)
	 (qplot-circle x y size :alu *black* :filled fill))
	;; Cycle displayed as small circle inside large circle.
	((member 'CYCLE status) (qplot-circle x y (+ size 1))
	                        (qplot-circle x y (- size 1)))
	;; Cross edge displayed as a circle with an X through it
	;; Added DJC with the Cross edge detect code  03/6/92
	((member 'CROSS-EDGE status) (qplot-circle x y (+ size 1))
		                     (qplot-line (- x (* .5 size)) (- y (* .5 size)) 
						 (+ x (* .5 size)) (+ y (* .5 size))))
	;; Transition state displayed as large circle with vertical line through it.
	((member 'TRANSITION status) (qplot-circle x y (+ size 1))
		                     (qplot-line x (- y size) x (+ y size)))
	;; Abstraction of chattering states displayed as a star
	((member 'CHATTER status)
	 (qplot-symbol x y 'ign))
	;;(qplot-box (- x size) (- y size) (* 2 size) (* 2 size) :thickness 1)
	;;(qplot-circle x y (+ size 1))
	;;(qplot-line (- x size) (- y size) (+ x size) (+ y size))
	;;(qplot-line (- x size) (+ y size) (+ x size) (- y size))
	;; States aggregating occurence branching displayed as boxes
	((member 'OCCURRENCE-BRANCHING status)
	 (qplot-box (- x size) (- y size) (* 2 size) (* 2 size) :thickness 2))
	;; Inconsistent state is plotted normally, later marked with "X".
	((intersection status '(INCONSISTENT SELF-INTERSECTION))
	 (qplot-circle x y size :alu *black* :filled fill))
	;; Unfinished state displayed with trailing dashed line.
	((member 'COMPLETE status)
	 (qplot-circle x y size :filled fill)
	 (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3)))
	;; Unknown status displayed as "?".
	(T (qplot-circle x y (+ size 5))		
	   (qplot-symbol x y '?)))     
      (cond ((intersection status '(INCONSISTENT SELF-INTERSECTION))
	     ;; Inconsistent state is marked with "X", in addition to whatever else.
	     (qplot-line (- x size -10) (- y size) (+ x size 10) (+ y size))
	     (qplot-line (- x size -10) (+ y size) (+ x size 10) (- y size)))))   
    ;; To mark a state in the display add (MARK-STATE <marking-type>) to the status slot.
    ;; Currently the <marking-type> can either be :arrow for an arrow or a string
    ;; added DJC 05/4/92
    (when *mark-states*   
      (let ((marking (cadar (member 'MARK-STATE status 
				    :test #'(lambda (s1 s2)
					      (when (listp s2)
						(equal s1 (car s2))))))))
	(cond ((equal marking :arrow)
	       (qplot-symbol x (+ y (* 3 size)) :inc))
	      ((stringp marking)
	       (unless (equal *plot-state-indices* :above-node)
		 (qplot-string marking (- x 3) (- y 5) :font axis-font))))))
    ))


;;;  Added DJC at Adam Farquhar's suggestion to display a key of the
;;;  symbols on a state tree.  03/30/92

(defmacro inc-y ()
  '(setf y (+ 30 y)))

(defun qplot-state-symbol-key ()
  "This function will display a key of the symbols used in the behavior tree"
  (let ((x 50)
	(y 50)
	(size 3)
	(x-str 50)
	(y-str 5))
    ;; Final state displayed as a small disc inside a larger circle.
    (qplot-new-behavior)
    (qplot-circle x y (+ size 1))
    (qplot-circle x y (- size 1) :alu *black* :filled t)
    (qplot-string "Final state" (+ x 50) (+ y 5))
    (inc-y)
    ;; Normal intermediate time-point state.
    (qplot-circle x y size :alu *black* :filled t)
    (qplot-string "Normal Intermediate time-point state" (+ x 50) (+ y 5))
    (inc-y)
    ;; Normal intermediate time-interval state.
    (qplot-circle x y size :alu *black* :filled nil)
    (qplot-string "Normal Intermediate time-interval state" (+ x 50) (+ y 5))
    (inc-y)
    ;; Cycle displayed as small circle inside large circle.
    (qplot-circle x y (+ size 1))
    (qplot-circle x y (- size 1))
    (qplot-string "Cycle state" (+ x 50) (+ y 5))
    (inc-y)
    ;; Cross edge displayed as a circle with an X through it
    ;; Added DJC with the Cross edge detect code  03/6/92
    (qplot-circle x y (+ size 1))
    (qplot-line (- x (* .5 size)) (- y (* .5 size)) 
		(+ x (* .5 size)) (+ y (* .5 size)))
    (qplot-string "Cross Edge state" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; Transition state displayed as large circle with vertical line through it.
    (qplot-circle x y (+ size 1))
    (qplot-line x (- y size) x (+ y size))
    (qplot-string "Transition State" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; Abstraction of chattering states displayed as a star
    (qplot-symbol x y 'ign)
    (qplot-string "Abstraction of a chatering state" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; States aggregating occurence branching displayed as boxes
    (qplot-box (- x 2)(- y 5) 14 10)    ;; x and y are adjusted to center the box
    (qplot-string "x" (+ x size)  (+ y size) :font axis-font)
    (qplot-string "Aggregate Interval, structure name = AGG-x" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; Unfinished state displayed with trailing dashed line.
    (qplot-circle x y size :filled nil)
    (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3))
    (qplot-string "Unfinished state" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; Unknown status displayed as "?".
    (qplot-circle x y (+ size 5))		
    (qplot-symbol x y '?)
    (qplot-string "Unknown state status" (+ x x-str) (+ y y-str))
    (inc-y)
    ;; Inconsistent state is marked with "X", in addition to whatever else.
    (qplot-line (- x size) (- y size) (+ x size) (+ y size))
    (qplot-line (- x size) (+ y size) (+ x size) (- y size))
    (qplot-string "Additional marking added to inconsistent state" (+ x x-str) (+ y y-str))
    (inc-y)
    (qplot-end-display))
  nil)





;;; Draw a dotted line connecting two plotted points.

(defun CONNECT-WITH-DOTS (x1 y1 x2 y2)
  (let ((dx (/ (- x2 x1) (+ ndots 1.0)))
	(dy (/ (- y2 y1) (+ ndots 1.0))))
    (do ((N  ndots  (- N 1))
	 (x  (+ x1 dx)  (+ x dx))
	 (y  (+ y1 dy)  (+ y dy)))
	((= N 0))
      (qplot-dot x y))))


;;;-----------------------------------------------------------------------------
;;; Draw the behavior tree.
;;; initial-state is a list of initial-states
;;; behaviors is a list of lists of behaviors.
;;;-----------------------------------------------------------------------------


(defun PLOT-STATE-TREE (initial-state depth width xloc yloc xsize ysize bnum
				      &key (tree-ystep nil)) ; BKay 3Feb92
  (let* ((xstep (min  (/ xsize depth) tree-xstep-max))
	 (ystep (if tree-ystep
		    tree-ystep
		  (min  (/ ysize width) tree-ystep-max)))
	 (*processed-states* ()))
    (declare (special *processed-states* *n*))
    (plot-trees initial-state xloc yloc xstep ystep nil nil)
    (if *plot-beh-numbers*
	(plot-beh-numbers depth width xloc yloc xstep ystep))
    (if (> bnum 0)
	(qplot-symbol (+ xloc xsize) (+ yloc (* (- bnum 1) ystep)) 'left))
    ))


(defun PLOT-BEH-NUMBERS (depth width xloc yloc xstep ystep)
  "Plot the number of each behavior (1, ..., width) on the state tree plot."
  (if *plot-beh-numbers*
      (loop with x = (if (eq *plot-beh-numbers* :right)
			 (+ xloc 5 (* depth xstep))	; right side of plot
			 (- xloc 25))			; left  edge of plot
	    for n from 1 to width
		  by (ceiling 8 ystep)	 ; at least 8 pixels between numbers
	    for y = (+ yloc 5 (* (1- n) ystep))
	    do (qplot-string n x y :font axis-font))))


;;;----------------------------------------------------------------------------
;;; plot-tree must return the y-level for the next call.
;;;   x,y  =  coordinates for this node
;;;   dx,dy  =  step amounts to later nodes
;;;   ox,oy  =  previous node to draw connections from.
;;;----------------------------------------------------------------------------

;;;  Modified so that it can plot aggregate intervals that occur in the code
;;;  15 May 1991  DJC

(defun PLOT-TREES (states x y dx dy ox oy &optional (previous-states nil))
  (declare (special *processed-states*))
  (if (null states)
      y
      (let* ((state (car states))
	     (other-states (cdr states))
	     (successors (get-successors state))
	     (ny nil)
	     (nx x)                  
	     (cycle-p (if *develop-graph*
			  (member state previous-states)
			  (member state *processed-states*))))
	(cond ((typep state 'aggregate-interval)                     ;;  added to handle occurrence branching
	       (qplot-agg-node state x y)
	       (setf nx (+ x 10)))
	      (t (qplot-state state x y (when cycle-p '(cycle)))))
	(when (numberp ox)
	  (draw-tree-limb ox oy x y
			  :thickness (if *envisionment*
					 (if cycle-p 1 2)
					 1)))
	(unless cycle-p
	  (push state *processed-states*))
	;;(unless (null successors) (push state previous-states)))
	(setq ny
	      (if (or cycle-p (null successors))
		  (+ y dy)
		  (plot-trees successors (+ nx dx) y dx dy nx y (cons state previous-states))))
	(plot-trees other-states nx ny dx dy ox oy previous-states))))


(defun DRAW-TREE-LIMB (x1 y1 x2 y2 &key (thickness 1))	; => L-shaped, not diagonal.
  (let ((size 3))				; radius of standard circle
    (if (= y1 y2)
	(qplot-hline (+ x1 size) y1 (- x2 x1 size size) :thickness thickness )
	(progn
	  (qplot-vline x1 (+ y1 size) (- y2 y1 size) :thickness thickness )
	  (qplot-hline x1 y2 (- x2 x1 size) :thickness thickness)))))


(defun TRANSITION-FOLLOWS (state &optional succ)
  (and state
       (eql (car (filtered-successor-slot-contents state)) 'transition-identity)
                ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
       (second (filtered-successor-slot-contents state))
       (if succ (member succ (cdr (filtered-successor-slot-contents state))) t)))



(defun TRANSITION-PRECEDES (state) ;;10/23/89 D.B.  
 (let ((pred (predecessor-of-state state)))
   (and pred				; Modified so it also finds completions of incomplete 
	(transition-follows pred))))	; states after region transitions - DRT.




;;; The remaining pages contain code for the new version of the QSIM display.

(proclaim '(special *time-points-only*))


;;; New version of Displays (old version --> boneyard).
;;; Cleaned up slightly 19 Oct 90 RSM

;;; Modified so that it can handle aggregate-intervals which are
;;; created when occurrence branching is eliminated.
;;; 15 May 1991  DJC
;;; Modified to display envisionments
;;; 04 Jun 1991  PF
;;; Modified to make numeric timeplots using qgraph.
;;; 25 Sep 1991  BKay
;;; Added a keyword states which can be list of states to display.
;;; 23 Oct 91
;;;  Modified to work off of single character strings as opposed
;;;  to the characters as it had previously been  DJC 04/20/92

(defun DISPLAYS (&key (state *initial-state*)
		 (states (list state))
		 (reference-states nil)
		 (trees *plot-small-state-tree*)
		 (tree-ystep nil)          ; BKay 3Feb92
		 (*plot-intervals* (display-block-plot-intervals *current-display-block*))
		 (layout)
		 (plot-mode 'time-plot)    ; BKay 25Sept91
		 (show-inconsistent-successors *show-inconsistent-successors*))    ;  28 Feb DJC
  (let* ((*show-inconsistent-successors* show-inconsistent-successors)    ;   28 Feb  DJC
	 (display-block *current-display-block*)
	 (*envisionment* (display-block-envisionment-p display-block))
	 (*develop-graph* nil)			; PF 02 May 1991
	 (*traverse-xedges* nil)
	 (initials (cond ((mapcan #'(lambda (state)    ;  DJC 23 Oct 91
				      (get-list-of-initial-states state :complete-partial-state nil))
				  states))
			 ; In case a state is incomplete and has no consistent successors DJC 03/15/92
			 ((incomplete-p state)
			  (cerror "Show the inconsistent successors" 
				  "The initial-state is incomplete and has no consistent successors.")
			    (let ((*show-inconsistent-successors* t))
			      (mapcan #'(lambda (state)    
				      (get-list-of-initial-states state :complete-partial-state nil))
				  states)))))
	 (qde (state-qde (car initials)))
	 (qspaces nil)				; Ignored in Display-Behavior
	 (behaviors (get-behaviors initials))
	 (visible-aggregates (unless *envisionment* (find-aggregates-in-behaviors behaviors)))
	 (all-aggregates (get-all-aggregates states))
	 (*agg-stack* nil)
	 (btotal (length behaviors))
;        The above line was deleted because the max function is not guaranteed
;	 to accept more than 256 arguments.  The loop below can take as many
;	 as needed.  BKay and AF 3Feb92
	 (bmax (loop for b in behaviors maximize (length b)))
	 (click-p nil)
	 (bnum nil)
	 (pplanes nil)
	 rows cols)
    (if (and *envisionment* (null *plot-state-indices*))	;  3 Oct 90 RSM
	;; setq, not let, permits user change during displays with break
	(setq *plot-state-indices* :above-node))		; 22 Oct 90 RSM
    (unless layout
      (setf layout (find-some-layout behaviors reference-states)))
    (compute-layout (if (eq plot-mode 'numeric-time-plot)
		        (display-block-numeric-time-plot-layout
			 *current-display-block*)
			layout))     ; If clause added BKay 25Sept91
    (loop							; 19 Oct 90 RSM
      for cmd = (get-display-command bnum btotal click-p plot-mode)
      do
      (setq click-p nil)
      (case cmd
	((Q) (close-postscript-file-if-needed)
	 (return-from displays t))		; BJK:  10-25-90
	((C) 
	 (click-input initials reference-states bmax btotal cols layout qspaces trees
		      plot-mode behaviors pplanes))
	((T)
	 (qplot-behavior-tree initials bmax btotal 0 :tree-ystep tree-ystep)
	 (setq click-p t))
	((K)   ; added DJC  03/30/92
	 (qplot-state-symbol-key))
	((CH)
	 (change-output-routing))
	((O V P L S I A  
	  G  ;BKay 25Sept91
	  B  ;DJC 05Feb92
	  Z  ; DJC 21Feb92  (selective continuation of the tree)
	  TR ; DJC 1May92  Truncate behavior tree or display
	  )
	 (loop
	   for (key values)
	   in (do-other-commands
		(if (equal cmd 'O) (get-other-command plot-mode all-aggregates) cmd)
		initials reference-states behaviors layout 
		display-block plot-mode ; BKay 25Sept91
		qspaces bmax btotal 
		bnum pplanes rows cols qde trees visible-aggregates all-aggregates)
	   do (case key
		(layout  (if (eq plot-mode 'time-plot)
			     (setq layout values))) ; If clause added BKay 22Oct91
		(pplanes (setq pplanes (first values)
			       rows (second values)
			       cols (third values)))
		(plot-mode (setq plot-mode values))
		;;  Recalculate the behavior information if the level of abstraction
		;;  is changed.
		(recalc-beh (setf behaviors (get-behaviors initials)
				  btotal (length behaviors)
				  bmax (apply #'max (mapcar #'length behaviors))
				  bnum nil
				  visible-aggregates (unless *envisionment* (find-aggregates-in-behaviors behaviors)))
			    (when values 
			      (setf all-aggregates (get-all-aggregates states)))))))
	(t
	  (setq bnum cmd)
	  (case plot-mode
	    (time-plot (time-plot-behavior initials reference-states behaviors
					   layout qspaces bmax btotal bnum trees :display-block display-block))
	    (phase-plot (phase-plot-behavior initials behaviors pplanes cols
					     bmax btotal bnum))
	    ;; Added 25Sept91 by BKay 
	    (numeric-time-plot
	     (numeric-time-plot-behavior
	      behaviors 
	      (display-block-numeric-time-plot-layout *current-display-block*)
	      (display-block-numeric-time-plot-options *current-display-block*)
	      btotal
	      bnum))
	    ))))))


(defun LAYOUT-FROM-STATE (initial)
  (qde-layout (state-qde (if (atom initial)	; ugly!
			     initial
			     (car initial)))))


#+symbolics
(defun CLICK-INPUT
       (initials reference-states bmax btotal cols layout qspaces trees plot-mode behaviors pplanes)
  (do ((list (select-behs-from-beh-tree initials bmax btotal nil) (cdr list)))
      ((null list) nil)
    (case plot-mode
      (time-plot (time-plot-behavior initials reference-states behaviors
				     layout qspaces bmax btotal (car list) trees))
      (phase-plot (phase-plot-behavior initials behaviors pplanes
				       cols bmax btotal (car list))))
    (when (cdr list)
      (format t "~%Behavior(s) remaining to be plotted: ~a.  " (cdr list))
      (when (null (y-or-n-p "Continue? "))
	(return nil)))))

#-symbolics
(defun CLICK-INPUT (&rest ignore)
  (declare (ignore ignore))
  (format *Qsim-Report* "~&This command not available on non-Symbolics machines."))



(defun change-output-routing ()
  (let* ((options '((S "Screen")
		    (F "New File")
		    (B "Screen and File")
		    (N "Nowhere")))
	 (cmd (car (general-menu options))))
    (when cmd
      (close-postscript-file-if-needed)
      (case cmd
	((S) (set-image-disposal :screen))
	((F) (if (get-and-open-postscript-file)
		 (set-image-disposal :ps)
		 (set-image-disposal :screen)))
	((B) (if (get-and-open-postscript-file)
		 (set-image-disposal :both)
		 (set-image-disposal :screen)))
	((N) (set-image-disposal :none))))))

		   
;;;  Modified to add a command to handle occurrence branching elimination
;;;  15 May 1991  DJC
;;;  Modified to add commands to handle multiple levels of abstraction
;;;  04 Jun 1991
;;;  Added the initials argument to the call to var-slice-viewer  DJC  07/20/91
;;;  Added disp-blk and plot-mode args for numeric plotting BKay 25Sept91
;;;  Modified to work off of single character strings as opposed
;;;  to the characters as it had previously been  DJC 04/20/92

(defun DO-OTHER-COMMANDS (cmd initials reference-states behaviors layout 
			  disp-blk plot-mode
			  qspaces bmax btotal bnum
			  pplanes rows cols qde trees
			  visible-aggregates all-aggregates)
  (case cmd
    ((V) (var-slice-viewer initials) (compute-layout layout)
     nil)
    ((P) (phase-slice-viewer initials) (compute-layout layout)
     nil)
    ((L) (modify-layout initials layout qde disp-blk plot-mode))
    ((S)
     (switch-plot-style initials reference-states behaviors layout disp-blk
			qspaces bmax btotal
			bnum pplanes rows cols plot-mode trees))
    ((I)			; Toggle switch for plotting time intervals in phase plots
     (setq *time-points-only* (not *time-points-only*))
     (format *Qsim-Report* "~&Time intervals are now~A plotted in phase plots."
	     (if *time-points-only* " not" ""))
     nil)
    ((O)
     (format *Qsim-Report* "No more options available yet.  Back to the basic prompt.")
     nil)
    ((A) (cond (all-aggregates (occ-branch-display-menu visible-aggregates all-aggregates))
	       (t (format *Qsim-Report*
			  "~&Attempting to create aggregate states for occurrence branches in the tree.")
		  (setq *aggregate-intervals* nil)
		  (mapc #'perform-occ-branch-aggregation initials)
		  (cond (*aggregate-intervals*
			 (setf *filter-occ-branching* t)
			 (list (list 'recalc-beh t)))
			(t (format *Qsim-Report* "~&Unable to abstract a portion of the tree."))))))
    ((B) (comparison-viewer initials) nil)    ; added DJC 09/13/91
    ((Z) (selective-continue initials) `((recalc-beh t)))   ; added DJC 02/21/92
    ;; BKay 25Sept91
    ((G) (do-numeric-command disp-blk initials))
    ((C) (change-description-level) `((recalc-beh t)))
    ((SW) (when (or *envisionment*
		    *cross-edge-envisionment*)
	    (setq *develop-graph* (not *develop-graph*))
	    (setq *traverse-xedges* (not *traverse-xedges*)) `((recalc-beh t))))
    ((F) (select-and-apply-focusing-technique) `((recalc-beh t)))
    ((TR) (set-truncation disp-blk) `((recalc-beh t)))   ; added DJC 05/1/92
    ((DD) (format *Qsim-Report* "~%  This option allows you to display the differences between ~
             ~%  the successors of a state.  Identify the state by selecting branches at each ~
             ~%  branch point. The top branch displayed at a branch point is numbered 1. ~
             ~%  A list of numebrs should be entered.  Each number selects the branch ~
             ~%  to be traversed at the next branch point.  The differences between the ~
             ~%  successors of the state at the next branch point will be displayed in a chart.~%")
     (successor-differences-chart (select-state initials)) nil)
    (t
     (format *Qsim-Report* "~&~A is an invalid command.  Back to the basic prompt." cmd)
     nil)))


;;; This function has been rewritten by BKay 25Sept91 to handle four different
;;; modes.  These modes are now selected from a list (rather than toggling as
;;; before).
;;;
(defun SWITCH-PLOT-STYLE (initials reference-states behaviors layout 
			  disp-blk qspaces bmax btotal
			  bnum pplanes rows cols plot-mode trees)
  (let (cmd)
    (format *Qsim-Report* "~&Current plotting mode is ~a.~
                           ~&1=Qualitative time plots, ~
                             2=Qualitative phase plots, 3=Numeric time plots, ~
                             4=Numeric phase plots, Q=Quit :"
	    (plot-mode-to-string plot-mode))
    (clear-input)
    (setf cmd (read-char))
    (case cmd
      ((#\1) (format *Qsim-Report* "~&Switching to time plots.")
             (compute-layout layout)
	     (when (and bnum (y-or-n-p "~&Proceed to time plot the current behavior? "))
	       (time-plot-behavior initials reference-states behaviors
				   layout qspaces bmax btotal bnum trees))
	     (list (list 'plot-mode 'time-plot)))
      ((#\2) (let ((list nil))
	       (format *Qsim-Report*
		"~&Switching to phase plots.  ~
		   Time intervals are currently~A plotted.  ~
		   Change mode using the I command."
		(if *time-points-only* " not" ""))
	       (cond (pplanes
		      (compute-rectangular-layout rows cols))
		     (t
		      (setq list (get-phase-plot-layout (car initials) nil))
		      (compute-rectangular-layout (second list) (caddr list))))
	       (when (and bnum (y-or-n-p "~&Proceed to phase plot the ~
                                            current behavior? "))
		 (if list			     
		     (phase-plot-behavior initials behaviors (car list)
					  (caddr list) bmax btotal bnum)
		     (phase-plot-behavior initials behaviors pplanes cols
					  bmax btotal bnum)))
	       (if list
		   (list (list 'plot-mode 'phase-plot)
			 (list 'pplanes list))
		 (list (list 'plot-mode 'phase-plot)))))
      ((#\3) (format *Qsim-Report* "~&Switching to numeric time plots. ~
                                      Change plotting modes with the G command.")
             (when (null (display-block-numeric-time-plot-layout disp-blk))
	       (modify-layout initials NIL NIL disp-blk 'numeric-time-plot))
	     (compute-layout (display-block-numeric-time-plot-layout disp-blk))
             (list (list 'plot-mode 'numeric-time-plot)))
      ((#\4) (format *Qsim-Report* "~&This option not implemented.  ~
                                      The mode will remain ~a."
		     (plot-mode-to-string plot-mode)))
      ((#\q #\Q))
      (t
       (format *Qsim-Report* "~&~A is an invalid command.  Back to the basic prompt." cmd)))))

;;; Old definition
;;;
#|
(defun SWITCH-PLOT-STYLE (initials reference-states behaviors layout qspaces bmax btotal
			  bnum pplanes rows cols plot-mode trees)
  (case plot-mode		; Switch between time and phase plots
    (time-plot 
      (let ((list nil))
	(format *Qsim-Report*
		"~&Switching from time plots to phase plots.  ~
		   Time intervals are currently~A plotted.  ~
		   Change mode using the I command."
		(if *time-points-only* " not" ""))
	(cond (pplanes
	       (compute-rectangular-layout rows cols))
	      (t
	       (setq list (get-phase-plot-layout (car initials) nil))
	       (compute-rectangular-layout (second list) (caddr list))))
	(when (and bnum (y-or-n-p "~&Proceed to phase plot the current behavior? "))
	  (if list			     
	      (phase-plot-behavior initials behaviors (car list)
				   (caddr list) bmax btotal bnum)
	      (phase-plot-behavior initials behaviors pplanes cols bmax btotal bnum)))
	(if list
	    (list (list 'plot-mode 'phase-plot)
		  (list 'pplanes list))
	    (list (list 'plot-mode 'phase-plot)))))
    (phase-plot (format *Qsim-Report* "~&Switching from phase plots to time plots.")
		(compute-layout layout)
		(when (and bnum (y-or-n-p "~&Proceed to time plot the current behavior? "))
		  (time-plot-behavior initials reference-states behaviors
				      layout qspaces bmax btotal bnum trees))
		(list (list 'plot-mode 'time-plot)))))
|#

;;; Return a string name for the plotting mode
;;; BKay 25Sept91
;;;
(defun plot-mode-to-string (plot-mode)
  (case plot-mode
    (time-plot "Qualitative time plots")
    (phase-plot "Qualitative phase plots")
    (numeric-time-plot "Numeric time plots")
    (numeric-phase-plot "Numeric phase plots")
    (t                  "???")))

;;; Added disp-blk and plot-mode args for numeric layout control.
;;; This function directly changes the display-block for numeric time plots.
;;; It returns a new layout for qualitative time and phase plots.
;;;
(defun MODIFY-LAYOUT (initials layout qde disp-blk plot-mode)
  (case plot-mode
      (time-plot			; Modified 1/17/90 by D.B.
       (let ((l nil))
	 (format *Qsim-Report* "~&Current layout is: ~a" layout)	
	 (format *Qsim-Report* "~%Remove TIME from layout only? ") 
	 (if (y-or-n-p)
	     (setq l (remove-time-from-layout layout))
	     (progn
	       (setq l (get-time-plot-layout (car initials)))
	       (compute-layout l)
	       (when (y-or-n-p "Do you want to use this as the QDE layout clause? ")
		 (setf (qde-layout qde) l))))
	 (list (list 'layout l))))
      (phase-plot 
       (let ((list (get-phase-plot-layout (car initials) t)))
	 (compute-rectangular-layout (second list) (caddr list))
	 (list (list 'pplanes list))))
      (numeric-time-plot
       (let ((l nil))
	 (format *Qsim-Report* "~&Current layout is: ~a"
		 (display-block-numeric-time-plot-layout disp-blk))
	 (setq l (get-time-plot-layout (car initials)))
	 (set-numeric-layout disp-blk l)
	 (compute-layout l)
	 (list nil nil))) ; Needed for return call.
      (t nil)))


;;; Old version
#|
(defun MODIFY-LAYOUT (initials layout qde)
  (let ((p (ask-for-plot)))			; change layout for behavior plots
    (case p
      ((#\T #\t)				; Modified 1/17/90 by D.B.
       (let ((l nil))
	 (format *Qsim-Report* "~&Current layout is: ~a" layout)	
	 (format *Qsim-Report* "~%Remove TIME from layout only? ") 
	 (if (y-or-n-p)
	     (setq l (remove-time-from-layout layout))
	     (progn
	       (setq l (get-time-plot-layout (car initials)))
	       (compute-layout l)
	       (when (y-or-n-p "Do you want to use this as the QDE layout clause? ")
		 (setf (qde-layout qde) l))))
	 (list (list 'layout l))))
      ((#\P #\p) 
       (let ((list (get-phase-plot-layout (car initials) t)))
	 (compute-rectangular-layout (second list) (caddr list))
	 (list (list 'pplanes list))))
      (t nil))))
|#

(defun REMOVE-TIME-FROM-LAYOUT (layout)   ; 1/17/90 by D.B.
  (mapcar #'(lambda (parameters)
	      (set-difference parameters '(time)))
	  layout))

;;; Modified to add a command for elimination of occurrence branchin
;;; 15 May 1991 DJC
;;; Modified to add commands handling abstraction-levels
;;; 04 Jun 1991

(defun GET-OTHER-COMMAND (plot-mode all-aggregates)
  (format *Qsim-Report* "~&~%V=Var slice viewer, P=Phase slice viewer, L=change Layout, ~
			   S=Switch to ~A plots, ~%G=Numeric graph options, ~
                           B=Beh Comparison, I=time Intervals mode, Z=Selective Continuation, "
	  (case plot-mode (time-plot "phase") (phase-plot "time")))
  (when (or *envisionment*
	    *cross-edge-envisionment*)
    (format *Qsim-Report* "SW=Switch display mode to ~a,"
	    (if *develop-graph* "spanning tree" "complete behaviors"))
    (when *envisionment*
      (format *Qsim-Report* "C=Change description level, F=Focus On ...")))
  (format *Qsim-Report* "~%~a, TR=Truncate Behaviors, DD=Display Differences: "
	  (if all-aggregates 
	      "A=Aggregation Menu"
	      "A=Aggregate Behaviors"))
  (clear-input)
  (read))



(defun ASK-FOR-PLOT ()
  (format *Qsim-Report*
	  "~&Change layout.  Enter T=Time plots, P=Phase plots, Other=no change: ")
  (read-char))



(defun PLOT-SMALL-STATE-TREE (initials bmax btotal bnum)
  "Plot a small state tree without state indices in the upper right corner of a
   time, phase, or other plot if the condition in *plot-small-state-tree* is met."
  (if (and *plot-small-state-tree*
	   (or (not (numberp *plot-small-state-tree*))
	       (<= btotal *plot-small-state-tree*)))
      (let ((*plot-state-indices* nil))
	(plot-state-tree initials bmax btotal (- xscreen rmargin xbsize)
			 ybloc xbsize ybsize bnum))))
  


(defun PHASE-PLOT-BEHAVIOR (initials behaviors pplanes cols bmax btotal bnum)
  (qplot-new-behavior)
  (qplot-label (nth (1- bnum) behaviors) bnum btotal)
  (do ((ypos tmargin (+ ypos *ysize* ysep))
       (plist pplanes (nthcdr cols plist)))
      ((null plist))
    (do* ((ct cols (1- ct))
	  (ps plist (cdr ps))
	  (parms (car ps) (car ps))	    
	  (xpos lmargin (+ xpos *xsize* xsep)))
	 ((zerop ct))
      (when (car ps)
	(let* ((label (format nil "~a vs ~a" (car parms) (second parms))))
	  (plot-one-phase-diagram parms (nth (1- bnum) behaviors) xpos ypos *xsize* *ysize* label)))))
  (plot-small-state-tree initials bmax btotal bnum)
  (qplot-end-display))



;;;
;;;  TRUNCATE BEHAVIOR CODE
;;;
;;; The following functions are used to specify a time qmag at which
;;; behavior displays are turncated.  This information is retained
;;; on the display block of the SIM.  Currently, it only applies to
;;; displaying a behavior and not the display of the behavior tree
;;; although this might be changed.
;;; Added DJC  01/5/92


(defun truncate-behavior (behavior time-value)
  "This function will truncate a behavior when the value of the time
variable exceeds the TIME-VALUE passed in as an argument."
  (loop for state in behavior
	when (and (not (aggregate-interval-p state))
		  (later-time? (qmag (state-time state)) time-value)) return beh 
	collect state into beh
	finally (return beh)))

(defun truncate-behaviors (behaviors time-value)
  (mapcar #'(lambda (beh)
	      (truncate-behavior beh time-value))
	  behaviors))

(defun truncate-behavior-if-needed (behavior display-block)
  "This function truncates the behavior if the display-block has
truncate? set."
  (let ((truncate? (display-block-truncate? display-block)))
    (if truncate? (truncate-behavior behavior truncate?)
	behavior)))

;;;
;;;  GETTING TRUNCATION INFO
;;;
;;;  These functions get info from the display menu.

(defun set-truncation (disp-blk)
  (let ((cmd (get-trunc-cmd)))
    (case cmd
      ((S) (setf (display-block-truncate? disp-blk) (get-trunc-point)))
      ((T) (setf (display-block-truncate? disp-blk) nil))
      (t (format *Qsim-Report* "~&Improper Input.  No change made to the truncation point")))))

(defun get-trunc-cmd ()
  (format *Qsim-Report* "~&S=Set Truncation Point, T=Turn Off Truncation: ")
  (clear-input)
  (cond ((car (member (read) '(S T))))
	(t nil)))

(defun get-trunc-point ()
  (format *Qsim-Report* "~&Please enter either a time point or a time interval: ")
  (read))





;;; Modified to add a command to eliminate occurrence branching
;;; 15 May 1991  DJC
;;; Modified to handle command entry on the Mac.  The mac does not
;;; implement the read-char unread-car cycle correctly.
;;; The Mac version will simply use read initially and will then handle the
;;; input appropriately.
;;; DJC 21Oct91


(defun GET-DISPLAY-COMMAND (bnum btotal click-p plot-mode)
  (let ((next-bnum (if (or (null bnum) (eql bnum btotal)) 1 (1+ bnum)))
	(return-strings '(C T O Q V P L S I A
		       G   ; BKay 25Sept91
		       B 
		       Z  ; DJC  02/21/92
		       K  ; DJC  03/30/92
		       TR ; DJC  05/1/92
		       CH ; DJC  07/6/92
		       )))
  (loop
    for cmd = (prompt-for-display-command next-bnum btotal click-p plot-mode)
    do (cond ((member cmd (if click-p return-strings (cdr return-strings))
			  :test #'equal)
		  
	      (return cmd))
	     ((member cmd '(N n) :test #'equal)
	      (return next-bnum))
	     ((numberp cmd)
	      (if (or (not (typep cmd 'fixnum))
		      (< cmd 1)
		      (> cmd btotal))
		  (format *Qsim-Report*
			  "~&Invalid behavior number: must be between 1 and ~A."
			  btotal)
		  (return cmd)))
	     (t (format *Qsim-Report*
			"~%Invalid Entry: Try again."))))))




;;; Modified to handle a bug in the Mac Lisp.  See comment in GET-DISPLAY-COMMAND
;;; DJC  21 Oct 91

(defun PROMPT-FOR-DISPLAY-COMMAND (bnum btotal click-p plot-mode)
  (format *Qsim-Report*
	  "~%~A.  Enter~A N=Next beh (~A of ~A), beh number, ~
                                          O=Other commands, K=Symbol Key, CH=Change Output Routing Q=Quit:"
	  (plot-mode-to-string plot-mode)
	  (if click-p
	      #+Symbolics " C=Click for beh," #-Symbolics ""
	      " T=beh Tree,")
	  bnum btotal)
  (clear-input)
  (read))


;;; Converts the input to a character or a number.  If this is not possible,
;;; it sends and error message to the user and the input is re-entered.
;;; This function is only used in the mac version.
;;; DJC  21 Oct 91

#+:ccl
(defun CONVERT-DISPLAY-INPUT (input)
  (cond ((numberp input) input)
	((eql (length (symbol-name input)) 1)
	 (character (symbol-name input)))
	(t (format *Qsim-Report*
		   "~&~A is not a valid entry.  Please re-enter your command. "
		   input))))


(defun GET-BEHAVIOR-NUMBER (cmd btotal)
  "Reads behavior number, echoing number to screen."
  (unread-char cmd)
  (let ((input (read)))
    (if (or (not (typep input 'fixnum))
	    (< input 1)
	    (> input btotal))
	(progn
	  (format *Qsim-Report*
		  "~&Invalid behavior number: must be between 1 and ~A."
		  btotal)
	  nil)
	input)))



(defun GET-TIME-PLOT-LAYOUT (initial)
   (loop for layout = (prompt-for-layout)
	 until (layout-okay-p layout initial)
	 finally (return layout)))


(defun PROMPT-FOR-LAYOUT ()
  (format *Qsim-Report*
	  "~&Enter new layout as list of lists of variables.  Each sublist will be ~
           a row in the display.  Use nils for spacing.~%Layout: ")
  (read))


;;; Check the LAYOUT to see that it is a list of lists of symbols.
;;; Variable L represents an element of the layout; each should be a list.
;;; Variable S represents an element of the L lists; each should be a symbol.
;;; At this time it is still an open question which variables should be
;;; considered "reachable" for a state, mostly because it is not clear which
;;; QDE's should be included when searching for "reachable" QDE's.
;;; This routine punts, checking only that the layout is a list of lists of symbols.
;;; Modified -ptb 10/3/90


(defun LAYOUT-OKAY-P (layout state)
  (declare (ignore state))
  (unless (listp layout)
    (format *Qsim-Report* "~&Input layout ~A is not a list." layout)
    (return-from layout-okay-p nil))
; (let ((var-list (get-var-list-from-qdes)))
    (dolist (L layout t)
      (unless (listp L)
	(format *Qsim-Report* "~&In layout, ~A is not a proper sublist." L)
	(return-from layout-okay-p nil))
      (dolist (S L t)
	(unless (or (null S) (symbolp S))
	  (format *Qsim-Report*
;		  "~&~A is an invalid variable.  The valid variables are: ~A" S var-list)
		  "~&~A is not a variable name." S)
	  (return-from layout-okay-p nil)))))


(defun GET-PHASE-PLOT-LAYOUT (initial prompt-only-p)
  (let* ((pplanes (or (and (null prompt-only-p) (get-phase-planes-from-qde initial))
		      (get-phase-planes-from-prompt initial)))
	 (n (length pplanes))
	 rows
	 cols)
    (cond ((< n 9) (setq rows 2 cols 4))
	  ((< n 19) (setq rows 3 cols 6))
	  (t (setq rows 4 cols 8)))
    (list pplanes rows cols)))


(defun GET-PHASE-PLANES-FROM-QDE (state)
  (let ((pplanes (sim-phase-planes *current-sim*)))	; BJK:  10-10-90
    (if (pplanes-okay-p pplanes state)
	pplanes
	(get-phase-planes-from-prompt state))))


(defun GET-PHASE-PLANES-FROM-PROMPT (state)
  (do* ((pplanes (prompt-for-pplanes) (prompt-for-pplanes))
	(check (pplanes-okay-p pplanes state)
	       (pplanes-okay-p pplanes state)))
       (check
	 (when (y-or-n-p "Do you want to use this as the QDE phase-planes clause? ")
	   (if (assoc 'phase-planes (qde-other (state-qde state)))
	       (rplacd (assoc 'phase-planes (qde-other (state-qde state)))
		       pplanes)
	       (if (qde-other (state-qde state))
		   (nconc (qde-other (state-qde state))
			  (list (cons 'phase-planes pplanes)))
		   (setf (qde-other (state-qde state))
			 (list (cons 'phase-planes pplanes))))))
	 pplanes)))


(defun PROMPT-FOR-PPLANES ()
  (format *Qsim-Report*
	  "~&Enter list of phase planes, each being a list of two phase variables ~
           (nils may be used for spacing out phase plots).~%Phase planes: ")
  (read))


;;; Like LAYOUT-OKAY-P above, this routine makes no attempt to see if the symbols
;;; used in pplanes are reachable.  It simply checks to see that the format of
;;; pplanes is correct.
;;; Modified -ptb 10/3/90


(defun PPLANES-OKAY-P (pplanes state)
  (declare (ignore state))
  (cond ((not (listp pplanes)) 
	 (format *Qsim-Report* "~&Input is not a list.") nil)
	((> (length pplanes) 32) 
	 (format *Qsim-Report* "~&Too many phase planes (> 32).") nil)
	(t
;	 (let ((var-list (get-var-list-from-qdes)))
	 (dolist (plane pplanes t)
	   (cond ((null plane))
		 ((or (not (listp plane)) (not (eql (length plane) 2)))
		  (format *Qsim-Report* "~&~a is not a sublist of length 2." plane)
		  (return nil))
;		 ((or (not (member (first  plane) var-list))
;		      (not (member (second plane) var-list)))
;		  (format *Qsim-Report* "~&Invalid phase variable in ~A.  ~
;			  Valid phase variables are: ~A" plane var-list)
		 ((or (not (symbolp (first  plane)))
		      (not (symbolp (second plane))))
		  (format *Qsim-Report* "~&Invalid phase variable in ~A." plane)
		  (return nil)))))))


(defun GET-VAR-LIST-FROM-STATE (state)
   (mapcar #'(lambda (pair) (car pair))
	   (state-qspaces state)))


(defun GET-VAR-LIST-FROM-QDES ()
  (apply #'append (mapcar #'(lambda (qde) (mapcar #'(lambda (pair) (car pair))
						  (qde-var-alist qde)))
			  (display-block-qdes-in-beh-tree *current-display-block*))))


(defun later-state? (s1 s2)
  "Returns true if state S1 has a later time qmag than S2.  It does not
mean to imply that S1 is a successor of S2"
  (later-time? (qmag (state-time s1))
	       (qmag (state-time s2))))

(defun later-time? (time1 time2 &optional (or-equal nil))
  "Returns t when t1 is later than t2.  It asumes that both
arguments are well-formed time values."
  (when (and time1 time2)
    (let* ((t1 (convert-time-to-string time1))
	   (t2 (convert-time-to-string time2))
	   (both-points? (and
			   (atom t1)
			   (atom t2)))
	   (lower-t1 (if (listp t1)
			 (subseq (car t1) 1)
			 (subseq t1 1)))
	   (upper-t2 (if (listp t2)
			 (subseq (cadr t2) 1)
			 (subseq t2 1))))
      (cond ((and both-points? or-equal)
	     (string>= lower-t1 upper-t2))
	    ((and both-points?)
	     (string> lower-t1 upper-t2))
	    (or-equal
	     (or (string>= lower-t1 upper-t2)
		 (equal t1 t2)))
	    (t (string>= lower-t1 upper-t2))))))

(defun convert-time-to-string (time-qmag)
  "Converts various formats for a time-qmag to strings"
  (let* ((time-int? (listp time-qmag))
	 (qmag-list (if time-int? time-qmag (list time-qmag)))
	 (converted
	   (cond ((symbolp (car qmag-list))
		  (mapcar #'symbol-name qmag-list))
		 ((stringp (car qmag-list))
		  qmag-list)
		 (t (mapcar #'(lambda (lm)
				(symbol-name (lmark-name lm)))
			    qmag-list)))))
    (if time-int? converted (car converted))))




;;;-----------------------------------------------------------------------------
;;; Functions to handle abstraction levels
;;;-----------------------------------------------------------------------------

(defun get-item-in-list (list)
  (loop with total = (length list)
	for input = (read)
	until (not (or (not (typep input 'fixnum))
		       (< input 1)
		       (> input total)))
	do (format *Qsim-Report*
		   "Invalid number: must be between 1 and ~A.~%"
		   total)
	finally (return (nth (1- input) list))))


(defun change-description-level ()
  (let ((levels (display-block-levels *current-display-block*)))
    (prompt-for-new-description-level levels)
    (setq *develop-graph* nil)
    (setq *abstraction-level* (get-item-in-list levels))))


(defun prompt-for-new-description-level (levels)
  (format *qsim-report* "~2%Select one of the following description levels:~%")
  (do* ((levels levels (cdr levels))
	(level (car levels) (car levels))
	(n 1 (1+ n)))
       ((null levels) t)
    (format *qsim-report* " ~3a: ~a~%" n level)))


(defun select-and-apply-focusing-technique ()
  (format *qsim-report* "~2%Focus on: 1=Qmags, 2=Interesting Variables, 3=Histories~%")
  (let ((focus (get-item-in-list '(qmags variables histories))))
    (setq *abstraction-level*
	  (ecase focus
	    (qmags (focus-on-qmags))
	    (variables (focus-on-variables (get-interesting-variables)))
	    (histories (focus-on-histories))))))

(defun get-interesting-variables ()
  (format *Qsim-Report*
	  "~%Enter a list of interesting variables:~%")
  (read))

(defun focus-on-histories ()
  (not-implemented-warning)
  *abstraction-level*)

(defun not-implemented-warning ()
  (format *qsim-report* "~%Not Implemented Yet!"))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                   ;;;
;;;   SELECTIVE-CONTINUE                                              ;;;
;;;                                                                   ;;;
;;;                                                                   ;;;
;;;  This function provides the user with the ability to specify      ;;;
;;;  specific behaviors for which the simulation should be extended.  ;;;
;;;  It prompts the user for a list of behaviors as well as the       ;;;
;;;  number of additional states which should be simulated.           ;;;
;;;  Those states on the agenda which are not in one of the           ;;;
;;;  specified behaviors are removed from the agenda on the           ;;;  
;;;  SIM and place on a new slot called -temp-susp-agenda.            ;;;
;;;  Once the simulation is complete, the states which                ;;;
;;;  were removed are restored.                                       ;;;
;;;                                                                   ;;;
;;;  DJC  21Feb92
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;




(defun selective-continue (&optional (initial-state-or-states *initial-state*))
  (let* ((sim (state-sim (if (listp initial-state-or-states)
			     (car initial-state-or-states)
			     initial-state-or-states)))
	 (final-states (get-last-states initial-state-or-states))
	 (btotal (length final-states))
	 (bnums (make-list-of-bnums (specify-behs-by-bnum btotal)))
	 (states-to-sim (mapcar #'(lambda (bnum)
				    (nth (1- bnum) final-states))
				bnums))
	 (inc-state-limit (when bnums (get-number-extra-states (sim-state-limit sim)))))
    (when (and bnums inc-state-limit)
      (let* ((agenda (sim-agenda sim))
	     (new-agenda (intersection states-to-sim agenda :test #'equal))
	     (suspended-states (set-difference agenda new-agenda :test #'equal)))
	(cond ((null new-agenda)
	       (format *qsim-report* "~&  These behaviors have been completely simulated.  They ~
                                      ~%  cannot be extended."))
	      (t (unwind-protect
		     (progn
		       (setf (sim-agenda sim) new-agenda
			     (sim--temp-susp-agenda sim) suspended-states)
		       (q-continue :new-state-limit (+ inc-state-limit (sim-state-limit sim))
				   :sim sim))
		   (progn
		     (setf (sim-agenda sim)
			   (append (sim--temp-susp-agenda sim)
				   (sim-agenda sim))
			   (sim--temp-susp-agenda sim) nil)
		     ;; Reset the state limit to the state count in case it
		     ;; wasn't reached
		     (when (> (sim-state-limit sim) (sim-state-count sim))
		       (setf (sim-state-limit sim) (sim-state-count sim)))))))))
    sim))


(defun get-last-states (states-or-state)
  "This function will return the final states in each branch of a 
   behavior tree."
  (let ((states (if (listp states-or-state)
		    states-or-state
		    (list states-or-state))))
    (mapcan #'(lambda (state)
		(let ((successors (successor-states state)))
		  (cond ((null successors)
			 (list state))
			(t (mapcan #'get-last-states successors)))))
	    states)))

(defun get-number-extra-states (cur-state-limit)
  "Prompts the user for the number of states to extend the simulation by."
  (format *qsim-report* "~&The current state limit is: ~a ~
                         ~%Increase the state limit by how many states? " cur-state-limit)
  (clear-input)
  ;(setf tmp (read-line))
  (loop for inp = (read)
	when (numberp inp)
	return inp
	do (format *qsim-report* "~&Input is not a number.  Try again: ")))

