;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-

;;; $Id: numeric-plot.lisp,v 1.13 92/07/09 12:46:54 clancy Exp $

(in-package 'QSIM)

;;; This file contains the functions used for drawing the numeric plots of
;;; qsim behaviors.  Presently, this is just numeric time plots, but
;;; it could ultimately include phase plots as well.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; QSIM-DISPLAY interface functions.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;---------------------------------------------------------------------------
;;; This function is called from the Displays function when a numeric time
;;; plot is needed.
;;; Inputs:  behaviors - A list of behaviors.
;;;          layout    - A screen layout (from the display-block).
;;;          btotal    - Total number of behaviors.
;;;          bnum      - The behavior number.
;;; Returns: nothing.
;;; Notes: btotal and bnum are only used for plotting the graph label.
;;;---------------------------------------------------------------------------
;;;
(defun NUMERIC-TIME-PLOT-BEHAVIOR
       (behaviors layout layout-options btotal bnum)
  (qplot-new-behavior)
  (qplot-label (nth (1- bnum) behaviors) bnum btotal)
  (qplot-time-qgraph (nth (1- bnum) behaviors) 
		     layout layout-options)
  (qplot-end-display))


;;; This variable is used to hold the qgraphs in the current display.
;;; It should probably be in the display-block instead.
;;; It is used so that the save function will know where to find the qgraphs.
;;;
(defvar *current-qgraphs* nil)


;;;---------------------------------------------------------------------------
;;; Figure out what to plot.
;;; This version is wasteful in that new qgraph structures are computed for each
;;; call even if we have already displayed this behavior.  This costs us extra
;;; qgraph structures and time to compute bounds and window layout.
;;; Inputs:  beh             - The behavior.
;;;          layout          - The variable layout.
;;;          layout-defaults - A list of ((option val) ...) for the options
;;; Returns: nothing.
;;;---------------------------------------------------------------------------
;;;
(defun qplot-time-qgraph (beh layout layout-options)
  (setf *current-qgraphs* nil)
  ;; *xsize* and *ysize* set by compute-layout call in switch-plot-style
  (do ((rows layout (cdr rows))
       (ypos tmargin (+ ypos *ysize* ysep)))
      ((null rows))
    (do ((varnames (car rows) (cdr varnames))
         (varname nil)
         (xpos lmargin (+ xpos *xsize* xsep)))
        ((null varnames))
      (setq varname (car varnames))
      (if varname
	  (qgraph-structure
	   (car (push
		 (make-qgraph-from-behavior beh
					    varname
					    (lookup-set varname
							layout-options))
		 *current-qgraphs*))
	   :xsize *xsize* :ysize *ysize*
	   :xpos xpos :ypos ypos)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Numeric graph control functions.
;;;
;;; These functions are used in user-written simulation functions to
;;; set the numeric layout and the graphing options for numeric time plots.
;;; The user accessible-functions are set-numeric-graph-options and
;;; set-numeric-layout.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;---------------------------------------------------------------------------
;;; Set the numeric-time-plot-options slot of the display-block.
;;; Inputs:  state-or-sim-or-db - A qsim state or sim structure or
;;;                               display-block.
;;;          options      - A list of (var optionlist) where var is a variable
;;;                         name or * (which stands for "all vars") and
;;;                         optionlist is a list of (option val) entries.
;;; Returns: nothing.
;;; Notes:   The display-block.numeric-time-plot-options slot has an
;;;          entry for each variable in the display.  New vars get their
;;;          options from the display-block.numeric-time-plot-defaults slot.
;;;---------------------------------------------------------------------------
;;;
(defun set-numeric-graph-options (state-or-sim-or-db options)
  (let ((disp-blk (cond ((display-block-p state-or-sim-or-db)
			  state-or-sim-or-db)
			 (T
			  (sim-display-block
			   (if (state-p state-or-sim-or-db)
			       (state-sim state-or-sim-or-db)
			       state-or-sim-or-db))))))
    (dolist (var-options options)
      (cond
	;; * means "set default option"
	((eq (car var-options) '*)
	 ;; Set the option for all existing vars and the default
	 (dolist (var-entry (display-block-numeric-time-plot-options disp-blk))
	   (set-numeric-graph-option var-entry (cdr var-options)))
	 ;; Set the defaults as well
	 (set-numeric-graph-default-option disp-blk (cdr var-options)))
	(T
	 (let ((var-entry (assoc (car var-options)
				 (display-block-numeric-time-plot-options disp-blk))))
	   (when (null var-entry)
	     ;; Add a var-entry for this fella if he's new.
	     (setf var-entry
		   (car
		    (push (cons (car var-options)
				(display-block-numeric-time-plot-defaults disp-blk))
			  (display-block-numeric-time-plot-options disp-blk)))))
	   (set-numeric-graph-option var-entry (cdr var-options))))))))


;;; Set the default numeric options for a display-block.
;;;
(defun set-numeric-graph-default-option (disp-blk options)
  (dolist (option options)
    (let ((option-entry (assoc
			 (car option)
			 (display-block-numeric-time-plot-defaults disp-blk))))
      (cond
	(option-entry
	 (setf (second option-entry) (second option)))
	(T
	 (setf (display-block-numeric-time-plot-defaults disp-blk)
	       (cons option
		     (display-block-numeric-time-plot-defaults disp-blk)))))
      )))


;;; Set the options for a single variable-entry
;;;
(defun set-numeric-graph-option (var-entry options)
  (dolist (option options)
    (let ((option-entry (assoc (car option)
			       (cdr var-entry))))
      (cond
	(option-entry
	 (setf (second option-entry) (second option)))
	(T
	 (setf (cdr var-entry)
	       (cons option (cdr var-entry))))))))


;;;---------------------------------------------------------------------------
;;; Set the numeric-layout slot of the display-block.
;;; Inputs:  state-or-sim-or-db - A qsim state or sim structure or
;;;                               display-block.
;;;          layout             - A layout list.
;;; Returns: nothing.
;;; Notes:   The display-block.numeric-time-plot-layout is updated.
;;;---------------------------------------------------------------------------
;;;
(defun set-numeric-layout (state-or-sim-or-db layout)
  (let* ((disp-blk (cond ((display-block-p state-or-sim-or-db)
			  state-or-sim-or-db)
			 (T
			  (sim-display-block
			   (if (state-p state-or-sim-or-db)
			       (state-sim state-or-sim-or-db)
			       state-or-sim-or-db))))))
    ;; Add a default setting for each new var in the layout.
    (dolist (row layout)
      (dolist (var row)
	(when var
	  (when (not
		 (assoc var
			(display-block-numeric-time-plot-options disp-blk)))
	    (setf (display-block-numeric-time-plot-options disp-blk)
		  (cons
		   (cons var
			 (display-block-numeric-time-plot-defaults disp-blk))
		   (display-block-numeric-time-plot-options disp-blk)))))))
    ;; Set the layout to a new value.
    (setf (display-block-numeric-time-plot-layout disp-blk) layout)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Numeric viewer command menu functions.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Menu for changing numeric time plot properties.
;;; 
(defun do-numeric-command (disp-blk initials)
  (let* (cmd)
    (format *Qsim-Report* "~&Change numeric plotting info.  Enter L=Change numeric layout, ~%O=Change graphing options, S=Save Qgraph file, Q=Quit : ")
    (clear-input)
    (setf cmd (read-char))
    (case cmd
      ((#\L #\l)  (modify-layout initials NIL NIL disp-blk 'numeric-time-plot))
      ((#\O #\o)  (change-numeric-graph-options disp-blk))
      ((#\S #\s)  (save-qgraph-plot disp-blk))
      ((#\Q #\q)  NIL)
      (T          
       (format *Qsim-Report* "~&~A is an invalid command.  Back to the basic prompt." cmd)))
    (list nil nil) ; Needed for do-other-command return.
    ))



;;; Change numeric options from the viewer menu entry.
;;;
(defun change-numeric-graph-options (disp-blk)
  (do ((options (display-block-numeric-time-plot-options disp-blk))
       option-list)
      (nil)
    (format *Qsim-Report* "~&Current options are: ~a~&Enter a list of the ~
          form~%~
          ((var (opt1 val1) (opt2 val2) ...) ...) or NIL to quit:~%"
	    options) 
    (clear-input)
    (setf option-list (read))
    (when (null option-list) (return NIL))
    (if (numeric-options-okay-p option-list)
	(set-numeric-graph-options disp-blk option-list)
        (format *Qsim-Report* "~&That list is incorrectly specified.~%~
                It should be a list of the form ((var (opt1 val1) ...) ...)"))
    ))


(defun numeric-options-okay-p (options)
  (and (listp options)
       (every #'(lambda (x) (and (listp x)
				 (atom (car x))
				 (every #'listp (cdr x))))
		  options)))

(defun save-qgraph-plot (disp-blk)
  (declare (ignore disp-blk))
  (let ()
;    (format *Qsim-Report* "~&Please enter a directory name: ")
;    (clear-input)
;    (setf dir (read))
;    (format *Qsim-Report* "~&Each graph in the last plotted display will be~%~
;            stored in a file var.q where var is the name of the variable.")
    (format *Qsim-Report* "~&This option is not yet implemented.~%~
            The variable *current-qgraphs* contains a list of current qgraphs.")))





;;;---------------------------------------------------------------------------
;;; This almost certainly doesn't belong here!
;;; It should probably be in states.lisp
;;;
;;; Make a dummy state for the purposes of numsim simulation.
;;; Inputs:  same as make-new-state
;;; Returns: An incomplete state from which a numsim can be hung.
;;;---------------------------------------------------------------------------
;;;
(defun make-plain-state (&rest keys)
  (apply #'make-new-state :completions NIL :allow-other-keys t keys))


;;;---------------------------------------------------------------------------
;;; Plot the monotonic envelopes for a QDE.
;;; Inputs:  qde-or-state  - A qde or a state.
;;;          env-con       - The envelope constraint.
;;;          envelopes     - A list of the envelope functions (this is a set of
;;;                          upper-envelope upper-inverse lower-envelope
;;;                          lower-inverse expected-function expected-inverse)
;;;          start         - The starting value for X in (M+ A X)
;;;          stop          - The ending value for X.
;;;          step          - X increment.
;;;          options       - A list of display options.
;;; Returns: nothing
;;;---------------------------------------------------------------------------
;;;
(defun graph-envelope (qde-or-state env-con &key
				    (envelopes '(upper-envelope
						 lower-envelope))
				    (start 0) (stop 10) (step .1)
				    (options NIL))
  (declare (ignore options))
  (let* ((qde (if (state-p qde-or-state) (state-qde qde-or-state)
		                         qde-or-state))
	 (m-env-clauses (when qde (lookup-set env-con (qde-m-envelopes qde)
					      :test #'equal)))
	 (datasets NIL))
    (when (null m-env-clauses)
      (error "No m-envelope clause for ~a in qde ~a" env-con qde))
    (dolist (env m-env-clauses)
      (when (member (first env) envelopes)
	(push (compute-envelope (second env) start stop step)
	      datasets)))
    (when (null datasets)
      (error "No ~a envelopes for ~a in qde ~a" envelopes env-con qde))
    (qgraph-display (list (list (make-qgraph :data datasets)))
		    :text (list
			   (format nil "Envelope plot for ~a from ~a"
				   env-con qde)
			   (format nil "Envelopes are ~a" envelopes)))))



(defun compute-envelope (fctn start stop step)
  (do ((x start (+ x step))
       (dataset (make-line :data (make-q))))
      ((>= x stop) dataset)
    (qpush (list x (funcall fctn x)) (line-data dataset))))
				    

