;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)


;;; NOTICE: this is a VARIANT version of QPLOT-PRIMITIVES that removes
;;; all calls to the PostScripting facility.  If you are having severe
;;; difficulty with the POS system, comment out the reference to the POS
;;; system in the system definition file; replace QPLOT-PRIMITIVES with
;;; this file, and recompile everything.



;  DEVICE DEPENDENT PLOTTING ROUTINES.
;    These functions may send their output to the screen, a postscript file, or both.

(defparameter *qplot-output* t)			; screen output stream (set on initialization)

;    All device-dependent functions should round their arguments.

(defparameter axis-font				; for labeling axis points
              '(:fix nil :small)		;   Symbolics version

              )
(defparameter label-font			; for labeling individual plots
	      '(:fix :bold :large)		;   Symbolics version

	      )

(defparameter *symbol-x-offset* -3)		; peculiar positional offsets to get
(defparameter *symbol-y-offset* +4)		; symbols in the right place.
 
(defparameter xscreen 1037.)                    ; horizontal screen size
(defparameter yscreen 700.)                     ; vertical screen size

(special *postscript-output-file*  bmargin)

(defparameter *black* tv:alu-seta)
(defparameter *white* tv:alu-andca)
 
;;; *Postscript-Bound* is a flag to test whether we already have opened
;;; a postscript output file.  Many of the higher level plotting
;;; routines call each other recursively - the *postscript-bound* flag
;;; stops us from dumping to multiple files simultaneously.

(defvar *postscript-bound* nil)

;;; DEVICE-INTERFACE is a macro that handles directing the output of the
;;; simulation to the screen, a postscript file, or both.
(defmacro device-interface (&rest plotting-exp)
  plotting-exp)

;;; Each output device will have a different set of tasks to do when starting a new behavior

(defun qplot-new-behavior (&key (box nil))
  (setq *qplot-output* *standard-output*	;  Symbolics version
;	               *the-standard-output*	;  Explorer version
	)
  (send *qplot-output* :new-behavior))

(defmethod (:new-behavior dw::dynamic-lisp-listener)
	   ()
  (send self :clear-history))

(defmethod (:new-behavior tv:lisp-listener)
	   ()
  (send self :clear-window))

(defun qplot-end-display ()
  ;; BMARGIN, the bottom margin, is usally 50 pixels.
  (send *qplot-output* :set-cursorpos
	0 (round (- yscreen bmargin))))

; Basic drawing:  dot, horizontal and vertical lines, heavy box.

(defun qplot-dot (x y &key (alu *black*))
  (graphics:draw-point (round x) (round y)
		       :alu alu :stream *qplot-output*))

(defun qplot-hline (x y length &key (alu *black*)(thickness 1))
  (graphics:draw-line (round x) (round y) (round (+ x length)) (round y)
		      :alu alu :stream *qplot-output*
		      :thickness		; Thicknesses below 1 pixel won't draw.
		      (max 1 (round thickness))))

(defun qplot-vline (x y length &key (alu *black*)(thickness 1))
  (graphics:draw-line 
    (round x) (round y) (round x) (round (+ y length))
    :alu alu :stream *qplot-output*
    :thickness					; Thicknesses below 1 pixel won't draw.
    (max 1 (round thickness))))

(defun qplot-line (x1 y1 x2 y2 &key (alu *black*)(thickness 1))
  (graphics:draw-line (round x1)
		      (round y1) (round x2) (round y2)
		      :stream *qplot-output*
		      :thickness		; Thicknesses below 1 pixel won't draw.
		      (max 1 (round thickness))))

(defun qplot-lines (&rest args)			; (car args) is an alu.
  (apply #'graphics:draw-lines 
	 (cons (mapcar 'round (car args))
	       (append '(:stream *qplot-output*)
		       (cdr args)))))

(defun qplot-dashed-line (x1 y1 x2 y2  &key (alu *black*)(thickness :thickness))
  (graphics:draw-line
    (round x1)(round y1)(round x2)(round y2)
    :alu alu :thickness thickness :stream *qplot-output*
    :dashed t))

(defun qplot-string (string x y &key (alu *black*))
  (graphics:draw-string string (round x) (round y)
			:alu alu :stream *qplot-output*))

(defun qplot-circle (x y radius &key (alu *black*) (filled nil))
						; tolerates floating point
  (graphics:draw-circle  x  y radius :alu alu	
			 :stream *qplot-output* :filled filled))


(defun qplot-ring (x y minor-r major-r &key (alu *black*))
  (when (< major-r minor-r)
    (Error "Wrong size radii"))
  (send *qplot-output* :draw-ring
	(round x)(round y) (round minor-r)(round major-r) alu))

(defun qplot-cubic-spline (points  &key (alu *black*)(thickness 1)
			   (start-relaxation :relaxed)
			   &allow-other-keys)
  (graphics:draw-cubic-spline			; tolerates floatingpt
    points :alu alu :thickness thickness :stream *qplot-output*
    :start-relaxation start-relaxation))

(defun qplot-polygon (points &key (alu *black*)(filled t)(thickness 1))
  ;; graphics:draw-polygon tolerates floatingpoint
  (graphics:draw-polygon points :alu alu :filled filled))

;; Draw a box, with (default) linethickness of 2.

(defun qplot-box (xpos ypos xsize ysize &key (alu *black*)(thickness 2))
  (graphics:draw-rectangle			; left top right bottom
    (round xpos) (round ypos) (round (+ xpos xsize))(round  (+ ysize ypos))
    :thickness thickness :filled nil :stream *qplot-output* :alu alu))


; Special symbols.  If you add any new symbols to this list; they must be added to 
; the list of exported functions from the POS package, (the parameter *EXPORTS*).

(defun qplot-symbol (x y symbol &key (alu *black*))
  (let ((xloc (round (+ x *symbol-x-offset*)))
	(yloc (round (+ y *symbol-y-offset*))))
    (case symbol
      (std (graphics:draw-circle
	     x y 2 :filled nil :stream *qplot-output* :alu alu))
      (inc (graphics:draw-string
	     "" xloc yloc :stream *qplot-output* :alu alu))
      (dec (graphics:draw-string
	     "" xloc yloc :stream *qplot-output* :alu alu))
      (ign (graphics:draw-string
	     "*" xloc yloc :stream *qplot-output* :alu alu))
      (left (graphics:draw-string
	      "" xloc yloc :stream *qplot-output* :alu alu))
      (right (graphics:draw-string
	       "" xloc yloc :stream *qplot-output* :alu alu))
      (t (graphics:draw-string "?" xloc yloc
			       :stream *qplot-output* :alu alu))
      )))

(defun qplot-special-char (x y char ascii &key (size 12)
			   (char-info '(nil nil nil))(alu *black*))
  (graphics:draw-string
    (format nil "~v~a~u" char-info char)
    x y :stream *qplot-output* :alu alu))


(defun qplot-vector (from-x from-y to-x to-y
		     &key (alu *black*)
		     (arrow-head-length 10) (arrow-base-width 5)
		     (filled nil)
		     (shaftthick 1))
  (graphics:draw-arrow  (round from-x) (round from-y) (round to-x) (round to-y)
			:arrow-head-length (round arrow-head-length)
			:arrow-base-width (round arrow-base-width)
			:alu alu :stream stream))



;; Plot a label for a behavior on the screen.  X,Y will specify the
;; starting point for the string, just below the left hand side of the box.

(defun qplot-box-label (x y string &optional ignore)
  (qplot-string
    (merge-font-info label-font string)
    x y :alu *flip*))				

;; Put a label for a landmark value on the right hand side of the graph,
;; placing it on the Y-axis.

(defun qplot-axis-label (x y string &optional ignore)
  (qplot-string
    (merge-font-info axis-font string)
    x y :alu *flip*))



;  Modified version of these routines can produce LaTeX or PostScript commands.

;  END OF DEVICE DEPENDENT PLOTTING ROUTINES.

