;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package 'QSIM)

; This sections lays out the screen, allocating space for each qualitative plot.

; A layout is a list of rows, where each row is a list of variables whose plots
; go in that row.  NIL will leave a blank space.
; The layout is an attribute of the structure description.

; The following translates the abstract layout descriptions into screen regions.

(defun compute-layout (layout)
  (compute-rectangular-layout (length layout)
			      (apply #'max (mapcar #'length layout))))

;;; Compute the width and height of a single paramter plot.

(defun compute-rectangular-layout (nrows ncols)
  (setq *xsize* (- (round (/ (+ (- xscreen lmargin rmargin) xsep)
			   ncols))
		 xsep))
  (setq *ysize* (- (round (/ (+ (- yscreen tmargin bmargin) ysep)
			   nrows))
		 ysep)))
;;; Try and find a layout - if none is available, find all the variables
;;; of all the qdes in the behavior, and arrange them in alphabetical
;;; order in rows of four.

(defun find-some-layout (behaviors refstates)
  (or (qde-layout (state-qde (caar behaviors)))
      (allvars-layout (qdes-of-behs behaviors refstates))))

(defun allvars-layout (qdes)
  (let* ((vars (remove-duplicates
		 (loop for qde in qdes
		       nconc (mapcar #'car (qde-var-alist qde)))))
	 (avars (sort vars #'string<)))
    (unless *include-time-in-layout*
      (setf avars (remove 'time avars)))
    (loop for avartail on avars by #'(lambda (list)
				       (nthcdr *columns-for-layout* list))
	  collect (loop for n from 0 below *columns-for-layout*
			collect (nth n avartail)))))

(defun qdes-of-behs (behaviors refstates)
  (loop with qdes = (remove-duplicates
		      (mapcar #'state-qde (mapcar #'second refstates)))
	for beh in behaviors
	do (loop for state in beh
		 do (pushnew (state-qde state) qdes))
	finally (return qdes)))
                    
;;; The allocation consists of an alist of (state xcoord), where the
;;; left edge of the plot is assumed to have xcoord=0.  The x values
;;; generated here will be added to the xpos values passed to
;;; plot-parameter to generate the absolute x coordinate of the plotted
;;; symbols.

;;; Possibly confusing detail: the horizontal (x) positions of the
;;; labelings for the vertical axis are figured here - that's why
;;; vtickshare, vlabelshare and vquantshare are set by allocate-h-axis.

;;; Modified to display aggregate intervals in the plot
;;; 15 May 1991  DJC

(defun allocate-h-axis (refpoints behavior xsize)
  (let* ((nortgaps (count-ortgaps behavior))
	 (totalshares nil)
	 (rbshare (if refpoints rbgap 0.0)))
    (set-vquantshare behavior)			;modified 10/17/89 by D.B.
    (setq refshare (if refpoints
		       (- (length refpoints) 1.0) 0.0))
    (setq behshare (+ (- (length behavior) nortgaps 1.0)
		      (size-of-agg-histories-in-beh behavior)))   ; added DJC
    (setq totalshares (+ lshare
			 refshare
			 rbshare
			 (* ortgap Nortgaps)
			 behshare
			 vtickshare
			 vlabelshare
			 vquantshare))
    (setq rshare (+ vtickshare vlabelshare vquantshare))
    (setq Hshare (/ xsize totalshares))
    (setq vtickpoint (- xsize (* Hshare rshare)))	
    (setq vlabelpoint (- xsize (* Hshare (+ vlabelshare vquantshare))))
    (append (refpoint-coords refpoints)
	    (reverse (allocate-behavior behavior
			 (* Hshare (+ lshare refshare rbshare)))))))



(defun refpoint-coords (refpoints)
  (loop for (name state) in refpoints
	for x from (* lshare Hshare) by hshare
	collect (list state x)))

(defun allocate-behavior (behavior leftTick)
  (cond ((allocate-behavior-with-hybrid-scaling      behavior leftTick))
	((allocate-behavior-with-qualitative-scaling behavior leftTick))
	(t (error "No usable allocation method for h axis"))))

;;; Eventually will do a numerical allocation of the time axis when possible,
;;; and without choking on the INF lmark (hence "hybrid" scaling).
(defun allocate-behavior-with-hybrid-scaling (behavior leftTick)
  behavior leftTick nil)

;(defun allocate-behavior-with-qualitative-scaling (behavior leftTick)
;  (do* ((L behavior (cdr L))			; allocate behavior points
;	(x  leftTick  (+ x (cond ((transition-precedes (car L)) (* Hshare ortgap))
;				 (t Hshare))))
;	(coords  (list (list (car L) (round x)))  (cons (list (car L) (round x)) coords)))
;       ((null (cdr L)) coords)))

;;; Modified to handle the existence of aggregate-intervals in
;;; the behavior tree.
;;;  15 May 1991 DJC

(defun allocate-behavior-with-qualitative-scaling (behavior leftTick)
  (do* ((L behavior (cdr L))			; allocate behavior points
	(state (car L) (car L))
	(x  leftTick  (+ x (cond (transition-p (* Hshare ortgap))
				 (prev-agg-interval 
				  (* Hshare (max-history-size prev-agg-interval)))
				 (t Hshare))))
	;; if the previous state in the behavior is an aggregate-interval
	;; then leave enough space for all of the values in the history
	(prev-agg-interval (when (typep state 'aggregate-interval)
			     state)
			   (when (typep state 'aggregate-interval)
			     state))
	(transition-p (transition-follows state (cadr L))
		      (transition-follows state (cadr L)))
	(coords  (list (list state (round x)))  (cons (list state (round x)) coords)))
       ((null (cdr L)) coords)))


(defun set-vquantshare (behavior)		;modified 10/17/89 by D.B.
  (setq vquantshare
	(if (lookup '*bindings* (state-other (car (last behavior))))
	    1.0
	    0.0)))



(defun count-ortgaps (behavior)
  (do ((L behavior (cdr L))
       (count 0))
      ((null L) count)
    (cond ((and (transition-follows (car L))
		(cdr L))
	   (setq count (+ count 1))))))


(defun allocate-v-axis-with-qualitative-scaling (qspace ysize)
  (let ((totalshares nil)
	(coords nil)
	(vlength (- ysize labelspace))) ; height of V axis
    (setq qsshare (float (* 2 (- (length qspace) 1))))
    (setq totalshares (+ tshare qsshare bshare))
    (setq *VshareSize* (/ vlength totalshares))
    (setq htickpoint (- vlength (* htickprop *VshareSize*)))
    (setq hlabelpoint (- vlength (* hlabelprop *VshareSize*))) 
    ;
    (do ((L (reverse qspace) (cdr L))
	 (y (* *VshareSize* tshare)
	    (+ y (* 2.0 *VshareSize*))))
	((null L))
      (setq coords (cons (list (car L) (round y))
			 coords)))
    ;
    (reverse coords)))