;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)


; Copyright (c) 1987 by Benjamin Kuipers.
;
; We use the CFILTER algorithm to generate all solutions to a set of
; constraint equations, using the DKB {+ 0 -} representation for qualitative
; values.  There are three types of solutions:
;  [X]o :  sign wrt zero                                   DKB-solve0
;  [X]* :  sign wrt reference value X*                     DKB-solve*
;  dX = [dX/dt]o :  magnitude + direction of change.       DKB-solve-d

; Later:  add quasi-equilibrium assumption explicitly.
;         do propagation before cfilter.

; The "solve0" version checks whether the sign is consistent with the qspace
; of each parameter.

(defun DKB-solve0 (qde givens)
  (cfilter givens (qde-qspaces qde) (qde-constraints qde)
	   #'meaningful-dkb-qvalues #'check-dkb-constraint))

; The "solve*" version takes a state, and determines what perturbations are
; possible around that state, eliminating perturbations that pass out of the
; qspace from the state.

(defun DKB-solve* (state givens)
  (cfilter givens (state-qspaces state) (state-constraints state)
	   #'(lambda (qspace given-values)
	       (meaningful-dkb-perturbations qspace
					     (qmag (lookup (car qspace)
							   (state-values state)))
					     given-values))
	   #'check-dkb-constraint))

; The "solve-d" version takes a state, and determines what perturbations,
; including slow changes to the equilibrium state, are possible.

(defun DKB-solve-d (state givens)
  (cfilter givens (state-qspaces state) (state-constraints state)
	   #'dkb-qvalue-pairs #'check-dkb-constraint))

(defun basic-qvalues (ignore given-values)
  (or given-values
      '(+ 0 -)))

(defun meaningful-dkb-qvalues (qspace given-values)
  (cond (given-values)
	((eql 0 (car (cadr qspace))) '(0 +))
	((eql 0 (car (last (cadr qspace)))) '(0 -))
	(t '(+ 0 -))))

(defun meaningful-dkb-perturbations (qspace reference given-values)
  (cond (given-values)
	((eql reference (car (cadr qspace))) '(0 +))
	((eql reference (car (last (cadr qspace)))) '(0 -))
	(t '(+ 0 -))))

; Generate qvalues with qdirs.

(defun dkb-qvalue-pairs (ignore given-values)
  (cond (given-values)
	(t (add-axis '(+ 0 -)
		     (add-axis '(inc std dec)
			       (list nil))))))


; A more general DKB constraint checker for qualitative values.
; Branches to handle atomic tuples like (+ + +), in addition to binary
; tuples like ((+ inc) (+ inc) (+ inc)).

(defun check-dkb-constraint (tuple constraint ignore)
  (cond ((not (member (constraint-type constraint) *known-constraint-types*)) nil)
	((member (constraint-type constraint) '(d/dt d//dt)) t)	; ignore d/dt constraint
	((atom (car tuple))
	 (atomic-values-satisfy-constraint tuple constraint))
	((eql (constraint-type constraint) 'MULT)
	 (test-dkb-mult tuple))
	((and (test-dkb-qmags tuple constraint)
	      (test-dkb-qdirs tuple constraint)))))

(defun atomic-values-satisfy-constraint (tuple constraint)
  (let* ((ctype (constraint-type constraint))
	 (legal-values (get-relation ctype)))
    (cond ((eql ctype 'MULT) (assoc tuple MULT-relation :test #'equal))
	  ((null legal-values) t)		; no table -> OK
	  ((member tuple legal-values :test #'equal) t)
	  (t nil))))

(defun test-dkb-qmags (tuple constraint)
  (let* ((ctype (constraint-type constraint))
	 (qmag-tuple (mapcar #'car tuple))
	 (legal-values (get-relation ctype)))
    (cond ((null legal-values) t)		; no table -> OK
	  ((member qmag-tuple legal-values :test #'equal) t)
	  (t nil))))

(defun test-dkb-qdirs (tuple constraint)
  (let* ((ctype (constraint-type constraint))
	 (qdir-tuple (mapcar #'cadr tuple))
	 (legal-values (get-qdir-relation ctype)))
    (cond ((null legal-values) t)		; no table -> OK
	  ((member qdir-tuple legal-values :test #'equal) t)
	  (t nil))))

(defun test-dkb-mult (tuple)
  (let* ((qmag-tuple (mapcar #'car tuple))
	 (qdir-tuple (mapcar #'cadr tuple))
	 (legal-qdir-table (cadr (assoc qmag-tuple MULT-relation :test #'equal))))
    (cond ((null legal-qdir-table) nil)
	  ((member qdir-tuple legal-qdir-table :test #'equal) t))))

; Show DKB solutions in a tabular format

(defun show-solutions (starting-point type givens solutions)
  (format *QSIM-Trace* "~%Solve (type ~a):  " type)
  (format *QSIM-Trace* "~a" (cond ((eql type '0) (qde-name starting-point))
		       ((member type '(* d)) (state-name starting-point))
		       (t starting-point)))
  (format *QSIM-Trace* "~% Params ~10TGiven: ~19TSolutions(~d):" (length solutions))
  (do ((params (cond ((eql type 0) (qde-qspaces starting-point))
		     ((member type '(* d)) (state-qspaces starting-point)))
	       (cdr params))
       (p nil))
      ((null params) t)
    (setq p (caar params))
    (format *QSIM-Trace* "~% ~a:" p)
    (if (lookup p givens)
	(format *QSIM-Trace* "~11T~a" (lookup p givens)))
    (format *QSIM-Trace* "~20T")
    (do ((L solutions (cdr L))
	 (N 0 (+ N 1)))
	((null L) t)
      (if (lookup p (car L))
	  (format *QSIM-Trace* "~3@T~a" (lookup p (car L))))))
  )


; test example:  bathtub and simple pressure regulator, +0- semantics.

(define-QDE ch-regulator
  (text "DKB pressure regulator from Q-E chapter")
  (quantity-spaces
    (Pin   (0 inf))
    (Pout  (0 inf))
    (P     (minf 0 inf))
    (Q     (minf 0 inf))
    (R     (0 inf)))
  (constraints
    ((add Pout P Pin))
    ((M+ Pout Q))
    ((M+ Pout R))
    ((mult Q R P)))
  )

(define-QDE ch-bathtub
  (text "Bathtub from Q-E chapter")
  (quantity-spaces
    (amount (0 FULL inf))
    (level  (0 TOP inf))
    (pressure (0 inf))
    (outflow  (0 inf))
    (drain    (0 OPEN inf))
    (inflow   (0 IF* inf))
    (netflow  (minf 0 inf)))
  (constraints
    ((m+ amount level)   (0 0) (full top) (inf inf))
    ((m+ level pressure) (0 0) (inf inf))
    ((mult pressure drain outflow))
    ((add netflow outflow inflow))
    ((d/dt amount netflow)))
  (independent inflow drain)
  (history amount)
  )

(defun foo1 ()
  (let* ((given '((netflow 0)))
	 (solns (dkb-solve0 ch-bathtub given)))
    (show-solutions ch-bathtub '0 given solns)))

(defun foo2 (given)
  (let* ((solns (dkb-solve0 ch-regulator given)))
    (show-solutions ch-regulator '0 given solns)))

(defun foo3 ()
  (let* ((given '((netflow (0 std))))
	 (solns (dkb-solve-d s194 given)))
    (show-solutions s194 'd given solns)))