;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :qsim)

(defun simple-U-tube-figure ()
  (declare (special u-tube))
  (let* ((init (make-new-state :from-qde U-tube
			       :assert-values `((amtA (Amax nil))
						(amtB (0 nil)))
			       :text "Tank A full; B empty"))
	 (nlayout '((amtA amtB total)
		    (pressureA pressureB)
		    (pAB flowAB)))
	 )
    (format *QSIM-Report* 
"~2%  This is the simple U-tube example:  a closed two-tank system, starting with
tank A full and tank B empty.  We get the usual three-way branch according to
whether the system reaches equilibrium before tank B overflows.

For dramatic effect, if tank B overflows, it bursts!
We get a region transition to a model where tank B has lost all its contents.
The contents of tank A now drain across the channel, unopposed by backpressure
from tank B, until the entire system is empty.  Meltdown! ~2%")

    (qsim init)
    (qsim-display init :layout nlayout)
    ))

(defun U-tube-w-ranges ()
  (declare (special u-tube))
  (let* ((sim (make-sim :Q2-constraints t
				:new-landmarks-across-constraints t))
	 (init (make-new-state :from-qde U-tube
			       :assert-values `((amtA (Amax nil))
						(amtB (0 nil)))
			       :text "Tank A full; B empty"
			       :sim sim))
	 (nlayout '((amtA amtB total)
		    (pressureA pressureB)
		    (pAB flowAB time)))
	 )
    (format *QSIM-Report* 
"~2%  This is the same as the simple U-tube example except that, in addition to
the qualitative information, we have some quantitative information about the sizes
of the two tanks and the envelopes bounding where the M+ functions can be.

In this case, we know that tank B is larger than tank A, so the overflow
scenarios are quantitatively impossible, though qualitatively plausible. 

The careful reader will notice that, in the inconsistent behaviors, the propagation
process stops as soon as an inconsistency is detected, and so is not complete.
Even in a consistent behavior, since propagation is local, there are stronger
quantitative conclusions that could be drawn from the same equations by global
algebraic methods.
~2%")

    (qsim init)
    (qsim-display init :layout nlayout :show-inconsistent-successors t)
    ))


; The U-tube:  a simple two-tank system.

(define-QDE U-tube
  (text "U-tube")
  (quantity-spaces
    (amtA       (0 Amax inf)   "amtA")
    (amtB       (0 Bmax inf)   "amtB")
    (total      (0 inf)        "amtA + amtB")
    (pressureA  (0 inf)        "pressureA")
    (pressureB  (0 inf)        "pressureB")
    (pAB        (minf 0 inf)   "pAB")
    (flowAB     (minf 0 inf)   "flowAB")
    (mflowAB    (minf 0 inf)   "-flowAB"))
  (constraints
    ((M+ amtA pressureA)       (0 0) (inf inf))
    ((M+ amtB pressureB)       (0 0) (inf inf))
    ((ADD amtA amtB total))
    ((ADD pAB pressureB pressureA))
    ((M+ pAB flowAB)           (0 0) (inf inf))
    ((d/dt amtB flowAB))
    ((minus flowAB mflowAB)    (inf minf) (minf inf))
    ((d/dt amtA mflowAB))
    ((constant total)))
  (transitions
    ((amtA (Amax inc)) -> t)
    ((amtB (Bmax inc)) -> tank-B-burst))
  (layout (amtA      amtB      total nil)
	  (pressureA pressureB)
	  (nil       pAB  nil)
	  (nil       flowAB nil)
	  (nil))
  ; "other" clauses can be at the top level, because of defother.
    (define-normal
      (amtA   ((0 amax) std))
      (amtB   ((0 bmax) std))
      (total  ((0 inf) std)))
    (initial-ranges ((amtA AMAX) (80 90))	; quantitative ranges
		    ((amtB BMAX) (100 110))
		    ((time t0)   (0 0)))
    (m-envelopes
      ((M+ amtA pressureA) (upper-envelope (lambda (x) x))
                           (upper-inverse  (lambda (y) y))
			   (lower-envelope (lambda (x) (/ x 2)))
			   (lower-inverse  (lambda (y) (* y 2))))
      ((M+ amtB pressureB) (upper-envelope (lambda (x) x))
                           (upper-inverse  (lambda (y) y))
			   (lower-envelope (lambda (x) (/ x 2)))
			   (lower-inverse  (lambda (y) (* y 2))))
      ((M+ pAB flowAB) (upper-envelope (lambda (x) (if (< x 10) x (+ (/ x 2) 5))))
                       (upper-inverse  (lambda (y) (if (< y 10) y (* 2 (- y 5)))))
		       (lower-envelope (lambda (x) (/ x 4)))
                       (lower-inverse  (lambda (y) (* y 4))))))





; The leaking U-tube results is amtB crosses BMAX.

(define-QDE U-tube-leaking
  (text "U-tube")
  (quantity-spaces
    (amtA       (0 Amax inf)   "amtA")
    (amtB       (0 Bmax inf)   "amtB")
    (total      (0 inf)        "amtA + amtB")
    (pressureA  (0 inf)        "pressureA")
    (pressureB  (0 inf)        "pressureB")
    (pAB        (minf 0 inf)   "pAB")
    (flowAB     (minf 0 inf)   "flowAB")
    (mflowAB    (minf 0 inf)   "-flowAB")
    (leakB      (0 inf)        "leak from B")
    (netflowB   (minf 0 inf)   "delta B")
    (mleakB     (minf 0)))
  (constraints
    ((M+ amtA pressureA)       (0 0) (inf inf))
    ((M+ amtB pressureB)       (0 0) (inf inf))
    ((ADD amtA amtB total))
    ((ADD pAB pressureB pressureA))
    ((M+ pAB flowAB)           (0 0) (inf inf))
    ((d/dt amtB netflowB))
    ((minus flowAB mflowAB)    (inf minf) (minf inf))
    ((d/dt amtA mflowAB))
    ((M+ pressureB leakB)      (0 0) (inf inf))
    ((add netflowB leakB flowAB))
    ((minus leakB mleakB))
    ((d/dt total mleakB)))
  (transitions
    ((amtA (Amax inc)) -> t)
    ((amtB (Bmax inc)) -> tank-B-burst))
  (layout (amtA      amtB      total )
	  (pressureA pressureB)
	  (nil       pAB  leakB)
	  (nil       flowAB netflowB))
  (ignore-qdirs netflowB))

(defun tank-B-leaking (old-state)
  (declare (special U-tube-leaking))
  (create-transition-state :from-state old-state
			   :to-qde U-tube-leaking
			   :inherit-qmag '(amtA amtB total)))


; The burst U-tube results if amtB crosses BMAX.

(define-QDE U-tube-burst
  (text "U-tube")
  (quantity-spaces
    (amtA       (0 Amax inf)   "amtA")
    (amtB       (0 Bmax inf)   "amtB")
    (total      (0 inf)        "amtA + amtB")
    (pressureA  (0 inf)        "pressureA")
    (pressureB  (0 inf)        "pressureB")
    (pAB        (minf 0 inf)   "pAB")
    (flowAB     (minf 0 inf)   "flowAB")
    (mflowAB    (minf 0 inf)   "-flowAB"))
   (constraints
     ((M+ amtA pressureA)       (0 0) (inf inf))
     ((M+ amtB pressureB)       (0 0) (inf inf))
     ((ADD amtA amtB total))
     ((ADD pAB pressureB pressureA))
     ((M+ pAB flowAB)           (0 0) (inf inf))
     ;;  ((d/dt amtB flowAB))  ;; amtB = 0, constant
     ((constant amtB))
     ((minus flowAB mflowAB)    (inf minf) (minf inf))
     ((d/dt amtA mflowAB)))
   (layout (amtA      amtB      total nil)
	   (pressureA pressureB)
	   (nil       pAB  nil)
	   (nil       flowAB nil)
	   (nil)))

(defun tank-B-burst (old-state)
  (declare (special U-tube-burst))
  (create-transition-state :from-state old-state
			   :to-qde U-tube-burst
			   :inherit-qmag '(amtA)
			   :assert '((amtB (0 std)))))