;;; -*- Syntax: Common-lisp; Package: qsim -*-
;;;     Copyright (c) 1987, Benjamin Kuipers.
;;;  $Id: thermostat-modes.lisp,v 1.3 92/07/20 15:34:36 clancy Exp $
(in-package :qsim)


;;;-----------------------------------------------------------------------------
;;;  Model:    HOME-HEATER-MODES
;;;
;;;  Intent:   This is a model of a thermostatically (on/off) controlled home-
;;;            heating furnace.  This model demonstrates four things:
;;;
;;;            1.  Discrete-variables:
;;;                In contrast to the continuous variables declared in the
;;;                quantity-spaces clause, discrete-variables:
;;;                -- can only have landmark values (never interval values),
;;;                -- have qdir always 'std (like independent variables),
;;;                -- have an unordered qspace.
;;;
;;;            2.  Modes:
;;;                A group of constraints can be made active or inactive
;;;                depending on the values of specified variables.  An active
;;;                constraint is processed normally; an inactive constraint is
;;;                effectively invisible during simulation.  The condition in
;;;                a mode clause is usually expressed in terms of a discrete
;;;                variable (but this is not a requirement).
;;;
;;;            3.  Transition conditions:
;;;                The simplest transition condition is a test of a variable's
;;;                value to a given landmark and qdir.  It is now valid to
;;;                combine such conditions using AND, OR, and NOT, nested
;;;                to any depth, if needed.
;;;
;;;            4.  Transition functions:
;;;                The new function |create-transition-state|, with its keyword
;;;                arguments, simplifies the job of specifying what should
;;;                change and what should remain the same at a region transition.
;;;-----------------------------------------------------------------------------


(define-QDE HOME-HEATER-MODES
  (text "Thermostatic control of home heater")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 inf))
    (TempIn  (0 LoTemp RoomTemp HiTemp inf))
    (TempOut (0 Cold RoomTemp Hot inf))
    (TempSet (0 RoomTemp inf))
    (dTemp   (minf 0 inf))
    (error   (minf Lo 0 Hi inf))
    (R       (0 inf))
    (HFout   (minf 0 inf))
    (netHF   (minf 0 inf))
    (Power   (0 On)) 
    (HFin    (0 On)))
  (discrete-variables
    (switch  (open closed stuck-open stuck-closed)))
  (constraints
    (mode (or (switch closed) (switch stuck-closed))
	  ((M+ HFin Power) (0 0) (On On)))
    (mode (or (switch open) (switch stuck-open))
	  ((zero-std HFin)))
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)  (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)  (RoomTemp 0 RoomTemp) (RoomTemp Lo LoTemp) (RoomTemp Hi HiTemp))
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF)))
   (transitions
     ((and (switch (open   std)) (error (Lo dec)))   turn-switch-on)
     ((and (switch (closed std)) (error (Hi inc)))   turn-switch-off))
  (independent HFin Power Mass TempOut TempSet R)
  (history TempIn)
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (TempSet "Temp(set)"      Ts)
    (dTemp   "dTemp(in,out)"  dT)
    (error   "error=in-set"  E)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (to environment)" HFo)
    (HFin    "Heat flow (from heater)" HFi)
    (netHF   "net Heat Flow"  nHF)))


(defun low-outside-temperature ()
  (let ((start (make-initial-state Home-Heater-Modes
				   '((Heat ((0 inf) nil))
				     (TempIn (RoomTemp nil))
				     (Mass ((0 inf) std)) (TempOut (Cold std))
				     (TempSet (RoomTemp std)) (R ((0 inf) std))
				     (Power (On std)) (Switch (open std)))
				   "Room temperature; furnace off.")))
    (qsim start)
    (qsim-display start)
    t))


(defun turn-switch-on (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-modes
			   :assert       '((switch (closed std))
					   (HFin   (nil std))
					   (netHF  ((0 inf) nil)))
			   :inherit-qmag :rest
			   :inherit-qdir nil))


(defun turn-switch-off (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-modes
			   :assert       '((switch (open std))
					   (HFin   (nil std))
					   (netHF  ((minf 0) nil)))
			   :inherit-qmag :rest
			   :inherit-qdir nil))

