;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

;;; $Id: numsim-examples.lisp,v 1.3 92/07/23 11:42:54 bert Exp $

(in-package :qsim)

;;; This file contains examples of models run using the numsim simulator.
;;; Numsim takes a QDE augmented with functional envelopes and values for
;;; initial state variable values and constants and simulates the resulting
;;; ODE.
;;;
;;; Qsim need not be run on the QDE to use numsim.  This independence lets
;;; you play with a QDE to try out some representative ODE instances and see
;;; what they look like before a tractible Qsim simulation has been developed.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The simple spring.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; The first model is the simple spring model found in EXAMPLES;springs.lisp.
;;; Note that the QDE is very similar except that the M- constraint has an
;;; m-envelope clause that defines a SPECIFIC M- function for the spring.
;;; This is done with the expected-function and expected-inverse slots.
;;; 
(define-QDE Numsim-Simple-Spring
   (text "Simple spring model (predicts possible increasing and decreasing oscillation)")
   (quantity-spaces
	      (A      (minf 0 inf)   "Accelleration")
	      (V      (minf 0 inf)   "Velocity")
	      (X      (minf 0 inf)   "Position"))
   (constraints
	      ((d/dt X V))
	      ((d/dt V A))
	      ((M- A X)         (0 0) (minf inf) (inf minf)))
   (m-envelopes 
     ((M- A X) (expected-function (lambda (a) (- (/ a 2.0))))
               (expected-inverse  (lambda (x) (- (* x 2.0))))))
   ;; Note that a layout clause is not needed.
   )


;;; This function does the simulation.  Note that an initial state is
;;; created to simplify the interface to the QSIM-DISPLAY function.
;;; The actual state information is not needed to run the simulation, but
;;; since the numsim simulation hangs off of a qsim state, some state must
;;; be created.  However, this state can simply be a place-holder
;;; (for instance (make-plain-state :from-qde qde :text "A dummy state"))
;;; that ties a sim and a qde together.  The advantage of creating a dummy
;;; state is that it does not require that qsim form an initial state
;;; via constraint satisfaction.
;;;
(defun Numsim-push-simple-spring ()
  (let* ((sim (make-sim))
	 (initial
	   (make-new-state :from-qde numsim-simple-spring
			   :assert-values '((x  (0 nil))
					    (v  ((0 inf) nil)))
			   :text "Start with initial velocity (using numsim)"
			   :sim sim)))
    (format *Qsim-Report*
	    "~&This illustrates the use of the NUMSIM simulator on a simple~
             ~%spring with spring constant 2.0 starting at x=0 with velocity ~
               5.~
             ~%This example is from the file EXAMPLES;numsim-examples.lisp")

    (numsim initial :initial-values '((x 0) (v 5.0)))

    ;; Set the layout to display a and x.
    (set-numeric-layout initial '((a v) (x nil)))
    (qsim-display initial :plot-mode 'numeric-time-plot)))
    


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A ball dropped in a gravity field.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This model demonstrates dropping an object in a gravity field.
;;; It augments the model given in EXAMPLES;ball.lisp by including 
;;; an expected-value clause for the value of the gravitational constant.
;;;
(define-QDE Numsim-Gravity
  (text  "Constant gravity (no friction).")
  (quantity-spaces
    (y   (0 y* inf))
    (v   (minf 0 inf))
    (a   (minf g 0)))
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((constant a)))
  (expected-values ((a g) -9.8))
  )


;;; Drop an object in a gravity field using NUMSIM to predict the trajectory.
;;; This example demonstrates the use of the :stop-condition keyword (which
;;; terminates the simulation when y=0 (i.e., when the ball hits the ground))
;;; and the use of a landmark as an initial value specification.  It also
;;; demonstrates the use of a dummy state to hold the results of the numsim
;;; simulation.
;;; 
(defun numsim-drop-thing ()
  (let ((dummy (make-plain-state :from-qde numsim-gravity
				 :text "Drop an object (using numsim)")))
    (format *Qsim-Report*
	    "~&This illustrates the use of the NUMSIM simulator on a ball~
             ~%falling in a gravity field.  The ball starts at height 50 with ~
               zero velocity.  ~
             ~%Simulation is terminated when the height of the ball reaches 0.~
             ~%This example is from the file EXAMPLES;numsim-examples.lisp")

    ;; y and x are assigned values directly, but a takes its value from the
    ;; expected-values clause of the qde.  This means that you don't need to 
    ;; remember the values of model constants when playing with the initial
    ;; conditions.
    (numsim dummy :initial-values '((y 50.0) (v 0) (a g))
	          :stop-condition '((<= y 0.0)))
    (set-numeric-layout dummy '((y v)))
    (qsim-display dummy :plot-mode 'numeric-time-plot)
    ))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A Two tank cascade
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This model demonstrates the use of the "set-numeric-graph-options"
;;; function to alter the behavior of individual graphs in a numeric plot.
;;; Play with this by uncommenting different lines that contain this command.
;;; It also shows how the results of a Q2 simulation and a numsim simulation
;;; are automatically superimposed.  
;;;
(define-QDE numsim-cascaded-tanks-w-ranges
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 ifa* inf))
    (amounta    (0 full inf))
    (outflowa   (0 inf))
    (netflowa   (minf 0 inf))
    (amountb    (0  full inf))
    (outflowb   (0 inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((constant inflowa))
    ((M+ amounta outflowa)            (0 0) (inf inf)   )
    ((ADD outflowa netflowa inflowa)                    )
    ((d/dt amounta netflowa))
    ((m+ amountb outflowb)            (0 0) (inf inf)   )
    ((add outflowb netflowb outflowa)                   )
    ((d/dt amountb netflowb)))
  (ignore-qdirs netflowb)
  (m-envelopes
      ;; Note that Q2 envelope definitions can coexist with numsim functions.
      ((M+ amountA outflowA) (upper-envelope (lambda (x) x))
                             (upper-inverse  (lambda (y) y))
			     (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
							     (+ (/ x 5) 6))))
			     (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
							     (* 5 (- y 6)))))
			     (expected-function (lambda (x) (/ x 1.5)))
			     (expected-inverse (lambda (y) (* 1.5 x))))
      ((M+ amountB outflowB) (upper-envelope (lambda (x) x))
                             (upper-inverse  (lambda (y) y))
			     (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
							     (+ (/ x 5) 6))))
			     (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
							     (* 5 (- y 6)))))
			     (expected-function (lambda (x) (/ x 1.5)))
			     (expected-inverse (lambda (y) (* 1.5 x))))
      )
  (initial-ranges ((inflowA ifa*) (5 10))
		  ((time t0)      (0 0))
		  ((amounta full) (50 100))
		  ((amountb full) (50 100)))
  (expected-values ((inflowA ifa*) 7.5)
		   ((amounta full) 75)
		   ((amountb full) 75))
  (layout
    (amounta amountb nil)
    (netflowa netflowb nil)
    (time nil nil))
)

(defun numsim-fill-cascade-from-empty ()
  (let* ((sim (make-sim :q2-constraints t))
	 (init (make-new-state :from-qde  numsim-cascaded-tanks-w-ranges
			       :assert-values '((inflowa (ifa* std))
						(amounta (0 nil))
						(amountb (0 nil)))
			       :text "Fill from empty (Q2 and numsim)"
			       :sim sim)))
    (format *Qsim-report*
	    "~&This illustrates the use of the NUMSIM simulator as a tool for ~
             ~%exploring particular behaviors of a QDE together with the ~
             ~%bounding regions computed by Q2. ~
             ~%First, we simulate the QDE using Q2.")
    (qsim init)
    (format *Qsim-report*
	    "~&Now we run NUMSIM to compute the behavior of a particular ODE ~
             ~%represented by the QDE.")

    ;; This uses the :use-state keyword to let the initial values come from
    ;; the initial state.
    (numsim init :use-state init :initial-values '((amountA :from-state :lb)
						   (amountB :from-state :lb)
						   (inflowa :from-state :expected)))

    ;; This runs numsim using user-specified conditions.
;    (numsim init :initial-values '((amountA 0) (amountb 0) (inflowa ifa*)))

    (set-numeric-layout init '((amounta amountb)
			       (netflowa netflowb)))
    ;; Note that the qualitative prediction for netflowb, while correct, is
    ;; rather uninteresting, but the numsim prediction is more informative.

    ;; In order to improve the resolution on the amount variables, we change
    ;; their scaling.  Note that this can also be done from the numeric
    ;; graphing options menu (select G then O from the main display menu)
    ;; by typing in the second argument to the call.
;    (set-numeric-graph-options init '((amounta (ylimits (0 25 nil nil)))
;				      (amountb (ylimits (0 25 nil nil)))))

    ;; This set of graph options does not plot Q2 timepoints at all.  The
    ;; dotted line represents the time interval ranges predictied by Q2.
;    (set-numeric-graph-options init '((amounta (ylimits (0 25 nil nil)))
;				      (amountb (ylimits (0 25 nil nil)))
;				      (* (rangestyle nil))))
				       
    
    (format *Qsim-Report*
	    "~&See the function numsim-fill-cascade-from-empty in the file ~
             ~%EXAMPLES;numsim-examples.lisp for information on changing the ~
             ~%plotting parameters of the graphs.")
    (qsim-display init :plot-mode 'numeric-time-plot)
    ))