
;;; -*- Syntax: Common-lisp; Package: qsim -*-

(in-package 'QSIM)

(defun describe-nic ()
  (format *qsim-report*
"~2%The Non-Intersection Constraint is described by Lee and Kuipers in AAAI-88.  It filters
out behaviors of second order systems having intersecting phase space trajectories.
It is a useful constraint for simulating oscillatory mechanisms and is applicable to
second order systems.

The filter can be invoked by setting *apply-nic-p* to T or specifying a Phase-Planes
clause in the Other clause of the QDE.  In the first case you will be prompted for the
phase planes you want to detect intersections in if a Phase-Planes clause is not already
present in the Other clause of the QDE.  A Phase-Planes clause is of the form:
(phase-planes (V1 V2) . . . ) where Vn is a variable of the system simulated.  Either
case, you have to determine the set of phase planes to check.  This is expected to be
automated in the future.

In specifying phase planes to be checked for trajectory intersection, care should be
taken to avoid using any phase plane whose phase variables are monotonically related.
To ensure that all trajectory intersections are detected, pick a maximal set of linearly
independent variables for the system being simulated (where no pair of variables are
monotonically related).  Check all possible phase planes between this set of variables
for trajectory intersection.  ~2%")
)

; The simple spring to illustrate the Non-Intersection Constraint.
; The Constraint rules out 2 of the 5 behaviors originally predicted.

(define-QDE nic-simple-spring
   (quantity-spaces
	      (A      (minf 0 inf))
	      (V      (minf 0 V* inf))
	      (X      (minf 0 inf)))
   (constraints
	      ((d/dt X V))
	      ((d/dt V A))
	      ((M- A X)         (0 0) (minf inf) (inf minf)))
   (independent)
   (history X V)
   (print-names  (A   "Accelleration")
		 (V   "Velocity")
		 (X   "Position"))
   (text "Simple spring model (predicts possible increasing and decreasing oscillation)")
   (layout  (nil x nil)
	    (nil v nil)
	    (nil a nil))
   (other (phase-planes (x v)(v a)))		; which portraits to look at
   ; Actually, looking at one of the portraits would be sufficient to eliminate spurious
   ; behaviors.  However, note that the (x a) portrait cannot be used since x and a are
   ; monotonic functions of each other.
   )

; Full tracing.

(defun simple-spring-w-tracing ()
  (let ((*state-limit* 25)			; setting max number of states to simulate
	(*prune-intersections-p* nil)		; do not prune behaviors with intersections
	(*nic-trace-mode* 'along-trajectory)	; full trace
	(*nic-trace-display*  'portrait-also)	; with phase portraits plotted each time
	(initial nil))
    (setq initial
	  (make-initial-state nic-simple-spring
			      '((x  (0 nil))
				(v  (V* nil)))
			      "Start with initial velocity"))
    (format *qsim-report* 
"~2%This simple spring example illustrates the application of the Non-Intersection
Constraint, as described in Lee and Kuipers [AAAI-88], with full tracing enabled.
Note that 2 of the 5 behaviors predicted are ruled out by the Constraint.  See
`examples;nic-examples' for details of this example.  ~2%")
    (qsim initial)
    (qsim-display initial)		; no reference points yet.
    ))

; Straight application. No tracing.

(defun simple-spring-w-o-tracing ()
  (let ((*state-limit* 25)			; setting max number of states to simulate
	(*prune-intersections-p* t)		; prune behaviors with intersections
	(*nic-trace-mode* nil)			; no tracing
	(initial nil))
    (setq initial
	  (make-initial-state nic-simple-spring
			      '((x  (0 nil))
				(v  (V* nil)))
			      "Start with initial velocity"))
    (format *qsim-report* 
"~2%This example applies the Non-Intersection Constraint, described in Lee & Kuipers
[AAAI-88], to the simple spring.  Note that only 3 behaviors are predicted (vs 5
without the constraint).  See `examples;nic-examples' for details of example.~2%")
    (qsim initial)
    (qsim-display initial)		; no reference points yet.
    ))

;;; *****************************************************************************************

; Applying the Non-Intersection Constraint to the damped spring.
; About half of the behaviors are pruned.

(define-QDE nic-damped-spring
  (text "Damped Spring with Non-Intersection Constraint")
  (quantity-spaces
    (x     (minf 0 inf))
    (v     (minf 0 inf))
    (a     (minf 0 inf))
    (ff    (minf 0 inf))
    (fs    (minf 0 inf))
    (f     (minf 0 inf)))
  (constraints
    ((d/dt x v))
    ((d/dt v a))
    ((m- x fs)      (0 0)     (minf inf)  (inf minf))
    ((m- v ff)      (0 0)     (minf inf)  (inf minf))
    ((add fs ff f)  (0 0 0) )
    ((m+ f a)       (0 0)     (inf inf)   (minf minf)))
  (history x v)
  (layout  (nil x nil)
	   (nil v nil)
	   (nil a nil))
  (other (phase-planes (v a)(x v)))		; which portraits to look at
)

; No initial landmarks.

(defun damped-spring-w-o-init-lmk ()
  (when (or (not *query-user*)			; Always run during tests
	    (y-or-n-p "~&~%This example uses a *state-limit* of 170.  Do you really want to run it? "))
    (let ((*state-limit* 170)			; setting max number of states to simualte
	  (*perform-acc-analysis* t)		; use curvature constraint
	(*new-landmarks-at-initial-state* nil)	; don't create landmarks for initial state
	  (*prune-intersections-p* nil)		; do not prune behaviors with intersections
	  (*intersection-count-p* t)		; count intersections
	  (initial nil))
      (setq initial 
	    (make-initial-state nic-damped-spring
				'((x ((0 inf) inc))
				  (v ((0 inf) dec))
				  (a ((minf 0) inc)))))
      (format *qsim-report* 
"~2%This example applies the Non-Intersection Constraint, described in Lee & Kuipers [AAAI-88],
to the damped spring.  With this and the curvature constraint, only strictly increasing and
strictly decreasing oscillations are predicted in addition to those reaching quiescence
after arbitrary number of oscillations.

Note that some of the increasing oscillations appear to have intersecting trajectories.
This however is only due to the inability of the phase diagram plotter to assign
appropriate coordinates for the (MINF 0) and (0 INF) qvals.  Run the With-Initial-Landmark
example which assigns initial landmarks to all the parameters to get rid of this problem.
This however leads to further proliferation in branching.

Note also that in some behaviors, x may be having an increasing oscillation while v has a
decreasing one.  These behaviors are not pruned because their trajectories do not
intersect yet at that point.  If simulation continues, v continuing with its decreasing
oscillation while x continuing with its increasing ocillation will result in intersections
and these behaviors being eliminated.

For details of this example, look in the file `examples;nic-examples'.~2%")
      (qsim initial)
      (qsim-display initial)		; no reference points
      )))

; With initial landmarks.

(defun damped-spring-w-init-lmk ()
  (when (or (not *query-user*)			; Always run during tests
	    (y-or-n-p "~&~%This example uses a *state-limit* of 530.  Do you really want to run it? "))
    (let ((*state-limit* 530)			; setting max number of states to simualte
	  (*perform-acc-analysis* t)		; use curvature constraint
	  (*prune-intersections-p* nil)		; do not prune behaviors with intersections
	  (*intersection-count-p* t)		; count intersections
	  (initial nil))
      (setq initial 
	    (make-initial-state nic-damped-spring
				'((x ((0 inf) inc))
				  (v ((0 inf) dec))
				  (a ((minf 0) inc)))))
      (format *qsim-report* 
"~2%This example applies the Non-Intersection Constraint, described in Lee & Kuipers [AAAI-88],
to the damped spring.  With this and the curvature constraint, only strictly increasing and
strictly decreasing oscillations are predicted in addition to those reaching quiescence
after arbitrary number of oscillations.  Note the proliferation in branching compared to
simulating without initial landmarks.

Note also that in some behaviors, x may be having an increasing oscillation while v has a
decreasing one.  These behaviors are not pruned because their trajectories do not
intersect yet at that point.  If simulation continues, v continuing with its decreasing
oscillation while x continuing with its increasing ocillation will result in intersections
and these behaviors being eliminated.

For details of this example, look in the file `examples;nic-examples'.~2%")
      (qsim initial)
      (qsim-display initial)		; no reference points
      )))