;;; -*- Mode:Common-Lisp; Package: QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim) ;changed DJC

;;;=============================================================================
;;;                   A bunch of models to test the
;;;
;;;           K I N E T I C   E N E R G Y   C O N S T R A I N T
;;;=============================================================================
;;;  In this file, the KET-constraints are derived automatically.
;;;=============================================================================

;;; by Pierre Fouche <fouche@cs.utexas.edu>, October 1989
;;; Revised August 1990

;;; Energy constraints are derived automatically.

;;;*****************************************************************************
;;; A small description of EC and how to use it...
;;;*****************************************************************************

(defun describe-EGF ()

  (format *qsim-report*
"~2%============================================================================
====

The Energy Constraint (EC) is described in the Tec-Report

           \"Reasoning About Energy in Qualitative Simulation\"
                   Pierre Fouche and Benjamin Kuipers
                              UT-TR-90-134


There is additional information in the code itself: >nq>energy-gf.lisp
and in the examples: >nq>examples>egf.lisp.

EC filters out behaviors that violate the law of conservation of energy. It is
applicable to any second- or higher-order system, even if it doesn't make sense
to talk about energy.

It is activated by specifying an ENERGY-CONSTRAINT clause in the QDE.
This clause is a list of tuples, each tuple being a list of four terms:
a \"POSITION\" variable X, a \"SPEED\" variable V, and two terms C and NC
representing the decomposition of dV/dt into CONSERVATIVE and NON-CONSERVATIVE
terms. A term is conservative iff it depends only on the \"position\" variable X
(for instance, C=M+(X)).

This clause can be asserted in the DEFINE-QDE macro, or it can be derived
automatically by a small algebraic manipulator which analyzes the QDE structure.

Automatic derivation is turned on by the switch *perform-energy-analysis*,
accessible from the EC control menu.
The filter itself is active when *check-energy-constraint* is T.

EC is traced by setting *trace-energy-filter* to T. In this case, behaviors
that don't satisfy the conservation of energy are not ruled out, but labeled as
inconsistent. You can look at the description of the final state to get an
explanation of the inconsistency:
\"KE variation\" must be equal to \"C work\" + \"NC work\".

Please, send comments and bug reports to Pierre Fouche <fouche@cs.utexas.edu>.

================================================================================
~2%"))


;;;*****************************************************************************
;;; The simplest second-order system: constant gravity without friction.
;;;*****************************************************************************

(define-QDE Gravity
  (text  "Constant gravity without friction.")
  (quantity-spaces
    (y   (0 inf)               "Position")
    (v   (minf -v0 0 v0 inf)   "Velocity")
    (-v- (0 v0 inf)            "Abs value of Velocity")
    (g   (g 0)                 "Accelleration")
    )
  (constraints
    ((d/dt v g))
    ((d/dt y v))
    ((abs-value v -v-) (v0 v0) (-v0 v0)))
  (independent g)
  (layout (y nil nil)
          (v -v- nil)
          (g nil nil))
  )


(defun gravity-w-tracing ()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (init
          (make-new-state :from-qde gravity
			  :assert-values '((y  (0 nil))
					   (v  (v0 nil))
					   (g  (g std)))
			  :text "Throw a ball upward")))
    (format *qsim-trace*
"~2%============================================================================
====

The simplest second-order system: constant gravity without friction.

Notice that two of the three possible behaviors are ruled out: the speed of the
ball when it touches the ground must be equal to its initial speed.

Look at the final state description to get an explanation.

================================================================================
~2%")
    (qsim init)
    (qsim-display init)
    ))


;;;*****************************************************************************
;;; A more complex model: gravity with friction.
;;; The number of simulated behaviors varies whether analytic functions are
;;; accepted.
;;;*****************************************************************************

(define-QDE Gravity-with-friction
            (text  "Constant gravity with friction.")
  (quantity-spaces
    (y   (0 inf)               "Position")
    (v   (minf -v0 0 v0 inf)   "Velocity")
    (-v- (0 v0 inf)            "Abs value of Velocity")
    (a   (minf g 0 inf)        "Accelleration")
    (g   (g 0)                 "Gravity")
    (ff  (minf 0 inf)          "Fluid friction"))
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((abs-value v -v-) (v0 v0) (-v0 v0))
    ((m- ff v) (0 0) (inf minf) (minf inf))
    ((add g ff a) (g 0 g)))
  (independent g)
  (layout (y nil nil)
          (v -v-)
          (a g ff))
  )

(defun gravity-w-friction-n-tracing-n-af()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*analytic-functions-only* t)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (init
          (make-new-state :from-qde Gravity-with-friction
			  :assert-values '((y  (0 nil))
					   (v  (v0 nil))
					   (g  (g std)))
			  :text "Throw a ball upward (w friction)")))
    (format *qsim-trace*
"~2%============================================================================
====

A ball thrown in the air which causes friction.

Only analytic functions are accepted and inconsistent behaviors are not deleted.
Notice that the ball terminal velocity is lower than its initial velocity.

================================================================================
~2%")
    (qsim init)
    (qsim-display init)
    ))

(defun gravity-w-friction-n-tracing()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*analytic-functions-only* nil)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (init
          (make-new-state :from-qde Gravity-with-friction
			  :assert-values '((y  (0 nil))
					   (v  (v0 nil))
					   (g  (g std)))
			  :text "Throw a ball upward (w friction)")))
    (format *qsim-trace*
"~2%============================================================================
====

A ball thrown in the air which causes friction.

Simulation is not restricted to analytic functions and inconsistent behaviors
are not deleted.

Notice that the ball terminal velocity is always lower than its initial
velocity.

================================================================================
~2%")
    (qsim init)
    (qsim-display init)
    ))


;;;*****************************************************************************
;;; Same example as above, but with a region transition: the bouncing ball.
;;;*****************************************************************************

(define-QDE Bouncing-ball
            (text  "Bouncing ball with friction")
  (quantity-spaces
    (y   (0 inf))
    (v   (minf 0 v0 inf))
    (a   (minf g 0 inf))
    (g   (g 0))
    (ff  (minf 0 inf))
    )
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((m- ff v) (0 0) (inf minf) (minf inf))
    ((add g ff a) (g 0 g)))
  (transitions
    ((y (0 dec)) reflect-speed))
  (independent g)
  (print-names (A   "Accelleration")
               (G   "Gravity")
               (FF  "Fluid friction")
               (V   "Velocity")
               (Y   "Position"))
  (layout (y nil nil)
          (v -v-)
          (a g ff))
  )


(defun reflect-speed (ball-state)
  (create-transition-state :from-state   ball-state
                           :to-qde       Bouncing-ball
                           :assert       '((y (0 inc)))
                           :inherit-qmag '(-v-)))

(defun bb-w-friction-n-tracing()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*analytic-functions-only* t)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (*state-limit* 50)
        (init
          (make-new-state :from-qde Bouncing-ball
			  :assert-values '((y  (0 nil))
					   (v  (v0 nil))
					   (g  (g std)))
			  :text "Throw an object upward (w friction)")))
    (format *qsim-trace*
"~2%============================================================================
====

A bouncing ball with friction. The bounce has no internal structure, the
velocity is simply reflected.

Simulation is restricted to analytic functions and inconsistent behaviors
are not deleted. Amoung 26 behaviors generated ONLY 2 ARE VALID with respect to
the law of conservation of energy.

================================================================================
~2%")
    (qsim init)
    (qsim-display init)
    ))

(defun bb-w-friction()
  (qsim-cleanup)
  (let ((*trace-energy-filter* nil)
        (*analytic-functions-only* t)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (*state-limit* 60)
        (init
          (make-new-state :from-qde Bouncing-ball
			  :assert-values '((y  (0 nil))
					   (v  (v0 nil))
					   (g  (g std)))
			  :text "Throw an object upward (w friction)")))
    (format *qsim-trace*
"~2%============================================================================
====

A bouncing ball with friction. The bounce has no internal structure, the
velocity is simply reflected.

Simulation is restricted to analytic functions only.
Notice that the bounce amplitude is always decreasing.

================================================================================
~2%")
    (qsim init)
    (qsim-display init)
    ))



;;;=============================================================================
;;; Now let's deal with systems that were more or less intractable so far:
;;;
;;;                              W E L C O M E
;;;                                  to the
;;;                         S P R I N G   W O R L D.
;;;
;;;=============================================================================

;;;*****************************************************************************
;;; No friction to begin with.
;;;*****************************************************************************

(define-QDE Simple-Spring
            (text "Simple spring")
  (quantity-spaces
    (A      (minf 0 inf)   "Accelleration")
    (V      (minf 0 inf)   "Velocity")
    (X      (minf 0 inf)   "Position")
    (-X-    (0 inf)        "Abs value of Position"))
  (constraints
    ((d/dt X V))
    ((d/dt V A))
    ((M- A X)         (0 0) (minf inf) (inf minf))
    ((abs-value x -x-)))
  (history X V)
  (layout  (x -x- nil)
           (v -v- nil)
           (a nil nil))
  )

(defun simple-spring-w-tracing- ()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*state-limit* 50)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (initial
          (make-new-state :from-qde Simple-Spring
			  :assert-values '((x  ((0 inf) std)))
			  :text "Start with initial velocity")))
    (format *qsim-trace*
"~2%============================================================================
====

The simple spring without friction.

Here 3 behaviors are kept among 13 possible.

Why 3 and not one ?

We don't know that the M+ function between the accelleration and the position
is symmetric. It can be weeker on one side and stronger on the other. Thus
the peaks on each side of the rest position are not related: there is three
possible
behaviors for the absolute value of the position.
We would get only one behavior without that variable.

Compare these results with those obtained by applying NIC.

================================================================================
~2%")

    (qsim initial)
    (qsim-display initial)
    ))

(defun simple-spring ()
  (qsim-cleanup)
  (let ((*trace-energy-filter* nil)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (initial
          (make-new-state :from-qde Simple-Spring
			  :assert-values '((x  ((0 inf) std)))
			  :text "Start with initial velocity")))
    (format *qsim-trace*
"~2%============================================================================
====

The simple spring without friction.

Only three behaviors are simulated.

Why 3 and not one ?

We don't know that the M+ function between the accelleration and the position
is symmetric. It can be weeker on one side and stronger on the other. Thus
the peaks on each side of the rest position are not related: there is three
possible
behaviors for the absolute value of the position.
We would get only one behavior without that variable.

Compare these results with those obtained with NIC.

================================================================================
~2%")

    (qsim initial)
    (qsim-display initial)
    ))


;;;*****************************************************************************
;;; The most impressive example: the spring with friction
;;;*****************************************************************************

(define-QDE Spring-with-friction
            (text "Spring with friction")
  (quantity-spaces
    (x     (minf 0 inf)   "Position")
    (v     (minf 0 inf)   "Velocity")
    (a     (minf 0 inf)   "Accelleration")
    (ff    (minf 0 inf)   "Fluid friction")
    (fs    (minf 0 inf)   "Spring force"))
  (constraints
    ((d/dt x v))
    ((d/dt v a))
    ((m- x fs)      (0 0)   (minf inf)  (inf minf))
    ((m- v ff)      (0 0)  (minf inf)  (inf minf))
    ((add fs ff a)))
  (history x v)
  (layout  (x nil fs)
           (v -v-  ff)
           (a nil nil))
  (no-new-landmarks a)
  (ignore-qdirs a))


(defun damped-spring-w-tracing ()
  (qsim-cleanup)
  (let ((*trace-energy-filter* t)
        (*state-limit* 50)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (initial
          (make-new-state :from-qde Spring-with-friction
			  :assert-values '((x ((0 inf) std))))))
    (format *qsim-trace*
"~2%============================================================================
====

The Spring with Friction.

Spurious behaviors are not ruled out. Look at the shape of the tree.
Notice that sometimes EC must backtrack very far to find an inconsistency
(see behavior 3).

================================================================================
~2%")
    (qsim initial)
    (qsim-display initial)
    ))

(defun damped-spring ()
  (qsim-cleanup)
  (let ((*trace-energy-filter* nil)
        (*perform-energy-analysis* t)
        (*trace-decomposition* t)
        (*state-limit* 40)
        (initial
          (make-new-state :from-qde Spring-with-friction
			  :assert-values '((x ((0 inf) std))))))
    (format *qsim-trace*
"~2%============================================================================
====

The Spring with Friction.

We get: - An infinite behavior with decreasing oscillations
        - An infinite set of quiescent behaviors. The system can reach
quiescence
each time it goes towards its rest position.

Compare these results with those obtained with NIC.

================================================================================
~2%")
    (qsim initial)
    (qsim-display initial)
    ))


;;;=============================================================================
;;;                  T H A T S   A L L ,   F O L K S . . .
;;;=============================================================================
