;;; -*- Syntax: Common-lisp; Package: qsim -*-
;       Copyright (c) 1987, Benjamin Kuipers.
(in-package :qsim)

; The home-heater with thermostat illustrates the heater going on and off.
;  - a good demonstration of region-transitions within the same structure.
;  - branches according to possible too-cold temperature for heater.

(define-QDE HOME-HEATER-THERMO
  (text "Thermostatic control of home heater")
  (quantity-spaces
    (heat    (0 inf)                   "Heat content")
    (mass    (0 inf)                   "Thermal mass")
    (TempIn  (0 RoomTemp inf)          "Temp(inside)")
    (TempOut (0 Cold RoomTemp Hot inf) "Temp(outside)")
    (TempSet (0 RoomTemp inf)          "Temp(set)")
    (dTemp   (minf 0 inf)              "dTemp(in,out)")
    (error   (minf Lo 0 Hi inf)        "error=in-set")
    (R       (0 inf)                   "Heat flow resistance")
    (HFout   (minf 0 inf)              "Heat flow (to environment)")
    (HFin    (0 On)                    "Heat flow (from heater)")
    (netHF   (minf 0 inf)              "net Heat Flow"))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)      (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)      (RoomTemp 0 RoomTemp) )
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF))
    ((constant HFin))
    ((constant Mass))
    ((constant TempOut))
    ((constant TempSet))
    ((constant R)))
  (transitions
    ((error (Lo dec)) -> turn-furnace-on)
    ((error (Hi inc)) -> turn-furnace-off))
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF)))

(defun low-outside-temperature ()
  (let ((start (make-new-state :from-qde Home-Heater-Thermo
			       :assert-values '((HFin (0 std))
						(Heat ((0 inf) nil))
						(TempIn (RoomTemp nil))
						(Mass ((0 inf) std))
						(TempOut (Cold std))
						(TempSet (RoomTemp std))
						(R ((0 inf) std)))
			       :text "Room temperature; furnace off.")))
    (qsim start)
    (qsim-display start)))

(defun turn-furnace-on (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-thermo
			   :assert       '((HFin  (On std))
					   (netHF ((0 inf) nil)))
			   :inherit-qmag :rest))

(defun turn-furnace-off (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-thermo
			   :assert       '((HFin  (0 std))
					   (netHF ((minf 0) nil)))
			   :inherit-qmag :rest))


; Heater with proportional control.
;   Modified to allow the heater or cooler to have a max value.

(define-QDE HOME-HEATER-PROP
  (text "Proportional control of home heater/cooler")
  (quantity-spaces
    (heat    (0 inf)                   "Heat content")
    (mass    (0 inf)                   "Thermal mass")
    (TempIn  (0 RoomTemp inf)          "Temp(inside)")
    (TempOut (0 Cold RoomTemp Hot inf) "Temp(outside)")
    (TempSet (0 RoomTemp inf)          "Temp(set)")
    (dTemp   (minf 0 inf)              "dTemp(in,out)")
    (error   (minf 0 inf)              "error=in-set")
    (R       (0 inf)                   "Heat flow resistance")
    (HFout   (minf 0 inf)              "Heat flow (to environment)")
    (HFin    (MaxCool 0 MaxHeat)       "Heat flow (from heater/cooler)")
    (netHF   (minf 0 inf)              "net Heat Flow"))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)      (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)      (RoomTemp 0 RoomTemp) )
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((m- error HFin)                 (0 0) ; (inf MaxCool) (minf MaxHeat)
     )
    ((d/dt Heat netHF))
    ((constant Mass))
    ((constant TempOut))
    ((constant TempSet))
    ((constant R)))
  (transitions
    ((HFin (MaxHeat inc)) -> constant-max-heating)
    ((HFin (MaxCool dec)) -> constant-max-cooling))
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF)))


(defun low-temperature-prop ()
  (let* ((normal (make-new-state :from-qde Home-Heater-Prop
				 :assert-values '((HFin (0 std))
						  (Heat ((0 inf) std))
						  (TempIn (RoomTemp nil))
						  (Mass ((0 inf) std))
						  (TempOut (RoomTemp std))
						  (TempSet (RoomTemp std))
						  (R ((0 inf) std)))
				 :text "Normal state; room temperature."))
	 (start (make-new-state :from-state normal
				:inherit '(Mass TempSet R TempIn)
				:assert-values '((HFin (0 nil))
						 (TempOut (Cold std)))
				:text "Suddenly cold outdoors; furnace off.")))
    (qsim start)
    (qsim-display start
		  :reference-states `((normal ,normal)))
    t))

; Switch to the constant-heat-input model when the system reaches MaxHeat.

(defun constant-max-heating (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       constant-home-heater
			   :assert       '((HFin  (MaxHeat std)))
			   :inherit-qmag :rest))

(defun constant-max-cooling (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       constant-home-heater
			   :assert       '((HFin  (MaxCool std)))
			   :inherit-qmag :rest))


; This models the heater system with constant input, either HFin=0 for the
; broken system, or HFin=MaxHeat for the system pinned to its top end.

(define-QDE CONSTANT-HOME-HEATER
  (text "Constant input from heater/cooler")
  (quantity-spaces
    (heat    (0 inf)                   "Heat content")
    (mass    (0 inf)                   "Thermal mass")
    (TempIn  (0 RoomTemp inf)          "Temp(inside)")
    (TempOut (0 Cold RoomTemp Hot inf) "Temp(outside)")
    (TempSet (0 RoomTemp inf)          "Temp(set)")
    (dTemp   (minf 0 inf)              "dTemp(in,out)")
    (error   (minf 0 inf)              "error=in-set")
    (R       (0 inf)                   "Heat flow resistance")
    (HFout   (minf 0 inf)              "Heat flow (to environment)")
    (HFin    (MaxCool 0 MaxHeat)       "Heat flow (from heater/cooler)")
    (netHF   (minf 0 inf)              "net Heat Flow"))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)      (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)      (RoomTemp 0 RoomTemp) )
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF))
    ((constant Mass))
    ((constant TempOut))
    ((constant TempSet))
    ((constant R))
    ((constant HFin)))
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF)))

; The broken furnace starts off at RoomTemp, but with no heat input.

(defun low-temperature-broken-heater ()
  (let* ((normal (make-new-state :from-qde Constant-Home-Heater
				 :assert-values '((HFin (0 std))
						  (Heat ((0 inf) std))
						  (TempIn (RoomTemp nil))
						  (Mass ((0 inf) std))
						  (TempOut (RoomTemp std))
						  (TempSet (RoomTemp std))
						  (R ((0 inf) std)))
				 :text "Normal state; room temperature."))
	 (start (make-new-state :from-state normal
				:inherit '(Mass TempSet R TempIn)
				:assert-values '((HFin (0 nil))
						 (TempOut (Cold std)))
				:text "Suddenly cold outdoors; furnace off.")))
    (qsim start)
    (qsim-display start
		  :reference-states `((normal ,normal)))))


 
